"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessClamscan = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const path = require("path");
const aws_ec2_1 = require("monocdk/aws-ec2");
const aws_efs_1 = require("monocdk/aws-efs");
const aws_events_1 = require("monocdk/aws-events");
const aws_events_targets_1 = require("monocdk/aws-events-targets");
const aws_iam_1 = require("monocdk/aws-iam");
const aws_lambda_1 = require("monocdk/aws-lambda");
const aws_lambda_destinations_1 = require("monocdk/aws-lambda-destinations");
const aws_lambda_event_sources_1 = require("monocdk/aws-lambda-event-sources");
const aws_s3_1 = require("monocdk/aws-s3");
const aws_sqs_1 = require("monocdk/aws-sqs");
const monocdk_1 = require("monocdk");
const monocdk_nag_1 = require("monocdk-nag");
/**
  An [aws-cdk](https://github.com/aws/aws-cdk) construct that uses [ClamAV®](https://www.clamav.net/).
  to scan objects in Amazon S3 for viruses. The construct provides a flexible interface for a system
  to act based on the results of a ClamAV virus scan.

  The construct creates a Lambda function with EFS integration to support larger files.
  A VPC with isolated subnets, a S3 Gateway endpoint will also be created.

  Additionally creates an twice-daily job to download the latest ClamAV definition files to the
  Virus Definitions S3 Bucket by utilizing an EventBridge rule and a Lambda function and
  publishes CloudWatch Metrics to the 'serverless-clamscan' namespace.

  __Important O&M__:
  When ClamAV publishes updates to the scanner you will see “Your ClamAV installation is OUTDATED” in your scan results.
  While the construct creates a system to keep the database definitions up to date, you must update the scanner to
  detect all the latest Viruses.

  Update the docker images of the Lambda functions with the latest version of ClamAV by re-running `cdk deploy`.

  Successful Scan Event format
  ```json
  {
     "source": "serverless-clamscan",
     "input_bucket": <input_bucket_name>,
     "input_key": <object_key>,
     "status": <"CLEAN"|"INFECTED"|"N/A">,
     "message": <scan_summary>,
   }
  ```

  Note: The Virus Definitions bucket policy will likely cause a deletion error if you choose to delete
  the stack associated in the construct. However since the bucket itself gets deleted, you can delete
  the stack again to resolve the error.
 */
class ServerlessClamscan extends monocdk_1.Construct {
    /**
     * Creates a ServerlessClamscan construct.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param props A `ServerlessClamscanProps` interface.
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this._efsRootPath = '/lambda';
        this._efsMountPath = `/mnt${this._efsRootPath}`;
        this._efsDefsPath = 'virus_database/';
        if (!props.onResult) {
            this.resultBus = new aws_events_1.EventBus(this, 'ScanResultBus');
            this.resultDest = new aws_lambda_destinations_1.EventBridgeDestination(this.resultBus);
            this.infectedRule = new aws_events_1.Rule(this, 'InfectedRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked INFECTED',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['INFECTED'],
                        },
                    },
                },
            });
            this.cleanRule = new aws_events_1.Rule(this, 'CleanRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked CLEAN',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['CLEAN'],
                        },
                    },
                },
            });
        }
        else {
            this.resultDest = props.onResult;
        }
        if (!props.onError) {
            this.errorDeadLetterQueue = new aws_sqs_1.Queue(this, 'ScanErrorDeadLetterQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            });
            this.errorDeadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorDeadLetterQueue.queueArn],
            }));
            this.errorQueue = new aws_sqs_1.Queue(this, 'ScanErrorQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                deadLetterQueue: {
                    maxReceiveCount: 3,
                    queue: this.errorDeadLetterQueue,
                },
            });
            this.errorQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorQueue.queueArn],
            }));
            this.errorDest = new aws_lambda_destinations_1.SqsDestination(this.errorQueue);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this.errorDeadLetterQueue, [
                { id: 'AwsSolutions-SQS3', reason: 'This queue is a DLQ.' },
            ]);
        }
        else {
            this.errorDest = props.onError;
        }
        const vpc = new aws_ec2_1.Vpc(this, 'ScanVPC', {
            subnetConfiguration: [
                {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
                    name: 'Isolated',
                },
            ],
        });
        vpc.addFlowLog('FlowLogs');
        this._s3Gw = vpc.addGatewayEndpoint('S3Endpoint', {
            service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
        });
        const fileSystem = new aws_efs_1.FileSystem(this, 'ScanFileSystem', {
            vpc: vpc,
            encrypted: props.efsEncryption === false ? false : true,
            lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_7_DAYS,
            performanceMode: aws_efs_1.PerformanceMode.GENERAL_PURPOSE,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            securityGroup: new aws_ec2_1.SecurityGroup(this, 'ScanFileSystemSecurityGroup', {
                vpc: vpc,
                allowAllOutbound: false,
            }),
        });
        const lambda_ap = fileSystem.addAccessPoint('ScanLambdaAP', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '755',
            },
            posixUser: {
                gid: '1000',
                uid: '1000',
            },
            path: this._efsRootPath,
        });
        const logs_bucket = (_b = props.defsBucketAccessLogsConfig) === null || _b === void 0 ? void 0 : _b.logsBucket;
        const logs_bucket_prefix = (_c = props.defsBucketAccessLogsConfig) === null || _c === void 0 ? void 0 : _c.logsPrefix;
        if (logs_bucket === true || logs_bucket === undefined) {
            this.defsAccessLogsBucket = new aws_s3_1.Bucket(this, 'VirusDefsAccessLogsBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: monocdk_1.RemovalPolicy.RETAIN,
                serverAccessLogsPrefix: 'access-logs-bucket-logs',
                blockPublicAccess: {
                    blockPublicAcls: true,
                    blockPublicPolicy: true,
                    ignorePublicAcls: true,
                    restrictPublicBuckets: true,
                },
            });
            this.defsAccessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:*'],
                resources: [
                    this.defsAccessLogsBucket.arnForObjects('*'),
                    this.defsAccessLogsBucket.bucketArn,
                ],
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: {
                    Bool: {
                        'aws:SecureTransport': false,
                    },
                },
            }));
        }
        else if (logs_bucket != false) {
            this.defsAccessLogsBucket = logs_bucket;
        }
        const defs_bucket = new aws_s3_1.Bucket(this, 'VirusDefsBucket', {
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            serverAccessLogsBucket: this.defsAccessLogsBucket,
            serverAccessLogsPrefix: logs_bucket === false ? undefined : logs_bucket_prefix,
            blockPublicAccess: {
                blockPublicAcls: true,
                blockPublicPolicy: true,
                ignorePublicAcls: true,
                restrictPublicBuckets: true,
            },
        });
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:*'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                Bool: {
                    'aws:SecureTransport': false,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                StringEquals: {
                    'aws:SourceVpce': this._s3Gw.vpcEndpointId,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:PutBucketPolicy', 's3:DeleteBucketPolicy'],
            resources: [defs_bucket.bucketArn],
            notPrincipals: [new aws_iam_1.AccountRootPrincipal()],
        }));
        this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
        }));
        this._scanFunction = new aws_lambda_1.DockerImageFunction(this, 'ServerlessClamscan', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/scan'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            onSuccess: this.resultDest,
            onFailure: this.errorDest,
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(lambda_ap, this._efsMountPath),
            vpc: vpc,
            vpcSubnets: { subnets: vpc.isolatedSubnets },
            allowAllOutbound: false,
            timeout: monocdk_1.Duration.minutes(15),
            memorySize: 10240,
            reservedConcurrentExecutions: props.reservedConcurrency,
            environment: {
                EFS_MOUNT_PATH: this._efsMountPath,
                EFS_DEF_PATH: this._efsDefsPath,
                DEFS_URL: defs_bucket.virtualHostedUrlForObject(),
                POWERTOOLS_METRICS_NAMESPACE: 'serverless-clamscan',
                POWERTOOLS_SERVICE_NAME: 'virus-scan',
            },
        });
        if (this._scanFunction.role) {
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this._scanFunction.role, [
                { id: 'AwsSolutions-IAM4', reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch. The AWSLambdaVPCAccessExecutionRole is required for functions with VPC access to manage elastic network interfaces.' },
            ]);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this._scanFunction.role, [{
                    id: 'AwsSolutions-IAM5',
                    reason: 'The EFS mount point permissions are controlled through a condition which limit the scope of the * resources.',
                }], true);
        }
        this._scanFunction.connections.allowToAnyIpv4(aws_ec2_1.Port.tcp(443), 'Allow outbound HTTPS traffic for S3 access.');
        defs_bucket.grantRead(this._scanFunction);
        const download_defs = new aws_lambda_1.DockerImageFunction(this, 'DownloadDefs', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/download_defs'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            timeout: monocdk_1.Duration.minutes(5),
            memorySize: 1024,
            environment: {
                DEFS_BUCKET: defs_bucket.bucketName,
                POWERTOOLS_SERVICE_NAME: 'freshclam-update',
            },
        });
        const stack = monocdk_1.Stack.of(this);
        if (download_defs.role) {
            const download_defs_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${download_defs.role.roleName}/${download_defs.functionName}`;
            const download_defs_assumed_principal = new aws_iam_1.ArnPrincipal(download_defs_role);
            defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:PutObject*'],
                resources: [defs_bucket.arnForObjects('*')],
                notPrincipals: [download_defs.role, download_defs_assumed_principal],
            }));
            defs_bucket.grantReadWrite(download_defs);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(download_defs.role, [{
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                }]);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(download_defs.role, [{
                    id: 'AwsSolutions-IAM5',
                    reason: 'The function is allowed to perform operations on all prefixes in the specified bucket.',
                }], true);
        }
        new aws_events_1.Rule(this, 'VirusDefsUpdateRule', {
            schedule: aws_events_1.Schedule.rate(monocdk_1.Duration.hours(12)),
            targets: [new aws_events_targets_1.LambdaFunction(download_defs)],
        });
        const init_defs_cr = new aws_lambda_1.Function(this, 'InitDefs', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_8,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../assets/lambda/code/initialize_defs_cr')),
            handler: 'lambda.lambda_handler',
            timeout: monocdk_1.Duration.minutes(5),
        });
        download_defs.grantInvoke(init_defs_cr);
        if (init_defs_cr.role) {
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(init_defs_cr.role, [{
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                }]);
        }
        new monocdk_1.CustomResource(this, 'InitDefsCr', {
            serviceToken: init_defs_cr.functionArn,
            properties: {
                FnName: download_defs.functionName,
            },
        });
        if (props.buckets) {
            props.buckets.forEach((bucket) => {
                this.addSourceBucket(bucket);
            });
        }
    }
    /**
     * Sets the specified S3 Bucket as a s3:ObjectCreate* for the ClamAV function.
       Grants the ClamAV function permissions to get and tag objects.
       Adds a bucket policy to disallow GetObject operations on files that are tagged 'IN PROGRESS', 'INFECTED', or 'ERROR'.
     * @param bucket The bucket to add the scanning bucket policy and s3:ObjectCreate* trigger to.
     */
    addSourceBucket(bucket) {
        this._scanFunction.addEventSource(new aws_lambda_event_sources_1.S3EventSource(bucket, { events: [aws_s3_1.EventType.OBJECT_CREATED] }));
        bucket.grantRead(this._scanFunction);
        this._scanFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
            resources: [bucket.arnForObjects('*')],
        }));
        if (this._scanFunction.role) {
            const stack = monocdk_1.Stack.of(this);
            const scan_assumed_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${this._scanFunction.role.roleName}/${this._scanFunction.functionName}`;
            const scan_assumed_principal = new aws_iam_1.ArnPrincipal(scan_assumed_role);
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [bucket.bucketArn, bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
                resources: [bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            // Need the assumed role for the not Principal Action with Lambda
            bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:GetObject'],
                resources: [bucket.arnForObjects('*')],
                notPrincipals: [this._scanFunction.role, scan_assumed_principal],
                conditions: {
                    StringEquals: {
                        's3:ExistingObjectTag/scan-status': [
                            'IN PROGRESS',
                            'INFECTED',
                            'ERROR',
                        ],
                    },
                },
            }));
        }
    }
}
exports.ServerlessClamscan = ServerlessClamscan;
_a = JSII_RTTI_SYMBOL_1;
ServerlessClamscan[_a] = { fqn: "monocdk-serverless-clamscan.ServerlessClamscan", version: "1.1.34" };
//# sourceMappingURL=data:application/json;base64,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