****A Python package for scraping and parsing Flik Menu Data for Hawai'i Preparatory Academy****

This package is designed to help integrate the menu service of Flik / Nutrislice with any 3rd party application that HPA cares to integrate it with by creating a couple helpful tools to pull and manage menu data.

Part of the difficulty in building this package was that there wasn't any documentation provided by Nutrislice. After having a couple emails back and forth with their IT people it became clear that they weren't going to give me access to the API links. 

I did some digging into the webpage that they create to serve the data to the school in a nice readable way. I discovered the API links in the webpages local storage on the client side browsing application. 
![Local Storage](assets/images/img_1.png)

The image below shows the webpage that serves the data in which the python package requests from.
![Webpage](assets/images/img.png)


# Documentation
The package can be installed using pip:

```
pip3 install hpa-menu-scraper
```
The package requires python 3.9 or higher. 
The package's only dependency is the **requests** package.
Pip should automatically resolve this though. 

**Importing into a python script**
The way I have found importing the package and its module to be most intuitive is to use:
```python
import hpa_menu.parse as parser

# Then using the package to create a python dict object of the week's menu data can be as easy as:
weeks_dictionary_data = parser.collect_week("YYYY-MM-DD") # Date must be in this format.
```
The ```collect_week()``` and ```collect_month()``` function's take 1 required argument and 3 optional arguments. 
The ```date``` argument for the ```collect_week()``` function must be a string in the "YYYY-MM-DD" format. 

The optional arguments for the `collect_month()` function are:
`full_week=False(default)`: True will include a whole weeks data even if some days in the begginning or ending week doesn't occur in the given month. For example if Feb 1 is a thursday, the dataset will also include the Mon Tues Wed of the previous January. False will not include another month's data.
`nutrition_info=True(default)`: True will include all the nutritional info for each dish in the nested dictionary with "nutrition info" as the key
`ingredients=True(default)`: True will include all the ingrediends that Flik provides for each dish in the nested dictionary with "ingredients" as the key.

## Examples

```python
import hpa_menu.parse as parser

# Collect the week of April 11, 2022 menu data. Without ingredients or nutrition info. 
menu_data_dict = parser.collect_week("2022-04-11", ingredients=False, nutrition_info=False)
"""
{'2022-04-11': {'Entree/Sides': [{'name': 'Pork Loin with Mango Chutney'}, {'name': 'Steamed Whole Grain 5 Blend'}, {'name': 'Steamed Butternut Squash'}], 'Vegetarian': [{'name': 'Beyond Meatless Penne Bake'}, {'name': 'Roasted Broccoli, Parmesan'}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza'}, {'name': 'Cheese Flatbread Pizza'}, {'name': 'Wild Mushroom Flatbread'}], 'Chefs Table': [{'name': 'Baked Potato'}, {'name': 'Bacon Bits'}, {'name': 'Shredded Cheddar Cheese'}, {'name': 'Sour Cream'}, {'name': 'Chopped Green Onions'}, {'name': 'Roasted Vegetables'}], 'Deli': [{'name': 'Potato Chips'}, {'name': 'Grilled Vegetables'}, {'name': 'Thinly Sliced Deli Turkey Breast'}, {'name': 'Meatless Burger'}, {'name': 'Sliced Provolone Cheese'}, {'name': 'Sliced Cheddar Cheese'}, 'Soup', {'name': 'Vegetable Miso Soup'}], 'Vegan': [{'name': 'Beyond Meatless Penne Bake'}, {'name': 'Roasted Broccoli, Parmesan'}]}, '2022-04-12': {'Entree/Sides': [{'name': 'Cheese Ravioli Pomodoro'}, {'name': 'Garlic, Cheese Bread'}, {'name': 'Zucchini, Squash, Herbs, Garlic'}], 'Vegetarian': [{'name': 'Cheese Ravioli Pomodoro'}, {'name': 'Zucchini, Squash, Herbs, Garlic'}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza'}, {'name': 'Cheese Flatbread Pizza'}], 'Chefs Table': [{'name': 'Beef Lo Mein'}, {'name': 'Veggie Stir Fry'}, {'name': 'Vegetable Spring Roll'}, {'name': 'Jasmine Rice, Garbanzo, Radish Seeds'}], 'Deli': [{'name': 'Potato Chips'}, {'name': 'Grilled Vegetables'}, {'name': 'Thinly Sliced Deli Turkey Breast'}, {'name': 'Meatless Burger'}, {'name': 'Sliced Provolone Cheese'}, {'name': 'Sliced Cheddar Cheese'}, 'Soup', {'name': 'Vegetable Miso Soup'}], 'Vegan': [{'name': 'Pasta Vegetable Pomodoro'}, {'name': 'Zucchini, Squash, Herbs, Garlic'}]}, '2022-04-13': {'Entree/Sides': [{'name': 'Coconut Curry Rotisserie Chicken'}, {'name': 'Fried Brown Rice'}, {'name': 'Sesame Broccoli'}], 'Vegetarian': [{'name': 'Broccoli with Asian Tofu'}, {'name': 'Steamed Rice'}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza'}, {'name': 'Cheese Flatbread Pizza'}, {'name': 'Mediterranean Grilled Flatbread'}], 'Chefs Table': [{'name': 'Ramen Noodles'}, {'name': 'Pad Thai Rice Noodles'}, {'name': 'Egg, Tofu, Ginger Miso Ramen Bowl'}, {'name': 'Pork Bely, Tonkotsu Ramen Bowl'}, {'name': 'Chopped Green Onions'}], 'Deli': [{'name': 'Potato Chips'}, {'name': 'Sliced Smoked Ham'}, {'name': 'Meatless Burger'}, {'name': 'Grilled Vegetables'}, {'name': 'Sliced Provolone Cheese'}, {'name': 'Sliced Cheddar Cheese'}, 'Soup', {'name': 'Vegetable Miso Soup'}], 'Vegan': [{'name': 'Broccoli with Asian Tofu'}, {'name': 'Steamed Rice'}]}, '2022-04-14': {'Entree/Sides': [{'name': 'BBQ Beef Brisket'}, {'name': 'Boiled New Potatoes, Parsley'}, {'name': 'Sauteed Chard with Mushrooms'}], 'Vegetarian': [{'name': 'Beyond Bratwurst, Sauerkraut, Roll'}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza'}, {'name': 'Cheese Flatbread Pizza'}], 'Chefs Table': [{'name': 'Beef Sausage Bolognese Sauce'}, {'name': 'Penne Pasta'}, {'name': 'Marinara Sauce'}, {'name': 'Parmesan Herbed Garlic Bread'}, {'name': 'Shredded Parmesan Cheese'}, {'name': 'Chicken Caesar Salad'}], 'Deli': [{'name': 'Potato Chips'}, {'name': 'Grilled Vegetables'}, {'name': 'Grilled Chicken'}, {'name': 'Meatless Burger'}, {'name': 'Sliced Provolone Cheese'}, {'name': 'Sliced Cheddar Cheese'}, 'Soup', {'name': 'Vegetable Miso Soup'}], 'Vegan': [{'name': 'Beyond Bratwurst, Sauerkraut, Roll'}]}, '2022-04-15': {'Entree/Sides': [{'name': 'Potato Chips'}, {'name': 'Grilled Vegetables'}, {'name': 'Grilled Chicken'}, {'name': 'Meatless Burger'}, {'name': 'Sliced Provolone Cheese'}, {'name': 'Sliced Cheddar Cheese'}], 'Vegetarian': [], 'Pizza, Flatbreads': [], 'Chefs Table': [], 'Deli': [{'name': 'Potato Chips'}, {'name': 'Grilled Vegetables'}, {'name': 'Grilled Chicken'}, {'name': 'Meatless Burger'}, {'name': 'Sliced Provolone Cheese'}, {'name': 'Sliced Cheddar Cheese'}], 'Vegan': []}}
"""


# Narrow down the menu to only 1 day (April 12)
day_menu = menu_data_dict["2022-04-12"]
"""
{'Entree/Sides': [{'name': 'Cheese Ravioli Pomodoro'}, {'name': 'Garlic, Cheese Bread'}, {'name': 'Zucchini, Squash, Herbs, Garlic'}], 'Vegetarian': [{'name': 'Cheese Ravioli Pomodoro'}, {'name': 'Zucchini, Squash, Herbs, Garlic'}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza'}, {'name': 'Cheese Flatbread Pizza'}], 'Chefs Table': [{'name': 'Beef Lo Mein'}, {'name': 'Veggie Stir Fry'}, {'name': 'Vegetable Spring Roll'}, {'name': 'Jasmine Rice, Garbanzo, Radish Seeds'}], 'Deli': [{'name': 'Potato Chips'}, {'name': 'Grilled Vegetables'}, {'name': 'Thinly Sliced Deli Turkey Breast'}, {'name': 'Meatless Burger'}, {'name': 'Sliced Provolone Cheese'}, {'name': 'Sliced Cheddar Cheese'}, 'Soup', {'name': 'Vegetable Miso Soup'}], 'Vegan': [{'name': 'Pasta Vegetable Pomodoro'}, {'name': 'Zucchini, Squash, Herbs, Garlic'}]}
"""

# Find the Vegan option for that day
vegan_option = day_menu["Vegan"]
"""
[{'name': 'Pasta Vegetable Pomodoro'}, {'name': 'Zucchini, Squash, Herbs, Garlic'}]
"""
```
```python

# Collect a full month's data - April 2022. No nutrition info, but with ingredients. Also, with a bit of March because April 1st is a Friday. 
import hpa_menu.parse as parser

menu_data_dict = parser.collect_month("2022-04", full_week=True, nutrition_info=False)
"""
{'2022-03-28': {'Entree/Sides': [{'name': 'Old Fashioned Meatloaf', 'ingredients': ['Beef', ' Breadcrumbs', ' Half & Half', ' Liquid Egg ', ' Garlic', ' Ketchup', ' Canola Oil', ' Onions', ' Salt', ' Hot Sauce (vinegar red pepper salt)', ' Worcestershire Sauce with Anchovies', ' Black Pepper']}, {'name': 'Skin-on Mashed Potatoes', 'ingredients': ['Unsalted Butter', ' Whole Milk', ' Potatoes', ' Salt', ' Black Pepper']}, {'name': 'Sauteed Kale', 'ingredients': ['Garlic ', ' Kale', ' Canola Oil', ' Salt', ' Black Pepper']}, {'name': 'Mushroom Beef Gravy', 'ingredients': ['Base Beef Low Sodium Gluten Free Minors', ' Carrots Fresh Bias Cut', ' Celery Fresh Chopped', ' Flour Arrowroot', ' Garlic Cloves Peeled Fresh Chopped', ' Leeks Fresh Chopped', ' Mushrooms', ' Canola Oil', ' Onions Red Fresh Chopped 1/8"', ' Spice Bay Leaf', ' Tomato Paste Canned', ' Water Cold', ' Water Tap', ' Burgundy Cooking Wine']}], 'Vegetarian': [{'name': 'Garbanzo Bean Vegetable Loaf', 'ingredients': ['Garbanzo Beans', ' Carrots', ' Celery', ' Garlic ', ' Canola Oil', ' Oil', ' Onions', ' Parsley', ' Salt', ' Poultry Seasoning', ' Black Pepper', ' Tofu']}], 'Pizza, Flatbreads': [{'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Lemongrass Chili Chicken, Coconut', 'ingredients': ['Green Beans', ' Chicken ', ' Cilantro', ' Garlic ', ' Lemongrass', ' Canola Oil', ' Onions', ' Chili Peppers', ' Fish Sauce', ' Sugar', ' Coconut Water']}, {'name': 'Thai Coconut Curry Vegetables, Tofu', 'ingredients': ['Base Vegetable Low Sodium Gluten Free No MSG Minors *Menu More', ' Basil', ' Coconut Milk', ' Eggplant', ' Makrut Lime Leaves', ' Onions', ' Tamarind Paste', ' Yellow Curry Paste', ' Red Bell Peppers', ' Fingerling Potatoes', ' Soy Sauce', ' Sugar', ' Tofu', ' Water Cold']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}, {'name': 'Macaroni Salad', 'ingredients': ['Celery', ' Light Mayonnaise', ' Onions', ' Parsley', ' Elbow Macaroni', ' Green Bell Peppers', ' Salt', ' Black Pepper']}, {'name': 'Papaya Tomato Som Tam Salad', 'ingredients': ['Basil', ' Green Beans', ' Garlic ', ' Lime', ' Papaya', ' Fish Sauce', ' Brown Sugar (sugar cane syrups)', ' Tomatoes', ' Water']}], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Sliced Smoked Ham', 'ingredients': ['Ham']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Sliced Swiss Cheese', 'ingredients': ['Swiss Cheese']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Garbanzo Bean Vegetable Loaf', 'ingredients': ['Garbanzo Beans', ' Carrots', ' Celery', ' Garlic ', ' Canola Oil', ' Oil', ' Onions', ' Parsley', ' Salt', ' Poultry Seasoning', ' Black Pepper', ' Tofu']}]}, '2022-03-29': {'Entree/Sides': [{'name': 'BBQ Pulled Pork', 'ingredients': ['Pork', ' Salt Kosher', ' Spice Chili Powder', ' Spice Cumin Ground', ' Spice Paprika Ground', ' Spice Pepper Black Ground', ' Spice Pepper Cayenne Ground', ' Sugar Brown Light', ' Sugar Granulated']}, {'name': 'Chili Spiced Potato Wedges', 'ingredients': ['Cilantro Fresh Chopped', ' Olive Oil', ' Green Onions', ' Potatoes', ' Salt Kosher', ' Spice Chili Powder', ' Spice Cumin Ground', ' Spice Garlic Powder', ' Spice Onion Powder', ' Spice Pepper Black Ground']}, {'name': 'Boston Baked Beans', 'ingredients': ['Navy Beans', ' Garlic', ' Ketchup', ' Dark Molasses (molasses potassium sorbate)', ' Spicy Brown Mustard', ' Canola Oil', ' Onions', ' Salt', ' Black Pepper', ' Brown Sugar (sugar cane syrups) ', ' Water']}, {'name': 'Cauliflower Mixed Vegetables', 'ingredients': ['Carrots', ' Cauliflower', ' Cilantro', ' Ginger ', ' Lemon Juice', ' Canola Oil', ' Salt', ' Spice Bay Leaf', ' Spice Cardamom Ground', ' Spice Cinnamon Stick', ' Spice Cloves Whole', ' Coriander', ' Cumin', ' Spice Cumin Seed', ' Spice Ginger Ground', ' Spice Pepper Black Whole', ' Cayenne Pepper', ' Turmeric', ' Zucchini']}, {'name': 'Corn Bread', 'ingredients': ['Baking Powder', ' Corn', ' Yellow Cornmeal', ' Egg', ' All Purpose Flour', ' Canola Oil', ' Oil', ' Green Chili Peppers', ' Salt', ' Sugar', ' Fat Free Yogurt']}], 'Vegetarian': [{'name': 'BBQ Tofu', 'ingredients': ['Oil', ' Salt Kosher', ' Barbecue Sauce', ' Spice Chili Powder', ' Spice Cumin Ground', ' Spice Paprika Ground', ' Spice Pepper Black Ground', ' Spice Pepper Cayenne Ground', ' Sugar Brown Light', ' Sugar Granulated', ' Tofu']}, {'name': 'Boston Baked Beans', 'ingredients': ['Navy Beans', ' Garlic', ' Ketchup', ' Dark Molasses (molasses potassium sorbate)', ' Spicy Brown Mustard', ' Canola Oil', ' Onions', ' Salt', ' Black Pepper', ' Brown Sugar (sugar cane syrups) ', ' Water']}, {'name': 'Cauliflower Mixed Vegetables', 'ingredients': ['Carrots', ' Cauliflower', ' Cilantro', ' Ginger ', ' Lemon Juice', ' Canola Oil', ' Salt', ' Spice Bay Leaf', ' Spice Cardamom Ground', ' Spice Cinnamon Stick', ' Spice Cloves Whole', ' Coriander', ' Cumin', ' Spice Cumin Seed', ' Spice Ginger Ground', ' Spice Pepper Black Whole', ' Cayenne Pepper', ' Turmeric', ' Zucchini']}], 'Pizza, Flatbreads': [{'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Grilled Chicken Breast', 'ingredients': ['Chicken', ' Canola Oil', ' Salt', ' Black Pepper']}, {'name': 'linguine', 'ingredients': ['Linguine', ' Salt', ' Water']}, {'name': 'Penne Pasta', 'ingredients': ['Penne ', ' Water']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}, {'name': 'Marinara Sauce', 'ingredients': ['Basil', ' Carrots', ' Garlic', ' Mushrooms', ' Canola Oil', ' Onions', ' Oregano', ' Parsley', ' Bell Peppers', ' Salt', ' Red Pepper', ' Spinach', ' Sugar', ' Tomato puree (tomato puree water tomato paste citric acid)', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)']}, {'name': 'Shredded Parmesan Cheese', 'ingredients': ['Parmesan Cheese']}], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Thinly Sliced Deli Turkey Breast', 'ingredients': ['Turkey']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Swiss Cheese', 'ingredients': ['Swiss Cheese']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'BBQ Tofu', 'ingredients': ['Oil', ' Salt Kosher', ' Barbecue Sauce', ' Spice Chili Powder', ' Spice Cumin Ground', ' Spice Paprika Ground', ' Spice Pepper Black Ground', ' Spice Pepper Cayenne Ground', ' Sugar Brown Light', ' Sugar Granulated', ' Tofu']}, {'name': 'Boston Baked Beans', 'ingredients': ['Navy Beans', ' Garlic', ' Ketchup', ' Dark Molasses (molasses potassium sorbate)', ' Spicy Brown Mustard', ' Canola Oil', ' Onions', ' Salt', ' Black Pepper', ' Brown Sugar (sugar cane syrups) ', ' Water']}, {'name': 'Cauliflower Mixed Vegetables', 'ingredients': ['Carrots', ' Cauliflower', ' Cilantro', ' Ginger ', ' Lemon Juice', ' Canola Oil', ' Salt', ' Spice Bay Leaf', ' Spice Cardamom Ground', ' Spice Cinnamon Stick', ' Spice Cloves Whole', ' Coriander', ' Cumin', ' Spice Cumin Seed', ' Spice Ginger Ground', ' Spice Pepper Black Whole', ' Cayenne Pepper', ' Turmeric', ' Zucchini']}]}, '2022-03-30': {'Entree/Sides': [{'name': 'Turkey Pot Pie Stew', 'ingredients': ['Broth Chicken Concentrate No MSG', ' Carrots', ' Half & Half', ' Arrowroot Flour', ' Mushrooms', ' Canola Oil', ' Pearl Onions', ' Parsley', ' Green Peas', ' Potatoes', ' Salt', ' Shallots', ' Bay Leaf', ' Fresh Ground Black Pepper', ' Thyme', ' Turkey Breast Skin On Petite Antibiotic Free RTC', ' Water Cold', ' White Wine']}, {'name': 'Gen-Ji-Mai Rice', 'ingredients': ['Medium Grain Brown Rice', ' Water']}, {'name': 'Roasted Broccoli', 'ingredients': ['Broccoli', ' Canola Oil', ' Salt', ' Black Pepper']}, {'name': 'Buttermilk Biscuit', 'ingredients': ['Baking Powder', ' Unsalted Butter', ' Liquid Egg Yolk', ' All Purpose Flour', ' Low Fat Buttermilk', ' Salt', ' Sugar']}], 'Vegetarian': [{'name': 'Vegetable Pot Pie', 'ingredients': ['Vegetable Stock', ' Beans Cannellini Dried', ' Unsalted Butter', ' Carrots', ' Whipping Cream', ' Puff Pastry Dough', ' Liquid Egg ', ' All Purpose Flour', ' Whole Milk', ' Mushrooms', ' Pearl Onions', ' Green Peas', ' Potatoes', ' Salt', ' Shallots', ' Black Pepper', ' Water']}, {'name': 'Gen-Ji-Mai Rice', 'ingredients': ['Medium Grain Brown Rice', ' Water']}, {'name': 'Roasted Broccoli', 'ingredients': ['Broccoli', ' Canola Oil', ' Salt', ' Black Pepper']}, {'name': 'Buttermilk Biscuit', 'ingredients': ['Baking Powder', ' Unsalted Butter', ' Liquid Egg Yolk', ' All Purpose Flour', ' Low Fat Buttermilk', ' Salt', ' Sugar']}], 'Pizza, Flatbreads': [{'name': 'Sausage Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Sausage', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Steakhouse Marinated Flank Steak', 'ingredients': ['Beef', ' Garlic Cloves Peeled Fresh', ' Mustard Dijon', ' Oil Canola', ' Oil', ' Onions Green Scallion Bunch Fresh Chopped', ' Parsley Fresh Chopped', ' Salt', ' Sauce Steak A-1', ' Sauce Worcestershire', ' Shallots Peeled Fresh Chopped', ' Spice Pepper Black Ground Coarse', ' Fresh Ground Black Pepper', ' Spice Pepper Red Chili Crushed Flakes', ' Thyme Fresh', ' Wine Red']}, {'name': 'Caesar Salad', 'ingredients': ['Bread Rye Deli Sliced 1.5 oz', ' Bread Wheat Loaf Fresh Cubed', ' Parmesan Cheese', ' Garlic Cloves Peeled Fresh Minced', ' Chopped Lettuce', ' Oil Canola', ' Parsley Italian Fresh Chopped', ' Roll Hoagie 8" Unsliced 3 oz', ' Asiago Caesar Dressing']}, {'name': 'Shredded Parmesan Cheese', 'ingredients': ['Parmesan Cheese']}, {'name': 'Cherry Tomatoes', 'ingredients': ['Tomatoes']}, {'name': 'Cilantro Lime Grilled Tofu', 'ingredients': ['Cilantro', ' Garlic ', ' Lime', ' Canola Oil', ' Oil', ' Salt', ' Fresh Ground Black Pepper', ' Tofu']}], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Thinly Sliced Deli Turkey Breast', 'ingredients': ['Turkey']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Swiss Cheese', 'ingredients': ['Swiss Cheese']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Vegetable Pot Pie', 'ingredients': ['Vegetable Stock', ' Beans Cannellini Dried', ' Unsalted Butter', ' Carrots', ' Whipping Cream', ' Puff Pastry Dough', ' Liquid Egg ', ' All Purpose Flour', ' Whole Milk', ' Mushrooms', ' Pearl Onions', ' Green Peas', ' Potatoes', ' Salt', ' Shallots', ' Black Pepper', ' Water']}, {'name': 'Gen-Ji-Mai Rice', 'ingredients': ['Medium Grain Brown Rice', ' Water']}, {'name': 'Roasted Broccoli', 'ingredients': ['Broccoli', ' Canola Oil', ' Salt', ' Black Pepper']}]}, '2022-03-31': {'Entree/Sides': [{'name': 'Macaroni & Cheese, Chicken, Broccoli', 'ingredients': ['Panko Breadcrumbs', ' Broccoli', ' Unsalted Butter', ' American Sharp Cheese', ' Cheddar Cheese', ' Chicken', ' Whipping Cream', ' All Purpose Flour', ' Whole Milk', ' Dijon Mustard', ' Oil', ' Olive Oil', ' Parsley', ' Elbow Macaroni', ' Salt', ' Black Pepper']}, {'name': 'Fried Cauliflower, Olive Oil, Tomato Sauce', 'ingredients': ['Cauliflower', ' Egg', ' All Purpose Flour', ' Garlic', ' Olive Oil', ' Onions', ' Salt', ' Tomatoes', ' Water']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}], 'Vegetarian': [{'name': 'Creamy Jalapeno Macaroni, Cheese', 'ingredients': ['Unsalted Butter', ' American Sharp Cheese', ' Cheddar Cheese', ' Whipping Cream', ' All Purpose Flour', ' Whole Milk', ' Elbow Macaroni', ' Jalapeno', ' Salt', ' Fresh Ground Black Pepper']}, {'name': 'Fried Cauliflower, Olive Oil, Tomato Sauce', 'ingredients': ['Cauliflower', ' Egg', ' All Purpose Flour', ' Garlic', ' Olive Oil', ' Onions', ' Salt', ' Tomatoes', ' Water']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}], 'Pizza, Flatbreads': [{'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Ramen Noodles', 'ingredients': ['Ramen Noodles', ' Water']}, {'name': 'Pad Thai Rice Noodles', 'ingredients': ['Rice Noodles']}, {'name': 'Pork Char Siu, Oyster Sauce', 'ingredients': ['Ginger', ' Honey', ' Ketchup', ' Pork', ' Hoisin Sauce', ' Oyster Sauce', ' Soy Sauce', ' Chinese Five Spice Seasoning (Anise Cinnamon Star Anise Cloves Ginger Sulfites)', ' Sugar', ' Rice Wine']}, {'name': 'Coconut Curry Broth', 'ingredients': ['Base Vegetable Low Sodium Gluten Free No MSG Minors *Menu More', ' Cilantro', ' Coconut Milk', ' Lime', ' Canola Oil', ' Green Onions', ' Red Curry Paste', ' Shallots', ' Water Cold']}, {'name': 'Ginger Miso Broth', 'ingredients': ['Ginger', ' Sesame Oil', ' Green Onions', ' Miso Paste', ' Soy Sauce', ' Water']}], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Sliced Deli Roast Beef', 'ingredients': ['Roast Beef']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Roasted Vegetable Penne Bake', 'ingredients': ['Basil Fresh Chopped', ' Whole Milk Mozzarella Cheese', ' Parmesan Cheese', ' Garlic Cloves Peeled Fresh Chopped', ' Mushrooms', ' Canola Oil', ' Onions', ' Onions Yellow Fresh Diced Fine', ' Oregano Fresh Chopped', ' Penne ', ' Green Bell Peppers', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Spice Pepper Red Chili Crushed Flakes', ' Squash', ' Zucchini', ' Sugar Granulated', ' Swiss Chard', ' Tomato Puree Canned', ' Tomatoes Crushed Canned Including Liquids', ' Tomatoes']}, {'name': 'Fried Cauliflower, Olive Oil, Tomato Sauce', 'ingredients': ['Cauliflower', ' Egg', ' All Purpose Flour', ' Garlic', ' Olive Oil', ' Onions', ' Salt', ' Tomatoes', ' Water']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}]}, '2022-04-01': {'Entree/Sides': [{'name': 'Fish and Chips', 'ingredients': ['Baking Powder', ' Beer', ' Cod ', ' All Purpose Flour', ' Lemon', ' Potatoes', ' Salt', ' Salt', ' Black Pepper', ' Turmeric']}, {'name': 'French Fries', 'ingredients': ['French Fries']}, {'name': 'Kale, Cranberry, Parmesan Salad', 'ingredients': ['Parmesan Cheese', ' Cranberries', ' Garlic ', ' Lemon Juice', ' Kale', ' Olive Oil', ' Green Onions', ' Salt', ' Black Pepper']}], 'Vegetarian': [{'name': 'Crispy Tofu Sticks', 'ingredients': ['Panko Breadcrumbs', ' All Purpose Flour', ' Low Fat Buttermilk', ' Salt', ' Paprika (Paprika and Silicon Dioxide)', ' Black Pepper', ' Cayenne Pepper', ' Tofu']}, {'name': 'French Fries', 'ingredients': ['French Fries']}, {'name': 'Kale, Cranberry, Parmesan Salad', 'ingredients': ['Parmesan Cheese', ' Cranberries', ' Garlic ', ' Lemon Juice', ' Kale', ' Olive Oil', ' Green Onions', ' Salt', ' Black Pepper']}], 'Pizza, Flatbreads': [{'name': 'Sausage Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Sausage', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Kalbi Chicken', 'ingredients': ['Chicken ', ' Garlic Cloves Peeled Fresh Minced', ' Ginger Root Fresh Minced', ' Juice Lemon Bulk', ' Oil Sesame', ' Sauce Soy Bulk', ' Spice Pepper Black Ground', ' Spice Pepper Red Chili Crushed Flakes', ' Spice Pepper White Ground', ' Sugar Granulated']}, {'name': 'Spicy Korean Braised Sesame Tofu', 'ingredients': ['Garlic ', ' Canola Oil', ' Sesame Oil', ' Green Onions', ' Onions', ' Salt', ' Soy Sauce', ' Korean Chili Pepper (hot pepper salt)', ' Sesame Seeds', ' Sugar', ' Tofu', ' Water']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}, {'name': 'Korean Cucumber Sesame Salad', 'ingredients': ['Cilantro', ' Cucumber', ' Sesame Oil', ' Green Onions', ' Korean Paste', ' Salt', ' Sesame Seeds', ' Sugar', ' Unseasoned Rice Wine Vinegar']}, {'name': 'Kimchi', 'ingredients': ['Cabbage ', ' Garlic', ' Ginger ', ' Green Onions', ' Korean Paste', ' Salt', ' Sugar', ' Water', ' Water']}], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Sliced Deli Roast Beef', 'ingredients': ['Roast Beef']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Crispy Tofu Sticks', 'ingredients': ['Panko Breadcrumbs', ' All Purpose Flour', ' Low Fat Buttermilk', ' Salt', ' Paprika (Paprika and Silicon Dioxide)', ' Black Pepper', ' Cayenne Pepper', ' Tofu']}, {'name': 'French Fries', 'ingredients': ['French Fries']}, {'name': 'Kale, Cranberry, Parmesan Salad', 'ingredients': ['Parmesan Cheese', ' Cranberries', ' Garlic ', ' Lemon Juice', ' Kale', ' Olive Oil', ' Green Onions', ' Salt', ' Black Pepper']}]}, '2022-04-04': {'Entree/Sides': [{'name': "Sweet n' Sour Pineapple Pork", 'ingredients': ['Baking Soda', ' Cornstarch', ' Ketchup Bulk', ' Canola Oil', ' Onions', ' Pineapple', ' Pork Shoulder Butt Bone In Raw', ' Salt Kosher', ' Sauce Steak A-1', ' Sauce Worcestershire', ' Sugar Granulated', ' Vinegar White', ' Water']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}, {'name': 'Braised Bok Choy, Sesame Seeds', 'ingredients': ['Bok Choy', ' Garlic', ' Ginger ', ' Orange Juice', ' Sesame Oil', ' Orange', ' Sesame Seeds', ' Water']}], 'Vegetarian': [{'name': 'Sweet and Sour Tofu', 'ingredients': ['Canola Oil', ' Green Onions', ' Onions', ' Green Bell Peppers', ' Red Bell Peppers', ' Pineapple', ' Sweet & Sour Sauce', ' Sesame Seeds', ' Sesame Seeds', ' Tofu']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}, {'name': 'Braised Bok Choy, Sesame Seeds', 'ingredients': ['Bok Choy', ' Garlic', ' Ginger ', ' Orange Juice', ' Sesame Oil', ' Orange', ' Sesame Seeds', ' Water']}], 'Pizza, Flatbreads': [{'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Tomato Basil WW Flatbread Pizza', 'ingredients': ['Basil', ' Part Skim Mozzarella Cheese', ' Whole Grain Flatbread', ' Canola Oil', ' Tomatoes']}], 'Chefs Table': [{'name': 'Avocado Bacon Tomato WW Toast, Egg', 'ingredients': ['Avocado', ' Bacon Sliced Raw 18/22 Frozen *Menu More', ' Whole Wheat Bread', ' Egg', ' Lemon Juice', ' Arugula', ' Canola Oil', ' Oil', ' Salt', ' Tomatoes']}], 'Deli': [{'name': 'Sliced Smoked Ham', 'ingredients': ['Ham']}, {'name': 'Grilled Chicken', 'ingredients': ['Chicken', ' Canola Oil']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Sharp Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Sweet and Sour Tofu', 'ingredients': ['Canola Oil', ' Green Onions', ' Onions', ' Green Bell Peppers', ' Red Bell Peppers', ' Pineapple', ' Sweet & Sour Sauce', ' Sesame Seeds', ' Sesame Seeds', ' Tofu']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}, {'name': 'Braised Bok Choy, Sesame Seeds', 'ingredients': ['Bok Choy', ' Garlic', ' Ginger ', ' Orange Juice', ' Sesame Oil', ' Orange', ' Sesame Seeds', ' Water']}]}, '2022-04-05': {'Entree/Sides': [{'name': 'Seasoned Wings', 'ingredients': ['Chicken ', ' Salt', ' Garlic', ' Black Pepper']}, {'name': 'South Carolina Barbeque Sauce', 'ingredients': ['Ketchup', ' Dark Molasses (molasses potassium sorbate)', ' Canola Oil', ' Onions', ' Salt', ' Hot Sauce (vinegar red pepper salt)', ' Worcestershire Sauce with Anchovies', ' Bay Leaf', ' Mustard', ' Paprika (Paprika and Silicon Dioxide)', ' Black Pepper', ' Sugar', ' Crushed tomatoes (tomatoes tomato juice citric acid calcium chloride)', ' Cider Vinegar (apple juice water)']}, {'name': 'Rice Pilaf', 'ingredients': ['Vegetable Base', ' Garlic ', ' Canola Oil', ' Onions', ' Long Grain White Rice', ' Black Pepper', ' Water']}, {'name': 'Sauteed Swiss Chard, Red Peppers', 'ingredients': ['Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Swiss Chard']}], 'Vegetarian': [{'name': 'BBQ Tofu', 'ingredients': ['Oil', ' Salt Kosher', ' Barbecue Sauce', ' Spice Chili Powder', ' Spice Cumin Ground', ' Spice Paprika Ground', ' Spice Pepper Black Ground', ' Spice Pepper Cayenne Ground', ' Sugar Brown Light', ' Sugar Granulated', ' Tofu']}, {'name': 'South Carolina Barbeque Sauce', 'ingredients': ['Ketchup', ' Dark Molasses (molasses potassium sorbate)', ' Canola Oil', ' Onions', ' Salt', ' Hot Sauce (vinegar red pepper salt)', ' Worcestershire Sauce with Anchovies', ' Bay Leaf', ' Mustard', ' Paprika (Paprika and Silicon Dioxide)', ' Black Pepper', ' Sugar', ' Crushed tomatoes (tomatoes tomato juice citric acid calcium chloride)', ' Cider Vinegar (apple juice water)']}, {'name': 'Montreal Eggplant', 'ingredients': ['Eggplant', ' Garlic ', ' Mushrooms', ' Canola Oil', ' Onions', ' Green Bell Peppers', ' Red Bell Peppers', ' Montreal Steak Seasoning']}], 'Pizza, Flatbreads': [{'name': 'Chicken, Broccoli, Flatbread', 'ingredients': ['Basil Fresh Chopped', ' Broccoli', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Chicken Breast Boneless Random FLIK ONLY Raw', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Juice Lemon Fresh Pressed', ' Oil Canola', ' Oil', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Black Pepper', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Peppers, Onions, WW Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Whole Grain Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Onions', ' Green Bell Peppers', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed No Added Salt Canned']}], 'Chefs Table': [{'name': 'All Natural Beef Burger, Bun', 'ingredients': ['Beef Patty', ' Hamburger Bun', ' Lettuce Leaf', ' Tomatoes']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Sharp Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Brioche Bun', 'ingredients': ['Egg Brioche Bun']}, {'name': 'French Fries', 'ingredients': ['French Fries']}], 'Deli': [{'name': 'Sliced Smoked Ham', 'ingredients': ['Ham']}, {'name': 'Grilled Chicken', 'ingredients': ['Chicken', ' Canola Oil']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Sharp Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'BBQ Tofu', 'ingredients': ['Oil', ' Salt Kosher', ' Barbecue Sauce', ' Spice Chili Powder', ' Spice Cumin Ground', ' Spice Paprika Ground', ' Spice Pepper Black Ground', ' Spice Pepper Cayenne Ground', ' Sugar Brown Light', ' Sugar Granulated', ' Tofu']}, {'name': 'South Carolina Barbeque Sauce', 'ingredients': ['Ketchup', ' Dark Molasses (molasses potassium sorbate)', ' Canola Oil', ' Onions', ' Salt', ' Hot Sauce (vinegar red pepper salt)', ' Worcestershire Sauce with Anchovies', ' Bay Leaf', ' Mustard', ' Paprika (Paprika and Silicon Dioxide)', ' Black Pepper', ' Sugar', ' Crushed tomatoes (tomatoes tomato juice citric acid calcium chloride)', ' Cider Vinegar (apple juice water)']}, {'name': 'Montreal Eggplant', 'ingredients': ['Eggplant', ' Garlic ', ' Mushrooms', ' Canola Oil', ' Onions', ' Green Bell Peppers', ' Red Bell Peppers', ' Montreal Steak Seasoning']}]}, '2022-04-06': {'Entree/Sides': [{'name': 'Turkey Meatloaf with Mushroom Gravy', 'ingredients': ['Base Beef Liquid Concentrate Knorr', ' Breadcrumbs', ' Carrots Fresh Bias Cut', ' Celery Fresh Chopped', ' Cornstarch', ' Liquid Egg ', ' Garlic', ' Ketchup', ' Leeks Fresh Chopped', ' Whole Milk', ' Mushrooms Button Fresh Sliced', ' Canola Oil', ' Onions Red Fresh Chopped 1/8"', ' Onions', ' Salt', ' Hot Sauce (vinegar red pepper salt)', ' Spice Bay Leaf', ' Black Pepper', ' Tomato Paste Canned', ' Ground Turkey', ' Water Cold', ' Water Tap', ' Wine Cooking Burgundy']}, {'name': 'Mashed Potatoes', 'ingredients': ['Base Vegetable Low Sodium Gluten Free No MSG Minors *Menu More', ' Olive Oil', ' Potatoes', ' Salt', ' Black Pepper', ' Water Cold']}, {'name': 'Roasted Broccolini', 'ingredients': ['Broccolini', ' Garlic ', ' Canola Oil', ' Salt']}], 'Vegetarian': [{'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Mashed Potatoes', 'ingredients': ['Base Vegetable Low Sodium Gluten Free No MSG Minors *Menu More', ' Olive Oil', ' Potatoes', ' Salt', ' Black Pepper', ' Water Cold']}, {'name': 'Balsamic Roasted Vegetables', 'ingredients': ['Basil', ' Carrots', ' Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Zucchini', ' Balsamic Vinegar']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Teriyaki Chicken', 'ingredients': ['Chicken ', ' Cornstarch', ' Ginger Root Fresh', ' Green Onions', ' Sauce Soy Bulk', ' Sugar Granulated', ' Water', ' Wine Cooking Mirin (Sweet Rice)', ' Wine Cooking Sherry']}, {'name': 'Gen-Ji-Mai Rice', 'ingredients': ['Medium Grain Brown Rice', ' Water']}, {'name': 'Teriyaki Marinated Tofu', 'ingredients': ['Cornstarch', ' Garlic', ' Ginger ', ' Honey', ' Apple Juice', ' Oil', ' Soy Sauce', ' Sugar', ' Tofu', ' Unseasoned Rice Wine Vinegar', ' Water']}, {'name': 'Macaroni Salad', 'ingredients': ['Celery', ' Mayonnaise', ' Onions', ' Parsley', ' Elbow Macaroni', ' Green Bell Peppers', ' Salt', ' Black Pepper']}], 'Deli': [{'name': 'Sliced Pastrami', 'ingredients': ['Beef Pastrami']}, {'name': 'Sliced Meatless Salami', 'ingredients': ['Meatless Salami']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Sharp Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Mashed Potatoes', 'ingredients': ['Base Vegetable Low Sodium Gluten Free No MSG Minors *Menu More', ' Olive Oil', ' Potatoes', ' Salt', ' Black Pepper', ' Water Cold']}, {'name': 'Balsamic Roasted Vegetables', 'ingredients': ['Basil', ' Carrots', ' Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Zucchini', ' Balsamic Vinegar']}]}, '2022-04-07': {'Entree/Sides': [{'name': 'Baked Salmon, No-Nut Pesto Crumbs', 'ingredients': ['White Bread', ' Salmon', ' Oil', ' Pesto Sauce']}, {'name': 'Sunshine Rice with Almonds', 'ingredients': ['Celery', ' Lemon Juice', ' Almonds', ' Canola Oil', ' Onions', ' Long Grain White Rice', ' Hot Sauce', ' Water']}, {'name': 'Balsamic Roasted Vegetables', 'ingredients': ['Basil', ' Carrots', ' Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Zucchini', ' Balsamic Vinegar']}], 'Vegetarian': [{'name': 'Caribbean Vegetable Stew', 'ingredients': ['Beans Garbanzo Dry', ' Cilantro', ' Coconut Milk', ' All Purpose Flour', ' Garlic', ' Canola Oil', ' Onions', ' Green Bell Peppers', ' Salt', ' Hot Sauce (vinegar red pepper salt)', ' Black Pepper', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)', ' Diced tomatoes (tomatoes tomato juice calcium chloride citric acid)', ' Water', ' Yuca Root']}, {'name': 'Sunshine Rice with Almonds', 'ingredients': ['Celery', ' Lemon Juice', ' Almonds', ' Canola Oil', ' Onions', ' Long Grain White Rice', ' Hot Sauce', ' Water']}], 'Pizza, Flatbreads': [{'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'White Pizza', 'ingredients': ['Part Skim Mozzarella Cheese', ' Part Skim Ricotta Cheese', ' Romano Cheese', ' Liquid Egg ', ' Flour High Gluten Bakers', ' Oil Canola', ' Parsley', ' Salt Kosher', ' Black Pepper', ' Sugar Granulated', ' Water', ' Yeast Bakers Active Dry']}], 'Chefs Table': [{'name': 'All Beef Hot Dog', 'ingredients': ['Beef Hot Dog']}, {'name': 'Beef Chili', 'ingredients': ['Beef', ' Garlic ', ' Canola Oil', ' Onions', ' Green Bell Peppers', ' Salt', ' Chili Powder', ' Cumin', ' Oregano', ' Black Pepper', ' Cayenne Pepper', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)', ' Water']}, {'name': 'Vegetarian Chili', 'ingredients': ['Heirloom Blend Beans', ' Lima Beans', ' Carrots Fresh Pieces 1"', ' Celery Fresh Chopped', ' Cilantro', ' Corn', ' Eggplant', ' Garlic Cloves Peeled Fresh', ' Garlic ', ' Leeks Fresh Chopped', ' Mushrooms', ' Canola Oil', ' Onions Yellow Fresh Chopped', ' Onions', ' Parsley Fresh Each Sprig', ' Green Bell Peppers', ' Red Bell Peppers', ' Chipotle Peppers', ' Salt', ' Spice Bay Leaf', ' Chili Powder (Chili Pepper Salt Cumin Oregano Silicon Dioxide Garlic)', ' Cumin', ' Black Pepper', ' Spice Pepper Black Whole', ' Zucchini', ' Thyme Fresh Sprig', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)', ' Tomatoes Fresh Chopped', ' Tomatoes', ' Water']}, {'name': 'Shredded Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Chopped Green Onions', 'ingredients': ['Green Onions']}], 'Deli': [{'name': 'Thinly Sliced Deli Turkey Breast', 'ingredients': ['Turkey']}, {'name': 'Thinly Sliced Deli Roast Beef', 'ingredients': ['Beef']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Sharp Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Caribbean Vegetable Stew', 'ingredients': ['Beans Garbanzo Dry', ' Cilantro', ' Coconut Milk', ' All Purpose Flour', ' Garlic', ' Canola Oil', ' Onions', ' Green Bell Peppers', ' Salt', ' Hot Sauce (vinegar red pepper salt)', ' Black Pepper', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)', ' Diced tomatoes (tomatoes tomato juice calcium chloride citric acid)', ' Water', ' Yuca Root']}, {'name': 'Sunshine Rice with Almonds', 'ingredients': ['Celery', ' Lemon Juice', ' Almonds', ' Canola Oil', ' Onions', ' Long Grain White Rice', ' Hot Sauce', ' Water']}]}, '2022-04-08': {'Entree/Sides': [{'name': 'BBQ Chicken Thighs', 'ingredients': ['Chicken ', ' Salt Kosher', ' Barbecue Sauce', ' Spice Chili Powder', ' Spice Cumin Ground', ' Spice Paprika Ground', ' Spice Pepper Black Ground', ' Spice Pepper Cayenne Ground', ' Sugar Brown Light', ' Sugar Granulated']}, {'name': 'Brown Rice Wheatberry Pilaf', 'ingredients': ['Vegetable Base', ' Carrots', ' Celery', ' Garlic', ' Wheat Berries', ' Canola Oil', ' Onions', ' Brown Rice', ' Salt', ' Bay Leaf', ' Black Pepper', ' Thyme', ' Water']}, {'name': 'Chili & Lime Roasted Corn on the Cob', 'ingredients': ['Corn', ' Lime', ' Olive Oil', ' Salt', ' Chili Powder (Chili Pepper Salt Cumin Oregano Silicon Dioxide Garlic)', ' Black Pepper']}], 'Vegetarian': [{'name': 'BBQ Tofu', 'ingredients': ['Oil', ' Salt Kosher', ' Barbecue Sauce', ' Spice Chili Powder', ' Spice Cumin Ground', ' Spice Paprika Ground', ' Spice Pepper Black Ground', ' Spice Pepper Cayenne Ground', ' Sugar Brown Light', ' Sugar Granulated', ' Tofu']}, {'name': 'Brown Rice Wheatberry Pilaf', 'ingredients': ['Vegetable Base', ' Carrots', ' Celery', ' Garlic', ' Wheat Berries', ' Canola Oil', ' Onions', ' Brown Rice', ' Salt', ' Bay Leaf', ' Black Pepper', ' Thyme', ' Water']}, {'name': 'Chili & Lime Roasted Corn on the Cob', 'ingredients': ['Corn', ' Lime', ' Olive Oil', ' Salt', ' Chili Powder (Chili Pepper Salt Cumin Oregano Silicon Dioxide Garlic)', ' Black Pepper']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Pork Bely, Tonkotsu Ramen Bowl', 'ingredients': ['Cornstarch', ' Egg Fresh Cage Free', ' Garlic Cloves Peeled Fresh', ' Garlic Cloves Peeled Fresh Chopped', ' Ginger Root Fresh', ' Ginger Root Fresh Minced', ' Honey Bulk', ' Juice Apple 100%', ' Leeks Fresh', ' Mushrooms Button Fresh', ' Noodles Ramen Tokyo Wavy Sun Noodle', ' Oil Canola', ' Green Onions', ' Onions Yellow Fresh Quartered', ' Pork Belly Trimmed Raw', ' Pork Bones Neck', ' Pork Feet', ' Salt Kosher', ' Sauce Soy Less Sodium Gluten Free Lee Kum Kee Bulk', ' Sesame Seeds', ' Sugar Granulated', ' Vinegar Rice Wine Unseasoned', ' Water', ' Wine Cooking Mirin (Sweet Rice)', ' Wine Rice (Sake)']}, {'name': 'Egg, Tofu, Ginger Miso Ramen Bowl', 'ingredients': ['Egg Fresh Cage Free', ' Flour All Purpose', ' Garlic Cloves Peeled Fresh Minced', ' Ginger Root Fresh', ' Ginger Root Fresh Grated', ' Kombu', ' Milk Buttermilk Low Fat 1%', ' Noodles Ramen Tokyo Wavy Sun Noodle', ' Sesame Oil', ' Green Onions', ' Onions Green Scallion Each Fresh Bias Cut', ' Onions Red Fresh', ' Paste Miso White', ' Salt Kosher', ' Sauce Soy Less Sodium Gluten Free Lee Kum Kee Bulk', ' Spice Pepper Black Ground', ' Sesame Seeds', ' Sugar Granulated', ' Syrup Maple 100% Pure', ' Tofu Firm Cubed 1"', ' Water', ' Wine Cooking Mirin (Sweet Rice)', ' Wine Rice (Sake)']}, {'name': 'Chopped Green Onions', 'ingredients': ['Green Onions']}], 'Deli': [{'name': 'Thinly Sliced Deli Turkey Breast', 'ingredients': ['Turkey']}, {'name': 'Thinly Sliced Deli Roast Beef', 'ingredients': ['Beef']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Sharp Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'BBQ Tofu', 'ingredients': ['Oil', ' Salt Kosher', ' Barbecue Sauce', ' Spice Chili Powder', ' Spice Cumin Ground', ' Spice Paprika Ground', ' Spice Pepper Black Ground', ' Spice Pepper Cayenne Ground', ' Sugar Brown Light', ' Sugar Granulated', ' Tofu']}, {'name': 'Brown Rice Wheatberry Pilaf', 'ingredients': ['Vegetable Base', ' Carrots', ' Celery', ' Garlic', ' Wheat Berries', ' Canola Oil', ' Onions', ' Brown Rice', ' Salt', ' Bay Leaf', ' Black Pepper', ' Thyme', ' Water']}, {'name': 'Chili & Lime Roasted Corn on the Cob', 'ingredients': ['Corn', ' Lime', ' Olive Oil', ' Salt', ' Chili Powder (Chili Pepper Salt Cumin Oregano Silicon Dioxide Garlic)', ' Black Pepper']}]}, '2022-04-11': {'Entree/Sides': [{'name': 'Pork Loin with Mango Chutney', 'ingredients': ['Garlic ', ' Ginger Root Fresh Minced', ' Juice Orange 100%', ' Mango Fresh Peeled Diced', ' Mint Fresh Chopped', ' Canola Oil', ' Onions Red Fresh Chopped Fine', ' Pork', ' Raisins', ' Salt', ' Sunflower Seeds', ' Spice Allspice Ground', ' Annatto Seed', ' Spice Cinnamon Ground', ' Cumin', ' Spice Pepper Red Chili Crushed Flakes', ' Sugar Brown Light', ' Sugar']}, {'name': 'Steamed Whole Grain 5 Blend', 'ingredients': ['Whole Grain 5 Blend', ' Water']}, {'name': 'Steamed Butternut Squash', 'ingredients': ['Squash']}], 'Vegetarian': [{'name': 'Beyond Meatless Penne Bake', 'ingredients': ['Basil', ' Meatless Beef Crumbles', ' Part Skim Mozzarella Cheese', ' Parmesan Cheese', ' Part Skim Ricotta Cheese', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Canola', ' Oil', ' Onions Yellow Fresh Diced Fine', ' Oregano Fresh Chopped', ' Pasta Penne Rigate Dry', ' Salt Kosher', ' Spice Pepper Red Chili Crushed Flakes', ' Sugar Granulated', ' Tomato Puree Canned', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Roasted Broccoli, Parmesan', 'ingredients': ['Broccoli', ' Parmesan Cheese', ' Canola Oil', ' Salt', ' Black Pepper']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Wild Mushroom Flatbread', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Flour All Purpose', ' Flour Whole Wheat', ' Garlic Cloves Peeled Fresh', ' Garlic Cloves Peeled Fresh Minced', ' Honey Bulk', ' Mushrooms Button Fresh Sliced', ' Oil Canola', ' Oil Cooking Spray', ' Olive Oil', ' Onions', ' Parsley Fresh Chopped', ' Parsley', ' Parsley Italian Fresh Chopped', ' Puree Thick and Easy', ' Rosemary Fresh Minced', ' Salt Sea', ' Seeds Pumpkin Hulled', ' Black Pepper', ' Spice Pepper Black Ground Fine', ' Thyme Fresh Minced', ' Water', ' Water Warm', ' Yeast Bakers Active Dry']}], 'Chefs Table': [{'name': 'Baked Potato', 'ingredients': ['Potatoes']}, {'name': 'Bacon Bits', 'ingredients': ['Bacon Bits']}, {'name': 'Shredded Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, {'name': 'Sour Cream', 'ingredients': ['Sour Cream']}, {'name': 'Chopped Green Onions', 'ingredients': ['Green Onions']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Thinly Sliced Deli Turkey Breast', 'ingredients': ['Turkey']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Beyond Meatless Penne Bake', 'ingredients': ['Basil', ' Meatless Beef Crumbles', ' Part Skim Mozzarella Cheese', ' Parmesan Cheese', ' Part Skim Ricotta Cheese', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Canola', ' Oil', ' Onions Yellow Fresh Diced Fine', ' Oregano Fresh Chopped', ' Pasta Penne Rigate Dry', ' Salt Kosher', ' Spice Pepper Red Chili Crushed Flakes', ' Sugar Granulated', ' Tomato Puree Canned', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Roasted Broccoli, Parmesan', 'ingredients': ['Broccoli', ' Parmesan Cheese', ' Canola Oil', ' Salt', ' Black Pepper']}]}, '2022-04-12': {'Entree/Sides': [{'name': 'Cheese Ravioli Pomodoro', 'ingredients': ['Basil', ' Garlic ', ' Olive Oil', ' Onions', ' Oregano', ' Parsley', ' Cheese Ravioli', ' Salt', ' Black Pepper', ' Sugar', ' Tomatoes']}, {'name': 'Garlic, Cheese Bread', 'ingredients': ['Unsalted Butter', ' Part Skim Mozzarella Cheese', ' Garlic ', ' Parsley', ' Demi Baguette Roll']}, {'name': 'Zucchini, Squash, Herbs, Garlic', 'ingredients': ['Basil', ' Garlic', ' Canola Oil', ' Oil', ' Oregano', ' Parsley', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}], 'Vegetarian': [{'name': 'Cheese Ravioli Pomodoro', 'ingredients': ['Basil', ' Garlic ', ' Olive Oil', ' Onions', ' Oregano', ' Parsley', ' Cheese Ravioli', ' Salt', ' Black Pepper', ' Sugar', ' Tomatoes']}, {'name': 'Zucchini, Squash, Herbs, Garlic', 'ingredients': ['Basil', ' Garlic', ' Canola Oil', ' Oil', ' Oregano', ' Parsley', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Beef Lo Mein', 'ingredients': ['Bamboo shoots (bamboo shoots water)', ' Base Vegetable Liquid Concentrate Knorr', ' Beef', ' Cabbage ', ' Carrots', ' Cornstarch', ' Garlic', ' Ginger ', ' Lo Mein Noodles', ' Canola Oil', ' Green Onions', ' Onions', ' Sauce Soy Bulk', ' Soy Sauce', ' Chinese Five Spice Seasoning (Anise Cinnamon Star Anise Cloves Ginger Sulfites)', ' Sugar Granulated', ' Water', ' Wine White Dry']}, {'name': 'Veggie Stir Fry', 'ingredients': ['Bamboo shoots (bamboo shoots water)', ' Base Vegetable Low Sodium Gluten Free No MSG Minors *Menu More', ' Cabbage ', ' Carrots', ' Cornstarch', ' Ginger ', ' Mushrooms', ' Canola Oil', ' Sesame Oil', ' Snow Peas ', ' Salt Kosher', ' Sauce Hoisin', ' Soy Sauce', ' White Pepper', ' Brown Sugar (sugar cane syrups)', ' Tofu', ' Water chestnuts (Water Chestnuts Water)', ' Water']}, {'name': 'Vegetable Spring Roll', 'ingredients': ['Avocado', ' Basil', ' Carrots', ' Cilantro', ' Cucumber', ' Ginger Root Fresh Minced', ' Noodles Bean Thread Cellophane Dry', ' Oil Canola', ' Onions Yellow Fresh Chopped', ' Paste Miso White', ' Orange Bell Peppers', ' Salt Kosher', ' Rice Paper Wrapper', ' Vinegar Rice Wine Unseasoned']}, {'name': 'Jasmine Rice, Garbanzo, Radish Seeds', 'ingredients': ['Jasmine Blend', ' Water']}], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Thinly Sliced Deli Turkey Breast', 'ingredients': ['Turkey']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Pasta Vegetable Pomodoro', 'ingredients': ['Basil', ' Parmesan Cheese', ' Eggplant', ' Garlic ', ' Mushrooms', ' Canola Oil', ' Black Olives', ' Onions', ' Penne ', ' Salt', ' Black Pepper', ' Red Pepper', ' Zucchini', ' Sugar', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)']}, {'name': 'Zucchini, Squash, Herbs, Garlic', 'ingredients': ['Basil', ' Garlic', ' Canola Oil', ' Oil', ' Oregano', ' Parsley', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}]}, '2022-04-13': {'Entree/Sides': [{'name': 'Coconut Curry Rotisserie Chicken', 'ingredients': ['Chicken', ' Coconut & Curry Seasoning']}, {'name': 'Fried Brown Rice', 'ingredients': ['Broccoli', ' Cilantro', ' Garlic ', ' Ginger ', ' Canola Oil', ' Onions', ' Rice Brown Dry', ' Soy Sauce', ' Edamame', ' Water']}, {'name': 'Sesame Broccoli', 'ingredients': ['Vegetable Stock', ' Broccoli', ' Garlic', ' Canola Oil', ' Sesame Oil', ' Salt', ' Black Pepper', ' Sesame Seeds', ' Water']}], 'Vegetarian': [{'name': 'Broccoli with Asian Tofu', 'ingredients': ['Broccoli', ' Garlic', ' Ginger ', ' Canola Oil', ' Oil', ' Sesame Oil', ' Soy Sauce', ' Red Pepper', ' Brown Sugar (sugar cane syrups)', ' Tofu']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Mediterranean Grilled Flatbread', 'ingredients': ['Cheddar Cheese', ' Liquid Egg White', ' Whole Grain Flatbread', ' Lemon Juice', ' Mushrooms', ' Canola Oil', ' Olive Oil', ' Kalamata Olives', ' Green Onions', ' Parsley', ' Potatoes', ' Salt', ' Black Pepper', ' Thyme']}], 'Chefs Table': [{'name': 'Ramen Noodles', 'ingredients': ['Ramen Noodles', ' Water']}, {'name': 'Pad Thai Rice Noodles', 'ingredients': ['Rice Noodles']}, {'name': 'Egg, Tofu, Ginger Miso Ramen Bowl', 'ingredients': ['Egg Fresh Cage Free', ' Flour All Purpose', ' Garlic Cloves Peeled Fresh Minced', ' Ginger Root Fresh', ' Ginger Root Fresh Grated', ' Kombu', ' Milk Buttermilk Low Fat 1%', ' Noodles Ramen Tokyo Wavy Sun Noodle', ' Sesame Oil', ' Green Onions', ' Onions Green Scallion Each Fresh Bias Cut', ' Onions Red Fresh', ' Paste Miso White', ' Salt Kosher', ' Sauce Soy Less Sodium Gluten Free Lee Kum Kee Bulk', ' Spice Pepper Black Ground', ' Sesame Seeds', ' Sugar Granulated', ' Syrup Maple 100% Pure', ' Tofu Firm Cubed 1"', ' Water', ' Wine Cooking Mirin (Sweet Rice)', ' Wine Rice (Sake)']}, {'name': 'Pork Bely, Tonkotsu Ramen Bowl', 'ingredients': ['Cornstarch', ' Egg Fresh Cage Free', ' Garlic Cloves Peeled Fresh', ' Garlic Cloves Peeled Fresh Chopped', ' Ginger Root Fresh', ' Ginger Root Fresh Minced', ' Honey Bulk', ' Juice Apple 100%', ' Leeks Fresh', ' Mushrooms Button Fresh', ' Noodles Ramen Tokyo Wavy Sun Noodle', ' Oil Canola', ' Green Onions', ' Onions Yellow Fresh Quartered', ' Pork Belly Trimmed Raw', ' Pork Bones Neck', ' Pork Feet', ' Salt Kosher', ' Sauce Soy Less Sodium Gluten Free Lee Kum Kee Bulk', ' Sesame Seeds', ' Sugar Granulated', ' Vinegar Rice Wine Unseasoned', ' Water', ' Wine Cooking Mirin (Sweet Rice)', ' Wine Rice (Sake)']}, {'name': 'Chopped Green Onions', 'ingredients': ['Green Onions']}], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Sliced Smoked Ham', 'ingredients': ['Ham']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Broccoli with Asian Tofu', 'ingredients': ['Broccoli', ' Garlic', ' Ginger ', ' Canola Oil', ' Oil', ' Sesame Oil', ' Soy Sauce', ' Red Pepper', ' Brown Sugar (sugar cane syrups)', ' Tofu']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}]}, '2022-04-14': {'Entree/Sides': [{'name': 'BBQ Beef Brisket', 'ingredients': ['Beef Stock', ' Beef', ' Salt', ' Bay Leaf', ' Ginger (Ginger and Sulfur Dioxide)', ' Black Pepper', ' Black Pepper', ' Cayenne Pepper', ' Brown Sugar (sugar cane syrups) ', ' Sugar', ' Water', ' Chablis Wine']}, {'name': 'Boiled New Potatoes, Parsley', 'ingredients': ['Parsley', ' New Potatoes', ' Salt', ' Black Pepper']}, {'name': 'Sauteed Chard with Mushrooms', 'ingredients': ['Garlic ', ' Ginger ', ' Mushrooms', ' Canola Oil', ' Black Pepper', ' Swiss Chard']}], 'Vegetarian': [{'name': 'Beyond Bratwurst, Sauerkraut, Roll', 'ingredients': ['Yellow Mustard', ' Hoagie Roll', ' Saurekraut (Cabbage water salt)', ' Vegan Brat Sausage']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Beef Sausage Bolognese Sauce', 'ingredients': ['Basil', ' Beef', ' Carrots', ' Garlic ', ' Onions', ' Salt', ' Italian Sausage', ' Black Pepper', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)', ' Water']}, {'name': 'Penne Pasta', 'ingredients': ['Canola Oil', ' Penne']}, {'name': 'Marinara Sauce', 'ingredients': ['Basil', ' Carrots', ' Garlic', ' Mushrooms', ' Canola Oil', ' Onions', ' Oregano', ' Parsley', ' Bell Peppers', ' Salt', ' Red Pepper', ' Spinach', ' Sugar', ' Tomato puree (tomato puree water tomato paste citric acid)', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)']}, {'name': 'Parmesan Herbed Garlic Bread', 'ingredients': ['Basil', ' French Bread', ' Parmesan Cheese', ' Garlic Cloves Peeled Fresh', ' Garlic ', ' Juice Lemon Fresh Pressed', ' Mayonnaise Regular Kraft', ' Oil Canola', ' Olive Oil', ' Salt Kosher', ' Parsley']}, {'name': 'Shredded Parmesan Cheese', 'ingredients': ['Parmesan Cheese']}, {'name': 'Chicken Caesar Salad', 'ingredients': ['Bread Rye Deli Sliced 1.5 oz', ' Bread Wheat Loaf Fresh Cubed', ' Parmesan Cheese', ' Chicken Breast Boneless Random FLIK ONLY Raw', ' Garlic Cloves Peeled Fresh Minced', ' Juice Lemon Fresh Pressed', ' Chopped Lettuce', " Mayonnaise Real Hellman's", ' Mustard Dijon', ' Mustard Whole Grain', ' Oil Canola', ' Onions Green Scallion Bunch Fresh Diced Fine', ' Parsley Italian Fresh Chopped', ' Paste Anchovy', ' Roll Hoagie 8" Unsliced 3 oz', ' Salt Kosher', ' Spice Pepper Black Cracked', ' Spice Pepper Black Ground', ' Tomatoes', ' Vinegar Red Wine']}], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Grilled Chicken', 'ingredients': ['Chicken', ' Canola Oil', ' Salt', ' Black Pepper']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}, 'Soup', {'name': 'Vegetable Miso Soup', 'ingredients': ['Carrots', ' Ginger ', ' Mushrooms', ' Mushrooms', ' Green Onions', ' Miso Paste', ' Dried Seaweed', ' Taro Root', ' Water']}], 'Vegan': [{'name': 'Beyond Bratwurst, Sauerkraut, Roll', 'ingredients': ['Yellow Mustard', ' Hoagie Roll', ' Saurekraut (Cabbage water salt)', ' Vegan Brat Sausage']}]}, '2022-04-15': {'Entree/Sides': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Grilled Chicken', 'ingredients': ['Chicken', ' Canola Oil', ' Salt', ' Black Pepper']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}], 'Vegetarian': [], 'Pizza, Flatbreads': [], 'Chefs Table': [], 'Deli': [{'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Grilled Vegetables', 'ingredients': ['Eggplant', ' Canola Oil', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}, {'name': 'Grilled Chicken', 'ingredients': ['Chicken', ' Canola Oil', ' Salt', ' Black Pepper']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Sliced Provolone Cheese', 'ingredients': ['Provolone Cheese']}, {'name': 'Sliced Cheddar Cheese', 'ingredients': ['Cheddar Cheese']}], 'Vegan': []}, '2022-04-18': {'Entree/Sides': [{'name': 'Steamed Fish with Ginger', 'ingredients': ['Cilantro', ' Striped Bass/Rockfish', ' Garlic ', ' Ginger ', ' Mushrooms', ' Canola Oil', ' Sesame Oil', ' Green Onions', ' Salt', ' Soy Sauce', ' White Pepper', ' Sugar', ' Sherry Cooking Wine']}, {'name': 'Gen-Ji-Mai Rice', 'ingredients': ['Medium Grain Brown Rice', ' Water']}, {'name': 'Braised Bok Choy, Sesame Seeds', 'ingredients': ['Bok Choy', ' Garlic', ' Ginger ', ' Orange Juice', ' Sesame Oil', ' Orange', ' Sesame Seeds', ' Water']}], 'Vegetarian': [{'name': 'Thai Curry with Tofu', 'ingredients': ['Basil', ' Coconut Milk', ' Eggplant', ' Makrut Lime Leaves', ' Onions', ' Tamarind Paste', ' Yellow Curry Paste', ' Red Bell Peppers', ' Fingerling Potatoes', ' Soy Sauce', ' Sugar', ' Tofu']}, {'name': 'Gen-Ji-Mai Rice', 'ingredients': ['Medium Grain Brown Rice', ' Water']}, {'name': 'Steamed Fresh Broccoli', 'ingredients': ['Broccoli']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed No Added Salt Canned']}], 'Chefs Table': [{'name': 'Corned Beef Reuben on Rye', 'ingredients': ['Corned Beef', ' Seeded Rye Bread', ' Cabbage Green Fresh Shredded', ' Swiss Cheese', ' Juice Lemon Fresh Pressed', ' Ketchup Fancy Pouch Dispenser', ' Oil Canola', ' Onions Yellow Fresh Grated', ' Salt Kosher', ' Sauce Worcestershire', ' Spice Caraway Seed', ' Spice Celery Seed', ' Spice Paprika Smoked', ' Sugar Granulated', ' Vinegar Apple Cider', ' Water']}, {'name': 'Traditional Coleslaw', 'ingredients': ['Cabbage', ' Carrots', ' Mayonnaise', ' Dijon Mustard', ' Salt', ' Black Pepper', ' Sugar', ' White Vinegar']}, {'name': 'French Fries', 'ingredients': ['French Fries']}], 'Deli': [{'name': 'Sliced Pastrami', 'ingredients': ['Pastrami']}, {'name': 'Sliced Smoked Ham', 'ingredients': ['Ham']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Fresh Fruit Salad', 'ingredients': ['Cantaloupe', ' Grapes', ' Honeydew Melon', ' Pineapple']}], 'Vegan': [{'name': 'Mongolian Seitan and Broccoli', 'ingredients': ['Broccoli', ' Cornstarch', ' Garlic ', ' Ginger ', ' Canola Oil', ' Green Onions', ' Soy Sauce', ' Chinese Five Spice Seasoning (Anise Cinnamon Star Anise Cloves Ginger Sulfites)', ' Seasoned Wheat Gluten Seitan', ' Red Pepper', ' Brown Sugar (sugar cane syrups)']}, {'name': 'Gen-Ji-Mai Rice', 'ingredients': ['Medium Grain Brown Rice', ' Water']}, {'name': 'Steamed Fresh Broccoli', 'ingredients': ['Broccoli']}]}, '2022-04-19': {'Entree/Sides': [{'name': 'Fried Hoisin Chicken Wings', 'ingredients': ['Chicken ', ' Garlic', ' Ginger ', ' Honey', ' Canola Oil', ' Green Onions', ' Hoisin Sauce', ' Soy Sauce', ' Red Pepper', ' Sugar']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}, {'name': 'Steamed Fresh Broccoli', 'ingredients': ['Broccoli']}], 'Vegetarian': [{'name': 'Vegetable Pad Thai with Peanuts', 'ingredients': ['Basil', ' Cabbage', ' Cilantro', ' Garlic ', ' Lime Juice', ' Rice Noodles', ' Peanuts', ' Canola Oil', ' Oil Cooking Spray', ' Green Onions', ' Tamarind Paste', ' Soy Sauce', ' Paprika (Paprika and Silicon Dioxide)', ' Brown Sugar (sugar cane syrups)', ' Tofu Firm Cubed 1"', ' Water']}], 'Pizza, Flatbreads': [{'name': 'Tomato, Cheese, WW Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Whole Grain Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed No Added Salt Canned', ' Tomatoes']}, {'name': 'Spinach, Sun Dried Tomato WW Flat', 'ingredients': ['Part Skim Mozzarella Cheese', ' Whole Grain Flatbread', ' Garlic', ' Canola Oil', ' Black Pepper', ' Spinach', ' Sun-Dried Tomatoes']}], 'Chefs Table': [{'name': 'grilled chicken breast', 'ingredients': ['Chicken', ' Canola Oil', ' Salt', ' Black Pepper']}, {'name': 'linguine', 'ingredients': ['Linguine', ' Salt', ' Water']}, {'name': 'Alfredo Sauce', 'ingredients': ['Unsalted Butter', ' Parmesan Cheese', ' Whipping Cream', ' All Purpose Flour', ' Garlic ', ' Onions', ' Parsley', ' Salt']}, {'name': 'Parmesan Herbed Garlic Bread', 'ingredients': ['Basil', ' French Bread', ' Parmesan Cheese', ' Garlic Cloves Peeled Fresh', ' Garlic ', ' Juice Lemon Fresh Pressed', ' Mayonnaise Regular Kraft', ' Oil Canola', ' Olive Oil', ' Salt Kosher', ' Parsley']}, {'name': 'Chopped Greek, Chickpea, Olive, Feta', 'ingredients': ['Beans Garbanzo Dry', ' Feta Cheese', ' Cucumber', ' Shredded Lettuce', ' Kalamata Olives', ' Onions', ' Parsley', ' Red Bell Peppers', ' Yellow Bell Peppers', ' Tomatoes']}], 'Deli': [{'name': 'Sliced Pastrami', 'ingredients': ['Pastrami']}, {'name': 'Sliced Smoked Ham', 'ingredients': ['Ham']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Fresh Fruit Salad', 'ingredients': ['Cantaloupe', ' Grapes', ' Honeydew Melon', ' Pineapple']}], 'Vegan': [{'name': 'Vegetable Pad Thai with Peanuts', 'ingredients': ['Basil', ' Cabbage', ' Cilantro', ' Garlic ', ' Lime Juice', ' Rice Noodles', ' Peanuts', ' Canola Oil', ' Oil Cooking Spray', ' Green Onions', ' Tamarind Paste', ' Soy Sauce', ' Paprika (Paprika and Silicon Dioxide)', ' Brown Sugar (sugar cane syrups)', ' Tofu Firm Cubed 1"', ' Water']}]}, '2022-04-20': {'Entree/Sides': [{'name': 'St. Louis BBQ Pork Ribs', 'ingredients': ['Pork', ' Salt', ' Hot Sauce (vinegar red pepper salt)', ' Chili Powder (Chili Pepper Salt Cumin Oregano Silicon Dioxide Garlic)', ' Cumin', ' Black Pepper', ' Sugar', ' White Vinegar']}, {'name': 'Garlic Chive Mashed Potatoes', 'ingredients': ['Unsalted Butter', ' Chives', ' Garlic Cloves Peeled Fresh', ' Fat Free Milk', ' Oil Canola', ' Potatoes', ' Salt', ' Fresh Ground Black Pepper', ' Fat Free Greek Yogurt']}, {'name': 'Sauteed Swiss Chard, Red Peppers', 'ingredients': ['Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Swiss Chard']}], 'Vegetarian': [{'name': 'Chickpeas Quinoa Burger Patty', 'ingredients': ['Garbanzo Beans', ' Panko Breadcrumbs', ' Carrots', ' Quinoa', ' Canola Oil', ' Onions', ' Green Bell Peppers', ' Bell Peppers', ' Salt', ' Cinnamon', ' Black Pepper', ' Spinach', ' Squash', ' Zucchini']}, {'name': 'Garlic Chive Mashed Potatoes', 'ingredients': ['Unsalted Butter', ' Chives', ' Garlic Cloves Peeled Fresh', ' Fat Free Milk', ' Oil Canola', ' Potatoes', ' Salt', ' Fresh Ground Black Pepper', ' Fat Free Greek Yogurt']}, {'name': 'Roasted Cauliflower with Garlic', 'ingredients': ['Cauliflower', ' Garlic ', ' Canola Oil', ' Salt', ' Black Pepper']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed No Added Salt Canned']}, {'name': 'Cheese Veggie Pizza, Cauli Crust', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Garlic Cloves Peeled Fresh Chopped', ' Mushrooms', ' Oil Olive Extra Virgin', ' Onions', ' Green Bell Peppers', ' Cauliflower Pizza Crust', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids', ' Tomatoes']}], 'Chefs Table': [{'name': 'Shio Broth, Chicken, Egg, Ramen Bowl', 'ingredients': ['Chicken Bones (for stock)', ' Chicken Breast Boneless Random FLIK ONLY Raw', ' Chicken Wings Jumbo RTC Frozen-IQF', ' Egg Fresh Cage Free', ' Flour All Purpose', ' Garlic Cloves Peeled Fresh', ' Garlic Cloves Peeled Fresh Minced', ' Ginger Root Fresh', ' Ginger Root Fresh Grated', ' Honey Bulk', ' Kombu', ' Leeks Fresh', ' Milk Buttermilk Low Fat 1%', ' Mushrooms Button Fresh', ' Noodles Ramen Tokyo Wavy Sun Noodle', ' Oil Canola', ' Oil Sesame', ' Onions Green Scallion Bunch Fresh', ' Green Onions', ' Onions Red Fresh', ' Onions Yellow Fresh Quartered', ' Salt Kosher', ' Sauce Soy Less Sodium Gluten Free Lee Kum Kee Bulk', ' Spice Pepper Black Ground', ' Sesame Seeds', ' Sugar Granulated', ' Water', ' Wine Cooking Mirin (Sweet Rice)', ' Wine Rice (Sake)']}, {'name': 'Egg, Tofu, Ginger Miso Ramen Bowl', 'ingredients': ['Egg Fresh Cage Free', ' Flour All Purpose', ' Garlic Cloves Peeled Fresh Minced', ' Ginger Root Fresh', ' Ginger Root Fresh Grated', ' Kombu', ' Milk Buttermilk Low Fat 1%', ' Noodles Ramen Tokyo Wavy Sun Noodle', ' Sesame Oil', ' Green Onions', ' Onions Green Scallion Each Fresh Bias Cut', ' Onions Red Fresh', ' Paste Miso White', ' Salt Kosher', ' Sauce Soy Less Sodium Gluten Free Lee Kum Kee Bulk', ' Spice Pepper Black Ground', ' Sesame Seeds', ' Sugar Granulated', ' Syrup Maple 100% Pure', ' Tofu Firm Cubed 1"', ' Water', ' Wine Cooking Mirin (Sweet Rice)', ' Wine Rice (Sake)']}, {'name': 'Cubed Tofu', 'ingredients': ['Tofu']}, {'name': 'Chopped Green Onions', 'ingredients': ['Green Onions']}], 'Deli': [{'name': 'Thinly Sliced Deli Turkey Breast', 'ingredients': ['Turkey']}, {'name': 'Thinly Sliced Deli Roast Beef', 'ingredients': ['Beef']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Fresh Fruit Salad', 'ingredients': ['Cantaloupe', ' Grapes', ' Honeydew Melon', ' Pineapple']}], 'Vegan': [{'name': 'Chickpeas Quinoa Burger Patty', 'ingredients': ['Garbanzo Beans', ' Panko Breadcrumbs', ' Carrots', ' Quinoa', ' Canola Oil', ' Onions', ' Green Bell Peppers', ' Bell Peppers', ' Salt', ' Cinnamon', ' Black Pepper', ' Spinach', ' Squash', ' Zucchini']}, {'name': 'Creamy Mashed Potatoes, Cashew', 'ingredients': ['Nuts Cashew Halves Unsalted', ' Olive Oil', ' Potatoes', ' Salt', ' White Pepper', ' Water', ' Dry Yeast']}, {'name': 'Roasted Cauliflower with Garlic', 'ingredients': ['Cauliflower', ' Garlic ', ' Canola Oil', ' Salt', ' Black Pepper']}]}, '2022-04-21': {'Entree/Sides': [{'name': 'Beef Stew', 'ingredients': ['Beef Bones (for stock)', ' Beef ', ' Carrots Fresh', ' Carrots', ' Celery Fresh', ' Celery', ' All Purpose Flour', ' Canola Oil', ' Onions Yellow Fresh', ' Onions', ' Parsley Fresh Each Sprig', ' Potatoes', ' Salt', ' Bay Leaf', ' Black Pepper', ' Spice Pepper Black Whole', ' Thyme Fresh Sprig', ' Tomato puree (tomato puree water tomato paste citric acid)', ' Tomatoes Fresh', ' Water']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}], 'Vegetarian': [{'name': 'Vegetable Curry', 'ingredients': ['Vegetable Stock', ' Carrots', ' Celery', ' All Purpose Flour', ' Olive Oil', ' Onions', ' Parsley', ' New Potatoes', ' Salt', ' Curry Powder', ' Black Pepper', ' Water']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}], 'Pizza, Flatbreads': [{'name': 'mediterrenean chicken flatbread bite', 'ingredients': ['Part Skim Mozzarella Cheese', ' Chicken', ' Cucumber', ' Flatbread', ' Canola Oil', ' Bell Peppers', ' Italian Dressing', ' Sauce Tzatziki', ' Tomatoes']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed No Added Salt Canned']}], 'Chefs Table': [{'name': 'Fajita Steak Salad, Chipotle Ranch', 'ingredients': ['Beef', ' Carrots', ' Cheddar Cheese', ' Monterey Jack Cheese', ' Cilantro Fresh Chopped', ' Juice Lemon Fresh Pressed', ' Lettuce', ' Oil', ' Onions', ' Peppers Chipotle in Adobo Sauce Canned', ' Salad Dressing Farmhouse Ranch Kraft Pure Bulk', ' Salt Kosher', ' Spice Chili Powder', ' Spice Cumin Seed', ' Spice Fennel Seed', ' Spice Mustard Seed', ' Spice Pepper Black Ground', ' Spice Pepper Red Chili Crushed Flakes', ' Tomatoes', ' Flour Tortilla']}], 'Deli': [{'name': 'Thinly Sliced Deli Turkey Breast', 'ingredients': ['Turkey']}, {'name': 'Thinly Sliced Deli Roast Beef', 'ingredients': ['Beef']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Fresh Fruit Salad', 'ingredients': ['Cantaloupe', ' Grapes', ' Honeydew Melon', ' Pineapple']}], 'Vegan': [{'name': 'Vegetable Curry', 'ingredients': ['Vegetable Stock', ' Carrots', ' Celery', ' All Purpose Flour', ' Olive Oil', ' Onions', ' Parsley', ' New Potatoes', ' Salt', ' Curry Powder', ' Black Pepper', ' Water']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}]}, '2022-04-22': {'Entree/Sides': [{'name': 'Chicken Tikka Masala', 'ingredients': ['Chicken', ' Cilantro', ' Cilantro', ' Whipping Cream', ' Garlic', ' Ginger ', ' Lemon Juice', ' Canola Oil', ' Onions', ' Green Bell Peppers', ' Red Bell Peppers', ' Yellow Bell Peppers', ' Jalapeno', ' Salt', ' Garam Masala', ' Coriander', ' Cumin', ' Paprika (Paprika and Silicon Dioxide)', ' Black Pepper', ' Cayenne Pepper', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)', ' Water', ' Fat Free Yogurt']}, {'name': 'Basmati Rice', 'ingredients': ['Basmati Rice', ' Water']}, {'name': 'Steamed Cauliflower', 'ingredients': ['Cauliflower', ' Canola Oil', ' Salt']}], 'Vegetarian': [{'name': 'Saag Aloo', 'ingredients': ['Whipping Cream', ' All Purpose Flour', ' Curry', ' Potatoes', ' Korma Sauce', ' Spinach', ' Water']}, {'name': 'Basmati Rice', 'ingredients': ['Basmati Rice', ' Water']}, {'name': 'Steamed Cauliflower', 'ingredients': ['Cauliflower', ' Canola Oil', ' Salt']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Mushroom, Cheese, Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Mushrooms', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed No Added Salt Canned']}], 'Chefs Table': [{'name': 'Grilled Pork Chop with Apple Chutney', 'ingredients': ['Apples Baking Fresh Peeled Cored Diced Medium 1/2"', ' Cranberries Dried', ' Ginger Root Fresh Minced', ' Juice Apple 100%', ' Lemon Fresh Zest Grated', ' Canola Oil', ' Pork', ' Salt', ' Black Pepper', ' Sugar Granulated']}, {'name': 'Panko Breaded Eggplant', 'ingredients': ['Panko Breadcrumbs', ' Liquid Egg ', ' Eggplant', ' All Purpose Flour', ' Salt', ' Fresh Ground Black Pepper']}, {'name': 'Vegan Mexican Rice', 'ingredients': ['Vegetable Base', ' Celery', ' Cilantro', ' Corn Kernels', ' Garlic', ' Canola Oil', ' Onions', ' Green Bell Peppers', ' Long Grain White Rice', ' Chipotle Salsa', ' Bay Leaf', ' Chili Powder (Chili Pepper Salt Cumin Oregano Silicon Dioxide Garlic)', ' Cumin', ' Oregano', ' Water']}, {'name': 'Cauliflower Mixed Vegetables', 'ingredients': ['Carrots', ' Cauliflower', ' Cilantro', ' Ginger ', ' Lemon Juice', ' Canola Oil', ' Salt', ' Spice Bay Leaf', ' Spice Cardamom Ground', ' Spice Cinnamon Stick', ' Spice Cloves Whole', ' Coriander', ' Cumin', ' Spice Cumin Seed', ' Spice Ginger Ground', ' Spice Pepper Black Whole', ' Cayenne Pepper', ' Turmeric', ' Zucchini']}], 'Deli': [{'name': 'Thinly Sliced Deli Turkey Breast', 'ingredients': ['Turkey']}, {'name': 'Thinly Sliced Deli Roast Beef', 'ingredients': ['Beef']}, {'name': 'Potato Chips', 'ingredients': ['Potato Chips']}, {'name': 'Fresh Fruit Salad', 'ingredients': ['Cantaloupe', ' Grapes', ' Honeydew Melon', ' Pineapple']}], 'Vegan': [{'name': 'Saag Aloo', 'ingredients': ['Whipping Cream', ' All Purpose Flour', ' Curry', ' Potatoes', ' Korma Sauce', ' Spinach', ' Water']}, {'name': 'Basmati Rice', 'ingredients': ['Basmati Rice', ' Water']}, {'name': 'Steamed Cauliflower', 'ingredients': ['Cauliflower', ' Canola Oil', ' Salt']}]}, '2022-04-25': {'Entree/Sides': [{'name': 'Shoyu Chicken', 'ingredients': ['Chicken', ' Chicken ', ' Garlic ', ' Ginger ', ' Soy Sauce', ' Sugar']}, {'name': 'Teriyaki Chicken Fried Rice', 'ingredients': ['Chicken Thigh Boneless Skinless *Menu More', ' Cornstarch', ' Egg', ' Garlic', ' Ginger Root Fresh', ' Ginger ', ' Canola Oil', ' Sesame Oil', ' Onions Green Scallion Bunch Fresh Chopped', ' Green Onions', ' Onions', ' Green Peas', ' Rice Jasmine Dry', ' Salt', ' Sauce Soy Bulk', ' Soy Sauce', ' White Pepper', ' Sugar', ' Water', ' Wine Cooking Mirin (Sweet Rice)', ' Wine Cooking Sherry']}, {'name': 'Sweet Chili Steamed Vegetables', 'ingredients': ['Asparagus', ' Mushrooms', ' Red Bell Peppers', ' Chili Sauce', ' Squash']}, {'name': 'Steamed Zucchini, Yellow Squash', 'ingredients': ['Canola Oil', ' Salt', ' Squash', ' Zucchini']}], 'Vegetarian': [{'name': 'Stir Fried Tofu, Lo Mein Noodles', 'ingredients': ['Bean Sprouts', ' Bok Choy', ' Liquid Egg ', ' Garlic ', ' Lo Mein Noodles', ' Canola Oil', ' Chili Paste', ' Soy Sauce', ' Sweet Soy Sauce', ' Shallots', ' Curry Powder', ' Tofu']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}, {'name': 'Sweet Chili Steamed Vegetables', 'ingredients': ['Asparagus', ' Mushrooms', ' Red Bell Peppers', ' Chili Sauce', ' Squash']}], 'Pizza, Flatbreads': [{'name': 'Sausage Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Sausage', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese, WW Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Whole Grain Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Kalua Pork and Cabbage', 'ingredients': ['Cabbage', ' Flavoring Liquid Smoke', ' Pork Boston Butt Boneless Raw', ' Salt Sea', ' Water']}, {'name': 'Kalua Pork', 'ingredients': ['Liquid Smoke', ' Pork', ' Salt']}, {'name': 'Chicken Long Rice Soup', 'ingredients': ['Chicken Stock', ' Chicken ', ' Garlic ', ' Ginger ', ' Rice Noodles', ' Green Onions', ' Salt', ' Water']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}, {'name': 'Oven Steamed Kabocha Squash', 'ingredients': ['Canola Oil', ' Salt', ' Black Pepper', ' Squash']}], 'Deli': [{'name': 'Turkey Breast, House Roasted', 'ingredients': ['Turkey']}, {'name': 'Rare Deli Roast Beef', 'ingredients': ['Roast Beef']}, {'name': 'Chips, Potato, 1.5 oz', 'ingredients': ['Ingredient Statement: Potatoes', ' Vegetable Oil (Sunflower', ' Corn and/or Canola Oil)', ' and Salt. Allergens: No Know Allergens']}, {'name': 'Fresh Fruit Platter', 'ingredients': ['Blueberries', ' Honeydew Melon', ' Pineapple', ' Raspberries', ' Strawberries']}], 'Vegan': [{'name': 'Stir Fried Tofu, Lo Mein Noodles', 'ingredients': ['Bean Sprouts', ' Bok Choy', ' Liquid Egg ', ' Garlic ', ' Lo Mein Noodles', ' Canola Oil', ' Chili Paste', ' Soy Sauce', ' Sweet Soy Sauce', ' Shallots', ' Curry Powder', ' Tofu']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}, {'name': 'Sweet Chili Steamed Vegetables', 'ingredients': ['Asparagus', ' Mushrooms', ' Red Bell Peppers', ' Chili Sauce', ' Squash']}]}, '2022-04-26': {'Entree/Sides': [{'name': 'Turkey Bolognese Sauce', 'ingredients': ['Basil', ' Carrots', ' Garlic Cloves Peeled Fresh Chopped', ' Garlic ', ' Oil Canola', ' Olive Oil', ' Onions', ' Onions Yellow Fresh Diced Fine', ' Oregano Fresh Chopped', ' Salt', ' Spice Pepper Red Chili Crushed Flakes', ' Sugar Granulated', ' Tomato Puree Canned', ' Tomatoes Crushed No Added Salt Canned', ' Ground Turkey', ' Red Wine']}, {'name': 'Spaghetti', 'ingredients': ['Spaghetti']}, {'name': 'Roasted Butternut Squash', 'ingredients': ['Canola Oil', ' Oil', ' Salt', ' Black Pepper', ' Squash', ' Brown Sugar (sugar cane syrups)']}, {'name': 'Parmesan Herbed Garlic Bread', 'ingredients': ['Basil', ' French Bread', ' Parmesan Cheese', ' Garlic Cloves Peeled Fresh', ' Garlic ', ' Juice Lemon Fresh Pressed', ' Mayonnaise Regular Kraft', ' Oil Canola', ' Olive Oil', ' Salt Kosher', ' Parsley']}], 'Vegetarian': [{'name': 'Vegan Lentil Bolognese', 'ingredients': ['Base Vegetable Low Sodium Gluten Free No MSG Minors *Menu More', ' Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Celery', ' Corn Kernels', ' Garlic ', ' Yellow Lentils', ' Mushrooms', ' Canola Oil', ' Onions', ' Oregano', ' Salt', ' Fresh Ground Black Pepper', ' Red Pepper', ' Tomato paste (tomato paste salt citric acid)', ' Crushed tomatoes (unpeeled tomatoes tomato puree salt citric acid)', ' Diced tomatoes (tomatoes tomato juice calcium chloride citric acid)', ' Water Cold']}, {'name': 'Roasted Butternut Squash', 'ingredients': ['Canola Oil', ' Oil', ' Salt', ' Black Pepper', ' Squash', ' Brown Sugar (sugar cane syrups)']}, {'name': 'Parmesan Herbed Garlic Bread', 'ingredients': ['Basil', ' French Bread', ' Parmesan Cheese', ' Garlic Cloves Peeled Fresh', ' Garlic ', ' Juice Lemon Fresh Pressed', ' Mayonnaise Regular Kraft', ' Oil Canola', ' Olive Oil', ' Salt Kosher', ' Parsley']}], 'Pizza, Flatbreads': [{'name': 'Cheese, WW Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Whole Grain Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Tomato Basil WW Flatbread Pizza', 'ingredients': ['Basil', ' Part Skim Mozzarella Cheese', ' Whole Grain Flatbread', ' Canola Oil', ' Tomatoes']}], 'Chefs Table': [{'name': 'Korean Fried Chicken, Rice Bowl, Egg', 'ingredients': ['Chicken Thigh Boneless Skinless *Menu More', ' Cilantro Fresh Chopped', ' Cornstarch', ' Cucumbers Fresh Peeled Each Slice', ' Egg Fresh Cage Free', ' Flour All Purpose', ' Garlic Cloves Peeled Fresh Minced', ' Ginger Root Fresh Chopped', ' Ginger Root Fresh Minced', ' Honey Bulk', ' Mushrooms Shiitake Fresh Stems Removed Sliced', ' Oil Canola', ' Oil Cooking Spray', ' Oil Sesame', ' Onions Green Scallion Bunch Fresh Chopped', ' Green Onions', ' Onions Green Scallion Bunch Fresh Sliced Thin', ' Paste Hot Red Pepper Korean (Gochujang)', ' Rice Sushi Calrose Dry', ' Salt Kosher', ' Sauce Hot & Sweet Gochujang', ' Sauce Soy Less Sodium Gluten Free Lee Kum Kee Bulk', ' Spice Garlic Granulated', ' Spice Ginger Ground', ' Spice Onion Granulated', ' Spice Pepper Black Ground Fresh', ' Spice Pepper Chili Korean Powder', ' Sesame Seeds', ' Spice Sesame Seeds Black', ' Spice Sesame Seeds Pre-Toasted', ' Sugar Granulated', ' Vinegar Rice Seasoned', ' Vinegar Rice Wine Unseasoned', ' Water']}, {'name': 'Spicy Korean Braised Sesame Tofu', 'ingredients': ['Garlic ', ' Canola Oil', ' Sesame Oil', ' Green Onions', ' Onions', ' Salt', ' Soy Sauce', ' Korean Chili Pepper (hot pepper salt)', ' Sesame Seeds', ' Sugar', ' Tofu', ' Water']}, {'name': 'Kimchi Fried Rice', 'ingredients': ['Canola Oil', ' Green Onions', ' Korean Paste', ' Rice', ' Sesame Seeds', ' Pickled Vegetables', ' Water']}, {'name': 'Soba Noodle, Kimchi, Cucumber Salad', 'ingredients': ['Cucumber', ' Lime', ' Orange Juice', ' Kimchi', ' Noodles Soba Buckwheat Dry', ' Canola Oil', ' Sesame Oil', ' Salt', ' Sesame Seeds', ' Water']}, {'name': 'Kimchi', 'ingredients': ['Cabbage ', ' Garlic', ' Ginger ', ' Green Onions', ' Korean Paste', ' Salt', ' Sugar', ' Water', ' Water']}], 'Deli': [{'name': 'Sliced Smoked Ham', 'ingredients': ['Ham']}, {'name': 'Sliced Pastrami', 'ingredients': ['Pastrami']}, {'name': 'Chips, Potato, 1.5 oz', 'ingredients': ['Ingredient Statement: Potatoes', ' Vegetable Oil (Sunflower', ' Corn and/or Canola Oil)', ' and Salt. Allergens: No Know Allergens']}, {'name': 'Fresh Fruit Platter', 'ingredients': ['Blueberries', ' Honeydew Melon', ' Pineapple', ' Raspberries', ' Strawberries']}], 'Vegan': [{'name': 'Lentil Bolognese over Zoodles', 'ingredients': ['Basil', ' Gold Beets', ' Carrots', ' Parmesan Cheese', ' Garlic ', ' Yellow Lentils', ' Olive Oil', ' Oregano', ' Salt', ' Shallots', ' Red Pepper', ' Zucchini', ' Sugar', ' Tomatoes', ' Water']}, {'name': 'Roasted Butternut Squash', 'ingredients': ['Canola Oil', ' Oil', ' Salt', ' Black Pepper', ' Squash', ' Brown Sugar (sugar cane syrups)']}, {'name': 'Parmesan Herbed Garlic Bread', 'ingredients': ['Basil', ' French Bread', ' Parmesan Cheese', ' Garlic Cloves Peeled Fresh', ' Garlic ', ' Juice Lemon Fresh Pressed', ' Mayonnaise Regular Kraft', ' Oil Canola', ' Olive Oil', ' Salt Kosher', ' Parsley']}]}, '2022-04-27': {'Entree/Sides': [{'name': 'Fried Chicken Tenders', 'ingredients': ['Breaded Chicken']}, {'name': 'Baked Tater Tots', 'ingredients': ['Tater Tot Potatoes']}, {'name': 'Vegetable Medley', 'ingredients': ['Broccoli', ' Carrots', ' Cauliflower', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini']}], 'Vegetarian': [{'name': 'Creamy Jalapeno Macaroni, Cheese', 'ingredients': ['Unsalted Butter', ' American Sharp Cheese', ' Reduced Fat Cheddar Cheese', ' Whipping Cream', ' All Purpose Flour', ' Whole Milk', ' Elbow Macaroni', ' Jalapeno', ' Salt', ' Fresh Ground Black Pepper']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese, WW Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Whole Grain Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Loco Moco', 'ingredients': ['Base Beef Low Sodium Gluten Free Minors', ' Beef Patty 20% Fat Angus All Natural Certified Humane Raw 4 oz Fresh', ' Carrots Fresh Bias Cut', ' Celery Fresh Chopped', ' Egg Liquid Whole w/Citric Acid Includes Cage Free Fresh', ' Flour Arrowroot', ' Garlic Cloves Peeled Fresh Chopped', ' Leeks Fresh Chopped', ' Oil Canola', ' Oil Cooking Spray', ' Onions Red Fresh Chopped 1/8"', ' Rice White Long Grain Parboiled Dry', ' Spice Bay Leaf', ' Tomato Paste Canned', ' Water', ' Water Cold', ' Water Tap', ' Wine Cooking Burgundy']}, {'name': 'Meatless Burger', 'ingredients': ['Meatless Burger']}, {'name': 'Macaroni Salad', 'ingredients': ['Celery', ' Mayonnaise', ' Onions', ' Parsley', ' Elbow Macaroni', ' Green Bell Peppers', ' Salt', ' Black Pepper']}, {'name': 'Mushroom Beef Gravy', 'ingredients': ['Base Beef Low Sodium Gluten Free Minors', ' Carrots Fresh Bias Cut', ' Celery Fresh Chopped', ' Flour Arrowroot', ' Garlic Cloves Peeled Fresh Chopped', ' Leeks Fresh Chopped', ' Mushrooms', ' Canola Oil', ' Onions Red Fresh Chopped 1/8"', ' Spice Bay Leaf', ' Tomato Paste Canned', ' Water Cold', ' Water Tap', ' Burgundy Cooking Wine']}], 'Deli': [{'name': 'Turkey Breast, House Roasted', 'ingredients': ['Turkey']}, {'name': 'Rare Deli Roast Beef', 'ingredients': ['Roast Beef']}, {'name': 'Chips, Potato, 1.5 oz', 'ingredients': ['Ingredient Statement: Potatoes', ' Vegetable Oil (Sunflower', ' Corn and/or Canola Oil)', ' and Salt. Allergens: No Know Allergens']}, {'name': 'Fresh Fruit Platter', 'ingredients': ['Blueberries', ' Honeydew Melon', ' Pineapple', ' Raspberries', ' Strawberries']}], 'Vegan': [{'name': 'Creamy Jalapeno Macaroni, Cheese', 'ingredients': ['Unsalted Butter', ' American Sharp Cheese', ' Reduced Fat Cheddar Cheese', ' Whipping Cream', ' All Purpose Flour', ' Whole Milk', ' Elbow Macaroni', ' Jalapeno', ' Salt', ' Fresh Ground Black Pepper']}]}, '2022-04-28': {'Entree/Sides': [{'name': 'Revved Up Beef Sloppy Joe', 'ingredients': ['Beef', ' Beets Red Fresh', ' Carrots', ' Grain Farro Dry', ' Ketchup', ' Lentils Yellow Dry', ' Mushrooms', ' Onions', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Spinach', ' Brown Sugar (sugar cane syrups)', ' Tomato puree (tomato puree water tomato paste citric acid)', ' Cider Vinegar (apple juice water)', ' Water']}, {'name': 'Brioche Bun', 'ingredients': ['Egg Brioche Bun']}, {'name': 'Garlic Roasted Potatoes', 'ingredients': ['Garlic', ' Canola Oil', ' Parsley', ' Potatoes', ' Salt', ' Paprika (Paprika and Silicon Dioxide)']}, {'name': 'Steamed Broccoli', 'ingredients': ['Broccoli']}, {'name': 'Chopped Coleslaw', 'ingredients': ['Cabbage', ' Carrots', ' Mayonnaise', ' Onions', ' Salt', ' Celery Seed', ' Black Pepper', ' Sugar', ' Cider Vinegar (apple juice water)']}], 'Vegetarian': [{'name': 'Vegetable Sloppy Joe', 'ingredients': ['Kidney Beans', ' Whole Wheat Hamburger Bun', ' Garlic ', ' Dark Molasses (molasses potassium sorbate)', ' Mushrooms', ' Dijon Mustard', ' Canola Oil', ' Onions', ' Red Bell Peppers', ' Jalapeno', ' Salt', ' Tomato Sauce', ' Black Pepper', ' Brown Sugar (sugar cane syrups) ', ' Tomato paste (tomato paste salt citric acid)']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}], 'Pizza, Flatbreads': [{'name': 'Sausage Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Sausage', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese, WW Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Whole Grain Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Grilled Chicken Caesar Salad', 'ingredients': ['Bread Rye Deli Sliced 1.5 oz', ' Bread Wheat Loaf Fresh Cubed', ' Cheese Parmesan Grated', ' Parmesan Cheese', ' Chicken Breast Boneless Random FLIK ONLY Raw', ' Garlic Cloves Peeled Fresh Minced', ' Chopped Lettuce', ' Oil Canola', ' Oil', ' Parsley Italian Fresh Chopped', ' Roll Hoagie 8" Unsliced 3 oz', ' Asiago Caesar Dressing', ' Salt Kosher', ' Spice Pepper Black Ground', ' Tomatoes']}, {'name': 'Caesar with Grilled Tofu', 'ingredients': ['Bagel Whole Wheat Sliced  La Marca 4 oz Fresh', ' Base Vegetable Low Sodium Gluten Free No MSG Minors *Menu More', ' Cornstarch', ' Garlic Cloves Peeled Fresh Crushed', ' Garlic Cloves Peeled Fresh Minced', ' Ginger Root Fresh Minced', ' Juice Lemon Fresh Pressed', ' Chopped Lettuce', ' Mustard Dijon', ' Oil Canola', ' Oil Cooking Spray', ' Parsley Fresh Minced', ' Salt Kosher', ' Salt Sea', ' Spice Pepper Black Ground', ' Spice Sesame Seeds Pre-Toasted', ' Tofu Firm Cubed 1"', ' Tofu Firm Each Slice 4 oz', ' Tomatoes', ' Water Boiling', ' Water Tap']}], 'Deli': [{'name': 'Sliced Smoked Ham', 'ingredients': ['Ham']}, {'name': 'Sliced Pastrami', 'ingredients': ['Pastrami']}, {'name': 'Chips, Potato, 1.5 oz', 'ingredients': ['Ingredient Statement: Potatoes', ' Vegetable Oil (Sunflower', ' Corn and/or Canola Oil)', ' and Salt. Allergens: No Know Allergens']}, {'name': 'Fresh Fruit Platter', 'ingredients': ['Blueberries', ' Honeydew Melon', ' Pineapple', ' Raspberries', ' Strawberries']}], 'Vegan': [{'name': 'Vegetable Sloppy Joe', 'ingredients': ['Kidney Beans', ' Whole Wheat Hamburger Bun', ' Garlic ', ' Dark Molasses (molasses potassium sorbate)', ' Mushrooms', ' Dijon Mustard', ' Canola Oil', ' Onions', ' Red Bell Peppers', ' Jalapeno', ' Salt', ' Tomato Sauce', ' Black Pepper', ' Brown Sugar (sugar cane syrups) ', ' Tomato paste (tomato paste salt citric acid)']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}]}, '2022-04-29': {'Entree/Sides': [{'name': 'Pork Tonkatsu', 'ingredients': ['Panko Breadcrumbs', ' Egg', ' All Purpose Flour', ' Canola Oil', ' Pork', ' Salt', ' Black Pepper']}, {'name': 'Gen-Ji-Mai Rice', 'ingredients': ['Medium Grain Brown Rice', ' Water']}, {'name': 'Roasted Asparagus', 'ingredients': ['Asparagus', ' Canola Oil', ' Salt', ' Fresh Ground Black Pepper']}, {'name': 'Roasted Vegetables', 'ingredients': ['Basil', ' Broccoli', ' Carrots', ' Cauliflower', ' Garlic', ' Canola Oil', ' Red Bell Peppers', ' Salt', ' Black Pepper', ' Squash', ' Zucchini', ' Balsamic Vinegar']}], 'Vegetarian': [{'name': 'Eggplant Katsu Sandwich', 'ingredients': ['White Bread', ' Panko Breadcrumbs', ' Cabbage ', ' Liquid Egg ', ' Japanese Eggplant', ' All Purpose Flour', ' Salt', ' Fresh Ground Black Pepper']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}], 'Pizza, Flatbreads': [{'name': 'Pepperoni Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Pepperoni', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}, {'name': 'Cheese, WW Flatbread Pizza', 'ingredients': ['Basil Fresh Chopped', ' Part Skim Mozzarella Cheese', ' Cheese Parmesan Pre-Shredded Fancy', ' Whole Grain Flatbread', ' Garlic Cloves Peeled Fresh Chopped', ' Oil Olive Extra Virgin', ' Salt Kosher', ' Spice Pepper Black Ground', ' Sugar Granulated', ' Tomatoes Crushed Canned Including Liquids']}], 'Chefs Table': [{'name': 'Pork Bely, Tonkotsu Ramen Bowl', 'ingredients': ['Cornstarch', ' Egg Fresh Cage Free', ' Garlic Cloves Peeled Fresh', ' Garlic Cloves Peeled Fresh Chopped', ' Ginger Root Fresh', ' Ginger Root Fresh Minced', ' Honey Bulk', ' Juice Apple 100%', ' Leeks Fresh', ' Mushrooms Button Fresh', ' Noodles Ramen Tokyo Wavy Sun Noodle', ' Oil Canola', ' Green Onions', ' Onions Yellow Fresh Quartered', ' Pork Belly Trimmed Raw', ' Pork Bones Neck', ' Pork Feet', ' Salt Kosher', ' Sauce Soy Less Sodium Gluten Free Lee Kum Kee Bulk', ' Sesame Seeds', ' Sugar Granulated', ' Vinegar Rice Wine Unseasoned', ' Water', ' Wine Cooking Mirin (Sweet Rice)', ' Wine Rice (Sake)']}, {'name': 'Ginger Miso Shiitake Ramen Bowl Tofu', 'ingredients': ['Bok Choy', ' Carrots Fresh Chopped', ' Celery Fresh Chopped', ' Garlic Cloves Peeled Fresh Minced', ' Ginger Root Fresh Chopped', ' Ginger Root Fresh Grated', ' Lemongrass Fresh', ' Mushrooms Shiitake Dried', ' Mushrooms', ' Mushrooms Wood Ear Dried', ' Noodles Ramen Tokyo Wavy Sun Noodle', ' Canola Oil', ' Oil Sesame', ' Onions Yellow Fresh Chopped', ' Paste Miso Red', ' Salt Kosher', ' Sauce Soy Less Sodium Gluten Free Lee Kum Kee Bulk', ' Dried Nori Seaweed ', ' Sesame Seeds', ' Spice Star Anise', ' Sugar Granulated', ' Syrup Maple 100% Pure', ' Tofu Firm Cubed 1"', ' Water']}], 'Deli': [{'name': 'Turkey Breast, House Roasted', 'ingredients': ['Turkey']}, {'name': 'Rare Deli Roast Beef', 'ingredients': ['Roast Beef']}, {'name': 'Chips, Potato, 1.5 oz', 'ingredients': ['Ingredient Statement: Potatoes', ' Vegetable Oil (Sunflower', ' Corn and/or Canola Oil)', ' and Salt. Allergens: No Know Allergens']}, {'name': 'Fresh Fruit Platter', 'ingredients': ['Blueberries', ' Honeydew Melon', ' Pineapple', ' Raspberries', ' Strawberries']}], 'Vegan': [{'name': 'Eggplant Katsu Sandwich', 'ingredients': ['White Bread', ' Panko Breadcrumbs', ' Cabbage ', ' Liquid Egg ', ' Japanese Eggplant', ' All Purpose Flour', ' Salt', ' Fresh Ground Black Pepper']}, {'name': 'Steamed Rice', 'ingredients': ['Long Grain White Rice', ' Water']}]}}
"""

# Save dict object as a json file
parser.save(menu_data_dict, file_name="april_menu_data.json")
```