"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudformation = require("@aws-cdk/aws-cloudformation");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * Base class for Actions that execute CloudFormation
 */
class CloudFormationAction extends action_1.Action {
    constructor(props, inputs) {
        super({
            ...props,
            provider: 'CloudFormation',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 0,
                maxInputs: 10,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs,
            outputs: props.outputFileName
                ? [props.output || new codepipeline.Artifact(`${props.actionName}_${props.stackName}_Artifact`)]
                : undefined,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        const singletonPolicy = SingletonPolicy.forRole(options.role);
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantReadWrite(singletonPolicy);
        }
        else if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackName: this.props.stackName,
                OutputFileName: this.props.outputFileName,
            },
        };
    }
}
/**
 * CodePipeline action to execute a prepared change set.
 */
class CloudFormationExecuteChangeSetAction extends CloudFormationAction {
    constructor(props) {
        super(props, undefined);
        this.props2 = props;
    }
    bound(scope, stage, options) {
        SingletonPolicy.forRole(options.role).grantExecuteChangeSet(this.props2);
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_EXECUTE',
                ChangeSetName: this.props2.changeSetName,
            },
        };
    }
}
exports.CloudFormationExecuteChangeSetAction = CloudFormationExecuteChangeSetAction;
// tslint:enable:max-line-length
/**
 * Base class for all CloudFormation actions that execute or stage deployments.
 */
class CloudFormationDeployAction extends CloudFormationAction {
    constructor(props, inputs) {
        super(props, (props.extraInputs || []).concat(inputs || []));
        this.props2 = props;
    }
    /**
     * Add statement to the service role assumed by CloudFormation while executing this action.
     */
    addToDeploymentRolePolicy(statement) {
        return this.getDeploymentRole('method addToRolePolicy()').addToPolicy(statement);
    }
    get deploymentRole() {
        return this.getDeploymentRole('property role()');
    }
    bound(scope, stage, options) {
        if (this.props2.deploymentRole) {
            this._deploymentRole = this.props2.deploymentRole;
        }
        else {
            const roleStack = cdk.Stack.of(options.role);
            const pipelineStack = cdk.Stack.of(scope);
            if (roleStack.account !== pipelineStack.account) {
                // pass role is not allowed for cross-account access - so,
                // create the deployment Role in the other account!
                this._deploymentRole = new iam.Role(roleStack, `${stage.pipeline.node.uniqueId}-${stage.stageName}-${this.actionProperties.actionName}-DeploymentRole`, {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                    roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
            }
            else {
                this._deploymentRole = new iam.Role(scope, 'Role', {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com')
                });
            }
            // the deployment role might need read access to the pipeline's bucket
            // (for example, if it's deploying a Lambda function),
            // and even if it has admin permissions, it won't be enough,
            // as it needs to be added to the key's resource policy
            // (and the bucket's, if the access is cross-account)
            options.bucket.grantRead(this._deploymentRole);
            if (this.props2.adminPermissions) {
                this._deploymentRole.addToPolicy(new iam.PolicyStatement({
                    actions: ['*'],
                    resources: ['*'],
                }));
            }
        }
        SingletonPolicy.forRole(options.role).grantPassRole(this._deploymentRole);
        const capabilities = this.props2.adminPermissions && this.props2.capabilities === undefined
            ? [cloudformation.CloudFormationCapabilities.NAMED_IAM]
            : this.props2.capabilities;
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                // None evaluates to empty string which is falsey and results in undefined
                Capabilities: parseCapabilities(capabilities),
                RoleArn: this.deploymentRole.roleArn,
                ParameterOverrides: cdk.Stack.of(scope).toJsonString(this.props2.parameterOverrides),
                TemplateConfiguration: this.props2.templateConfiguration
                    ? this.props2.templateConfiguration.location
                    : undefined,
                StackName: this.props2.stackName,
            },
        };
    }
    getDeploymentRole(member) {
        if (this._deploymentRole) {
            return this._deploymentRole;
        }
        else {
            throw new Error(`Cannot use the ${member} before the Action has been added to a Pipeline`);
        }
    }
}
/**
 * CodePipeline action to prepare a change set.
 *
 * Creates the change set if it doesn't exist based on the stack name and template that you submit.
 * If the change set exists, AWS CloudFormation deletes it, and then creates a new one.
 */
class CloudFormationCreateReplaceChangeSetAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateReplaceChangeSet(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_REPLACE',
                ChangeSetName: this.props3.changeSetName,
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateReplaceChangeSetAction = CloudFormationCreateReplaceChangeSetAction;
/**
 * CodePipeline action to deploy a stack.
 *
 * Creates the stack if the specified stack doesn't exist. If the stack exists,
 * AWS CloudFormation updates the stack. Use this action to update existing
 * stacks.
 *
 * AWS CodePipeline won't replace the stack, and will fail deployment if the
 * stack is in a failed state. Use `ReplaceOnFailure` for an action that
 * will delete and recreate the stack to try and recover from failed states.
 *
 * Use this action to automatically replace failed stacks without recovering or
 * troubleshooting them. You would typically choose this mode for testing.
 */
class CloudFormationCreateUpdateStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateUpdateStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: this.props3.replaceOnFailure ? 'REPLACE_ON_FAILURE' : 'CREATE_UPDATE',
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateUpdateStackAction = CloudFormationCreateUpdateStackAction;
/**
 * CodePipeline action to delete a stack.
 *
 * Deletes a stack. If you specify a stack that doesn't exist, the action completes successfully
 * without deleting a stack.
 */
class CloudFormationDeleteStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, undefined);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantDeleteStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'DELETE_ONLY',
            },
        };
    }
}
exports.CloudFormationDeleteStackAction = CloudFormationDeleteStackAction;
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends cdk.Construct {
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ]
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`
        });
    }
}
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGlwZWxpbmUtYWN0aW9ucy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInBpcGVsaW5lLWFjdGlvbnMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSw4REFBOEQ7QUFDOUQsMERBQTBEO0FBQzFELHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFDckMsc0NBQW1DO0FBc0RuQzs7R0FFRztBQUNILE1BQWUsb0JBQXFCLFNBQVEsZUFBTTtJQUdoRCxZQUFZLEtBQWdDLEVBQUUsTUFBMkM7UUFDdkYsS0FBSyxDQUFDO1lBQ0osR0FBRyxLQUFLO1lBQ1IsUUFBUSxFQUFFLGdCQUFnQjtZQUMxQixRQUFRLEVBQUUsWUFBWSxDQUFDLGNBQWMsQ0FBQyxNQUFNO1lBQzVDLGNBQWMsRUFBRTtnQkFDZCxTQUFTLEVBQUUsQ0FBQztnQkFDWixTQUFTLEVBQUUsRUFBRTtnQkFDYixVQUFVLEVBQUUsQ0FBQztnQkFDYixVQUFVLEVBQUUsQ0FBQzthQUNkO1lBQ0QsTUFBTTtZQUNOLE9BQU8sRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDM0IsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sSUFBSSxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQyxTQUFTLFdBQVcsQ0FBQyxDQUFDO2dCQUNoRyxDQUFDLENBQUMsU0FBUztTQUNkLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO0lBQ3JCLENBQUM7SUFFUyxLQUFLLENBQUMsTUFBcUIsRUFBRSxNQUEyQixFQUFFLE9BQXVDO1FBRXpHLE1BQU0sZUFBZSxHQUFHLGVBQWUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTlELElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDcEQsT0FBTyxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDaEQ7YUFBTSxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sSUFBSSxFQUFFLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzFELE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQzNDO1FBRUQsT0FBTztZQUNMLGFBQWEsRUFBRTtnQkFDYixTQUFTLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTO2dCQUMvQixjQUFjLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjO2FBQzFDO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQVlEOztHQUVHO0FBQ0gsTUFBYSxvQ0FBcUMsU0FBUSxvQkFBb0I7SUFHNUUsWUFBWSxLQUFnRDtRQUMxRCxLQUFLLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRXhCLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDO0lBQ3RCLENBQUM7SUFFUyxLQUFLLENBQUMsS0FBb0IsRUFBRSxLQUEwQixFQUFFLE9BQXVDO1FBRXZHLGVBQWUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUV6RSxNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDeEQsT0FBTztZQUNMLEdBQUcsWUFBWTtZQUNmLGFBQWEsRUFBRTtnQkFDYixHQUFHLFlBQVksQ0FBQyxhQUFhO2dCQUM3QixVQUFVLEVBQUUsb0JBQW9CO2dCQUNoQyxhQUFhLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxhQUFhO2FBQ3pDO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQXZCRCxvRkF1QkM7QUFrR0QsZ0NBQWdDO0FBRWhDOztHQUVHO0FBQ0gsTUFBZSwwQkFBMkIsU0FBUSxvQkFBb0I7SUFJcEUsWUFBWSxLQUFzQyxFQUFFLE1BQTJDO1FBQzdGLEtBQUssQ0FBQyxLQUFLLEVBQUUsQ0FBQyxLQUFLLENBQUMsV0FBVyxJQUFJLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztRQUU3RCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztJQUN0QixDQUFDO0lBRUQ7O09BRUc7SUFDSSx5QkFBeUIsQ0FBQyxTQUE4QjtRQUM3RCxPQUFPLElBQUksQ0FBQyxpQkFBaUIsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUNuRixDQUFDO0lBRUQsSUFBVyxjQUFjO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVTLEtBQUssQ0FBQyxLQUFvQixFQUFFLEtBQTBCLEVBQUUsT0FBdUM7UUFFdkcsSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsRUFBRTtZQUM5QixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDO1NBQ25EO2FBQU07WUFDTCxNQUFNLFNBQVMsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDN0MsTUFBTSxhQUFhLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDMUMsSUFBSSxTQUFTLENBQUMsT0FBTyxLQUFLLGFBQWEsQ0FBQyxPQUFPLEVBQUU7Z0JBQy9DLDBEQUEwRDtnQkFDMUQsbURBQW1EO2dCQUNuRCxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQzNDLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsUUFBUSxJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsaUJBQWlCLEVBQUU7b0JBQ3ZHLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyw4QkFBOEIsQ0FBQztvQkFDbkUsUUFBUSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsa0JBQWtCO2lCQUM5QyxDQUFDLENBQUM7YUFDTjtpQkFBTTtnQkFDTCxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO29CQUNqRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsOEJBQThCLENBQUM7aUJBQ3BFLENBQUMsQ0FBQzthQUNKO1lBRUQsc0VBQXNFO1lBQ3RFLHNEQUFzRDtZQUN0RCw0REFBNEQ7WUFDNUQsdURBQXVEO1lBQ3ZELHFEQUFxRDtZQUNyRCxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7WUFFL0MsSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFO2dCQUNoQyxJQUFJLENBQUMsZUFBZSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3ZELE9BQU8sRUFBRSxDQUFDLEdBQUcsQ0FBQztvQkFDZCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7aUJBQ2pCLENBQUMsQ0FBQyxDQUFDO2FBQ0w7U0FDRjtRQUVELGVBQWUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7UUFFMUUsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksS0FBSyxTQUFTO1lBQ3pGLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQywwQkFBMEIsQ0FBQyxTQUFTLENBQUM7WUFDdkQsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDO1FBRTdCLE1BQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN4RCxPQUFPO1lBQ0wsR0FBRyxZQUFZO1lBQ2YsYUFBYSxFQUFFO2dCQUNiLEdBQUcsWUFBWSxDQUFDLGFBQWE7Z0JBQzdCLDBFQUEwRTtnQkFDMUUsWUFBWSxFQUFFLGlCQUFpQixDQUFDLFlBQVksQ0FBQztnQkFDN0MsT0FBTyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsT0FBTztnQkFDcEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsa0JBQWtCLENBQUM7Z0JBQ3BGLHFCQUFxQixFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMscUJBQXFCO29CQUN0RCxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRO29CQUM1QyxDQUFDLENBQUMsU0FBUztnQkFDYixTQUFTLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTO2FBQ2pDO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxNQUFjO1FBQ3RDLElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN4QixPQUFPLElBQUksQ0FBQyxlQUFlLENBQUM7U0FDN0I7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsa0JBQWtCLE1BQU0saURBQWlELENBQUMsQ0FBQztTQUM1RjtJQUNILENBQUM7Q0FDRjtBQWlCRDs7Ozs7R0FLRztBQUNILE1BQWEsMENBQTJDLFNBQVEsMEJBQTBCO0lBR3hGLFlBQVksS0FBc0Q7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMscUJBQXFCO1lBQ3RDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLENBQUM7WUFDckUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBRW5DLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDO0lBQ3RCLENBQUM7SUFFUyxLQUFLLENBQUMsS0FBb0IsRUFBRSxLQUEwQixFQUFFLE9BQXVDO1FBRXZHLG1GQUFtRjtRQUNuRixNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFeEQsZUFBZSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsMkJBQTJCLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRS9FLE9BQU87WUFDTCxHQUFHLFlBQVk7WUFDZixhQUFhLEVBQUU7Z0JBQ2IsR0FBRyxZQUFZLENBQUMsYUFBYTtnQkFDN0IsVUFBVSxFQUFFLG9CQUFvQjtnQkFDaEMsYUFBYSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsYUFBYTtnQkFDeEMsWUFBWSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLFFBQVE7YUFDaEQ7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBNUJELGdHQTRCQztBQTJCRDs7Ozs7Ozs7Ozs7OztHQWFHO0FBQ0gsTUFBYSxxQ0FBc0MsU0FBUSwwQkFBMEI7SUFHbkYsWUFBWSxLQUFpRDtRQUMzRCxLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxxQkFBcUI7WUFDdEMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLHFCQUFxQixDQUFDLFFBQVEsQ0FBQztZQUNyRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFFbkMsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUM7SUFDdEIsQ0FBQztJQUVTLEtBQUssQ0FBQyxLQUFvQixFQUFFLEtBQTBCLEVBQUUsT0FBdUM7UUFFdkcsbUZBQW1GO1FBQ25GLE1BQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUV4RCxlQUFlLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFMUUsT0FBTztZQUNMLEdBQUcsWUFBWTtZQUNmLGFBQWEsRUFBRTtnQkFDYixHQUFHLFlBQVksQ0FBQyxhQUFhO2dCQUM3QixVQUFVLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLGVBQWU7Z0JBQ2pGLFlBQVksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUFRO2FBQ2hEO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQTNCRCxzRkEyQkM7QUFTRDs7Ozs7R0FLRztBQUNILE1BQWEsK0JBQWdDLFNBQVEsMEJBQTBCO0lBRzdFLFlBQVksS0FBMkM7UUFDckQsS0FBSyxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztRQUV4QixJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztJQUN0QixDQUFDO0lBRVMsS0FBSyxDQUFDLEtBQW9CLEVBQUUsS0FBMEIsRUFBRSxPQUF1QztRQUV2RyxtRkFBbUY7UUFDbkYsTUFBTSxZQUFZLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXhELGVBQWUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVwRSxPQUFPO1lBQ0wsR0FBRyxZQUFZO1lBQ2YsYUFBYSxFQUFFO2dCQUNiLEdBQUcsWUFBWSxDQUFDLGFBQWE7Z0JBQzdCLFVBQVUsRUFBRSxhQUFhO2FBQzFCO1NBQ0YsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQXhCRCwwRUF3QkM7QUFFRDs7Ozs7Ozs7R0FRRztBQUNILE1BQU0sZUFBZ0IsU0FBUSxHQUFHLENBQUMsU0FBUztJQWlCekMsWUFBcUMsSUFBZTtRQUNsRCxLQUFLLENBQUMsSUFBZ0MsRUFBRSxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7UUFEM0IsU0FBSSxHQUFKLElBQUksQ0FBVztRQUY1QyxlQUFVLEdBQTJDLEVBQUUsQ0FBQztRQUk5RCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQztJQUM3QixDQUFDO0lBbkJEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQWU7UUFDbkMsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzNELE9BQVEsS0FBeUIsSUFBSSxJQUFJLGVBQWUsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBYU0scUJBQXFCLENBQUMsS0FBb0U7UUFDL0YsSUFBSSxDQUFDLFlBQVksQ0FBQztZQUNoQixPQUFPLEVBQUU7Z0JBQ1AsK0JBQStCO2dCQUMvQixrQ0FBa0M7Z0JBQ2xDLGlDQUFpQzthQUNsQztZQUNELFVBQVUsRUFBRSxFQUFFLG9CQUFvQixFQUFFLEVBQUUsOEJBQThCLEVBQUUsS0FBSyxDQUFDLGFBQWEsRUFBRSxFQUFFO1NBQzlGLENBQUMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVNLDJCQUEyQixDQUFDLEtBQW9FO1FBQ3JHLElBQUksQ0FBQyxZQUFZLENBQUM7WUFDaEIsT0FBTyxFQUFFO2dCQUNQLGdDQUFnQztnQkFDaEMsZ0NBQWdDO2dCQUNoQyxrQ0FBa0M7Z0JBQ2xDLCtCQUErQjthQUNoQztZQUNELFVBQVUsRUFBRSxFQUFFLG9CQUFvQixFQUFFLEVBQUUsOEJBQThCLEVBQUUsS0FBSyxDQUFDLGFBQWEsRUFBRSxFQUFFO1NBQzlGLENBQUMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVNLHNCQUFzQixDQUFDLEtBQXlFO1FBQ3JHLE1BQU0sT0FBTyxHQUFHO1lBQ2QsK0JBQStCO1lBQy9CLDRCQUE0QjtZQUM1Qiw0QkFBNEI7WUFDNUIsNkJBQTZCO1lBQzdCLGlDQUFpQztZQUNqQywrQkFBK0I7WUFDL0IsK0JBQStCO1NBQ2hDLENBQUM7UUFDRixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixPQUFPLENBQUMsSUFBSSxDQUFDLDRCQUE0QixDQUFDLENBQUM7U0FDNUM7UUFDRCxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDN0UsQ0FBQztJQUVNLGdCQUFnQixDQUFDLEtBQTZDO1FBQ25FLElBQUksQ0FBQyxZQUFZLENBQUM7WUFDaEIsT0FBTyxFQUFFO2dCQUNQLCtCQUErQjtnQkFDL0IsNEJBQTRCO2FBQzdCO1NBQ0YsQ0FBQyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRU0sYUFBYSxDQUFDLElBQWU7UUFDbEMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQzlFLENBQUM7SUFFTyxZQUFZLENBQUMsUUFBMkI7UUFDOUMsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzdCLElBQUksQ0FBQyxDQUFDLEdBQUcsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDN0IsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUMsRUFBRSxPQUFPLEVBQUUsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFDOUUsSUFBSSxRQUFRLENBQUMsVUFBVSxFQUFFO2dCQUN2QixJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7YUFDekQ7WUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDN0M7UUFDRCxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFNUIsU0FBUyxNQUFNLENBQUMsS0FBd0I7WUFDdEMsTUFBTSxPQUFPLEdBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDO1lBQ3ZELE1BQU0sVUFBVSxHQUFHLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUN0RCxPQUFPLEdBQUcsT0FBTyxPQUFPLFVBQVUsRUFBRSxDQUFDO1lBRXJDLFNBQVMsZ0JBQWdCLENBQUMsSUFBeUI7Z0JBQ2pELElBQUksSUFBSSxJQUFJLElBQUksRUFBRTtvQkFBRSxPQUFPLEVBQUUsQ0FBQztpQkFBRTtnQkFDaEMsSUFBSSxNQUFNLEdBQUcsRUFBRSxDQUFDO2dCQUNoQixLQUFLLE1BQU0sRUFBRSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUU7b0JBQ3pDLE1BQU0sSUFBSSxHQUFHLEVBQUUsTUFBTSxDQUFDO29CQUN0QixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7b0JBQzNCLEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTt3QkFDckQsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDO3dCQUNuQyxNQUFNLElBQUksR0FBRyxLQUFLLE1BQU0sQ0FBQztxQkFDMUI7aUJBQ0Y7Z0JBQ0QsT0FBTyxNQUFNLENBQUM7WUFDaEIsQ0FBQztRQUNILENBQUM7SUFDSCxDQUFDO0lBRU8saUJBQWlCLENBQUMsS0FBNkM7UUFDckUsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDbEMsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQ3BCLE9BQU8sRUFBRSxnQkFBZ0I7WUFDekIsUUFBUSxFQUFFLE9BQU87WUFDakIsWUFBWSxFQUFFLEdBQUcsS0FBSyxDQUFDLFNBQVMsSUFBSTtTQUNyQyxDQUFDLENBQUM7SUFDTCxDQUFDOztBQXRHdUIsb0JBQUksR0FBRyxzQ0FBc0MsQ0FBQztBQWdIeEUsU0FBUyxpQkFBaUIsQ0FBQyxZQUFxRTtJQUM5RixJQUFJLFlBQVksS0FBSyxTQUFTLEVBQUU7UUFDOUIsT0FBTyxTQUFTLENBQUM7S0FDbEI7U0FBTSxJQUFJLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1FBQ3BDLE1BQU0sVUFBVSxHQUFHLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUMzQyxPQUFPLENBQUMsVUFBVSxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztLQUNyRDtTQUFNLElBQUksWUFBWSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7UUFDbEMsT0FBTyxZQUFZLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0tBQy9CO0lBRUQsT0FBTyxTQUFTLENBQUM7QUFDbkIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNsb3VkZm9ybWF0aW9uIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZvcm1hdGlvbic7XG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZSc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBBY3Rpb24gfSBmcm9tICcuLi9hY3Rpb24nO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgY29tbW9uIHRvIGFsbCBDbG91ZEZvcm1hdGlvbiBhY3Rpb25zXG4gKi9cbmludGVyZmFjZSBDbG91ZEZvcm1hdGlvbkFjdGlvblByb3BzIGV4dGVuZHMgY29kZXBpcGVsaW5lLkNvbW1vbkF3c0FjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzdGFjayB0byBhcHBseSB0aGlzIGFjdGlvbiB0b1xuICAgKi9cbiAgcmVhZG9ubHkgc3RhY2tOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEEgbmFtZSBmb3IgdGhlIGZpbGVuYW1lIGluIHRoZSBvdXRwdXQgYXJ0aWZhY3QgdG8gc3RvcmUgdGhlIEFXUyBDbG91ZEZvcm1hdGlvbiBjYWxsJ3MgcmVzdWx0LlxuICAgKlxuICAgKiBUaGUgZmlsZSB3aWxsIGNvbnRhaW4gdGhlIHJlc3VsdCBvZiB0aGUgY2FsbCB0byBBV1MgQ2xvdWRGb3JtYXRpb24gKGZvciBleGFtcGxlXG4gICAqIHRoZSBjYWxsIHRvIFVwZGF0ZVN0YWNrIG9yIENyZWF0ZUNoYW5nZVNldCkuXG4gICAqXG4gICAqIEFXUyBDb2RlUGlwZWxpbmUgYWRkcyB0aGUgZmlsZSB0byB0aGUgb3V0cHV0IGFydGlmYWN0IGFmdGVyIHBlcmZvcm1pbmdcbiAgICogdGhlIHNwZWNpZmllZCBhY3Rpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIG91dHB1dCBhcnRpZmFjdCBnZW5lcmF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IG91dHB1dEZpbGVOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgb3V0cHV0IGFydGlmYWN0IHRvIGdlbmVyYXRlXG4gICAqXG4gICAqIE9ubHkgYXBwbGllZCBpZiBgb3V0cHV0RmlsZU5hbWVgIGlzIHNldCBhcyB3ZWxsLlxuICAgKlxuICAgKiBAZGVmYXVsdCBBdXRvbWF0aWNhbGx5IGdlbmVyYXRlZCBhcnRpZmFjdCBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgb3V0cHV0PzogY29kZXBpcGVsaW5lLkFydGlmYWN0O1xuXG4gIC8qKlxuICAgKiBUaGUgQVdTIHJlZ2lvbiB0aGUgZ2l2ZW4gQWN0aW9uIHJlc2lkZXMgaW4uXG4gICAqIE5vdGUgdGhhdCBhIGNyb3NzLXJlZ2lvbiBQaXBlbGluZSByZXF1aXJlcyByZXBsaWNhdGlvbiBidWNrZXRzIHRvIGZ1bmN0aW9uIGNvcnJlY3RseS5cbiAgICogWW91IGNhbiBwcm92aWRlIHRoZWlyIG5hbWVzIHdpdGggdGhlIHtAbGluayBQaXBlbGluZVByb3BzI2Nyb3NzUmVnaW9uUmVwbGljYXRpb25CdWNrZXRzfSBwcm9wZXJ0eS5cbiAgICogSWYgeW91IGRvbid0LCB0aGUgQ29kZVBpcGVsaW5lIENvbnN0cnVjdCB3aWxsIGNyZWF0ZSBuZXcgU3RhY2tzIGluIHlvdXIgQ0RLIGFwcCBjb250YWluaW5nIHRob3NlIGJ1Y2tldHMsXG4gICAqIHRoYXQgeW91IHdpbGwgbmVlZCB0byBgY2RrIGRlcGxveWAgYmVmb3JlIGRlcGxveWluZyB0aGUgbWFpbiwgUGlwZWxpbmUtY29udGFpbmluZyBTdGFjay5cbiAgICpcbiAgICogQGRlZmF1bHQgdGhlIEFjdGlvbiByZXNpZGVzIGluIHRoZSBzYW1lIHJlZ2lvbiBhcyB0aGUgUGlwZWxpbmVcbiAgICovXG4gIHJlYWRvbmx5IHJlZ2lvbj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFXUyBhY2NvdW50IHRoaXMgQWN0aW9uIGlzIHN1cHBvc2VkIHRvIG9wZXJhdGUgaW4uXG4gICAqICoqTm90ZSoqOiBpZiB5b3Ugc3BlY2lmeSB0aGUgYHJvbGVgIHByb3BlcnR5LFxuICAgKiB0aGlzIGlzIGlnbm9yZWQgLSB0aGUgYWN0aW9uIHdpbGwgb3BlcmF0ZSBpbiB0aGUgc2FtZSByZWdpb24gdGhlIHBhc3NlZCByb2xlIGRvZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYWN0aW9uIHJlc2lkZXMgaW4gdGhlIHNhbWUgYWNjb3VudCBhcyB0aGUgcGlwZWxpbmVcbiAgICovXG4gIHJlYWRvbmx5IGFjY291bnQ/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQmFzZSBjbGFzcyBmb3IgQWN0aW9ucyB0aGF0IGV4ZWN1dGUgQ2xvdWRGb3JtYXRpb25cbiAqL1xuYWJzdHJhY3QgY2xhc3MgQ2xvdWRGb3JtYXRpb25BY3Rpb24gZXh0ZW5kcyBBY3Rpb24ge1xuICBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBDbG91ZEZvcm1hdGlvbkFjdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBDbG91ZEZvcm1hdGlvbkFjdGlvblByb3BzLCBpbnB1dHM6IGNvZGVwaXBlbGluZS5BcnRpZmFjdFtdIHwgdW5kZWZpbmVkKSB7XG4gICAgc3VwZXIoe1xuICAgICAgLi4ucHJvcHMsXG4gICAgICBwcm92aWRlcjogJ0Nsb3VkRm9ybWF0aW9uJyxcbiAgICAgIGNhdGVnb3J5OiBjb2RlcGlwZWxpbmUuQWN0aW9uQ2F0ZWdvcnkuREVQTE9ZLFxuICAgICAgYXJ0aWZhY3RCb3VuZHM6IHtcbiAgICAgICAgbWluSW5wdXRzOiAwLFxuICAgICAgICBtYXhJbnB1dHM6IDEwLFxuICAgICAgICBtaW5PdXRwdXRzOiAwLFxuICAgICAgICBtYXhPdXRwdXRzOiAxLFxuICAgICAgfSxcbiAgICAgIGlucHV0cyxcbiAgICAgIG91dHB1dHM6IHByb3BzLm91dHB1dEZpbGVOYW1lXG4gICAgICAgID8gW3Byb3BzLm91dHB1dCB8fCBuZXcgY29kZXBpcGVsaW5lLkFydGlmYWN0KGAke3Byb3BzLmFjdGlvbk5hbWV9XyR7cHJvcHMuc3RhY2tOYW1lfV9BcnRpZmFjdGApXVxuICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICB9KTtcblxuICAgIHRoaXMucHJvcHMgPSBwcm9wcztcbiAgfVxuXG4gIHByb3RlY3RlZCBib3VuZChfc2NvcGU6IGNkay5Db25zdHJ1Y3QsIF9zdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgY29uc3Qgc2luZ2xldG9uUG9saWN5ID0gU2luZ2xldG9uUG9saWN5LmZvclJvbGUob3B0aW9ucy5yb2xlKTtcblxuICAgIGlmICgodGhpcy5hY3Rpb25Qcm9wZXJ0aWVzLm91dHB1dHMgfHwgW10pLmxlbmd0aCA+IDApIHtcbiAgICAgIG9wdGlvbnMuYnVja2V0LmdyYW50UmVhZFdyaXRlKHNpbmdsZXRvblBvbGljeSk7XG4gICAgfSBlbHNlIGlmICgodGhpcy5hY3Rpb25Qcm9wZXJ0aWVzLmlucHV0cyB8fCBbXSkubGVuZ3RoID4gMCkge1xuICAgICAgb3B0aW9ucy5idWNrZXQuZ3JhbnRSZWFkKHNpbmdsZXRvblBvbGljeSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHtcbiAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgU3RhY2tOYW1lOiB0aGlzLnByb3BzLnN0YWNrTmFtZSxcbiAgICAgICAgT3V0cHV0RmlsZU5hbWU6IHRoaXMucHJvcHMub3V0cHV0RmlsZU5hbWUsXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciB0aGUgQ2xvdWRGb3JtYXRpb25FeGVjdXRlQ2hhbmdlU2V0QWN0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENsb3VkRm9ybWF0aW9uRXhlY3V0ZUNoYW5nZVNldEFjdGlvblByb3BzIGV4dGVuZHMgQ2xvdWRGb3JtYXRpb25BY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBjaGFuZ2Ugc2V0IHRvIGV4ZWN1dGUuXG4gICAqL1xuICByZWFkb25seSBjaGFuZ2VTZXROYW1lOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQ29kZVBpcGVsaW5lIGFjdGlvbiB0byBleGVjdXRlIGEgcHJlcGFyZWQgY2hhbmdlIHNldC5cbiAqL1xuZXhwb3J0IGNsYXNzIENsb3VkRm9ybWF0aW9uRXhlY3V0ZUNoYW5nZVNldEFjdGlvbiBleHRlbmRzIENsb3VkRm9ybWF0aW9uQWN0aW9uIHtcbiAgcHJpdmF0ZSByZWFkb25seSBwcm9wczI6IENsb3VkRm9ybWF0aW9uRXhlY3V0ZUNoYW5nZVNldEFjdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBDbG91ZEZvcm1hdGlvbkV4ZWN1dGVDaGFuZ2VTZXRBY3Rpb25Qcm9wcykge1xuICAgIHN1cGVyKHByb3BzLCB1bmRlZmluZWQpO1xuXG4gICAgdGhpcy5wcm9wczIgPSBwcm9wcztcbiAgfVxuXG4gIHByb3RlY3RlZCBib3VuZChzY29wZTogY2RrLkNvbnN0cnVjdCwgc3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2UsIG9wdGlvbnM6IGNvZGVwaXBlbGluZS5BY3Rpb25CaW5kT3B0aW9ucyk6XG4gIGNvZGVwaXBlbGluZS5BY3Rpb25Db25maWcge1xuICAgIFNpbmdsZXRvblBvbGljeS5mb3JSb2xlKG9wdGlvbnMucm9sZSkuZ3JhbnRFeGVjdXRlQ2hhbmdlU2V0KHRoaXMucHJvcHMyKTtcblxuICAgIGNvbnN0IGFjdGlvbkNvbmZpZyA9IHN1cGVyLmJvdW5kKHNjb3BlLCBzdGFnZSwgb3B0aW9ucyk7XG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLmFjdGlvbkNvbmZpZyxcbiAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgLi4uYWN0aW9uQ29uZmlnLmNvbmZpZ3VyYXRpb24sXG4gICAgICAgIEFjdGlvbk1vZGU6ICdDSEFOR0VfU0VUX0VYRUNVVEUnLFxuICAgICAgICBDaGFuZ2VTZXROYW1lOiB0aGlzLnByb3BzMi5jaGFuZ2VTZXROYW1lLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbi8vIHRzbGludDpkaXNhYmxlOm1heC1saW5lLWxlbmd0aCBCZWNhdXNlIG9mIGxvbmcgVVJMcyBpbiBkb2N1bWVudGF0aW9uXG4vKipcbiAqIFByb3BlcnRpZXMgY29tbW9uIHRvIENsb3VkRm9ybWF0aW9uIGFjdGlvbnMgdGhhdCBzdGFnZSBkZXBsb3ltZW50c1xuICovXG5pbnRlcmZhY2UgQ2xvdWRGb3JtYXRpb25EZXBsb3lBY3Rpb25Qcm9wcyBleHRlbmRzIENsb3VkRm9ybWF0aW9uQWN0aW9uUHJvcHMge1xuICAvKipcbiAgICogSUFNIHJvbGUgdG8gYXNzdW1lIHdoZW4gZGVwbG95aW5nIGNoYW5nZXMuXG4gICAqXG4gICAqIElmIG5vdCBzcGVjaWZpZWQsIGEgZnJlc2ggcm9sZSBpcyBjcmVhdGVkLiBUaGUgcm9sZSBpcyBjcmVhdGVkIHdpdGggemVyb1xuICAgKiBwZXJtaXNzaW9ucyB1bmxlc3MgYGFkbWluUGVybWlzc2lvbnNgIGlzIHRydWUsIGluIHdoaWNoIGNhc2UgdGhlIHJvbGUgd2lsbCBoYXZlXG4gICAqIGZ1bGwgcGVybWlzc2lvbnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IEEgZnJlc2ggcm9sZSB3aXRoIGZ1bGwgb3Igbm8gcGVybWlzc2lvbnMgKGRlcGVuZGluZyBvbiB0aGUgdmFsdWUgb2YgYGFkbWluUGVybWlzc2lvbnNgKS5cbiAgICovXG4gIHJlYWRvbmx5IGRlcGxveW1lbnRSb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBBY2tub3dsZWRnZSBjZXJ0YWluIGNoYW5nZXMgbWFkZSBhcyBwYXJ0IG9mIGRlcGxveW1lbnRcbiAgICpcbiAgICogRm9yIHN0YWNrcyB0aGF0IGNvbnRhaW4gY2VydGFpbiByZXNvdXJjZXMsIGV4cGxpY2l0IGFja25vd2xlZGdlbWVudCB0aGF0IEFXUyBDbG91ZEZvcm1hdGlvblxuICAgKiBtaWdodCBjcmVhdGUgb3IgdXBkYXRlIHRob3NlIHJlc291cmNlcy4gRm9yIGV4YW1wbGUsIHlvdSBtdXN0IHNwZWNpZnkgYEFub255bW91c0lBTWAgb3IgYE5hbWVkSUFNYFxuICAgKiBpZiB5b3VyIHN0YWNrIHRlbXBsYXRlIGNvbnRhaW5zIEFXUyBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgcmVzb3VyY2VzLiBGb3IgbW9yZVxuICAgKiBpbmZvcm1hdGlvbiBzZWUgdGhlIGxpbmsgYmVsb3cuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvdXNpbmctaWFtLXRlbXBsYXRlLmh0bWwjdXNpbmctaWFtLWNhcGFiaWxpdGllc1xuICAgKiBAZGVmYXVsdCBOb25lLCB1bmxlc3MgYGFkbWluUGVybWlzc2lvbnNgIGlzIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGNhcGFiaWxpdGllcz86IGNsb3VkZm9ybWF0aW9uLkNsb3VkRm9ybWF0aW9uQ2FwYWJpbGl0aWVzW107XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gZ3JhbnQgZnVsbCBwZXJtaXNzaW9ucyB0byBDbG91ZEZvcm1hdGlvbiB3aGlsZSBkZXBsb3lpbmcgdGhpcyB0ZW1wbGF0ZS5cbiAgICpcbiAgICogU2V0dGluZyB0aGlzIHRvIGB0cnVlYCBhZmZlY3RzIHRoZSBkZWZhdWx0cyBmb3IgYHJvbGVgIGFuZCBgY2FwYWJpbGl0aWVzYCwgaWYgeW91XG4gICAqIGRvbid0IHNwZWNpZnkgYW55IGFsdGVybmF0aXZlcy5cbiAgICpcbiAgICogVGhlIGRlZmF1bHQgcm9sZSB0aGF0IHdpbGwgYmUgY3JlYXRlZCBmb3IgeW91IHdpbGwgaGF2ZSBmdWxsIChpLmUuLCBgKmApXG4gICAqIHBlcm1pc3Npb25zIG9uIGFsbCByZXNvdXJjZXMsIGFuZCB0aGUgZGVwbG95bWVudCB3aWxsIGhhdmUgbmFtZWQgSUFNXG4gICAqIGNhcGFiaWxpdGllcyAoaS5lLiwgYWJsZSB0byBjcmVhdGUgYWxsIElBTSByZXNvdXJjZXMpLlxuICAgKlxuICAgKiBUaGlzIGlzIGEgc2hvcnRoYW5kIHRoYXQgeW91IGNhbiB1c2UgaWYgeW91IGZ1bGx5IHRydXN0IHRoZSB0ZW1wbGF0ZXMgdGhhdFxuICAgKiBhcmUgZGVwbG95ZWQgaW4gdGhpcyBwaXBlbGluZS4gSWYgeW91IHdhbnQgbW9yZSBmaW5lLWdyYWluZWQgcGVybWlzc2lvbnMsXG4gICAqIHVzZSBgYWRkVG9Sb2xlUG9saWN5YCBhbmQgYGNhcGFiaWxpdGllc2AgdG8gY29udHJvbCB3aGF0IHRoZSBDbG91ZEZvcm1hdGlvblxuICAgKiBkZXBsb3ltZW50IGlzIGFsbG93ZWQgdG8gZG8uXG4gICAqL1xuICByZWFkb25seSBhZG1pblBlcm1pc3Npb25zOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBJbnB1dCBhcnRpZmFjdCB0byB1c2UgZm9yIHRlbXBsYXRlIHBhcmFtZXRlcnMgdmFsdWVzIGFuZCBzdGFjayBwb2xpY3kuXG4gICAqXG4gICAqIFRoZSB0ZW1wbGF0ZSBjb25maWd1cmF0aW9uIGZpbGUgc2hvdWxkIGNvbnRhaW4gYSBKU09OIG9iamVjdCB0aGF0IHNob3VsZCBsb29rIGxpa2UgdGhpczpcbiAgICogYHsgXCJQYXJhbWV0ZXJzXCI6IHsuLi59LCBcIlRhZ3NcIjogey4uLn0sIFwiU3RhY2tQb2xpY3lcIjogey4uLiB9fWAuIEZvciBtb3JlIGluZm9ybWF0aW9uLFxuICAgKiBzZWUgW0FXUyBDbG91ZEZvcm1hdGlvbiBBcnRpZmFjdHNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2NvbnRpbnVvdXMtZGVsaXZlcnktY29kZXBpcGVsaW5lLWNmbi1hcnRpZmFjdHMuaHRtbCkuXG4gICAqXG4gICAqIE5vdGUgdGhhdCBpZiB5b3UgaW5jbHVkZSBzZW5zaXRpdmUgaW5mb3JtYXRpb24sIHN1Y2ggYXMgcGFzc3dvcmRzLCByZXN0cmljdCBhY2Nlc3MgdG8gdGhpc1xuICAgKiBmaWxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyB0ZW1wbGF0ZSBjb25maWd1cmF0aW9uIGJhc2VkIG9uIGlucHV0IGFydGlmYWN0c1xuICAgKi9cbiAgcmVhZG9ubHkgdGVtcGxhdGVDb25maWd1cmF0aW9uPzogY29kZXBpcGVsaW5lLkFydGlmYWN0UGF0aDtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCB0ZW1wbGF0ZSBwYXJhbWV0ZXJzLlxuICAgKlxuICAgKiBUZW1wbGF0ZSBwYXJhbWV0ZXJzIHNwZWNpZmllZCBoZXJlIHRha2UgcHJlY2VkZW5jZSBvdmVyIHRlbXBsYXRlIHBhcmFtZXRlcnNcbiAgICogZm91bmQgaW4gdGhlIGFydGlmYWN0IHNwZWNpZmllZCBieSB0aGUgYHRlbXBsYXRlQ29uZmlndXJhdGlvbmAgcHJvcGVydHkuXG4gICAqXG4gICAqIFdlIHJlY29tbWVuZCB0aGF0IHlvdSB1c2UgdGhlIHRlbXBsYXRlIGNvbmZpZ3VyYXRpb24gZmlsZSB0byBzcGVjaWZ5XG4gICAqIG1vc3Qgb2YgeW91ciBwYXJhbWV0ZXIgdmFsdWVzLiBVc2UgcGFyYW1ldGVyIG92ZXJyaWRlcyB0byBzcGVjaWZ5IG9ubHlcbiAgICogZHluYW1pYyBwYXJhbWV0ZXIgdmFsdWVzICh2YWx1ZXMgdGhhdCBhcmUgdW5rbm93biB1bnRpbCB5b3UgcnVuIHRoZVxuICAgKiBwaXBlbGluZSkuXG4gICAqXG4gICAqIEFsbCBwYXJhbWV0ZXIgbmFtZXMgbXVzdCBiZSBwcmVzZW50IGluIHRoZSBzdGFjayB0ZW1wbGF0ZS5cbiAgICpcbiAgICogTm90ZTogdGhlIGVudGlyZSBvYmplY3QgY2Fubm90IGJlIG1vcmUgdGhhbiAxa0IuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIG92ZXJyaWRlc1xuICAgKi9cbiAgcmVhZG9ubHkgcGFyYW1ldGVyT3ZlcnJpZGVzPzogeyBbbmFtZTogc3RyaW5nXTogYW55IH07XG5cbiAgLyoqXG4gICAqIFRoZSBsaXN0IG9mIGFkZGl0aW9uYWwgaW5wdXQgQXJ0aWZhY3RzIGZvciB0aGlzIEFjdGlvbi5cbiAgICogVGhpcyBpcyBlc3BlY2lhbGx5IHVzZWZ1bCB3aGVuIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCB0aGUgYHBhcmFtZXRlck92ZXJyaWRlc2AgcHJvcGVydHkuXG4gICAqIEZvciBleGFtcGxlLCBpZiB5b3UgaGF2ZTpcbiAgICpcbiAgICogICBwYXJhbWV0ZXJPdmVycmlkZXM6IHtcbiAgICogICAgICdQYXJhbTEnOiBhY3Rpb24xLm91dHB1dEFydGlmYWN0LmJ1Y2tldE5hbWUsXG4gICAqICAgICAnUGFyYW0yJzogYWN0aW9uMi5vdXRwdXRBcnRpZmFjdC5vYmplY3RLZXksXG4gICAqICAgfVxuICAgKlxuICAgKiAsIGlmIHRoZSBvdXRwdXQgQXJ0aWZhY3RzIG9mIGBhY3Rpb24xYCBhbmQgYGFjdGlvbjJgIHdlcmUgbm90IHVzZWQgdG9cbiAgICogc2V0IGVpdGhlciB0aGUgYHRlbXBsYXRlQ29uZmlndXJhdGlvbmAgb3IgdGhlIGB0ZW1wbGF0ZVBhdGhgIHByb3BlcnRpZXMsXG4gICAqIHlvdSBuZWVkIHRvIG1ha2Ugc3VyZSB0byBpbmNsdWRlIHRoZW0gaW4gdGhlIGBleHRyYUlucHV0c2AgLVxuICAgKiBvdGhlcndpc2UsIHlvdSdsbCBnZXQgYW4gXCJ1bnJlY29nbml6ZWQgQXJ0aWZhY3RcIiBlcnJvciBkdXJpbmcgeW91ciBQaXBlbGluZSdzIGV4ZWN1dGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGV4dHJhSW5wdXRzPzogY29kZXBpcGVsaW5lLkFydGlmYWN0W107XG59XG4vLyB0c2xpbnQ6ZW5hYmxlOm1heC1saW5lLWxlbmd0aFxuXG4vKipcbiAqIEJhc2UgY2xhc3MgZm9yIGFsbCBDbG91ZEZvcm1hdGlvbiBhY3Rpb25zIHRoYXQgZXhlY3V0ZSBvciBzdGFnZSBkZXBsb3ltZW50cy5cbiAqL1xuYWJzdHJhY3QgY2xhc3MgQ2xvdWRGb3JtYXRpb25EZXBsb3lBY3Rpb24gZXh0ZW5kcyBDbG91ZEZvcm1hdGlvbkFjdGlvbiB7XG4gIHByaXZhdGUgX2RlcGxveW1lbnRSb2xlPzogaWFtLklSb2xlO1xuICBwcml2YXRlIHJlYWRvbmx5IHByb3BzMjogQ2xvdWRGb3JtYXRpb25EZXBsb3lBY3Rpb25Qcm9wcztcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogQ2xvdWRGb3JtYXRpb25EZXBsb3lBY3Rpb25Qcm9wcywgaW5wdXRzOiBjb2RlcGlwZWxpbmUuQXJ0aWZhY3RbXSB8IHVuZGVmaW5lZCkge1xuICAgIHN1cGVyKHByb3BzLCAocHJvcHMuZXh0cmFJbnB1dHMgfHwgW10pLmNvbmNhdChpbnB1dHMgfHwgW10pKTtcblxuICAgIHRoaXMucHJvcHMyID0gcHJvcHM7XG4gIH1cblxuICAvKipcbiAgICogQWRkIHN0YXRlbWVudCB0byB0aGUgc2VydmljZSByb2xlIGFzc3VtZWQgYnkgQ2xvdWRGb3JtYXRpb24gd2hpbGUgZXhlY3V0aW5nIHRoaXMgYWN0aW9uLlxuICAgKi9cbiAgcHVibGljIGFkZFRvRGVwbG95bWVudFJvbGVQb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0RGVwbG95bWVudFJvbGUoJ21ldGhvZCBhZGRUb1JvbGVQb2xpY3koKScpLmFkZFRvUG9saWN5KHN0YXRlbWVudCk7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGRlcGxveW1lbnRSb2xlKCk6IGlhbS5JUm9sZSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0RGVwbG95bWVudFJvbGUoJ3Byb3BlcnR5IHJvbGUoKScpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGJvdW5kKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgaWYgKHRoaXMucHJvcHMyLmRlcGxveW1lbnRSb2xlKSB7XG4gICAgICB0aGlzLl9kZXBsb3ltZW50Um9sZSA9IHRoaXMucHJvcHMyLmRlcGxveW1lbnRSb2xlO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCByb2xlU3RhY2sgPSBjZGsuU3RhY2sub2Yob3B0aW9ucy5yb2xlKTtcbiAgICAgIGNvbnN0IHBpcGVsaW5lU3RhY2sgPSBjZGsuU3RhY2sub2Yoc2NvcGUpO1xuICAgICAgaWYgKHJvbGVTdGFjay5hY2NvdW50ICE9PSBwaXBlbGluZVN0YWNrLmFjY291bnQpIHtcbiAgICAgICAgLy8gcGFzcyByb2xlIGlzIG5vdCBhbGxvd2VkIGZvciBjcm9zcy1hY2NvdW50IGFjY2VzcyAtIHNvLFxuICAgICAgICAvLyBjcmVhdGUgdGhlIGRlcGxveW1lbnQgUm9sZSBpbiB0aGUgb3RoZXIgYWNjb3VudCFcbiAgICAgICAgdGhpcy5fZGVwbG95bWVudFJvbGUgPSBuZXcgaWFtLlJvbGUocm9sZVN0YWNrLFxuICAgICAgICAgIGAke3N0YWdlLnBpcGVsaW5lLm5vZGUudW5pcXVlSWR9LSR7c3RhZ2Uuc3RhZ2VOYW1lfS0ke3RoaXMuYWN0aW9uUHJvcGVydGllcy5hY3Rpb25OYW1lfS1EZXBsb3ltZW50Um9sZWAsIHtcbiAgICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdjbG91ZGZvcm1hdGlvbi5hbWF6b25hd3MuY29tJyksXG4gICAgICAgICAgICByb2xlTmFtZTogY2RrLlBoeXNpY2FsTmFtZS5HRU5FUkFURV9JRl9ORUVERUQsXG4gICAgICAgICAgfSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aGlzLl9kZXBsb3ltZW50Um9sZSA9IG5ldyBpYW0uUm9sZShzY29wZSwgJ1JvbGUnLCB7XG4gICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2Nsb3VkZm9ybWF0aW9uLmFtYXpvbmF3cy5jb20nKVxuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgICAgLy8gdGhlIGRlcGxveW1lbnQgcm9sZSBtaWdodCBuZWVkIHJlYWQgYWNjZXNzIHRvIHRoZSBwaXBlbGluZSdzIGJ1Y2tldFxuICAgICAgLy8gKGZvciBleGFtcGxlLCBpZiBpdCdzIGRlcGxveWluZyBhIExhbWJkYSBmdW5jdGlvbiksXG4gICAgICAvLyBhbmQgZXZlbiBpZiBpdCBoYXMgYWRtaW4gcGVybWlzc2lvbnMsIGl0IHdvbid0IGJlIGVub3VnaCxcbiAgICAgIC8vIGFzIGl0IG5lZWRzIHRvIGJlIGFkZGVkIHRvIHRoZSBrZXkncyByZXNvdXJjZSBwb2xpY3lcbiAgICAgIC8vIChhbmQgdGhlIGJ1Y2tldCdzLCBpZiB0aGUgYWNjZXNzIGlzIGNyb3NzLWFjY291bnQpXG4gICAgICBvcHRpb25zLmJ1Y2tldC5ncmFudFJlYWQodGhpcy5fZGVwbG95bWVudFJvbGUpO1xuXG4gICAgICBpZiAodGhpcy5wcm9wczIuYWRtaW5QZXJtaXNzaW9ucykge1xuICAgICAgICB0aGlzLl9kZXBsb3ltZW50Um9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogWycqJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgfSkpO1xuICAgICAgfVxuICAgIH1cblxuICAgIFNpbmdsZXRvblBvbGljeS5mb3JSb2xlKG9wdGlvbnMucm9sZSkuZ3JhbnRQYXNzUm9sZSh0aGlzLl9kZXBsb3ltZW50Um9sZSk7XG5cbiAgICBjb25zdCBjYXBhYmlsaXRpZXMgPSB0aGlzLnByb3BzMi5hZG1pblBlcm1pc3Npb25zICYmIHRoaXMucHJvcHMyLmNhcGFiaWxpdGllcyA9PT0gdW5kZWZpbmVkXG4gICAgICA/IFtjbG91ZGZvcm1hdGlvbi5DbG91ZEZvcm1hdGlvbkNhcGFiaWxpdGllcy5OQU1FRF9JQU1dXG4gICAgICA6IHRoaXMucHJvcHMyLmNhcGFiaWxpdGllcztcblxuICAgIGNvbnN0IGFjdGlvbkNvbmZpZyA9IHN1cGVyLmJvdW5kKHNjb3BlLCBzdGFnZSwgb3B0aW9ucyk7XG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLmFjdGlvbkNvbmZpZyxcbiAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgLi4uYWN0aW9uQ29uZmlnLmNvbmZpZ3VyYXRpb24sXG4gICAgICAgIC8vIE5vbmUgZXZhbHVhdGVzIHRvIGVtcHR5IHN0cmluZyB3aGljaCBpcyBmYWxzZXkgYW5kIHJlc3VsdHMgaW4gdW5kZWZpbmVkXG4gICAgICAgIENhcGFiaWxpdGllczogcGFyc2VDYXBhYmlsaXRpZXMoY2FwYWJpbGl0aWVzKSxcbiAgICAgICAgUm9sZUFybjogdGhpcy5kZXBsb3ltZW50Um9sZS5yb2xlQXJuLFxuICAgICAgICBQYXJhbWV0ZXJPdmVycmlkZXM6IGNkay5TdGFjay5vZihzY29wZSkudG9Kc29uU3RyaW5nKHRoaXMucHJvcHMyLnBhcmFtZXRlck92ZXJyaWRlcyksXG4gICAgICAgIFRlbXBsYXRlQ29uZmlndXJhdGlvbjogdGhpcy5wcm9wczIudGVtcGxhdGVDb25maWd1cmF0aW9uXG4gICAgICAgICAgPyB0aGlzLnByb3BzMi50ZW1wbGF0ZUNvbmZpZ3VyYXRpb24ubG9jYXRpb25cbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgU3RhY2tOYW1lOiB0aGlzLnByb3BzMi5zdGFja05hbWUsXG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIGdldERlcGxveW1lbnRSb2xlKG1lbWJlcjogc3RyaW5nKTogaWFtLklSb2xlIHtcbiAgICBpZiAodGhpcy5fZGVwbG95bWVudFJvbGUpIHtcbiAgICAgIHJldHVybiB0aGlzLl9kZXBsb3ltZW50Um9sZTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3QgdXNlIHRoZSAke21lbWJlcn0gYmVmb3JlIHRoZSBBY3Rpb24gaGFzIGJlZW4gYWRkZWQgdG8gYSBQaXBlbGluZWApO1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHRoZSBDbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2xvdWRGb3JtYXRpb25DcmVhdGVSZXBsYWNlQ2hhbmdlU2V0QWN0aW9uUHJvcHMgZXh0ZW5kcyBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGNoYW5nZSBzZXQgdG8gY3JlYXRlIG9yIHVwZGF0ZS5cbiAgICovXG4gIHJlYWRvbmx5IGNoYW5nZVNldE5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogSW5wdXQgYXJ0aWZhY3Qgd2l0aCB0aGUgQ2hhbmdlU2V0J3MgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGVcbiAgICovXG4gIHJlYWRvbmx5IHRlbXBsYXRlUGF0aDogY29kZXBpcGVsaW5lLkFydGlmYWN0UGF0aDtcbn1cblxuLyoqXG4gKiBDb2RlUGlwZWxpbmUgYWN0aW9uIHRvIHByZXBhcmUgYSBjaGFuZ2Ugc2V0LlxuICpcbiAqIENyZWF0ZXMgdGhlIGNoYW5nZSBzZXQgaWYgaXQgZG9lc24ndCBleGlzdCBiYXNlZCBvbiB0aGUgc3RhY2sgbmFtZSBhbmQgdGVtcGxhdGUgdGhhdCB5b3Ugc3VibWl0LlxuICogSWYgdGhlIGNoYW5nZSBzZXQgZXhpc3RzLCBBV1MgQ2xvdWRGb3JtYXRpb24gZGVsZXRlcyBpdCwgYW5kIHRoZW4gY3JlYXRlcyBhIG5ldyBvbmUuXG4gKi9cbmV4cG9ydCBjbGFzcyBDbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb24gZXh0ZW5kcyBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvbiB7XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJvcHMzOiBDbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb25Qcm9wcztcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogQ2xvdWRGb3JtYXRpb25DcmVhdGVSZXBsYWNlQ2hhbmdlU2V0QWN0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcywgcHJvcHMudGVtcGxhdGVDb25maWd1cmF0aW9uXG4gICAgICA/IFtwcm9wcy50ZW1wbGF0ZVBhdGguYXJ0aWZhY3QsIHByb3BzLnRlbXBsYXRlQ29uZmlndXJhdGlvbi5hcnRpZmFjdF1cbiAgICAgIDogW3Byb3BzLnRlbXBsYXRlUGF0aC5hcnRpZmFjdF0pO1xuXG4gICAgdGhpcy5wcm9wczMgPSBwcm9wcztcbiAgfVxuXG4gIHByb3RlY3RlZCBib3VuZChzY29wZTogY2RrLkNvbnN0cnVjdCwgc3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2UsIG9wdGlvbnM6IGNvZGVwaXBlbGluZS5BY3Rpb25CaW5kT3B0aW9ucyk6XG4gIGNvZGVwaXBlbGluZS5BY3Rpb25Db25maWcge1xuICAgIC8vIHRoZSBzdXBlciBjYWxsIG9yZGVyIGlzIHRvIHByZXNlcnZlIHRoZSBleGlzdGluZyBvcmRlciBvZiBzdGF0ZW1lbnRzIGluIHBvbGljaWVzXG4gICAgY29uc3QgYWN0aW9uQ29uZmlnID0gc3VwZXIuYm91bmQoc2NvcGUsIHN0YWdlLCBvcHRpb25zKTtcblxuICAgIFNpbmdsZXRvblBvbGljeS5mb3JSb2xlKG9wdGlvbnMucm9sZSkuZ3JhbnRDcmVhdGVSZXBsYWNlQ2hhbmdlU2V0KHRoaXMucHJvcHMzKTtcblxuICAgIHJldHVybiB7XG4gICAgICAuLi5hY3Rpb25Db25maWcsXG4gICAgICBjb25maWd1cmF0aW9uOiB7XG4gICAgICAgIC4uLmFjdGlvbkNvbmZpZy5jb25maWd1cmF0aW9uLFxuICAgICAgICBBY3Rpb25Nb2RlOiAnQ0hBTkdFX1NFVF9SRVBMQUNFJyxcbiAgICAgICAgQ2hhbmdlU2V0TmFtZTogdGhpcy5wcm9wczMuY2hhbmdlU2V0TmFtZSxcbiAgICAgICAgVGVtcGxhdGVQYXRoOiB0aGlzLnByb3BzMy50ZW1wbGF0ZVBhdGgubG9jYXRpb24sXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciB0aGUgQ2xvdWRGb3JtYXRpb25DcmVhdGVVcGRhdGVTdGFja0FjdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDbG91ZEZvcm1hdGlvbkNyZWF0ZVVwZGF0ZVN0YWNrQWN0aW9uUHJvcHMgZXh0ZW5kcyBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIElucHV0IGFydGlmYWN0IHdpdGggdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlIHRvIGRlcGxveVxuICAgKi9cbiAgcmVhZG9ubHkgdGVtcGxhdGVQYXRoOiBjb2RlcGlwZWxpbmUuQXJ0aWZhY3RQYXRoO1xuXG4gIC8qKlxuICAgKiBSZXBsYWNlIHRoZSBzdGFjayBpZiBpdCdzIGluIGEgZmFpbGVkIHN0YXRlLlxuICAgKlxuICAgKiBJZiB0aGlzIGlzIHNldCB0byB0cnVlIGFuZCB0aGUgc3RhY2sgaXMgaW4gYSBmYWlsZWQgc3RhdGUgKG9uZSBvZlxuICAgKiBST0xMQkFDS19DT01QTEVURSwgUk9MTEJBQ0tfRkFJTEVELCBDUkVBVEVfRkFJTEVELCBERUxFVEVfRkFJTEVELCBvclxuICAgKiBVUERBVEVfUk9MTEJBQ0tfRkFJTEVEKSwgQVdTIENsb3VkRm9ybWF0aW9uIGRlbGV0ZXMgdGhlIHN0YWNrIGFuZCB0aGVuXG4gICAqIGNyZWF0ZXMgYSBuZXcgc3RhY2suXG4gICAqXG4gICAqIElmIHRoaXMgaXMgbm90IHNldCB0byB0cnVlIGFuZCB0aGUgc3RhY2sgaXMgaW4gYSBmYWlsZWQgc3RhdGUsXG4gICAqIHRoZSBkZXBsb3ltZW50IGZhaWxzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVwbGFjZU9uRmFpbHVyZT86IGJvb2xlYW47XG59XG5cbi8qKlxuICogQ29kZVBpcGVsaW5lIGFjdGlvbiB0byBkZXBsb3kgYSBzdGFjay5cbiAqXG4gKiBDcmVhdGVzIHRoZSBzdGFjayBpZiB0aGUgc3BlY2lmaWVkIHN0YWNrIGRvZXNuJ3QgZXhpc3QuIElmIHRoZSBzdGFjayBleGlzdHMsXG4gKiBBV1MgQ2xvdWRGb3JtYXRpb24gdXBkYXRlcyB0aGUgc3RhY2suIFVzZSB0aGlzIGFjdGlvbiB0byB1cGRhdGUgZXhpc3RpbmdcbiAqIHN0YWNrcy5cbiAqXG4gKiBBV1MgQ29kZVBpcGVsaW5lIHdvbid0IHJlcGxhY2UgdGhlIHN0YWNrLCBhbmQgd2lsbCBmYWlsIGRlcGxveW1lbnQgaWYgdGhlXG4gKiBzdGFjayBpcyBpbiBhIGZhaWxlZCBzdGF0ZS4gVXNlIGBSZXBsYWNlT25GYWlsdXJlYCBmb3IgYW4gYWN0aW9uIHRoYXRcbiAqIHdpbGwgZGVsZXRlIGFuZCByZWNyZWF0ZSB0aGUgc3RhY2sgdG8gdHJ5IGFuZCByZWNvdmVyIGZyb20gZmFpbGVkIHN0YXRlcy5cbiAqXG4gKiBVc2UgdGhpcyBhY3Rpb24gdG8gYXV0b21hdGljYWxseSByZXBsYWNlIGZhaWxlZCBzdGFja3Mgd2l0aG91dCByZWNvdmVyaW5nIG9yXG4gKiB0cm91Ymxlc2hvb3RpbmcgdGhlbS4gWW91IHdvdWxkIHR5cGljYWxseSBjaG9vc2UgdGhpcyBtb2RlIGZvciB0ZXN0aW5nLlxuICovXG5leHBvcnQgY2xhc3MgQ2xvdWRGb3JtYXRpb25DcmVhdGVVcGRhdGVTdGFja0FjdGlvbiBleHRlbmRzIENsb3VkRm9ybWF0aW9uRGVwbG95QWN0aW9uIHtcbiAgcHJpdmF0ZSByZWFkb25seSBwcm9wczM6IENsb3VkRm9ybWF0aW9uQ3JlYXRlVXBkYXRlU3RhY2tBY3Rpb25Qcm9wcztcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogQ2xvdWRGb3JtYXRpb25DcmVhdGVVcGRhdGVTdGFja0FjdGlvblByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMsIHByb3BzLnRlbXBsYXRlQ29uZmlndXJhdGlvblxuICAgICAgPyBbcHJvcHMudGVtcGxhdGVQYXRoLmFydGlmYWN0LCBwcm9wcy50ZW1wbGF0ZUNvbmZpZ3VyYXRpb24uYXJ0aWZhY3RdXG4gICAgICA6IFtwcm9wcy50ZW1wbGF0ZVBhdGguYXJ0aWZhY3RdKTtcblxuICAgIHRoaXMucHJvcHMzID0gcHJvcHM7XG4gIH1cblxuICBwcm90ZWN0ZWQgYm91bmQoc2NvcGU6IGNkay5Db25zdHJ1Y3QsIHN0YWdlOiBjb2RlcGlwZWxpbmUuSVN0YWdlLCBvcHRpb25zOiBjb2RlcGlwZWxpbmUuQWN0aW9uQmluZE9wdGlvbnMpOlxuICBjb2RlcGlwZWxpbmUuQWN0aW9uQ29uZmlnIHtcbiAgICAvLyB0aGUgc3VwZXIgY2FsbCBvcmRlciBpcyB0byBwcmVzZXJ2ZSB0aGUgZXhpc3Rpbmcgb3JkZXIgb2Ygc3RhdGVtZW50cyBpbiBwb2xpY2llc1xuICAgIGNvbnN0IGFjdGlvbkNvbmZpZyA9IHN1cGVyLmJvdW5kKHNjb3BlLCBzdGFnZSwgb3B0aW9ucyk7XG5cbiAgICBTaW5nbGV0b25Qb2xpY3kuZm9yUm9sZShvcHRpb25zLnJvbGUpLmdyYW50Q3JlYXRlVXBkYXRlU3RhY2sodGhpcy5wcm9wczMpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLmFjdGlvbkNvbmZpZyxcbiAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgLi4uYWN0aW9uQ29uZmlnLmNvbmZpZ3VyYXRpb24sXG4gICAgICAgIEFjdGlvbk1vZGU6IHRoaXMucHJvcHMzLnJlcGxhY2VPbkZhaWx1cmUgPyAnUkVQTEFDRV9PTl9GQUlMVVJFJyA6ICdDUkVBVEVfVVBEQVRFJyxcbiAgICAgICAgVGVtcGxhdGVQYXRoOiB0aGlzLnByb3BzMy50ZW1wbGF0ZVBhdGgubG9jYXRpb24sXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciB0aGUgQ2xvdWRGb3JtYXRpb25EZWxldGVTdGFja0FjdGlvbi5cbiAqL1xuLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm5vLWVtcHR5LWludGVyZmFjZVxuZXhwb3J0IGludGVyZmFjZSBDbG91ZEZvcm1hdGlvbkRlbGV0ZVN0YWNrQWN0aW9uUHJvcHMgZXh0ZW5kcyBDbG91ZEZvcm1hdGlvbkRlcGxveUFjdGlvblByb3BzIHtcbn1cblxuLyoqXG4gKiBDb2RlUGlwZWxpbmUgYWN0aW9uIHRvIGRlbGV0ZSBhIHN0YWNrLlxuICpcbiAqIERlbGV0ZXMgYSBzdGFjay4gSWYgeW91IHNwZWNpZnkgYSBzdGFjayB0aGF0IGRvZXNuJ3QgZXhpc3QsIHRoZSBhY3Rpb24gY29tcGxldGVzIHN1Y2Nlc3NmdWxseVxuICogd2l0aG91dCBkZWxldGluZyBhIHN0YWNrLlxuICovXG5leHBvcnQgY2xhc3MgQ2xvdWRGb3JtYXRpb25EZWxldGVTdGFja0FjdGlvbiBleHRlbmRzIENsb3VkRm9ybWF0aW9uRGVwbG95QWN0aW9uIHtcbiAgcHJpdmF0ZSByZWFkb25seSBwcm9wczM6IENsb3VkRm9ybWF0aW9uRGVsZXRlU3RhY2tBY3Rpb25Qcm9wcztcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogQ2xvdWRGb3JtYXRpb25EZWxldGVTdGFja0FjdGlvblByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMsIHVuZGVmaW5lZCk7XG5cbiAgICB0aGlzLnByb3BzMyA9IHByb3BzO1xuICB9XG5cbiAgcHJvdGVjdGVkIGJvdW5kKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgLy8gdGhlIHN1cGVyIGNhbGwgb3JkZXIgaXMgdG8gcHJlc2VydmUgdGhlIGV4aXN0aW5nIG9yZGVyIG9mIHN0YXRlbWVudHMgaW4gcG9saWNpZXNcbiAgICBjb25zdCBhY3Rpb25Db25maWcgPSBzdXBlci5ib3VuZChzY29wZSwgc3RhZ2UsIG9wdGlvbnMpO1xuXG4gICAgU2luZ2xldG9uUG9saWN5LmZvclJvbGUob3B0aW9ucy5yb2xlKS5ncmFudERlbGV0ZVN0YWNrKHRoaXMucHJvcHMzKTtcblxuICAgIHJldHVybiB7XG4gICAgICAuLi5hY3Rpb25Db25maWcsXG4gICAgICBjb25maWd1cmF0aW9uOiB7XG4gICAgICAgIC4uLmFjdGlvbkNvbmZpZy5jb25maWd1cmF0aW9uLFxuICAgICAgICBBY3Rpb25Nb2RlOiAnREVMRVRFX09OTFknLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogTWFuYWdlcyBhIGJ1bmNoIG9mIHNpbmdsZXRvbi15IHN0YXRlbWVudHMgb24gdGhlIHBvbGljeSBvZiBhbiBJQU0gUm9sZS5cbiAqIERlZGljYXRlZCBtZXRob2RzIGNhbiBiZSB1c2VkIHRvIGFkZCBzcGVjaWZpYyBwZXJtaXNzaW9ucyB0byB0aGUgcm9sZSBwb2xpY3lcbiAqIHVzaW5nIGFzIGZldyBzdGF0ZW1lbnRzIGFzIHBvc3NpYmxlIChhZGRpbmcgcmVzb3VyY2VzIHRvIGV4aXN0aW5nIGNvbXBhdGlibGVcbiAqIHN0YXRlbWVudHMgaW5zdGVhZCBvZiBhZGRpbmcgbmV3IHN0YXRlbWVudHMgd2hlbmV2ZXIgcG9zc2libGUpLlxuICpcbiAqIFN0YXRlbWVudHMgY3JlYXRlZCBvdXRzaWRlIG9mIHRoaXMgY2xhc3MgYXJlIG5vdCBjb25zaWRlcmVkIHdoZW4gYWRkaW5nIG5ld1xuICogcGVybWlzc2lvbnMuXG4gKi9cbmNsYXNzIFNpbmdsZXRvblBvbGljeSBleHRlbmRzIGNkay5Db25zdHJ1Y3QgaW1wbGVtZW50cyBpYW0uSUdyYW50YWJsZSB7XG4gIC8qKlxuICAgKiBPYnRhaW4gYSBTaW5nbGV0b25Qb2xpY3kgZm9yIGEgZ2l2ZW4gcm9sZS5cbiAgICogQHBhcmFtIHJvbGUgdGhlIFJvbGUgdGhpcyBwb2xpY3kgaXMgYm91bmQgdG8uXG4gICAqIEByZXR1cm5zIHRoZSBTaW5nbGV0b25Qb2xpY3kgZm9yIHRoaXMgcm9sZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZm9yUm9sZShyb2xlOiBpYW0uSVJvbGUpOiBTaW5nbGV0b25Qb2xpY3kge1xuICAgIGNvbnN0IGZvdW5kID0gcm9sZS5ub2RlLnRyeUZpbmRDaGlsZChTaW5nbGV0b25Qb2xpY3kuVVVJRCk7XG4gICAgcmV0dXJuIChmb3VuZCBhcyBTaW5nbGV0b25Qb2xpY3kpIHx8IG5ldyBTaW5nbGV0b25Qb2xpY3kocm9sZSk7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBVVUlEID0gJzgzODllNzVmLTA4MTAtNDgzOC1iZjY0LWQ2Zjg1YTk1Y2Y4Myc7XG5cbiAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcblxuICBwcml2YXRlIHN0YXRlbWVudHM6IHsgW2tleTogc3RyaW5nXTogaWFtLlBvbGljeVN0YXRlbWVudCB9ID0ge307XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHJvbGU6IGlhbS5JUm9sZSkge1xuICAgIHN1cGVyKHJvbGUgYXMgdW5rbm93biBhcyBjZGsuQ29uc3RydWN0LCBTaW5nbGV0b25Qb2xpY3kuVVVJRCk7XG4gICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHJvbGU7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnRFeGVjdXRlQ2hhbmdlU2V0KHByb3BzOiB7IHN0YWNrTmFtZTogc3RyaW5nLCBjaGFuZ2VTZXROYW1lOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZyB9KTogdm9pZCB7XG4gICAgdGhpcy5zdGF0ZW1lbnRGb3Ioe1xuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVTdGFja3MnLFxuICAgICAgICAnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVDaGFuZ2VTZXQnLFxuICAgICAgICAnY2xvdWRmb3JtYXRpb246RXhlY3V0ZUNoYW5nZVNldCcsXG4gICAgICBdLFxuICAgICAgY29uZGl0aW9uczoge8KgU3RyaW5nRXF1YWxzSWZFeGlzdHM6IHsgJ2Nsb3VkZm9ybWF0aW9uOkNoYW5nZVNldE5hbWUnOiBwcm9wcy5jaGFuZ2VTZXROYW1lIH0gfSxcbiAgICB9KS5hZGRSZXNvdXJjZXModGhpcy5zdGFja0FybkZyb21Qcm9wcyhwcm9wcykpO1xuICB9XG5cbiAgcHVibGljIGdyYW50Q3JlYXRlUmVwbGFjZUNoYW5nZVNldChwcm9wczogeyBzdGFja05hbWU6IHN0cmluZywgY2hhbmdlU2V0TmFtZTogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcgfSk6IHZvaWQge1xuICAgIHRoaXMuc3RhdGVtZW50Rm9yKHtcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkNyZWF0ZUNoYW5nZVNldCcsXG4gICAgICAgICdjbG91ZGZvcm1hdGlvbjpEZWxldGVDaGFuZ2VTZXQnLFxuICAgICAgICAnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVDaGFuZ2VTZXQnLFxuICAgICAgICAnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVTdGFja3MnLFxuICAgICAgXSxcbiAgICAgIGNvbmRpdGlvbnM6IHsgU3RyaW5nRXF1YWxzSWZFeGlzdHM6IHsgJ2Nsb3VkZm9ybWF0aW9uOkNoYW5nZVNldE5hbWUnOiBwcm9wcy5jaGFuZ2VTZXROYW1lIH0gfSxcbiAgICB9KS5hZGRSZXNvdXJjZXModGhpcy5zdGFja0FybkZyb21Qcm9wcyhwcm9wcykpO1xuICB9XG5cbiAgcHVibGljIGdyYW50Q3JlYXRlVXBkYXRlU3RhY2socHJvcHM6IHsgc3RhY2tOYW1lOiBzdHJpbmcsIHJlcGxhY2VPbkZhaWx1cmU/OiBib29sZWFuLCByZWdpb24/OiBzdHJpbmcgfSk6IHZvaWQge1xuICAgIGNvbnN0IGFjdGlvbnMgPSBbXG4gICAgICAnY2xvdWRmb3JtYXRpb246RGVzY3JpYmVTdGFjayonLFxuICAgICAgJ2Nsb3VkZm9ybWF0aW9uOkNyZWF0ZVN0YWNrJyxcbiAgICAgICdjbG91ZGZvcm1hdGlvbjpVcGRhdGVTdGFjaycsXG4gICAgICAnY2xvdWRmb3JtYXRpb246R2V0VGVtcGxhdGUqJyxcbiAgICAgICdjbG91ZGZvcm1hdGlvbjpWYWxpZGF0ZVRlbXBsYXRlJyxcbiAgICAgICdjbG91ZGZvcm1hdGlvbjpHZXRTdGFja1BvbGljeScsXG4gICAgICAnY2xvdWRmb3JtYXRpb246U2V0U3RhY2tQb2xpY3knLFxuICAgIF07XG4gICAgaWYgKHByb3BzLnJlcGxhY2VPbkZhaWx1cmUpIHtcbiAgICAgIGFjdGlvbnMucHVzaCgnY2xvdWRmb3JtYXRpb246RGVsZXRlU3RhY2snKTtcbiAgICB9XG4gICAgdGhpcy5zdGF0ZW1lbnRGb3IoeyBhY3Rpb25zIH0pLmFkZFJlc291cmNlcyh0aGlzLnN0YWNrQXJuRnJvbVByb3BzKHByb3BzKSk7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnREZWxldGVTdGFjayhwcm9wczogeyBzdGFja05hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nIH0pOiB2b2lkIHtcbiAgICB0aGlzLnN0YXRlbWVudEZvcih7XG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdjbG91ZGZvcm1hdGlvbjpEZXNjcmliZVN0YWNrKicsXG4gICAgICAgICdjbG91ZGZvcm1hdGlvbjpEZWxldGVTdGFjaycsXG4gICAgICBdXG4gICAgfSkuYWRkUmVzb3VyY2VzKHRoaXMuc3RhY2tBcm5Gcm9tUHJvcHMocHJvcHMpKTtcbiAgfVxuXG4gIHB1YmxpYyBncmFudFBhc3NSb2xlKHJvbGU6IGlhbS5JUm9sZSk6IHZvaWQge1xuICAgIHRoaXMuc3RhdGVtZW50Rm9yKHsgYWN0aW9uczogWydpYW06UGFzc1JvbGUnXSB9KS5hZGRSZXNvdXJjZXMocm9sZS5yb2xlQXJuKTtcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGVtZW50Rm9yKHRlbXBsYXRlOiBTdGF0ZW1lbnRUZW1wbGF0ZSk6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQge1xuICAgIGNvbnN0IGtleSA9IGtleUZvcih0ZW1wbGF0ZSk7XG4gICAgaWYgKCEoa2V5IGluIHRoaXMuc3RhdGVtZW50cykpIHtcbiAgICAgIHRoaXMuc3RhdGVtZW50c1trZXldID0gbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoeyBhY3Rpb25zOiB0ZW1wbGF0ZS5hY3Rpb25zIH0pO1xuICAgICAgaWYgKHRlbXBsYXRlLmNvbmRpdGlvbnMpIHtcbiAgICAgICAgdGhpcy5zdGF0ZW1lbnRzW2tleV0uYWRkQ29uZGl0aW9ucyh0ZW1wbGF0ZS5jb25kaXRpb25zKTtcbiAgICAgIH1cbiAgICAgIHRoaXMucm9sZS5hZGRUb1BvbGljeSh0aGlzLnN0YXRlbWVudHNba2V5XSk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLnN0YXRlbWVudHNba2V5XTtcblxuICAgIGZ1bmN0aW9uIGtleUZvcihwcm9wczogU3RhdGVtZW50VGVtcGxhdGUpOiBzdHJpbmcge1xuICAgICAgY29uc3QgYWN0aW9ucyA9IGAke3Byb3BzLmFjdGlvbnMuc29ydCgpLmpvaW4oJ1xceDFGJyl9YDtcbiAgICAgIGNvbnN0IGNvbmRpdGlvbnMgPSBmb3JtYXRDb25kaXRpb25zKHByb3BzLmNvbmRpdGlvbnMpO1xuICAgICAgcmV0dXJuIGAke2FjdGlvbnN9XFx4MUQke2NvbmRpdGlvbnN9YDtcblxuICAgICAgZnVuY3Rpb24gZm9ybWF0Q29uZGl0aW9ucyhjb25kPzogU3RhdGVtZW50Q29uZGl0aW9uKTogc3RyaW5nIHtcbiAgICAgICAgaWYgKGNvbmQgPT0gbnVsbCkgeyByZXR1cm4gJyc7IH1cbiAgICAgICAgbGV0IHJlc3VsdCA9ICcnO1xuICAgICAgICBmb3IgKGNvbnN0IG9wIG9mIE9iamVjdC5rZXlzKGNvbmQpLnNvcnQoKSkge1xuICAgICAgICAgIHJlc3VsdCArPSBgJHtvcH1cXHgxRWA7XG4gICAgICAgICAgY29uc3QgY29uZGl0aW9uID0gY29uZFtvcF07XG4gICAgICAgICAgZm9yIChjb25zdCBhdHRyaWJ1dGUgb2YgT2JqZWN0LmtleXMoY29uZGl0aW9uKS5zb3J0KCkpIHtcbiAgICAgICAgICAgIGNvbnN0IHZhbHVlID0gY29uZGl0aW9uW2F0dHJpYnV0ZV07XG4gICAgICAgICAgICByZXN1bHQgKz0gYCR7dmFsdWV9XFx4MUZgO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmVzdWx0O1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgc3RhY2tBcm5Gcm9tUHJvcHMocHJvcHM6IHsgc3RhY2tOYW1lOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZyB9KTogc3RyaW5nIHtcbiAgICByZXR1cm4gY2RrLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICByZWdpb246IHByb3BzLnJlZ2lvbixcbiAgICAgIHNlcnZpY2U6ICdjbG91ZGZvcm1hdGlvbicsXG4gICAgICByZXNvdXJjZTogJ3N0YWNrJyxcbiAgICAgIHJlc291cmNlTmFtZTogYCR7cHJvcHMuc3RhY2tOYW1lfS8qYFxuICAgIH0pO1xuICB9XG59XG5cbmludGVyZmFjZSBTdGF0ZW1lbnRUZW1wbGF0ZSB7XG4gIGFjdGlvbnM6IHN0cmluZ1tdO1xuICBjb25kaXRpb25zPzogU3RhdGVtZW50Q29uZGl0aW9uO1xufVxuXG50eXBlIFN0YXRlbWVudENvbmRpdGlvbiA9IHsgW29wOiBzdHJpbmddOiB7IFthdHRyaWJ1dGU6IHN0cmluZ106IHN0cmluZyB9IH07XG5cbmZ1bmN0aW9uIHBhcnNlQ2FwYWJpbGl0aWVzKGNhcGFiaWxpdGllczogY2xvdWRmb3JtYXRpb24uQ2xvdWRGb3JtYXRpb25DYXBhYmlsaXRpZXNbXSB8IHVuZGVmaW5lZCk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gIGlmIChjYXBhYmlsaXRpZXMgPT09IHVuZGVmaW5lZCkge1xuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH0gZWxzZSBpZiAoY2FwYWJpbGl0aWVzLmxlbmd0aCA9PT0gMSkge1xuICAgIGNvbnN0IGNhcGFiaWxpdHkgPSBjYXBhYmlsaXRpZXMudG9TdHJpbmcoKTtcbiAgICByZXR1cm4gKGNhcGFiaWxpdHkgPT09ICcnKSA/IHVuZGVmaW5lZCA6IGNhcGFiaWxpdHk7XG4gIH0gZWxzZSBpZiAoY2FwYWJpbGl0aWVzLmxlbmd0aCA+IDEpIHtcbiAgICByZXR1cm4gY2FwYWJpbGl0aWVzLmpvaW4oJywnKTtcbiAgfVxuXG4gIHJldHVybiB1bmRlZmluZWQ7XG59XG4iXX0=