"use strict";
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const cpactions = require("../../lib");
function _assertActionMatches(test, stack, actions, provider, category, configuration) {
    const configurationStr = configuration
        ? `, configuration including ${JSON.stringify(stack.resolve(configuration), null, 2)}`
        : '';
    const actionsStr = JSON.stringify(actions.map(a => ({ owner: a.actionProperties.owner, provider: a.actionProperties.provider,
        category: a.actionProperties.category, configuration: stack.resolve(a.actionConfig.configuration)
    })), null, 2);
    test.ok(_hasAction(stack, actions, provider, category, configuration), `Expected to find an action with provider ${provider}, category ${category}${configurationStr}, but found ${actionsStr}`);
}
function _hasAction(stack, actions, provider, category, configuration) {
    for (const action of actions) {
        if (action.actionProperties.provider !== provider) {
            continue;
        }
        if (action.actionProperties.category !== category) {
            continue;
        }
        if (configuration && !action.actionConfig.configuration) {
            continue;
        }
        if (configuration) {
            for (const key of Object.keys(configuration)) {
                if (!_.isEqual(stack.resolve(action.actionConfig.configuration[key]), stack.resolve(configuration[key]))) {
                    continue;
                }
            }
        }
        return true;
    }
    return false;
}
function _assertPermissionGranted(test, stack, statements, action, resource, conditions) {
    const conditionStr = conditions
        ? ` with condition(s) ${JSON.stringify(stack.resolve(conditions))}`
        : '';
    const resolvedStatements = stack.resolve(statements.map(s => s.toStatementJson()));
    const statementsStr = JSON.stringify(resolvedStatements, null, 2);
    test.ok(_grantsPermission(stack, resolvedStatements, action, resource, conditions), `Expected to find a statement granting ${action} on ${JSON.stringify(stack.resolve(resource))}${conditionStr}, found:\n${statementsStr}`);
}
function _grantsPermission(stack, statements, action, resource, conditions) {
    for (const statement of statements.filter(s => s.Effect === 'Allow')) {
        if (!_isOrContains(stack, statement.Action, action)) {
            continue;
        }
        if (!_isOrContains(stack, statement.Resource, resource)) {
            continue;
        }
        if (conditions && !_isOrContains(stack, statement.Condition, conditions)) {
            continue;
        }
        return true;
    }
    return false;
}
function _isOrContains(stack, entity, value) {
    const resolvedValue = stack.resolve(value);
    const resolvedEntity = stack.resolve(entity);
    if (_.isEqual(resolvedEntity, resolvedValue)) {
        return true;
    }
    if (!Array.isArray(resolvedEntity)) {
        return false;
    }
    for (const tested of entity) {
        if (_.isEqual(tested, resolvedValue)) {
            return true;
        }
    }
    return false;
}
function _stackArn(stackName, scope) {
    return cdk.Stack.of(scope).formatArn({
        service: 'cloudformation',
        resource: 'stack',
        resourceName: `${stackName}/*`,
    });
}
class PipelineDouble extends cdk.Resource {
    constructor(scope, id, { pipelineName, role }) {
        super(scope, id);
        this.pipelineName = pipelineName || 'TestPipeline';
        this.pipelineArn = cdk.Stack.of(this).formatArn({ service: 'codepipeline', resource: 'pipeline', resourceName: this.pipelineName });
        this.role = role;
        this.artifactBucket = new BucketDouble(scope, 'BucketDouble');
    }
    onEvent(_id, _options) {
        throw new Error('Method not implemented.');
    }
    onStateChange(_id, _options) {
        throw new Error('Method not implemented.');
    }
}
class FullAction {
    constructor(actionProperties, actionConfig) {
        this.actionProperties = actionProperties;
        this.actionConfig = actionConfig;
        // empty
    }
}
class StageDouble {
    constructor({ name, pipeline, actions }) {
        this.actions = [];
        this.stageName = name || 'TestStage';
        this.pipeline = pipeline;
        const stageParent = new cdk.Construct(pipeline, this.stageName);
        const fullActions = new Array();
        for (const action of actions) {
            const actionParent = new cdk.Construct(stageParent, action.actionProperties.actionName);
            fullActions.push(new FullAction(action.actionProperties, action.bind(actionParent, this, {
                role: pipeline.role,
                bucket: pipeline.artifactBucket,
            })));
        }
        this.fullActions = fullActions;
    }
    get node() {
        throw new Error('StageDouble is not a real construct');
    }
    addAction(_action) {
        throw new Error('addAction() is not supported on StageDouble');
    }
    onStateChange(_name, _target, _options) {
        throw new Error('onStateChange() is not supported on StageDouble');
    }
}
class RoleDouble extends iam.Role {
    constructor(scope, id, props = { assumedBy: new iam.ServicePrincipal('test') }) {
        super(scope, id, props);
        this.statements = new Array();
    }
    addToPolicy(statement) {
        super.addToPolicy(statement);
        this.statements.push(statement);
        return true;
    }
}
class BucketDouble extends s3.Bucket {
    grantRead(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantReadWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
}
module.exports = nodeunit.testCase({
    'CreateReplaceChangeSet': {
        'works'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Stack');
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            const action = new cpactions.CloudFormationCreateReplaceChangeSetAction({
                actionName: 'Action',
                changeSetName: 'MyChangeSet',
                stackName: 'MyStack',
                templatePath: artifact.atPath('path/to/file'),
                adminPermissions: false,
            });
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [action],
            });
            cdk.ConstructNode.prepare(stack.node);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
            const stackArn = _stackArn('MyStack', stack);
            const changeSetCondition = { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } };
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStacks', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteChangeSet', stackArn, changeSetCondition);
            // TODO: revert "as any" once we move all actions into a single package.
            test.deepEqual(stage.fullActions[0].actionProperties.inputs, [artifact], 'The input was correctly registered');
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_CREATE_REPLACE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet'
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/file')
                    }),
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/other/file')
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [
                        { 'Fn::GetAtt': ['PipelineTestStageActionARole9283FBE3', 'Arn'] },
                        { 'Fn::GetAtt': ['PipelineTestStageActionBRoleCABC8FA5', 'Arn'] }
                    ],
                },
                {
                    Action: [
                        'cloudformation:CreateChangeSet',
                        'cloudformation:DeleteChangeSet',
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks'
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] }
                    ],
                }
            ]);
            test.done();
        }
    },
    'ExecuteChangeSet': {
        'works'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'Action',
                        changeSetName: 'MyChangeSet',
                        stackName: 'MyStack',
                    }),
                ],
            });
            const stackArn = _stackArn('MyStack', stack);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:ExecuteChangeSet', stackArn, { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } });
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_EXECUTE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet'
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                    }),
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: [
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks',
                        'cloudformation:ExecuteChangeSet',
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // tslint:disable-next-line:max-line-length
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] }
                    ],
                }
            ]);
            test.done();
        }
    },
    'the CreateUpdateStack Action sets the DescribeStack*, Create/Update/DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'Action',
            templatePath: new codepipeline.Artifact('TestArtifact').atPath('some/file'),
            stackName: 'MyStack',
            adminPermissions: false,
            replaceOnFailure: true,
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:UpdateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
    'the DeleteStack Action sets the DescribeStack*, DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationDeleteStackAction({
            actionName: 'Action',
            adminPermissions: false,
            stackName: 'MyStack',
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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