"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsiteOrigin = exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using by default a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps = (props) => {
            this.validateEncryptionSettings(props);
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
            props.websiteBucket && this.validateBucketConfig(props.websiteBucket);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error(`validateRuntimeConfig only accepts non-null RuntimeOptions.`);
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error(`RuntimeOptions.jsonFileName must be a json file.`);
            }
        };
        this.validateBucketConfig = (bucket) => {
            if (bucket.isWebsite) {
                throw new Error("Website buckets cannot be configured as websites as this will break Cloudfront hosting!");
            }
        };
        this.validateEncryptionSettings = ({ defaultWebsiteBucketEncryption, defaultWebsiteBucketEncryptionKey, }) => {
            if (defaultWebsiteBucketEncryptionKey &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS) {
                throw new Error("Bucket encryption should be set to KMS if providing a defaultWebsiteBucketEncryptionKey.");
            }
            if (defaultWebsiteBucketEncryption &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.S3_MANAGED) {
                throw new Error("Only KMS and S3_MANAGED encryption are supported on the default bucket.");
            }
        };
        this.validateProps(props);
        // S3 Bucket to hold website files
        this.websiteBucket =
            props.websiteBucket ??
                new aws_s3_1.Bucket(this, "WebsiteBucket", {
                    versioned: true,
                    enforceSSL: true,
                    autoDeleteObjects: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                    encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                    serverAccessLogsPrefix: "access-logs",
                });
        // Web ACL
        const { distributionProps } = props;
        const webAclArn = distributionProps?.webAclId ??
            new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", props.webAclProps).webAclArn;
        // Cloudfront Distribution
        const defaultRootObject = distributionProps?.defaultRootObject ?? "index.html";
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            ...distributionProps,
            webAclId: webAclArn,
            defaultBehavior: {
                ...distributionProps?.defaultBehavior,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            defaultRootObject,
            errorResponses: distributionProps?.errorResponses ?? [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
                {
                    httpStatus: 403,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
            ],
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(props.runtimeOptions?.jsonFileName ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, props.runtimeOptions?.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
        });
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsite", version: "0.2.15" };
/**
 * If passing in distributionProps, the default behaviour.origin is a required parameter. An instance of this class can be passed in
 * to make the compiler happy.
 */
class StaticWebsiteOrigin {
    bind(_scope, _options) {
        throw new Error("This should never be called");
    }
}
exports.StaticWebsiteOrigin = StaticWebsiteOrigin;
_b = JSII_RTTI_SYMBOL_1;
StaticWebsiteOrigin[_b] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsiteOrigin", version: "0.2.15" };
//# sourceMappingURL=data:application/json;base64,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