"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IamRoles = void 0;
const cdk = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
/**
 * Create all required IAM roles
 */
class IamRoles extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const urlsuffix = cdk.Stack.of(this).urlSuffix;
        const stack = cdk.Stack.of(this);
        const region = cdk.Stack.of(this).region;
        // ComputeNodeIAMRole
        const computeNodeIamRole = new iam.Role(this, 'ComputeNodeIamRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal(`ec2.${urlsuffix}`), new iam.ServicePrincipal(`ssm.${urlsuffix}`)),
        });
        computeNodeIamRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // PolicyName: ComputeNodePermissions
        const computeNodePermissionsPolicy = new iam.Policy(this, 'ComputeNodePermissionsPolicy');
        computeNodePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                's3:GetObject',
                's3:ListBucket',
                's3:PutObject',
            ],
            resources: [
                // S3InstallBucket/*
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: `${props.s3InstallBucketName}/*`,
                }),
                // S3InstallBucket
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: `${props.s3InstallBucketName}`,
                }),
            ],
        }));
        computeNodePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                's3:GetObject',
                's3:ListBucket',
            ],
            resources: [
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: `dcv-license.${region}/*`,
                }),
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: 'ec2-linux-nvidia-drivers/*',
                }),
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: 'ec2-linux-nvidia-drivers',
                }),
            ],
        }));
        computeNodePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: ['ses:SendEmail'],
            resources: [
                stack.formatArn({ service: 'ses', resource: 'identity' }),
            ],
        }));
        computeNodePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: ['ec2:CreateTags'],
            resources: [
                stack.formatArn({ service: 'ec2', resource: 'volume' }),
                stack.formatArn({ service: 'ec2', resource: 'network-interface' }),
            ],
        }));
        computeNodePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeVolumes',
                'ec2:DescribeNetworkInterfaces',
                'fsx:CreateDataRepositoryTask',
                'fsx:DescribeFileSystems',
                'tag:GetResources',
                'tag:GetTagValues',
                'tag:GetTagKeys',
            ],
            resources: ['*'],
        }));
        // attach to the role
        computeNodePermissionsPolicy.attachToRole(computeNodeIamRole);
        this.computeNodeIamRole = computeNodeIamRole;
        // ComputeNodeInstanceProfile
        this.computeNodeInstanceProfileName = new iam.CfnInstanceProfile(this, 'ComputeNodeInstanceProfile', {
            roles: [this.computeNodeIamRole.roleName],
        }).ref;
        // SpotFleetIAMRole
        const spotFleetIAMRole = new iam.Role(this, 'SpotFleetIAMRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal(`spotfleet.${urlsuffix}`)),
        });
        // PolicyName: SpotFleetPermissions
        const spotFleetPermissionsPolicy = new iam.Policy(this, 'SpotFleetPermissionsPolicy');
        spotFleetPermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeImages',
                'ec2:DescribeSubnets',
                'ec2:DescribeInstanceStatus',
            ],
            resources: ['*'],
        }));
        spotFleetPermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'ec2:RequestSpotInstances',
                'ec2:TerminateInstances',
                'ec2:CreateTags',
            ],
            resources: [
                stack.formatArn({ service: 'ec2', resource: 'instance' }),
            ],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({ service: 'ec2', resource: 'vpc', resourceName: props.network.vpc.vpcId }),
                },
            },
        }));
        spotFleetPermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'iam:PassRole',
            ],
            resources: [
                computeNodeIamRole.roleArn,
            ],
            conditions: {
                StringEquals: {
                    'iam:PassedToService': [
                        'ec2.amazonaws.com',
                        'ec2.amazonaws.com.cn',
                    ],
                },
            },
        }));
        spotFleetPermissionsPolicy.attachToRole(spotFleetIAMRole);
        // SchedulerIAMRole
        const schedulerIAMRole = new iam.Role(this, 'SchedulerIAMRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal(`ec2.${urlsuffix}`), new iam.ServicePrincipal(`ssm.${urlsuffix}`)),
        });
        schedulerIAMRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // PolicyName: SchedulerReadPermissions
        const schedulerReadPermissionsPolicy = new iam.Policy(this, 'SchedulerReadPermissionsPolicy');
        schedulerReadPermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'pricing:GetProducts',
                'budgets:ViewBudget',
                'ec2:DescribeInstances',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
                'ec2:DescribeImages',
                'ec2:DescribeInstanceAttribute',
                'ec2:DescribeInstanceTypes',
                'ec2:DescribeReservedInstances',
                'ec2:DescribeSpotInstanceRequests',
                'ec2:DescribeVpcClassicLink',
                'ec2:DescribeVolumes',
                'ec2:DescribePlacementGroups',
                'ec2:DescribeKeyPairs',
                'ec2:DescribeLaunchTemplates',
                'ec2:DescribeLaunchTemplateVersions',
                'ec2:DescribeNetworkInterfaces',
                'ec2:DescribeSpotFleetRequests',
                'ec2:DescribeSpotFleetInstances',
                'fsx:DescribeFileSystems',
                'autoscaling:DescribeAutoScalingGroups',
                'autoscaling:DescribeScalingActivities',
                'autoscaling:DescribeLaunchConfigurations',
                'elasticloadbalancing:DescribeRules',
                'elasticloadbalancing:DescribeListeners',
                'elasticloadbalancing:DescribeTargetGroups',
                'savingsplans:DescribeSavingsPlans',
                'servicequotas:ListServiceQuotas',
            ],
            resources: ['*'],
        }));
        // PolicyName: SchedulerWritePermissions
        const schedulerWritePermissionsPolicy = new iam.Policy(this, 'SchedulerWritePermissionsPolicy');
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
                'autoscaling:DeleteAutoScalingGroup',
                'autoscaling:CreateAutoScalingGroup',
                'autoscaling:DetachInstances',
                'ec2:DeleteLaunchTemplate',
                'ec2:CreateLaunchTemplate',
                'fsx:CreateDataRepositoryTask',
            ],
            resources: ['*'],
            conditions: {
                StringLikeIfExists: {
                    'autoscaling:LaunchConfigurationName': props.network.clusterId,
                },
            },
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'ec2:CreateTags',
            ],
            resources: [
                stack.formatArn({ service: 'ec2', resource: 'volume' }),
                stack.formatArn({ service: 'ec2', resource: 'network-interface' }),
                stack.formatArn({ service: 'ec2', resource: 'instance' }),
            ],
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'cloudformation:CreateStack',
                'cloudformation:DeleteStack',
                'cloudformation:DescribeStacks',
            ],
            resources: ['*'],
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'ec2:RunInstances',
                'ec2:TerminateInstances',
                'ec2:CreatePlacementGroup',
                'ec2:DeletePlacementGroup',
            ],
            resources: [
                stack.formatArn({ service: 'ec2', resource: 'subnet' }),
                stack.formatArn({ service: 'ec2', resource: 'key-pair' }),
                stack.formatArn({ service: 'ec2', resource: 'instance' }),
                stack.formatArn({ service: 'ec2', resource: 'snapshot', account: '' }),
                stack.formatArn({ service: 'ec2', resource: 'launch-template' }),
                stack.formatArn({ service: 'ec2', resource: 'volume' }),
                stack.formatArn({ service: 'ec2', resource: 'security-group' }),
                stack.formatArn({ service: 'ec2', resource: 'placement-group' }),
                stack.formatArn({ service: 'ec2', resource: 'network-interface' }),
                stack.formatArn({ service: 'ec2', resource: 'image', account: '' }),
            ],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({ service: 'ec2', resource: 'vpc', resourceName: props.network.vpc.vpcId }),
                },
            },
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [
                stack.formatArn({
                    service: 'lambda',
                    resource: 'function',
                    resourceName: `${props.network.clusterId}-Metrics`,
                }),
            ],
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: ['fsx:CreateFileSystem'],
            resources: [
                stack.formatArn({
                    service: 'fsx',
                    resource: 'file-system',
                }),
            ],
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: ['fsx:DeleteFileSystem'],
            resources: [
                stack.formatArn({
                    service: 'fsx',
                    resource: 'file-system',
                }),
            ],
            conditions: {
                StringLike: {
                    'aws:ResourceTag/soca:ClusterId': props.network.clusterId,
                },
            },
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'iam:CreateServiceLinkedRole',
                'iam:AttachRolePolicy',
                'iam:PutRolePolicy',
            ],
            resources: [
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/s3.data-source.lustre.fsx.amazonaws.com',
                }),
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/autoscaling.amazonaws.com',
                }),
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/spotfleet.amazonaws.com',
                }),
            ],
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'ses:SendEmail',
            ],
            resources: [
                stack.formatArn({
                    service: 'ses',
                    resource: 'identity',
                }),
            ],
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'ec2:CreatePlacementGroup',
                'ec2:DeletePlacementGroup',
                'ec2:RequestSpotFleet',
                'ec2:ModifySpotFleetRequest',
                'ec2:CancelSpotFleetRequests',
            ],
            resources: ['*'],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({ service: 'ec2', resource: 'vpc', resourceName: props.network.vpc.vpcId }),
                },
            },
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                's3:GetObject',
                's3:ListBucket',
                's3:PutObject',
            ],
            resources: [
                // S3InstallBucket/*
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: `${props.s3InstallBucketName}/*`,
                }),
                // S3InstallBucket
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: `${props.s3InstallBucketName}`,
                }),
            ],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({ service: 'ec2', resource: 'vpc', resourceName: props.network.vpc.vpcId }),
                },
            },
        }));
        schedulerWritePermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'iam:PassRole',
                'iam:CreateServiceLinkedRole',
            ],
            resources: [
                computeNodeIamRole.roleArn,
                spotFleetIAMRole.roleArn,
            ],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({ service: 'ec2', resource: 'vpc', resourceName: props.network.vpc.vpcId }),
                },
            },
        }));
        // attach to the role
        schedulerWritePermissionsPolicy.attachToRole(schedulerIAMRole);
        this.schedulerIAMRole = schedulerIAMRole;
        // SchedulerIAMInstanceProfile
        this.schedulerIamInstanceProfileName = new iam.CfnInstanceProfile(this, 'SchedulerIamInstanceProfileName', {
            roles: [this.schedulerIAMRole.roleName],
        }).ref;
        // LambdaSolutionMetricRole
        const lambdaSolutionMetricRole = new iam.Role(this, 'LambdaSolutionMetricRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal(`lambda.${urlsuffix}`)),
        });
        // PolicyName: SolutionMetric
        const solutionMetricPolicy = new iam.Policy(this, 'SolutionMetricPolicy');
        spotFleetPermissionsPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
            ],
            resources: [
                stack.formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    resourceName: `/aws/lambda/${props.network.clusterId}*`,
                }),
            ],
        }));
        solutionMetricPolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DeleteLogStream',
            ],
            resources: [
                stack.formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    resourceName: `/aws/lambda/${props.network.clusterId}*:log-stream:*`,
                }),
            ],
        }));
        solutionMetricPolicy.attachToRole(lambdaSolutionMetricRole);
        // LambdaACMIAMRole
        const lambdaACMIAMRole = new iam.Role(this, 'LambdaACMIAMRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal(`lambda.${urlsuffix}`)),
        });
        // PolicyName: ${clusterId}-LambdaACMIamRole-Policy
        const lambdaACMIamRolePolicy = new iam.Policy(this, 'LambdaACMIamRolePolicy');
        lambdaACMIamRolePolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
            ],
            resources: [
                stack.formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    resourceName: `/aws/lambda/${props.network.clusterId}*`,
                }),
            ],
        }));
        lambdaACMIamRolePolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                stack.formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    resourceName: `/aws/lambda/${props.network.clusterId}*:log-stream:*`,
                }),
            ],
        }));
        lambdaACMIamRolePolicy.addStatements(new iam.PolicyStatement({
            actions: [
                'acm:ImportCertificate',
                'acm:ListCertificates',
                'acm:AddTagsToCertificate',
            ],
            resources: ['*'],
        }));
        lambdaACMIamRolePolicy.attachToRole(lambdaACMIAMRole);
    }
}
exports.IamRoles = IamRoles;
//# sourceMappingURL=data:application/json;base64,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