# Client for AICosmos

This package implements the client for AICosmos. Before using this package, please make sure that you have a valid account for AICosmos. 

## Login & Create Session
By using this client, you can chat with our backend in "base" mode. To login, you will need the server's address and your API key. Our conversation system is based on sessions, you can either start a new session, or continue with an existing one.

```Python
from aicosmos_client.client import AICosmosClient 

# Login
client = AICosmosClient(
    base_url="https://aicosmos.ai/api", api_key="xxx"
)

# Create a new session
try:
    new_session_id = client.create_session()
except Exception as e:
    print(f"Error creating new session: {e}")
    exit(0)

# Check all existing sessions
try:
    my_sessions = client.get_my_sessions()
except Exception as e:
    print(f"Error getting my sessions: {e}")
    exit(0)
# [{"session_id", "title"}, ...]
print(my_sessions)
```

## Chatting
Our framework is different from "chat completions", where you give an llm the conversation history. Instead, your conversation history, along with other tool execution results, are stored in our database. This gives you a clean and simple interface to use, without worrying about constructing complicated contexts. The following code starts a new conversation in "base" mode. We support both stream and non-stream responses.

### Non-Stream Response
In this case, you only need to submit your prompt. The whole conversation history is returned.

```Python
# prompt = "Hello"
try:
    conversation_history = client.chat(session_id, prompt)
except Exception as e:
    print(f"Error chatting: {e}")
    exit(0)
print(conversation_history)
```

### Stream Response
In this scenario, our server only returns the **diff** between current history and old history. There are two types of diffs:
- added: a new item is added into conversation history
- updated: certain fields of existing history has changed

The diffs are very intuitive. You can track the conversation history dinamically using them:

```Python
# get existing conversation history
tracked_history = client.get_session_history(session)

for diff in client.chat_stream(
    session_id,
    prompt,
):
    # add new items
    for add in diff["added"]:
        index = add["index"]
        item = add["item"]
        tracked_history.append(item)
    # update existing items
    for update in diff["updated"]:
        index = update["index"]
        field = update["field"]
        # new_suffix is a string
        new_suffix = update["new_suffix"]
        if field in tracked_history[index].keys():
            tracked_history[index][field] += new_suffix
        else:
            tracked_history[index] = new_suffix

# now `tracked_history` is up to date!
```

### File Upload & Download
To upload a file to current session, just provide session id and local file path:

```Python
client.upload_file(session_id, "./my_file.txt")
```

To download an uploaded file, just provide session id, filename and where you want to save it.

```Python
client.download_file(session_id, "my_file.txt", "./my_local_path.txt")
```

## Other Modes
Apart from "base" mode, we also support "code" mode and "lean" mode. In these modes, we provide you with a web code editor, which you can open and interact with your web browser.

```Python
try:
    conversation_history = client.chat(
        new_session_id, "De Morgan's law. Start now.", mode="lean"
    )
except Exception as e:
    print(f"Error chatting: {e}")
    exit(0)
print(conversation_history)

# open this link with your web browser (e.g. Edge, Chrome)
url = client.get_browser_url(new_session_id)
print(url)
```
