import os
import json
import requests


class AICosmosClient:
    def __init__(
        self,
        base_url: str,
        api_key: str,
    ):
        self.session = requests.Session()
        self.base_url = base_url
        self.api_key: str = api_key
        self.access_token: str = None

        self._login()

    def _login(self):
        login_data = {"api_key": self.api_key}
        login_url = f"{self.base_url}/user/api_login"
        headers = {"Content-Type": "application/x-www-form-urlencoded"}
        try:
            response = self.session.post(
                url=login_url, data=login_data, headers=headers
            )
            if response.status_code == 200:
                token_data = response.json()
                self.access_token = token_data["access_token"]
                return
            else:
                raise ValueError(f"Status code: {response.status_code}")
        except Exception as e:
            raise ValueError(f"Error: {e}")

    def _get_auth_headers(self):
        if not self.access_token:
            raise ValueError("Not logged in")
        return {
            "Authorization": f"Bearer {self.access_token}",
        }

    def create_session(self):
        if not self.access_token:
            raise ValueError("Not logged in")
        try:
            response = self.session.post(
                f"{self.base_url}/sessions/create", headers=self._get_auth_headers()
            )
            if response.status_code == 200:
                response_json = response.json()
                return response_json["session_id"]
            else:
                raise ValueError(f"Status code: {response.status_code}")
        except Exception as e:
            raise ValueError(f"Error: {e}")

    def delete_session(self, session_id: str):
        if not self.access_token:
            raise ValueError("Not logged in")
        try:
            response = self.session.delete(
                f"{self.base_url}/sessions/{session_id}",
                headers=self._get_auth_headers(),
            )
            if response.status_code == 200:
                return
            else:
                raise ValueError(f"Status code: {response.status_code}")
        except Exception as e:
            raise ValueError(f"Error: {e}")

    def get_my_sessions(self):
        """
        Retrieves session ids and titles.
        """
        if not self.access_token:
            raise ValueError("Not logged in")
        try:
            response = self.session.get(
                f"{self.base_url}/sessions/my_sessions",
                headers=self._get_auth_headers(),
            )
            if response.status_code == 200:
                sessions = response.json()
                self.active_sessions = sessions
                return [
                    {
                        "session_id": session["session_id"],
                        "title": session["environment_info"].get("title", None),
                    }
                    for session in sessions
                ]
            else:
                raise ValueError(f"Status code: {response.status_code}")
        except Exception as e:
            raise ValueError(f"Error: {e}")

    def get_session_history(self, session_id: str):
        """
        Retrieves the conversation history of the given session
        """
        try:
            response = self.session.get(
                f"{self.base_url}/sessions/{session_id}/history",
                headers=self._get_auth_headers(),
            )
            success = response.status_code == 200
            if success:
                return response.json()
            else:
                raise ValueError(f"Status code: {response.status_code}")
        except Exception as e:
            raise ValueError(f"Error: {e}")

    def chat(self, session_id: str, prompt: str, mode: str = "base"):
        """
        Returns conversation history. Available modes: base, code, lean
        """
        if not self.access_token:
            raise ValueError("Not logged in")
        data = {
            "user_input": prompt,
            "session_id": session_id,
            "mode": mode,
        }
        try:
            response = self.session.post(
                f"{self.base_url}/chat",
                json=data,
                headers=self._get_auth_headers(),
            )
            success = response.status_code == 200
            if success:
                return response.json()
            else:
                raise ValueError(f"Status code: {response.status_code}")
        except Exception as e:
            raise ValueError(f"Error: {e}")

    def chat_stream(self, session_id: str, prompt: str, mode: str = "base"):
        """
        Returns diff of conversation history. Available modes: base, code, lean
        """
        if not self.access_token:
            raise ValueError("Not logged in")

        data = {
            "user_input": prompt,
            "session_id": session_id,
            "mode": mode,
        }
        try:
            response = self.session.post(
                f"{self.base_url}/chat/stream",
                json=data,
                headers=self._get_auth_headers(),
                stream=True,
            )
            success = response.status_code == 200
            if success:
                for chunk in response.iter_content(chunk_size=None):
                    yield json.loads(chunk.decode())
            else:
                raise ValueError(f"Status code: {response.status_code}")
        except Exception as e:
            raise ValueError(f"Error: {e}")

    def get_browser_url(self, session_id: str):
        """
        Retrieves the url that can be opened in a browser.
        """
        try:
            response = self.session.get(
                f"{self.base_url}/sessions/{session_id}/status",
                headers=self._get_auth_headers(),
            )
            success = response.status_code == 200
            if success:
                return response.json().get("browser_url", None)
            else:
                raise ValueError(f"Status code: {response.status_code}")
        except Exception as e:
            raise ValueError(f"Error: {e}")

    def upload_file(self, session_id: str, file_path: str):
        """
        Read and upload a file to the server.
        """
        if not os.path.exists(file_path):
            raise ValueError(f"Error: {file_path} does not exist")

        try:
            with open(file_path, "rb") as file:
                filename = os.path.basename(file_path)
                files = {"file_upload": (filename, file, "text/plain")}
                data = {"mode": "base"}
                response = self.session.post(
                    f"{self.base_url}/sessions/{session_id}/files",
                    headers=self._get_auth_headers(),
                    files=files,
                    data=data,
                )
                success = response.status_code == 200
                if success:
                    return True
                else:
                    raise ValueError(
                        f"Status code: {response.status_code}. Error: {response.json()}"
                    )
        except Exception as e:
            raise ValueError(f"Error: {e}")

    def download_file(self, session_id: str, file_name: str, save_path: str):
        """
        Download an uploaded file and save it locally.
        """
        try:
            response = self.session.get(
                f"{self.base_url}/files/{session_id}/{file_name}",
                headers=self._get_auth_headers(),
            )
            success = response.status_code == 200
            if success:
                with open(save_path, "wb") as file:
                    file.write(response.content)
                return True
            else:
                raise ValueError(
                    f"Status code: {response.status_code}. Error: {response.json()}"
                )
        except Exception as e:
            raise ValueError(f"Error: {e}")
