# -*- coding: UTF-8 -*-
# Copyright 2023 Rumma & Ko Ltd
# License: GNU Affero General Public License v3 (see file COPYING for details)

import asyncio
import os
import pickle
import signal
import socket
import sys
import time

from django.conf import settings
from django.core.management import execute_from_command_line as django_execute_from_command_line
from pathlib import Path
from subprocess import Popen, PIPE, DEVNULL

from lino.utils.socks import send_through_socket, get_from_socket, unlink

sock = None

try:
    worker_sock_file = settings.SITE.site_dir / 'worker_sock'
    log_sock_file = settings.SITE.site_dir / 'log_sock'
except:
    worker_sock_file = None
    log_sock_file = None

DEFAULT_TIMEOUT = 1

SKIP_COMMANDS = [
    'runworker',
    'initiate_linod',
    'install',
    'dump_settings'
]

SKIP_FLAGS = [
    '--help',
    '-h'
]

SKIP_ARG = SKIP_COMMANDS + SKIP_FLAGS


def get_pid() -> bytes:
    return str(os.getpid()).encode()


def get_socket() -> socket.socket:
    sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
    sock.connect(str(worker_sock_file))
    return sock


STDIO = dict(stdout=PIPE, stderr=PIPE, stdin=DEVNULL)


def check_and_start_linod(argv=None):
    if argv:
        for c in SKIP_ARG:
            if c in argv:
                return

    def add_client():
        global sock
        sock = get_socket()
        send_through_socket(sock, b'add')
        send_through_socket(sock, get_pid())
        sock.close()

    if log_sock_file.exists():
        if worker_sock_file.exists():
            add_client()
        return

    sys.stdout.write("Run worker process ...\n")
    from lino.modlib.linod.utils import LINOD
    from channels.layers import get_channel_layer
    p = Popen(['python', 'manage.py', 'runworker', LINOD], start_new_session=True, **STDIO)
    channel_layer = get_channel_layer()

    async def start_worker_processes():
        await asyncio.sleep(DEFAULT_TIMEOUT)
        await channel_layer.send(LINOD, {'type': 'log.server'})
        await asyncio.sleep(DEFAULT_TIMEOUT)
        await channel_layer.send(LINOD, {'type': 'dev.worker'})
        await asyncio.sleep(DEFAULT_TIMEOUT)
        await channel_layer.send(LINOD, {'type': 'run.schedule'})
        await asyncio.sleep(DEFAULT_TIMEOUT)

    loop = asyncio.get_event_loop()
    loop.run_until_complete(start_worker_processes())

    def try_add_client(count=0):
        try:
            add_client()
        except (FileNotFoundError, ConnectionRefusedError):
            if count < 5:
                time.sleep(DEFAULT_TIMEOUT)
                sys.stdout.write(f'Worker not up yet. Waited for worker process: {count} seconds\n')
                try_add_client(count + 1)
            else:
                out, err = p.communicate(timeout=DEFAULT_TIMEOUT)
                p.kill()
                sys.stderr.write("\nWorker process failed\n\n")
                sys.stdout.write(out.decode() + '\n\n')
                sys.stderr.write(err.decode() + '\n')
                sys.exit()

    try_add_client()


def check_and_start_utility(argv=None):
    if settings.SITE.use_linod:
        check_and_start_linod(argv)


def stop_utility(argv=None):
    global sock
    if sock is not None:
        sock = get_socket()
        send_through_socket(sock, b'exists')
        send_through_socket(sock, get_pid())
        if get_from_socket(sock) == b'true':
            send_through_socket(sock, b'remove_get')
            send_through_socket(sock, get_pid())
            data = pickle.loads(get_from_socket(sock))
            sock.close()
            if data['clients'] == 0:
                sys.stdout.write('Terminate worker process ...\n')
                os.kill(data['pid'], signal.SIGKILL)
                unlink(str(worker_sock_file))
                unlink(str(log_sock_file))
        else:
            send_through_socket(sock, b'close')
            sock.close()


def execute_from_command_line(argv=None):
    check_and_start_utility(argv)
    try:
        django_execute_from_command_line(argv)
    finally:
        stop_utility(argv)
