# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['lk_qtquick_scaffold',
 'lk_qtquick_scaffold.experimental_features',
 'lk_qtquick_scaffold.pyside',
 'lk_qtquick_scaffold.qmlside',
 'lk_qtquick_scaffold.qmlside.hot_reloader',
 'lk_qtquick_scaffold.qmlside.layout_helper',
 'lk_qtquick_scaffold.qmlside.resource_manager',
 'lk_qtquick_scaffold.qmlside.widgets_backend',
 'lk_qtquick_scaffold.qt_core',
 'lk_qtquick_scaffold.style']

package_data = \
{'': ['*'],
 'lk_qtquick_scaffold': ['themes/*',
                         'themes/Theme/*',
                         'themes/Theme/LightClean/*',
                         'themes/Theme/LightClean/LCBackground/*',
                         'themes/Theme/LightClean/LCButtons/*',
                         'themes/Theme/LightClean/LCComplex/*',
                         'themes/Theme/LightClean/LCStyle/*',
                         'themes/Theme/LightClean/rss/*',
                         'themes/Theme/SimpleUI/*',
                         'widgets/LKWidgets/*',
                         'widgets/LKWidgets/LKButtons/*',
                         'widgets/LKWidgets/LKButtons/.assets/*',
                         'widgets/LKWidgets/LKSidebar/*'],
 'lk_qtquick_scaffold.style': ['stylesheet/*']}

install_requires = \
['lk-lambdex', 'lk-logger', 'lk-utils', 'qtpy']

extras_require = \
{'qt5': ['pyside2'], 'qt6': ['pyside6']}

setup_kwargs = {
    'name': 'lk-qtquick-scaffold',
    'version': '2.0.0a1',
    'description': 'A flexible toolset to improve QML coding experience for PyQt/PySide development.',
    'long_description': '# LK QtQuick Scaffold\n\nUsing Python and QtQuick QML to build desktop applications from a series of\npredefined tools.\n\n# Highlights\n\n- A layout engine to extend QML layouts.\n- Integrate Qt logging with python console.\n- Execute Python snippet in QML, and vice versa.\n- Easy-to-use register handler to register Python functions to QML side.\n- Well type-annotated signal and slot (and more elegant writing style!)\n- Hot loader to verify layout changing on the fly.\n- Stylesheet manager to thoroughly control application appearance (color, size, \n  motion, typography, and so on).\n- A set of built-in widgets/themes to quickly produce beautiful user interface.\n\n# How to install\n\n`lk-qtquick-scaffold` requires Python 3.8+ interpreter version.\n\nUse pip install:\n\n```shell\n# the 1.x version. (1.3.0)\npip install lk-qtquick-scaffold\n\n# the next big version is coming soon. currently i\'ve released a preview version.\npip install lk-qtquick-scaffold>=2.0.0a0\n```\n\nNote: many features in this document are based on 2.0, currently the 2.0 formal\nrelease is still in progress, you may install the alpha version to taste the\nnewest features.\n\n## Install Qt backend\n\nInstalling lk-qtquick-scaffold doesn\'t include Python for Qt\'s library. You \nneed to manually install one of the follows:\n\n```shell\n# choose one to install\npip install pyside6\npip install pyqt6\npip install pyside2\npip install pyqt5\n```\n\nlk-qtquick-scaffold auto detects the Qt backend you\'ve installed (you can also \nexplicitly set the specific one), it uses \n[qtpy](https://github.com/spyder-ide/qtpy) to provide an uniform layer \noverrides PySide6/PyQt6/PySide2/PyQt5.\n\n# Examples quick through\n\n## Hello world\n\nview.qml\n\n```qml\nimport QtQuick\nimport QtQuick.Window\n\nWindow {\n    visible: true\n    width: 400\n    height: 300\n    \n    Text {\n        anchors.centerIn: parent\n        text: \'Hello world!\'\n    }\n}\n```\n\nmain.py\n\n```python\nfrom lk_qtquick_scaffold import app\napp.run(\'view.qml\')\n```\n\n![](examples/hello_world/screenshot.png)\n\n## Hot loader\n\nThe `app.run` method accepts `debug` (bool type) parameter, to enable hot \nloader mode:\n\n```python\nfrom lk_qtquick_scaffold import app\napp.run(\'view.qml\', debug=True)\n```\n\nIt starts a floating window that includes a button "RELOAD", each time when you \nmodify "view.qml", click "RELOAD" to refresh your GUI:\n\n![](examples/hot_reloader/screenshot.gif)\n\nBTW you can run "view.qml" in command line:\n\n```shell\n# see help\npy -m lk_qtquick_scaffold -h\n\n# run\npy -m lk_qtquick_scaffold view.qml\n\n# run in debug mode\npy -m lk_qtquick_scaffold view.qml --debug\n```\n\nIt has the same result like above "main.py" does.\n\n## Register funtions to QML\n\n```python\nfrom lk_qtquick_scaffold import QObject, app, pyside, slot\n\nclass MyObject(QObject):\n    @slot(result=str)\n    def hello(self):\n        return \'hello world\'\n\n# 1. register QObject subclasses by `app.register`\napp.register(MyObject())  \n#   it will be available as \'MyObject\' in QML side.\n# 1.1. or use alias\napp.register(MyObject(), name=\'PyObject\')\n#   it will be available as \'PyObject\' in QML side.\n\n# 2. register regular function by `pyside.register`.\ndef foo(a: int, b: int, c: int):\n    return a + b + c\npyside.register(foo)\n#   it will be available as \'pyside.call("foo", ...)\' in QML side.\n# 2.1. or use alias\npyside.register(foo, name=\'add_abc\')\n#   it will be available as \'pyside.call("add_abc", ...)\' in QML side.\n```\n\nview.qml\n\n```qml\nimport QtQuick\n\nItem {\n    Component.onCompleted: {\n        console.log(MyObject.hello())  // -> \'hello world\'\n        console.log(PyObject.hello())  // -> \'hello world\'\n\n        console.log(pyside.call("foo", [1, 2, 3]))  // -> 6\n        console.log(pyside.call("add_abc", [1, 2, 3]))  // -> 6\n    }\n}\n```\n\n## Integrate qt logging in python console\n\nWhen you use `console.log` in QML side, it will be printed in Python console:\n\n![](examples/console_print/screenshot.png)\n\n## Signal and Slot\n\nThe `signal` and `slot` wrap on Qt\'s `Signal` and `Slot` decorators, but\nextended their functionalities:\n\n1.  You can get the correct type hint in IDE:\n\n    ![](examples/signal_slot/screenshot_1.png)\n\n2.  The `slot` accepts more types as alias to "QObject" and "QVariant" -- it is\n    more convenient and more readable:\n\n    ```python\n    from lk_qtquick_scaffold import QObject, slot\n    \n    class MyObject(QObject):\n\n        @slot(int, dict, result=list)  # <- here\n        def foo(self, index, data):\n            return [index, len(data)]\n    \n    \'\'\'\n    it is more readable than:\n        @Slot(int, QJSValue, result=\'QVariant\')\n        def foo(self, index, data):\n            return [index, len(data)]\n    \'\'\'\n    ```\n\n    Here is a full alias list (which is documented in \n    `lk_qtquick_scaffold/qt_core/signal_slot.py`):\n\n    **slot(\\*args)**\n\n    | Alias         | Real value    | Note              |\n    | ------------- | ------------- |------------------ |\n    | `bool`        | `bool`        | basic type        |\n    | `float`       | `float`       | basic type        |\n    | `int`         | `int`         | basic type        |\n    | `str`         | `str`         | basic type        |\n    | `QObject`     | `QObject`     | object            |\n    | `object`      | `QObject`     | object            |\n    | `\'item\'`      | `QObject`     | object (string)   |\n    | `\'object\'`    | `QObject`     | object (string)   |\n    | `\'qobject\'`   | `QObject`     | object (string)   |\n    | `dict`        | `QJSValue`    | qjsvalue          |\n    | `list`        | `QJSValue`    | qjsvalue          |\n    | `set`         | `QJSValue`    | qjsvalue          |\n    | `tuple`       | `QJSValue`    | qjsvalue          |\n    | `...`         | `QJSValue`    | qjsvalue          |\n    | `\'any\'`       | `QJSValue`    | qjsvalue (string) |\n\n    **slot(result=...)**\n\n    | Alias     | Real value    | Note          |\n    | --------- | ------------- |-------------- |\n    | `None`    | `None`        | basic type    |\n    | `bool`    | `bool`        | basic type    |\n    | `float`   | `float`       | basic type    |\n    | `int`     | `int`         | basic type    |\n    | `str`     | `str`         | basic type    |\n    | `dict`    | `\'QVariant\'`  | qvariant      |\n    | `list`    | `\'QVariant\'`  | qvariant      |\n    | `set`     | `\'QVariant\'`  | qvariant      |\n    | `tuple`   | `\'QVariant\'`  | qvariant      |\n    | `...`     | `\'QVariant\'`  | qvariant      |\n\n3.  `slot` decorator is non-intrusive -- it means the method been decorated can \n    be called in Python side as usual.\n\n    ```python\n    from lk_qtquick_scaffold import QObject, slot\n\n    class MyObject(QObject):\n        @slot(int, str, result=list)\n        def foo(self, index, name):\n            return [index, name]\n\n    my_obj = MyObject()\n    # you can call it like a regular method! (just \'ignore\' its docorator.)\n    my_obj.foo(1, \'hello\')  # -> [1, \'hello\']\n    ```\n\n## Built-in widgets library\n\n`lk-qtquick-scaffold` provides a set of built-in widgets under its `~/widgets` \ndirectory. \n\nBasically, you can use it in QML by importing "LKWidgets" (or "LKWidgets 1.0" \nfor Qt 5.x):\n\n```qml\nimport LKWidgets\n\nLKWindow {\n    color: \'#DBDBF7\'  // moon white\n\n    LKRectangle {\n        anchors.fill: parent\n        anchors.margins: 32\n        color: \'#ECDEC8\'  // parchment yellow\n\n        LKColumn {\n            anchors.centerIn: parent\n            alignment: \'hcenter\'  // horizontally center children\n\n            LKGhostButton {\n                text: \'SUNDAY\'\n            }\n\n            LKButton {\n                text: \'MONDAY\'\n            }\n\n            LKGhostButton {\n                text: \'TUESDAY\'\n            }\n\n            LKButton {\n                text: \'WEDNESDAY\'\n            }\n\n            LKGhostButton {\n                text: \'THURSDAY\'\n            }\n\n            LKButton {\n                text: \'FRIDAY\'\n            }\n\n            LKGhostButton {\n                text: \'SATURDAY\'\n            }\n        }\n    }\n}\n```\n\n![](examples/lk_widgets/screenshot_1.gif)\n\nThe dark theme:\n\n![](examples/lk_widgets/screenshot_2.png)\n\nMore screenshots: see `examples/lk_widgets/screenshot_*`.\n\nAll widget names are started with \'LK\', the full list is in\n`lk_qtquick_scaffold/widgets/LKWidgets/qmldir` file.\n\nNote: the widgets documentation is not ready. Currently you may have a look at \nthe `examples/lk_widgets` screenshots, or view its source code for more details.\n\n## High-level model, human-readable API\n\n*TODO*\n\n## Layout engine\n\nLayout engine is powered by `lk_qtquick_scaffold.qmlside.layout_helper`, which \nis registered as `pylayout` in QML side.\n\n```qml\n// some_view.qml\nimport QtQuick\n\nColumn {\n    height: 100\n    \n    Item { id: item1; height: 20  }\n    Item { id: item2; height: 0.4 }\n    Item { id: item3; height: 0   }\n    Item { id: item4; height: 0   }\n\n    Component.onCompleted: {\n        // horizontally center children\n        pylayout.auto_align(this, \'hcenter\')\n\n        // auto size children:\n        //  width > 1: as pixels\n        //  width > 0 and < 1: as percent of left spared space\n        //  width = 0: as stretch to fill the left spared space\n        pylayout.auto_size_children(this, \'vertical\')\n        //  the result is:\n        //      item1: 20px\n        //      item2: (100 - 20) * 0.4 = 32px\n        //      item3: (100 - 20 - 32) * 0.5 = 24px\n        //      item4: (100 - 20 - 32) * 0.5 = 24px\n        //          (item 3 and 4 share the left space equally.)\n    }\n}\n```\n\n## Executing Python snippet in QML, and vice versa\n\ntest.py\n\n```python\nfrom lk_qtquick_scaffold import eval_js\n\ndef foo(item1: QObject, item2: QObject):\n    eval_js(\'\'\'\n        $a.widthChanged.connect(() => {\n            $b.width = $a.width * 2\n        })\n    \'\'\', {\'a\': item1, \'b\': item2})\n```\n\nview.qml\n\n```qml\nimport QtQuick\n\nListView {\n    model: pyside.eval(`\n        import os\n        files = os.listdir(input(\'target folder: \'))\n        return files\n    `)\n}\n```\n\n## Style manager\n\n`lk-qtquick-scaffold` exposes a list of built-in style controlers to QML side\nas follows:\n\n| Style         | Description                                               |\n| ------------- | --------------------------------------------------------- |\n| `pycolor`     | All color specifications defined in a canonical name form |\n| `pyfont`      | Font related specifications                               |\n| `pysize`      | Width, height, radius, padding, margin, spacing, etc.     |\n| `pymotion` | Animation related specifications (duration, easing type, etc.) |\n\nUsage examples (seen in all LKWidgets):\n\n![](examples/lk_widgets/screenshot_3.png)\n\n![](examples/lk_widgets/screenshot_4.png)\n\nYou can overwrite the style by giving a YAML file to load, for example a \n"dark-theme.yaml":\n\n```yaml\n# this is dark theme color scheme\n\n# == general ==\n\nblue_1: \'#e4e5f8\'\nblue_3: \'#5294eb\'\nblue_5: \'#3844e6\'\nblue_7: \'#0f143b\'\ndark_1: \'#424141\'\ndark_2: \'#242529\'\ndark_3: \'#15141a\'\ndark_5: \'#050408\'\ngrey_3: \'#e8eaed\'\ngrey_5: \'#a9acb0\'\n\n# == widgets spec ==\n\nborder_active: \'#797171\'\nborder_default: \'#575757\'\nborder_glow: \'$border_active\'\nbutton_bg_active: \'$blue_5\'\nbutton_bg_default: \'$panel_bg\'\nbutton_bg_hovered: \'$dark_1\'\nbutton_bg_pressed: \'$dark_3\'\nbutton_bg_selected: \'$button_bg_pressed\'\ninput_bg_active: \'$dark_2\'\ninput_bg_default: \'$panel_bg\'\ninput_border_active: \'$border_active\'\ninput_border_default: \'$border_default\'\ninput_indicator_active: \'$blue_5\'\npanel_bg: \'$dark_3\'\nprog_bg: \'$blue_1\'\nprog_fg: \'$blue_5\'\nsidebar_bg: \'$panel_bg\'\ntext_default: \'$grey_3\'\ntext_disabled: \'$grey_5\'\ntext_hint: \'$grey_5\'\nwin_bg_default: \'$dark_5\'\n```\n\nThe dollar symbol (`$`) is a simple pointer to the other key. \n\nYou don\'t need to write all colors in the file, `lk-qtquick-scaffold` has a \ngreat deduction algorithm to automatically call back "defaults" when required\ncolors are missing from your sheet.\n\nFinally load it by calling `pycolor.update_from_file()`:\n\n```python\nfrom lk_qtquick_scaffold import pycolor\npycolor.update_from_file(\'dark-theme.yaml\')\n```\n\nWarning: currently color name style is under refactoring, it is very unstable \nto learn from its style.\n\n# Gallery\n\n![](gallery/widgets_demo/viscous-indicator-anim.gif)\n\n![](gallery/widgets_demo/swipe-view.gif)\n\n![](gallery/widgets_demo/breathing-circle-avatar.gif)\n\n[![](gallery/widgets_demo/password-eye-open.gif)](https://uimovement.com/media/resource_image/image_5213.gif.mp4)\n\nhttps://user-images.githubusercontent.com/27986259/180829198-7110831e-c060-436a-a9be-c41452f49932.mp4\n\nhttps://user-images.githubusercontent.com/27986259/180829267-cd497bcc-de38-4d00-bb19-c4a84b251031.mp4\n\n*TODO:AddMoreWidgetsDemo*\n',
    'author': 'Likianta',
    'author_email': 'likianta@foxmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/likianta/lk-qtquick-scaffold',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
