from mpl_toolkits.mplot3d import Axes3D
import sys

import matplotlib.pyplot as plt
from matplotlib.ticker import LinearLocator, FormatStrFormatter
import numpy as np
from numpy import exp, cos, sin, sqrt, linspace
from . import colormaps as cmaps
from .problem import Problem
from sklearn.model_selection import train_test_split
from random import uniform
from .individual import Individual
from abc import abstractmethod


class BenchmarkFunction(Problem):
    """
    The general class for standardized artap benchmarks:

     - contains the information from the global optimum of the function
     - bounds and the dimensionality of the problem
     - some references for the origin

    The functions should defined by the eval and the eval constraints functions, which mimics the structure of a
    standardized artap problem.

    """

    def __init__(self, **kwargs):

        super().__init__(**kwargs)

        self.dimension: int
        self.global_optimum: float
        self.global_optimum_coords: list
        self.robust_optimum:float
        self.robust_optimum_coords:list

    def generate_paramlist(self, dimension, lb, ub, **kwargs):
        """Defines an n-dimensional list for the optimization"""

        param_list = []

        for i in range(0, dimension):

            if 'dimension' in kwargs:
                dict = {'name': str(i), 'bounds': [lb, ub], 'initial_value': kwargs['initial_value']}
            else:
                dict = {'name': str(i), 'bounds': [lb, ub]}

            param_list.append(dict)

        return param_list

    def generate_objective_functions(self, criteria='minimize', **kwargs):
        """Defines an n-dimensional objective function"""

        objective_functions = []

        if 'm' in kwargs:
            for i in range(0,kwargs['m']):
                dict = {'name': 'f_' + str(i), 'criteria': criteria}
                objective_functions.append(dict)

        else:
            dict = {'name': 'f_1', 'criteria': criteria}
            objective_functions.append(dict)

        return objective_functions

    def plot_1d(self, px=0):
        """

        :param px:
        :return:
        """
        x = linspace(self.parameters[px]['bounds'][0], self.parameters[px]['bounds'][1], num=2000)
        y = np.zeros([2000])
        for i, xi in enumerate(x):
            y[i] = self.evaluate(Individual([xi]))[0]

        plt.figure()
        plt.plot(x, y, 'black', '--')
        plt.xlabel("x")
        plt.ylabel("f(x)")
        plt.grid(True)
        return

    @abstractmethod
    def pareto_front(self, x):
        pass

    def plot_2d(self, px=0, py=1, cz=0, f=0):
        """
        Makes a 3 dimensional surface plot from a 2 variable function.

        :param px: defines the examined parameter: x, the default value is 0
        :param py: defines the examined parameter: y, the default value is 1
        :param cz: in case of multi-objective test functions to select the optimal values

        :return:
        """
        fig = plt.figure()
        ax = fig.gca(projection='3d')

        # use the generator functions
        # TODO: make a simple algorithm which can solve this
        x = linspace(self.parameters[px]['bounds'][0],
                     self.parameters[px]['bounds'][1], num=1000)
        y = linspace(self.parameters[py]['bounds'][0],
                     self.parameters[py]['bounds'][1], num=1000)

        x, y = np.meshgrid(x, y)
        z = self.evaluate(Individual([x, y]))[f]

        # Plot the surface
        surf = ax.plot_surface(x, y, z, cmap=cmaps.viridis,
                               linewidth=0, antialiased=True)

        # Customize the z axis
        ax.zaxis.set_major_locator(LinearLocator(10))
        ax.zaxis.set_major_formatter(FormatStrFormatter('%.02f'))

        # Add a color bar which maps values to colors.
        fig.colorbar(surf, shrink=0.5, aspect=5)

        # plt.show()

    def set_init_values(self, **kwargs):
        if 'initial_value' in kwargs:

            for elem in self.parameters:
                elem.update({'initial_value': kwargs['initial_value']})

    def set_dimension(self, **kwargs):

        if 'dimension' in kwargs:
            self.dimension = kwargs['dimension']

    def get_data_1d(self, n=100, index_x=0, index_y=0):
        x = np.linspace(self.parameters[index_x]['bounds'][0], self.parameters[index_x]['bounds'][1], n)
        y = np.zeros([n])
        for i, xi in enumerate(x):
            y[i] = self.evaluate(Individual([xi]))[index_y]
        return[x, y]

    def get_data_2d(self, n=100, indices_x=[0, 1], index_y=0):
        x = linspace(self.parameters[indices_x[0]]['bounds'][0],
                     self.parameters[indices_x[0]]['bounds'][1], num=1000)
        y = linspace(self.parameters[indices_x[1]]['bounds'][0],
                     self.parameters[indices_x[1]]['bounds'][1], num=1000)

        x, y = np.meshgrid(x, y)
        z = self.evaluate(Individual([x, y]))[index_y]
        return[x, y, z]

class Rosenbrock(BenchmarkFunction):
    """
    .. math::
        f(x, y)=\sum_{i=1}^{n}[b (x_{i+1} - x_i^2)^ 2 + (a - x_i)^2]

    Complexity: difficult, unimodal function
    Implementation is based on [4]

    The unconstrained Rosenbrock function --also known as Rosenbrock's valley or Rosenbrock's banana function --
    The global minimum is inside a long, narrow, parabolic shaped flat valley. To find the valley is trivial
    f*(X) = f(X*) is at the point X* = (1,1). However to converge to hte global minimum is difficult.
    Algorithms that cannot find the good directions, underperforms this problem.

    General settings:
    n = [5,30] dimensions
    -5.0 <= xi <= 10.0

    Sources:
    [1] https://en.wikipedia.org/wiki/Rosenbrock_function
    [2] J. G. Digalakis and K. G. Margaritis, “On benchmarking functions for genetic algorithms,”
        Int. J. Comput. Math., vol. 77, no. 4, pp. 481–506, 2001, doi: 10.1080/00207160108805080.
    [3] K. Hussain, M. N. Mohd Salleh, S. Cheng, and R. Naseem, “Common Benchmark Functions
        for Metaheuristic Evaluation: A Review,” JOIV  Int. J. Informatics Vis., vol. 1, no. 4–2,
        p. 218, 2017, doi: 10.30630/joiv.1.4-2.65.
    [4] http://benchmarkfcns.xyz/benchmarkfcns/rosenbrockfcn.html
    """

    def set(self, **kwargs):
        self.name = 'Rosenbrock function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-5., ub=10.)

        self.global_optimum = 0.
        self.global_optimum_coords = [1.0 for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        """
        :param x: a two dimensional array/list/tuple, which contains the X[x,y]
        :return: f(X)
        """
        x = x.vector
        scores = 0
        for i in range(0, self.dimension - 1):
            a = 1. - x[i]
            b = x[i + 1] - x[i] ** 2.
            scores += (a * a + b * b * 100.0)
        return [scores]


class Ackley(BenchmarkFunction):
    """
    Ackley function is declared by the following formula [1]:

    .. math::
        f(\textbf{x}) = f(x_1, ..., x_n)= -a.exp(-b\sqrt{\frac{1}{n}\sum_{i=1}^{n}x_i^2})-exp(\frac{1}{n}\sum_{i=1}^{n}cos(cx_i))+ a + exp(1)


        -32 \leq x_i \leq 32$$

        \text{minimum at }f(0, \cdots, 0) = 0$$

    minimum: x = 0...0
             f(x) = 0

    [1] https://www.cs.unm.edu/~neal.holts/dga/benchmarkFunction/ackley.html
   """

    def set(self, **kwargs):
        self.name = 'Ackley function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-32., ub=32.)

        self.global_optimum = 0.
        self.global_optimum_coords = [0.0 for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        firstSum = 0.0
        secondSum = 0.0
        x = x.vector

        for c in x:
            firstSum += c ** 2.0
            secondSum += np.cos(2.0 * np.pi * c)
        n = float(len(x))
        return [-20.0 * np.exp(-0.2 * np.sqrt(firstSum / n)) - np.exp(secondSum / n) + 20.0 + np.e]


# class Ackley4Modified:
#     """
#     Ackley 4 or Modified Ackley Function (Continuous, Differentiable, Non-Separable, Scalable, Multimodal)
#
#     f(x)=\sum_{i=1}^{n-1}( e^{-0.2}\sqrt{x_i^2+x_{i+1}^2} + 3(cos(2x_i) + sin(2x_{i+1})))
#
#     Description:
#
#         - not convex
#         - defined on n-dimensional space
#         - non-separable
#         - differentiable
#         - continuous
#         - scalable
#
#     Search Domain
#
#         The function can be defined on any input domain but it is usually evaluated on xi∈[−35,35] for i=1,…,n
#
#     Solution
#
#     On the 2-dimensional space, the function has one global minima at f(x∗)=−4.590101633799122
#     located at x∗=(−1.51,−0.755)
#
#     """
#
#     @classmethod
#     def eval(cls, x):
#         dim = len(x)
#
#         value = 0.
#         for i in range(0, dim - 1):
#             value += exp(-0.2) * sqrt(x[i] ** 2. + x[i + 1] ** 2.) + 3. * (cos(2 * x[i]) + sin(2 * x[i + 1]))
#
#         return value

class Sphere(BenchmarkFunction):
    """
    Sphere function is declared by the following formula [1]:

    .. math::
        f(\textbf{x}) = f(x_1, x_2, ..., x_n) = {\sum_{i=1}^{n} x_i^{2}}


        -5,12 \leq x_i \leq 5.12$$

        \text{minimum at }f(0, \cdots, 0) = 0$$

    minimum: x = 0...0
             f(x) = 0

    [1] http://www.sfu.ca/~ssurjano/spheref.html
   """

    def set(self, **kwargs):
        self.name = 'Sphere function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-5.12, ub=5.12)

        self.global_optimum = 0.
        self.global_optimum_coords = [0.0 for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        sum = 0.0
        x = x.vector

        for c in x:
            sum += c ** 2.0

        return [sum]


class Schwefel(BenchmarkFunction):
    """
    Schwefel function is declared by the following formula [1]:

    .. math::
        f(\textbf{x}) = f(x_1, x_2, ..., x_n) = 418.9829d -{\sum_{i=1}^{n} x_i sin(\sqrt{|x_i|})}


        -500 \leq x_i \leq 500$$

        \text{minimum at }f(0, \cdots, 0) = 0$$

    minimum: x = 420.9687,...,420.9687
             f(x) = 0

    [1] http://benchmarkfcns.xyz/benchmarkfcns/schwefelfcn.html
   """

    def set(self, **kwargs):
        self.name = 'Schwefel function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-500., ub=500.)

        self.global_optimum = 0.
        self.global_optimum_coords = [420.9687 for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        alpha = 418.982887
        fitness = 0
        x = x.vector

        fitness = 0.0
        for c in x:
            fitness -= c * np.sin(np.sqrt(np.abs(c)))
            fitness += alpha
        return [fitness]


class ModifiedEasom(BenchmarkFunction):
    """
    Generalized Easom's funcion, or Xin-She Yangs extended function [1]:

    .. math::
        f(\mathbf x)=f(x_1, ..., x_n) =exp(-\sum_{i=1}^{n}(x_i / \beta)^{2m}) - 2exp(-\sum_{i=1}^{n}x_i^2) \prod_{i=1}^{n}cos^ 2(x_i)

        -2.pi \leq x_i \leq 2pi$$

    [1]X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
       Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
   """

    def set(self, **kwargs):
        self.name = 'Modified Easom'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-2. * np.pi, ub=2. * np.pi)

        self.global_optimum = -1.
        self.global_optimum_coords = [np.pi for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        summa = 0.0
        product = -1.0
        x = x.vector

        for c in x:
            product *= -1. * np.cos(c) ** 2.
            summa += (c - np.pi) ** 2.
        return [product * np.exp(-summa)]


class EqualityConstr(BenchmarkFunction):
    """
    Equality Constrained Function [1] eq 7-8:

    .. math::
        f(\mathbf x)= 

    [1] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
       Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
   """

    def set(self, **kwargs):
        self.name = 'Equality Constrained Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=0.0, ub=1.0)

        self.global_optimum = -1.
        self.global_optimum_coords = [1. / np.sqrt(self.dimension) for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        x = x.vector

        product = 1.0
        summa = 0.0
        for c in x:
            product *= c * np.sqrt(self.dimension)
            summa += c * c
        if summa.any() == 1.:
            return [-1.0 * product]
        else:
            return [0.]


class Griewank(BenchmarkFunction):
    """
    Griewank function strongly multimodal:

    A minimization problem [1-2]:

    .. math::
        $$f(x_1 \cdots x_n) = 1 + \frac{1}{4000} \sum_{i=1}^n x^2_i - \prod_{i=1}^n cos(\frac{x_i}{\sqrt{i}})$$

        $$-512 \leq x_i \leq 512$$

        $$\text{minimum at }f(0, \cdots, 0) = 0$$

    [1] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
       Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
    [2] https://www.cs.unm.edu/~neal.holts/dga/benchmarkFunction/griewank.html
   """

    def set(self, **kwargs):
        self.name = 'Griewank Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-512.0, ub=512.0)

        self.global_optimum = 0.0
        self.global_optimum_coords = [0.0 for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        """F6 Griewank's function
        multimodal, symmetric, inseparable"""
        summa = 0
        produkt = 1.
        x = x.vector

        for i, c in enumerate(x):
            summa += c ** 2 / 4000.0
            produkt *= np.cos(c / np.sqrt(i + 1))
        return [summa - produkt + 1.]


class Michaelwicz(BenchmarkFunction):
    """
    Works for 2, 5 and 10 dimensions only.

    Michalewicz function has d! local minima, and it is multimodal. The parameter m defines the steepness of they
    valleys and ridges; a larger m leads to a more difficult search. The recommended value of m is m = 10.

    [1] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
       Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
    [2] http://www.sfu.ca/~ssurjano/michal.html
    [3] http://www.smaree.com/2016/09/30/global-minimum-of-the-michalewicz-function/
   """

    def set(self, **kwargs):
        self.name = 'Michaelwicz Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=0.0, ub=np.pi)

        if self.dimension == 2:
            self.global_optimum = -1.8013
            self.global_optimum_coords = [2.20, 1.57]

        elif self.dimension == 5:
            self.global_optimum = -4.687658

        elif self.dimension == 10:
            self.global_optimum = -9.66015

        else:
            raise ValueError  # shorthand for 'raise ValueError()'

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        f = 0.
        m = 10  # m is generally selected to 10
        x = x.vector

        for i, c in enumerate(x):
            f += np.sin(c) * np.sin((i + 1) * c * c / np.pi) ** (2. * m)
        return [-f]


class Perm(BenchmarkFunction):
    """

    Usually, evaluated on a hypercube, which relates to the d-dimension. 
    The global minima is 0, located in (1, 1/2, ...1/d) point in d dimensions [2] is the correct reference now.

    [1] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
       Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
    [2] http://www.sfu.ca/~ssurjano/perm0db.html
   """

    def set(self, **kwargs):
        self.name = 'Perm Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-float(self.dimension), ub=float(self.dimension))

        self.global_optimum = 0.0
        self.global_optimum_coords = [1. / float(x + 1) for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        b = 10  # optional, with the default value of 10
        f = 0.
        x = x.vector

        for i in range(1, self.dimension + 1):
            for j, d in enumerate(x):
                f += (j + 1 + b) * (d ** i - 1. / ((j + 1.) ** i)) ** 2.
        return [f]


class Rastrigin(BenchmarkFunction):
    """
    Rastrigin function  is multimodal, symmetric, separable:

    .. math::

     $$f(x_1 \cdots x_n) = 10n + \sum_{i=1}^n (x_i^2 -10cos(2\pi x_i))$$

     $$-5.12 \leq x_i \leq 5.12$$

     $$\text{minimum at }f(0, \cdots, 0) = 0$$

    [1] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
       Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
    [2] https://www.cs.unm.edu/~neal.holts/dga/benchmarkFunction/rastrigin.html
   """

    def set(self, **kwargs):
        self.name = 'Rastrigin Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-5.120, ub=5.120)

        self.global_optimum = 0.0
        self.global_optimum_coords = [0.0 for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        fitness = 10 * self.dimension
        x = x.vector

        for c in x:
            fitness += c ** 2 - (10 * np.cos(2 * np.pi * c))
        return [fitness]


class SixHump(BenchmarkFunction):
    """
    Six Hump Camel Back function  is a two variable function with two separate optimum:


    [1] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
       Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
    [2] https://www.cs.unm.edu/~neal.holts/dga/benchmarkFunction/rastrigin.html
   """

    def set(self, **kwargs):
        self.name = 'Six hump camel back Function'

        self.set_dimension = 2.0
        self.parameters = [{'name': 'x', 'bounds': [-3., 3.]},
                           {'name': 'y', 'bounds': [-2., 2.]}]

        # there are 2 global optimums!!!
        self.global_optimum = -1.0316
        self.global_optimum_coords = [0.0898, -0.7126]  # [-0.0898, 0.7126]]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        x = x.vector

        return [((4 - 2.1 * x[0] ** 2 + x[0] ** 4 / 3.) * x[0] ** 2 + x[0] * x[1]
                 - 4 * x[1] ** 2 + 4 * x[1] ** 4)]


class Schubert(BenchmarkFunction):
    """
    It is a two variable function, which has 18 global optimums for n=5:

    [1] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
       Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
    [2] http://www.sfu.ca/~ssurjano/shubert.html
   """

    def set(self, **kwargs):
        self.name = 'Schubert Function'

        self.set_dimension = 2.0
        self.parameters = [{'name': 'x', 'bounds': [-10., 10.]},
                           {'name': 'y', 'bounds': [-10., 10.]}]

        self.global_optimum = -186.7309
        # self.global_optimum_coords =  TODO: find these coorcdinates

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        n = 5
        f1 = 0.0
        f2 = 0.0
        x = x.vector

        for i in range(1, n + 1):
            f1 += i * np.cos(i + (i + 1) * x[0])
            f2 += i * np.cos(i + (i + 1) * x[1])
        return [f1 * f2]


class Zakharov(BenchmarkFunction):
    """
    Zakharov function [1]:

    .. math::

     $$f(x_1 \cdots x_n) = 10n + \sum_{i=1}^n (x_i^2 -10cos(2\pi x_i))$$

     $$-5.12 \leq x_i \leq 5.12$$

     $$\text{minimum at }f(0, \cdots, 0) = 0$$

    [1] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
       Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
   """

    def set(self, **kwargs):
        self.name = 'Zakharov Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-5.0, ub=10.0)

        self.global_optimum = 0.0
        self.global_optimum_coords = [0.0 for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        f1 = 0.0
        f2 = 0.0
        f3 = 0.0
        x = x.vector

        for i, c in enumerate(x):
            f1 += c ** 2
            f2 += 0.5 * (i + 1) * c
            f3 += 0.5 * (i + 1) * c
        return [f1 + f2 ** 2. + f3 ** 2.]


class XinSheYang(BenchmarkFunction):
    """
        Xin She Yang function [1-2]:

        .. math::
            f(\mathbf{x})=f(x_1, ..., x_n)=(\sum_{i=1}^{n}|x_i|)exp(-\sum_{i=1}^{n}sin(x_i^2))

        [1] Momin Jamil and Xin-She Yang, A literature survey of benchmark functions for global optimization problems,
            Int. Journal of Mathematical Modelling and Numerical Optimisation}, Vol. 4, No. 2, pp. 150–194 (2013),
            arXiv:1308.4008
        [2] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
            Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549
       """

    def set(self, **kwargs):
        self.name = 'Xin She Yang - 1  Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-2.0 * np.pi, ub=2.0 * np.pi)

        self.global_optimum = 0.0
        self.global_optimum_coords = [0.0 for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        f1 = 0.0
        f2 = 0.0
        x = x.vector

        for c in x:
            f1 = np.fabs(c)
            f2 = np.sin(c ** 2.)
        return [f1 * np.exp(-f2)]


class XinSheYang2(BenchmarkFunction):
    """
        Xin She Yang function [1-2]:

        .. math::


       [1] Momin Jamil and Xin-She Yang, A literature survey of benchmark functions for global optimization problems,
           Int. Journal of Mathematical Modelling and Numerical Optimisation}, Vol. 4, No. 2, pp. 150–194 (2013),
           arXiv:1308.4008
       [2] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
           Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549

       """

    def set(self, **kwargs):
        self.name = 'Xin She Yang - 2  Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-20.0, ub=20.0)

        self.global_optimum = -1.0
        self.global_optimum_coords = [0.0 for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        f1 = 0.0
        f2 = 0.0
        f3 = 1.0
        beta = 15.
        m = 5.
        x = x.vector

        for c in x:
            f1 = -1. * (c / beta) ** (2. * m)
            f2 = -1. * c ** 2.
            f3 = np.cos(c) ** 2.
        return [(np.exp(f1) - 2. * np.exp(f2)) * f3]


class XinSheYang3(BenchmarkFunction):
    """
        Xin She Yang function [1-2]:

        .. math::
            f(\mathbf x)=f(x_1, ...,x_n)=\sum_{i=1}^{n}\epsilon_i|x_i|^i

       [1] Momin Jamil and Xin-She Yang, A literature survey of benchmark functions for global optimization problems,
           Int. Journal of Mathematical Modelling and Numerical Optimisation}, Vol. 4, No. 2, pp. 150–194 (2013),
           arXiv:1308.4008
       [2] X. S. Yang, “Test Problems in Optimization,” Engineering Optimization: An Introduction with Metaheuristic
           Applications John Wliey & Sons, 2010. [Available Online]: http://arxiv.org/abs/1008.0549

       """

    def set(self, **kwargs):
        self.name = 'Xin She Yang - 3  Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=-5.0, ub=5.0)

        self.global_optimum = 0.0
        self.global_optimum_coords = [1. / float(x + 1) for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        f1 = 0.0
        x = x.vector

        for i, c in enumerate(x):
            eps = uniform(0, 1)
            f1 = eps * np.fabs(c - 1. / (i + 1.))
        return [f1]


class BinhAndKorn:
    """
    This problem is often attributed to Binh and Korn, but is also mentioned in A Osyczka, H Tamura,
    Pareto set distribution method for multicriteria optimization using genetic algorithm.

    The problem is searching the minimum of the following function:

    $min(f_{1}(x,y),f_{2}(x,y))$

    where

        $f_{1}(x,y) = 4x^2 + 4y^2$
        $f_{2}(x,y) = (x-5)^2 + (y-5)^2 $

    subject to the following constraints:

    $g_{1}(x,y) = (x-5)^2 + y^2 \leq 25$
    $g_{2}(x,y) = (x-8)^2 + (y+3)^2 \leq 7.7$

    search domain: 0 <= x <=5, 0 <= y <= 3

    """

    @staticmethod
    def approx(x):
        """
        Estimates the pareto front of the Binh_and_Korn function in the following domain: [10;100]

        The approximate function is:

        f = a0+a1*x+a2*x^2+a3*x^3+a4*x^4+a5*x^5

        a0 = 4.564170954344949e+01 +/- 2.827448422117511e-01
        a1 = -1.939843031431697e+00 +/- 5.958429263576211e-02
        a2 = 5.327835893656892e-02 +/- 3.214348835707173e-03
        a3 = -7.960654974842228e-04 +/- 6.602271055507837e-05
        a4 = 5.666751361667045e-06 +/- 5.691036855808526e-07
        a5 = -1.505297721151948e-08 +/- 1.733740155631940e-09

        fitted on a platypus calculation: algorithm nsga II, 10 000 evaluations

        :param x: the value of the f1 function
        :return:
        """

        a0 = 4.564170954344949e+01
        a1 = -1.939843031431697e+00
        a2 = 5.327835893656892e-02
        a3 = -7.960654974842228e-04
        a4 = 5.666751361667045e-06
        a5 = -1.505297721151948e-08

        return a0 + a1 * x + a2 * x ** 2. + a3 * x ** 3. + a4 * x ** 4. + a5 * x ** 5.

    @classmethod
    def eval(cls, x):
        f1 = 4 * pow(x[0], 2) + 4 * pow(x[1], 2)
        f2 = pow(x[0] - 5, 2) + pow(x[1] - 5, 2)
        target = [f1, f2]

        return target

    @classmethod
    def constraints(cls, x):
        # 0 <= x <=5, 0 <= y <= 3
        g1 = min(0, 25 - pow(x[0] - 5, 2) - pow(x[1], 2))
        g2 = min(0, pow(x[0] - 8, 2) + pow(x[1] + 3, 2) - 7.7)
        violation = [g1, g2]
        return violation


class Booth(BenchmarkFunction):
    """
    Booth function:
    .. math::
        f(x,y)=(x+2y-7)^2+(2x+y-5)^2

    [1] http://benchmarkfcns.xyz/benchmarkfcns/boothfcn.html
    """

    def set(self, **kwargs):
        self.name = 'Booth Function'

        self.set_dimension = 2.0
        self.parameters = [{'name': 'x', 'bounds': [-5., 5.]},  # the search space reduced because of the surrogate test
                           {'name': 'y', 'bounds': [-5., 5.]}]

        self.global_optimum = 0.0
        self.global_optimum_coords = [1., 3.]
        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        x = x.vector
        return [(x[0] + 2 * x[1] - 7) ** 2 + (2 * x[0] + x[1] - 5) ** 2]


class GramacyLee(BenchmarkFunction):
    """
    Gramacy  & Lee function is a simple 1D, not convex test function:
    .. math::
        f(x) = \frac{sin(10\pi x)}{2x} + (x-1)^4
    [1] http://benchmarkfcns.xyz/benchmarkfcns/gramacyleefcn.html
    [2] https://al-roomi.org/benchmarks/unconstrained/1-dimension/258-gramacy-lee-s-function-no-1
    """

    def set(self, **kwargs):
        self.name = 'Gramacy and Lee Function'

        self.set_dimension = 1.0
        self.parameters = [{'name': 'x', 'bounds': [0.5, 2.5]}]

        self.global_optimum = -0.869011134989500
        self.global_optimum_coords = [0.548563444114526]
        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        x = x.vector
        f = np.sin(10.0 * np.pi * x[0]) / (2. * x[0]) + (x[0] - 1.) ** 4
        return [f]


class AlpineFunction(BenchmarkFunction):
    """
        Alpine function is a simple n dimension test problem [1-3]:

        .. math::
            f(\mathbf x)=f(x_1, ..., x_n) = \sum{i=1}^{n}|x_i sin(x_i)+0.1x_i|

    [1] Momin Jamil and Xin-She Yang, A literature survey of benchmark functions for global optimization problems,
        Int. Journal of Mathematical Modelling and Numerical Optimisation}, Vol. 4, No. 2, pp. 150–194 (2013),
        arXiv:1308.4008
    [2] M. Clerc, “The Swarm and the Queen, Towards a Deterministic and Adaptive Particle Swarm Optimization, ”
        IEEE Congress on Evolutionary Computation, Washington DC, USA, pp. 1951-1957, 1999.
    [3] http://benchmarkfcns.xyz/benchmarkfcns/alpinen1fcn.html
    """

    def set(self, **kwargs):
        self.name = 'Xin She Yang - 3  Function'

        self.set_dimension(**kwargs)
        self.parameters = self.generate_paramlist(self.dimension, lb=0.0, ub=10.0)

        self.global_optimum = 0.0
        self.global_optimum_coords = [0. for x in range(self.dimension)]

        # single objective problem
        self.costs = [{'name': 'f_1', 'criteria': 'minimize'}]

    def evaluate(self, x):
        f1 = 0.0
        x = x.vector

        for c in x:
            f1 += np.abs(c * np.sin(c) + 0.1 * c)
        return [f1]


class SurrogateBenchmarkData:
    def __init__(self, n_test=5, n_train=None, verbose=1):
        self.n_test = n_test
        self.n_train = n_train
        self.verbose = verbose

        self.x_data = [[]]
        self.y_data = [[]]

    def _train_data(self):
        n = self.length()

        if self.n_train is None:
            x_data = self.x_data
            y_data = self.y_data
        else:
            x_data = self.x_data[:self.n_train + self.n_test]
            y_data = self.y_data[:self.n_train + self.n_test]

        if self.n_test > 0:
            x_train, x_test, y_train, y_test = train_test_split(x_data, y_data, train_size=n, test_size=self.n_test,
                                                                random_state=4)
        else:
            x_train = x_data
            y_train = y_data
            x_test = []
            y_test = []

        if self.verbose == 1:
            print("Number of train samples: {}, number of test samples: {}".format(n, self.n_test))

        return x_train, y_train, x_test, y_test

    def length(self):
        assert len(self.x_data) == len(self.y_data)

        if self.n_train is None:
            return len(self.x_data) - self.n_test
        else:
            return self.n_train

    def limits(self):
        # TODO: for more parameters
        lims = []
        for i in range(len(self.x_data[0])):
            lims.append([sys.float_info.max, -sys.float_info.max])

        for i in range(self.length()):
            for j in range(len(self.x_data[0])):
                lims[j][0] = min(self.x_data[i][j], lims[j][0])
                lims[j][1] = max(self.x_data[i][j], lims[j][1])

        return lims


class SurrogateBenchmarkTeamSingle(SurrogateBenchmarkData):
    def __init__(self, n_test=5, n_train=None, verbose=1):
        super().__init__(n_test, n_train, verbose)

    def values(self, set="250"):
        if set == "100":
            self.x_data = [[0.03524622653769729, 0.012453844637180172, 0.03550642124447178, 0.02110548108032166,
                            0.045538484031520485, 0.04734044034610589, 0.0269070185501492, 0.014411083695101346,
                            0.038585623141211854, 0.03922199779276903],
                           [0.04173143136917885, 0.04726105620830657, 0.02791987968371437, 0.03324481445335426,
                            0.006647427726004693, 0.038849607752053664, 0.03527108283953232, 0.03517783761324652,
                            0.008903932366147383, 0.04665478675644693],
                           [0.01716249754930431, 0.028630590850059798, 0.044008092442600594, 0.01761511572630732,
                            0.04997791266975473, 0.017828191291329978, 0.044382333066898046, 0.030674270038144217,
                            0.036094722142314885, 0.03143208503789327],
                           [0.03489903000471019, 0.012722919164628267, 0.026318136876675544, 0.023358312213997184,
                            0.007333430835061327, 0.041165327566304456, 0.029105235297508334, 0.049309534956937226,
                            0.025101761693960223, 0.011142863991466075],
                           [0.02321919389545712, 0.019272812597651987, 0.015739499362001128, 0.03691919103401917,
                            0.03793974939056681, 0.027939898267822533, 0.04468016958148079, 0.01219926456234847,
                            0.022310478054286886, 0.010527329365531935],
                           [0.007083311977702306, 0.03776684475815727, 0.04082440924289525, 0.037554281211105475,
                            0.013138035832881798, 0.014129106327895349, 0.01099216579021851, 0.04588669683512349,
                            0.021250140924185118, 0.007811438987591604],
                           [0.03375669314042771, 0.040674512677896395, 0.010686017149843775, 0.039490359505027994,
                            0.025123761340440774, 0.02547573049473017, 0.0184183448988869, 0.02791163721956003,
                            0.031779290569826334, 0.036790177491819405],
                           [0.021297833530245093, 0.04221373151105264, 0.01141749825589142, 0.04319518939999616,
                            0.03257393275561994, 0.0191435689770608, 0.0288321829859359, 0.03759857528376719,
                            0.042276154924576696, 0.03578355258006228],
                           [0.00679124010531329, 0.01430709648175547, 0.02500848698548833, 0.00923158273442853,
                            0.018092549560341814, 0.031240395939826842, 0.03512243323896948, 0.022663630185372492,
                            0.02401360324914321, 0.022424403326611633],
                           [0.02836529976936044, 0.03365094341577045, 0.012372067786457268, 0.03880460006871125,
                            0.04193670076722619, 0.03197309445835113, 0.04720798591888873, 0.044660750930183735,
                            0.014110652948085972, 0.04042582323673426],
                           [0.01602569579735631, 0.028115696039519834, 0.039340282366942925, 0.03394050152640426,
                            0.02124002416373558, 0.01098707542416084, 0.023846780621924007, 0.01539331477689388,
                            0.027487625509125726, 0.005290931878276189],
                           [0.029831603373267363, 0.022312945303290307, 0.013632118635708523, 0.024010302558102,
                            0.016435640861197445, 0.012051100094860226, 0.04897889022429164, 0.04640542949966959,
                            0.013978359274488875, 0.03988587674103783],
                           [0.0376608558923794, 0.02591820663759142, 0.03774459169507666, 0.026316202024659827,
                            0.020391388028323205, 0.024380974993445142, 0.031149042196332504, 0.04006072077309003,
                            0.011680674511236807, 0.04489668076075836],
                           [0.030567145247595053, 0.015269079747629054, 0.010015053171586328, 0.015332552158212238,
                            0.03715350440337289, 0.008642108022443525, 0.03572350804608994, 0.013548159161659256,
                            0.041653745356630315, 0.013589381126200045],
                           [0.010961944793176321, 0.018898948164970543, 0.009147082396750288, 0.030490061487875978,
                            0.0304131732092672, 0.0217781279185875, 0.007541644865840649, 0.019698223279116873,
                            0.02079176279654965, 0.03341108491193506],
                           [0.022186378132585612, 0.018304120431694302, 0.014186133660312406, 0.016916926088628045,
                            0.04298543505835895, 0.01606788403526356, 0.01749776963743383, 0.031278164936160986,
                            0.04395577457926063, 0.04372035825373543],
                           [0.010697329921985254, 0.006460940262459037, 0.03358316219482534, 0.047211585808874204,
                            0.03688643990122491, 0.013253633082395608, 0.014708212446339877, 0.04444755905318127,
                            0.023276455731842574, 0.024596698366247432],
                           [0.0220467078221415, 0.00855198017158166, 0.019172620124138798, 0.011225668027361774,
                            0.023885114605490695, 0.007957293214113955, 0.027880372788292432, 0.020646901175250237,
                            0.020270318735612185, 0.013516004270307046],
                           [0.013756938958150548, 0.04336334788741935, 0.02409066634373459, 0.02907697288510443,
                            0.026618510893980936, 0.041606229014651275, 0.006872678834692244, 0.007449788476944061,
                            0.04367510307570675, 0.04445264343703838],
                           [0.008285479709422611, 0.021448171628438747, 0.03833633061122411, 0.04171378118255841,
                            0.04146677551951747, 0.012872376742376315, 0.029577446165643676, 0.029390929054487883,
                            0.03074494817498672, 0.03545040158382397],
                           [0.04982738792457931, 0.04522380472032298, 0.012831333579333179, 0.013417342893438985,
                            0.0412290141422835, 0.009504568292593902, 0.021701125293336582, 0.008872592948782555,
                            0.04457734025316356, 0.02408516639766051],
                           [0.01904470934957504, 0.020965880370398624, 0.037367038283474915, 0.027826589311523316,
                            0.0190102757814607, 0.04399102318484299, 0.04840724176284572, 0.005857006336829643,
                            0.026555637270255783, 0.006831898463746316],
                           [0.011781148739502628, 0.009509863864806869, 0.02797985426461153, 0.006369457070306256,
                            0.01851793093110828, 0.04863518635885137, 0.04769721856012947, 0.04055858108780585,
                            0.013379890551480977, 0.017419942925158333],
                           [0.012501930207625245, 0.025693764571349142, 0.02682832317546024, 0.011511121671017293,
                            0.01602333525347307, 0.03324951045786085, 0.01931691888560798, 0.01300330739142773,
                            0.011916926959942436, 0.024988212168699425],
                           [0.04485712631748988, 0.02522907730643831, 0.020270270429036105, 0.007643224475607382,
                            0.04507606531931151, 0.04531807039656909, 0.04613302919973406, 0.02986763796760229,
                            0.016844584861900987, 0.04163192195111281],
                           [0.02358486139075597, 0.04927379025856709, 0.029240121327275267, 0.021592193815740544,
                            0.03882470642960234, 0.006551844319146693, 0.017628909601450884, 0.021998163759111178,
                            0.033327737135288034, 0.01627943317053146],
                           [0.023984318402989983, 0.03912372979732469, 0.019614516984076544, 0.031202439511932373,
                            0.027551545432381117, 0.005624245492074761, 0.036069460129502386, 0.036045817394733136,
                            0.014852993320991624, 0.019962514035480394],
                           [0.02025875602365704, 0.010898062815517265, 0.04660133434795835, 0.04501504183597427,
                            0.02227020680473064, 0.036408184172594955, 0.04247844936881986, 0.03510617343430502,
                            0.043236307276358056, 0.015818741683852302],
                           [0.043918631157342354, 0.03539370752200319, 0.005495889239798708, 0.022423520724938945,
                            0.03833086330588108, 0.017207167014877674, 0.04951543555141841, 0.019287855109495457,
                            0.030566925891460966, 0.0065852973777894295],
                           [0.04584359181955955, 0.021810461389613713, 0.044316163752484164, 0.049413489418509704,
                            0.04739975125582344, 0.015803576226061843, 0.04225153485534926, 0.01385409671828177,
                            0.006666096696580713, 0.03391758345970995],
                           [0.04927127933660478, 0.03128743589934941, 0.038891378858619985, 0.008331116084259474,
                            0.02971305474785882, 0.046816118955504867, 0.02037780853994484, 0.03256153137537318,
                            0.018097325035683124, 0.045450884622682916],
                           [0.016711790996298638, 0.03190830809058858, 0.02469521112932955, 0.03701396340326111,
                            0.04871676426462641, 0.014964927034818644, 0.03841960181598592, 0.045311125784447244,
                            0.03566916744972431, 0.011380189635176643],
                           [0.0401296538376276, 0.030030906923287762, 0.017386649047746543, 0.029449302756944712,
                            0.010916818493712017, 0.023827926351766494, 0.006080884532863556, 0.00513036888595058,
                            0.019188560277795114, 0.0373179048673459],
                           [0.025756030547442233, 0.04837016763472093, 0.03242166045580418, 0.016229308226133457,
                            0.04076032255242787, 0.03922662613654046, 0.015302135493677552, 0.0176359835724973,
                            0.034044230307911295, 0.04894710581710138],
                           [0.0182468501586806, 0.024421954373636895, 0.01349635298665013, 0.036142603835401986,
                            0.046736948129312644, 0.0072340777848518955, 0.04548206952951351, 0.00825883484269882,
                            0.036510643510262376, 0.03887418159482388],
                           [0.03579361246278381, 0.043232733720394836, 0.01888388031117393, 0.04024935879026254,
                            0.011818974755648002, 0.02000686630842286, 0.030923061997767287, 0.04698322447106264,
                            0.028055691266942746, 0.007300867204595133],
                           [0.008989007277866543, 0.015998020071497848, 0.04324989786540669, 0.030762413590222477,
                            0.03306762632425285, 0.048017126271888, 0.03192601137454693, 0.015042386767410435,
                            0.04493841518003766, 0.014995542355493621],
                           [0.033899075761502084, 0.030683945319274878, 0.02340797004447708, 0.03192083735356224,
                            0.031037541993875664, 0.04959155619009672, 0.020151859377248667, 0.009366325626448761,
                            0.017437963874713244, 0.026004323691580664],
                           [0.04816326951828528, 0.046699602951179996, 0.008360814270179357, 0.043333430546750006,
                            0.00537892769502696, 0.023041484898622906, 0.00814313338848459, 0.025324264473281913,
                            0.010561696310706366, 0.047052299487715596],
                           [0.042111547198033845, 0.03973924242656668, 0.033814480289660935, 0.04435892740768047,
                            0.01557200873020875, 0.008419727430767503, 0.011917131038193538, 0.021035823503774302,
                            0.0465173751257452, 0.019309631293746743],
                           [0.040614423293447784, 0.036833227851678856, 0.016069926313153387, 0.0457264468251896,
                            0.011504168931684681, 0.04702586113017778, 0.04584362329500179, 0.012532865151628974,
                            0.02255699142458926, 0.029901553303864535],
                           [0.01448788629338053, 0.010478296338709578, 0.007592187001244014, 0.017600255587980545,
                            0.034162738709531196, 0.040002950536722655, 0.043312587112448245, 0.010917520389920825,
                            0.03953420327686009, 0.010057648735161847],
                           [0.006031708912378956, 0.04628578325437398, 0.023855130567612258, 0.03272184284719731,
                            0.026481872215959292, 0.034142484038730554, 0.012711710413255684, 0.03321699912440808,
                            0.04573642639740918, 0.04358042364890018],
                           [0.03807044894333262, 0.02661827318699021, 0.02217861758182662, 0.010740578713328626,
                            0.02271228726282303, 0.029596565324276042, 0.00916800693886353, 0.02849356190941163,
                            0.015605897281624329, 0.025554449141976982],
                           [0.005300406279695637, 0.015743982443136406, 0.018150626754472605, 0.018818251215051162,
                            0.01457004053360992, 0.02924624161975043, 0.022793931435355647, 0.006221102399604802,
                            0.04023146996216265, 0.006032106790285914],
                           [0.029648562638286132, 0.02376468655347479, 0.01704839781924826, 0.014244357170938103,
                            0.01525101462257462, 0.03800612669044168, 0.046612835330813875, 0.021598387817605286,
                            0.006157104358859354, 0.028429245400384825],
                           [0.021199793168333446, 0.03393924090171397, 0.010987805053944822, 0.03441450121883999,
                            0.03450805542491526, 0.025740159757835814, 0.039941710320363415, 0.01813988619316035,
                            0.04243987480274667, 0.011816079362621272],
                           [0.01980770645232218, 0.03228401219344982, 0.009022949379173497, 0.02181037298712031,
                            0.017589281135290905, 0.03232160788273563, 0.013472641639529746, 0.02346751465713011,
                            0.007256960006673891, 0.031909043349714235],
                           [0.007316607633227544, 0.01141253211098273, 0.031021971426867325, 0.012915393581422067,
                            0.03214897769571308, 0.028466462592142132, 0.01626683593950635, 0.04368809270958953,
                            0.020618759388005677, 0.01418631523406972],
                           [0.012892917076258264, 0.030209412504092995, 0.04504011369836765, 0.02468237267539928,
                            0.04788526942722965, 0.009646112935169041, 0.027988497143458337, 0.045983835672009234,
                            0.035427263180287036, 0.015604788580332405],
                           [0.03144890428285361, 0.005589972245340797, 0.021425951702372204, 0.03546329909852893,
                            0.039532001120502064, 0.019622940931655973, 0.00999413012541771, 0.041749045174312154,
                            0.028959202223385418, 0.0382275597440947],
                           [0.04713043684260662, 0.03285259763625027, 0.043410751441808386, 0.005332723804668981,
                            0.04479901475964836, 0.040104928771298766, 0.049910024653015105, 0.04813208366175192,
                            0.04988336710218524, 0.04603157126201786],
                           [0.048652160729144675, 0.012111412680857659, 0.0063226390820692365, 0.048691615449709964,
                            0.009677474855260321, 0.014548658474412154, 0.01551068184573965, 0.042675201629077,
                            0.02943111172325604, 0.0480669620843364],
                           [0.026898398631738003, 0.010203639137052526, 0.007878564217490919, 0.028575085459815297,
                            0.027217157011408383, 0.049366473514165315, 0.03710001242023887, 0.04042726899543094,
                            0.03254672168410765, 0.029196907066704133],
                           [0.03693717921242519, 0.02618957944520824, 0.021790343937896292, 0.04116142055725218,
                            0.012277245934311051, 0.036568605847323694, 0.03889783729973375, 0.03691369108681314,
                            0.02700943141642708, 0.027581460739468863],
                           [0.04773472203256523, 0.017801599826639754, 0.025536457462488907, 0.03984330311002459,
                            0.037756911655253404, 0.028094591092187395, 0.03305552107819066, 0.025723033243959113,
                            0.04745801609315873, 0.03623997013709774],
                           [0.04314365943455129, 0.03435250208402401, 0.04831411406824726, 0.034720323723129096,
                            0.024006968033976978, 0.040718521944853515, 0.024606231171626983, 0.031669673182178344,
                            0.015240208711300995, 0.023494780706216287],
                           [0.03999520790475931, 0.020463599291856777, 0.03787920264783935, 0.038535736589304544,
                            0.008692477387286375, 0.006237109588668684, 0.01260889058782022, 0.048694636094918954,
                            0.012344024332072403, 0.005696920177601282],
                           [0.015460579091054149, 0.03847860270515576, 0.041630445902512754, 0.012261966101945531,
                            0.035412114767110245, 0.007270998335675944, 0.04407440322343449, 0.023097411171776324,
                            0.01300319905112544, 0.04084801229588176],
                           [0.03217517362768551, 0.03699003363566482, 0.0470952145361344, 0.012008300455137986,
                            0.048407041778250746, 0.03283533506668564, 0.024979764543452125, 0.04844489671371968,
                            0.005312103819828473, 0.04134416215578198],
                           [0.04114344361878902, 0.035667153049342905, 0.04744196312248378, 0.04741757339432806,
                            0.04632972054755502, 0.0354256399751102, 0.014056195071074896, 0.007739424846761691,
                            0.03020417405726356, 0.01298554647471897],
                           [0.011421818913488813, 0.04395822633918978, 0.04984633549391244, 0.01945060005092226,
                            0.0336399974355852, 0.030516358054631626, 0.030342572303495283, 0.03930470204453075,
                            0.038892040332188085, 0.012445176859079834],
                           [0.028497892157410085, 0.013347338529530909, 0.020385547181775265, 0.030145137477918946,
                            0.04051360608562286, 0.04437127896160634, 0.025279206603236765, 0.038603278883753846,
                            0.025538449978519923, 0.008334538584078646],
                           [0.014075956630520269, 0.04970078534741413, 0.014474052267625558, 0.04533969129633871,
                            0.014138375481987427, 0.022457325138366995, 0.00825338732137349, 0.028057862163941155,
                            0.046158574341273864, 0.014547524062417782],
                           [0.017767654822489213, 0.04187989185086885, 0.04558780450382422, 0.02006592807494877,
                            0.04326085732397113, 0.043034957362007145, 0.03349925886980769, 0.036228332386792246,
                            0.037188799292733915, 0.021627819240491807],
                           [0.018619580602924273, 0.04567917982847245, 0.039986290256685715, 0.008789899343213276,
                            0.021888707407569333, 0.03105076567374306, 0.0393511911173709, 0.027125836005557815,
                            0.02177906400171995, 0.04250963766802606],
                           [0.005819790818212877, 0.041151855333747246, 0.04208210752457748, 0.01033568020123194,
                            0.010301607353151927, 0.02119134550639512, 0.04076776058741293, 0.03050421063868284,
                            0.019580010497239488, 0.032093026049896674],
                           [0.04435985232125221, 0.0170976166021146, 0.02860774845962664, 0.024928810733125018,
                            0.009451675382231465, 0.0348633334568022, 0.021438773786185295, 0.016225290945350714,
                            0.009912428444821878, 0.04564733917899512],
                           [0.029185045441345616, 0.04773402942255898, 0.030475629862395263, 0.014816384280244729,
                            0.044458220201842924, 0.04196498079345823, 0.016072948499527492, 0.04211614495468902,
                            0.005625254809617833, 0.018413140924972853],
                           [0.04263018491267673, 0.02781855871343381, 0.03272120900994342, 0.042098723705077994,
                            0.025908714261588407, 0.010247819610541293, 0.011389540137853382, 0.026535597903018616,
                            0.03216278959329907, 0.022771289954422583],
                           [0.015022827373287584, 0.04460406049648553, 0.005436485443877335, 0.005974153744665748,
                            0.04725265605234752, 0.010502864349761953, 0.03444745927645782, 0.033611040585794345,
                            0.048378288639775426, 0.034937219020864836],
                           [0.031750794535828034, 0.03962051195299776, 0.009794908917003364, 0.025838786991513638,
                            0.013757825949662655, 0.013610784654534615, 0.04290823411459062, 0.01001655700100049,
                            0.018035289232608928, 0.018631444790348915],
                           [0.046072298440173715, 0.005079489183476137, 0.021002603988334645, 0.027439246408331217,
                            0.00780324019422373, 0.018849797961966407, 0.03742917670330182, 0.03439226506105231,
                            0.03501782358781139, 0.021978504000787935],
                           [0.04675561184797124, 0.023910945338987487, 0.03606897726053867, 0.03233839798092882,
                            0.04928787415417426, 0.005049054738062044, 0.03824805622742159, 0.04382201976728967,
                            0.009494153938813123, 0.048394963026826394],
                           [0.009381425032250087, 0.04052355960385422, 0.040150967823492804, 0.0099099317021876,
                            0.019868459025241928, 0.02418365441790267, 0.03383881522402128, 0.03877206523868934,
                            0.041429041556575466, 0.023424131734520674],
                           [0.00780889899926277, 0.04423995320808999, 0.04901450596108507, 0.015713988839142167,
                            0.0424269405056608, 0.011440029301050213, 0.026555630712376193, 0.02249545776405633,
                            0.03146481926406262, 0.03036099808442382],
                           [0.032891951792849636, 0.007579304337165156, 0.049141681044963276, 0.03827446692148434,
                            0.03622991126598047, 0.03754779619584437, 0.024175619082606643, 0.011461463791356416,
                            0.03792969583135139, 0.019552512675869164],
                           [0.02286349005266532, 0.008965133403643298, 0.027102150344580023, 0.046518326878484305,
                            0.024690735760231522, 0.03873189056450606, 0.006481733696084897, 0.010644008943154305,
                            0.008341615336711308, 0.03426261565378593],
                           [0.013397799877233954, 0.009493191848934334, 0.016632462810309962, 0.043783469505773796,
                            0.016950919195621812, 0.045045323648374914, 0.04031922326016186, 0.04144571660763261,
                            0.0166655449890573, 0.020664438771536883],
                           [0.025284034252491726, 0.03481611610886159, 0.042649929224009696, 0.018263650609119733,
                            0.008512453811713745, 0.018262644863851308, 0.008877407927329382, 0.023942317392807153,
                            0.007294975949716552, 0.04994394276194047],
                           [0.04354216564622288, 0.014872970959532324, 0.01774957700622675, 0.013700829991132622,
                            0.019733470593069024, 0.02997067443882757, 0.019461783988839115, 0.016304004893577584,
                            0.037673863869534245, 0.009062641912528833],
                           [0.03301911608999793, 0.028987475013619424, 0.01510836374938987, 0.02064207652047698,
                            0.034883165718473545, 0.04355488804035809, 0.023192922436227593, 0.009859311045093178,
                            0.011078917109029926, 0.026257848932857715],
                           [0.04827147867115744, 0.016381839569681114, 0.03588040460010511, 0.007997321513708008,
                            0.031364582230451174, 0.021349043246429403, 0.022275324847298557, 0.014620129765317704,
                            0.02600736716251913, 0.042281148582418646],
                           [0.03851774012728415, 0.029432752336577023, 0.007225951703534142, 0.016373410006635825,
                            0.020811603945290504, 0.012347527552846559, 0.013840638089879325, 0.04962566915609081,
                            0.028847447591297863, 0.049312977183404084],
                           [0.009748334500500801, 0.03788431322561363, 0.03306095353141038, 0.005831404906087958,
                            0.03989534728159581, 0.022565572281541, 0.01882009719479755, 0.01880587425369212,
                            0.049296623309086816, 0.009955627153987592],
                           [0.02457392891714881, 0.006342865929962017, 0.03159024557829883, 0.0357738621337693,
                            0.023170654342611025, 0.026935682666383423, 0.029846405446354585, 0.020239069122450007,
                            0.027810770920576443, 0.0169898309964344],
                           [0.03962401287748464, 0.022688359627310476, 0.04101861786254833, 0.04602331964073809,
                            0.017742335744874767, 0.03363926003642491, 0.03668930767784075, 0.034185885260506595,
                            0.03991287918473621, 0.029606857622066864],
                           [0.010027834291508369, 0.013775017448082516, 0.034788357503784104, 0.026990840873491802,
                            0.012852458474218651, 0.045689117639495035, 0.02726216257920265, 0.02681453333113031,
                            0.033429267737917136, 0.0267836196269775],
                           [0.02771898999064252, 0.017213336424191455, 0.04535886589682223, 0.048196846852032155,
                            0.03570705891672025, 0.01639339666070311, 0.010600241228768776, 0.017370294010344667,
                            0.016205861265829402, 0.03333285435448394],
                           [0.030754222285800433, 0.03329189018281158, 0.046061451312637273, 0.04855693563575414,
                            0.007182794564357095, 0.03567464956077491, 0.03218985848975083, 0.007117552216478213,
                            0.04775832168408435, 0.032467561890849476],
                           [0.02073046802230338, 0.023378538051875612, 0.011858503583707571, 0.006934537016299652,
                            0.005659298615544342, 0.03461238874150203, 0.021119771996322513, 0.03217881908669221,
                            0.04089153945610422, 0.028030443826133046],
                           [0.02505616401448437, 0.007864450208935822, 0.04775204560285964, 0.02352209152857612,
                            0.044035549269053294, 0.016763901182311355, 0.04162267150088278, 0.025066132833006204,
                            0.047181547816197586, 0.04758327431856279],
                           [0.027256434127391208, 0.036465296135879885, 0.02601476439604137, 0.04093951316240836,
                            0.010599053756120077, 0.027429341503917983, 0.0410199998954332, 0.04753132462366373,
                            0.04548356434721534, 0.02083242649673613],
                           [0.03429835601877308, 0.020242637958165874, 0.03693245209682213, 0.0253732237176862,
                            0.028764665306937033, 0.02499191604787103, 0.017081159132312184, 0.03803971230291043,
                            0.010098761287230069, 0.03764752585919858],
                           [0.03729444897024068, 0.027407894788031215, 0.022574835300216527, 0.042570554326757916,
                            0.03003307661482945, 0.036994837257254096, 0.005506911232464379, 0.0247658030779006,
                            0.023478493435015957, 0.00863353038253172],
                           [0.0162748289641932, 0.04798287222391132, 0.03141891240708849, 0.02814773920843172,
                            0.028291649614393836, 0.04893528685592765, 0.005294615178835572, 0.017067272934630527,
                            0.034482184959924896, 0.03874653864104898],
                           [0.03895948083594054, 0.0427484577903408, 0.034595582566939044, 0.0192865750720225,
                            0.025452994002989785, 0.026425908844305873, 0.03267289316778968, 0.02929138596497091,
                            0.00807153128651313, 0.04281151040152838],
                           [0.02656115700901534, 0.049054311845487704, 0.02985103161143371, 0.04991096919787087,
                            0.03192409522067889, 0.0204913499449883, 0.04810092091261951, 0.03707286999078388,
                            0.024749807448060112, 0.027308698861813648],
                           [0.045283679540330635, 0.0068582513868374806, 0.029585104146657642, 0.033538582311795406,
                            0.02914645203410987, 0.04629791908908904, 0.02598217143991443, 0.04306919959710279,
                            0.018802804666965545, 0.030662399713235186],
                           [0.036457865768348884, 0.019737454969873135, 0.006707431008640726, 0.022580534885542812,
                            0.006265852706813922, 0.042484513980751884, 0.009827446039159478, 0.006535162116542452,
                            0.04873373930519299, 0.017968360951101398]]
            self.y_data = [[0.0009273854122466732], [0.001094952255392254], [0.0009452629595010824],
                           [0.0007679201720319299], [0.0008542013044527868], [0.00075902914962367],
                           [0.0008779675178084151], [0.0009107815768113685], [0.0005655125027020883],
                           [0.000992029327605792], [0.0007851225756764787], [0.0007544931430854854],
                           [0.0009671313033674343], [0.0006196150698161511], [0.0005292546202777633],
                           [0.0007214870793818802], [0.0007130735213138191], [0.0004685267894529637],
                           [0.0009066491359882736], [0.0008339438443283838], [0.000813432296758832],
                           [0.0009382886334371326], [0.0005345889339679398], [0.0006023433039280321],
                           [0.0009061727995751773], [0.0009252598670256578], [0.0009441542763215104],
                           [0.0009448651257459011], [0.0008645059638646516], [0.0010945369462356685],
                           [0.0009270971909655204], [0.0009727723317281483], [0.0008942397051562556],
                           [0.0009520121612245822], [0.0008352982296539092], [0.0009319612782060882],
                           [0.000855918506263346], [0.0009715912325588801], [0.0008784593005657973],
                           [0.0009693159810311305], [0.000930951778340703], [0.0005699532611037331],
                           [0.0009307436087590594], [0.0007575644256194425], [0.0007058268970006095],
                           [0.0007990768126473285], [0.0008772298224267072], [0.0006574779403412244],
                           [0.0006420520978883337], [0.0008896312364371768], [0.0008403452152503209],
                           [0.0011883353798279768], [0.0005608774674603995], [0.0007428828150991017],
                           [0.0009597451316074346], [0.0010333877434099302], [0.0010792810536224898],
                           [0.0009384700584185009], [0.0008057692556477114], [0.0010567954585035135],
                           [0.0011550402623947478], [0.0009578863052190499], [0.0009062470097947178],
                           [0.0007509930766420177], [0.001059535759775344], [0.0008358796706052595],
                           [0.0007098100396302569], [0.0007936713613718942], [0.0010314549686618194],
                           [0.0009487841126769122], [0.0007379263957942669], [0.0007003993745229473],
                           [0.000750239436494431], [0.0011582734923111394], [0.0007670751148777981],
                           [0.0008000732225452185], [0.0009119799036870152], [0.0007793964016131342],
                           [0.000707066835675234], [0.0008490100130786317], [0.0007122549760810213],
                           [0.0008581756124420828], [0.0007379498957059362], [0.0006441688859369715],
                           [0.0006353278540118829], [0.0008127088119610307], [0.0010669359679513323],
                           [0.0007066913599354095], [0.0008604135284181082], [0.0011105368496010424],
                           [0.0006279667014774805], [0.000854731867299582], [0.0009916410978333713],
                           [0.0009149723712394092], [0.0010192863609509812], [0.000974834424524773],
                           [0.0010091344929528555], [0.0010908585635974676], [0.000925672558032448],
                           [0.0007671713483288732]]
        elif set == "150":
            self.x_data = [[0.04470804274665621, 0.01675894714295027, 0.04091105322415139, 0.04269062722261752,
                            0.040266235516088995, 0.0394980095965184, 0.034158068944789725, 0.024795771478053525,
                            0.01404577556038698, 0.026733329395264304],
                           [0.021347099600228406, 0.013692833524277734, 0.010858713935914043, 0.04016861809410105,
                            0.03817482041391048, 0.02780001841706371, 0.03052821433422338, 0.013247439372687294,
                            0.04020758261613658, 0.03854915364410304],
                           [0.02295029474983862, 0.010349308633197746, 0.02037736320725573, 0.005016121107225027,
                            0.04636576840051327, 0.010418131509860987, 0.017577150287796536, 0.007530017261321473,
                            0.03363053284148507, 0.049759245442950246],
                           [0.007525760691695074, 0.030524062174798212, 0.01641793728065722, 0.023633942823810657,
                            0.007113618311579787, 0.04022999417016682, 0.045054421214752866, 0.034964281108436544,
                            0.029586612031378273, 0.04859667219361853],
                           [0.01027885160021461, 0.029859261576081292, 0.007220818141541654, 0.022202803634170723,
                            0.046642522743147605, 0.03099986267786942, 0.026457982190719508, 0.04928985997150786,
                            0.03678783255777759, 0.03815888819805368],
                           [0.030009424428038885, 0.01832395742672777, 0.006270094246711097, 0.0061098157706178805,
                            0.030205802596674106, 0.02877167058660055, 0.037055508275718636, 0.02669087754649531,
                            0.03311478452014784, 0.024955894205066335],
                           [0.04632457414014195, 0.04356480623253032, 0.018757690151111287, 0.049957745299726415,
                            0.018760810046328017, 0.04745688829762071, 0.018005118660009804, 0.0171326947212421,
                            0.03708025014653483, 0.021392435716639194],
                           [0.02977837769200059, 0.046487907980660055, 0.014362729532425598, 0.04934672204038583,
                            0.023954114572719714, 0.03979171321564502, 0.032437403287270954, 0.03966210657342239,
                            0.011405803298025874, 0.03637443784815817],
                           [0.008737680753024476, 0.018123752950354653, 0.021739293890426, 0.01625846902169058,
                            0.03322049913399747, 0.021424827758966238, 0.01690398019840454, 0.04364858539835993,
                            0.006422408913921283, 0.015621479400146514],
                           [0.02596507471785939, 0.027684411345747718, 0.00952064004113712, 0.02312612921365413,
                            0.03271124966006184, 0.02818255398996511, 0.04623873232425587, 0.011536242909444529,
                            0.026809984777285356, 0.04750307842612305],
                           [0.04884778235578856, 0.04836726233645144, 0.028478756909574163, 0.02343989090978705,
                            0.030835951989629566, 0.03507401817000661, 0.04053047034016722, 0.022826196566934787,
                            0.023964129773833998, 0.034326619418201976],
                           [0.028947238573872568, 0.04270956432310709, 0.02476182214566081, 0.045127760861253044,
                            0.03456337156752625, 0.034325834829300704, 0.019356853266627765, 0.023050567370214634,
                            0.021526184629878344, 0.041953758395070526],
                           [0.025481365859466976, 0.01129749523737137, 0.014827943422398182, 0.01261599794003572,
                            0.03379329956961125, 0.047703276657466584, 0.012088793881758632, 0.018007209730756167,
                            0.020053960011540993, 0.018077476543319145],
                           [0.04559398670312718, 0.011536455748392474, 0.025530554493347866, 0.044346163396100036,
                            0.029959083828891137, 0.043427933075800555, 0.015558906244717785, 0.028596036548952933,
                            0.046349878914787083, 0.022993826309060033],
                           [0.039942599095131745, 0.022014519335439187, 0.024883744960440428, 0.021823433035719684,
                            0.01573630498181107, 0.03188598488757594, 0.03341973048029049, 0.03581579848905422,
                            0.03516148441222226, 0.04302580677177693],
                           [0.030391169953340336, 0.024334154344473297, 0.010437898414933183, 0.030943594698415358,
                            0.012123726198701796, 0.007375370625289824, 0.01761777734694694, 0.013602468130908641,
                            0.0060097577203535955, 0.03776325857457347],
                           [0.02469129750710315, 0.01327612623740277, 0.02645643883824257, 0.04097116855438236,
                            0.006447582360152904, 0.008120722609317241, 0.018638574612918304, 0.04142881501547103,
                            0.016375627988236195, 0.0391106291509003],
                           [0.007139948559045402, 0.038668609041056255, 0.03860114571715498, 0.02915238993421517,
                            0.009354043082453214, 0.01869832031266404, 0.04287354853389851, 0.010085228979675752,
                            0.0229220591259431, 0.0065589545494063],
                           [0.04284798956180401, 0.007419404843789777, 0.04781571641709398, 0.04670664008493839,
                            0.02878334050721973, 0.040979288374497165, 0.04006218894784192, 0.02766586914464109,
                            0.009721614835363851, 0.044173075114993014],
                           [0.02259461979249904, 0.020481923548926747, 0.025151400784554946, 0.012215391039845046,
                            0.005703706272780083, 0.038073410340349535, 0.02296342696723227, 0.04268293478197114,
                            0.02240822008495452, 0.0233629631952485],
                           [0.017976575819271274, 0.049064214162617345, 0.032865614084114265, 0.020759914104490306,
                            0.005424319337163421, 0.009248069171574572, 0.013324247351559102, 0.01994882628416113,
                            0.01888801980400694, 0.005171809645886625],
                           [0.0452229296945461, 0.01491851074923933, 0.04550867202053423, 0.008892881975380781,
                            0.017874652240056677, 0.0365902252793594, 0.007607356177276363, 0.03984225358171302,
                            0.049213517333215044, 0.033820702876183995],
                           [0.021587545536955337, 0.02814634627097371, 0.027722033781998806, 0.005594409209360801,
                            0.03836751325071631, 0.04545489143291218, 0.029640092397792625, 0.018793046813221644,
                            0.029197405913978233, 0.0063373113688676205],
                           [0.020971407961265547, 0.04466302121221236, 0.04226389831470292, 0.022756055914158614,
                            0.03897171220155689, 0.0179605260676766, 0.035064976903594086, 0.014099266641812932,
                            0.008163414222427386, 0.030067522086196723],
                           [0.03705011365477002, 0.017458235172720684, 0.0336779410964496, 0.04778037623686229,
                            0.02154404270760498, 0.04234110229117527, 0.02958172806717995, 0.02588407415821486,
                            0.03108216366752357, 0.02409754325494044],
                           [0.03816312513980555, 0.023493503905597225, 0.03872165761640065, 0.037431215677151776,
                            0.03979620328396185, 0.04323044242097009, 0.041379042009568295, 0.03717902459923599,
                            0.04404086328894153, 0.023161161244520678],
                           [0.009110329356400979, 0.04856382008425761, 0.013909071239880028, 0.02124496757758272,
                            0.03783828156112636, 0.006202905827306777, 0.026907905575566438, 0.008755602733047874,
                            0.012995873586380425, 0.043510720481729825],
                           [0.03762491323357706, 0.04635886324357529, 0.01183786890110069, 0.030751245411372925,
                            0.016094255090469765, 0.04890835560721981, 0.03793436899338571, 0.025271373804312464,
                            0.020999923604380914, 0.023610334296679518],
                           [0.02087201441207563, 0.019643300772990608, 0.049126270460336366, 0.01312881430915537,
                            0.02815597720914833, 0.007687300920067991, 0.03444590452621717, 0.034120003436620444,
                            0.04745053358674876, 0.012001258890391955],
                           [0.037130229517265995, 0.029328774536585882, 0.01232688900020613, 0.008217272552541071,
                            0.04437733624245102, 0.044552329518465245, 0.020831740016102794, 0.00794837320501501,
                            0.014419439858980265, 0.03103607748129727],
                           [0.029153506829768416, 0.02712325058200128, 0.02081051543076029, 0.04428905014280394,
                            0.02649090607216977, 0.04619110256935117, 0.014181351416894753, 0.005674918479466579,
                            0.029776431473430797, 0.048291373772041915],
                           [0.041711339878466036, 0.01702421381056467, 0.047008557979546624, 0.01603079025024034,
                            0.04417761527231235, 0.04132425087517088, 0.030920464140064968, 0.032713947066318715,
                            0.007923893652665059, 0.013341699313862303],
                           [0.01442068197822691, 0.04503306486735558, 0.006772365933185668, 0.019552170267561276,
                            0.041883044244614094, 0.006823524373859873, 0.02866898501918285, 0.04983562269817458,
                            0.010938027857842128, 0.025311996147547856],
                           [0.043223251456985655, 0.04442143070100965, 0.02727720287926608, 0.0397298320045866,
                            0.02945773424803808, 0.032873837677468896, 0.019055281917321306, 0.040293179857468775,
                            0.041266010424882836, 0.03199761912018197],
                           [0.012147022094165807, 0.023778999485227142, 0.03149520159475017, 0.005638439050891248,
                            0.019431409280627317, 0.03490491743724148, 0.030391125016361396, 0.022705269231185866,
                            0.043736978879648045, 0.00549006828213449],
                           [0.01671879618511651, 0.021790532045294027, 0.048428034227933896, 0.028781688880233018,
                            0.012606085287996678, 0.03621502569553174, 0.042049573708764006, 0.005134752378742002,
                            0.016946896165740456, 0.04727595129798963],
                           [0.03055938230564852, 0.006559964397108011, 0.03220748905923947, 0.04572062413972676,
                            0.02511802042747522, 0.022454385204544677, 0.0248036607349567, 0.0484048091199509,
                            0.012520425781928062, 0.03967553649296339],
                           [0.028560442381136045, 0.049215869622852376, 0.012919031273602118, 0.015022136502378002,
                            0.023014817501861306, 0.01017664359265759, 0.042385993168283745, 0.044541709671472494,
                            0.007265136503736173, 0.006113798225802614],
                           [0.005729076372751687, 0.033852282506078434, 0.03938549482822907, 0.047498325750204906,
                            0.019387915806588562, 0.01114870997177382, 0.005082048744430185, 0.030249411513734382,
                            0.0181493844160761, 0.00958109819408377],
                           [0.01465341842910637, 0.04119580990400779, 0.0074645997147039485, 0.02437626011663613,
                            0.04873080774744702, 0.021933280254314444, 0.005843145766532928, 0.03604777707055195,
                            0.03922833897944995, 0.01060139791862743],
                           [0.015134450023427233, 0.015278874313217831, 0.043599530490808884, 0.04228755530112638,
                            0.029780091428347606, 0.005350646092115184, 0.024870451179433782, 0.014860989811467808,
                            0.01196312617607958, 0.02019620458822061],
                           [0.00992625807762464, 0.03320819843274303, 0.04971625870402671, 0.03372640078473902,
                            0.04700040147620024, 0.012586264740363777, 0.0199796763842587, 0.03359506244703677,
                            0.009136784892162414, 0.021591926306532713],
                           [0.02710139635525141, 0.025712949046224086, 0.033450327248920726, 0.039400159291924516,
                            0.021468191613985662, 0.03034880146327007, 0.021621840862758583, 0.028358821761011184,
                            0.045944091461426924, 0.012504383267671334],
                           [0.03575334022636758, 0.03803460426580735, 0.04651758389230217, 0.020103021121393748,
                            0.045650743766627554, 0.03338337267626236, 0.021119614557842437, 0.019209418331938764,
                            0.03182422440573492, 0.048199358162657246],
                           [0.038596255511756945, 0.049456442113164155, 0.005987261897843613, 0.041841823628959565,
                            0.025984301771555546, 0.022244641463178397, 0.038454759136286466, 0.036538302789970334,
                            0.04690631666835294, 0.017219606519892756],
                           [0.03380421668310999, 0.04749589767799233, 0.011436057645787319, 0.03129464038337537,
                            0.012809280429183183, 0.02699439410259838, 0.03713408431725252, 0.00898752705324062,
                            0.024692948901869627, 0.008622261402729292],
                           [0.01886368624213717, 0.03370555515446954, 0.044536231564222685, 0.01883490036862139,
                            0.013442205900834111, 0.049692080899552724, 0.033950266695104765, 0.04294587596767699,
                            0.023531671919414382, 0.04943199855631295],
                           [0.010533229866457439, 0.01098327723028195, 0.02020391197237731, 0.01204679313462861,
                            0.0446445430596961, 0.029775809126298944, 0.024363904467366464, 0.00962428782970529,
                            0.039932369035146006, 0.022037974964601366],
                           [0.03155041194289281, 0.025369615125567232, 0.038967069128597785, 0.032591075332302,
                            0.03700463617396247, 0.01615672707922181, 0.039720278946244796, 0.020063332084273335,
                            0.011249890145324095, 0.03221877990862635],
                           [0.03475655202660904, 0.026667529214347327, 0.026632176833897337, 0.01106500231852195,
                            0.005997814459797327, 0.018837457000107995, 0.01001432500432378, 0.010915066849590586,
                            0.00569764437853537, 0.04154159086513517],
                           [0.02758979602356371, 0.04322302992051839, 0.013256954501528011, 0.03886316164834853,
                            0.01623869877089016, 0.035724294549143325, 0.026843436091377935, 0.01625722166879469,
                            0.045187835042717714, 0.020405711573550568],
                           [0.030845827538677036, 0.011654410954428089, 0.03630746079821507, 0.017099921629672814,
                            0.0221146356793554, 0.009562747041054783, 0.04694455187517307, 0.011978146213358375,
                            0.034717628625567036, 0.025919051927611746],
                           [0.026204270892783064, 0.04203666878272831, 0.04315733928808918, 0.034617415847537095,
                            0.0181265886468224, 0.015540242999404007, 0.008346391453825902, 0.006610006687820072,
                            0.014952663250596205, 0.01762955911374445],
                           [0.03770327489522261, 0.030100312730691872, 0.041631960525601114, 0.028131037758897134,
                            0.026296473391747392, 0.020328381486751045, 0.03679507418546243, 0.010677674068164956,
                            0.02229582387825059, 0.032700421943968536],
                           [0.0135564628605651, 0.03743625159051217, 0.018367199136171845, 0.0294266288358868,
                            0.028087606973708937, 0.04214598230398739, 0.006986329254872222, 0.031151474966413416,
                            0.03965250144386015, 0.042632712622887],
                           [0.011550381681098686, 0.03648247932493672, 0.018110852436568146, 0.03324117731659431,
                            0.04915104239060432, 0.007995016315097357, 0.03541301331060757, 0.04530258950843384,
                            0.032030951843922725, 0.042382451734187664],
                           [0.04349959366531904, 0.03491626224416053, 0.023253923046454756, 0.027332097268801033,
                            0.046099984875854115, 0.04647879804686025, 0.04368357777413271, 0.04685438877713222,
                            0.02058521262066504, 0.04629603422774015],
                           [0.009419171490466022, 0.03834267865098626, 0.020939720315521583, 0.016945291474217306,
                            0.03293784799290432, 0.03543955801027203, 0.009307332125343612, 0.04623035195427178,
                            0.0497161558637889, 0.0324934802020792],
                           [0.010872648444777852, 0.032968076555119276, 0.033035009784515984, 0.01561766153728112,
                            0.043819203659571114, 0.02167678807750763, 0.03611887187350933, 0.036313107393989366,
                            0.03740512053745808, 0.04318135239011207],
                           [0.04441586373569742, 0.02418569473407162, 0.03486352965992928, 0.028508341155910086,
                            0.03138262792897285, 0.04591160216563045, 0.033791150893469867, 0.041896232531331075,
                            0.006942957360556032, 0.016949209846859336],
                           [0.02181162216092591, 0.014401061519769977, 0.01683220057871076, 0.03708910110569571,
                            0.031454580311862525, 0.030008085458841587, 0.044372324616541864, 0.04487208685123734,
                            0.01920521899077806, 0.04558869713186643],
                           [0.032675814854628275, 0.028594863831669887, 0.04888322070007572, 0.04864765746658108,
                            0.02492887422677309, 0.0406323275685973, 0.043859363755143176, 0.021426628295795527,
                            0.030602048596316454, 0.024577396310399405],
                           [0.03877450893917675, 0.03591927198482593, 0.008830035785894003, 0.007478652357945922,
                            0.011113192423847392, 0.019517194347672163, 0.009620222725508932, 0.04720454752690017,
                            0.037202765778177264, 0.029310229486642764],
                           [0.018398408283921756, 0.010692079288742154, 0.029769372731617255, 0.02782541047600744,
                            0.009618235641219941, 0.025499231543726623, 0.01125209228096182, 0.03840869415373132,
                            0.024434851988468897, 0.015208699564354805],
                           [0.03224144330661498, 0.02482423822081881, 0.015010620273489164, 0.03180831267109008,
                            0.006951317742838244, 0.011785428605688597, 0.01823851559380249, 0.008336645918871224,
                            0.020777632881367334, 0.016022659694892888],
                           [0.03539596693363765, 0.03523543869302036, 0.03700693750057935, 0.03824512504280942,
                            0.04248910341632882, 0.03390287365922948, 0.04823228584871993, 0.015459907768619625,
                            0.04561277201972616, 0.01756319905809239],
                           [0.04124804069836674, 0.022907441232599586, 0.043046718930146284, 0.018608637785572757,
                            0.04040804453520763, 0.012045300025542608, 0.039444882503735755, 0.03150309208787396,
                            0.005412233635976061, 0.012621557951566697],
                           [0.01711934020653899, 0.03775064365348753, 0.02124200466439789, 0.013459747292332933,
                            0.037446048594173144, 0.029569856449636753, 0.0477429134139061, 0.03292968188633524,
                            0.04769142915360266, 0.013422280205577508],
                           [0.03119660807263012, 0.026552746845728645, 0.01567289985334788, 0.01425535715865242,
                            0.025603471173257862, 0.0100032250891261, 0.009039282749550272, 0.04402559873084114,
                            0.04097623847216861, 0.014018958002319622],
                           [0.03945316869588442, 0.04030423440875908, 0.034307644341677, 0.007805773521999658,
                            0.04333599405761814, 0.02311487083073188, 0.042717245103507884, 0.017668622955665264,
                            0.030421228289002476, 0.04079192042841692],
                           [0.01371134722152385, 0.029216676291768936, 0.022315006428498204, 0.015377006620120655,
                            0.042558955390247295, 0.04007701890869613, 0.02046898351400351, 0.02995496604284343,
                            0.01584165906155066, 0.01861314220764881],
                           [0.014118601908739762, 0.03920278882692765, 0.03548729067931918, 0.007364774199324934,
                            0.01971916574352279, 0.04702393485495605, 0.025596108639154524, 0.03244515410951638,
                            0.03465762150880551, 0.01634677340489119],
                           [0.024845224935842448, 0.023215628239116343, 0.034406134776773616, 0.03982152368008767,
                            0.011545390745928374, 0.038311087585670756, 0.048969965958449824, 0.015673077042180728,
                            0.04140807521942313, 0.04382381563694535],
                           [0.006529049879494122, 0.01485912733675793, 0.019403447856673178, 0.02417572240550904,
                            0.012313284234334648, 0.017489298567708908, 0.008823530272276186, 0.02438731297714161,
                            0.04224208792833546, 0.044795401068335745],
                           [0.04972924789155607, 0.04537194360884473, 0.032377975540612954, 0.049095996844764456,
                            0.007730845976824538, 0.024271947508202205, 0.012801426916889978, 0.04231897621913465,
                            0.01181625196085554, 0.022267575276248276],
                           [0.013405231542261465, 0.006036947555522683, 0.035611357032638034, 0.04108705098069662,
                            0.048079147077252686, 0.04392404055584039, 0.02763581767066672, 0.028978158073623975,
                            0.036288814124735086, 0.016489309340240295],
                           [0.04963424570728969, 0.009488508096217985, 0.005471068529967285, 0.01373325682880755,
                            0.014344196366764032, 0.006302567146303052, 0.027355826963919933, 0.023850550823536813,
                            0.03164006316536183, 0.02249744554408794],
                           [0.005392391728392798, 0.016275513411572434, 0.018890431600999905, 0.025561608601956082,
                            0.01678816104263951, 0.015870634488185327, 0.03919001867711604, 0.029832917206592802,
                            0.04298297072665451, 0.028308742549158786],
                           [0.034150333115807334, 0.016024627470074933, 0.009048991923506979, 0.010213995373192295,
                            0.036110382550426805, 0.04103869064610139, 0.02225831275355708, 0.02784184060958417,
                            0.0327138661646265, 0.007285725197389152],
                           [0.018787957246806312, 0.03434977468182983, 0.02300295835282325, 0.03505267520659572,
                            0.021810720684335812, 0.01094757827275903, 0.007733606144039814, 0.031753669006539476,
                            0.028213913281400774, 0.03763864039667972],
                           [0.008049676195975769, 0.03559477887915821, 0.030209161853353917, 0.048095481091682005,
                            0.018859507735309748, 0.015225507636178408, 0.010658313510629971, 0.04120176861558674,
                            0.010627807665651471, 0.005730229002152249],
                           [0.008443904557094414, 0.006315368074535868, 0.02901867390359707, 0.04539441829729661,
                            0.0319627682863088, 0.036959619624514835, 0.01968315001599621, 0.04052788736026777,
                            0.038935629529509894, 0.04761582607539282],
                           [0.020087160423738773, 0.016431123193681528, 0.04044307342322872, 0.010673787436961728,
                            0.04958335813286931, 0.048543665977502634, 0.02876323281634736, 0.0184940005503281,
                            0.019785657794700626, 0.04070107018881252],
                           [0.01957646330573697, 0.03561075413292498, 0.005696349083678442, 0.04617761661335474,
                            0.035025292607587306, 0.020855126790336128, 0.02921375672275132, 0.03793707489762785,
                            0.047958698455256674, 0.013057187629370113],
                           [0.028284990839121883, 0.007923949541649001, 0.008373696871724387, 0.014800356594006327,
                            0.014731427727949116, 0.04296512640774306, 0.013819671771518195, 0.020962624170118738,
                            0.04266406639767483, 0.04450263648918641],
                           [0.03443494379854305, 0.01895057156314707, 0.0427547275711351, 0.037153005946159494,
                            0.01709480222104965, 0.04815454887164471, 0.03494910718932012, 0.026534605824947155,
                            0.03418597079734508, 0.014904385343307493],
                           [0.02658109896187772, 0.024654885459808283, 0.046996915841740976, 0.043450609257588445,
                            0.020122206245348835, 0.023849297860048507, 0.024070032797821286, 0.008148486320850905,
                            0.013427910539285447, 0.026513660789841263],
                           [0.02741129735231516, 0.0210100817642128, 0.04941948686847452, 0.016435765727103452,
                            0.008967362276818979, 0.028020368951689866, 0.028189457974590017, 0.02410401487098539,
                            0.0324579126682457, 0.0209679686073798],
                           [0.023593018195424333, 0.04563220533756545, 0.01621519524102706, 0.026471971372284152,
                            0.03679461362800306, 0.013354349002029295, 0.03020241608825237, 0.024958052043543857,
                            0.049667564062376296, 0.013722063291164807],
                           [0.03350628049308267, 0.04377802091170404, 0.027154711398945056, 0.018461190821418786,
                            0.013732254316152041, 0.037413187906712365, 0.014516837596096888, 0.038265994059399,
                            0.030062288462939412, 0.03602037212148996],
                           [0.0063345334494456585, 0.030994622692404942, 0.005039188117582543, 0.009919607569979717,
                            0.032406504222838625, 0.017147919061215034, 0.006592299489439577, 0.047658759816839875,
                            0.0452580834210472, 0.027772343203964867],
                           [0.017422887285806238, 0.005637118898223537, 0.01730906617233218, 0.034108440188323395,
                            0.04786678777539596, 0.014826292058822858, 0.048667722770415194, 0.030994946647419988,
                            0.005259634801234747, 0.007656701120262942],
                           [0.022145642062552613, 0.01204507477991085, 0.037950732199966995, 0.011531327999774865,
                            0.010357587954726154, 0.030588423766539568, 0.0314193141823028, 0.02732102200681336,
                            0.04481632556586923, 0.03312717826497702],
                           [0.046837670168901765, 0.008983741961383775, 0.02343557216046215, 0.024692961488704604,
                            0.03425567303656192, 0.04277866408811166, 0.013587560670532135, 0.02225321186768186,
                            0.04430765231760403, 0.008161939963216127],
                           [0.005209815110046446, 0.027956995327271725, 0.011220281335689818, 0.04385015982828784,
                            0.033805115148574504, 0.026474298837584696, 0.041103067395522934, 0.019476708567048133,
                            0.023171393160985744, 0.009423444975243401],
                           [0.009651856031044376, 0.04241398108233866, 0.022533387168614635, 0.046677014300661915,
                            0.006761121185561763, 0.032241612040098976, 0.014767839138593291, 0.035248573723884076,
                            0.0333845630188186, 0.046521546285828046],
                           [0.0472441507541136, 0.031145663812486783, 0.015982789303990864, 0.009715214005119027,
                            0.03487576283701568, 0.048225033841380736, 0.04323171445992437, 0.026158192578000096,
                            0.025314871786374173, 0.041263378600768466],
                           [0.012499232775497327, 0.04156751081673573, 0.03078001346991654, 0.02688218485838332,
                            0.008642312238011193, 0.021102088382706018, 0.04485973354143162, 0.034679867139360304,
                            0.008869070203575706, 0.033474964660854895],
                           [0.04276293150006078, 0.009673838577113927, 0.01524566456975978, 0.03408755979808494,
                            0.010875045727654659, 0.049754463428832975, 0.03285194374861882, 0.043724815652704134,
                            0.031177940973353333, 0.03452892371712064],
                           [0.02047761865861951, 0.014021193561074466, 0.03957626140251938, 0.02696315378639328,
                            0.018447727587126473, 0.039019184046593575, 0.015096658346694634, 0.013716972214679442,
                            0.024874720811807204, 0.030719297645982527],
                           [0.04193124846711448, 0.031942642784433366, 0.048143686106473654, 0.049541888334626694,
                            0.04074575215726398, 0.026814063094439837, 0.03640912284622729, 0.016810332349631786,
                            0.009947982973786896, 0.03671418133601653],
                           [0.03666115058017561, 0.03720665354071811, 0.024311170793753657, 0.013054869545677021,
                            0.03861427089476647, 0.04418871118853085, 0.04792184484967875, 0.045718145932427186,
                            0.04334445647699189, 0.024430644746579147],
                           [0.007841381673390088, 0.013882231395888501, 0.009955752160780916, 0.03667161004000991,
                            0.02747975751463275, 0.04160533012944409, 0.03758109250645578, 0.015867797332115297,
                            0.010278138228087732, 0.031375905220835214],
                           [0.04597484950314745, 0.04673369656783639, 0.02574586703725004, 0.04713622231077389,
                            0.02277804634014903, 0.0052223132636590765, 0.02027957647004173, 0.01737176656895366,
                            0.028024569480387968, 0.035438107951506446],
                           [0.026669741875103466, 0.04817377443621387, 0.045207257236727444, 0.032113732999311154,
                            0.04368889446270241, 0.016739218382493048, 0.012848735276280637, 0.011259282560200088,
                            0.027262185255039015, 0.04026326144060722],
                           [0.043912618930973056, 0.039031385752146985, 0.046113890472429135, 0.048228581369618266,
                            0.03209964205528326, 0.0346778302707635, 0.01037909986754149, 0.0071367903616003725,
                            0.026069568021148855, 0.03505149144353102],
                           [0.023164610148700332, 0.03269229792265722, 0.045928302700949906, 0.025958132655491486,
                            0.04991953163279133, 0.03140293403759932, 0.032204883592300164, 0.04509326421483036,
                            0.046406138326403244, 0.007730416004392395],
                           [0.0477991662742815, 0.008679207908054612, 0.04429750017526891, 0.025083185893477907,
                            0.0428758850562646, 0.03737035124769543, 0.005569557785982064, 0.03216238585271082,
                            0.006628032784530789, 0.018323568430489993],
                           [0.019262374808154098, 0.012833818279782742, 0.008283644729330889, 0.04046524230122279,
                            0.0488112953558631, 0.029019056824519048, 0.023728519027840952, 0.04793900641908842,
                            0.013222231883226077, 0.02562408517050085],
                           [0.015453150067600615, 0.04069622065558595, 0.014056292090297212, 0.04588524568344891,
                            0.024530556256983752, 0.008825117190243726, 0.015437423133288216, 0.02357632232879356,
                            0.038165388414329544, 0.008601739129128082],
                           [0.013060461597572154, 0.01263987833788819, 0.010222119143774177, 0.026148372935133012,
                            0.035487059671790736, 0.013747073554411519, 0.049753974873879246, 0.01189451605234347,
                            0.038428890929593736, 0.019824425336589475],
                           [0.04805657431654362, 0.03710242310052826, 0.03808918977598022, 0.00837586538142166,
                            0.037394975390219395, 0.01492729649984706, 0.011622145399292527, 0.020906424261310813,
                            0.018337199544235974, 0.046712132133197914],
                           [0.011799009394238464, 0.03024916083115945, 0.02894936588959813, 0.03848139436423353,
                            0.04722083635823392, 0.033642476392535765, 0.0260414401929062, 0.029058697502192966,
                            0.033821476167727724, 0.019437282317519285],
                           [0.027960219980613835, 0.012435120180529738, 0.022035795681173035, 0.03261820755123639,
                            0.020517423199184346, 0.028483200269806584, 0.04020783261939752, 0.025672654633553883,
                            0.008480068612732625, 0.020782540013600758],
                           [0.016416615400298065, 0.04306444378625546, 0.03675444493376332, 0.03488183013458737,
                            0.027734026221136673, 0.014164393191413986, 0.006260417333687179, 0.00631517053764149,
                            0.048597822059200976, 0.04914275789240588],
                           [0.03957014380206539, 0.015736246776219178, 0.030054566827494113, 0.017504872530363775,
                            0.009979810606161415, 0.0067733294630661, 0.04948967056763749, 0.036956100198540356,
                            0.04703220479173708, 0.015102681327946815],
                           [0.04644912621081559, 0.031590315067737934, 0.006951116526584553, 0.01784284584177653,
                            0.02905831163937563, 0.026264514062318736, 0.025809184920333753, 0.043350880142627536,
                            0.026425328018473897, 0.03986447838858505],
                           [0.048506028827393015, 0.0053140004686063605, 0.037208212760198686, 0.02059634921441724,
                            0.03634835274468746, 0.027345416765204616, 0.010898320534640682, 0.03537707259316205,
                            0.04822934620088565, 0.010719563992599369],
                           [0.016402828785040267, 0.04774439864717819, 0.04022363339972134, 0.03773114262852663,
                            0.013402130513269203, 0.019709404603456635, 0.03179782845730127, 0.020604380124290597,
                            0.028840655406196958, 0.02695120658125639],
                           [0.024141986554849577, 0.020054863568111727, 0.044679890751744424, 0.036089644457974135,
                            0.017369117647074203, 0.033114998896704946, 0.04656605019857303, 0.03407511938028476,
                            0.019576231355764466, 0.03925734670692772],
                           [0.025155197100589267, 0.046087093899848025, 0.029421193794521596, 0.006605003962342967,
                            0.04128604321038246, 0.023471363394662367, 0.023401273401980612, 0.03337997432803329,
                            0.018561107733517514, 0.02747927604238987],
                           [0.006987186226353571, 0.008596532494014271, 0.041906501346455495, 0.041521365695056996,
                            0.01543255362417141, 0.0378913572738207, 0.03879167651433359, 0.04216171147636367,
                            0.015212040226800793, 0.009874369437795256],
                           [0.019743068316673572, 0.02601606948115828, 0.03519389573154373, 0.019925133891975092,
                            0.027060441575354824, 0.02503086798786222, 0.016296256700913867, 0.03929871728738278,
                            0.028535110721339605, 0.028704041326023968],
                           [0.03650166421858382, 0.044207651831639945, 0.04861821959895817, 0.0448069251138452,
                            0.015197779538916378, 0.00903920552216297, 0.04587186217672871, 0.021964106049079286,
                            0.02554333663266511, 0.03874790553393451],
                           [0.006199112341967037, 0.0366983609984212, 0.007904217780371243, 0.04306032864317771,
                            0.035736807376375676, 0.014560718259790134, 0.04172344930539537, 0.0055672527215542295,
                            0.02133129186494306, 0.00908495838338337],
                           [0.04088194209911126, 0.04080354120267771, 0.037562284216382386, 0.014381740982450465,
                            0.020622703674346887, 0.025803069193078475, 0.03800669905728131, 0.00922905011609106,
                            0.016418798451749008, 0.02783396853204464],
                           [0.04228766491972371, 0.008234169371723454, 0.009253976579311367, 0.029882159060247823,
                            0.04147108859603244, 0.016441550130889757, 0.012249839101201887, 0.006846452592683254,
                            0.017700502461286658, 0.04175107421564985],
                           [0.04501314144965794, 0.007024927358600417, 0.01213492634543993, 0.008928767533500618,
                            0.016498301005544377, 0.013104012754580344, 0.03305986845629844, 0.04582110479965121,
                            0.0406465745903196, 0.03706398119781613],
                           [0.04065300252364106, 0.022469034077309186, 0.047458063432079894, 0.03905449086477074,
                            0.02677598011065319, 0.012390153204091571, 0.04718906911489175, 0.037544785147389376,
                            0.021867967743587492, 0.04520185136185582],
                           [0.047303530535782325, 0.04169523685441804, 0.04494185146503847, 0.027548125810916296,
                            0.010657934220455082, 0.045747967059882745, 0.03577690062665747, 0.03875910820934567,
                            0.025820891782909815, 0.030260221449942674],
                           [0.023694370447308263, 0.017700801834393324, 0.012508721307176545, 0.011780912308053702,
                            0.023672051169489852, 0.032407665702456706, 0.015957141993726536, 0.04907935398078081,
                            0.01392127037550384, 0.035641728314741335],
                           [0.01596306025943312, 0.021332935915984973, 0.028003139763451725, 0.030269139412021032,
                            0.024339595946687928, 0.022805105679002288, 0.022639259830130872, 0.014984649232971659,
                            0.03773739161125947, 0.0335898319500477],
                           [0.03237469301649474, 0.039603666832931046, 0.04380358929234709, 0.03305815649103639,
                            0.011749007770352482, 0.019135993380704337, 0.011404961660113773, 0.012224392204494665,
                            0.027532235696041286, 0.01125173390429347],
                           [0.017744736901403806, 0.028998141156741104, 0.039961265430073035, 0.010740826659150503,
                            0.04547590937035727, 0.03888848659996632, 0.016575380113375844, 0.005989510603611082,
                            0.017379936538521212, 0.03494611633600057],
                           [0.0493515762118646, 0.019900243751646698, 0.01362833682679843, 0.007032977859931528,
                            0.02099295850259612, 0.01842753725226007, 0.00729539146518517, 0.016568820390338683,
                            0.02704343373974173, 0.029170696665738367],
                           [0.041506220892043214, 0.025542421303024178, 0.024171253010808857, 0.03566441954882199,
                            0.04203222552435367, 0.03149371488141969, 0.04548273642177377, 0.038932893657581194,
                            0.01484470382766398, 0.019216864421788896],
                           [0.044153316575261774, 0.03988217308010245, 0.03399475125558405, 0.01928421207641484,
                            0.022615223795423217, 0.03617874969734088, 0.02136830502996556, 0.03078015395829013,
                            0.04881717300114743, 0.014584067878565869],
                           [0.01556672128188048, 0.007242730287756653, 0.017071393610426613, 0.04319451958607872,
                            0.04490349267491493, 0.024182627811260328, 0.04733507478312566, 0.013064453668160905,
                            0.04367485600517349, 0.04494631184871196],
                           [0.048494755805843046, 0.032253758950625266, 0.026114126278461033, 0.03148620802464106,
                            0.008475563018087485, 0.025211736110767637, 0.02301538660080616, 0.027119187234668937,
                            0.04205742043643, 0.029682965119671587],
                           [0.029360430625349824, 0.047040730695817184, 0.04152337434271506, 0.03537480207185055,
                            0.0284224468480928, 0.02016255633415921, 0.01717557154462377, 0.047524418417748515,
                            0.03887408842463915, 0.011621128885305143],
                           [0.03316008822236858, 0.005134911022206023, 0.019463768058067626, 0.021635481700497037,
                            0.048315795212238735, 0.024550143995412402, 0.00818659236634499, 0.0465176033905918,
                            0.015641368017839403, 0.010205908648740165],
                           [0.012586138944132918, 0.02223496428682407, 0.0199769134719895, 0.04192056350729714,
                            0.02332162130191587, 0.04502349151338821, 0.04411559370109325, 0.04081917600954838,
                            0.03550889837357161, 0.02879444600037573],
                           [0.024261433204612975, 0.049971329022497374, 0.03113331011938353, 0.025257908755129774,
                            0.007626698184045934, 0.017696068853277326, 0.031390989976780484, 0.029568158195057388,
                            0.035722423963050684, 0.03730359671540024],
                           [0.036026899602160364, 0.03237732741172943, 0.0360802525671986, 0.0064964206554194775,
                            0.04736584644268742, 0.005730956965498877, 0.04919359424271477, 0.048563652773024546,
                            0.01244249930740184, 0.006883259153214787],
                           [0.03508242939870623, 0.010080542733865085, 0.03194701194733885, 0.02253266240681011,
                            0.03065945856229284, 0.011545368160667854, 0.028004057366130108, 0.021547407916795748,
                            0.017237937717307555, 0.04887215312275277],
                           [0.011053976503428245, 0.02062061332215232, 0.04126576384725308, 0.03634831805231265,
                            0.014215348778127607, 0.044800201864359006, 0.006173577569578576, 0.010343877015361852,
                            0.009365816298582349, 0.03168482992196253],
                           [0.04032364060027171, 0.03446527833868401, 0.023640776116476796, 0.009418282982761927,
                            0.03928989678383293, 0.013635518766314619, 0.04577987314877338, 0.014500424546339912,
                            0.007626955466584576, 0.01893534668939093],
                           [0.031807933782836195, 0.019125938866605803, 0.028359937533567397, 0.029926542150627938,
                            0.005279168265274256, 0.04681437583577237, 0.021934686488140894, 0.018874424137403204,
                            0.02380601339481336, 0.026220100919893057],
                           [0.03902680290347149, 0.018584713853213797, 0.031077576434120367, 0.02118973491556917,
                            0.008021537505463285, 0.008321922934248664, 0.040928658292820656, 0.04949362754245803,
                            0.04176796542599698, 0.011445071345660602],
                           [0.033304687141433045, 0.0274367122327329, 0.01788599595640225, 0.017911330780732655,
                            0.03990679774727526, 0.04920557862610335, 0.025153410749957535, 0.012685215929799074,
                            0.0360654859092087, 0.046063908251590874]]
            self.y_data = [[0.0010513906465791584], [0.0007572528727593507], [0.0006090001979030312],
                           [0.0006492715170522156], [0.0007187371589402677], [0.0006625236947640185],
                           [0.001004256844373366], [0.0010002196212914691], [0.0006839492660402516],
                           [0.0007955768481118044], [0.0010736134781399172], [0.0010126171136197193],
                           [0.0006164578410574846], [0.0009436022527590363], [0.0009176819803647342],
                           [0.000672304030810635], [0.000728025127465586], [0.0007284193613200438],
                           [0.0009841644802988817], [0.0007364526202352069], [0.0007953740987730012],
                           [0.0008156041288777845], [0.0008888848199829049], [0.0009614483939505408],
                           [0.0010010288888928128], [0.0011602364021351735], [0.0006504948092963444],
                           [0.0009210946814347165], [0.0008061094450811789], [0.0007743224431968398],
                           [0.000976790028440539], [0.0009969192860070416], [0.0006868398603487421],
                           [0.00110565728122876], [0.0006600435175168706], [0.0008743053219293259],
                           [0.0008626022008991462], [0.0007965458123598094], [0.00087986909866938],
                           [0.0007140243099886903], [0.0007885300761682327], [0.0008602558794205282],
                           [0.0009825691116462376], [0.0010408899575548], [0.0009383955451858003],
                           [0.0008161612689812634], [0.0009255305931864902], [0.0005792981184799269],
                           [0.0009673550477329639], [0.000817638629377021], [0.000891690142227115],
                           [0.0007200119532953171], [0.0009663432752742274], [0.0009749198538831954],
                           [0.0008505378707514114], [0.0008262418183825268], [0.0011272724619542107],
                           [0.0007947300433426925], [0.0008631742286107668], [0.0011097502070706734],
                           [0.0008657488693952035], [0.0011099680388569648], [0.0006086807734492416],
                           [0.0006167096813219376], [0.0007314228229667059], [0.0011447120812525316],
                           [0.0009982555492538496], [0.0009132342246959452], [0.0007307511966622899],
                           [0.0009442867849741399], [0.0007951627563825976], [0.0007682760930226896],
                           [0.0009368945777215582], [0.0005537990569789432], [0.001037088394709579],
                           [0.000902431634028268], [0.00044069452242429446], [0.0007491495712432859],
                           [0.0006803022519524654], [0.0008132239482268993], [0.0007671916150359315],
                           [0.0008324263347490224], [0.0008165776973568194], [0.0008455071855910044],
                           [0.0004892572277203984], [0.001010611335119616], [0.0009423441226806687],
                           [0.0007882919354792177], [0.0009006812852235911], [0.0008987063962367954],
                           [0.0007253111637297914], [0.0008437976097755323], [0.0006779205745731066],
                           [0.0008356129474664782], [0.0008591152292423321], [0.000707734620758053],
                           [0.0009015735415187523], [0.0007758344039998777], [0.0007909915969564169],
                           [0.0007579834444137218], [0.0011314481994030875], [0.0010642555635322014],
                           [0.0006366622905512253], [0.0010340326050228073], [0.0009898458955955203],
                           [0.0011522058169740116], [0.001111361252856993], [0.0009743903760293763],
                           [0.0007252876399518072], [0.0007292895507774906], [0.0005913255034625381],
                           [0.0008493654012700541], [0.0009451495821536475], [0.0008170179503165314],
                           [0.000977215387320829], [0.0008113605233159143], [0.0007955176392711258],
                           [0.0009442237601086892], [0.0008846237967620924], [0.000962920381025045],
                           [0.0008821991061165101], [0.0007897558943267051], [0.0008621056183308521],
                           [0.0010176610774765135], [0.0007564791661879409], [0.0009237548316251073],
                           [0.0005941486955209127], [0.0005262333007217162], [0.0010150268378301649],
                           [0.0010567840540746955], [0.000644961676387989], [0.0007954928093425927],
                           [0.000890946210567851], [0.000834308167361002], [0.0006334806963401506],
                           [0.0010679435590298712], [0.001030661300393146], [0.0007577589898188565],
                           [0.000924975196098578], [0.0010585897221517373], [0.0008429937841860997],
                           [0.0009057148788099875], [0.0008876164374478018], [0.0010720086682211622],
                           [0.0007329979151903922], [0.0007649875458064087], [0.0008752197198415134],
                           [0.0008898265747100272], [0.0008723540270788122], [0.0009119375007600166]]
        elif set == "250":
            self.x_data = [[7.90984312799614, -3.7953080566000965], [6.686429317139762, -0.4143030060810915],
                           [-9.582011246984475, -2.185413606317824], [4.494280749724844, -1.1442059123862816],
                           [1.7827810485770197, -5.635513852539285], [3.1258030975273954, -4.283810456076051],
                           [3.7514731855170496, -2.822450392112799], [-1.6596292787735916, -5.716262963618613],
                           [-2.6929127429626645, 5.573685894874515], [7.811503293310899, -6.2642624705398795],
                           [5.17049270553305, -9.972549372404982], [-2.0633714270212202, 6.258736300914588],
                           [0.9859869742114888, 1.3567705547957978], [-3.7629834938502267, 8.796878351959958],
                           [8.510773118786034, 7.976759025886643], [5.6764763133398315, 0.4568505258705091],
                           [6.216497509948294, -3.545665796665549], [4.093522685106306, 8.873590601255113],
                           [2.6786268430265903, -3.858979049058549], [-6.067780482669762, -5.33168445930542],
                           [9.572980302435255, 1.090553901521865], [7.458267692013347, 7.566281970648397],
                           [1.5710416586388511, -4.480304154120035], [-0.43047391987181527, -4.12912453359331],
                           [-5.6846143145916255, 9.983471546353108], [4.677165631174239, 3.977140297614149],
                           [8.468885817360281, 7.841231376804668], [2.825652099266522, 1.9348893747824576],
                           [-4.738017632594374, 8.517278651887711], [-7.087289060731813, -7.0242754480534755],
                           [6.9971211703474445, 1.2422239851662766], [-0.30963720733407385, -4.5947030811462275],
                           [-8.900953674559105, -2.6818045866875355], [8.800419568134792, 5.503003083915264],
                           [-1.583776721647844, 4.004352679790461], [-9.437569635793434, 9.71588999474583],
                           [-1.5086168932026904, -0.20558796418678504], [5.800923987535194, 1.1995761037970691],
                           [-5.5239313385912805, 0.9916988589319473], [-2.5649087982011674, 4.429324868315835],
                           [-9.500081047157792, 7.66322918663694], [8.17601301743571, 6.872202633065221],
                           [3.3433945414207358, -0.3808988768068051], [-5.499499885826246, -3.6845938002252208],
                           [-0.1184026454566407, -9.767778290322942], [-6.308763341337049, 5.1617746415852395],
                           [-7.847567788850774, -5.434566445519451], [-5.137350988086076, 3.124582538051687],
                           [4.790889073643264, 1.4301226405287153], [-6.538229564857635, -7.153156526778341],
                           [9.754724618842694, 9.885239224779475], [-5.07908245846196, 3.042483065932476],
                           [1.6947144942701193, -8.247723656398152], [-0.6391764297902913, -7.645259294098778],
                           [-8.269427773517291, 4.688445613023113], [1.329955123415976, -0.9437565948438884],
                           [4.973366027246268, -4.8963467742087445], [7.345975507324987, 0.7332881127427271],
                           [-9.294119367634515, 8.319354295172403], [4.579518611332363, -0.7496718604482275],
                           [-6.681283083982721, -2.485261403080738], [-7.948911170219255, 5.839249631707091],
                           [-0.5105514383065941, 2.9189278491050032], [-3.562260934535556, -6.837939747348514],
                           [2.938667747629143, 4.992550998750451], [5.941443138245379, -1.3565247668386498],
                           [-7.781389355857514, -6.934838146710417], [-7.413773015844198, 2.1671193186026336],
                           [-3.404425229966529, 0.11130863108096456], [8.888682523379316, 8.574247606605251],
                           [-5.23047742453379, -9.48614628431497], [-1.0887731058618542, -1.0048500323652796],
                           [-1.980777044251802, 0.2864147269541846], [-9.088457688133362, 2.688553142408715],
                           [-0.8349949374772052, 5.979560195201323], [-8.77356345055937, 1.4891456733887196],
                           [4.166837079953456, -4.024472338473503], [8.33067478770112, -9.290161220931116],
                           [3.8247081130508747, -9.707614798258396], [-4.4380227914300185, -8.604312341420412],
                           [-1.0369702049603866, 0.8557231719927394], [0.07792456034928286, -3.953072099348123],
                           [-8.117186529596506, -1.1092345591315613], [-9.121635312691483, -5.236944976296178],
                           [-4.841473545737111, 2.058339245771524], [6.295107333285891, -6.124713059631363],
                           [-6.623163382105502, 9.184650922641634], [-1.2671192289195314, -0.2928660966338743],
                           [9.462721481803442, 4.894714794990037], [-9.938433915477972, -5.859531524429318],
                           [-2.1401555283492595, -0.6212875364412405], [1.2410143488820005, -0.07313199349796662],
                           [-8.163921837757934, 8.078511547253644], [6.871116020453517, 7.0139848256287145],
                           [1.9653719780860701, -9.53792353268689], [7.600561373665457, 2.405515149016818],
                           [5.860682689999237, -4.343493491203017], [-7.31959869273771, 1.5278117527490842],
                           [-6.84912999969403, 8.962591999692236], [-6.890179054708496, -7.727197093738107],
                           [-1.887102322354771, 7.110498950569468], [2.507809148882691, 7.765663279827031],
                           [-7.744035594184679, 6.652735388723496], [-1.4105248396358032, 1.8923153870390692],
                           [-4.073816400056085, -7.293433017746603], [-5.2988624105192415, 0.018585510535176653],
                           [9.87513258607489, -0.0854157259247259], [-7.133125231171336, -9.212042420661497],
                           [3.4199718512995467, -7.056192702036385], [8.646205907132934, 2.1494545575316977],
                           [-5.390919383852541, -2.3370141759322616], [-6.741217766812728, 4.5034697161601365],
                           [1.5182917006116696, -0.6414632777301694], [6.51058542523133, -7.498012500676741],
                           [1.1622480343102772, 3.570370734154171], [0.3548950256057726, 8.228837979928379],
                           [7.944458339618446, 6.196768155304241], [-4.244066700581194, 2.2918600290653313],
                           [-8.96566198208425, -8.975581352635862], [-9.780748493760482, -4.424911437971241],
                           [-0.34996315511457254, 7.285233434201089], [3.9228139365693906, 6.102280940612047],
                           [7.560832781650781, -6.511565298382422], [0.6203958279996122, 3.6192726081620314],
                           [3.4441682348403013, 1.7283838903711235], [1.8689032379271548, 9.676006157537177],
                           [0.21708473404572892, -4.820567347672711], [4.361853007497354, -1.274550787329492],
                           [-5.920058016582795, 3.2902671029528676], [-3.2432934745384507, -6.056587450127736],
                           [-5.794199570094837, -6.755359354913117], [2.59582784740914, -2.7261838917565484],
                           [4.927900697485422, 7.233744622222424], [0.66012442287205, -8.53392015234348],
                           [-2.9988589553142813, 1.7843361989874253], [-3.875126761202904, 9.803386503583063],
                           [8.245425782715479, -9.435921074192619], [9.234347051966473, 3.2293137189937333],
                           [0.7222750505317137, -7.971207091438018], [9.411593207025287, 2.989721679614238],
                           [-4.53673445271447, 3.4189315713097006], [3.249925760149173, -8.70523868029999],
                           [-1.7207624926840275, -9.877438909396902], [-6.232712732515585, -2.951725710622929],
                           [4.403890207592326, 5.733870789653665], [5.306399223708331, -1.998921886340824],
                           [-1.1821523809913934, -8.945017187429912], [-0.7418316515570282, 0.1896464564893705],
                           [-4.150273261854163, 3.5112617964450763], [3.650624248676726, -2.1318162081555334],
                           [-8.808662232067775, -6.3561790881724285], [-2.1674635589319458, 7.197026155826833],
                           [1.6557405580547808, 0.49100847605576803], [-4.167198505802018, -9.10046971062314],
                           [-8.034564536851784, -8.842973158134091], [2.1715918380321, 2.4802262522827547],
                           [9.928901834909464, -3.4352104669220216], [2.7457684016029233, 1.664848931652351],
                           [8.609196883367993, -9.144352417360594], [-9.266151978086633, 5.305430831524841],
                           [6.63500235911043, 9.314282908580882], [7.196067339072179, 8.117554759054485],
                           [-3.7306901642977595, 0.6854978467172437], [4.283947862208713, -2.473728323109211],
                           [-2.7798911990496897, -3.602906536150323], [0.24715540724452545, -6.162572679090161],
                           [3.0884412983161003, -8.055256971902876], [-2.473174496672713, 9.209992391070507],
                           [-1.830899200747714, -1.4911279949716896], [-4.716320149167924, -6.694610696940873],
                           [0.500781729001238, -1.8548243901475008], [-2.3374598342555437, -7.430431422619903],
                           [6.9321294798023985, 5.214013610473325], [-8.68167983596586, 0.596730498644698],
                           [7.215109770068629, 6.517478521382223], [6.425377938324761, -2.2918206197137936],
                           [9.65752287057914, 9.487238748924618], [-2.50947790980499, 4.319184108809733],
                           [-9.881412893425797, 5.895387489829872], [-1.2950034947778448, 4.135902498730882],
                           [-6.371674372930039, -3.0718200106054896], [6.3400929179110825, -1.6469638578974415],
                           [0.40603946813936354, 0.39863113352843627], [8.0024400888365, 2.3650687815958786],
                           [-3.3434606310036123, -8.143194362878832], [8.14715407732277, 3.759515484224618],
                           [2.306421232230784, 9.555214665469862], [5.267996153643498, -7.80609947186682],
                           [5.546670221739472, -2.0788704326891816], [-3.1590846107526804, -5.011274991837624],
                           [-8.413628841672118, -9.652793164935215], [8.768033230308987, 5.614377520123043],
                           [2.0070218811520153, -1.8146477622559516], [5.4361092355403855, 4.216251699143038],
                           [-0.002519736785673743, 8.652392644446543], [7.116213904167893, -8.411276541258403],
                           [-6.963668353980875, 7.727437161215963], [9.834951333493017, 4.587100155020437],
                           [2.416342754814094, -8.785232690571233], [3.5877423770164683, -5.940822789198126],
                           [-2.271106291685112, -5.129947444619952], [-8.596787694994655, -0.5417318832307529],
                           [-2.9316079765630807, -2.9959396768828235], [9.035003924504199, 2.5988437739979293],
                           [0.8509490862445226, -5.760926681841059], [4.07808562799659, -5.580083918574224],
                           [5.702385441552364, 4.840866769341407], [-4.977598375453428, -6.409215657418145],
                           [-9.722613435306652, -1.7520664643580055], [-6.159905846707619, -8.192119219095535],
                           [-9.677540524880568, -4.210227958751664], [-3.9479394585370517, -1.362853782485404],
                           [4.83919545644493, 3.889733520500613], [-0.16778932835931215, 8.8800140054847],
                           [-4.344958996846521, -2.6087065691462215], [-8.320912143093874, 9.437370612647584],
                           [-4.569712898142154, 5.091705891339286], [6.721392255055505, -5.445081603064444],
                           [3.007521602690101, 4.3728457002534515], [0.9321329965227143, 8.466631950672586],
                           [7.7359319622952825, 0.8832327381356144], [9.315346572089965, 7.435239558266005],
                           [-8.551529641826708, 2.8325578014070523], [-7.551990716666522, -7.209468578207462],
                           [-3.1089905670513893, 6.434486437922445], [2.3913014313651804, -3.297357577139765],
                           [6.106293207023722, 6.068184398619966], [-7.511656269222411, 6.723571126681485],
                           [-3.4895888526633847, 5.371605895586075], [-0.684523631629439, -3.513506080426085],
                           [-5.652391672312319, -3.16752808605335], [9.14145705797688, 2.7988207686493283],
                           [1.0786495368072284, -1.5446860847035975], [6.070808362421584, -5.108834557636518],
                           [1.4352037932271955, -0.8477526847765304], [-7.63069900953574, 3.7966253457037418],
                           [-0.9258289408602831, -4.702627839306556], [9.055909648891575, 6.577400534418896],
                           [7.377146614313514, -3.2773276358136307], [2.087636855201069, 7.4406854598064776],
                           [5.470812071025277, 9.085360663808817], [-5.8529501508578115, 6.898929812221947],
                           [3.8942233179786747, 8.362598828704929], [5.114197352159497, -8.348755823582954],
                           [-4.954833233893333, 4.738882454278228], [0.08328784933210898, 6.340035079326356],
                           [-2.801570741833519, -4.76151941342517], [-6.4298345521141185, -7.847167897486459],
                           [-7.276777575136025, -6.623918794258655], [-3.6058059223418892, -7.580466687711521]]
            self.y_data = [[93.97461676792847], [64.64317528978128], [1133.314222318032], [31.074051470397016],
                           [321.84649562910954], [163.99309321286816], [79.19513727295636], [600.6905651935358],
                           [25.272138562537393], [156.28724656376284], [495.5848528936878], [20.156276514594005],
                           [13.686209805125998], [60.56556470061467], [704.9336795005567], [46.541373568574336],
                           [77.12434033059353], [365.70543191617776], [157.20757758771427], [1067.9445688771298],
                           [254.75272801816203], [548.7229336751527], [247.23267173989797], [345.93757498916574],
                           [93.8102059534713], [101.12660495532944], [685.3777179328256], [6.781159566501941],
                           [63.560112864953496], [1478.0054184338437], [110.94342824046485], [376.5437683978196],
                           [1101.6012071408704], [491.7536432641423], [17.662947668720008], [209.4496357259171],
                           [147.177501521855], [62.302400532449056], [337.7909179316833], [32.994419205016094],
                           [268.2727670106478], [554.741392297819], [21.227636175733572], [782.2085983301936],
                           [935.5710708730396], [164.0572618821646], [1344.110021681358], [182.29613569466431],
                           [36.56692703040503], [1411.8520598030786], [1102.4855984534472], [182.71917725122026],
                           [572.4579251830804], [719.6378890240243], [318.65830190751467], [67.90033076242129],
                           [139.69904172918473], [111.97095464743384], [233.25757377743298], [26.98880106265078],
                           [782.5217773956911], [237.45620606432314], [12.421400940232678], [947.0624020213866],
                           [69.54661544586426], [44.7656863374273], [1577.002495732021], [413.4876849784373],
                           [240.50169833099363], [818.3054990737998], [1595.9707102054956], [168.93076867245637],
                           [145.951625624266], [534.5060359110973], [17.485103894151422], [607.15756159725],
                           [118.89746787605564], [303.17287840508544], [560.1498279789292], [1325.9969039189068],
                           [78.67847773382826], [297.26722977913835], [799.7617581428193], [1518.4446448544793],
                           [219.05318188707918], [169.96207016865728], [104.63979936933734], [139.63654400997433],
                           [504.31354394428183], [1765.9783775946896], [205.84274943742128], [41.58578957760815],
                           [176.53105195115876], [441.44295245486614], [693.8272675031529], [188.2126434708337],
                           [102.21053076649594], [454.899426556074], [111.39744760869374], [1563.7544616142582],
                           [31.218358682087374], [182.41086018127035], [193.48598542091793], [56.54878621365567],
                           [1076.307794828142], [393.05865738615444], [222.37105390502032], [1870.9850932346922],
                           [340.23075094648425], [243.91189692800185], [619.5052732159105], [217.8248766743085],
                           [52.54568389046372], [240.07253352468095], [2.4985682386425747], [111.79934011242221],
                           [469.82272755475105], [169.7167548203317], [2168.4016037683577], [1497.138062146883],
                           [54.6488725509163], [163.37406145844105], [168.341690755366], [0.7573617277885764],
                           [13.090479835821348], [273.02668219541846], [357.85388228507594], [32.90590902672024],
                           [223.78796817772823], [807.5746461940698], [1236.8797863910095], [103.57521269308735],
                           [299.80281471433966], [697.0949295894189], [126.23370172659209], [84.92567139126889],
                           [314.91329321977173], [354.3912707375106], [626.5984044344909], [353.0833732222275],
                           [135.59855739297527], [499.68654816085785], [1146.4465293263634], [783.0528406492178],
                           [169.7489582771209], [45.45264485273141], [945.7529046815783], [93.82162474266131],
                           [112.868412801062], [57.98660066109003], [1652.912513232693], [31.887850513556174],
                           [20.46319071698273], [1365.810770146229], [1965.365690417049], [3.3422594752062813],
                           [146.01120828613264], [5.504762064823787], [287.65571571692413], [364.2012124493817],
                           [642.7650938714353], [576.5727970991181], [226.27528881491614], [59.926569184679536],
                           [489.095907442088], [477.7815117797095], [448.19542706764526], [80.58763532987527],
                           [242.6325499714307], [1076.6489507872138], [138.48163001076668], [878.1518564518351],
                           [305.53060542148944], [683.6946306847692], [429.8933440059327], [57.50944147430733],
                           [1034.4921868865263], [33.246229651058336], [381.8948222609943], [11.931376003813828],
                           [814.1187462623079], [52.032577043870745], [47.960428948239795], [211.61798170998887],
                           [1102.3823624601491], [301.7203693977684], [291.8988516984763], [305.9747569624474],
                           [47.60008323543651], [673.9491232196907], [2196.4175035381295], [498.35512204765575],
                           [82.1869855455253], [148.95506988905612], [119.43986065923242], [279.7760581454586],
                           [127.66144509754052], [515.0518808841579], [570.9532503694061], [248.08124565223648],
                           [596.7322870162471], [795.1249856251434], [445.6336435359492], [297.82473712016304],
                           [394.3268555156148], [204.18039194113203], [196.75781728485396], [1071.2812059547205],
                           [1095.419446553036], [1523.7150804116345], [1445.8863041737372], [390.2800811209729],
                           [104.9821245662707], [124.75854225096397], [539.957333383215], [161.57839061508773],
                           [83.78306626107394], [133.72766727645606], [51.62238071321129], [146.47532263453402],
                           [135.20021316039947], [739.12612522368], [469.09340992804476], [1585.3392133602092],
                           [30.499323853238486], [137.8699797492487], [302.7763223207122], [178.01631170973386],
                           [43.72432981581773], [314.0941459642463], [739.6940880978452], [318.5158439660556],
                           [100.44228714672298], [128.38470798277342], [61.57656281370272], [320.6144180160033],
                           [433.8680126104785], [619.0310981392254], [80.11269650167489], [143.15203175616995],
                           [502.750906213676], [97.06952409507255], [309.83444265981825], [355.0762237016038],
                           [109.58071493782606], [35.48617156456012], [609.5133394647022], [1509.0587670388718],
                           [1442.8645820237991], [1055.6512266714867]]
        else:
            raise Exception("Dataset '{}' not available.".format(set))

        return self._train_data()

    def limits(self):
        return [[5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3],
                [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3]]


class SurrogateBenchmarkTeamMulti(SurrogateBenchmarkData):
    def __init__(self, n_test=5, n_train=None, verbose=1):
        super().__init__(n_test, n_train, verbose)

    def values(self, set="300F1F3"):
        if set == "300F1F3":
            self.x_data = [[0.03507625874709347, 0.03457922453471541, 0.04462600168613375, 0.03720276897968462,
                            0.020563267690229217, 0.04201981601827307, 0.017587625959223606, 0.009400202177229231,
                            0.006223979993741034, 0.029899232469225715],
                           [0.03240671022608214, 0.04797106749633357, 0.020936723791686557, 0.008456974870031294,
                            0.045331468781083156, 0.029694161523204903, 0.038865959461467, 0.026237174947994518,
                            0.047526689843497054, 0.03942369789478368],
                           [0.016747761328407124, 0.025440353008615524, 0.028891476153539375, 0.006797224417949114,
                            0.036815314648671785, 0.01722931376302568, 0.008965460666019345, 0.007201836230155929,
                            0.025023053193771884, 0.03732730739339237],
                           [0.02388671632135772, 0.0498766512856664, 0.029343663991648103, 0.011745401936620005,
                            0.04607566932016879, 0.03554431266869683, 0.029319421208468418, 0.013519124352280688,
                            0.01928212888995016, 0.0251569425042478],
                           [0.031423300708437615, 0.007677217619954763, 0.034091629262530364, 0.03327907354404095,
                            0.024695839671001824, 0.029482655380478208, 0.0100244654664486, 0.007349566509994623,
                            0.033411995035183675, 0.009976127467339207],
                           [0.0059681964372267846, 0.025380060944313606, 0.0297236148039789, 0.03696859882932578,
                            0.033220242116974816, 0.017688886821453493, 0.031656039739263425, 0.04158347417770518,
                            0.033074490364683656, 0.01278662473138727],
                           [0.04648082550069279, 0.040476006986024306, 0.0348803633899001, 0.027246279364169637,
                            0.021922065093781463, 0.043591908632955825, 0.03032303712657518, 0.03797439028072148,
                            0.01750685574714621, 0.038094563727725926],
                           [0.012583181396127081, 0.011957335315865125, 0.026101333892605657, 0.007718470069118419,
                            0.017919754622763934, 0.04425697732322216, 0.03231062439061754, 0.011755723224698374,
                            0.03264403013382315, 0.02196813542082958],
                           [0.02192747823223129, 0.046380731465438675, 0.028399722000515996, 0.04894870451822543,
                            0.017493001123192242, 0.03526208418093231, 0.04588616758239281, 0.006890559665395025,
                            0.012041908947631796, 0.014248815751642111],
                           [0.036677121911740544, 0.024988401033940035, 0.012866599255082559, 0.022153840462164318,
                            0.04814346719839895, 0.04876426088269313, 0.0064947170834529855, 0.012844467724463973,
                            0.022128448139418198, 0.017239110810037234],
                           [0.03701799495932883, 0.03722720931266452, 0.015520659618705697, 0.009966241018365934,
                            0.013030611793175984, 0.03336258725283455, 0.0341297808464549, 0.017456712276761658,
                            0.018875846193162463, 0.01674796182978659],
                           [0.03946581353497069, 0.005937386280225478, 0.048450505155285, 0.02683725050622067,
                            0.030336532283699907, 0.024046397331425506, 0.01646112444625127, 0.037348818170240015,
                            0.014041109726185854, 0.011764159994688043],
                           [0.03633666044495574, 0.03572880060903705, 0.026682888805329423, 0.03782752132312105,
                            0.0073252099392500485, 0.03953703836898393, 0.02347747033567276, 0.04822363080181739,
                            0.03441644026005263, 0.022525079011507528],
                           [0.02733037975917565, 0.010265526040410155, 0.034608908220302964, 0.049717573023945937,
                            0.026730532247924123, 0.03949141808952607, 0.04308232906087847, 0.031221964428004758,
                            0.04770261404293426, 0.02370744426072576],
                           [0.009081792545060387, 0.01790683934313682, 0.03725745696401489, 0.03274628175187681,
                            0.035056571200869066, 0.030879172666016128, 0.009405485010040376, 0.014271091037523404,
                            0.019245329759112844, 0.04354322160176041],
                           [0.04407235248836716, 0.007248856072075409, 0.030554236742423035, 0.045887259485671184,
                            0.03563937513692826, 0.03908733617955919, 0.027630621661783494, 0.020104026667175767,
                            0.04077138854641784, 0.031105545253006236],
                           [0.007137499056794892, 0.03334942850428647, 0.032935459242464514, 0.005548834682653192,
                            0.023053556899989423, 0.006536620731408266, 0.03566915577497578, 0.0250812477646257,
                            0.0398337279648978, 0.014336095232874962],
                           [0.023127457345922003, 0.03297014107478902, 0.026578566086417805, 0.014347203288339043,
                            0.03301669990170756, 0.045925774028512056, 0.00850536448413501, 0.04034833401493998,
                            0.005230392997269143, 0.048795837989436375],
                           [0.04726475509608159, 0.026406446862649615, 0.024352982981943293, 0.0434644148365561,
                            0.046847812536144104, 0.025015376101271754, 0.03196114162812205, 0.019571225595701125,
                            0.029259249857096752, 0.032489885046660866],
                           [0.038122846558382203, 0.04608401762096784, 0.04044836966287013, 0.02828431070189696,
                            0.018843246615451066, 0.036376307688369296, 0.016592325488757, 0.04995760300311679,
                            0.034039214620698345, 0.04793705112064948],
                           [0.0415725090494397, 0.015339069756553851, 0.046249138530688544, 0.04437631174103983,
                            0.01449328471585493, 0.038568418217499005, 0.012777241071216054, 0.02605223929389465,
                            0.04551260027398195, 0.020096481186859437],
                           [0.04859079798957078, 0.04898300324991088, 0.031058213854682933, 0.038591242394973975,
                            0.02344237675462779, 0.031438001929511204, 0.03918892259389248, 0.02970532926477921,
                            0.013522150664200951, 0.016681924301864465],
                           [0.04310406100434275, 0.040968219532412066, 0.012006419349583703, 0.04190479675703686,
                            0.023532704773130672, 0.01433113306494756, 0.04830641628678266, 0.024067758626321236,
                            0.010411691197270714, 0.017367623912397802],
                           [0.007062235563149983, 0.01667293366072273, 0.031980518976248797, 0.006948829413751481,
                            0.04259758210457526, 0.04629076799012428, 0.03149020620077882, 0.006713631332744241,
                            0.03187126509523836, 0.04085364382205913],
                           [0.005282433006249842, 0.033652429095008506, 0.012783466968538444, 0.019165677842815698,
                            0.042901471813134695, 0.025843979572948213, 0.020253126916712903, 0.04940416668555232,
                            0.012385484496142692, 0.04862039412166013],
                           [0.03260133460966177, 0.04031649497537705, 0.032801006770662076, 0.016230253159426684,
                            0.011465640144196819, 0.04254136032452199, 0.02556724834800428, 0.01643800095924312,
                            0.02617939448219639, 0.009762447239468801],
                           [0.03692273927956777, 0.048568207934733584, 0.04811279413529162, 0.015966985820952757,
                            0.02837812576347422, 0.01410729174706396, 0.02256812842143527, 0.030711805659356148,
                            0.010168286389506166, 0.01241037607428142],
                           [0.023461057276344562, 0.03640033162279069, 0.0360375653138708, 0.02472846014416272,
                            0.015577220228687842, 0.026294476414331736, 0.0076418496783031975, 0.005651398293629924,
                            0.04164572487404009, 0.030306509423356637],
                           [0.028231916831150342, 0.030622828221005836, 0.016471417365001715, 0.046001850308328936,
                            0.027110106906100967, 0.028690092944267352, 0.020345599567586737, 0.03939850305169574,
                            0.018063632934451564, 0.04185176833498074],
                           [0.008715518877065698, 0.022480942644780528, 0.043002434800147024, 0.04404806640443841,
                            0.04710301576061325, 0.02437490523988334, 0.014302114561774354, 0.018048900016922387,
                            0.033222125441448314, 0.015079156725501728],
                           [0.028859325716659195, 0.021458699472231904, 0.04914415562594094, 0.03682618459677618,
                            0.04763483791505027, 0.03127732509295184, 0.02671957825868623, 0.010483236786285696,
                            0.011213821704312621, 0.014887732670777167],
                           [0.015352102744858837, 0.02400256846507302, 0.037091434291158965, 0.014719637089254778,
                            0.022672983179552002, 0.04430526627631395, 0.019020106625917382, 0.03703742796896468,
                            0.0484254213195498, 0.033978202649563194],
                           [0.01128590068899319, 0.03868940304960281, 0.036822750013212384, 0.03908250617544697,
                            0.020882681254711582, 0.03379989720599625, 0.03324224870794403, 0.041828664999288355,
                            0.01657540791455778, 0.03226210633452895],
                           [0.007398774695486417, 0.046642011730624426, 0.021696100602801357, 0.041666378437323355,
                            0.03497454975711409, 0.02938133149085825, 0.04930153477368702, 0.03676634270543168,
                            0.023654843665927922, 0.027447242385966023],
                           [0.01700658569408076, 0.04475210519010924, 0.047047828582626974, 0.020938801029825777,
                            0.027952170958776376, 0.025887822835394852, 0.020044207152547684, 0.03451550053779006,
                            0.04457209538516037, 0.04934230618421564],
                           [0.012007792458742035, 0.03108110180458666, 0.03248744059438257, 0.03622648490056833,
                            0.02389541122140732, 0.0273307138261691, 0.028393168961628423, 0.02539261228712639,
                            0.04799212040698033, 0.017480719310484515],
                           [0.024677153640667585, 0.01427302134712427, 0.039642783692128473, 0.03871871511028653,
                            0.013519993172727628, 0.049332215131720626, 0.04864617531486764, 0.0390367087082522,
                            0.022104593302622644, 0.020725778188433793],
                           [0.015190738739271261, 0.04787652860269933, 0.007120843629207558, 0.0052751565832295875,
                            0.03384832610622925, 0.00926804698749369, 0.015666788922418382, 0.048783938236705314,
                            0.016035581914508587, 0.036045747838881544],
                           [0.029918995579956566, 0.014621560298179351, 0.009933517661063956, 0.02363376123819467,
                            0.04094225425042017, 0.03802037760495749, 0.012849870783319884, 0.02245651885693721,
                            0.03772080199238243, 0.014019997463107635],
                           [0.04004757118306592, 0.03182190802081733, 0.01003033219181751, 0.04507739127945256,
                            0.031073161279683403, 0.03629643254835053, 0.04767798751492818, 0.04460641323335716,
                            0.04230654705041231, 0.043708623533149395],
                           [0.012948756748610405, 0.034295955564152596, 0.03914031044137554, 0.01826917251477627,
                            0.03918805713456112, 0.048628824722586325, 0.04873438788526864, 0.023224442463464622,
                            0.04805494250919179, 0.04326908711222978],
                           [0.03988276281790323, 0.023824829804111384, 0.010959920312374635, 0.021659570838182433,
                            0.04409114727850347, 0.03124464061275341, 0.043350168596386095, 0.02122275884491832,
                            0.017341341188394654, 0.04101948823384483],
                           [0.016590393794386274, 0.015012295968161202, 0.04085679174225162, 0.02696616314260907,
                            0.019105533081836705, 0.0430224076359461, 0.024365129947552435, 0.022807068734179654,
                            0.03129463294999165, 0.01586716863319276],
                           [0.01951605194267508, 0.025186940416180983, 0.013340638872695227, 0.02544565199182057,
                            0.016455253121882455, 0.02175989347738705, 0.045531817025703726, 0.030606479492285708,
                            0.037141405002735245, 0.03900483537654599],
                           [0.04277106728543678, 0.01073969193082644, 0.00978685534955661, 0.02666967437103653,
                            0.04212906102149903, 0.03575415950049082, 0.04951877787250784, 0.02258111735710668,
                            0.02038989518089396, 0.04305540802883133],
                           [0.02214571139080436, 0.027255405869632838, 0.023592609995284102, 0.03130610033022621,
                            0.03340746926287629, 0.009821871474128035, 0.010994124309101757, 0.00994811903960759,
                            0.024951398278874364, 0.04227276990297057],
                           [0.023191400667322313, 0.010199706915054414, 0.0366953641658018, 0.013523124174278846,
                            0.011015909807002627, 0.03514797738320122, 0.03078921578140103, 0.026331402539388944,
                            0.01484578529639351, 0.033711999575729255],
                           [0.018386449737494818, 0.03882134809324653, 0.006876676432851692, 0.009722820582866453,
                            0.041986989078929414, 0.00511555320661447, 0.012577833476207624, 0.01308017978087321,
                            0.015771098030630572, 0.044840370710072026],
                           [0.040850781412384386, 0.009979979535154948, 0.030076865674649005, 0.007687566687605531,
                            0.006839693186253621, 0.00855139445775106, 0.016891367783187537, 0.015086309876420179,
                            0.011549253545688401, 0.0318161329919318],
                           [0.02208523661128038, 0.015693170864094212, 0.04143540204171311, 0.028820052441035127,
                            0.03149935215081213, 0.013843778939961731, 0.04431465723652928, 0.0137533255722132,
                            0.041541721478856215, 0.04446427730852879],
                           [0.04956691575008424, 0.01229510586199871, 0.038521275577575974, 0.033565493725770994,
                            0.04065429257413078, 0.03242643985614489, 0.04724100273047525, 0.03179624003710781,
                            0.04726256094707335, 0.018242310782314003],
                           [0.01834163952222664, 0.04326116561033388, 0.03474534910402122, 0.017081689069041752,
                            0.029983833029139612, 0.01780065552839817, 0.014484407202075892, 0.0268576566053781,
                            0.04472791324740308, 0.026391693248609172],
                           [0.012200802328501424, 0.039515888009696534, 0.017803166308440796, 0.034948017893776336,
                            0.016888133207210658, 0.022015264462430763, 0.027024195948877353, 0.01875401417467007,
                            0.03158989881627166, 0.031261913458353024],
                           [0.04913437465445245, 0.02438944381304467, 0.00665113557793188, 0.018804452929482796,
                            0.03618848644476988, 0.047690760105093946, 0.0321589708233244, 0.0076394090958945525,
                            0.02257422739468626, 0.021048636104803543],
                           [0.026304353575893884, 0.04761179047116333, 0.022430817712416173, 0.04244209416782879,
                            0.005857358218895453, 0.04642772116401637, 0.043663336237685854, 0.04338037503938705,
                            0.015280863649984676, 0.010830888183748128],
                           [0.04573151748291993, 0.03586517430307429, 0.0248431417817383, 0.037993482252587044,
                            0.021217508303767653, 0.015440967007473022, 0.04984045938281511, 0.020763317700603096,
                            0.03251132001239313, 0.02866826778762661],
                           [0.03734062246398378, 0.022830646848407295, 0.04982233567599301, 0.01800741428488699,
                            0.046675222929681674, 0.00881979833485046, 0.023779926361026663, 0.036111078056191466,
                            0.02656085372342556, 0.023103929916940127],
                           [0.033857143391608856, 0.036019254599783446, 0.0331592083425354, 0.020886188988824255,
                            0.024000432252410827, 0.04912611941561096, 0.02862521922571781, 0.04756096283887986,
                            0.01705407212659509, 0.026218691823591524],
                           [0.03453267585394579, 0.04398582866063427, 0.02615838028647335, 0.030692306738625656,
                            0.019748471421503416, 0.03463123332093459, 0.03540308266217631, 0.044991379759813446,
                            0.011122444528554392, 0.007749339724140781],
                           [0.017734176999709642, 0.04106241277254157, 0.03879318904929048, 0.01041305151415129,
                            0.005329686933015808, 0.0057143990269981635, 0.023336013936119435, 0.03564080059454829,
                            0.026678734127569426, 0.03841969184531314],
                           [0.04257512803223275, 0.03538974376494433, 0.045135873902873054, 0.006082041891567013,
                            0.02979619725694169, 0.031965872126553274, 0.036055806974175444, 0.018281829722963577,
                            0.03577578704461541, 0.006809016921248686],
                           [0.012431866786758054, 0.01980632062045408, 0.010848165863127895, 0.015694092952147368,
                            0.03289858543922968, 0.04098248874436723, 0.03706372698599233, 0.012540964461914368,
                            0.023842537354419865, 0.021936307423704733],
                           [0.030848473301839645, 0.029138751800598824, 0.02033285194863911, 0.01068388409698717,
                            0.006601279799939875, 0.009159651154253287, 0.009549058709541279, 0.04908373284268618,
                            0.021069869372111548, 0.04822975596716101],
                           [0.036564484880739816, 0.005587247522678959, 0.033915648459100424, 0.02488632666115504,
                            0.01812710941492981, 0.007968151161103382, 0.035890091690813725, 0.035241838083547736,
                            0.006038930785475505, 0.007909145445096329],
                           [0.04048316512994925, 0.0326276054940655, 0.022579935160335428, 0.03842556192487398,
                            0.04583010873021402, 0.03207486061658491, 0.00748265318681547, 0.0140323084075963,
                            0.0353694080670705, 0.04264042297804416],
                           [0.025108684445810853, 0.036132317435386786, 0.02194532494994329, 0.032091505556622865,
                            0.04689644225276602, 0.011926378047645042, 0.019371953695170013, 0.016632018577479354,
                            0.03238594995371039, 0.025940081347403117],
                           [0.034242157216028926, 0.028876825469067297, 0.019999804896867264, 0.005674479224350061,
                            0.013693302768823919, 0.02744685161439864, 0.030834310934561425, 0.010030270623866704,
                            0.03670415954180919, 0.015417978336859879],
                           [0.03294726873964478, 0.011772337983219795, 0.01683693265110528, 0.022073839404856545,
                            0.03378296191824909, 0.031804563455439055, 0.03384196337755334, 0.03992817133723903,
                            0.046345755213541985, 0.005043317253560581],
                           [0.025768974119949332, 0.043091867849064804, 0.04147108235566523, 0.014883760072779702,
                            0.01943610570411605, 0.04410751970339038, 0.009165122778036246, 0.032376318130296976,
                            0.042804878072449926, 0.005566869876902456],
                           [0.033658746013639225, 0.038328998602469695, 0.006790320451574639, 0.02329206230803535,
                            0.041133687129694195, 0.008342172360421098, 0.028862563502774274, 0.007782555962514826,
                            0.0065336972328871945, 0.028452635903741495],
                           [0.024313546846875124, 0.036905186811855395, 0.015201466825087134, 0.02610733145049753,
                            0.045550772573176655, 0.030309455488440794, 0.0460456994217726, 0.0410243901272635,
                            0.037406754901462176, 0.00990412957637097],
                           [0.02171775996099755, 0.02255695946190467, 0.008298553310117184, 0.023385575400056445,
                            0.049029885067411794, 0.028711189727899, 0.01582920869115761, 0.023717939788469044,
                            0.04479986798950875, 0.014002906865527291],
                           [0.02776450911804137, 0.0235762049517764, 0.028666914102488446, 0.040660425549417525,
                            0.014807501323013943, 0.020756668481260742, 0.006600165122158656, 0.009267566436214358,
                            0.020925345074823735, 0.047094673174451485],
                           [0.04898689296563386, 0.020896966966967946, 0.018388900048423855, 0.043957508352606076,
                            0.014145742319511719, 0.04551847651912653, 0.037558611880322564, 0.022286750815007448,
                            0.02695086587823658, 0.02128753363243293],
                           [0.020379028373728597, 0.018089685069708636, 0.030402117217602383, 0.021310903624119203,
                            0.03885430070580626, 0.021589405575889883, 0.029654391953655053, 0.0442261768779901,
                            0.015548012428463005, 0.045119712696742593],
                           [0.010165144693066765, 0.04373270927963575, 0.020221226859848035, 0.03608521701447969,
                            0.044355338556243226, 0.00784971865745606, 0.023261040878571653, 0.031960893469751235,
                            0.035685867467644954, 0.006970232559333021],
                           [0.00568030394981774, 0.03775875998630208, 0.036555622349749115, 0.036505285913936494,
                            0.04171426492996281, 0.04329319842793771, 0.04635979400999029, 0.012714503342145207,
                            0.030877220837891173, 0.0386063166630443],
                           [0.014345333027282615, 0.02602346700551249, 0.009045378187102781, 0.04949271801379059,
                            0.04345621654017605, 0.019567185566339282, 0.02632609442941285, 0.040101666483264964,
                            0.018321069363658765, 0.027312511046337886],
                           [0.04104954169050578, 0.04729083975575672, 0.033424194410109784, 0.007914797541386858,
                            0.04424586855522607, 0.04949367158144037, 0.03488318036231461, 0.03600113926173494,
                            0.013301227900135666, 0.013451705591802785],
                           [0.0395856969512947, 0.04281187544528743, 0.02208567779364426, 0.042508633620192394,
                            0.020154685800690207, 0.033880827932919515, 0.03378182374747811, 0.047218360195251614,
                            0.03684310682273912, 0.043165383533033126],
                           [0.03822028411839425, 0.02588804669177972, 0.04015488953519425, 0.049231090916625454,
                            0.039038019236932044, 0.04070855557435227, 0.04473521139203277, 0.01853124547326278,
                            0.023174930457542053, 0.03007495390744068],
                           [0.026062678650786663, 0.029772404973150507, 0.016348249697664817, 0.02117491751670488,
                            0.01788599921157326, 0.044767354311393476, 0.01784530630842491, 0.02895247294037733,
                            0.010282778075257556, 0.022386804860045382],
                           [0.027174132020117304, 0.01514336825032083, 0.017190448620174235, 0.013357448956798613,
                            0.04050912754361613, 0.016737691121123298, 0.029563301729531172, 0.005205174814465154,
                            0.02014539119310823, 0.017802476121497372],
                           [0.049910982878554375, 0.015458847512081, 0.02340694059342092, 0.04450367244644696,
                            0.03519318804997125, 0.010689694444337688, 0.045391081051952616, 0.032708098019258,
                            0.028691900642867713, 0.03361579125106893],
                           [0.03062788481630511, 0.03966195738843512, 0.009176993387661644, 0.04496372999849573,
                            0.01415981651087006, 0.0107202144098931, 0.048074428009350315, 0.03805418013070627,
                            0.012150243566470462, 0.04606262278540731],
                           [0.04294068779202055, 0.04443510279624213, 0.03770641255555408, 0.02851664407817587,
                            0.048506202631385414, 0.011752857702235331, 0.03775744561221135, 0.0477283494971502,
                            0.020495927776370004, 0.021518733706784643],
                           [0.03098141961391512, 0.04741872998556594, 0.03427486607357167, 0.04779936108967273,
                            0.006043611708830531, 0.028405096863188452, 0.00785376616341375, 0.032548212662235836,
                            0.045698766974362835, 0.04656919093617869],
                           [0.005564773958483677, 0.01358041808742776, 0.04111882334356549, 0.01989108050387,
                            0.038611448143681615, 0.0052949115785885644, 0.03461372857021392, 0.04130857747666716,
                            0.02832634710750725, 0.03648028269198364],
                           [0.02147605197157592, 0.04149951334109341, 0.04076684660818843, 0.04158279244553674,
                            0.03411200477626059, 0.04490787354412403, 0.01951839673182495, 0.018843508708394394,
                            0.028438601971719612, 0.02441010708126246],
                           [0.0372079990987082, 0.03486158399298269, 0.022266189573741255, 0.0308576535753076,
                            0.049423183612690394, 0.020511704456720094, 0.02522965534062872, 0.040587098234824766,
                            0.014676731861220715, 0.008768613037604787],
                           [0.0484469505068391, 0.020228015637440696, 0.012971292201542046, 0.02778216489607943,
                            0.02214562099269525, 0.04894841917098274, 0.0329841082130857, 0.011039912059372655,
                            0.0164467032632043, 0.04198439124478338],
                           [0.02498842080862329, 0.04426289671765421, 0.03210140288419907, 0.014992005357242617,
                            0.04780303418833563, 0.04464008624635085, 0.04188029215041464, 0.02992801661077112,
                            0.024456590135296417, 0.021743139421776997],
                           [0.03577136720753115, 0.045709794909426636, 0.01829327270759308, 0.03964204489214713,
                            0.04747882556837838, 0.03418701828191234, 0.047137584628295114, 0.04804225162697818,
                            0.030372714315190927, 0.012937355097395575],
                           [0.02856685326156781, 0.03110890950570798, 0.04581206005008845, 0.04047503124849443,
                            0.027784930103341484, 0.0228447422526828, 0.021479973036750855, 0.04226939312575584,
                            0.0496463873610339, 0.006254774970993339],
                           [0.020827417664620027, 0.04845623270908912, 0.0381293222714825, 0.016452500014764445,
                            0.009249054902725633, 0.040577939094857, 0.033621390162781234, 0.019718681330365254,
                            0.030634718313562147, 0.04886337778249976],
                           [0.013128058077269876, 0.04353377184455116, 0.029991136984055734, 0.042245228439061075,
                            0.042473484742922056, 0.0071220235497219085, 0.039899690559609687, 0.03533177441486695,
                            0.04860885845074309, 0.012544218868297848],
                           [0.031248032673261747, 0.006478475497081724, 0.027489597533863406, 0.02254045196167265,
                            0.02875557576623685, 0.016894280403094573, 0.04886932885805789, 0.008627939613831306,
                            0.04025373407934816, 0.044258986700552436],
                           [0.03267119024204508, 0.03406098772319026, 0.005091179579093063, 0.006392189187080509,
                            0.012013381040156604, 0.011079362880765286, 0.010710246427320137, 0.01460723484297078,
                            0.030803476710056257, 0.04972537781055224],
                           [0.03602527474316163, 0.007031043310373387, 0.043222776540243014, 0.0301890494428361,
                            0.0074220453888922, 0.023367706241789476, 0.043947906927703076, 0.01333512253990881,
                            0.04343918277212268, 0.01500613185470362],
                           [0.008319961571455434, 0.014781776997909852, 0.015369330937859051, 0.03924078735165815,
                            0.038022265891471396, 0.0331272756391858, 0.04149006941798485, 0.01814651160168722,
                            0.03368225847323244, 0.03417561238055754],
                           [0.010681184830603862, 0.008264612391447478, 0.019142414527421017, 0.03146705210061474,
                            0.034287966992604994, 0.030632382935234, 0.013463968657383103, 0.014408418489698926,
                            0.019893006591754036, 0.0392647572939495],
                           [0.011833505376742885, 0.010485791156527507, 0.043406932770748974, 0.009483565670385805,
                            0.030851255108844018, 0.0281314084049434, 0.005703321176135155, 0.03018257028754456,
                            0.024506107289449682, 0.03885874298990875],
                           [0.02749505884344435, 0.02813565223875943, 0.01129532534699729, 0.04546377636357045,
                            0.016185525758872877, 0.031695453836290015, 0.009752594174664437, 0.0369345554180116,
                            0.028184670851480124, 0.03214626867025611],
                           [0.02482596260173863, 0.03394429332527109, 0.0165818608930482, 0.018582274119628858,
                            0.032457888436742445, 0.03573187252590185, 0.04790355464743919, 0.006248756998433514,
                            0.030148312545913, 0.040444507911290443],
                           [0.035240797391839614, 0.023382503310436047, 0.025707632685331685, 0.010948183085652024,
                            0.015693819195039958, 0.012169846133490638, 0.04018117077095112, 0.02774708779060567,
                            0.037876614629844214, 0.019655769594885523],
                           [0.040103343985745464, 0.029588051304319032, 0.04894679301352584, 0.010125317964722914,
                            0.009205327056779141, 0.018635016994475784, 0.013374121568093228, 0.0327632465820383,
                            0.0116687265173811, 0.036577767676779005],
                           [0.030346922167532618, 0.0051215421375496025, 0.014628290428912998, 0.04725444134361376,
                            0.018462948896765672, 0.04248791522358735, 0.005161355547755243, 0.04889233950055097,
                            0.009023217458307881, 0.04960493902386774],
                           [0.01027500011366124, 0.012122285887408733, 0.02453063435737485, 0.03744216462679866,
                            0.01637005899985947, 0.022438015121410963, 0.03038120490323472, 0.012126232631110553,
                            0.015004350291051682, 0.013795332535869367],
                           [0.03540388799341173, 0.018908788633187113, 0.03355096833303701, 0.016922964115032742,
                            0.0335652594818429, 0.01564722346440501, 0.019664652366264176, 0.028405103435355586,
                            0.04914726929241594, 0.022871288104970865],
                           [0.006891308063151464, 0.027494429374489735, 0.03155142073610984, 0.03802883174961135,
                            0.047302596758282335, 0.04168028442460802, 0.019225035469986666, 0.021912257870370603,
                            0.01789113680014105, 0.02494457072749266],
                           [0.015456307094695556, 0.03167018889827825, 0.04738527728328834, 0.02390445802374243,
                            0.03858324214276021, 0.032898085767264586, 0.034391511973795315, 0.009176868996850614,
                            0.006422475975453671, 0.04031436906250527],
                           [0.028330710687351907, 0.014500770898356562, 0.025063342476901784, 0.00921743701458664,
                            0.012793040281549534, 0.027023862606899225, 0.01681808982296353, 0.03209533667325547,
                            0.039715401692694136, 0.007167734635826864],
                           [0.04554314549709965, 0.018684732293885002, 0.03570561173197911, 0.028972230899096705,
                            0.03195848676971904, 0.04044833919130051, 0.019984589708679522, 0.031624344356318036,
                            0.018519093189568184, 0.01700393387352766],
                           [0.014155224426699035, 0.010706831272791567, 0.017636261270853062, 0.018412221424936256,
                            0.03841633978371678, 0.005990224103042949, 0.03271575669425389, 0.014766502905102067,
                            0.027553954324506566, 0.034679453325039064],
                           [0.03205438935218665, 0.00923160369804758, 0.008862676488637804, 0.02157226755465414,
                            0.007254429449118331, 0.018500315792908575, 0.048397940453110196, 0.04271734413098555,
                            0.005475480947291179, 0.03770160283887563],
                           [0.01888331449296487, 0.04257099664113134, 0.02075511744902647, 0.017657133206074274,
                            0.03988848276435272, 0.042149195506161155, 0.023031882408055534, 0.03661159602932075,
                            0.03840418091002854, 0.02137542038415314],
                           [0.006164365833796657, 0.02471248037269436, 0.005580285618668833, 0.03648365567203569,
                            0.044959096709172745, 0.009469135429691293, 0.04264700806806184, 0.02741792506940367,
                            0.04781936101974551, 0.03001965025281379],
                           [0.0378087869317292, 0.027521778105837842, 0.04824763738199086, 0.015912680401339092,
                            0.01598254738863531, 0.012615048569806607, 0.0359997276614384, 0.045978637587388606,
                            0.03619338061151867, 0.042714017844097556],
                           [0.027043306116164484, 0.039479165434623466, 0.02311529376166582, 0.031242055762652127,
                            0.02592516049828615, 0.009570381510733684, 0.025871481226695785, 0.011541515112279827,
                            0.04265297499699768, 0.023511308589113322],
                           [0.013390406906211117, 0.0069397965244198155, 0.046692597784960745, 0.04793762674203554,
                            0.028181011064636653, 0.015835699919912863, 0.029965174657680112, 0.018424801806536432,
                            0.04141107627270067, 0.020220138738979997],
                           [0.04674568380266614, 0.013897307490787388, 0.046516638589508774, 0.026563260078656774,
                            0.03264277428678918, 0.0187978268664934, 0.029266567650119275, 0.032281644158329154,
                            0.017956962792919175, 0.01880142773859409],
                           [0.024592084805146623, 0.02929535930620205, 0.021618867791969244, 0.015103783410768085,
                            0.028474357074970695, 0.04735972106211893, 0.03314928707261739, 0.037659334580908144,
                            0.039262176329579866, 0.01865247215696889],
                           [0.04303976155890523, 0.01129724730930155, 0.015689291527961047, 0.01379974301014789,
                            0.01763006208988999, 0.04144043022814688, 0.029128880423121374, 0.04412723342772681,
                            0.013078245966683399, 0.02686033645590161],
                           [0.04026327281625647, 0.03696872334005201, 0.010537878467894668, 0.031559006574137974,
                            0.04822766278626119, 0.014922426564520065, 0.010568055351445942, 0.005792144819760452,
                            0.022341771143295563, 0.007545797102637571],
                           [0.04207347380523189, 0.01947429963180937, 0.013639647218260512, 0.014119517809742719,
                            0.04629010049348122, 0.011247002205881642, 0.038452337031889376, 0.03355556988314906,
                            0.042048560608636376, 0.03170124112139201],
                           [0.017117405990108442, 0.02997651469927, 0.020527297181860218, 0.03223226393511817,
                            0.02681932770306774, 0.037765730606419134, 0.046675681518126476, 0.027235015099560788,
                            0.014554829432726325, 0.03622504387307864],
                           [0.041870779937547675, 0.04810120505851408, 0.047722980480029986, 0.012893912896123697,
                            0.025035311195505033, 0.03867661688844884, 0.024813159708240943, 0.005464065763794758,
                            0.015375513861014, 0.023811430495560826],
                           [0.02673386823279173, 0.02428041578262484, 0.03990808981105941, 0.013148880375959224,
                            0.007654022579592468, 0.030150416872069628, 0.024697677053395862, 0.027605921585846266,
                            0.03856081549710458, 0.041638627255847833],
                           [0.01351469277414075, 0.011458865814416937, 0.04037133886523965, 0.023059806353721454,
                            0.028655267283337522, 0.009017903383370034, 0.04421048304673492, 0.015488184912216055,
                            0.049072281271648206, 0.03692871348813614],
                           [0.01145743843642941, 0.0162655540911694, 0.014837393909446741, 0.017379508151192144,
                            0.006272602192737049, 0.0070967383603076275, 0.04324244641808322, 0.019070352459985035,
                            0.04876305412097527, 0.03291257397540396],
                           [0.033182173172517135, 0.016752462114427977, 0.02448209206238403, 0.006576888993795287,
                            0.03931844252586137, 0.01246377084375939, 0.022464295210785072, 0.03548938039047452,
                            0.039378946482998126, 0.03267070378123869],
                           [0.020611996280284442, 0.016407603893679453, 0.04377195096175681, 0.035539058634425164,
                            0.014412332250446767, 0.046149561776672315, 0.031359554414529305, 0.034943812681072266,
                            0.008484208461688656, 0.01171549538969019],
                           [0.00654945978828433, 0.030775849534090387, 0.025994308231306, 0.013671203863030855,
                            0.029616792723414078, 0.019199818700675376, 0.031013780768842734, 0.02024740849773647,
                            0.03436158813303772, 0.019235613185283672],
                           [0.023913944586088115, 0.03255034864107914, 0.013965555599331322, 0.024584872060137722,
                            0.019894033761025453, 0.013987269496318844, 0.03818933387975224, 0.04161990204650025,
                            0.023482534544226455, 0.04809707243931948],
                           [0.03915465947858619, 0.037978001684163244, 0.020758331841749543, 0.036022715720199674,
                            0.02959168938949879, 0.029222850302401283, 0.03677187647065699, 0.022045809742034492,
                            0.04065116363809361, 0.011314833603185612],
                           [0.008964311800646375, 0.03208484141332947, 0.012213371535147318, 0.02804129427455378,
                            0.04537666921486683, 0.01031044706194146, 0.017401792578352598, 0.013159947606221679,
                            0.006099443986700665, 0.008914598011184584],
                           [0.017888532265767802, 0.006748129313719688, 0.04776486651199209, 0.043552890480675434,
                            0.037858483613643475, 0.026614040972302612, 0.04689696835133916, 0.04813265712759287,
                            0.018777706511826992, 0.04523900816886563],
                           [0.048135302733248085, 0.017007988509172088, 0.01179482417355932, 0.012270202009411556,
                            0.009927195328573661, 0.015702808792785107, 0.015254298622167782, 0.049269528633639835,
                            0.045808471697772246, 0.023937154961075537],
                           [0.011140824780335339, 0.04650753716149825, 0.026800450356888633, 0.018979014531753807,
                            0.04798645811121118, 0.011556099307824497, 0.013863836238419934, 0.017723634289735238,
                            0.049532655450079545, 0.006623179176605824],
                           [0.02281120926832585, 0.030115498871284504, 0.019034417125146474, 0.009907267129885971,
                            0.011710034158635018, 0.026077156467863488, 0.025446439064910245, 0.021613860206598317,
                            0.01619099778952979, 0.00608204121289035],
                           [0.02118557065033765, 0.04750977070320661, 0.02811745059369519, 0.0257114841184238,
                            0.010627697706172896, 0.04817880556529085, 0.022882246699676114, 0.025213278104950852,
                            0.009611858852101453, 0.04726777667833653],
                           [0.026497580472981043, 0.017448830387561603, 0.019701116548979526, 0.0054467344710335515,
                            0.04230218800919513, 0.006360954336503176, 0.02165948565649868, 0.01680850280834574,
                            0.013253845662958372, 0.02058656932405356],
                           [0.017396676989127443, 0.009895481656153247, 0.03608638948564633, 0.04994555519376679,
                            0.03479508077093429, 0.02107671428056365, 0.04228638125592467, 0.02572563349489121,
                            0.01507355647308992, 0.010524864832660585],
                           [0.038393524074446767, 0.041408847953303456, 0.04527382313508595, 0.02395038400499107,
                            0.02316592642247739, 0.04752723294046715, 0.024208603781639866, 0.046460059204543526,
                            0.03390587219411843, 0.030659663542045294],
                           [0.0411636152591135, 0.041840820795236384, 0.006289165041008803, 0.041444803310198175,
                            0.01181460168563112, 0.02010010225092686, 0.026598244408895733, 0.021758845980297672,
                            0.04548379464855401, 0.046747148404922965],
                           [0.017555883310050705, 0.01252332334038883, 0.011119386147507161, 0.029117648077724673,
                            0.01290501563842594, 0.03606808521860673, 0.01705345082065608, 0.028787195562030823,
                            0.009807196793249293, 0.04699251173159469],
                           [0.03161819735797, 0.005171859901711739, 0.02011453876364667, 0.037646680300266,
                            0.029160101605171088, 0.04458879724297027, 0.015655330044383898, 0.037539954139210574,
                            0.03504436333439503, 0.02345113040764939],
                           [0.0442245063659092, 0.04984899885629267, 0.03925817115722229, 0.046778859791756146,
                            0.005097009852411294, 0.006282945329957875, 0.0394150870870159, 0.0114408516609192,
                            0.019826573431206463, 0.02925393843116361],
                           [0.03929060815820741, 0.044975360094231316, 0.03504717222143514, 0.04643841601781224,
                            0.026353774840661582, 0.04528275366307375, 0.04117499082014101, 0.015597438270630443,
                            0.027991736645659834, 0.03386532075550804],
                           [0.029885232091360667, 0.043123484773794574, 0.017579037164636966, 0.007260820259120575,
                            0.027975817629088025, 0.038237169318924735, 0.020916481424128105, 0.024597549907404457,
                            0.036516776648204116, 0.00916878700908921],
                           [0.035574488789061595, 0.04452006616606307, 0.04968943332495741, 0.01679463920915773,
                            0.02461680485860068, 0.03105362094450844, 0.03245783684945001, 0.0211857101619871,
                            0.009975358878541722, 0.0333752367473127],
                           [0.02904650905201254, 0.03749111304242934, 0.03520389531433806, 0.025125985475757878,
                            0.04006849285138287, 0.03787110006245389, 0.011420978120008946, 0.04566842215355204,
                            0.009133511724954961, 0.009314676970283943],
                           [0.02122968436108542, 0.04542648655732557, 0.03241835729833942, 0.008770358863153822,
                            0.01726948540226228, 0.017939212491339097, 0.04006717107111204, 0.016306182621165472,
                            0.040495017274047765, 0.014458120583126177],
                           [0.043917915120468026, 0.045861434491750674, 0.04692073892758867, 0.02060284731620346,
                            0.049145475498737254, 0.023627427672721513, 0.006322819871347917, 0.034813384902757115,
                            0.021312877260214712, 0.025643315921422768],
                           [0.006762600082407223, 0.0308712288426733, 0.038631518852993485, 0.04928551675884634,
                            0.016644151508995567, 0.026518749046421496, 0.03500905108761567, 0.0314992411173461,
                            0.026004944470478717, 0.04146160912454727],
                           [0.013727851845754757, 0.029694378430273803, 0.049982939379518906, 0.009552629687233156,
                            0.008624731295221934, 0.04225066540568731, 0.009944383411752727, 0.03136457314605341,
                            0.007154541077138754, 0.03283695120300022],
                           [0.009425159328793304, 0.04078365373554621, 0.0197958771582724, 0.03236899160612798,
                            0.026243032797664656, 0.042921666615662414, 0.0298267368446643, 0.027909772123757417,
                            0.02772152845698482, 0.017059270075731776],
                           [0.04437985419161166, 0.010957030251586716, 0.019274724183420223, 0.008535204855770305,
                            0.02940628052664039, 0.032703417636145184, 0.02462890661480756, 0.04857954134322213,
                            0.0296705647602936, 0.022733989283185103],
                           [0.018512716743886506, 0.03902537808130411, 0.018723891085888785, 0.015373949547795664,
                            0.011188418792862594, 0.02125503115471168, 0.02407291887650679, 0.012255223673442703,
                            0.04248669438620865, 0.03973156865345442],
                           [0.012332287922823192, 0.017302259619478847, 0.03634580046629588, 0.012785142594043468,
                            0.036042628293494854, 0.032280183662327336, 0.044517310205009375, 0.006464038566465161,
                            0.019061458420366974, 0.04625143719845765],
                           [0.00787515573560635, 0.02651694288114824, 0.04127390079769856, 0.04958104765076204,
                            0.024358194377740977, 0.013580732976568463, 0.0136255585200501, 0.0384291620017501,
                            0.016930822187996478, 0.01102853249895942],
                           [0.03278219935270488, 0.01695330430600597, 0.023925954666502474, 0.02564267306157941,
                            0.027229908674383813, 0.03495442732070456, 0.014059317582564412, 0.010634955293830014,
                            0.04931157440642887, 0.030995171451804446],
                           [0.020024614348596724, 0.04355268922133499, 0.045432642911136474, 0.015566110994352393,
                            0.020747770177143316, 0.025651926487585387, 0.04174463128245326, 0.013880293473510122,
                            0.02158296227824562, 0.049855927633760344],
                           [0.04519221251238573, 0.019941399290274216, 0.03069013115270886, 0.048333675510418395,
                            0.008219072215939465, 0.042747762476877296, 0.04785259610080462, 0.04480944084982907,
                            0.027355839407791016, 0.012108115954720378],
                           [0.028801174609589322, 0.009191344998545457, 0.04226051907475352, 0.01532281253712688,
                            0.012146634361996825, 0.03997263760195996, 0.04644946732885574, 0.019191600627287184,
                            0.017656941486996928, 0.03953586905249713],
                           [0.020280734966115523, 0.03801270509115078, 0.047922252715910996, 0.03515801129018972,
                            0.03820307471677271, 0.020293956044157038, 0.03894974812797316, 0.04452486407822588,
                            0.0352070249668297, 0.03915913895505409],
                           [0.04943755121232968, 0.028297242215647337, 0.04170781724461292, 0.0330963250636477,
                            0.009738425034556884, 0.013403646342859218, 0.0450156807043445, 0.048396089986953154,
                            0.011388912632527105, 0.026127877976316938],
                           [0.02267948717891737, 0.04695107218404896, 0.011477630913399529, 0.006260679806661633,
                            0.022872303848761478, 0.024134427291023275, 0.040641260536609214, 0.04983020467606765,
                            0.03473501179219016, 0.03695892521233795],
                           [0.006443525795910121, 0.04677674503896035, 0.021430356086565565, 0.043250784875802924,
                            0.046508109440743864, 0.03763938698002846, 0.04965118694829587, 0.011171684381088485,
                            0.045219433551632014, 0.03473730549945015],
                           [0.010493971459457907, 0.04888321583443371, 0.013431782075203229, 0.04028934071594663,
                            0.022553065274034563, 0.04178171257459555, 0.03527463171572586, 0.026002688342874867,
                            0.04997623198121072, 0.03614579406312406],
                           [0.019908077647185513, 0.039287136119427045, 0.011421692057893104, 0.008956306983378272,
                            0.047272779452333026, 0.023295466447036027, 0.022131639920183268, 0.030388166828119022,
                            0.014308930089396807, 0.03546501181617827],
                           [0.02799854205862166, 0.049697941592626124, 0.008022216794363194, 0.030256551130443586,
                            0.00809518512356207, 0.040171256202964384, 0.03865533781475379, 0.04627780781292508,
                            0.006810860369300852, 0.039819204130935766],
                           [0.03137555324864491, 0.008410348114506885, 0.037643493384393704, 0.027526843530904833,
                            0.006739505477268351, 0.03726436536020099, 0.011698919709455025, 0.025440714124664116,
                            0.031179685381305167, 0.020866830304867807],
                           [0.008581759420780507, 0.008914673096429572, 0.03028570944293656, 0.01027755268035708,
                            0.04613283162322194, 0.019014647583074484, 0.03406262008629205, 0.01605770704753104,
                            0.021819607908092124, 0.02778964047782809],
                           [0.006262882107756636, 0.02795888800367421, 0.014326621556257819, 0.01193662234498656,
                            0.048354704150625566, 0.005579341664272534, 0.02795095254546779, 0.029241703920666138,
                            0.008652199885841146, 0.049198197318125754],
                           [0.01010556099894359, 0.028646659654395758, 0.01949982486335091, 0.03382723532496524,
                            0.021529734558770573, 0.012256241285149922, 0.03443478491414443, 0.012000327201535356,
                            0.02677599204937307, 0.030942346160216527],
                           [0.029717333070401315, 0.006522739471416796, 0.043606742146445186, 0.035743404464405666,
                            0.013195144313384505, 0.04028818084322834, 0.04042214697381267, 0.038230544474773615,
                            0.041278251336779447, 0.046511286068387565],
                           [0.029328622835840534, 0.036332138433016234, 0.03894531552752423, 0.0050123953375120715,
                            0.02895244884190216, 0.021498134801458273, 0.03953141800166331, 0.015008269809789924,
                            0.04357578130838726, 0.045854385001398296],
                           [0.03890864906665038, 0.03185725246485948, 0.016993481805038835, 0.007148887478126294,
                            0.03040731788819458, 0.027933056936324776, 0.017959660221938054, 0.020692069247677317,
                            0.007592509839995178, 0.03147033180569377],
                           [0.037985992668789985, 0.016079974906203547, 0.014049570855063306, 0.022870321179087744,
                            0.00842396931503897, 0.01937552582221276, 0.02881361216032644, 0.013604703393320903,
                            0.024042133484197023, 0.02223909899352479],
                           [0.043497431627188145, 0.03377521115852707, 0.018511690053967703, 0.03041885111652346,
                            0.03532822456949253, 0.03076155632228181, 0.008730503982231644, 0.010110234786610519,
                            0.013839036549236162, 0.04208647008655602],
                           [0.0493963569680501, 0.03502985500432746, 0.04556813510642397, 0.02148434661978835,
                            0.02172664752469163, 0.00812138774668761, 0.005576931872918753, 0.03082251450042672,
                            0.04318649685981368, 0.013373468801302177],
                           [0.030404405455647814, 0.005417706372876077, 0.00526353967011557, 0.022739258312978863,
                            0.03185144294850971, 0.034419812031954954, 0.007257237173835548, 0.012489902328558244,
                            0.03958842456268499, 0.025104606856140396],
                           [0.03016996178525489, 0.009507369165984065, 0.006207314299071821, 0.02437555652282193,
                            0.025194885264265665, 0.038876450371473296, 0.021276397412607496, 0.042853803393839665,
                            0.03867010898596995, 0.018050336891557477],
                           [0.039667870347433014, 0.019251959707710257, 0.04729959002446152, 0.03727739740881939,
                            0.04308961704201794, 0.019524406966176522, 0.02842752804875715, 0.008067383407314188,
                            0.04893431850459768, 0.016046013641506502],
                           [0.03356953205980159, 0.009597590641886228, 0.029491670850261425, 0.04265727642035009,
                            0.006139904522808859, 0.014198864930790544, 0.03182815860106481, 0.04913543618407509,
                            0.043941934261434105, 0.027827654430523905],
                           [0.012717624563750234, 0.02050420922520509, 0.023612399730001433, 0.007513947794736925,
                            0.03164651476305524, 0.010547380977711896, 0.027258572836965693, 0.03295207083742279,
                            0.031088942683242735, 0.025832335240647215],
                           [0.019162482586271387, 0.00755240724790175, 0.018175593750814827, 0.012527684868587009,
                            0.012541006901793928, 0.04516390871055098, 0.023672547698471585, 0.04684895298196437,
                            0.032781843562438906, 0.03819263051779108],
                           [0.04984147501531854, 0.026788393183767578, 0.026389502350830957, 0.04477651687016148,
                            0.007869288921945763, 0.016412088883375402, 0.04750012436329631, 0.035771077813205154,
                            0.026026607145831857, 0.011945125484645713],
                           [0.014817600206669054, 0.020404317404390727, 0.0256690241100762, 0.012168626963116363,
                            0.02114242904735943, 0.00977660981173967, 0.02190292811676232, 0.045401849341012275,
                            0.01391501520195557, 0.04367217785348751],
                           [0.046693030615376205, 0.04276938236610036, 0.02761774388718427, 0.010776174263282928,
                            0.04926068516299274, 0.023806913041997366, 0.011521106143871678, 0.008966617173576826,
                            0.04188338562208568, 0.027143008227962218],
                           [0.01868390257765944, 0.03992865995356449, 0.0427964175565099, 0.019654258863199256,
                            0.03657799172345969, 0.017345216074453752, 0.01178221073019534, 0.03371354787536063,
                            0.040027173725844516, 0.005438621611553767],
                           [0.007638777103335407, 0.04169944411527977, 0.012487314116618627, 0.02500180711830609,
                            0.042773722687946505, 0.03660476563559434, 0.014632791241985563, 0.03413136981685483,
                            0.034249443487031364, 0.042381259698105606],
                           [0.02255196582563994, 0.025581339388123132, 0.04280196484003643, 0.04846006527258921,
                            0.020189804916347025, 0.04126133186487824, 0.01506680945180755, 0.02426551932417951,
                            0.03727347606883956, 0.028970662412195293],
                           [0.044454103242514344, 0.04623335295656458, 0.012551211108377463, 0.04375170435868416,
                            0.018329716941960384, 0.01001436172223806, 0.020823565246742417, 0.0061984116790001565,
                            0.008773737903731663, 0.005907450600916821],
                           [0.019315025298214022, 0.0257273377310018, 0.01529210775681631, 0.008109921840002803,
                            0.03980071790478404, 0.02460648915448659, 0.031232101308173787, 0.033236610086350316,
                            0.03891760650945243, 0.028239910104548804],
                           [0.0437385114619498, 0.022268898700914135, 0.028079424435640616, 0.031000057363562424,
                            0.01936644941263779, 0.02777277844348004, 0.02405265715922644, 0.03951815372099571,
                            0.025795974527424263, 0.0057513081016474935],
                           [0.04631313894522888, 0.034226024385256563, 0.04184234817509577, 0.005939793380146895,
                            0.04446216641248348, 0.04957597096111054, 0.04374885219034548, 0.015685352307694193,
                            0.012226209896387667, 0.0064143688866745725],
                           [0.04786290269512249, 0.008821266285359286, 0.03225949098199152, 0.02023158337314697,
                            0.0220787252518127, 0.03704977164488012, 0.030154661167133376, 0.027959442780161336,
                            0.027271073434204186, 0.005169761944652857],
                           [0.03431137363845603, 0.026171096840383944, 0.023780972960421105, 0.04709506983035284,
                            0.012339901306631231, 0.04687986731010863, 0.036517051866501324, 0.010807395961790547,
                            0.03885197280639218, 0.035272554062823744],
                           [0.01399830194428697, 0.012444842623199014, 0.007618010601319234, 0.044606994046213176,
                            0.0102968298297543, 0.005380873951777899, 0.005362441145976155, 0.023083090111005795,
                            0.005326281235332872, 0.02959698411932179],
                           [0.04540012549639657, 0.008750508804626751, 0.013167702426487016, 0.03667387204698012,
                            0.01919633515572237, 0.00872708067183115, 0.006101685919541118, 0.0169864173946245,
                            0.013622868398258037, 0.03489648654749698],
                           [0.019774964068479405, 0.037688884190687444, 0.02928297725623202, 0.025267269088843,
                            0.03667845667302572, 0.008196538802359658, 0.019749896181392154, 0.0339741314773305,
                            0.02977583701579152, 0.04771402908906692],
                           [0.03196432916488926, 0.048731966657175656, 0.04193683417530919, 0.03403544556808927,
                            0.009489833681275258, 0.041012800678054816, 0.022840704908421, 0.02049435141778734,
                            0.0067222459968865455, 0.009474267628018293],
                           [0.02159833716658902, 0.032239318567429356, 0.008381533732851604, 0.03427272326910363,
                            0.02553809116201156, 0.028285721176249154, 0.017174615686013226, 0.029842590839172057,
                            0.01729114801229714, 0.013647020518960482],
                           [0.008172208949876336, 0.01959554650010701, 0.03547012360498639, 0.01177756001345524,
                            0.03397557940039671, 0.03543493271946021, 0.018841474469296234, 0.029123826677616923,
                            0.025385118618639072, 0.04572691569695158],
                           [0.012981962041249791, 0.04407575726920341, 0.008602921548592182, 0.024326214114634405,
                            0.04511798497770677, 0.04723370496081717, 0.018224393538233883, 0.03884765774582994,
                            0.0322466329083019, 0.04405460053869227],
                           [0.026454663636482557, 0.03912696818967115, 0.01801594231265585, 0.047632347057132714,
                            0.02601479993331284, 0.04906397681237694, 0.0059675443884829645, 0.03723240166651933,
                            0.03466011389146142, 0.04789420179846187],
                           [0.03561721401397214, 0.03233033906818708, 0.007427372349424425, 0.03291545820643613,
                            0.03549149739513942, 0.027104065114550432, 0.04294544434953486, 0.024462052591043098,
                            0.045145351996641506, 0.0371502605257528],
                           [0.04161366704390129, 0.030340093835844458, 0.04569542568002597, 0.04215112780002698,
                            0.03110697641948795, 0.04579955595493207, 0.046846697780258906, 0.01777535737516741,
                            0.020816866177677223, 0.025535577797122413],
                           [0.04590803894704191, 0.02707350492197537, 0.02523476927456025, 0.028220671311359143,
                            0.04883042791332842, 0.023033818892512572, 0.02804790066140026, 0.024723452631841416,
                            0.04218603883914568, 0.03197570505027897],
                           [0.040959238030844614, 0.021022350554016363, 0.010309800800958627, 0.019398660997901607,
                            0.03757348696132841, 0.04672730513882234, 0.040717641591185946, 0.044308425445399195,
                            0.04690806678201155, 0.046117086414899644],
                           [0.03457392046279225, 0.024634814228469398, 0.007929626472189972, 0.005810872771831495,
                            0.010859453449552248, 0.014845504984797763, 0.012458734765900181, 0.015266418297436555,
                            0.021767939177685852, 0.01890201368481146],
                           [0.02926056009984124, 0.006335049834032135, 0.009388071856938872, 0.045312253747718835,
                            0.022338278750545094, 0.010141893488203334, 0.044067161026489506, 0.026477755100907538,
                            0.016380147840990816, 0.029698771062777633],
                           [0.02444124476226198, 0.03449012328152919, 0.005388061953783402, 0.0418189037139393,
                            0.03207682207349478, 0.006795715653877511, 0.027198900125469064, 0.02359156913727002,
                            0.04431667424086875, 0.007685602095290209],
                           [0.04233582215402333, 0.021253143747988056, 0.03540617385026174, 0.03945202038607447,
                            0.03716902600094228, 0.016666999131173463, 0.018658610648894094, 0.029561151391305145,
                            0.016734926260300948, 0.0377155173533436],
                           [0.016429387484232668, 0.041968480751398946, 0.044412623248531735, 0.01418463353519445,
                            0.027427317991469788, 0.011396020549709791, 0.03201162344225101, 0.019359652878486962,
                            0.008025538502295601, 0.012285534500466776],
                           [0.0461169863828547, 0.01320121209236419, 0.02297282346402524, 0.013859662952008425,
                            0.03753159581760457, 0.0374364313485198, 0.018374636671519978, 0.04963999580836249,
                            0.020752663028596918, 0.044929653724974976],
                           [0.0348353605276561, 0.011626605701702396, 0.04424854387980575, 0.029804927496234687,
                            0.021502845941580393, 0.02492457414440504, 0.020527972029024195, 0.04198441840030915,
                            0.03493635257598022, 0.024225073527497056],
                           [0.03327443951204401, 0.04912809283326613, 0.007370902794338729, 0.0487750396062324,
                            0.041489024509602776, 0.006168575420890765, 0.030609580110544148, 0.0069617153710889515,
                            0.044974585342804844, 0.02941189164104407],
                           [0.005836966870657984, 0.02329692099449547, 0.04459309371856948, 0.024077687556031343,
                            0.036365580711322586, 0.024340212902192317, 0.006934276273641156, 0.028670066920879887,
                            0.02334386716602726, 0.040125577136217075],
                           [0.023449946823073806, 0.012921117198040899, 0.04390132535688806, 0.011034676831210986,
                            0.024332633234212104, 0.041464098230672666, 0.008259501641523644, 0.011782310986976284,
                            0.008281983599638528, 0.016132974457698412],
                           [0.028420058604637, 0.006190782778820384, 0.042415980406561644, 0.017760342475409134,
                            0.025739775239623477, 0.028082402093697743, 0.011252900425919182, 0.042144114308632005,
                            0.012782992005683301, 0.049536180578635125],
                           [0.010972122737438512, 0.008462552958467168, 0.037152757026517554, 0.030619773385736272,
                            0.020995357153220436, 0.03971277226545987, 0.03810235359392447, 0.021017234562864924,
                            0.010977501856369917, 0.04080344428680572],
                           [0.009248746711829592, 0.024923069505199714, 0.007778796133198228, 0.0206320707023898,
                            0.030558828051885676, 0.018109559211979435, 0.039742018490998926, 0.03391867993565977,
                            0.03590677069412612, 0.015328987465545483],
                           [0.02959298076731013, 0.02780858428432737, 0.049389890035817695, 0.028641831948846164,
                            0.0418877934523711, 0.0076725445066338355, 0.040270797163782575, 0.04589545355195262,
                            0.028763941285830978, 0.010878798618418253],
                           [0.03174302958810781, 0.024196661914196158, 0.04943552498856417, 0.04298344971628572,
                            0.04973889793254842, 0.0275347818940253, 0.007365491761216356, 0.023441267191413387,
                            0.0056431482142384505, 0.015806737301489796],
                           [0.03616917533222368, 0.035260071161751484, 0.017335158090838777, 0.04906171311170714,
                            0.03943200111605474, 0.0068860709395945205, 0.011909563508636518, 0.033146142113575756,
                            0.02020991159784234, 0.03544108132101999],
                           [0.047354287022541165, 0.023035238440520717, 0.023190905149619634, 0.03472049228626103,
                            0.044685621940086975, 0.03408809069507309, 0.03481151989960842, 0.04377756708351357,
                            0.012654448956937257, 0.04743819607044992],
                           [0.005060499129400217, 0.015639317116796725, 0.028450896689743408, 0.03287837228590166,
                            0.015391244741852547, 0.024680643949781957, 0.040957941462738415, 0.026707022185542467,
                            0.03763378044221751, 0.04006924729376534],
                           [0.04242760360413332, 0.049425100209002766, 0.03330789979339835, 0.03585023795053739,
                            0.011405286982949289, 0.022582865748092287, 0.007971231789582426, 0.04313800646361839,
                            0.0432760571328361, 0.028844391743789933],
                           [0.04608064733092483, 0.017700234148806976, 0.035863610132392844, 0.04632526634830558,
                            0.017449224675565294, 0.014644185990293414, 0.03623051020973353, 0.021358611867054594,
                            0.024276869827814882, 0.04464104298343986],
                           [0.014281235410580757, 0.019388776277248075, 0.03791140993670208, 0.029588482807719436,
                            0.0330605112253195, 0.02292812801800476, 0.007016443989457052, 0.008885869730258563,
                            0.03918566816881127, 0.01933268507761038],
                           [0.027878908044099735, 0.042432139660326114, 0.045952421064142435, 0.04745102844478006,
                            0.005276264479686425, 0.02528401647014983, 0.022106167062908664, 0.03853793091979959,
                            0.01065688123942777, 0.010122150044425845],
                           [0.009919091493449047, 0.02871636617364964, 0.01162199877721704, 0.03184488400984731,
                            0.035761546998429254, 0.034840231132836554, 0.014842665803762454, 0.04398797887865041,
                            0.015825350808354278, 0.023271886379483513],
                           [0.026830888422347633, 0.00570766181803421, 0.03088696339284264, 0.011201174525027741,
                            0.04382850696680357, 0.03050221617131068, 0.0434584417332746, 0.030315905401733123,
                            0.042645919754863856, 0.013255379627574209],
                           [0.022385451410316212, 0.0452342419552999, 0.040611326769525065, 0.04009245680741943,
                            0.0256347772136763, 0.03893957572638064, 0.013225299866650074, 0.039235832761515335,
                            0.046016937981641595, 0.008392331036885074],
                           [0.015992028502677648, 0.01589315561063673, 0.027892211209900725, 0.019794702068151676,
                            0.010165367774109795, 0.020923025943129347, 0.01012950922290783, 0.04248848804927573,
                            0.014228035759627791, 0.02909634404426003],
                           [0.02763748208656754, 0.029318049451170013, 0.02914858911889275, 0.0261958994735695,
                            0.010486979048037269, 0.023487475901798036, 0.01631807737009444, 0.02036213558608029,
                            0.009267491972559421, 0.008681328714555476],
                           [0.047496413511336884, 0.022012904641136786, 0.009596376447350594, 0.027368166845148083,
                            0.0055161185821983125, 0.022180831738519854, 0.026303867308414778, 0.02292155461569345,
                            0.029093082626196573, 0.041377652134328716],
                           [0.007439692886012924, 0.03038913049090245, 0.028762149506032397, 0.01716821310283073,
                            0.018538882631958516, 0.010864463852751971, 0.027405181266183964, 0.04532947468732769,
                            0.007900874769452497, 0.008171059747993448],
                           [0.03858910799534244, 0.03726249248840176, 0.03818034724618878, 0.044266764918349605,
                            0.032279906074725126, 0.017092272504293682, 0.046175983568776544, 0.030987848170349493,
                            0.04097030477705016, 0.01655618847665339],
                           [0.00810701341161837, 0.007748966028104784, 0.03115875240145704, 0.014601811965664981,
                            0.04969559630554181, 0.005858628098881031, 0.04900190874390472, 0.005137284655215428,
                            0.04674675795316839, 0.04537807081464956],
                           [0.016254436911888676, 0.005905049253845873, 0.02129871842059398, 0.04072645151212318,
                            0.010857707487124378, 0.036874007208054164, 0.021076361313878935, 0.025582367714991307,
                            0.02246101330529786, 0.028058750359859462],
                           [0.015648388018349633, 0.013076092073091839, 0.005971280229438475, 0.008668397404787259,
                            0.006473988456414769, 0.013169941760393211, 0.022373678617481366, 0.01466634240912363,
                            0.0462347853029064, 0.03432759354612876],
                           [0.025898282182679526, 0.022974779729705234, 0.043318418009869984, 0.042940979009334274,
                            0.01969894434070318, 0.01894158369793608, 0.04107938671696169, 0.047316084074919204,
                            0.021429270432929422, 0.011573755081832903],
                           [0.02559177242585765, 0.036667229025862176, 0.022786017448299868, 0.029182710782349863,
                            0.015067743864043966, 0.017506672371443967, 0.02497975849629047, 0.009748297163386859,
                            0.009875021616183501, 0.018391772014639037],
                           [0.04143646074092881, 0.013493973670322682, 0.04897589045710939, 0.016282058927207053,
                            0.007073315725008879, 0.012837518235825521, 0.018059831697696444, 0.010266119366168426,
                            0.010839649919846393, 0.028255602200908572],
                           [0.03641213754633206, 0.016142697898448348, 0.009267558081239847, 0.011485031534184882,
                            0.04315051316987066, 0.026426111542942964, 0.0368662955251757, 0.043497651260809865,
                            0.03545448048482898, 0.02532727461020777],
                           [0.02544167930496633, 0.021580056156826757, 0.04749447572110551, 0.018155327835822983,
                            0.010016503632357551, 0.049946384611719546, 0.045704081574330556, 0.02942484802785963,
                            0.007040393195094058, 0.02664925446351005],
                           [0.016321947141466106, 0.011367148141646655, 0.016248081770418282, 0.02012210248489131,
                            0.02705065094446396, 0.039948563049572186, 0.047399989739954906, 0.0471493716257454,
                            0.022955104485305265, 0.043995695399437505],
                           [0.017249814575237086, 0.03335378481718017, 0.0421209348871046, 0.012966988631754599,
                            0.01382073927180874, 0.043228815257831066, 0.020653593776462177, 0.01752123541672735,
                            0.00844374126387933, 0.01764498140835669],
                           [0.024101333042560062, 0.04825767298029825, 0.029831608958585065, 0.023546807348722654,
                            0.04017348434776886, 0.04354668603064932, 0.025753839901028246, 0.008552106473626118,
                            0.04700680804759851, 0.00954282646482052],
                           [0.03769151950296249, 0.04712454135035415, 0.013790426029279549, 0.01946580426133198,
                            0.0499839266182196, 0.0197429247768044, 0.006727328433304055, 0.027153826873928297,
                            0.040132166814320205, 0.037490160974396766],
                           [0.015948203924990426, 0.011081808647324746, 0.038450390959496185, 0.022630004549856444,
                            0.015048905539135942, 0.04786753925955714, 0.03727599403717956, 0.019946306016116796,
                            0.04304875128481952, 0.027613049054353914],
                           [0.010799913770463136, 0.04063497858757139, 0.014253126265209796, 0.034424214645046275,
                            0.012481750455237021, 0.03435599819076724, 0.02814846240797378, 0.019500969514009384,
                            0.030324200938452534, 0.04749585988897971],
                           [0.04893351040348999, 0.0331502116506189, 0.02119608545185108, 0.022364773495630656,
                            0.034488028602470217, 0.0288550372227933, 0.008823757706075434, 0.03349840937198577,
                            0.03211020810735296, 0.018164844207873707],
                           [0.014707434027433831, 0.0451182582683968, 0.03974573332602354, 0.03895498216949546,
                            0.013886450246375956, 0.03331418966678125, 0.04204166583558061, 0.02824446357305828,
                            0.046554410477733864, 0.04065406677871251],
                           [0.019671650068365215, 0.0366089425929175, 0.027141973169945745, 0.03823729164326146,
                            0.014648328662279983, 0.0074133518254229595, 0.012192910653353903, 0.023935163641297646,
                            0.04103276652902801, 0.04901513861713719],
                           [0.048715512834754705, 0.044607281184170705, 0.007024776852829697, 0.04731114977410828,
                            0.04479267943618372, 0.039247220371183354, 0.01300136501286803, 0.016157533852077226,
                            0.03303227364276093, 0.01027855291046691],
                           [0.014988215461523569, 0.008132449073971407, 0.025393290868961915, 0.04096452437499974,
                            0.04869873119927467, 0.03300696318657049, 0.015371590725706902, 0.04552222840817834,
                            0.029911622299104604, 0.01558862129794028],
                           [0.011538462782029215, 0.04126291632191172, 0.010218618795693335, 0.03248973245155994,
                            0.007821432824902535, 0.03717522468199516, 0.04923432134633387, 0.022127130197192816,
                            0.02713150997097297, 0.020417647187934707],
                           [0.019000154842360752, 0.03547737880071773, 0.005851995694646983, 0.03345669243042699,
                            0.04119446594554422, 0.01614148483966762, 0.04985809868801293, 0.037777237806138814,
                            0.03620335787558969, 0.04116356803112613],
                           [0.04369973625247125, 0.03860022462799503, 0.0336951580598008, 0.029727299791673024,
                            0.03237497638189412, 0.048209393486592395, 0.025282207299423497, 0.007452378293569126,
                            0.026371906150402104, 0.048485852522232885],
                           [0.03871445314952189, 0.009664425748667799, 0.044003037771835105, 0.04108791952870312,
                            0.045765319947963214, 0.03249511115454251, 0.0050516622622448154, 0.017105768300896714,
                            0.005835993330399061, 0.04551286875782036],
                           [0.009636890377914363, 0.017551692669206056, 0.03130484529217659, 0.04811131529448821,
                            0.02487400064773814, 0.03604269832071599, 0.012258211273591906, 0.023789591025173366,
                            0.029530190296906716, 0.035043452115713705],
                           [0.047623652594978215, 0.01437196666244674, 0.024079089449936933, 0.0341386856229639,
                            0.03065977876942921, 0.049740932675925414, 0.04249401361978695, 0.024901989491402224,
                            0.007389496307059453, 0.010605335780354684],
                           [0.04482957349765231, 0.02852922622847079, 0.044901258475596235, 0.029946230163767934,
                            0.037793184229077674, 0.018236127340678858, 0.008321169131021279, 0.040763411013081584,
                            0.023140833390222465, 0.03323561905104303],
                           [0.034920546054188116, 0.02360880966889136, 0.04478667699598496, 0.038835523153517564,
                            0.0395925428910084, 0.025432915329073075, 0.04520282916360795, 0.008280047289475503,
                            0.03806090027049623, 0.035865798315217134],
                           [0.04061546597859185, 0.03284845908593454, 0.006380804208328487, 0.045502415915684345,
                            0.0370651127422444, 0.04595437912876336, 0.005854787065046704, 0.03916822100904829,
                            0.007412607487045712, 0.027045709549775674],
                           [0.04463952235032961, 0.049323189942483324, 0.03446425372359831, 0.03511684150998897,
                            0.015876823341827682, 0.013075711364388728, 0.03281339876665184, 0.03969184474777724,
                            0.01837780848317323, 0.03245012433319008],
                           [0.02094281941111059, 0.02667548287775969, 0.0121359332462922, 0.034666467215667296,
                            0.00889987955583112, 0.04660109389548768, 0.014341612694460444, 0.04013648537084963,
                            0.02280772645308614, 0.03672771061133504],
                           [0.048269043965405004, 0.04552979225343144, 0.03266570379002869, 0.03541613342072058,
                            0.04084211225626625, 0.04391489010709909, 0.03856967908530223, 0.026954768375868958,
                            0.028930144198261974, 0.006944569653054268],
                           [0.04532626797135226, 0.04004493562827871, 0.04852709930395886, 0.029442208600224493,
                            0.009058437440345169, 0.03361468151886859, 0.02687060471029929, 0.006549190446647042,
                            0.04971589520326736, 0.019093610627583017],
                           [0.009747255555345048, 0.042297889104874795, 0.01892434262407949, 0.03969230415034344,
                            0.024156192121553254, 0.02913359461588592, 0.010425461513005331, 0.038672112629815625,
                            0.048218141740767925, 0.04287692987298655],
                           [0.017990169016560996, 0.014102930227661053, 0.014981293750806781, 0.025875868271315484,
                            0.018757657817321523, 0.03670817839682626, 0.011054284626220136, 0.005946700713321166,
                            0.025593888201624355, 0.02120493390790308],
                           [0.04799325087588625, 0.021824096139664705, 0.046797143582225184, 0.048614516083915496,
                            0.03634902069457697, 0.01512109829516839, 0.010351963898190352, 0.015891095351270115,
                            0.009491680942133316, 0.030542223340721316],
                           [0.022908646929880247, 0.042189501357825954, 0.036454931678264, 0.027132428065986466,
                            0.04335255293933045, 0.02517601342573461, 0.013847891678908583, 0.041284787715042626,
                            0.01949588737288385, 0.016304782686801793],
                           [0.032190865148788494, 0.018514641397573008, 0.034234861467449376, 0.026341979257133552,
                            0.017123227228377006, 0.04708403774069552, 0.03344159805854261, 0.045153475032754575,
                            0.019587907428877675, 0.03309330666008962],
                           [0.008826359721443343, 0.013804508056367518, 0.01584686593240758, 0.02787936570464045,
                            0.023646427076001106, 0.03843233872126712, 0.01863760531178988, 0.009518015176725337,
                            0.038222962910880245, 0.030449025572212185],
                           [0.037411594159344246, 0.013307326067009385, 0.027666347814681653, 0.021898231524915122,
                            0.02648615055605808, 0.007392700081647743, 0.0261074239460095, 0.01092731460031773,
                            0.011851668722672496, 0.019726775467941254],
                           [0.03880687169713109, 0.020613069634720343, 0.042506475390086645, 0.01885816161726137,
                            0.02913687684812735, 0.020627908091914393, 0.01621232145067236, 0.028325350848124015,
                            0.046406855189944925, 0.008108281333599143],
                           [0.04199262733444973, 0.018303536791970738, 0.02550759793659948, 0.04657447919140387,
                            0.02040821161416805, 0.04844280655970673, 0.045094073202940975, 0.034264090228584745,
                            0.025505851188989705, 0.024648322496659127],
                           [0.011627805993903994, 0.03146798496524305, 0.005644861248225955, 0.040207822316895896,
                            0.009633195437626606, 0.022359792124569317, 0.04479226806690334, 0.035122784819636016,
                            0.024151676502550797, 0.03571635534751206],
                           [0.014590768535307296, 0.022186477469817386, 0.03750431776845509, 0.017566062548548346,
                            0.03737935317136114, 0.011834643323276606, 0.03744937358359609, 0.04696965562544942,
                            0.03142526919532474, 0.019480012032353704],
                           [0.013565830777237393, 0.031347599349846794, 0.02721350607718644, 0.03993380812154905,
                            0.025276315213639537, 0.029888857456426474, 0.02153015037791119, 0.040898674849738914,
                            0.04735850390857539, 0.024714057307869054],
                           [0.034027785954532624, 0.02699447045698247, 0.0462520243065106, 0.04692349907545913,
                            0.030090697719137076, 0.01607162220809745, 0.015989493628187103, 0.04361202124829744,
                            0.024793086325951754, 0.022610078284551013],
                           [0.04327504895316891, 0.01847395952488612, 0.03947753095658706, 0.012360617864587111,
                            0.03465296894979066, 0.030013296245809742, 0.017701938633684907, 0.046652089375642185,
                            0.012839886459052218, 0.034424482929543254],
                           [0.005342627278474841, 0.007385086208925909, 0.016042826208250927, 0.00910591549366506,
                            0.008543780557878715, 0.04540933923747193, 0.03921101697463935, 0.04052364206404105,
                            0.03700288829494274, 0.019987404895921632],
                           [0.020515960238163964, 0.03818886739422132, 0.010618367008266488, 0.0069737683118912906,
                            0.027522341197703956, 0.047977697919305964, 0.037964628609332945, 0.03623773844784678,
                            0.03361153391907893, 0.011252257103491804],
                           [0.046928589145630605, 0.012735965608161414, 0.03172528471109771, 0.011346197532180413,
                            0.04027039702337897, 0.026872192483586927, 0.041316606981698735, 0.043069871572086665,
                            0.027837111370193242, 0.013034254157626077],
                           [0.007736430089361309, 0.02110058912833098, 0.04876953245001066, 0.008180961860318227,
                            0.031289658814345575, 0.01636412389998763, 0.03546781791815909, 0.01718511970625432,
                            0.005116085730021025, 0.005975438282789297],
                           [0.04709128498440344, 0.02780411304369971, 0.008672246229928841, 0.01668523651635733,
                            0.016793759574467854, 0.019927328658172976, 0.014994463263979387, 0.03636969283026815,
                            0.03182962438430069, 0.014729023402439468],
                           [0.025395701550793816, 0.007948245186034932, 0.03998010575431144, 0.046157389484275876,
                            0.022736233624398156, 0.014484700322418418, 0.037121191140989544, 0.008367178670831942,
                            0.044073660208952654, 0.02413202948942006],
                           [0.03071539544051428, 0.040236969374164905, 0.014555658473035991, 0.03192505618623753,
                            0.013283810945518528, 0.01270043770051141, 0.036359884463952205, 0.04618361213370948,
                            0.04378681185889039, 0.026505668043682207],
                           [0.04501098399783847, 0.017995374340400702, 0.022249498324084706, 0.04328020034744065,
                            0.005682195486636417, 0.0218073536047454, 0.009295165127389694, 0.042509594329485166,
                            0.029360300524788267, 0.03788991119103986],
                           [0.01570023632164763, 0.03471434444281271, 0.02474335952654956, 0.020453631422190206,
                            0.015268700187957961, 0.0203658663667997, 0.04265758304001992, 0.007957500679345728,
                            0.0441648075784985, 0.03847627215400677],
                           [0.033488261629230916, 0.021739948119261478, 0.03156016063207305, 0.03369562995475976,
                            0.043852469305194416, 0.043850806860204164, 0.04212333918619587, 0.03461098170986191,
                            0.025113500003015213, 0.04444614253538363],
                           [0.0181722559962993, 0.01901854047927134, 0.026926362709403913, 0.0412790019962988,
                            0.04355478196781896, 0.013526714086712577, 0.008148193004338689, 0.005432619454479919,
                            0.007712621438662706, 0.007259762799578234],
                           [0.023619124584562265, 0.02005408697410415, 0.017044573712188225, 0.04577156079216256,
                            0.0413824715913861, 0.01527845119039654, 0.027722924151447812, 0.047761729947770636,
                            0.036451789667169616, 0.008585532293489781]]
            self.y_data = [[0.0010820129274286075, 277.17837825554915], [0.0009323107875810338, 336.8506288361638],
                           [0.0007381082757183801, 210.43910080354814], [0.0009044338693569956, 283.75003247910496],
                           [0.0007820706634125296, 221.41187066540982], [0.0009329757837946907, 268.05022896631294],
                           [0.0010755995661809604, 338.49629584969296], [0.0005358953921165279, 219.215565789671],
                           [0.001030515430207179, 277.47917346759766], [0.0008678148024340049, 252.30043450139195],
                           [0.0007242612508529203, 233.3356051012411], [0.0008812225423461756, 254.68909742919243],
                           [0.0010179549498820109, 312.0807398997275], [0.0009796627489023822, 333.8586891738281],
                           [0.0007246374215506441, 249.39324187941114], [0.0009458361873356964, 322.10099823340755],
                           [0.0005478103037624586, 223.48162585497099], [0.0009544528781149688, 278.845771211468],
                           [0.0010453028123889455, 326.6332905422273], [0.001061704553606444, 356.68529308115916],
                           [0.0009331050061536823, 305.03729383702796], [0.00109600831785665, 321.2019629980146],
                           [0.0008993113682316428, 276.00082450422605], [0.0006203635343529821, 262.481614159393],
                           [0.0008261514896471113, 270.2926305187635], [0.0008646754420141856, 253.90318101275895],
                           [0.0009382754701132452, 267.91474122566297], [0.0009549074226557724, 247.7445932695182],
                           [0.0009248972826317145, 296.78771646456994], [0.0008426868424776744, 270.3771804725751],
                           [0.0010242710469329573, 278.504897839672], [0.0008488214784360192, 296.6051506102066],
                           [0.0009272206146521327, 304.47156634428234], [0.0009675077143221631, 318.9291102452205],
                           [0.0009591741494669367, 332.0594235505277], [0.0008999931402809357, 282.28756577207565],
                           [0.0009329663894175023, 310.6771376088314], [0.0006137750743152876, 235.11169756064453],
                           [0.0007356369351803808, 244.11765572851937], [0.0010339774276268338, 372.64636783503437],
                           [0.0009856175850875453, 355.75393709621704], [0.0008665444007931381, 294.59662852737245],
                           [0.0008176324361491943, 255.88758563010748], [0.0007767214723566774, 273.988966719912],
                           [0.0007938696558141296, 303.39570789818606], [0.0008230406142326422, 235.6955798535061],
                           [0.0006450809392531208, 235.451886305574], [0.0005983854241272961, 207.1793191297868],
                           [0.0005223516950510278, 179.32934515102644], [0.0008633009479562989, 297.4509746450242],
                           [0.001064885418855673, 351.57163784267595], [0.0008562918455894083, 273.6760021666275],
                           [0.0007543980915951748, 252.00129460822862], [0.0008175744024235935, 266.27989694348435],
                           [0.00106163251579207, 304.22959842101966], [0.000994953222336292, 312.87515601499814],
                           [0.0009325040071136358, 293.05182859538695], [0.001030337787675515, 316.5072930055581],
                           [0.0010325174150095315, 289.015142956802], [0.0008309802841538924, 243.12215679925728],
                           [0.00100763779045822, 287.86729763811525], [0.0006370128909604022, 228.0450566321156],
                           [0.0007296532996638133, 234.69730899375793], [0.000815712716880506, 212.12897410464075],
                           [0.0010457481571748732, 311.5460296955495], [0.000878212625373908, 268.4306562619377],
                           [0.0008042133942227495, 222.92014062663333], [0.0008637285068243292, 274.37711133440945],
                           [0.0009268268774164982, 278.6724986626511], [0.000864993265614577, 223.17743946775198],
                           [0.001007063888026803, 312.7687340228019], [0.0007438882013462038, 252.04984626304932],
                           [0.0008676827114292385, 240.11997333364664], [0.0009354670637019085, 299.9782493782694],
                           [0.0008592221848907737, 285.1737345237057], [0.0007919752731517423, 260.2873894360304],
                           [0.0010622378455693443, 330.0652704107769], [0.0007651459261973642, 273.9916396628783],
                           [0.0010376984488353845, 321.0561666504136], [0.0010866811727953424, 362.0360718425305],
                           [0.0011188353076173304, 349.7572273035563], [0.0008028273961881564, 235.47896654537877],
                           [0.0007111693141691395, 202.82856037075234], [0.0009377548825230351, 319.5701968899594],
                           [0.000782332298340364, 293.65207100359515], [0.0010890920032824846, 341.35836414812957],
                           [0.0010983803521672944, 327.59302207093555], [0.0008334112643807253, 264.79039146198915],
                           [0.0010382870638917633, 315.55569717998054], [0.001004242143270448, 284.3904127844292],
                           [0.000842582897688313, 282.9775781860253], [0.0010343587973370037, 326.79588452066486],
                           [0.0011419231905929687, 359.5722292348485], [0.001068488730287323, 316.2430549164169],
                           [0.000849928344857814, 306.53063424674707], [0.0009550293789389592, 314.8782459310994],
                           [0.0008016386107809071, 275.41640308700073], [0.0006604770669136767, 207.15462644323023],
                           [0.0007148473603101995, 262.98623976173394], [0.000752561416735904, 276.35585026300174],
                           [0.0005803758573408335, 221.50576481071167], [0.0005750411661985022, 233.44320023123072],
                           [0.0007924715630042722, 267.2888815023467], [0.0009256787385492942, 286.8692840054072],
                           [0.0007523439379995601, 248.60342458807648], [0.0007766717266881372, 250.98771266385978],
                           [0.0009068251196423124, 270.98391172844384], [0.0005974088163360437, 194.48527946778023],
                           [0.0008673410832931923, 274.087405219923], [0.0009091713197185363, 276.92187197424374],
                           [0.0009748437271465991, 280.2027852186342], [0.0006335073291301921, 212.72572679038566],
                           [0.000985957335240697, 288.44450751117347], [0.0005442803256628047, 215.03276953094124],
                           [0.0005439271729393772, 231.76805070581563], [0.0009431784047322142, 301.3273197912688],
                           [0.0007138733031542275, 275.27296404608444], [0.0009098986333506634, 318.9742424837821],
                           [0.0009000422347077321, 259.9526430092124], [0.0007605689823605838, 268.998330418074],
                           [0.0009260449009029487, 283.4700934548634], [0.0009380991651998151, 295.16744359124993],
                           [0.0007189140298267816, 256.09123199778577], [0.000854385861092261, 228.72673896626188],
                           [0.000804892996790897, 292.6017498094742], [0.0009092282047560177, 289.12911003933596],
                           [0.0010189370227040235, 283.76497628477983], [0.0007856364573652528, 274.37873504629187],
                           [0.0006952032047863508, 271.7775371929617], [0.00045699219706748145, 217.29766211572985],
                           [0.0007459083580730948, 262.7791555782416], [0.0008965361160678929, 263.39557474424544],
                           [0.0007578353764669639, 230.6658234257016], [0.0007997639530825145, 280.284867053779],
                           [0.0010440609143250908, 303.5119318705695], [0.0006353019329772587, 182.56671748294477],
                           [0.0009169505084396758, 339.4732833177257], [0.0006002389500912959, 249.10777546140523],
                           [0.0007598940832006765, 250.7136894043523], [0.0006298393583935397, 188.98892129507],
                           [0.0008786017732836573, 286.3059396721919], [0.0007572213717275464, 190.06580767811042],
                           [0.000827062062238224, 262.80633443275815], [0.0011067986182853608, 354.95393725311897],
                           [0.00077513272273158, 303.2411417847852], [0.0005664611983399343, 221.92969664023494],
                           [0.0009372535315107277, 279.9909530973747], [0.0011597077213652196, 291.42694196385463],
                           [0.0011283436347862472, 356.0175714873607], [0.0008078694108439973, 255.2611562257514],
                           [0.0009911407122179452, 299.2431959296404], [0.001058375081880876, 280.34468476717444],
                           [0.0007699791933338829, 254.38007652295707], [0.0010988008263678945, 318.16823698321207],
                           [0.0009124660004251206, 302.6886108923252], [0.0007128068453284988, 235.13364488163495],
                           [0.0008573109736919524, 274.0556887432053], [0.0007291204431164425, 270.8695136466829],
                           [0.0006780342899322876, 242.6257909960619], [0.0007547227293975503, 263.38254694876275],
                           [0.0007836883740372405, 243.20004962828074], [0.0008458391358321325, 266.48948611650883],
                           [0.000878961599830979, 298.03956880811864], [0.0010166882699504556, 327.250245471749],
                           [0.0007192217478458314, 270.52900167960894], [0.0010809043402967699, 357.71151095237957],
                           [0.000989136058515287, 306.60956841379544], [0.00074725493165944, 296.54100344001716],
                           [0.0010256464987221528, 342.82851859847773], [0.0008704996315505613, 324.8324340366071],
                           [0.0006847857318888065, 252.43520736113035], [0.0008578134227185934, 295.80490303049646],
                           [0.0006463424249740711, 238.14625863621177], [0.0005519726817645388, 222.93674936654344],
                           [0.0006673570781485024, 229.46211349767415], [0.0006784343560769888, 230.01870701084533],
                           [0.0009046693139723317, 335.515773163035], [0.0009874258077957646, 304.03890989849225],
                           [0.0007988749045038983, 230.96321675449866], [0.0006084038496430827, 207.48490277776543],
                           [0.0009645898845428726, 267.05921025261506], [0.001048064870639928, 274.2861410044305],
                           [0.0007438449316165748, 214.53633571330215], [0.0006412495633411363, 255.1821840928463],
                           [0.001037463029736533, 307.5860850987942], [0.0007328719662239313, 288.3880229029966],
                           [0.0005678586871457375, 223.67399865273924], [0.0005793358937671122, 256.6190588265858],
                           [0.0009586943880612746, 293.3202000323788], [0.0006021619682478203, 228.87057805656912],
                           [0.00094023293580961, 290.43804650443025], [0.0008995663817176216, 265.9480006918008],
                           [0.000793399048978648, 291.60069502105813], [0.0009687874410784114, 306.4229393646359],
                           [0.0008934093451228705, 217.0376157607301], [0.0007261557279136065, 264.4778276938309],
                           [0.0009518649488475817, 267.34421332747866], [0.0010872194639945435, 300.43422554042405],
                           [0.0008362310673712646, 258.85867955846487], [0.0009562315154313256, 312.02725658386163],
                           [0.0005060478746514122, 157.71464960818722], [0.0005647788282404985, 203.52308286196464],
                           [0.000878912727133002, 288.1029838432084], [0.0010300229100983283, 266.70277987590663],
                           [0.000741818849048858, 228.27110013067718], [0.0007003885538427207, 263.5032866507857],
                           [0.0008023319838724166, 315.7118296432617], [0.0010313043536867663, 332.06295968655763],
                           [0.0009013773647767592, 320.5890556106819], [0.0011284010978299397, 347.6813458644446],
                           [0.0010319358710039668, 325.234328443444], [0.0009393504295421397, 354.0420627173683],
                           [0.0004925796468130137, 167.04929829546717], [0.0006206092134531247, 239.399942808445],
                           [0.0008874736583202864, 247.8036170377842], [0.0009520974767554804, 294.9533909243071],
                           [0.0007932184978893373, 227.50081288403769], [0.0008414867974377428, 304.8156612908313],
                           [0.0008901633000395446, 288.61667422419504], [0.0009749939596511559, 298.1638471418568],
                           [0.0008685761215337177, 257.5842502500346], [0.0007060429394772953, 201.5605683599354],
                           [0.0007644866606941344, 264.32552891761895], [0.0007098374224438043, 258.8158698268049],
                           [0.0006155090393755711, 236.14752690281273], [0.0010636878851403267, 310.80261561529073],
                           [0.0011249375184000506, 277.8889905020856], [0.0009605601701965097, 284.8508882932449],
                           [0.0010583451572805452, 345.75636754461175], [0.0008497744282530069, 267.4689652977402],
                           [0.0010275446369614992, 318.2286814179403], [0.000906333867687072, 304.5702035143318],
                           [0.0008126884638187682, 231.57921064171185], [0.0010827599615407025, 275.69790742974646],
                           [0.0007242087899543007, 250.6320004556795], [0.0008542272171153384, 278.63305781381223],
                           [0.0010593903056016632, 319.7682315293097], [0.0005850541016459405, 206.60286818467733],
                           [0.0007988541586648563, 200.90350872332473], [0.000710840373553405, 253.86696845032404],
                           [0.0007052106545350224, 201.9691229414466], [0.0011206162094633187, 342.36121622013104],
                           [0.0006888318329726212, 263.4347903782061], [0.0006635658679948162, 229.09486348765782],
                           [0.0003042379876400934, 180.61048781811533], [0.0009819729770295692, 295.17148327687147],
                           [0.0008214818247422745, 209.79699520835734], [0.0006984156150147195, 207.5204209410222],
                           [0.0007555931039861355, 284.0297516540411], [0.0009092784278251683, 281.453004523593],
                           [0.0007481231958451293, 292.5586547829016], [0.0007641989102353299, 227.0046293063418],
                           [0.0010435407092141856, 300.3131735956576], [0.000899438301414564, 299.3026256349015],
                           [0.0007696523368013163, 278.9109532728307], [0.0007173351154954592, 272.4194746637103],
                           [0.000961900121752942, 281.58486622014203], [0.000979673809204642, 343.22165436959204],
                           [0.0008249843447285769, 269.8975174030096], [0.0009087374251864694, 304.1683458714036],
                           [0.0007869447546049868, 277.57823689623194], [0.0006653934966771779, 259.41699656052623],
                           [0.0008174239199680429, 316.12443495423184], [0.0011057254667898638, 333.89913286605423],
                           [0.0009992294320844944, 285.2365599525946], [0.0007618951826070785, 268.14288731362507],
                           [0.0009974578841720288, 286.00494097428367], [0.001035195499216299, 309.6966325206121],
                           [0.0010935616201505787, 334.5865891314603], [0.001029109195694356, 287.84796217046335],
                           [0.0009887896507511384, 315.829518563409], [0.0007141665772236473, 263.9352113318799],
                           [0.0011787172351718522, 348.0368371054385], [0.0010865030702174876, 308.2429314508248],
                           [0.0008222526113373829, 304.14422392865737], [0.0006387714645339, 192.21590492956088],
                           [0.001002680698527825, 282.97608920375205], [0.0009831851260147376, 288.1474238506586],
                           [0.0009516188246948932, 306.7658994202814], [0.0006240197248880396, 225.26347418373476],
                           [0.000752377094505675, 202.7755329405972], [0.0009157705536277963, 269.6021721015817],
                           [0.0010284231321634528, 330.74159654547316], [0.0005707891495327867, 260.724546823067],
                           [0.0008309356437861677, 276.3859332534899], [0.0009302482390189287, 301.7273034399803],
                           [0.0010045392875739079, 307.36477920770665], [0.000885914636275017, 289.87181989275217],
                           [0.0007039726687926836, 228.55452721421216], [0.0007289285697861987, 270.86316014930344],
                           [0.0008887031129556722, 295.1364705856407], [0.0006377476245100208, 197.18575787957812],
                           [0.0006928457870792764, 234.89677288801786], [0.0008084801774504925, 270.39643543242687],
                           [0.000866653551117747, 296.2533046202169], [0.0009038844250552163, 275.08057727384937],
                           [0.000788902257952721, 264.50230171982867], [0.0010840828877788046, 354.48123993518044],
                           [0.0009051302415781995, 191.03085393286446], [0.0009068296615265488, 283.6722449046777]]
        elif set == "300F1F2":
            self.x_data = [[0.020468215111057578, 0.00531162657003023, 0.02919409621255804, 0.025166866705370317,
                            0.019558121170663474, 0.017400043397254598, 0.04100230460096901, 0.030919271848794932,
                            0.019082555065623435, 0.04402269010167836],
                           [0.028125588999550366, 0.03199312574674581, 0.0344163542325855, 0.04265762050173422,
                            0.03119859264201305, 0.03836049237505442, 0.030154050600347104, 0.033469003857789316,
                            0.014000289658914783, 0.04301300928651473],
                           [0.01143829889757867, 0.045652131577791785, 0.01877401485499111, 0.032749442714683566,
                            0.02367085100920463, 0.02467417690193331, 0.01697061041801776, 0.04352377306925334,
                            0.045287747199419984, 0.04764345245147588],
                           [0.03775566132811516, 0.029756519600353196, 0.027346535293922617, 0.029122087989503904,
                            0.01817364718823545, 0.00789846444210041, 0.03738713636474863, 0.03363022175120623,
                            0.01404851269985622, 0.024098116658219447],
                           [0.03618549235321378, 0.017949959358291162, 0.041712888968931286, 0.031649993096254804,
                            0.012734271488224917, 0.04495514116032311, 0.02389050399946471, 0.012809865589973261,
                            0.022924010641848007, 0.025514739433436715],
                           [0.023451842046260926, 0.030350152578226014, 0.005671981163627332, 0.03826533440061867,
                            0.009318379125677376, 0.03570713549749931, 0.04894625098140069, 0.016708717694298134,
                            0.020662625434109492, 0.029179219197673502],
                           [0.028266185442337544, 0.022645476022146213, 0.03254538156501529, 0.03986953755403499,
                            0.0202809032071206, 0.027887389026024056, 0.0076606236181140965, 0.038536260235147314,
                            0.026551011292771685, 0.029695566460541948],
                           [0.02633167192526085, 0.009634163351991881, 0.046971474881706665, 0.045520095248539075,
                            0.008051481062689662, 0.04534916078139932, 0.025799781422347894, 0.017472124045074706,
                            0.04008748834018926, 0.006532065081864772],
                           [0.007837299030953397, 0.04825638245905926, 0.010033996523604483, 0.04098316107640912,
                            0.03395525378166698, 0.013533268137305241, 0.04332350607581697, 0.027833940711597453,
                            0.02374419330419012, 0.04680046379921248],
                           [0.04426667001256213, 0.04881508887500272, 0.022000512314518522, 0.0076056884661526054,
                            0.036522799937947735, 0.010661276677030481, 0.037077206263059215, 0.02974464830446424,
                            0.006261935172829167, 0.03992907209486422],
                           [0.02239996620704956, 0.02699541061809628, 0.02285005463910908, 0.01969193653473535,
                            0.03631266472255215, 0.008996103541507957, 0.02752427676336179, 0.01936366410080639,
                            0.020839057050576558, 0.032670930433636086],
                           [0.029609168086987953, 0.035905378154111926, 0.0239678916171879, 0.04323683614006718,
                            0.02990528534789002, 0.04121834911568102, 0.037552345621158056, 0.018728312186242945,
                            0.010054525628789831, 0.03679537297836054],
                           [0.020433341957710634, 0.030946628773809718, 0.01261333069127164, 0.017461306640749248,
                            0.015632164616484822, 0.04332944875021322, 0.025581330685228176, 0.014643612370202192,
                            0.007255348102295157, 0.022643531246602185],
                           [0.03706772595857771, 0.010631443307455425, 0.03628237057833627, 0.028134087563677915,
                            0.046931733539150006, 0.01017976386325099, 0.03311910345575856, 0.03308490809948634,
                            0.03094258851644097, 0.014283199703401205],
                           [0.04189488160281712, 0.03270405658603547, 0.021640077660012084, 0.02165510376968015,
                            0.028377867847120583, 0.03425422036927948, 0.0379342927461829, 0.007446225080859456,
                            0.011981955831317528, 0.01397936701631745],
                           [0.017669353536394324, 0.01354574327778206, 0.03499762615804142, 0.028985108127107723,
                            0.007402578251779169, 0.023343883257618436, 0.012610739978297501, 0.04867687044316123,
                            0.013298852795264236, 0.025915243721002717],
                           [0.049712701738913755, 0.04599513190568019, 0.04044232647605342, 0.00666683697818434,
                            0.03830785163873554, 0.031721996539782006, 0.04022531238220923, 0.04143870028283797,
                            0.007848444438238785, 0.021670480712978384],
                           [0.02281504104119808, 0.024772262957385213, 0.030375713179509982, 0.0330331984990034,
                            0.018049892586554886, 0.047734617359376855, 0.04237147719697456, 0.022816539358605507,
                            0.04670066066661311, 0.02142553925629042],
                           [0.038955252925653924, 0.02544244550154879, 0.03212070781535382, 0.0355447776855609,
                            0.024205584586960108, 0.022718290403974217, 0.021135888846076496, 0.007239641413574678,
                            0.008498612728363155, 0.008270912562388013],
                           [0.009774248089384981, 0.01878901983988228, 0.028841909614256717, 0.03773125996214782,
                            0.04926994375744995, 0.03362351123049062, 0.022494776891450803, 0.04104820656577505,
                            0.006850555004147783, 0.03575070912715313],
                           [0.024331398251385212, 0.015452161887701606, 0.04540759412629708, 0.044316732380934795,
                            0.006021143708935514, 0.032391658860590915, 0.021844614166972642, 0.03144991162866454,
                            0.029416889938554083, 0.017666893032461082],
                           [0.01951848611123663, 0.035422559558626776, 0.03600341332583944, 0.029674847580385844,
                            0.01994600397587084, 0.02445105992809743, 0.014015715296152524, 0.010908112731282608,
                            0.009450102295539506, 0.02224729542055914],
                           [0.03933377750699596, 0.020020734994580007, 0.045741886212481445, 0.006425212582254235,
                            0.026472294624597822, 0.028107891999564934, 0.044310462757705185, 0.01967553268048064,
                            0.030387187476858434, 0.022714213648750096],
                           [0.011287425000243666, 0.032579145870930956, 0.02762927425835983, 0.030701088878130486,
                            0.041717580607092276, 0.044194713091527, 0.04232936872321126, 0.010021252668322583,
                            0.03718561279692615, 0.03842177409484292],
                           [0.031253815745347045, 0.006674459192959897, 0.04820105688482869, 0.007297621147293675,
                            0.01982879121221966, 0.01749330291284143, 0.04948577477310295, 0.04469004049373868,
                            0.03908343979885693, 0.016133429836872513],
                           [0.0346055093163807, 0.02690070363474995, 0.031423172333158035, 0.04132778866916758,
                            0.04502826726785021, 0.02795498293227705, 0.013080704850754212, 0.0054531278621424715,
                            0.042244610491399354, 0.03937828793028257],
                           [0.02405326869425694, 0.012166144639449004, 0.034712550397384793, 0.03865057912763065,
                            0.042095355949293575, 0.04778687394155341, 0.048000378631683376, 0.015360956132233233,
                            0.04455115412728966, 0.028959246548154554],
                           [0.03830709441573963, 0.01498946816656243, 0.02258246110496943, 0.03703976364229035,
                            0.03549336374401875, 0.038002993592793424, 0.040076706380812975, 0.049074396256565025,
                            0.04460291289825871, 0.023680654092454505],
                           [0.005529043884229782, 0.024162457689339615, 0.047238788605452484, 0.02493920800591478,
                            0.02356282337047125, 0.030574055547955486, 0.02357576266920092, 0.04232344698275078,
                            0.03244537318940256, 0.03504664185786232],
                           [0.0230020425437549, 0.02761748184615309, 0.03612848609913649, 0.04514702626008526,
                            0.019175422291899856, 0.04359794077582981, 0.009704581942281012, 0.006328792460996473,
                            0.02277380988326037, 0.010657014941008899],
                           [0.047303057122314265, 0.02622352821922383, 0.017876146018795114, 0.032409319478912964,
                            0.032132386124251955, 0.011819301283973089, 0.017429573575086007, 0.007576266964073811,
                            0.006067862791784331, 0.03144889304716981],
                           [0.030991939085511296, 0.044956212276287606, 0.04887508271542588, 0.02286514874554761,
                            0.025075826936868686, 0.027646969885919248, 0.030537940006752955, 0.0184999483749054,
                            0.03576980178920302, 0.01791219431619295],
                           [0.02655815598214545, 0.04703429069037429, 0.048954338849147516, 0.009596266087713325,
                            0.010473502453920688, 0.044564046226836955, 0.02897099687895526, 0.038056203688004164,
                            0.02050009791261783, 0.03106933227127881],
                           [0.023507212648770284, 0.04768786238839776, 0.03436076962753941, 0.008811963566953466,
                            0.033520501080810686, 0.031201296158787157, 0.006172544863436704, 0.04047886163568186,
                            0.013705738520541307, 0.0323357457273056],
                           [0.03577278251332123, 0.02713800459279907, 0.03223087698149303, 0.009103411767314095,
                            0.02099493130438782, 0.016539416098071014, 0.031206026307073344, 0.047900013769289665,
                            0.040549237874155035, 0.014068351911151276],
                           [0.02747257532704284, 0.02966094622123974, 0.008372144032386704, 0.009879359999117855,
                            0.034240726848427376, 0.04156789241357078, 0.01562782922981704, 0.035991730002683006,
                            0.043281188094577955, 0.025599666495822668],
                           [0.033837617595234445, 0.038169957555052106, 0.035628799137538955, 0.04459651606561243,
                            0.049560840901069235, 0.03300352515222016, 0.04115607601103181, 0.02738470674991918,
                            0.046164672802637974, 0.04926617262091229],
                           [0.007964921365817668, 0.04332438972858842, 0.04109615247365466, 0.02851873503030309,
                            0.016502643644919986, 0.02526763784635818, 0.008497676233723004, 0.04308868220696874,
                            0.024495344260412104, 0.007955498586451946],
                           [0.029865931306891764, 0.025086518676027098, 0.040327360986782385, 0.006260930740529026,
                            0.03428539075984321, 0.025797784189220376, 0.03176312457991033, 0.04958580271869704,
                            0.03631148713762621, 0.04340753680263445],
                           [0.04559288054445026, 0.037193465954130374, 0.01338423436884054, 0.03757281444297016,
                            0.032377092535437246, 0.012086834445931106, 0.008093450772324852, 0.019473072481570094,
                            0.010179254172272629, 0.04417153854547903],
                           [0.03631344442939623, 0.04839592452783699, 0.009098531315658596, 0.035371654915966035,
                            0.047809377057901524, 0.04481711606655201, 0.04565102911292439, 0.04620373095647936,
                            0.0234739755170358, 0.03284788111003372],
                           [0.04717401971942925, 0.026706713184404852, 0.015352264764060815, 0.037353031359638064,
                            0.02395592416684318, 0.013248405040819894, 0.0073425673340840915, 0.013439804706234952,
                            0.030266367104923305, 0.04054422611303632],
                           [0.021729621873888327, 0.0419431156763579, 0.014595718868579483, 0.04870684857194975,
                            0.03167893702854992, 0.011471508608736289, 0.04754733729901561, 0.011232732339199539,
                            0.03849708465155634, 0.020185264017787908],
                           [0.02127805691954564, 0.007555030828836016, 0.025311180352706835, 0.03787321029796427,
                            0.035200201070408915, 0.005874877137417728, 0.005494002676264204, 0.04674450891354026,
                            0.031991587293667294, 0.031110383180513165],
                           [0.012266591684315913, 0.016221736369661746, 0.025615935270627836, 0.019443884372471135,
                            0.03962921087449164, 0.01903052200504492, 0.006931297154138702, 0.005087977345062685,
                            0.02896358186312201, 0.015212140954844692],
                           [0.0287926304260759, 0.006866124378174353, 0.010906276921172742, 0.00819133248215236,
                            0.022078711936844863, 0.03389561265275176, 0.009233768452277094, 0.009760300065565171,
                            0.04309842489153825, 0.049621910947519665],
                           [0.044397551606907344, 0.030605264034533626, 0.01595985042632113, 0.021967834327305685,
                            0.033802245286687994, 0.03960797902020726, 0.015151593712278751, 0.01973284836681252,
                            0.035130586095440966, 0.02651059619349842],
                           [0.013962591895500504, 0.03399905082984598, 0.007597151052413025, 0.02721317309317626,
                            0.04708969087496214, 0.04907690907158761, 0.026647209610935977, 0.04501774584241883,
                            0.04073802478019634, 0.04879748686575383],
                           [0.01673671694159178, 0.04584014320367488, 0.04844559015414617, 0.008333157955776547,
                            0.02896975030988804, 0.04630676202539116, 0.046693042542135695, 0.035433186457589985,
                            0.021674046019091998, 0.015079892982067064],
                           [0.015077867696490825, 0.025893005311642653, 0.024612309188421232, 0.02657990007167707,
                            0.02915517764966521, 0.016644804552094484, 0.025384361552210216, 0.03465025386302333,
                            0.03614798237779776, 0.005443619777993505],
                           [0.005263199171274746, 0.025609521567376, 0.016897252400365048, 0.019929437918177095,
                            0.02530300522358844, 0.030990147738923356, 0.03324157913040879, 0.007721702904381746,
                            0.03205820053631143, 0.049757692107935306],
                           [0.042135436426474826, 0.008714256412595437, 0.02190822094649359, 0.012303327797359868,
                            0.048655417189052176, 0.041837125931764495, 0.005177894534453171, 0.03050155032097986,
                            0.023262633712503974, 0.02644823290768149],
                           [0.033281266060329766, 0.026601781494957937, 0.01935709740584881, 0.024126183840804245,
                            0.007593796390814205, 0.02459434151038484, 0.018247128229053312, 0.022017753511615112,
                            0.0475704954995188, 0.040274532477538746],
                           [0.01383873763345686, 0.04094276695346334, 0.010217036810318953, 0.03252725128926038,
                            0.04749430167253099, 0.03375520629599875, 0.01169722525322618, 0.015585865002955224,
                            0.04694829390554735, 0.017193669401321426],
                           [0.0232345356833854, 0.008106922757478143, 0.027040263584260166, 0.025508924818268268,
                            0.021654164239128023, 0.028568663228565192, 0.01536306445847228, 0.013094175878853621,
                            0.048836606539454396, 0.04736948501012849],
                           [0.035661936065014485, 0.008159460484253495, 0.02439326669942722, 0.04615203924755282,
                            0.01085806108551592, 0.027297970364297337, 0.0298215947113874, 0.010286757786037368,
                            0.012845819749972755, 0.04799852893188153],
                           [0.026784688365521284, 0.04557200351800219, 0.04289860770797078, 0.014941898701928896,
                            0.03255007946548787, 0.011386501145841622, 0.03867284799651437, 0.03729567712453214,
                            0.03884858093429965, 0.03909257454731916],
                           [0.026617904310961987, 0.030462780469306523, 0.01816128378526895, 0.02114171221124106,
                            0.006415724133597708, 0.006843546428230139, 0.048557954629391525, 0.04889568359354434,
                            0.017734749057494295, 0.01960546057553229],
                           [0.01316441648557527, 0.04176212918466026, 0.0478779881981848, 0.007051857308176615,
                            0.040324051039527034, 0.023459007435709348, 0.014774867400576194, 0.02309944247816644,
                            0.01171422071874131, 0.044893349397110524],
                           [0.04942167220454585, 0.03308356597969529, 0.011733185955657581, 0.023512287972509968,
                            0.03756036128633249, 0.02184309897983818, 0.036075812334592144, 0.049846867173045366,
                            0.04679936048356987, 0.04494869199010815],
                           [0.009516906076369742, 0.03000984386432785, 0.011528646363216651, 0.012753605577187774,
                            0.03328317134215556, 0.01771448942604094, 0.038991969121377326, 0.014467867270004885,
                            0.017194397842194432, 0.04058663784597819],
                           [0.021656288120447918, 0.014308547544529787, 0.04802704254091187, 0.031089394805039562,
                            0.016714735964681555, 0.045857665995960965, 0.03368124508249623, 0.04096121045463551,
                            0.04016131729445469, 0.038841027410344564],
                           [0.04713507813118542, 0.008989396034815994, 0.04151523935185582, 0.04108538858633166,
                            0.0204550410621705, 0.032210696095698, 0.04163201977287118, 0.010774244467175328,
                            0.008141680801575629, 0.006967591390265752],
                           [0.022005146830135075, 0.021013450175358106, 0.009244562596634181, 0.02311067349090504,
                            0.005493517497397968, 0.04109397248340066, 0.03082708677449966, 0.03585469449703048,
                            0.04552718507245443, 0.008151390244206268],
                           [0.010179800948671137, 0.045367569888856644, 0.008996257587645072, 0.012065318040817239,
                            0.026443410167907466, 0.041436810280716524, 0.017305154123041403, 0.0062026651106173725,
                            0.013195650432251033, 0.01906955414253648],
                           [0.005714759814272063, 0.007302191832861724, 0.011036298319578183, 0.04386911966670615,
                            0.025484662578280977, 0.023304990236161047, 0.03907582647112806, 0.042728977481806915,
                            0.037032735958008406, 0.04647922143495771],
                           [0.017105573953324515, 0.017537884468498754, 0.03763375189095884, 0.042080927467708175,
                            0.030118045998108068, 0.042701469282740205, 0.02960612794075245, 0.0443214617850703,
                            0.011372610636368685, 0.03787744191582266],
                           [0.02888844436128895, 0.04777455233119194, 0.04673380018865724, 0.020819330817348623,
                            0.028668120086062277, 0.012275243839487453, 0.03202433920012793, 0.02544820507202958,
                            0.03255359104558765, 0.04360604978626036],
                           [0.023812350524335092, 0.029514465171867982, 0.027750573768005972, 0.048184767848975116,
                            0.025588081057306762, 0.014161542654331217, 0.026196437992562462, 0.046412949363226756,
                            0.044323740388647885, 0.005742278541171496],
                           [0.04369515595183064, 0.048655340491508506, 0.0168371204390194, 0.027802709242070123,
                            0.037250494834066485, 0.03646340917662852, 0.011105261594239682, 0.024729400001635413,
                            0.024344108722542748, 0.009522060000778454],
                           [0.01916860654693326, 0.04405663869559246, 0.02803838211873151, 0.014221926180329313,
                            0.021249322565337993, 0.011703348008002361, 0.03678572385234575, 0.01821191599199348,
                            0.04199865319845633, 0.02035113319502705],
                           [0.04335773493672895, 0.005650359868202991, 0.03244128447201072, 0.03417462440388371,
                            0.009516157364300826, 0.012733974382354182, 0.014600859845341235, 0.0073884613530126875,
                            0.027615504237162518, 0.03467078971196087],
                           [0.030807065186877106, 0.049370078102904794, 0.006960232673301428, 0.01328910050375903,
                            0.024397693672225572, 0.033240544833899204, 0.008673694359404807, 0.02138361743536638,
                            0.015394609474918736, 0.03054520112581619],
                           [0.006897704459903395, 0.019446053708002044, 0.01119787094337718, 0.01546688214833395,
                            0.03739773512789246, 0.046736701383806784, 0.024351867228010398, 0.014436312267959328,
                            0.005269841777397532, 0.03960451196129988],
                           [0.03494191578304167, 0.021921227671768854, 0.030907048698918863, 0.030136688610121858,
                            0.01359935767449692, 0.03944829474313536, 0.020931793127583868, 0.03834213120987481,
                            0.036465186026068845, 0.0338926448495482],
                           [0.009477713890202223, 0.04142368594785426, 0.031720228928639786, 0.04082854691632338,
                            0.04880958308821102, 0.045498398060582874, 0.035742498604931595, 0.036383721184480196,
                            0.033166433789443846, 0.030033925976424777],
                           [0.04594265725875115, 0.008531788145725068, 0.01318671630369023, 0.02471660451323506,
                            0.006584754445674109, 0.020726782383512218, 0.026871792331820712, 0.02376978139401032,
                            0.018625183576659318, 0.035885133773955157],
                           [0.04086980770085548, 0.014600852993763683, 0.04606639640127416, 0.02435933451441644,
                            0.048449979521400156, 0.006315102500475459, 0.01582425997010686, 0.010445139523911372,
                            0.04344536692704894, 0.04880434936434475],
                           [0.03792610520739166, 0.006447774023900914, 0.04708701292481273, 0.04651317606684978,
                            0.005176451568067726, 0.048404216877925905, 0.02881413856555472, 0.03887439606777858,
                            0.045670127594189744, 0.01777026042299997],
                           [0.03010128532415588, 0.03695079998697433, 0.006923191861479003, 0.00933658243085464,
                            0.012143778253022312, 0.03407014259666621, 0.0071942551139442205, 0.030310026947302473,
                            0.030033729668542165, 0.046870361209733404],
                           [0.03399341484539586, 0.01967610531025797, 0.027108460274381196, 0.026969585355414303,
                            0.049413617936412364, 0.019802686375952724, 0.029464224454783382, 0.01490200781837937,
                            0.03527873765073524, 0.02453018667344342],
                           [0.017332695201600634, 0.033000318483147315, 0.0075566511407016606, 0.03626794101405984,
                            0.01855573372910028, 0.044132096336659495, 0.036037016318666036, 0.028997364250015593,
                            0.009852816807881352, 0.0319526691820909],
                           [0.045684576934312736, 0.046530640541673864, 0.04648568986579431, 0.026824210768707172,
                            0.008771777140479373, 0.0408870877315627, 0.012412869682574428, 0.02512202547058038,
                            0.008620292727559613, 0.017376616370916958],
                           [0.007429768286233573, 0.00800914990886651, 0.022530132269010936, 0.038556077956466414,
                            0.03668771521492464, 0.04790638813245053, 0.013806632675708242, 0.0441750578986349,
                            0.010452077674615892, 0.028187104078224848],
                           [0.013516456853894663, 0.030665367363659397, 0.016271792891235207, 0.017966873844766824,
                            0.029506827246023697, 0.014352363426081653, 0.035594895844704544, 0.040796968717275915,
                            0.018072878724980975, 0.018431598187365394],
                           [0.027311556664066424, 0.018336732296385317, 0.049754034724368304, 0.0390929605121421,
                            0.034615156334674865, 0.028374945509116664, 0.040840759988071136, 0.03371506044118825,
                            0.015572719944934886, 0.04833105774234877],
                           [0.01864873425351342, 0.02200290496336166, 0.028979181614191378, 0.04950834299210236,
                            0.012309490360383382, 0.04253661820876787, 0.02036616769752287, 0.021092356381030245,
                            0.041569817959312236, 0.0382070385664676],
                           [0.046608229468330875, 0.010321559621661883, 0.03637095696245966, 0.013898133783899543,
                            0.022736624414601194, 0.03273649416624821, 0.03637615719326974, 0.03503578609962975,
                            0.014301169181103916, 0.01372134364311917],
                           [0.02718123981299993, 0.02384534630850822, 0.04428020371866751, 0.026277711012653665,
                            0.03847820269539317, 0.046942202236261964, 0.01907151574637986, 0.01267956860073877,
                            0.041779627668004686, 0.01705742810756815],
                           [0.013607689153732908, 0.03358131082762297, 0.026769467004757786, 0.03337576638854893,
                            0.015057580799853199, 0.017302709802913235, 0.015740941431564782, 0.0256474458453255,
                            0.028650127412973066, 0.043715298352204945],
                           [0.03543818589962036, 0.03564216780548692, 0.00814819282748589, 0.04045505980384423,
                            0.0480519955009001, 0.034995829898970335, 0.039245185379177636, 0.025931583834889697,
                            0.005707137364214103, 0.013643587858422741],
                           [0.03752560945468375, 0.049745756171861506, 0.04531176637375195, 0.030204975795272745,
                            0.02251348524025454, 0.01811868342736371, 0.03282234849967292, 0.03677277431465579,
                            0.011595708722316201, 0.013135066088457478],
                           [0.03444667430500506, 0.03332892868497649, 0.03399110819804433, 0.01552240218115078,
                            0.04576000348289613, 0.024175921519075196, 0.010024221298983836, 0.03518367369755342,
                            0.03828632325008172, 0.009680762046519485],
                           [0.048775161966495124, 0.016003153888961317, 0.02795309497671808, 0.0320033750105203,
                            0.02672452515141788, 0.020463051601436542, 0.005076029697871292, 0.0406476697628356,
                            0.027024688459808488, 0.018234872981633263],
                           [0.022154448261518668, 0.022444950793971664, 0.017062380822815738, 0.026312305699070365,
                            0.025788450527757512, 0.020114159982215957, 0.03261737237314017, 0.02766371965281533,
                            0.0425245437274793, 0.03410691816937254],
                           [0.0066771901222341095, 0.023392313033760753, 0.035276027065291475, 0.03522579357658044,
                            0.04398465809443902, 0.026786364437103888, 0.04545229019301013, 0.04200547454545988,
                            0.031297352379023535, 0.032488352925544854],
                           [0.01007312695570153, 0.03965840301025121, 0.04938233837891026, 0.016626458083930422,
                            0.015445641694595443, 0.021558047604268618, 0.03197147943390302, 0.038229918179828355,
                            0.040666145586057675, 0.03229739237732162],
                           [0.020114242231873088, 0.041239661171664645, 0.047394472262986216, 0.025593518044424696,
                            0.041181582481236156, 0.01503425785700781, 0.043164625407959276, 0.028699763013147275,
                            0.010939144353627854, 0.008574770325099868],
                           [0.008175881798820556, 0.02580612826986117, 0.0158195172425971, 0.01740949809481938,
                            0.008373829215295423, 0.009462186370923803, 0.03944724339087666, 0.03283226894706115,
                            0.007596595523239562, 0.01995077121555392],
                           [0.04890412018483239, 0.011750004395601053, 0.031148590228305188, 0.04259679170235689,
                            0.020136624255693132, 0.022642177428421024, 0.04464419730120372, 0.040105323581554705,
                            0.02682339645111305, 0.006199260389932059],
                           [0.04448320653344052, 0.04215831253658566, 0.0400207133686663, 0.039534966921348794,
                            0.03272045489620263, 0.01258093172529028, 0.02728502970542153, 0.03245172250090996,
                            0.025171050934720218, 0.04550536327351541],
                           [0.017806796228423526, 0.0366635714403961, 0.016616620140900106, 0.02456815716186765,
                            0.0321673591041323, 0.02207637706090195, 0.02087030639847299, 0.037906609294849486,
                            0.011825372975641206, 0.030872395535130166],
                           [0.047656200720304145, 0.005020762219156621, 0.006475367205567448, 0.03995464950227858,
                            0.048333859365728725, 0.04391177838810034, 0.017973570383806706, 0.021328456425989205,
                            0.033342106318329956, 0.038681586973424595],
                           [0.04924683082768743, 0.017753036895329716, 0.011892138087729747, 0.0240110286679984,
                            0.041112365031582984, 0.036293912726137155, 0.02715876548150284, 0.009612662790895083,
                            0.03756474207412741, 0.041516341199669994],
                           [0.03549014926070997, 0.009972235584109146, 0.013875914073010944, 0.022203309831165127,
                            0.03022423848446073, 0.0051320521618107426, 0.04300096190372958, 0.0069063935098118544,
                            0.020293830230931117, 0.03761014992585595],
                           [0.0258544076650947, 0.04431366495940259, 0.046601252434833276, 0.04233900062104657,
                            0.030394934780566405, 0.03165933009570401, 0.008843007630312387, 0.014148536034438528,
                            0.037262497460464554, 0.022992209766351017],
                           [0.03310918403035846, 0.034821716701790094, 0.033595137030902535, 0.04850869740791591,
                            0.038180467389607224, 0.034737068577442305, 0.02838102354776612, 0.04941553233558812,
                            0.047029275325357785, 0.011852654088447562],
                           [0.006631784815038903, 0.03116319124092071, 0.03740062718454337, 0.02662194534442453,
                            0.011213899535824499, 0.015952999178433926, 0.021785935810547066, 0.03192975523658625,
                            0.007556181544993696, 0.015713472830690432],
                           [0.04342240022005782, 0.03579706169193788, 0.008467792267901295, 0.028640529278691314,
                            0.024816412875077517, 0.01685370388838601, 0.011979849911745482, 0.04923257549893403,
                            0.008313618149698722, 0.041819778457855145],
                           [0.02753621642483577, 0.0221990792594598, 0.025412588527316964, 0.008708441508779667,
                            0.009780200793160231, 0.04057595815370845, 0.021240319968515154, 0.0301583533680755,
                            0.027840441655399575, 0.006836592944974015],
                           [0.046202770953360124, 0.046814900145299015, 0.04120799773751201, 0.010846508994214897,
                            0.008938145735296187, 0.03724344628848514, 0.03296701892930178, 0.04759317086598412,
                            0.04897759035387499, 0.011660926343325124],
                           [0.02856737701679325, 0.028946365689512685, 0.020281422642390287, 0.00779876113383582,
                            0.04773215635222621, 0.04551558784548166, 0.028497683084690674, 0.023559853074791314,
                            0.03671056811759452, 0.0484537181164556],
                           [0.013096627599193816, 0.018861054602901, 0.04854007217985305, 0.010312971584061787,
                            0.03389000735940046, 0.018583642353199613, 0.0274163519855162, 0.013990509430500328,
                            0.027969973415685148, 0.009913902191370613],
                           [0.016908687757623924, 0.011397735975844193, 0.012451150970182337, 0.017707980719712076,
                            0.013715950096271731, 0.028718834952820772, 0.011817010969645339, 0.04927396994685758,
                            0.015253674615543122, 0.042492097645924515],
                           [0.047941174884811075, 0.027494875296296486, 0.021120350872969262, 0.011199120456461637,
                            0.03560884288564291, 0.039087539451954346, 0.025197237275466362, 0.013112023380242248,
                            0.04731849744597191, 0.010968817136752726],
                           [0.0394979095051796, 0.009063376084811274, 0.019676795120695038, 0.03902635878639927,
                            0.017603841988007993, 0.0419567077516759, 0.019651375212602823, 0.03609976004759763,
                            0.04940196021026875, 0.02839917201770396],
                           [0.017506547613416704, 0.015338683530965094, 0.011457269448111873, 0.010951805372100807,
                            0.016955814970538455, 0.00599967937636029, 0.005879326250572707, 0.011616028601539072,
                            0.016416191652161884, 0.045410180096369],
                           [0.024901922812318643, 0.04448471964086579, 0.026577839967701494, 0.034888310865844284,
                            0.021869563100227234, 0.04208414084127988, 0.028695216501455546, 0.02055953023388416,
                            0.010273496545326562, 0.012809511587503928],
                           [0.02846328723496476, 0.03624102156085623, 0.023010552680240385, 0.014788117319299587,
                            0.013335458387655041, 0.03815652223473525, 0.015327157781710586, 0.02719919767895303,
                            0.03937937521461117, 0.016840993589735065],
                           [0.025327094286261273, 0.01696089000987533, 0.014975189107693628, 0.011766839757312023,
                            0.04204544005899189, 0.0306752295955828, 0.043470884304380925, 0.02580395371141313,
                            0.03775765917771576, 0.02935271178634105],
                           [0.00904312531551223, 0.019306286741213388, 0.03127506966520342, 0.01251741133948713,
                            0.014454194952081573, 0.012399609047135753, 0.03779852104654171, 0.008233519704531407,
                            0.009731651616693216, 0.023457272570374846],
                           [0.04528999121914051, 0.007222763695939063, 0.023307847560680463, 0.043005722629721924,
                            0.03774798978280718, 0.03904432804180717, 0.009027095693448756, 0.04037908762430566,
                            0.04854760811340805, 0.018864997736910315],
                           [0.018340586314610205, 0.02901229870450462, 0.018553228312238206, 0.036611000777959375,
                            0.030603717462169677, 0.023838217275523828, 0.024068216683824504, 0.03956769443392072,
                            0.04638047543894213, 0.032048481193699595],
                           [0.008704850201910785, 0.03922295921594157, 0.0387202764598184, 0.018807298795195673,
                            0.04093121038334847, 0.04567717088663125, 0.010701985804946166, 0.04362356771672224,
                            0.03535978432820522, 0.007419863767473372],
                           [0.0269361114977356, 0.034378774622577137, 0.04994071263438907, 0.035736946727989126,
                            0.027973963071390703, 0.035901248134716165, 0.02919388317540059, 0.03445986756527906,
                            0.02916200484152777, 0.014905327569272903],
                           [0.02065449135946079, 0.032803852239008335, 0.01347068496866815, 0.0187748434398181,
                            0.04583638661722331, 0.028447559499304404, 0.049214549573268644, 0.02802034216765519,
                            0.0250540752628527, 0.03773737028741742],
                           [0.029514809750446548, 0.010451597410810472, 0.015753276120966494, 0.030641401244609113,
                            0.03452721094387893, 0.022267681798582392, 0.0098198106973975, 0.027604884527459975,
                            0.02344945850819386, 0.03439529570288897],
                           [0.01591689659240278, 0.009786707558212144, 0.035142673046790675, 0.015251556899893735,
                            0.0465317688157069, 0.02096699000071566, 0.0065142824988470936, 0.020418313123036734,
                            0.02259903258017972, 0.016628118893365382],
                           [0.025095133666534188, 0.021789984294084686, 0.0285316794413353, 0.01292964612426337,
                            0.0072040212498147935, 0.016122637446237355, 0.0067412102736431456, 0.02097461015446401,
                            0.014353724907868813, 0.037006452839877864],
                           [0.027767724390666788, 0.03345799992736307, 0.026178675434559342, 0.03606922132648724,
                            0.0447736863533775, 0.00772969973139471, 0.008226731539499505, 0.024140403331703718,
                            0.046017045054482404, 0.011344219889935381],
                           [0.03209368659562316, 0.006283622285714248, 0.009867236739037076, 0.017117238688166454,
                            0.04523183971478816, 0.03338685807110292, 0.017541374963816976, 0.030520787310248564,
                            0.016889862843287896, 0.023983370329822118],
                           [0.04805800026829543, 0.03909481937650794, 0.04064782466087757, 0.018350731485329858,
                            0.011330617700553514, 0.018849127546455055, 0.036637048628204105, 0.01572181709273304,
                            0.033721649262292916, 0.04117444879110982],
                           [0.015048493088248988, 0.03210034634980247, 0.03887937056428823, 0.013004450492363186,
                            0.012400148950164417, 0.04930672902236156, 0.04594883626067915, 0.03894917636589247,
                            0.02386390086651266, 0.010319952545788194],
                           [0.04059295211437227, 0.012674513934133016, 0.023862706126228574, 0.020203378825584078,
                            0.006853628974437075, 0.01600423860058289, 0.04206188992326064, 0.008112908535793306,
                            0.048230739039502174, 0.020025670451112827],
                           [0.022427701368672556, 0.01122310155369068, 0.012149050243927855, 0.016979614144450022,
                            0.041886328171552016, 0.02936375775727009, 0.01620261591990298, 0.03568597821557022,
                            0.04781753762072327, 0.013459060645450441],
                           [0.013271953984624513, 0.03154429492838318, 0.03459893149498212, 0.04664438196629985,
                            0.013545340170318904, 0.037618423132460485, 0.008361419791670014, 0.043383385566176845,
                            0.03470021372258409, 0.04531190695984024],
                           [0.007226543110507501, 0.030978982018843616, 0.009708540964255961, 0.038411817580559275,
                            0.007499130658472946, 0.006173031477855685, 0.0388047987679172, 0.037698393635928654,
                            0.04771084682095901, 0.021582222097562154],
                           [0.03963941206053113, 0.016360729266916316, 0.0447988737675333, 0.03602435202751128,
                            0.04444663955333071, 0.0070588786190976145, 0.048158750725136736, 0.02948106830042334,
                            0.03388294217762213, 0.040197123799277514],
                           [0.02563188960059861, 0.023925761986776966, 0.013099537714218266, 0.029338680981256485,
                            0.02743574046757036, 0.044377977492587636, 0.03246464539714653, 0.04701442921776357,
                            0.014938697926066363, 0.0348644434835949],
                           [0.04263818799922992, 0.035228440028120415, 0.006070806010975009, 0.01133542531709686,
                            0.010329319418344536, 0.03600204577236631, 0.046197403060361616, 0.015243053499457392,
                            0.03423469367356968, 0.005534232802161335],
                           [0.02992692925368324, 0.042826555401015584, 0.0451514236495048, 0.02789363244341947,
                            0.024687582547699954, 0.02599971432349231, 0.012299373465136971, 0.0221755778064175,
                            0.01082092218074917, 0.02959709658484974],
                           [0.01176928498855057, 0.013662725217409147, 0.0378059145530362, 0.015151558996817166,
                            0.032871763487030144, 0.027460676101598244, 0.023293782306397906, 0.040036610471793674,
                            0.04086516649377603, 0.021005067031352975],
                           [0.026067867339128324, 0.020256482625830623, 0.007953806372645702, 0.014123759897648077,
                            0.010064685459773022, 0.01128197080163007, 0.029340687097257144, 0.046984961897783145,
                            0.04721578895319997, 0.04108895966850057],
                           [0.008533564407648016, 0.035026606322218, 0.028225173205029257, 0.04240436540922736,
                            0.012985502625762854, 0.005723971212585284, 0.016593575164265212, 0.047972686435132215,
                            0.012988630236426053, 0.04898409624554752],
                           [0.028004753491303897, 0.015127775695697693, 0.006713229302099067, 0.035071310945332224,
                            0.03592310253279449, 0.01826825435492034, 0.016071154548713634, 0.017263256484970846,
                            0.014550786251092782, 0.022079402945630943],
                           [0.014685100958132715, 0.022919621002073617, 0.010655052158409329, 0.008529162614094782,
                            0.00625928140962572, 0.012824071458489808, 0.02013232903318448, 0.01105816298115915,
                            0.03438787635073685, 0.01003020030421833],
                           [0.012147410032072958, 0.008377620993037252, 0.009483747226566887, 0.025903827812870747,
                            0.007854771094082849, 0.026020815275801575, 0.041921755799592314, 0.007875929873310256,
                            0.029565385842564382, 0.017512062966452836],
                           [0.017908821970651644, 0.048553900055242655, 0.012739470559446697, 0.04201518572431041,
                            0.04408746057851359, 0.010848859020050514, 0.04903940488497372, 0.017745402007901874,
                            0.0058576377704131054, 0.020591159941830695],
                           [0.022572358152178063, 0.012842979472384539, 0.03799884384694137, 0.023753319499722137,
                            0.01632596318254433, 0.018384293947842637, 0.018867604362604203, 0.02627414879176196,
                            0.019379441494259197, 0.015466370060128472],
                           [0.010599730575391, 0.023281145176658222, 0.01362522058019985, 0.021484855484764774,
                            0.007899257342983303, 0.005523815689769373, 0.04699921456317495, 0.031115024131623703,
                            0.00955244758997217, 0.0067158106907441025],
                           [0.03651383707587583, 0.042974733605808016, 0.03970521366270053, 0.01647171236346237,
                            0.036167023042848176, 0.030914892905063326, 0.04600449624147559, 0.04484654606602449,
                            0.026009902188117228, 0.02859728361243965],
                           [0.019283283318195597, 0.034119695283881535, 0.013723821525183533, 0.04122925417411367,
                            0.03335847314145755, 0.036077483630586604, 0.04636522898104988, 0.01516423415264299,
                            0.035514914292375924, 0.012451772326406811],
                           [0.014790581147935578, 0.03391129478030335, 0.02294672966893368, 0.01238375383814094,
                            0.04512580230762878, 0.02963201501310552, 0.034495714387140095, 0.02293723142584145,
                            0.04404991375341567, 0.02190560178060846],
                           [0.01715759282299301, 0.005925969090396711, 0.024985616914823483, 0.04408417873715736,
                            0.00570617334273501, 0.01880307340973311, 0.02294562885566799, 0.01499945356966461,
                            0.030056276790494888, 0.04430820061902458],
                           [0.04323872528155716, 0.024515871298733557, 0.025900864840143104, 0.027096525226806986,
                            0.043746911853981496, 0.04375389463748111, 0.04363926423065746, 0.02701048306796673,
                            0.02578426104689419, 0.04467015021343752],
                           [0.01982831841757083, 0.00876083209035221, 0.04966731448067141, 0.01007928414659372,
                            0.02872648337679319, 0.03466719363285802, 0.019359508419676486, 0.018589983020886016,
                            0.03181952576581612, 0.02571528020792701],
                           [0.04054026845454176, 0.031339540693910505, 0.043260754836347295, 0.03327153867283912,
                            0.023795796016925386, 0.04232834625993223, 0.03148120978324205, 0.01649377731321313,
                            0.04137776591441576, 0.048125752772679845],
                           [0.03152204734026505, 0.04039644820816887, 0.042413610708667754, 0.022476015140711025,
                            0.032000288648188234, 0.04940740880033364, 0.04562286665625444, 0.023924985267862617,
                            0.033357679346304896, 0.0304198945296851],
                           [0.030707820257447466, 0.02370736951182383, 0.03821675596474558, 0.020961796301230445,
                            0.03943726263646489, 0.037908292697474054, 0.04684288035370767, 0.045167979150746516,
                            0.01598839349098523, 0.008844982794770586],
                           [0.015790569480954468, 0.005180152672827029, 0.032687966255555236, 0.031150856156483286,
                            0.038657634466040966, 0.04744876295123588, 0.04742098267115146, 0.03495204728356338,
                            0.03657347397592847, 0.0072847727503711525],
                           [0.04072717234249868, 0.025162030694565605, 0.043492713024055246, 0.0484441166820346,
                            0.040459890307762196, 0.03138584389747022, 0.044490806055896365, 0.0476677409776101,
                            0.02717078180078463, 0.023064474776436587],
                           [0.04824802597244019, 0.04532760844560822, 0.029892714742251053, 0.03649771839666327,
                            0.04617079582182054, 0.02521793932491534, 0.010334895640707952, 0.016186463177772715,
                            0.03069960538366243, 0.02802382794368895],
                           [0.015589288973887734, 0.015510725422278882, 0.030525567686492106, 0.045844842576368144,
                            0.011902995439564551, 0.0216974996834837, 0.03431712739972476, 0.009491912130994904,
                            0.03110623022098597, 0.04287680692671934],
                           [0.01808675572075161, 0.04014015926438804, 0.04745736363082325, 0.04327288341655511,
                            0.005012891262933678, 0.043223319720520854, 0.04525076498660708, 0.01805032693936998,
                            0.006805908223539543, 0.03137917857877228],
                           [0.04153757126027405, 0.01151917196903567, 0.04316769529703081, 0.04628371690425619,
                            0.046613212061371805, 0.037338928816315585, 0.02369327230502721, 0.012256857690464761,
                            0.018759590384103117, 0.0269158067472384],
                           [0.0317876798627482, 0.03903129637424332, 0.04231052447670787, 0.03884479389794372,
                            0.027667673515311592, 0.013097133490561061, 0.006237209117666445, 0.0354682367205866,
                            0.044964478245697895, 0.012720706361159542],
                           [0.041639942114312774, 0.02057790100957335, 0.04954826306627202, 0.03680214366691402,
                            0.026208323682586434, 0.043482139428838454, 0.00957548850025761, 0.04120088170599158,
                            0.023008537211896454, 0.04944108447859538],
                           [0.042495270860423234, 0.013120516659622626, 0.04433595516860105, 0.03802540833950778,
                            0.04557801543123526, 0.03670609614032839, 0.01999848412794838, 0.012506013108669264,
                            0.04386860243624089, 0.012195842639194123],
                           [0.03848286201449585, 0.04791857762277613, 0.02858226723588886, 0.044695133383679954,
                            0.043267522500036716, 0.04885662906193613, 0.027663894968007318, 0.03868492007425899,
                            0.037853036917227155, 0.029794360698439288],
                           [0.011146918492928991, 0.013724411156569194, 0.017501495825328412, 0.024349515311043415,
                            0.012891907375681078, 0.00883819557476366, 0.03532793177181993, 0.030657312665755835,
                            0.04034403913699767, 0.03634464400006365],
                           [0.03434109877884778, 0.01838414298966134, 0.008804385985180838, 0.033864518979431685,
                            0.025987805001670992, 0.014108507729082627, 0.026016471468290298, 0.039661494542019546,
                            0.007327061069245204, 0.011174503988253396],
                           [0.024430683541699336, 0.03639581344094833, 0.00524202989046027, 0.013812114970246453,
                            0.045405409039839945, 0.020244892344565432, 0.03528251245253308, 0.018945975373429385,
                            0.0380270374988699, 0.016333076353199177],
                           [0.042869415136832505, 0.039643349154127046, 0.020803319268876293, 0.01349332655660146,
                            0.013242051028194143, 0.008127379597450553, 0.04925909578664442, 0.019909432056061577,
                            0.0347790752657906, 0.01332972920229479],
                           [0.04642097664440372, 0.04685588951434869, 0.03665734823280767, 0.014727328662468415,
                            0.005907555248138414, 0.013798002766361315, 0.018768804947367802, 0.02668520338366824,
                            0.03990831454482874, 0.0491112226042304],
                           [0.03169303702637536, 0.009887713947090265, 0.023678414607570752, 0.007467269830180272,
                            0.024292310754290936, 0.019467761866199803, 0.02143600152148046, 0.04323778816146985,
                            0.03274328611378546, 0.009225005525603671],
                           [0.030370953439077607, 0.026382017524130014, 0.021398803535618297, 0.01154993328849628,
                            0.018371433069579553, 0.011081914363338335, 0.03511323982363575, 0.025072758191277886,
                            0.04868925478304641, 0.02515393192436534],
                           [0.008335080188299206, 0.040546252416365385, 0.0328088826504245, 0.017762200192339598,
                            0.021162167723773932, 0.014580598611364164, 0.005758116164408248, 0.046067625459006986,
                            0.04926437681859794, 0.023337838455054764],
                           [0.032448002979302476, 0.03461110351562014, 0.005139054828873425, 0.020466830599390164,
                            0.0424313867028299, 0.029129098441724798, 0.03496626277162442, 0.00882184418345534,
                            0.025347472226384993, 0.03729668664025293],
                           [0.006229281507259953, 0.04723674722002786, 0.0303305556928513, 0.03940626093128601,
                            0.04063883437204555, 0.03925562702045383, 0.048372775849844916, 0.029185126779047693,
                            0.04290939790832562, 0.03598566985817004],
                           [0.03996955135909268, 0.028491699791736873, 0.03746417382248629, 0.009495153694258504,
                            0.009381596326419873, 0.007597259306741018, 0.006388040299201703, 0.005539520614452051,
                            0.024761192301128238, 0.027875047811857775],
                           [0.03592760130380402, 0.04963070377593703, 0.0486761935285491, 0.04772585824636314,
                            0.029653312403222283, 0.03253527876909144, 0.01139364165248875, 0.023629672969127314,
                            0.033584788284278036, 0.014618394945278269],
                           [0.04932796959848659, 0.01258485305514431, 0.005606164119416388, 0.04501978326962002,
                            0.038947455199684226, 0.03752546342685647, 0.03631212995659314, 0.031720316302988355,
                            0.04213344458989339, 0.04457644311820191],
                           [0.04143639455347707, 0.029250517978505322, 0.006284113474045878, 0.008084901296204707,
                            0.006746639097524341, 0.039747926967360396, 0.039685239668071984, 0.00859737983056923,
                            0.02635642956539687, 0.009012674556251061],
                           [0.0069967700053904045, 0.047565514756171455, 0.02635354647864324, 0.016009476736297476,
                            0.03305320372439452, 0.023625917702951773, 0.007018195426291841, 0.022568170545580822,
                            0.02661822015114563, 0.03033032576320484],
                           [0.025225385395686802, 0.04252804480253113, 0.03014542902160232, 0.03462697258238534,
                            0.04150853580025766, 0.02768001837600145, 0.010873565183665262, 0.021767841339060453,
                            0.008919495987585072, 0.03451419908099874],
                           [0.04492404515239179, 0.04235906779352695, 0.0073030440509957596, 0.03438259689987691,
                            0.042604593562534585, 0.046505831559166794, 0.03758824484650108, 0.028224872316063125,
                            0.041649770979658725, 0.04210953848681949],
                           [0.018683922226003475, 0.046680243178394296, 0.02838174940885223, 0.028791718689855854,
                            0.047421330071461355, 0.0407194003128737, 0.040290826713196407, 0.026482423396098097,
                            0.015197534510369062, 0.02883252010529777],
                           [0.043095962300567, 0.028100518034424343, 0.035401185687008827, 0.04032956361375183,
                            0.011079671237990826, 0.02390873492007047, 0.04409454195255185, 0.020684676269220274,
                            0.017401631826068756, 0.007614775173672935],
                           [0.006468889665899357, 0.04948100443639832, 0.02667465308669857, 0.02500856974614155,
                            0.009932850112230909, 0.04235898503804903, 0.03718828505977758, 0.00868746494985427,
                            0.031606427587792106, 0.015582124218058293],
                           [0.02599302160110397, 0.043527070712952795, 0.017240677399305133, 0.0371806981821565,
                            0.043483683508697225, 0.013567468536904857, 0.03821733250337868, 0.03410713375941704,
                            0.026266540142698965, 0.03164238099104716],
                           [0.0210947121474251, 0.038364619145565645, 0.014107348248572016, 0.010409375483125134,
                            0.027097348179288248, 0.04712579718929609, 0.021592954615545716, 0.0245969247647277,
                            0.005935941509073865, 0.010227392442067655],
                           [0.01538263862736191, 0.03883346623274047, 0.010382794526842057, 0.045691293583652336,
                            0.04275149703430941, 0.02490241686151585, 0.04258630537303067, 0.0370207926031031,
                            0.023907860062116742, 0.008422220632687027],
                           [0.04515136364051466, 0.043792015243608616, 0.014192559500819966, 0.04345403748021082,
                            0.017098441215041678, 0.03208545488728206, 0.01491860637875175, 0.04817712108899283,
                            0.028213768751553195, 0.0331674022152439],
                           [0.00919525392401978, 0.031572175064975935, 0.017918021063468875, 0.016182321731378107,
                            0.014749593944672533, 0.0314744488599079, 0.03385168559900603, 0.015859139793804804,
                            0.011238481875314, 0.03517076826203961],
                           [0.03342789969830189, 0.03741468175511502, 0.0458134299005297, 0.03402009871351697,
                            0.014035238864861653, 0.02985559269372474, 0.047890384313387435, 0.011839345937994347,
                            0.03400228594071326, 0.0380102250651239],
                           [0.03695004812262724, 0.03446871113670333, 0.02967922702490621, 0.0052247064200032015,
                            0.03700899645583303, 0.04724348066923988, 0.0074317714097928336, 0.03000337257672664,
                            0.02202647296306645, 0.016100260696669033],
                           [0.03678819012518859, 0.019202284852323236, 0.039386760377133866, 0.03692446582469363,
                            0.014475599619388418, 0.009168817003619458, 0.026415724747721893, 0.03780474272962497,
                            0.029822640893295237, 0.048501457704705375],
                           [0.014591834109632742, 0.016534525198645574, 0.016222878594891646, 0.04828616836469315,
                            0.017621970914610544, 0.014761729859422108, 0.027874257685688526, 0.006372657718808235,
                            0.01586549186433002, 0.033050035864667884],
                           [0.021905584422253904, 0.027217266594509768, 0.024761876910299768, 0.009711707859979091,
                            0.04986583207384541, 0.010413741952643607, 0.024739211375803274, 0.04849235131920112,
                            0.012521069431124406, 0.03405047016100466],
                           [0.03906419073748674, 0.012415004072543416, 0.025203720789128376, 0.0435593892820055,
                            0.04368098068165444, 0.022485346064096585, 0.018193695640496336, 0.009286244320754443,
                            0.03222460361116195, 0.043318295355376714],
                           [0.011746160112115548, 0.011091153102575916, 0.008683571325494766, 0.029495838371141097,
                            0.012519155616585141, 0.020352698950602644, 0.010515800621422777, 0.01132181180333136,
                            0.006437347466322947, 0.042292736769059085],
                           [0.04780069998173167, 0.009219172677430627, 0.0392467248156731, 0.02740377908641272,
                            0.03979494875415552, 0.04617163400312595, 0.01362040526487512, 0.01328611913355502,
                            0.02490981459614153, 0.03617348013943744],
                           [0.008123332208677147, 0.03371541794915601, 0.042639864336650346, 0.02341844358091675,
                            0.03309232039314257, 0.01636903048343167, 0.007924143799830167, 0.008946035340596677,
                            0.02912057144497092, 0.033383362096351635],
                           [0.04748748804852903, 0.04730820044845752, 0.021785013422959375, 0.019132497219165923,
                            0.018921621599191307, 0.03185812453571201, 0.022345943288979774, 0.031351227982035185,
                            0.016160159336890457, 0.03977780729706746],
                           [0.03251178282548519, 0.01706101448749985, 0.01549908525252747, 0.020363558088374464,
                            0.010620139974486407, 0.014652619947802863, 0.022175624906935806, 0.02607298984153898,
                            0.02217209489505683, 0.046596068458274034],
                           [0.031908647825211814, 0.04482608347302012, 0.018393907311331704, 0.030949039337450066,
                            0.022884939781539954, 0.013938780099934045, 0.039598505569232154, 0.04626466122426102,
                            0.027698648955603117, 0.010541870174208011],
                           [0.029350591352877292, 0.03775553518615523, 0.043719069363991954, 0.005966136636803555,
                            0.029347899137929363, 0.005318473901305988, 0.009085899966765383, 0.031050911763170378,
                            0.015765364945441618, 0.02231395701704233],
                           [0.023620001287743258, 0.009359483785465156, 0.029467318329577944, 0.019307173496583334,
                            0.048000430258844096, 0.009515891622674962, 0.026984933572066983, 0.042911277799137865,
                            0.04810698424888377, 0.023171765608098534],
                           [0.02090858761654347, 0.02770133629334938, 0.01831834953746355, 0.007889766446132972,
                            0.028140050251389374, 0.0221765991833061, 0.04043803110405569, 0.03906911795274597,
                            0.021641771154348947, 0.045800604091932144],
                           [0.010416678458621032, 0.03793165698279029, 0.005357243383122826, 0.01566538901055986,
                            0.027929975725048677, 0.023096283447740142, 0.013246239250661214, 0.016591515379385257,
                            0.020124742001645903, 0.04280129966494781],
                           [0.012788278610805889, 0.01900825221815599, 0.03422528506129086, 0.04729076846086603,
                            0.029862497132645566, 0.026514444167261703, 0.04284894436016072, 0.04584681968530258,
                            0.011033377836564044, 0.04140422115291157],
                           [0.023058256237638824, 0.04399940986324865, 0.04016330041918303, 0.005080603117752815,
                            0.03491398104901271, 0.040045341864139515, 0.034031172740374836, 0.019191471290492413,
                            0.03603531159570868, 0.042596477736541946],
                           [0.046358014675969055, 0.017818607736363488, 0.04395747429229237, 0.005829241029777856,
                            0.016683124380426722, 0.005200651117956997, 0.030306466915438843, 0.03661464213907885,
                            0.005090182595693544, 0.0301431636975622],
                           [0.032889802142968445, 0.04623167588947986, 0.02429066759611954, 0.01019869133107324,
                            0.02175423771985827, 0.006625026391789455, 0.028123290834043163, 0.010706010728849573,
                            0.03902060361203935, 0.03755032077975127],
                           [0.018834168499581472, 0.03942227869844037, 0.03983383260559758, 0.0428100516081356,
                            0.030688144655158758, 0.0485062601257023, 0.017755941698274227, 0.01210051249940899,
                            0.03933183479283753, 0.04638445915853673],
                           [0.03056041779152236, 0.03759257529108607, 0.0366005750697856, 0.020631892502207302,
                            0.027009282640311376, 0.02641750375615283, 0.023964856578300026, 0.037227445354505,
                            0.0386585988477621, 0.04604459662682718],
                           [0.012419087881714167, 0.03703773656029617, 0.024846102884127856, 0.049133009671587076,
                            0.03743310053107589, 0.04752077466835828, 0.009494830498448388, 0.032018971327987665,
                            0.009239722098387238, 0.008696028406467276],
                           [0.04677425050099796, 0.04311911432874717, 0.019969688591523125, 0.02978635139417161,
                            0.04223280883254018, 0.01930223442774412, 0.012856881552766406, 0.04384250501892262,
                            0.01710129928829977, 0.024722516262693745],
                           [0.03484055128265013, 0.023113339805512265, 0.038386342235461286, 0.005401638584295629,
                            0.03134640756044108, 0.040277626651505964, 0.026557745081425048, 0.03404446144757719,
                            0.044289024903750425, 0.040077752190681486],
                           [0.014012256826217352, 0.016758453099948594, 0.010808189054267586, 0.006111630051461744,
                            0.019042470025272464, 0.04042850912861643, 0.025455037232476045, 0.016312067256119826,
                            0.005396766756093934, 0.045164032397425724],
                           [0.03872241213709443, 0.02277202327839479, 0.044556308469085426, 0.014559638766047395,
                            0.022573840557378728, 0.010008178966811557, 0.029115591925928972, 0.007014702026984695,
                            0.016734975727248767, 0.009466736342239136],
                           [0.04404112993905618, 0.04069125003618314, 0.019069274431826284, 0.026115764119378892,
                            0.03990362228986361, 0.041716631787353076, 0.03579457273909513, 0.047279419269428496,
                            0.03147944291742222, 0.015826458794738655],
                           [0.008762838356233748, 0.01987508641753536, 0.014816522742694342, 0.01068422760939676,
                            0.03688148664398665, 0.015426771586047543, 0.017095386408499055, 0.014283247860243094,
                            0.020373123882911733, 0.026120923049234662],
                           [0.005829486336636015, 0.04994120265972566, 0.04412052033079155, 0.01195942466542867,
                            0.008731682747246103, 0.008438704722250459, 0.04495078556563886, 0.009811285525867044,
                            0.03688084235015041, 0.010756119199715964],
                           [0.04272761357629337, 0.007681750191763918, 0.040983580169520924, 0.006601186663526549,
                            0.01589793312648897, 0.030142944566899955, 0.024928680854923255, 0.012573685832080333,
                            0.027260240768042194, 0.020896378804933134],
                           [0.01896635733933095, 0.01398666541620848, 0.02216384800624855, 0.04167467312730931,
                            0.04143609992452048, 0.027130044431265408, 0.016809601988894054, 0.008379492735642381,
                            0.049676104840005286, 0.03693406007505278],
                           [0.03359674584478351, 0.00606435856428158, 0.03350343517685703, 0.006952226153367241,
                            0.01871204901131894, 0.00850083163279318, 0.0438908144231299, 0.03164530039030832,
                            0.04108438711661421, 0.021065866208347203],
                           [0.024510717520045487, 0.010965859312899284, 0.04501404257276819, 0.022737295516923838,
                            0.0497133517848057, 0.025595544066567062, 0.04868950598939358, 0.023252629447687388,
                            0.04653569744263246, 0.04776086990323713],
                           [0.03377953458588898, 0.014877060798937004, 0.04307291553269222, 0.025740012955137673,
                            0.034811327099332, 0.04018522188719354, 0.028099770865987424, 0.02239597704561444,
                            0.021259854197715138, 0.019550779488756193],
                           [0.005109230075589886, 0.024256260906530314, 0.037219567144505014, 0.011084172177569178,
                            0.010188378057453354, 0.017962560659441445, 0.04729168775168664, 0.02492065225425394,
                            0.019126548597139297, 0.02724016030873975],
                           [0.0499872264824053, 0.01578220724608491, 0.039632658667298516, 0.04903951302608984,
                            0.03932221150124168, 0.008616604704271582, 0.0346148124577519, 0.04162744688117539,
                            0.01634848744660001, 0.03712229224787633],
                           [0.011916652913009161, 0.005791136424618239, 0.046266690895896204, 0.02387969739257014,
                            0.040834038895559505, 0.00931805082508649, 0.011201860837575907, 0.047429492283186915,
                            0.017846720861538802, 0.01253648043881666],
                           [0.006003601120607258, 0.039841461407444885, 0.005985847423598251, 0.047551870688879595,
                            0.020683334921946576, 0.04968444484105832, 0.041813237403034574, 0.037404359814932765,
                            0.013729141404005208, 0.019165069557273314],
                           [0.012947408076344626, 0.044723415666352885, 0.04761868976798498, 0.01439916643665625,
                            0.03141629987523862, 0.04306829464680261, 0.03420488002314209, 0.045425354970194974,
                            0.022295461424182816, 0.04389248590973355],
                           [0.0053872594148395275, 0.027898665930223877, 0.04563325493609173, 0.037451302988316296,
                            0.029017352247932785, 0.007519093577662157, 0.007806631867994166, 0.005176600441550583,
                            0.020922611199312127, 0.04072663379641977],
                           [0.0214553311125957, 0.04913324946215417, 0.033002862236459304, 0.02531096397034037,
                            0.04924945357463482, 0.01921171223306217, 0.01147201098666873, 0.011512028578867661,
                            0.02567269388696986, 0.04997847093343963],
                           [0.026198927281335938, 0.04175072486435096, 0.02048278519622126, 0.03979456770431986,
                            0.00913418139157282, 0.04971931959972772, 0.033476696856833535, 0.041592436575500887,
                            0.02148765491787998, 0.00911376190466672],
                           [0.03724211357427991, 0.029315971083017287, 0.046110072482825004, 0.04425433713927332,
                            0.04318740319663871, 0.015524044591402202, 0.04066567779979656, 0.048206926898015805,
                            0.03842869273178805, 0.03663882732673997],
                           [0.01667445380415266, 0.012327211589288942, 0.0413074692086983, 0.028304237383731577,
                            0.03110293694868698, 0.010931479753118428, 0.046512355067008716, 0.0190270655251186,
                            0.04923005112352888, 0.04161653563069649],
                           [0.02479677298817738, 0.038050764982467386, 0.03102513187524288, 0.04708710593428631,
                            0.014196593313512335, 0.03708833783525589, 0.03128277488329507, 0.01734636997003995,
                            0.01490270857081518, 0.012022613583062301],
                           [0.030235216213526313, 0.007082973316544405, 0.007749394421090892, 0.03443849716219311,
                            0.02848789306736962, 0.04506396650441203, 0.014295698740059083, 0.010252900824967198,
                            0.01992966787964868, 0.02061986017694009],
                           [0.02922168169493548, 0.018082193088623123, 0.009596894631454668, 0.045483226718148285,
                            0.044602178357912345, 0.03539696075585631, 0.04277002837146326, 0.0066878264912792005,
                            0.027356200416042694, 0.00643307674638662],
                           [0.03643870805708788, 0.021512891086631892, 0.020065751044575763, 0.00551966361927437,
                            0.04626184655291875, 0.028924330131141478, 0.02257611389439959, 0.025398804061322215,
                            0.012082095961843783, 0.01875416967484882],
                           [0.0061566868747421856, 0.00777909564356357, 0.021239087221537663, 0.03352364396975099,
                            0.04728700149052222, 0.03882611041914923, 0.012780862474694583, 0.028341096438794056,
                            0.009098323514433756, 0.04705818444061962],
                           [0.007308825658479663, 0.02445037721311853, 0.039104273251372085, 0.03309667212729586,
                            0.011547060236167087, 0.0242323538259776, 0.04994114443025109, 0.028780917220574367,
                            0.021854784665633176, 0.025373680055380477],
                           [0.010331206272978975, 0.04903609796313997, 0.031893265760917396, 0.0401340388231393,
                            0.015316286437514339, 0.01570437816835138, 0.04512476177857368, 0.03625808372602799,
                            0.03977290804190325, 0.02443948391954817],
                           [0.009872666869860063, 0.028296669010155696, 0.0203873036230351, 0.007171821089317866,
                            0.016093162426392922, 0.00731838929936556, 0.019449272248309106, 0.032652592575620636,
                            0.030559576648764714, 0.01850768081473857],
                           [0.03229419216370146, 0.020884636842314626, 0.022320183988112315, 0.017222899934131834,
                            0.02681283478488545, 0.02080529369148572, 0.02056911939086087, 0.005765363222229641,
                            0.04591476655289037, 0.03531531912255183],
                           [0.039687113365614396, 0.025290637386108542, 0.040803506736485105, 0.048836903043247566,
                            0.03180776390490933, 0.011960700377594889, 0.010769980094018031, 0.03430631836824289,
                            0.031064313901631814, 0.007772729008113667],
                           [0.03992050485057065, 0.04419631767283281, 0.030687517536522674, 0.02917581521852262,
                            0.023414553337076274, 0.037709873987660375, 0.03362294086757893, 0.020131218470470776,
                            0.0195658220553962, 0.04199207532468904],
                           [0.04598045630261411, 0.03733996023804372, 0.020635175998356747, 0.04530715058059671,
                            0.014811677000976502, 0.030331293558891328, 0.04769790147476487, 0.024347531973952088,
                            0.024561682307181868, 0.03352064976485503],
                           [0.02540784763367206, 0.01329376365410117, 0.005831002067450855, 0.019085833489753047,
                            0.022187619406739082, 0.026173148932127167, 0.04376897883407012, 0.02341409349997081,
                            0.0082388326853238, 0.04612735643152319],
                           [0.04467562981556489, 0.04100603139993887, 0.033660401448589373, 0.04693533327377245,
                            0.025245359605886142, 0.021134839971978455, 0.020616852340224145, 0.029026669061443626,
                            0.02833233561466102, 0.012982454148449032],
                           [0.04373352872495395, 0.03226065381549463, 0.018843723527851016, 0.04777390524524972,
                            0.037996857678741947, 0.009767209850405478, 0.039927871974520766, 0.04559177157591775,
                            0.007977180791760777, 0.022464404479374034],
                           [0.011001396210815688, 0.020448781928976374, 0.007211015321928358, 0.040644959854387666,
                            0.016138573510288082, 0.029245687739242923, 0.04137263063332093, 0.04579678840524958,
                            0.016692188078447034, 0.040875615844633055],
                           [0.04550054724790139, 0.03996228494957334, 0.014679306278588187, 0.021280966111468874,
                            0.023288680066284016, 0.017019042131022427, 0.018566027355192145, 0.02931588979424113,
                            0.0329693547036976, 0.028487696135026823],
                           [0.019885626303629635, 0.012054119505886984, 0.029117028184877377, 0.02329461811453198,
                            0.044541978643386916, 0.026641053236365713, 0.013555408268363347, 0.022476212867441704,
                            0.01822851475715445, 0.04316317830123527],
                           [0.04022906114764718, 0.03490688392398833, 0.031643115047326094, 0.04599629499733613,
                            0.01201941829906145, 0.03864096097100446, 0.019712854217464753, 0.0065645121495459865,
                            0.024073696220952137, 0.005025700015413076],
                           [0.04105310672306777, 0.03547542013870518, 0.04917768542573394, 0.03173161083353454,
                            0.0485028906330986, 0.03851138158346338, 0.0385309525221923, 0.04445202110155239,
                            0.04268364928705378, 0.026875900096814977],
                           [0.048596354098024104, 0.03658490085552626, 0.006553960452417306, 0.028055650202984568,
                            0.04023460243917465, 0.015141175167018288, 0.03803713350955397, 0.03986203355644792,
                            0.04793136826275444, 0.03335245853190657],
                           [0.032730786620249724, 0.02603239543167589, 0.015580960547613213, 0.04939354465374922,
                            0.02459732382344007, 0.04994918554390341, 0.014720575833780232, 0.02678534756116507,
                            0.028462514382568464, 0.01147569677521251],
                           [0.038189181875092015, 0.03180363774763045, 0.04200650953852682, 0.04487353462202118,
                            0.035858488815131186, 0.025078545773584187, 0.01394507372271377, 0.01810037615145729,
                            0.02874542136082197, 0.026187481067698293],
                           [0.03505062740717129, 0.04278912125593421, 0.01515422246185559, 0.04158819075914071,
                            0.017774055776095754, 0.026922987072396088, 0.030803182979583947, 0.005932930358711197,
                            0.019503615239154396, 0.011013747720926871],
                           [0.03521313918020118, 0.005601046753210021, 0.012932248784603313, 0.04382556382626371,
                            0.039993406756398585, 0.04598107628814767, 0.02509046525955461, 0.039306513081415244,
                            0.005573970443103555, 0.031812221196175],
                           [0.01622069032318666, 0.03859685809342213, 0.014367331227940094, 0.048031660815734734,
                            0.02054120203671272, 0.03686306437998371, 0.01214530614704537, 0.02978083390082365,
                            0.010625937678933748, 0.017004963723521078],
                           [0.04689142000408602, 0.022319863386066407, 0.029443908149743528, 0.034741953882369746,
                            0.02731577833171775, 0.033134548835591096, 0.03474131259213653, 0.03259449619303298,
                            0.03961913016485228, 0.029077129531794765],
                           [0.01839574302865395, 0.04220459600756815, 0.04268789611599023, 0.030366310642928225,
                            0.046093836981572094, 0.013281003207212677, 0.02990471796235017, 0.02640482375389861,
                            0.012498490248747905, 0.036471832027027716],
                           [0.011595697596571367, 0.011801909395557808, 0.03859658901174864, 0.008973718347063675,
                            0.02751366039768431, 0.009855359279865899, 0.022030372169285332, 0.04400118674675557,
                            0.01474073854615342, 0.03935002646260049],
                           [0.0376374475502119, 0.04358910354606104, 0.0174049904087581, 0.01588171460578275,
                            0.03647946011413684, 0.03654257535729058, 0.04150679590659755, 0.036890854399000896,
                            0.04363047139231686, 0.04719150657118887],
                           [0.042281428708025134, 0.017388707611857443, 0.00824933778239887, 0.016735817018286706,
                            0.013860340481776184, 0.048652228654464766, 0.03836293919279311, 0.021626065607358448,
                            0.01847220337370124, 0.018173475388832058],
                           [0.04390291333938459, 0.020636122993521886, 0.033881269731218105, 0.029925054035066984,
                            0.0208047494517343, 0.04818692023925879, 0.019227109433102685, 0.020881504378666493,
                            0.013476111490192663, 0.019261288406638245],
                           [0.04840382398443666, 0.030148824563445132, 0.023197939850455405, 0.005758987427433909,
                            0.030807221652529446, 0.01968112167259412, 0.005424086659838573, 0.011958533744073317,
                            0.0412230631458628, 0.026742058873946477],
                           [0.04124934138706181, 0.021080268077234614, 0.043605523898039955, 0.047382985122399426,
                            0.026077241885237763, 0.039936101728898436, 0.03690495747272888, 0.03230073619562196,
                            0.019731785991948053, 0.023899019206482686],
                           [0.02080258827692606, 0.021210101281871622, 0.048190886360184745, 0.020074474242814094,
                            0.017388944950181673, 0.0466935268388838, 0.04823134727400356, 0.0057446043586346284,
                            0.03284168851464952, 0.03852265134537035],
                           [0.034199303214489045, 0.010731645789267549, 0.01761821075840501, 0.039222875284838074,
                            0.03883485935856826, 0.04923791129111862, 0.04416693729131508, 0.03333481936418653,
                            0.03497660671145157, 0.04754432387749269],
                           [0.00769235797930681, 0.03619988266216615, 0.01644414978445513, 0.04998872619943993,
                            0.049026277542830655, 0.015342333514634696, 0.024401131244194352, 0.013669744191637065,
                            0.0070497503135712, 0.02709434127258643],
                           [0.01599933719644903, 0.02822895361634078, 0.04175970951434357, 0.018558913574655955,
                            0.0061527861728898755, 0.03412116984453176, 0.03049765532667523, 0.042066801982526686,
                            0.025494057937820513, 0.04585119302823544],
                           [0.014271790409199701, 0.0249236763148691, 0.012213117104361338, 0.011736091695710122,
                            0.035433983129186816, 0.008255927081404806, 0.04975308687122862, 0.0425970081796326,
                            0.045469300274133914, 0.01648747240608918],
                           [0.03731587081627487, 0.01983082659581604, 0.010411873659490315, 0.03366900528336806,
                            0.023062859326845955, 0.017761881757757246, 0.040921743151295466, 0.04665602016327729,
                            0.04320939451099982, 0.014398409252431573],
                           [0.014426251115421649, 0.048129464887976516, 0.026153305769406814, 0.035864104805706606,
                            0.017243342667819465, 0.048320508331019495, 0.049618671132005576, 0.04413162349501227,
                            0.008894293848240309, 0.014534324853010528],
                           [0.01642830596402218, 0.01855630102449137, 0.023591752771606873, 0.01635343037095868,
                            0.01824237912982002, 0.03036496662811555, 0.032330479824535706, 0.03290929967410783,
                            0.04236757769526706, 0.03551302048388809],
                           [0.03299892706552598, 0.017231659032585535, 0.030007647769458003, 0.03154502442324105,
                            0.04670635867613615, 0.006398490836284773, 0.024550516826655613, 0.0244329420369976,
                            0.012262837490025453, 0.02431452612190801],
                           [0.03805787407656821, 0.015829978550772665, 0.035501674923167824, 0.013251227012512794,
                            0.04283435328380561, 0.025465519335143738, 0.025917064548546243, 0.0091022618989936,
                            0.04509391554786887, 0.03477691054100679],
                           [0.04204814443181919, 0.046285603422012295, 0.03801225652678169, 0.018360498585396064,
                            0.019506457256442386, 0.04287209239532307, 0.017819931936546383, 0.041900920690916006,
                            0.04374999287531825, 0.030674984032234603],
                           [0.049563247490866484, 0.006577177560776335, 0.019164587817434908, 0.019855527890954704,
                            0.0053203773246621525, 0.030038181087302177, 0.023103703791826666, 0.013807161457233756,
                            0.04849178959111378, 0.021287166770401798],
                           [0.015266262907993464, 0.023477880373527125, 0.02576124266159111, 0.032182379996385846,
                            0.019345358489584713, 0.029590258681423905, 0.04487596065085574, 0.04989930452686145,
                            0.049738751670242914, 0.005963481938055651],
                           [0.009301246881737665, 0.02876168736119158, 0.03689809044477766, 0.03285302220881822,
                            0.011746073712583153, 0.01993501379547364, 0.03161937695602978, 0.045227362979789625,
                            0.01796489449519819, 0.007112788143366458],
                           [0.02914644477950473, 0.03235793427119994, 0.02416789834173064, 0.0497905782522779,
                            0.0214626996023832, 0.022869773917447576, 0.010234907838336854, 0.039471417212917984,
                            0.006606882846680613, 0.012211447626554578],
                           [0.04481081482025662, 0.028654233852438604, 0.027481281278103786, 0.022337801496170557,
                            0.008510795051739777, 0.016871022360906103, 0.016272694520774393, 0.02185952320253532,
                            0.04480135169664522, 0.03898622540873182],
                           [0.010796306590054835, 0.014054740890651455, 0.03900249553845563, 0.04675746451833353,
                            0.03914064371208133, 0.007221083652890356, 0.02278919907515344, 0.01610563184035065,
                            0.03563928283073872, 0.024829084680027973],
                           [0.016386192265528166, 0.0129625257630752, 0.037090311952301946, 0.03206261674329985,
                            0.03504834534033722, 0.021394150055124524, 0.020197818952908755, 0.04855626170589669,
                            0.01746190792152245, 0.005251099959262916],
                           [0.02413791733556524, 0.01418009820236061, 0.03585662936765647, 0.04961035755678205,
                            0.022292461385832418, 0.03285690826826881, 0.013339255797398933, 0.034762247390299705,
                            0.04990543863858117, 0.024960761246849227],
                           [0.02030065505638935, 0.04507356103997628, 0.0447431025958544, 0.01812623224084133,
                            0.007092295385754654, 0.006662353941336588, 0.032216141053653426, 0.016917670721464856,
                            0.03747978722348479, 0.027582677461056915],
                           [0.01963122102613836, 0.014727316137662445, 0.019811620199922726, 0.021695882104669315,
                            0.015779755634785408, 0.034513737514199734, 0.02338561238464482, 0.04238538470859948,
                            0.025968823536122187, 0.005900661510769529],
                           [0.027851414645867364, 0.038608533656777716, 0.04218852859740293, 0.031307720557603815,
                            0.01510708038361799, 0.010327353057854597, 0.005982177511960103, 0.020263585026944766,
                            0.0344991778925035, 0.019803583142504157],
                           [0.03138354490594714, 0.016576338652075858, 0.01954067488485227, 0.027538118339307718,
                            0.04295461952032487, 0.044632947489609845, 0.031043892072892643, 0.03381453865886708,
                            0.02967913353195915, 0.027456067548166214],
                           [0.012648277213949139, 0.02147141723593203, 0.02104209150238283, 0.04175438111690916,
                            0.0380752572098937, 0.03511219731206188, 0.014325732577487673, 0.01779038257458537,
                            0.012712711626257492, 0.006312587371308425],
                           [0.038798071221229184, 0.040786062425047544, 0.011973230919468097, 0.022668067055476082,
                            0.010864379122508525, 0.035510234146772277, 0.018376795032121522, 0.02732495797820408,
                            0.021118718645008677, 0.027660777105358823],
                           [0.040398176337111416, 0.041589887557204185, 0.03313944326562041, 0.013562457446489234,
                            0.04423786178446942, 0.02122480870954843, 0.01655409545458892, 0.017089483310473264,
                            0.022476209248644827, 0.03365572632857359],
                           [0.048960081369487936, 0.010242854772025092, 0.03330655009788278, 0.02194347781464191,
                            0.008220460572221562, 0.03526244126477144, 0.047048377591589564, 0.028526910596301427,
                            0.018880333232398495, 0.015006107089722793]]
            self.y_data = [[0.0007925424056634186, 0.0005665900911329277],
                           [0.0010585048722607602, 2.8494086496458716e-06],
                           [0.0008414107857886328, 2.8002946087861272e-05],
                           [0.0009251870560912125, 2.384788397875836e-05],
                           [0.000874573156497512, 8.062597191488177e-06],
                           [0.0007638498146239093, 0.0003703263009294565],
                           [0.0009308519077971741, 1.1638228072977165e-05],
                           [0.0007864811543508003, 2.2304400947606054e-05],
                           [0.0007491363378327119, 9.534515353957354e-05],
                           [0.0009426003453837156, 5.6693756291933275e-05],
                           [0.0007637592096321781, 1.2130102980903709e-05],
                           [0.0010310354715801174, 1.727594055837286e-06],
                           [0.0007087635853838984, 1.0214029282239149e-05],
                           [0.0008681399157594405, 1.376450905520804e-05],
                           [0.0009871100149544995, 1.1972027081289327e-05],
                           [0.0006600383157709415, 4.411301452868887e-05],
                           [0.0011150290136542483, 0.00011941591570269189],
                           [0.0009844470596521951, 1.0390571314487557e-05],
                           [0.0009924202951985919, 1.7780738978293172e-05],
                           [0.0008890215322758985, 4.1344186535449104e-05],
                           [0.0008712156727370767, 0.00010132154128213327],
                           [0.0008561045818881447, 8.381566483689625e-06],
                           [0.0008890942059352698, 0.00016129868006170442],
                           [0.0009333952496798223, 2.1249592561484834e-05],
                           [0.0006761969398906298, 9.488863813051396e-05],
                           [0.0010801998890405827, 1.3152742457748038e-05],
                           [0.000985491878546432, 1.609348230264967e-05],
                           [0.0010407607851237884, 1.3881360860358527e-05],
                           [0.0009448438028736406, 0.0004416951486649948],
                           [0.000980012794338712, 1.8530270631060155e-05],
                           [0.0009318687505264468, 1.7917542301062093e-05],
                           [0.0010243445967227825, 6.603101392164839e-06],
                           [0.0008622056007327381, 3.090527605691687e-05],
                           [0.0009306879741384426, 2.2924441992576388e-05],
                           [0.0008443141609667974, 3.066068249429251e-05],
                           [0.0007245272093153513, 5.363406370392003e-05],
                           [0.0011849585828018208, 4.675871037442442e-06],
                           [0.0008105349715943244, 8.094003205300774e-05],
                           [0.0009490669794237794, 0.00018975019639924567],
                           [0.0008778935581885564, 1.8461703640382713e-05],
                           [0.001044474062145572, 2.9164971161063217e-05],
                           [0.0008351480431144039, 1.6071860440487194e-05],
                           [0.0008837319240273249, 6.812666550480133e-06],
                           [0.0007736249258360838, 8.434890991008527e-05],
                           [0.0007095981348908593, 2.8623123810378887e-05],
                           [0.0004229746063234756, 8.238149709813923e-05],
                           [0.0009095265551539573, 9.751977705592244e-06],
                           [0.0008380747797696737, 7.092838555805115e-05],
                           [0.0009123386682751461, 4.220635337701501e-05],
                           [0.0008254765804077775, 1.71460820743383e-05],
                           [0.0008237728388182426, 0.0005858396818730046], [0.0007841947168334, 4.003196446303115e-05],
                           [0.000783387769347537, 4.1827511111575526e-05],
                           [0.0007836587539256941, 2.2815579760238418e-05],
                           [0.000689810913593528, 4.677874357943257e-05],
                           [0.0006928092388136206, 3.2041946816423834e-05],
                           [0.0009628308413665072, 9.091327506849174e-06],
                           [0.0008674853886132909, 0.00011556832649864892],
                           [0.0007316818498915116, 9.06142487908154e-05],
                           [0.0009841384143746672, 1.7708658169128936e-05],
                           [0.0005633713619061733, 4.909505572408975e-05],
                           [0.0009395683797016946, 1.7337711014965126e-05],
                           [0.0009045219642133674, 2.2720866604547203e-05],
                           [0.0007007077593375118, 0.00012193579125573799],
                           [0.0005690170033905004, 3.887874627370344e-05],
                           [0.0007657104749808448, 0.00037999721152594636],
                           [0.0009503806278811835, 9.151835128402212e-06],
                           [0.0009675609365212856, 7.1313307453442594e-06],
                           [0.000974510097771328, 8.50051762275781e-06], [0.0009608494311319217, 4.102701188378303e-06],
                           [0.0007919129343864603, 1.263242483445386e-05],
                           [0.0006769311078530377, 0.00034059633553839994],
                           [0.0006713181899141524, 8.926997665601477e-05],
                           [0.0006864306990085552, 0.00016352391301309408],
                           [0.0009352471463769254, 1.0653548245634068e-05],
                           [0.0010293617808291551, 4.7273146880616684e-05],
                           [0.000520421542537526, 4.0163695023880504e-05],
                           [0.0009245775882230998, 4.506212105650066e-05],
                           [0.0008722753745296028, 0.00019590559689400478],
                           [0.0006122954119062695, 0.00010077946292202919],
                           [0.00092631687823538, 7.996082822598985e-06], [0.0007589181095245227, 6.239775485629166e-05],
                           [0.0010302838487190114, 2.901873927404364e-05],
                           [0.0007497058135891877, 0.00013209109631584398],
                           [0.0007242117041892783, 2.1283653718063533e-05],
                           [0.001019036356802633, 7.293821879143868e-06],
                           [0.0008582616016361702, 1.3730368922431366e-05],
                           [0.000820213479118376, 2.5165289411630813e-05],
                           [0.0009949122986752052, 3.4406844061796946e-06],
                           [0.0007421884604245471, 1.6985562432437836e-05],
                           [0.0009712596286981783, 3.67522088506879e-05],
                           [0.0010387433319199418, 1.3354019906832657e-06],
                           [0.0009491915123565882, 4.859175025186338e-06],
                           [0.000942916560522447, 2.4216395413043535e-05],
                           [0.0008333603263210304, 1.6122495796027574e-05],
                           [0.0009872862002234886, 0.00018510885686371818],
                           [0.0008135829258114036, 4.074898102162636e-05],
                           [0.0009821072262702643, 1.975128047594417e-06],
                           [0.0004793268163100407, 5.593606323953503e-05],
                           [0.0009433825535621887, 1.987828938098789e-05],
                           [0.0010534018290836015, 5.373172841701729e-06],
                           [0.0008173150306250249, 1.0431010198263167e-05],
                           [0.0008265869873916913, 0.0011118198905213012],
                           [0.0008559272110632151, 1.0563162846433177e-05],
                           [0.0007147781881168918, 6.779868933842494e-05],
                           [0.0010599274833073813, 1.3673728897275495e-05],
                           [0.0011716453929469881, 5.2637387954432755e-06],
                           [0.0007457008995842067, 0.00017798836201445315],
                           [0.0007848619037893339, 2.6597924273268394e-05],
                           [0.000681965331031664, 4.688892939979759e-05],
                           [0.0009748936108733166, 3.0194125866290043e-05],
                           [0.0008531581576957838, 5.701214513967124e-05],
                           [0.0006798851273004801, 2.6634373650710846e-05],
                           [0.0005063954575633604, 2.9812152247067138e-05],
                           [0.000896860712254886, 1.350104914658771e-05],
                           [0.0008383657569342121, 3.714845483750536e-05],
                           [0.00043456623412789166, 5.144773393037422e-05],
                           [0.0009744203372013663, 7.845464058544707e-07],
                           [0.0008146308216879444, 1.3369494298034185e-05],
                           [0.0007863820402647026, 2.4578230232840748e-05],
                           [0.0005324596070001466, 4.273231395330076e-05],
                           [0.0009129105767939171, 6.665709843411428e-05],
                           [0.0009160502845906562, 1.45972192736778e-05],
                           [0.0008932693526968915, 5.8815602362153435e-05],
                           [0.0010852198973956076, 6.620142478393173e-06],
                           [0.0008747444663238607, 1.830670004911971e-05],
                           [0.0006970161943219365, 1.987971633843321e-05],
                           [0.0006366455020916801, 2.4819537319569796e-05],
                           [0.0006851004048795766, 6.154148784580448e-05], [0.0010096105731193, 4.050551665063854e-05],
                           [0.0006620670021369798, 0.00018266085204902104],
                           [0.000899936693532784, 1.1642578199481923e-05],
                           [0.000837992528312776, 1.974583936573093e-05],
                           [0.0007366124872955382, 6.0151649113520754e-05],
                           [0.0006911156793848897, 3.425109608484562e-05],
                           [0.0008581536236389556, 5.5080450604157145e-06],
                           [0.0005325864908892018, 9.724743663874842e-05],
                           [0.001024795887810873, 2.714545578929375e-05],
                           [0.0008922134914688346, 1.4081077236374736e-05],
                           [0.0006908700178775859, 0.00023803112422652093],
                           [0.0009681276011244109, 6.427420301312383e-06],
                           [0.0007473117554994532, 3.522188098520935e-05],
                           [0.0005459521956177326, 5.645094074761669e-05],
                           [0.0007138153496185944, 5.9302141537103946e-05],
                           [0.0006019593937385818, 0.00012203842448384563],
                           [0.0003824726825447726, 9.106354640836211e-05],
                           [0.00039147187342213855, 3.9523149668613855e-05],
                           [0.0008713035640954209, 2.263113121279153e-06],
                           [0.0007007821742201448, 1.9918142490526043e-05],
                           [0.0005289708663153738, 4.602352690675059e-05],
                           [0.0010895372322683196, 7.784580884561608e-06],
                           [0.0009281035525757939, 1.3669532048878744e-05],
                           [0.0008461364007825968, 2.100512268542016e-05],
                           [0.0005142740640675288, 0.0002234216906170518],
                           [0.0010674571040720955, 7.499111398769335e-06],
                           [0.0006639583499858664, 4.12064021566654e-05],
                           [0.0010800993361943898, 4.454121097571768e-06],
                           [0.0011088928821601713, 6.281005074717913e-06],
                           [0.0010447117337918491, 6.166977292837267e-06],
                           [0.0009908474328398234, 0.0006972119228249687],
                           [0.0011734535385022986, 5.322822964716142e-06],
                           [0.0010607998339840633, 8.432843440663979e-06],
                           [0.000728286363528645, 1.003876157206595e-05],
                           [0.0010978802184722096, 0.00020117472017046036],
                           [0.0009581809478185817, 1.2076960448404152e-05],
                           [0.0010706742107879006, 2.274293707123082e-05],
                           [0.0010295078377422075, 6.2340605970630305e-06],
                           [0.0009929019464142226, 8.89104174257169e-06], [0.0011952403637113024, 4.69199336408724e-06],
                           [0.0005368992700338172, 2.6778245999333798e-05],
                           [0.0007407855082797821, 2.8855561383426108e-05],
                           [0.0007644638627602774, 0.0006278089763466673],
                           [0.0008003585344939782, 1.9401640972987943e-05],
                           [0.0009003877403885162, 0.0001170417735715546],
                           [0.0006145823767484273, 7.469747097079194e-05],
                           [0.0007449178090528109, 1.7618499669040572e-05],
                           [0.0007917731373564069, 6.61310447591173e-05], [0.000905159076578529, 0.0008065776519163843],
                           [0.001092919182818895, 0.00024376567214998965],
                           [0.0008772844626700118, 5.578826539826172e-05],
                           [0.0011005088759159698, 4.75910396007134e-06],
                           [0.0008311368245230549, 0.0004109440818708806],
                           [0.0007342718240152611, 0.00018017190411205163],
                           [0.0007734856307151691, 0.00014317577459111287],
                           [0.0010051737664051692, 1.0675457141850279e-05],
                           [0.0009979489614306738, 7.411940232951084e-05],
                           [0.0010293006555458614, 6.729131251218018e-06],
                           [0.0009402037413601612, 1.084515595369716e-05],
                           [0.0008135866976490847, 0.00019351276437479665],
                           [0.0009644191433647279, 9.155220030372248e-06],
                           [0.0007702841369004863, 1.6170671056767797e-05],
                           [0.0008889527560157718, 2.8074695062118673e-05],
                           [0.0009515647322301069, 1.096111965225368e-05],
                           [0.0006227342642606577, 5.158406911755844e-05],
                           [0.0010148692721997585, 4.403765772358281e-06],
                           [0.0010416099088280861, 0.0005851023332755385],
                           [0.0008659535985236261, 1.2885281871424912e-05],
                           [0.0006454905828760025, 1.2817332867824986e-05],
                           [0.0007655451500322049, 2.3557662070007896e-05],
                           [0.0008674584813143287, 8.856861705353588e-06],
                           [0.00036619421110539437, 4.1503682430357475e-05],
                           [0.0008675219136518872, 2.462599635899522e-05],
                           [0.0007626814671464603, 6.652649036210925e-05],
                           [0.0009251180540016106, 8.228900687756101e-06],
                           [0.0006140804104382597, 2.0937704874241867e-05],
                           [0.0009517674154145132, 9.269237062582967e-06],
                           [0.0009872028048595759, 0.00021160772016642752],
                           [0.0007647916640761233, 2.3224222580949718e-05],
                           [0.0007242942238451349, 5.648662879002465e-05],
                           [0.0005650697723750336, 0.0005105574349513307],
                           [0.0009101601927383088, 1.7508959649759813e-05],
                           [0.001018409708674942, 0.0009459022826620646],
                           [0.0009648025530663635, 0.0002477617537785257],
                           [0.0008450946690102171, 3.521924011464604e-05],
                           [0.0010100229484474531, 2.8329962182638002e-06],
                           [0.0010052316423376532, 8.652363536356032e-06],
                           [0.0009146122576147075, 7.135890470149718e-06],
                           [0.0009724876325779789, 4.397487125099371e-06],
                           [0.0009911920912535725, 0.0004363095385437614],
                           [0.00046910739524457174, 0.00022122681971643138],
                           [0.0008476044767019245, 1.4225962430368518e-05],
                           [0.0011021372225760624, 8.186677144436385e-06],
                           [0.0004604793427321793, 6.06349411792995e-05],
                           [0.0006468378372755016, 0.00029357779052072554],
                           [0.0005756452371128695, 0.00012284563991282955], [0.0008170584072086, 8.89309801840898e-06],
                           [0.0005482602814881866, 0.0001762745439604965],
                           [0.0009305983456823297, 2.3878962098497554e-05],
                           [0.0009466126963690213, 1.2603930611780905e-05],
                           [0.0006764148623841389, 0.0009909617025753038],
                           [0.0009962862769668538, 1.4278805307888455e-05],
                           [0.0006701864648210985, 0.00030220968744527335],
                           [0.0007893465475069403, 0.00021970952057019614],
                           [0.0009455409728252226, 1.9632801903985985e-05],
                           [0.000881100573793759, 0.00042697886007071284],
                           [0.0009177294798619482, 7.35600801934417e-06],
                           [0.0009418370301121488, 2.0591976290781975e-05],
                           [0.0011398432065756346, 5.017765649891877e-06],
                           [0.0007931351391755164, 1.663738810813122e-05],
                           [0.000941510775244343, 5.652964620783635e-06],
                           [0.0005862799760302288, 9.736763349480767e-05],
                           [0.0008703745032229044, 1.749101639101409e-05],
                           [0.0008506600091280249, 0.000449558814710851],
                           [0.0007592287956162036, 0.00026550687944006145],
                           [0.0007922546014249352, 0.00012306860623678966],
                           [0.0008787151600585496, 3.4313786267294635e-05],
                           [0.0004297475533260073, 6.929081662098814e-05],
                           [0.000837012522170824, 9.725297699417409e-06],
                           [0.0009935208313893466, 1.0760232925558377e-05],
                           [0.0010328489013296494, 5.039788736642683e-06],
                           [0.0010025644276306092, 8.686549888097799e-06],
                           [0.0005828396730252758, 0.00030751615631483843],
                           [0.0010472781081333495, 5.892334119966341e-06],
                           [0.0010570737285726338, 1.3229994198808528e-05],
                           [0.0006307719336259602, 9.556855561132093e-05],
                           [0.0008496184282434318, 1.3687248600976854e-05],
                           [0.0007407260184792085, 1.8075189297066648e-05],
                           [0.0010063098850405265, 8.987545676563895e-06],
                           [0.0012255251510939005, 4.294489677796414e-06],
                           [0.0008899756147019955, 0.00013947776444988335],
                           [0.0009260307358539252, 1.0176360828797875e-05],
                           [0.001022458825267972, 3.7015049925785683e-06],
                           [0.0009282333594430593, 9.597761471532313e-06], [0.0009546822238929543, 0.00038750718436829],
                           [0.0008026069660628784, 6.311741485892521e-06],
                           [0.0010596810667644946, 8.298993272397523e-06],
                           [0.0009458078014478136, 4.221131080976692e-06],
                           [0.0005141473796859439, 3.7181423105348075e-05],
                           [0.0010224475054160386, 1.2157984392006952e-05],
                           [0.0006769892407971154, 5.020810645817439e-05],
                           [0.000933215595510814, 4.585626567132628e-06],
                           [0.0009532278467103888, 0.0003088177271633614],
                           [0.0010808178350017485, 6.513432672299816e-06],
                           [0.0009435359880822255, 1.0388111358431685e-05],
                           [0.0009507836170601683, 2.6017097598967806e-05],
                           [0.0008109965722747468, 9.67345352799021e-05],
                           [0.0008262454612199842, 9.115306747669838e-05],
                           [0.0006603521930245668, 1.5521193991290136e-05],
                           [0.0008592746760517122, 2.565102166362613e-05],
                           [0.000981376506420099, 8.929482784899657e-06],
                           [0.0007643300873186709, 2.208644013691686e-05],
                           [0.0009160211466588642, 4.115441900477233e-05],
                           [0.0008631146348311388, 5.196687393262204e-06],
                           [0.0010416360557500748, 7.751011494063575e-06],
                           [0.0005949753895545541, 0.00016957499221161758],
                           [0.000929488323447122, 1.7207472887754074e-05],
                           [0.0007585937957494972, 4.6161018987638794e-05],
                           [0.0009805365210837482, 1.070957562415537e-05],
                           [0.0008136033003220738, 2.9416537342186925e-05],
                           [0.0007024612542802225, 2.1246089769292253e-05],
                           [0.0008133180894785864, 2.094639902420677e-05],
                           [0.0009106004429259544, 1.2477218265504069e-05],
                           [0.0008827061131435547, 8.411544933034531e-05],
                           [0.0007291780354019351, 2.251763336970168e-05],
                           [0.0009345639146532636, 2.5279167259611963e-05],
                           [0.0009524191471242692, 1.4419219132885989e-05],
                           [0.0007718999781590319, 1.5157688618864634e-05],
                           [0.0007716200338429244, 1.9186387637357206e-05],
                           [0.000903718774025823, 8.62185574569425e-06],
                           [0.0007726106927071618, 2.3026169806608247e-05]]
        else:
            raise Exception("Dataset '{}' not available.".format(set))

        return self._train_data()

    def limits(self):
        return [[5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3],
                [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3], [5.01e-3, 50e-3]]


class SurrogateBenchmarkLaser(SurrogateBenchmarkData):
    def __init__(self, n_test=5, n_train=None, verbose=1):
        super().__init__(n_test, n_train, verbose)

    def values(self, set="1850"):
        if set == "200":
            self.x_data = [
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2833336.6064214385, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1002.8735112225839, 1222.315873787476, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016459337889, 2.110682e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2533132.283573338, 6921050.360550507,
                 3461589.4238329353, 9.457999534029, 0.018048349898, 2.019091e-06],
                [1039.018276207625, 1145.028891106496, 1313.910403182829, 2933184.899608702, 5890961.479807982,
                 5730384.392981274, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1006.917429294439, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1119.645645441935, 1166.051027068192, 1410.400490357985, 2839590.7843750915, 3105552.0740890396,
                 4280687.504420381, 9.013062737532, 0.015566562652, 1.816343e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5793876.509322431, 12.013233977881999, 0.013809671326999999, 1.936649e-06],
                [1181.71642134566, 1106.1331358152229, 1300.392678923062, 3188724.243480558, 3611549.2338148607,
                 6040859.706667053, 11.608820336322, 0.015677646923999998, 2.337928e-06],
                [1086.4310962436139, 1143.313661827018, 1366.172986059074, 4433250.634960789, 3537428.9649966285,
                 4778094.621137584, 10.76567418015, 0.01753600924, 2.087574e-06],
                [1000.135095386044, 1105.4306348685059, 1330.571207498223, 4846321.381285451, 2701805.7086814954,
                 4484561.382528685, 10.352115864391, 0.014366939106, 1.835322e-06],
                [1154.407498959948, 1219.449161088396, 1351.174773511498, 4799249.275372577, 4503915.849723133,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1025.149660893128, 1274.573086050776, 1441.884859713586, 3756596.421046057, 6182273.026197797,
                 4498231.51928829, 11.583196813951, 0.016789544937, 2.281151e-06],
                [1065.247568510084, 1208.273246309111, 1365.658155840244, 3883964.276353225, 5756819.268304925,
                 5427939.298501339, 8.703442787445999, 0.016377523318, 2.36645e-06],
                [1113.362945387948, 1156.142543911252, 1318.7628524468, 4025728.5804803325, 6917121.327522967,
                 4324002.329611495, 8.410287456499, 0.013058706477, 1.860162e-06],
                [1144.62055680547, 1220.5901686895759, 1444.561092205831, 4253401.780289518, 6454896.318523467,
                 2945971.4495624257, 10.977852933097, 0.013716394277, 2.048221e-06],
                [1115.4642213739219, 1237.480378115084, 1399.160793550262, 3448300.45522647, 6948646.520626062,
                 6514459.910768375, 8.720091398814, 0.013200356943, 1.888512e-06],
                [1198.424856114746, 1296.995725535306, 1440.535501239776, 4037142.379113212, 6651700.51664973,
                 5937108.9775888715, 11.349016071859, 0.01691485517, 2.435554e-06],
                [1046.908391145076, 1110.0747175909369, 1429.941399224388, 5217719.52396076, 4029724.500115867,
                 4760881.312296346, 9.197976880151, 0.01708413309, 1.674435e-06],
                [1030.6783189917, 1173.084221673461, 1424.795222279961, 3227458.8804032593, 6221198.969017923,
                 5665900.526135622, 11.973154310207, 0.013935315602, 2.393006e-06],
                [1033.654551699753, 1201.128628532626, 1431.635745495852, 5001605.802510871, 3077964.2470758413,
                 5838982.254382255, 10.867178599420999, 0.017908623275, 2.002622e-06],
                [1157.156120746182, 1276.253071343592, 1382.704475395262, 4627392.441237274, 5111018.915760348,
                 5818605.271772643, 10.829110164955999, 0.016290290978999998, 1.648879e-06],
                [1013.999867239225, 1274.596607440347, 1328.137494820482, 6148326.188947396, 2880023.7632703707,
                 2716413.8509415397, 9.367603433475999, 0.017335893655, 1.7532879999999999e-06],
                [1183.3845823801569, 1103.53856891706, 1339.1973498816849, 5857130.671192093, 4830487.666190833,
                 3242830.1212443644, 11.73045420814, 0.013025100232999999, 2.041424e-06],
                [1187.13774787835, 1288.6560756130389, 1319.030401699622, 5120143.465953905, 3847769.4226577096,
                 4135483.2447417285, 12.07666433942, 0.014994717939, 2.339993e-06],
                [1008.737167229319, 1298.779343298153, 1401.16581949804, 4938520.93202682, 3669736.7410551626,
                 5506106.1604879005, 11.351561456138, 0.013820405011999999, 2.390425e-06],
                [1004.223595632002, 1105.781346904021, 1370.626336314514, 5514030.984856181, 6546703.664882269,
                 2517980.1310129967, 9.693377070051, 0.017067955495, 2.300258e-06],
                [1056.940680224295, 1171.0463258805419, 1421.258491743623, 6842973.116589421, 3002959.567692171,
                 2729190.8445358654, 8.92034402772, 0.014743255164, 1.993947e-06],
                [1005.092668316944, 1271.3524158120829, 1392.078489256121, 5166982.2406184245, 4268826.744940132,
                 6275433.024060723, 11.694209398863, 0.015654044555, 1.877544e-06],
                [1159.790454303531, 1294.810925782156, 1301.699194654326, 5531315.329960264, 2527259.94357461,
                 6233021.66283012, 9.304254074063, 0.014271952417, 2.110928e-06],
                [1079.637238641862, 1109.022579301891, 1371.99597925231, 6263782.484921139, 5348705.76036157,
                 3685271.6903267894, 8.457364096471, 0.015867152215999998, 1.7637589999999999e-06],
                [1063.999782361264, 1126.827411745328, 1435.651928395808, 6696749.080333681, 3327256.9565666015,
                 3819948.963790721, 10.957739291304, 0.015772118421, 1.998117e-06],
                [1128.898066684834, 1102.192643163407, 1382.4605764156959, 4115124.870206611, 6542036.732067945,
                 3988349.6840567538, 11.098528613383, 0.015783746379, 1.68582e-06],
                [1171.946382206198, 1250.465713604113, 1440.78402641335, 6267756.415151311, 2576120.882945207,
                 4298359.081961768, 10.643833324662, 0.013468011217, 2.175839e-06],
                [1191.342898668022, 1174.48878885235, 1446.59633611697, 2609878.0348966713, 6065752.76943621,
                 6274645.192937591, 11.345134400646, 0.01575507788, 2.44526e-06],
                [1081.971918344206, 1242.156732390483, 1421.184002985495, 5581516.932538617, 5506585.320418177,
                 4635067.7086390015, 11.895874262146, 0.014871805856999999, 2.060084e-06],
                [1099.912724382713, 1255.155437617339, 1407.960618786552, 5760511.412062648, 6957126.272013029,
                 3264769.430405523, 10.613935376776, 0.01855738889, 2.254727e-06],
                [1120.556841321501, 1151.844507270716, 1439.410715884273, 6948886.461354318, 3017680.9523862735,
                 4313526.957920614, 11.651913842295, 0.018208637796, 1.990718e-06],
                [1124.9533335742, 1186.166614541836, 1406.490558394534, 5104064.771557352, 6854157.806043102,
                 6792639.098163593, 9.324635019941, 0.017829849372, 2.3500399999999998e-06],
                [1060.306370914016, 1177.691586694892, 1356.64807625923, 6935598.916319065, 3041099.6477138954,
                 6442069.761881914, 8.822829285853, 0.015536299970999999, 1.731994e-06],
                [1192.40385198054, 1172.90338393294, 1398.198640963345, 5836571.517527281, 6658504.582887757,
                 3825984.626864104, 8.657999694988, 0.014541138015, 2.451221e-06],
                [1020.4033522193449, 1203.85747443386, 1323.879764465933, 6530553.91558025, 6469441.034533508,
                 3886526.433409815, 12.481427340769999, 0.016501171603, 2.424672e-06],
                [1066.242402098398, 1285.208512243742, 1341.080973157573, 6167119.824479279, 6559695.60009571,
                 3392404.364998077, 10.478922292791, 0.013199752915, 2.29902e-06],
                [1022.8133585878439, 1114.458955555204, 1441.513600695975, 6710705.024531956, 4886156.461802257,
                 6256204.592895824, 10.404065688781, 0.015312967018, 2.313376e-06],
                [1080.996695972347, 1286.585721993858, 1387.3638276057409, 6919538.349295578, 6599153.740642592,
                 3747963.890594435, 11.498552909894, 0.017503021367, 2.148159e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1056.2370978971562, 1228.0621474080876, 1331.3214771996602, 3627678.357077309, 2998055.7538120616,
                 2532740.0207836498, 9.697531154957, 0.01319801981388231, 2.340356e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2833336.6064214385, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1006.917429294439, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1002.3679828289813, 1233.8201449012863, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016812713242489365, 2.110682e-06],
                [1154.407498959948, 1236.655253905034, 1351.174773511498, 4487711.842441496, 2586063.573298348,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1017.0400643510857, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1002.8735112225839, 1222.315873787476, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016459337889, 2.110682e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3092961.8274809974,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1011.1651636778292, 1219.449161088396, 1351.174773511498, 4531951.9996198295, 3494078.511359096,
                 2756512.928705282, 11.845560061499354, 0.013221990874991763, 1.716072e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2533132.283573338, 6921050.360550507,
                 3461589.4238329353, 9.457999534029, 0.018048349898, 2.019091e-06],
                [1128.0511659807232, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.1650025557814735e-06],
                [1103.683044712124, 1117.8172959384456, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5785882.1544099115, 12.013233977881999, 0.013809671326999999, 1.912446838761154e-06],
                [1119.645645441935, 1166.051027068192, 1410.400490357985, 2839590.7843750915, 3105552.0740890396,
                 4280687.504420381, 9.013062737532, 0.015566562652, 1.816343e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5793876.509322431, 12.013233977881999, 0.013809671326999999, 1.936649e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3625963.6402100134, 3381410.7237955173,
                 5793876.509322431, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1039.018276207625, 1145.028891106496, 1313.910403182829, 2933184.899608702, 5890961.479807982,
                 5730384.392981274, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1039.018276207625, 1153.8582415325982, 1313.910403182829, 2933184.899608702, 2850090.527418211,
                 2951607.045186893, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1181.71642134566, 1106.1331358152229, 1300.392678923062, 3188724.243480558, 3611549.2338148607,
                 6040859.706667053, 11.608820336322, 0.015677646923999998, 2.337928e-06],
                [1113.362945387948, 1156.142543911252, 1318.7628524468, 4025728.5804803325, 6917121.327522967,
                 4324002.329611495, 8.410287456499, 0.013058706477, 1.860162e-06],
                [1000.135095386044, 1105.4306348685059, 1330.571207498223, 4846321.381285451, 2701805.7086814954,
                 4484561.382528685, 10.352115864391, 0.014366939106, 1.835322e-06],
                [1086.4310962436139, 1143.313661827018, 1366.172986059074, 4433250.634960789, 3537428.9649966285,
                 4778094.621137584, 10.76567418015, 0.01753600924, 2.087574e-06],
                [1170.0160238258507, 1219.449161088396, 1351.174773511498, 4892238.0689851735, 3553816.363672882,
                 2825500.380147861, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1154.407498959948, 1219.449161088396, 1351.174773511498, 4799249.275372577, 4503915.849723133,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1144.62055680547, 1220.5901686895759, 1444.561092205831, 4195862.806659156, 6454896.318523467,
                 2945971.4495624257, 10.977852933097, 0.014840773008576945, 2.048221e-06],
                [1084.1351758402184, 1241.226049743086, 1421.184002985495, 4028227.4624709627, 5506585.320418177,
                 4635067.7086390015, 11.026913378266935, 0.014871805856999999, 2.002752315241874e-06],
                [1048.5536313074438, 1280.282124741661, 1447.05386618055, 3756596.421046057, 6182273.026197797,
                 4498231.51928829, 11.583196813951, 0.016789544937, 2.281151e-06],
                [1144.62055680547, 1220.5901686895759, 1444.561092205831, 4253401.780289518, 6431058.822726591,
                 2945971.4495624257, 10.977852933097, 0.013716394277, 1.974682253310644e-06],
                [1115.4642213739219, 1237.480378115084, 1399.160793550262, 3448300.45522647, 6948646.520626062,
                 6514459.910768375, 8.720091398814, 0.013200356943, 1.888512e-06],
                [1144.62055680547, 1220.5901686895759, 1444.561092205831, 4253401.780289518, 6454896.318523467,
                 2945971.4495624257, 10.977852933097, 0.013716394277, 2.048221e-06],
                [1038.2632837344158, 1274.573086050776, 1441.884859713586, 3756596.421046057, 6182273.026197797,
                 4498231.51928829, 11.583196813951, 0.016789544937, 2.281151e-06],
                [1025.149660893128, 1274.573086050776, 1441.884859713586, 3756596.421046057, 6182273.026197797,
                 4498231.51928829, 11.583196813951, 0.016789544937, 2.281151e-06],
                [1065.247568510084, 1208.273246309111, 1365.658155840244, 3883964.276353225, 5756819.268304925,
                 5427939.298501339, 8.703442787445999, 0.016377523318, 2.36645e-06],
                [1183.3845823801569, 1103.53856891706, 1339.1973498816849, 4009970.812727008, 4766329.348673246,
                 3242830.1212443644, 11.73045420814, 0.013025100232999999, 2.041424e-06],
                [1113.362945387948, 1275.0473319678974, 1317.0402368068549, 4025728.5804803325, 6917121.327522967,
                 6244999.228967555, 8.410287456499, 0.012965542767530727, 1.8588072182133293e-06],
                [1075.8304613644482, 1291.3389576897396, 1382.704475395262, 4627392.441237274, 5111018.915760348,
                 3799434.3905045544, 10.829110164955999, 0.016290290978999998, 1.6468622167278855e-06],
                [1030.6783189917, 1173.084221673461, 1424.795222279961, 3227458.8804032593, 6221198.969017923,
                 5665900.526135622, 11.973154310207, 0.013935315602, 2.393006e-06],
                [1198.424856114746, 1296.995725535306, 1440.535501239776, 4037142.379113212, 6651700.51664973,
                 5937108.9775888715, 11.349016071859, 0.01691485517, 2.435554e-06],
                [1154.407498959948, 1295.8346354373668, 1351.174773511498, 4799249.275372577, 4503915.849723133,
                 2756512.928705282, 11.43677145574, 0.014938196750102613, 2.3286824690076844e-06],
                [1183.3845823801569, 1103.53856891706, 1339.1973498816849, 5857130.671192093, 4830487.666190833,
                 3242830.1212443644, 11.73045420814, 0.013025100232999999, 2.041424e-06],
                [1046.908391145076, 1110.0747175909369, 1429.941399224388, 5217719.52396076, 4029724.500115867,
                 4760881.312296346, 9.197976880151, 0.01708413309, 1.674435e-06],
                [1144.62055680547, 1220.5901686895759, 1442.215360592326, 4389167.385786755, 6407687.243641006,
                 4342385.035752849, 10.977852933097, 0.014303694536669386, 2.1339146227475467e-06],
                [1009.1146043961726, 1298.779343298153, 1401.16581949804, 4902261.4984828215, 3669736.7410551626,
                 5506106.1604879005, 11.351561456138, 0.013820405011999999, 1.9265682162822142e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1181.71642134566, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 6040859.706667053, 11.216198135126099, 0.015677646923999998, 2.337928e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 5911610.327873674,
                 4700731.238906128, 9.427550057683757, 0.015939573082178627, 1.8770206541805113e-06],
                [1056.2370978971562, 1228.0621474080876, 1331.3214771996602, 3627678.357077309, 2998055.7538120616,
                 2532740.0207836498, 9.697531154957, 0.01319801981388231, 2.340356e-06],
                [1112.3524010915671, 1257.584793961963, 1310.427373334314, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.41622197839763, 0.016637333018644673, 1.6652576526071669e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1138.8994908064808, 1264.2034079590524, 1326.3993608570645, 3614288.9880884974, 3448028.61050569,
                 2613897.3925167606, 10.676487893570695, 0.013473647569965668, 2.369752188687355e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2833336.6064214385, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1006.917429294439, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1006.7804880189733, 1107.897956860713, 1436.8036091630076, 3991524.1814932027, 2715483.6685275612,
                 5499059.652247055, 10.200095357793, 0.013625707946, 1.8527773810483195e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2522732.7973377593, 6921050.360550507,
                 3461589.4238329353, 11.919213951914514, 0.018048349898, 2.019091e-06],
                [1002.3679828289813, 1233.8201449012863, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016812713242489365, 2.110682e-06],
                [1154.407498959948, 1236.655253905034, 1351.174773511498, 4487711.842441496, 2586063.573298348,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1017.0400643510857, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1002.8735112225839, 1222.315873787476, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016459337889, 2.110682e-06],
                [1089.956830650977, 1114.607364489448, 1419.872158616546, 3617780.108636307, 3381410.7237955173,
                 5793876.509322431, 9.551867507206268, 0.013809671326999999, 1.937855678932045e-06],
                [1148.9327261066758, 1153.6304141964065, 1304.551838382366, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.76748429421807, 0.018402202852, 1.7815010318259866e-06],
                [1098.218987361545, 1228.2800522789857, 1362.310913050378, 3949022.787889535, 2765020.178343959,
                 5685500.530032963, 9.477858635073746, 0.017868050845, 2.167855466685067e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3092961.8274809974,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1011.1651636778292, 1219.449161088396, 1351.174773511498, 4531951.9996198295, 3494078.511359096,
                 2756512.928705282, 11.845560061499354, 0.013221990874991763, 1.716072e-06],
                [1011.1651636778292, 1219.449161088396, 1351.174773511498, 4531951.9996198295, 3494078.511359096,
                 2756512.928705282, 11.845560061499354, 0.013705077041655573, 1.718482291015711e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2500575.673267536, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2533132.283573338, 6921050.360550507,
                 3461589.4238329353, 9.457999534029, 0.018048349898, 2.019091e-06],
                [1154.5397873167135, 1264.202938587985, 1362.310913050378, 3921988.0613203924, 3092961.8274809974,
                 5772927.094828865, 10.634894158475426, 0.017868050845, 2.165278e-06],
                [1119.645645441935, 1166.051027068192, 1410.400490357985, 2839590.7843750915, 3105552.0740890396,
                 4280687.504420381, 9.013062737532, 0.015566562652, 1.816343e-06],
                [1103.683044712124, 1117.8172959384456, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5785882.1544099115, 12.013233977881999, 0.013809671326999999, 1.912446838761154e-06],
                [1128.0511659807232, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.1650025557814735e-06],
                [1170.0160238258507, 1219.449161088396, 1351.174773511498, 4565079.831118726, 3553816.363672882,
                 2822066.382887337, 11.43677145574, 0.013222653340200709, 1.7776952010619029e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5793876.509322431, 12.013233977881999, 0.013809671326999999, 1.936649e-06],
                [1128.0511659807232, 1257.4239927153367, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.1650025557814735e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3625963.6402100134, 3381410.7237955173,
                 5793876.509322431, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1039.018276207625, 1145.028891106496, 1313.910403182829, 2933184.899608702, 5890961.479807982,
                 5730384.392981274, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1143.4387951608267, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5920157.272578891, 10.713570074941, 0.018024370678357375, 2.1650025557814735e-06],
                [1104.4239043486102, 1114.607364489448, 1419.872158616546, 3625963.6402100134, 3381410.7237955173,
                 5793876.509322431, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1039.018276207625, 1153.8582415325982, 1313.910403182829, 2933184.899608702, 2850090.527418211,
                 2951607.045186893, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1183.4996967827772, 1219.449161088396, 1351.174773511498, 4892238.0689851735, 3021703.552791283,
                 2825500.380147861, 11.266590332781, 0.013341161615, 1.716072e-06],
                [1025.56871224164, 1109.5634116524457, 1306.648311432572, 4065276.9287967673, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1181.71642134566, 1106.1331358152229, 1300.392678923062, 3188724.243480558, 3611549.2338148607,
                 6040859.706667053, 11.608820336322, 0.015677646923999998, 2.337928e-06],
                [1000.135095386044, 1105.4306348685059, 1330.5493242905914, 4713546.073109317, 2662353.0558696883,
                 4484561.382528685, 10.334296325294885, 0.014366939106, 2.0449849189212174e-06],
                [1000.6018390865513, 1100.7447212558154, 1381.4951036145428, 4846321.381285451, 2701805.7086814954,
                 4484561.382528685, 10.352115864391, 0.014327089983809603, 1.9053230239915903e-06],
                [1039.018276207625, 1122.5444686286983, 1326.5430142598552, 2791094.7810776485, 2697731.1452343613,
                 3158915.6229445105, 11.255935596768929, 0.015413925091, 1.7812039681740136e-06],
                [1113.362945387948, 1156.142543911252, 1318.7628524468, 4025728.5804803325, 6917121.327522967,
                 4324002.329611495, 8.410287456499, 0.013058706477, 1.860162e-06],
                [1170.0160238258507, 1219.449161088396, 1351.174773511498, 4892238.0689851735, 3553816.363672882,
                 2825500.380147861, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2727002.312869729, 5177667.749296468,
                 6055735.99685296, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1181.71642134566, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 6040859.706667053, 11.216198135126099, 0.015677646923999998, 2.337928e-06],
                [1056.2370978971562, 1228.0621474080876, 1331.3214771996602, 3627678.357077309, 2998055.7538120616,
                 2532740.0207836498, 9.697531154957, 0.01319801981388231, 2.340356e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 2562820.6700545694,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1064.317474852996, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.562097953987161, 0.012851621732462407, 2.065492e-06],
                [1115.4440971511028, 1257.584793961963, 1335.811778824276, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.41622197839763, 0.01810061756308735, 1.6652576526071669e-06],
                [1121.556326096255, 1225.477552275013, 1419.57153976711, 2824159.735864557, 4668788.44424598,
                 4235528.204404335, 11.244808146052, 0.01805712000324544, 2.432848e-06],
                [1134.3052340247398, 1266.4455156803888, 1430.258879312566, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1003.6429065603127, 1107.897956860713, 1436.8036091630076, 3773196.911885251, 2715483.6685275612,
                 5499059.652247055, 10.200095357793, 0.013625707946, 1.8527773810483195e-06],
                [1112.3524010915671, 1257.584793961963, 1310.427373334314, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.41622197839763, 0.016637333018644673, 1.6652576526071669e-06],
                [1128.0891752962004, 1257.584793961963, 1310.427373334314, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.399765828035274, 0.017857864436581865, 1.6872256181843123e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 5911610.327873674,
                 4700731.238906128, 9.427550057683757, 0.015939573082178627, 1.8770206541805113e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 11.198032372030648, 0.018402202852, 1.7812109999999999e-06],
                [1006.917429294439, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1138.8994908064808, 1264.2034079590524, 1326.3993608570645, 3614288.9880884974, 3448028.61050569,
                 2613897.3925167606, 10.676487893570695, 0.013473647569965668, 2.369752188687355e-06],
                [1089.945804052426, 1238.8150696891787, 1419.872158616546, 3617780.108636307, 3381410.7237955173,
                 5793876.509322431, 9.940957317476279, 0.013890054540421578, 1.937855678932045e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2833336.6064214385, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1135.680586055101, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4087908.14487714, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1017.0400643510857, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1098.218987361545, 1228.3923530007805, 1362.8611286943965, 3750480.6814249475, 2857960.75831493,
                 2974008.506144651, 9.477858635073746, 0.013821536692075082, 1.7741876259115089e-06],
                [1154.407498959948, 1236.655253905034, 1351.174773511498, 4427474.235537298, 2542437.981751782,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1006.7804880189733, 1107.897956860713, 1436.8036091630076, 3991524.1814932027, 2715483.6685275612,
                 5499059.652247055, 10.200095357793, 0.013625707946, 1.8527773810483195e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2522732.7973377593, 6921050.360550507,
                 3461589.4238329353, 11.919213951914514, 0.018048349898, 2.019091e-06],
                [1017.0400643510857, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1002.8735112225839, 1222.315873787476, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016459337889, 2.110682e-06],
                [1015.7284995164233, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5607520.642758014, 10.386857748765365, 0.013625707946, 2.3722169999999998e-06],
                [1002.3679828289813, 1233.8201449012863, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016812713242489365, 2.110682e-06],
                [1147.501727915812, 1114.1127163538993, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.01837329413225449, 1.7812109999999999e-06],
                [1154.407498959948, 1236.655253905034, 1351.174773511498, 4487711.842441496, 2586063.573298348,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1157.058011260272, 1240.3487101897, 1307.5075760627185, 2522732.7973377593, 6921050.360550507,
                 3461589.4238329353, 11.919213951914514, 0.016585063159245916, 2.019091e-06],
                [1135.736010842282, 1117.8172959384456, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5785882.1544099115, 12.013233977881999, 0.013809671326999999, 1.8866730357254492e-06],
                [1154.407498959948, 1236.655253905034, 1342.9555486456536, 4487711.842441496, 2586063.573298348,
                 2756512.928705282, 11.43677145574, 0.013355360412473294, 1.716072e-06],
                [1157.058011260272, 1240.3487101897, 1325.7833258513153, 2526832.2962593758, 6921050.360550507,
                 3555721.1998439874, 10.398248441320256, 0.018048349898, 2.019091e-06],
                [1089.956830650977, 1114.607364489448, 1419.872158616546, 3617780.108636307, 3381410.7237955173,
                 5793876.509322431, 9.551867507206268, 0.013809671326999999, 1.937855678932045e-06],
                [1148.9327261066758, 1153.6304141964065, 1304.551838382366, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.76748429421807, 0.018402202852, 1.7815010318259866e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2624686.8443433163, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.01801296639855309, 2.4490968385460116e-06],
                [1098.218987361545, 1228.2800522789857, 1362.310913050378, 3949022.787889535, 2765020.178343959,
                 5685500.530032963, 9.477858635073746, 0.017868050845, 2.167855466685067e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06]]
            self.y_data = [[2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [84.39742275948912, 33.78314907324829, 265.4014460620532],
                           [932.6069641175573, 400.59055218545626, 749.2704358802002],
                           [1219.4217619547032, 483.40873541839017, 827.8547810464862],
                           [590.3969996969831, 361.1751758113917, 897.3239825360729],
                           [28.069171949228203, 1003.1312033418135, 1027.3516684209444],
                           [3293.6417749350526, 4000.296651387464, 2749.9811204802127],
                           [2689.888797413033, 1995.423712500253, 2913.153419949472],
                           [3641.408349259386, 4149.027032556554, 2979.8648961738068],
                           [64.95870380673843, 2037.0352581288441, 3142.3452838768544],
                           [5050.042406848, 3353.631362261292, 4046.1731460186083],
                           [3211.138664604769, 5346.419128787817, 4918.9261583296375],
                           [4258.619318468715, 5989.02266715898, 5220.525994374762],
                           [8936.189342070667, 9965.239758350204, 7563.000813830158],
                           [6021.767634834487, 8903.495264146026, 7814.127457858662],
                           [6017.482913124648, 9878.662058584368, 9219.928973267353],
                           [10383.663899872603, 11692.453275702828, 9778.423392895807],
                           [9959.714499578859, 12100.473650714735, 10815.113656904918],
                           [4728.725795408572, 9833.320115679087, 11373.539981017966],
                           [7040.3231430446585, 11659.214148088358, 11790.152463711856],
                           [6625.1594545319385, 10968.926234932243, 12625.282931427295],
                           [15642.10277555957, 16893.988266075045, 14863.564593947294],
                           [18607.931674621763, 19082.778723217332, 15013.161810006111],
                           [10626.98560266889, 14816.17950387497, 15506.383314966086],
                           [19339.61783872236, 19682.730138362174, 15674.580866406783],
                           [22189.138934738672, 21973.758051439312, 17867.900673260734],
                           [23690.59769921251, 25783.93771324694, 20809.23996609462],
                           [14967.7196505778, 22248.082939376614, 22731.8285500197],
                           [23423.407106244136, 28591.62813777676, 26664.35093389845],
                           [19075.645612367698, 27207.437259519545, 28128.538674419888],
                           [29997.930337349753, 33405.527835033296, 28873.62124640369],
                           [24820.89628524226, 33586.311728094224, 31719.51145267233],
                           [37282.62405854246, 38046.09208233556, 33387.387843928715],
                           [28410.627000789853, 35686.40139112738, 36220.30055895746],
                           [40471.242966725076, 43007.140870292154, 37868.621210244215],
                           [37398.14699282434, 42565.10811820573, 37870.35351039514],
                           [45479.57029289774, 47732.36757601205, 41595.563139189384],
                           [32552.29129704946, 44403.59499708624, 45751.504789978455],
                           [45786.72576366185, 50791.11058567032, 48223.58538251199],
                           [51020.88136265659, 54357.122848508094, 50465.40361704311],
                           [61881.73733921197, 59644.849846935205, 51400.45602940709],
                           [60457.47777367473, 60647.47618547739, 51432.39988001411],
                           [62646.79677225086, 58733.17045634589, 53097.700319228185],
                           [71247.0421208742, 72098.19782720233, 67199.0151309894],
                           [63957.85300621107, 77501.03448860586, 81158.49902713434],
                           [93260.27723239115, 94410.9579982488, 86854.8936256963],
                           [69247.51335865381, 84142.67184010833, 89262.52599409055],
                           [96023.63022667098, 99075.37552357498, 97413.0256048264],
                           [115089.56740480401, 109259.95186664716, 99354.46311566826],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [23.975982777695172, 6.757366123987301, 191.0233853494245],
                           [84.39742275948912, 33.78314907324829, 265.4014460620532],
                           [28.069171949228203, 1003.1312033418135, 1027.3516684209444],
                           [3293.5437591554783, 130.18670022608515, 25.37306097654482],
                           [932.6069641175573, 400.59055218545626, 749.2704358802002],
                           [64.95870380673843, 2037.0352581288441, 3142.3452838768544],
                           [590.3969996969831, 361.1751758113917, 897.3239825360729],
                           [1195.5075793930594, 526.1504936826086, 932.2775739448363],
                           [615.7980966174894, 2371.6328956458747, 2197.19994390353],
                           [84.94494447429759, 2139.8419082286405, 3265.87568834472],
                           [1219.4217619547032, 483.40873541839017, 827.8547810464862],
                           [2724.5059366381524, 3420.7700687544716, 2292.325941125391],
                           [2144.00329464219, 4811.868522654193, 4324.293371510733],
                           [2689.888797413033, 1995.423712500253, 2913.153419949472],
                           [3227.723554235312, 3931.1954332489513, 2694.6979836887685],
                           [3102.5415684431687, 5156.512214548655, 4696.911509918105],
                           [5050.042406848, 3353.631362261292, 4046.1731460186083],
                           [3211.138664604769, 5346.419128787817, 4918.9261583296375],
                           [3293.6417749350526, 4000.296651387464, 2749.9811204802127],
                           [3636.4519255889963, 7701.036499127755, 8649.018536861058],
                           [3641.408349259386, 4149.027032556554, 2979.8648961738068],
                           [7094.934117160581, 4963.715116596285, 5681.255547927741],
                           [4258.619318468715, 5989.02266715898, 5220.525994374762],
                           [4728.725795408572, 9833.320115679087, 11373.539981017966],
                           [6021.767634834487, 8903.495264146026, 7814.127457858662],
                           [8936.189342070667, 9965.239758350204, 7563.000813830158],
                           [5108.537042399923, 8846.416734250804, 8264.988876899557],
                           [6017.482913124648, 9878.662058584368, 9219.928973267353],
                           [6749.343731353359, 9786.717933043677, 8863.214695950994],
                           [8699.036586466838, 11186.733448158204, 9991.249310163385],
                           [9685.64062202062, 10986.969888115693, 9156.021071890882],
                           [6854.662907936784, 11100.471712774037, 10984.746603997097],
                           [6625.1594545319385, 10968.926234932243, 12625.282931427295],
                           [7040.3231430446585, 11659.214148088358, 11790.152463711856],
                           [9955.857397038104, 11269.85863024913, 9404.157063167368],
                           [10383.663899872603, 11692.453275702828, 9778.423392895807],
                           [9959.714499578859, 12100.473650714735, 10815.113656904918],
                           [7855.450025109316, 13741.228884635286, 13987.2032403042],
                           [9539.580737744978, 15726.640872161857, 18121.942676224597],
                           [15272.943154076545, 16049.89530306775, 12544.873510185207],
                           [10626.98560266889, 14816.17950387497, 15506.383314966086],
                           [15642.10277555957, 16893.988266075045, 14863.564593947294],
                           [11972.885691299613, 16740.618540004456, 15604.373301592324],
                           [14967.7196505778, 22248.082939376614, 22731.8285500197],
                           [18607.931674621763, 19082.778723217332, 15013.161810006111],
                           [15600.764007884087, 20229.29363534731, 19865.958258264185],
                           [17299.3462527832, 22185.51377411193, 20908.129208334474],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [23.975982777695172, 6.757366123987301, 191.0233853494245],
                           [84.39742275948912, 33.78314907324829, 265.4014460620532],
                           [0.8807537900138819, 308.8706699734264, 244.54305628933315],
                           [348.7682075032219, 79.03123825619325, 251.45129387305434],
                           [3293.5437591554783, 130.18670022608515, 25.37306097654482],
                           [222.55731422260303, 146.9065009953099, 554.3482836092844],
                           [28.069171949228203, 1003.1312033418135, 1027.3516684209444],
                           [875.379542803012, 99.28324447428258, 466.83123685949204],
                           [932.6069641175573, 400.59055218545626, 749.2704358802002],
                           [64.95870380673843, 2037.0352581288441, 3142.3452838768544],
                           [590.3969996969831, 361.1751758113917, 897.3239825360729],
                           [419.4517590701555, 1833.403663275858, 1907.3039740444506],
                           [666.5041189662938, 369.17520959179217, 852.6852888192585],
                           [1195.5075793930594, 526.1504936826086, 932.2775739448363],
                           [615.7980966174894, 2371.6328956458747, 2197.19994390353],
                           [84.94494447429759, 2139.8419082286405, 3265.87568834472],
                           [1219.4217619547032, 483.40873541839017, 827.8547810464862],
                           [806.4250159418375, 2573.897296351246, 2788.011669077916],
                           [2189.475081270695, 1751.4863945013165, 2779.906539602099],
                           [1511.207677762512, 2088.7074393216253, 1254.3211433445492],
                           [2724.5059366381524, 3420.7700687544716, 2292.325941125391],
                           [2144.00329464219, 4811.868522654193, 4324.293371510733],
                           [2265.587989319462, 4618.719697500087, 3911.8825835162193],
                           [3337.817233042934, 2204.27579141391, 2869.87751326041],
                           [2689.888797413033, 1995.423712500253, 2913.153419949472],
                           [2850.780079119669, 3539.006290856695, 2384.2371130748134],
                           [5050.042406848, 3353.631362261292, 4046.1731460186083],
                           [3102.5415684431687, 5156.512214548655, 4696.911509918105],
                           [3227.723554235312, 3931.1954332489513, 2694.6979836887685],
                           [3155.6670736396673, 6458.660214702686, 6019.507672510742],
                           [3293.6417749350526, 4000.296651387464, 2749.9811204802127],
                           [3211.138664604769, 5346.419128787817, 4918.9261583296375],
                           [3333.657830110825, 4036.580611849707, 2778.6004285400895],
                           [3636.4519255889963, 7701.036499127755, 8649.018536861058],
                           [3641.408349259386, 4149.027032556554, 2979.8648961738068],
                           [3881.6456447322603, 4524.469986396184, 3141.099951069475],
                           [3641.863977178802, 7708.587433808557, 8656.844594135235],
                           [7094.934117160581, 4963.715116596285, 5681.255547927741],
                           [4028.4909074635952, 7529.888366003115, 7053.717520267663],
                           [3904.834571608711, 8901.031785138372, 10520.56571402266],
                           [4258.619318468715, 5989.02266715898, 5220.525994374762],
                           [4714.951634369974, 7990.049683599002, 7473.374103872339],
                           [4896.420563119286, 7895.896204411419, 7067.661070062295],
                           [8830.145285436984, 6453.843953175076, 7234.323435982808],
                           [4728.725795408572, 9833.320115679087, 11373.539981017966],
                           [5108.537042399923, 8846.416734250804, 8264.988876899557],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [1698.424036289286, 0.01299795111754957, 168.37538497510388],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [10.666841821962958, 0.10829104953011984, 120.397148384552],
                           [1725.6082432411808, 14.39254270781067, 32.963287653975904],
                           [24.29419694458095, 6.387867606153993, 188.9083418160392],
                           [0.8807537900138819, 308.8706699734264, 244.54305628933315],
                           [3293.5437591554783, 130.18670022608515, 25.37306097654482],
                           [23.975982777690717, 6.757366123987301, 191.0233853494245],
                           [23.975982777695172, 6.757366123987301, 191.0233853494245],
                           [3614.1945490081284, 403.56408090706645, 64.64230859699205],
                           [84.39742275948912, 33.78314907324829, 265.4014460620532],
                           [49.500236767375256, 748.8184597207388, 1194.8425780038576],
                           [59.29231378399418, 73.37614553766984, 507.52095741347256],
                           [273.34395127563, 47.55919777853988, 219.62646505048573],
                           [231.49253129291893, 59.0836955422837, 264.88592492996935],
                           [28.066275586976047, 763.7119287944752, 825.5639218708797],
                           [222.55731422260303, 146.9065009953099, 554.3482836092844],
                           [121.64342076581877, 120.4858584700321, 595.5160139781838],
                           [348.7682075032219, 79.03123825619325, 251.45129387305434],
                           [28.069171949228203, 1003.1312033418135, 1027.3516684209444],
                           [451.4713618411107, 250.38998334942517, 710.4584995815518],
                           [64.95870380673843, 2037.0352581288441, 3142.3452838768544],
                           [875.379542803012, 99.28324447428258, 466.83123685949204],
                           [103.35966736125393, 1054.0901300295363, 1140.2307760836932],
                           [932.6069641175573, 400.59055218545626, 749.2704358802002],
                           [573.8775186027166, 348.50456444541703, 878.1386230306048],
                           [84.94494447429759, 2139.8419082285986, 3265.87568834472],
                           [2215.8745136846023, 497.1308621557844, 469.140767742917],
                           [372.330578765052, 1872.1123080729162, 1732.8198568681098],
                           [419.4517590701555, 1833.403663275858, 1907.3039740444506],
                           [666.5041189662938, 369.17520959179217, 852.6852888192585],
                           [84.94494447429759, 2139.8419082286405, 3265.87568834472],
                           [590.3969996969831, 361.1751758113917, 897.3239825360729],
                           [1219.4217619547032, 483.40873541839017, 827.8547810464862],
                           [156.00522571860756, 2373.563916579705, 3498.689531915568],
                           [1195.5075793930594, 526.1504936826086, 932.2775739448363],
                           [645.5498562605851, 401.99901838595474, 957.277825073818],
                           [615.7980966174894, 2371.6328956458747, 2197.19994390353],
                           [1272.8274396706872, 656.4073205730202, 1099.5969877098196],
                           [1049.756553208089, 2433.9776637581167, 2272.2434364372602],
                           [686.8351067732438, 2487.4607641016514, 2300.204812129034],
                           [1253.8496497005235, 844.2912509695493, 1537.8776451832111],
                           [806.4250159418375, 2573.897296351246, 2788.011669077916],
                           [2189.475081270695, 1751.4863945013165, 2779.906539602099],
                           [2287.613789100325, 1348.6367817905837, 1878.2952537142526],
                           [1511.207677762512, 2088.7074393216253, 1254.3211433445492],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068]]
        elif set == "1850":
            self.x_data = [
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2833336.6064214385, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1002.8735112225839, 1222.315873787476, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016459337889, 2.110682e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2533132.283573338, 6921050.360550507,
                 3461589.4238329353, 9.457999534029, 0.018048349898, 2.019091e-06],
                [1039.018276207625, 1145.028891106496, 1313.910403182829, 2933184.899608702, 5890961.479807982,
                 5730384.392981274, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1006.917429294439, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1119.645645441935, 1166.051027068192, 1410.400490357985, 2839590.7843750915, 3105552.0740890396,
                 4280687.504420381, 9.013062737532, 0.015566562652, 1.816343e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5793876.509322431, 12.013233977881999, 0.013809671326999999, 1.936649e-06],
                [1181.71642134566, 1106.1331358152229, 1300.392678923062, 3188724.243480558, 3611549.2338148607,
                 6040859.706667053, 11.608820336322, 0.015677646923999998, 2.337928e-06],
                [1086.4310962436139, 1143.313661827018, 1366.172986059074, 4433250.634960789, 3537428.9649966285,
                 4778094.621137584, 10.76567418015, 0.01753600924, 2.087574e-06],
                [1000.135095386044, 1105.4306348685059, 1330.571207498223, 4846321.381285451, 2701805.7086814954,
                 4484561.382528685, 10.352115864391, 0.014366939106, 1.835322e-06],
                [1154.407498959948, 1219.449161088396, 1351.174773511498, 4799249.275372577, 4503915.849723133,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1025.149660893128, 1274.573086050776, 1441.884859713586, 3756596.421046057, 6182273.026197797,
                 4498231.51928829, 11.583196813951, 0.016789544937, 2.281151e-06],
                [1065.247568510084, 1208.273246309111, 1365.658155840244, 3883964.276353225, 5756819.268304925,
                 5427939.298501339, 8.703442787445999, 0.016377523318, 2.36645e-06],
                [1113.362945387948, 1156.142543911252, 1318.7628524468, 4025728.5804803325, 6917121.327522967,
                 4324002.329611495, 8.410287456499, 0.013058706477, 1.860162e-06],
                [1144.62055680547, 1220.5901686895759, 1444.561092205831, 4253401.780289518, 6454896.318523467,
                 2945971.4495624257, 10.977852933097, 0.013716394277, 2.048221e-06],
                [1115.4642213739219, 1237.480378115084, 1399.160793550262, 3448300.45522647, 6948646.520626062,
                 6514459.910768375, 8.720091398814, 0.013200356943, 1.888512e-06],
                [1198.424856114746, 1296.995725535306, 1440.535501239776, 4037142.379113212, 6651700.51664973,
                 5937108.9775888715, 11.349016071859, 0.01691485517, 2.435554e-06],
                [1046.908391145076, 1110.0747175909369, 1429.941399224388, 5217719.52396076, 4029724.500115867,
                 4760881.312296346, 9.197976880151, 0.01708413309, 1.674435e-06],
                [1030.6783189917, 1173.084221673461, 1424.795222279961, 3227458.8804032593, 6221198.969017923,
                 5665900.526135622, 11.973154310207, 0.013935315602, 2.393006e-06],
                [1033.654551699753, 1201.128628532626, 1431.635745495852, 5001605.802510871, 3077964.2470758413,
                 5838982.254382255, 10.867178599420999, 0.017908623275, 2.002622e-06],
                [1157.156120746182, 1276.253071343592, 1382.704475395262, 4627392.441237274, 5111018.915760348,
                 5818605.271772643, 10.829110164955999, 0.016290290978999998, 1.648879e-06],
                [1013.999867239225, 1274.596607440347, 1328.137494820482, 6148326.188947396, 2880023.7632703707,
                 2716413.8509415397, 9.367603433475999, 0.017335893655, 1.7532879999999999e-06],
                [1183.3845823801569, 1103.53856891706, 1339.1973498816849, 5857130.671192093, 4830487.666190833,
                 3242830.1212443644, 11.73045420814, 0.013025100232999999, 2.041424e-06],
                [1187.13774787835, 1288.6560756130389, 1319.030401699622, 5120143.465953905, 3847769.4226577096,
                 4135483.2447417285, 12.07666433942, 0.014994717939, 2.339993e-06],
                [1008.737167229319, 1298.779343298153, 1401.16581949804, 4938520.93202682, 3669736.7410551626,
                 5506106.1604879005, 11.351561456138, 0.013820405011999999, 2.390425e-06],
                [1004.223595632002, 1105.781346904021, 1370.626336314514, 5514030.984856181, 6546703.664882269,
                 2517980.1310129967, 9.693377070051, 0.017067955495, 2.300258e-06],
                [1056.940680224295, 1171.0463258805419, 1421.258491743623, 6842973.116589421, 3002959.567692171,
                 2729190.8445358654, 8.92034402772, 0.014743255164, 1.993947e-06],
                [1005.092668316944, 1271.3524158120829, 1392.078489256121, 5166982.2406184245, 4268826.744940132,
                 6275433.024060723, 11.694209398863, 0.015654044555, 1.877544e-06],
                [1159.790454303531, 1294.810925782156, 1301.699194654326, 5531315.329960264, 2527259.94357461,
                 6233021.66283012, 9.304254074063, 0.014271952417, 2.110928e-06],
                [1079.637238641862, 1109.022579301891, 1371.99597925231, 6263782.484921139, 5348705.76036157,
                 3685271.6903267894, 8.457364096471, 0.015867152215999998, 1.7637589999999999e-06],
                [1063.999782361264, 1126.827411745328, 1435.651928395808, 6696749.080333681, 3327256.9565666015,
                 3819948.963790721, 10.957739291304, 0.015772118421, 1.998117e-06],
                [1128.898066684834, 1102.192643163407, 1382.4605764156959, 4115124.870206611, 6542036.732067945,
                 3988349.6840567538, 11.098528613383, 0.015783746379, 1.68582e-06],
                [1171.946382206198, 1250.465713604113, 1440.78402641335, 6267756.415151311, 2576120.882945207,
                 4298359.081961768, 10.643833324662, 0.013468011217, 2.175839e-06],
                [1191.342898668022, 1174.48878885235, 1446.59633611697, 2609878.0348966713, 6065752.76943621,
                 6274645.192937591, 11.345134400646, 0.01575507788, 2.44526e-06],
                [1081.971918344206, 1242.156732390483, 1421.184002985495, 5581516.932538617, 5506585.320418177,
                 4635067.7086390015, 11.895874262146, 0.014871805856999999, 2.060084e-06],
                [1099.912724382713, 1255.155437617339, 1407.960618786552, 5760511.412062648, 6957126.272013029,
                 3264769.430405523, 10.613935376776, 0.01855738889, 2.254727e-06],
                [1120.556841321501, 1151.844507270716, 1439.410715884273, 6948886.461354318, 3017680.9523862735,
                 4313526.957920614, 11.651913842295, 0.018208637796, 1.990718e-06],
                [1124.9533335742, 1186.166614541836, 1406.490558394534, 5104064.771557352, 6854157.806043102,
                 6792639.098163593, 9.324635019941, 0.017829849372, 2.3500399999999998e-06],
                [1060.306370914016, 1177.691586694892, 1356.64807625923, 6935598.916319065, 3041099.6477138954,
                 6442069.761881914, 8.822829285853, 0.015536299970999999, 1.731994e-06],
                [1192.40385198054, 1172.90338393294, 1398.198640963345, 5836571.517527281, 6658504.582887757,
                 3825984.626864104, 8.657999694988, 0.014541138015, 2.451221e-06],
                [1020.4033522193449, 1203.85747443386, 1323.879764465933, 6530553.91558025, 6469441.034533508,
                 3886526.433409815, 12.481427340769999, 0.016501171603, 2.424672e-06],
                [1066.242402098398, 1285.208512243742, 1341.080973157573, 6167119.824479279, 6559695.60009571,
                 3392404.364998077, 10.478922292791, 0.013199752915, 2.29902e-06],
                [1022.8133585878439, 1114.458955555204, 1441.513600695975, 6710705.024531956, 4886156.461802257,
                 6256204.592895824, 10.404065688781, 0.015312967018, 2.313376e-06],
                [1080.996695972347, 1286.585721993858, 1387.3638276057409, 6919538.349295578, 6599153.740642592,
                 3747963.890594435, 11.498552909894, 0.017503021367, 2.148159e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1056.2370978971562, 1228.0621474080876, 1331.3214771996602, 3627678.357077309, 2998055.7538120616,
                 2532740.0207836498, 9.697531154957, 0.01319801981388231, 2.340356e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2833336.6064214385, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1006.917429294439, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1002.3679828289813, 1233.8201449012863, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016812713242489365, 2.110682e-06],
                [1154.407498959948, 1236.655253905034, 1351.174773511498, 4487711.842441496, 2586063.573298348,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1017.0400643510857, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1002.8735112225839, 1222.315873787476, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016459337889, 2.110682e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3092961.8274809974,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1011.1651636778292, 1219.449161088396, 1351.174773511498, 4531951.9996198295, 3494078.511359096,
                 2756512.928705282, 11.845560061499354, 0.013221990874991763, 1.716072e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2533132.283573338, 6921050.360550507,
                 3461589.4238329353, 9.457999534029, 0.018048349898, 2.019091e-06],
                [1128.0511659807232, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.1650025557814735e-06],
                [1103.683044712124, 1117.8172959384456, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5785882.1544099115, 12.013233977881999, 0.013809671326999999, 1.912446838761154e-06],
                [1119.645645441935, 1166.051027068192, 1410.400490357985, 2839590.7843750915, 3105552.0740890396,
                 4280687.504420381, 9.013062737532, 0.015566562652, 1.816343e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5793876.509322431, 12.013233977881999, 0.013809671326999999, 1.936649e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3625963.6402100134, 3381410.7237955173,
                 5793876.509322431, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1039.018276207625, 1145.028891106496, 1313.910403182829, 2933184.899608702, 5890961.479807982,
                 5730384.392981274, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1039.018276207625, 1153.8582415325982, 1313.910403182829, 2933184.899608702, 2850090.527418211,
                 2951607.045186893, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1181.71642134566, 1106.1331358152229, 1300.392678923062, 3188724.243480558, 3611549.2338148607,
                 6040859.706667053, 11.608820336322, 0.015677646923999998, 2.337928e-06],
                [1113.362945387948, 1156.142543911252, 1318.7628524468, 4025728.5804803325, 6917121.327522967,
                 4324002.329611495, 8.410287456499, 0.013058706477, 1.860162e-06],
                [1000.135095386044, 1105.4306348685059, 1330.571207498223, 4846321.381285451, 2701805.7086814954,
                 4484561.382528685, 10.352115864391, 0.014366939106, 1.835322e-06],
                [1086.4310962436139, 1143.313661827018, 1366.172986059074, 4433250.634960789, 3537428.9649966285,
                 4778094.621137584, 10.76567418015, 0.01753600924, 2.087574e-06],
                [1170.0160238258507, 1219.449161088396, 1351.174773511498, 4892238.0689851735, 3553816.363672882,
                 2825500.380147861, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1154.407498959948, 1219.449161088396, 1351.174773511498, 4799249.275372577, 4503915.849723133,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1144.62055680547, 1220.5901686895759, 1444.561092205831, 4195862.806659156, 6454896.318523467,
                 2945971.4495624257, 10.977852933097, 0.014840773008576945, 2.048221e-06],
                [1084.1351758402184, 1241.226049743086, 1421.184002985495, 4028227.4624709627, 5506585.320418177,
                 4635067.7086390015, 11.026913378266935, 0.014871805856999999, 2.002752315241874e-06],
                [1048.5536313074438, 1280.282124741661, 1447.05386618055, 3756596.421046057, 6182273.026197797,
                 4498231.51928829, 11.583196813951, 0.016789544937, 2.281151e-06],
                [1144.62055680547, 1220.5901686895759, 1444.561092205831, 4253401.780289518, 6431058.822726591,
                 2945971.4495624257, 10.977852933097, 0.013716394277, 1.974682253310644e-06],
                [1115.4642213739219, 1237.480378115084, 1399.160793550262, 3448300.45522647, 6948646.520626062,
                 6514459.910768375, 8.720091398814, 0.013200356943, 1.888512e-06],
                [1144.62055680547, 1220.5901686895759, 1444.561092205831, 4253401.780289518, 6454896.318523467,
                 2945971.4495624257, 10.977852933097, 0.013716394277, 2.048221e-06],
                [1038.2632837344158, 1274.573086050776, 1441.884859713586, 3756596.421046057, 6182273.026197797,
                 4498231.51928829, 11.583196813951, 0.016789544937, 2.281151e-06],
                [1025.149660893128, 1274.573086050776, 1441.884859713586, 3756596.421046057, 6182273.026197797,
                 4498231.51928829, 11.583196813951, 0.016789544937, 2.281151e-06],
                [1065.247568510084, 1208.273246309111, 1365.658155840244, 3883964.276353225, 5756819.268304925,
                 5427939.298501339, 8.703442787445999, 0.016377523318, 2.36645e-06],
                [1183.3845823801569, 1103.53856891706, 1339.1973498816849, 4009970.812727008, 4766329.348673246,
                 3242830.1212443644, 11.73045420814, 0.013025100232999999, 2.041424e-06],
                [1113.362945387948, 1275.0473319678974, 1317.0402368068549, 4025728.5804803325, 6917121.327522967,
                 6244999.228967555, 8.410287456499, 0.012965542767530727, 1.8588072182133293e-06],
                [1075.8304613644482, 1291.3389576897396, 1382.704475395262, 4627392.441237274, 5111018.915760348,
                 3799434.3905045544, 10.829110164955999, 0.016290290978999998, 1.6468622167278855e-06],
                [1030.6783189917, 1173.084221673461, 1424.795222279961, 3227458.8804032593, 6221198.969017923,
                 5665900.526135622, 11.973154310207, 0.013935315602, 2.393006e-06],
                [1198.424856114746, 1296.995725535306, 1440.535501239776, 4037142.379113212, 6651700.51664973,
                 5937108.9775888715, 11.349016071859, 0.01691485517, 2.435554e-06],
                [1154.407498959948, 1295.8346354373668, 1351.174773511498, 4799249.275372577, 4503915.849723133,
                 2756512.928705282, 11.43677145574, 0.014938196750102613, 2.3286824690076844e-06],
                [1183.3845823801569, 1103.53856891706, 1339.1973498816849, 5857130.671192093, 4830487.666190833,
                 3242830.1212443644, 11.73045420814, 0.013025100232999999, 2.041424e-06],
                [1046.908391145076, 1110.0747175909369, 1429.941399224388, 5217719.52396076, 4029724.500115867,
                 4760881.312296346, 9.197976880151, 0.01708413309, 1.674435e-06],
                [1144.62055680547, 1220.5901686895759, 1442.215360592326, 4389167.385786755, 6407687.243641006,
                 4342385.035752849, 10.977852933097, 0.014303694536669386, 2.1339146227475467e-06],
                [1009.1146043961726, 1298.779343298153, 1401.16581949804, 4902261.4984828215, 3669736.7410551626,
                 5506106.1604879005, 11.351561456138, 0.013820405011999999, 1.9265682162822142e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1181.71642134566, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 6040859.706667053, 11.216198135126099, 0.015677646923999998, 2.337928e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 5911610.327873674,
                 4700731.238906128, 9.427550057683757, 0.015939573082178627, 1.8770206541805113e-06],
                [1056.2370978971562, 1228.0621474080876, 1331.3214771996602, 3627678.357077309, 2998055.7538120616,
                 2532740.0207836498, 9.697531154957, 0.01319801981388231, 2.340356e-06],
                [1112.3524010915671, 1257.584793961963, 1310.427373334314, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.41622197839763, 0.016637333018644673, 1.6652576526071669e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1138.8994908064808, 1264.2034079590524, 1326.3993608570645, 3614288.9880884974, 3448028.61050569,
                 2613897.3925167606, 10.676487893570695, 0.013473647569965668, 2.369752188687355e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2833336.6064214385, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1006.917429294439, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1006.7804880189733, 1107.897956860713, 1436.8036091630076, 3991524.1814932027, 2715483.6685275612,
                 5499059.652247055, 10.200095357793, 0.013625707946, 1.8527773810483195e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2522732.7973377593, 6921050.360550507,
                 3461589.4238329353, 11.919213951914514, 0.018048349898, 2.019091e-06],
                [1002.3679828289813, 1233.8201449012863, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016812713242489365, 2.110682e-06],
                [1154.407498959948, 1236.655253905034, 1351.174773511498, 4487711.842441496, 2586063.573298348,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1017.0400643510857, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1002.8735112225839, 1222.315873787476, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016459337889, 2.110682e-06],
                [1089.956830650977, 1114.607364489448, 1419.872158616546, 3617780.108636307, 3381410.7237955173,
                 5793876.509322431, 9.551867507206268, 0.013809671326999999, 1.937855678932045e-06],
                [1148.9327261066758, 1153.6304141964065, 1304.551838382366, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.76748429421807, 0.018402202852, 1.7815010318259866e-06],
                [1098.218987361545, 1228.2800522789857, 1362.310913050378, 3949022.787889535, 2765020.178343959,
                 5685500.530032963, 9.477858635073746, 0.017868050845, 2.167855466685067e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3092961.8274809974,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1011.1651636778292, 1219.449161088396, 1351.174773511498, 4531951.9996198295, 3494078.511359096,
                 2756512.928705282, 11.845560061499354, 0.013221990874991763, 1.716072e-06],
                [1011.1651636778292, 1219.449161088396, 1351.174773511498, 4531951.9996198295, 3494078.511359096,
                 2756512.928705282, 11.845560061499354, 0.013705077041655573, 1.718482291015711e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2500575.673267536, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2533132.283573338, 6921050.360550507,
                 3461589.4238329353, 9.457999534029, 0.018048349898, 2.019091e-06],
                [1154.5397873167135, 1264.202938587985, 1362.310913050378, 3921988.0613203924, 3092961.8274809974,
                 5772927.094828865, 10.634894158475426, 0.017868050845, 2.165278e-06],
                [1119.645645441935, 1166.051027068192, 1410.400490357985, 2839590.7843750915, 3105552.0740890396,
                 4280687.504420381, 9.013062737532, 0.015566562652, 1.816343e-06],
                [1103.683044712124, 1117.8172959384456, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5785882.1544099115, 12.013233977881999, 0.013809671326999999, 1.912446838761154e-06],
                [1128.0511659807232, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.1650025557814735e-06],
                [1170.0160238258507, 1219.449161088396, 1351.174773511498, 4565079.831118726, 3553816.363672882,
                 2822066.382887337, 11.43677145574, 0.013222653340200709, 1.7776952010619029e-06],
                [1137.9157350997789, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.165278e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5793876.509322431, 12.013233977881999, 0.013809671326999999, 1.936649e-06],
                [1128.0511659807232, 1257.4239927153367, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5685500.530032963, 10.713570074941, 0.017868050845, 2.1650025557814735e-06],
                [1103.683044712124, 1114.607364489448, 1419.872158616546, 3625963.6402100134, 3381410.7237955173,
                 5793876.509322431, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1039.018276207625, 1145.028891106496, 1313.910403182829, 2933184.899608702, 5890961.479807982,
                 5730384.392981274, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1143.4387951608267, 1264.202938587985, 1362.310913050378, 3935633.4189007236, 3381585.1063012956,
                 5920157.272578891, 10.713570074941, 0.018024370678357375, 2.1650025557814735e-06],
                [1104.4239043486102, 1114.607364489448, 1419.872158616546, 3625963.6402100134, 3381410.7237955173,
                 5793876.509322431, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1039.018276207625, 1153.8582415325982, 1313.910403182829, 2933184.899608702, 2850090.527418211,
                 2951607.045186893, 11.257555580852, 0.015413925091, 1.781494e-06],
                [1183.4996967827772, 1219.449161088396, 1351.174773511498, 4892238.0689851735, 3021703.552791283,
                 2825500.380147861, 11.266590332781, 0.013341161615, 1.716072e-06],
                [1025.56871224164, 1109.5634116524457, 1306.648311432572, 4065276.9287967673, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1181.71642134566, 1106.1331358152229, 1300.392678923062, 3188724.243480558, 3611549.2338148607,
                 6040859.706667053, 11.608820336322, 0.015677646923999998, 2.337928e-06],
                [1000.135095386044, 1105.4306348685059, 1330.5493242905914, 4713546.073109317, 2662353.0558696883,
                 4484561.382528685, 10.334296325294885, 0.014366939106, 2.0449849189212174e-06],
                [1000.6018390865513, 1100.7447212558154, 1381.4951036145428, 4846321.381285451, 2701805.7086814954,
                 4484561.382528685, 10.352115864391, 0.014327089983809603, 1.9053230239915903e-06],
                [1039.018276207625, 1122.5444686286983, 1326.5430142598552, 2791094.7810776485, 2697731.1452343613,
                 3158915.6229445105, 11.255935596768929, 0.015413925091, 1.7812039681740136e-06],
                [1113.362945387948, 1156.142543911252, 1318.7628524468, 4025728.5804803325, 6917121.327522967,
                 4324002.329611495, 8.410287456499, 0.013058706477, 1.860162e-06],
                [1170.0160238258507, 1219.449161088396, 1351.174773511498, 4892238.0689851735, 3553816.363672882,
                 2825500.380147861, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2727002.312869729, 5177667.749296468,
                 6055735.99685296, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1181.71642134566, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 6040859.706667053, 11.216198135126099, 0.015677646923999998, 2.337928e-06],
                [1056.2370978971562, 1228.0621474080876, 1331.3214771996602, 3627678.357077309, 2998055.7538120616,
                 2532740.0207836498, 9.697531154957, 0.01319801981388231, 2.340356e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 2562820.6700545694,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1064.317474852996, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.562097953987161, 0.012851621732462407, 2.065492e-06],
                [1115.4440971511028, 1257.584793961963, 1335.811778824276, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.41622197839763, 0.01810061756308735, 1.6652576526071669e-06],
                [1121.556326096255, 1225.477552275013, 1419.57153976711, 2824159.735864557, 4668788.44424598,
                 4235528.204404335, 11.244808146052, 0.01805712000324544, 2.432848e-06],
                [1134.3052340247398, 1266.4455156803888, 1430.258879312566, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1003.6429065603127, 1107.897956860713, 1436.8036091630076, 3773196.911885251, 2715483.6685275612,
                 5499059.652247055, 10.200095357793, 0.013625707946, 1.8527773810483195e-06],
                [1112.3524010915671, 1257.584793961963, 1310.427373334314, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.41622197839763, 0.016637333018644673, 1.6652576526071669e-06],
                [1128.0891752962004, 1257.584793961963, 1310.427373334314, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.399765828035274, 0.017857864436581865, 1.6872256181843123e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 5911610.327873674,
                 4700731.238906128, 9.427550057683757, 0.015939573082178627, 1.8770206541805113e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 11.198032372030648, 0.018402202852, 1.7812109999999999e-06],
                [1006.917429294439, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1138.8994908064808, 1264.2034079590524, 1326.3993608570645, 3614288.9880884974, 3448028.61050569,
                 2613897.3925167606, 10.676487893570695, 0.013473647569965668, 2.369752188687355e-06],
                [1089.945804052426, 1238.8150696891787, 1419.872158616546, 3617780.108636307, 3381410.7237955173,
                 5793876.509322431, 9.940957317476279, 0.013890054540421578, 1.937855678932045e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2833336.6064214385, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.018099038041, 2.432848e-06],
                [1135.680586055101, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4087908.14487714, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1017.0400643510857, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1098.218987361545, 1228.3923530007805, 1362.8611286943965, 3750480.6814249475, 2857960.75831493,
                 2974008.506144651, 9.477858635073746, 0.013821536692075082, 1.7741876259115089e-06],
                [1154.407498959948, 1236.655253905034, 1351.174773511498, 4427474.235537298, 2542437.981751782,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1006.7804880189733, 1107.897956860713, 1436.8036091630076, 3991524.1814932027, 2715483.6685275612,
                 5499059.652247055, 10.200095357793, 0.013625707946, 1.8527773810483195e-06],
                [1157.058011260272, 1240.3487101897, 1332.891017396078, 2522732.7973377593, 6921050.360550507,
                 3461589.4238329353, 11.919213951914514, 0.018048349898, 2.019091e-06],
                [1017.0400643510857, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1147.501727915812, 1122.316640929817, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.018402202852, 1.7812109999999999e-06],
                [1002.8735112225839, 1222.315873787476, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016459337889, 2.110682e-06],
                [1015.7284995164233, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5607520.642758014, 10.386857748765365, 0.013625707946, 2.3722169999999998e-06],
                [1002.3679828289813, 1233.8201449012863, 1434.5695663290069, 2867030.052889139, 5041021.649316114,
                 3083413.354012654, 9.962370627938, 0.016812713242489365, 2.110682e-06],
                [1147.501727915812, 1114.1127163538993, 1304.635397750834, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.765864310135, 0.01837329413225449, 1.7812109999999999e-06],
                [1154.407498959948, 1236.655253905034, 1351.174773511498, 4487711.842441496, 2586063.573298348,
                 2756512.928705282, 11.43677145574, 0.013341161615, 1.716072e-06],
                [1157.058011260272, 1240.3487101897, 1307.5075760627185, 2522732.7973377593, 6921050.360550507,
                 3461589.4238329353, 11.919213951914514, 0.016585063159245916, 2.019091e-06],
                [1135.736010842282, 1117.8172959384456, 1419.872158616546, 3589704.202519048, 3381410.7237955173,
                 5785882.1544099115, 12.013233977881999, 0.013809671326999999, 1.8866730357254492e-06],
                [1154.407498959948, 1236.655253905034, 1342.9555486456536, 4487711.842441496, 2586063.573298348,
                 2756512.928705282, 11.43677145574, 0.013355360412473294, 1.716072e-06],
                [1157.058011260272, 1240.3487101897, 1325.7833258513153, 2526832.2962593758, 6921050.360550507,
                 3555721.1998439874, 10.398248441320256, 0.018048349898, 2.019091e-06],
                [1089.956830650977, 1114.607364489448, 1419.872158616546, 3617780.108636307, 3381410.7237955173,
                 5793876.509322431, 9.551867507206268, 0.013809671326999999, 1.937855678932045e-06],
                [1148.9327261066758, 1153.6304141964065, 1304.551838382366, 2667086.1590371584, 3207967.3793078577,
                 4107081.9288311386, 10.76748429421807, 0.018402202852, 1.7815010318259866e-06],
                [1121.556326096255, 1225.477552275013, 1423.939707863312, 2624686.8443433163, 4056131.6666533365,
                 4235528.204404335, 11.244808146052, 0.01801296639855309, 2.4490968385460116e-06],
                [1098.218987361545, 1228.2800522789857, 1362.310913050378, 3949022.787889535, 2765020.178343959,
                 5685500.530032963, 9.477858635073746, 0.017868050845, 2.167855466685067e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.3930516661924304e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1181.71642134566, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5326574.538707027, 11.216198135126099, 0.015677646923999998, 2.337928e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.258879312566, 2849919.1458090344, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2727002.312869729, 5177667.749296468,
                 6055735.99685296, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1086.6744945525136, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6170892.134969252, 9.867582517813236, 0.016533863886783313, 1.6399981991408293e-06],
                [1181.71642134566, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 6040859.706667053, 11.216198135126099, 0.015677646923999998, 2.337928e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1056.2370978971562, 1228.0621474080876, 1331.3214771996602, 3627678.357077309, 2998055.7538120616,
                 2532740.0207836498, 9.697531154957, 0.01319801981388231, 2.340356e-06],
                [1133.7319732346987, 1231.6287501442703, 1433.860821727598, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.017327964330254093, 1.8465579641647771e-06],
                [1162.985408391334, 1282.3691608818883, 1426.265554917064, 2849919.1458090344, 5723333.212275367,
                 4700731.238906128, 9.667633942498014, 0.015939573082178627, 1.8666212688497504e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 5244155.843156379, 10.835973545147, 0.016005238592, 1.8812150677270692e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 2562820.6700545694,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1006.7804880189733, 1116.961606566379, 1434.546466688771, 3753295.247361683, 2715483.6685275612,
                 5499059.652247055, 10.366563701365079, 0.013625707946, 1.8527773810483195e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1003.6429065603127, 1107.897956860713, 1436.8036091630076, 3773196.911885251, 2715483.6685275612,
                 5499059.652247055, 10.200095357793, 0.013625707946, 1.8527773810483195e-06],
                [1064.317474852996, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.562097953987161, 0.012851621732462407, 2.065492e-06],
                [1115.4440971511028, 1257.584793961963, 1335.811778824276, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.41622197839763, 0.01810061756308735, 1.6652576526071669e-06],
                [1147.501727915812, 1114.1127163538993, 1304.635397750834, 3029706.7719576024, 3369551.2205330944,
                 4107081.9288311386, 10.885113287182477, 0.018026000660431068, 1.7812109999999999e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1121.556326096255, 1225.477552275013, 1419.57153976711, 2824159.735864557, 4668788.44424598,
                 4235528.204404335, 11.244808146052, 0.01805712000324544, 2.432848e-06],
                [1134.3052340247398, 1266.4455156803888, 1430.258879312566, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.369921485045, 0.013701864499999999, 2.065492e-06],
                [1128.0891752962004, 1257.584793961963, 1310.427373334314, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.399765828035274, 0.017857864436581865, 1.6872256181843123e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 5911610.327873674,
                 4700731.238906128, 9.427550057683757, 0.015939573082178627, 1.8770206541805113e-06],
                [1112.3524010915671, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.41622197839763, 0.016637333018644673, 1.6652576526071669e-06],
                [1003.3748195496676, 1125.693111980194, 1380.7390220955515, 3726676.39046288, 4479343.791478679,
                 2709285.3525210335, 10.096188090219801, 0.013701864499999999, 2.065492e-06],
                [1089.945804052426, 1238.8150696891787, 1419.872158616546, 3617780.108636307, 3381410.7237955173,
                 5793876.509322431, 9.940957317476279, 0.013890054540421578, 1.9366399480294868e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 5911610.327873674,
                 4700731.238906128, 9.427550057683757, 0.015939573082178627, 1.8770206541805113e-06],
                [1015.9697556802806, 1253.4752967539239, 1435.1650459022958, 3991524.1814932027, 2715754.5081717256,
                 4135288.1126174694, 10.386857748765365, 0.013679303451418096, 2.3722169999999998e-06],
                [1053.745648916394, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4690134.6429963,
                 2709285.3525210335, 11.369921485045, 0.013408772908143086, 2.065492e-06],
                [1112.3524010915671, 1257.584793961963, 1310.427373334314, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.41622197839763, 0.016637333018644673, 1.6652576526071669e-06],
                [1056.2370978971562, 1228.0621474080876, 1331.3214771996602, 3627678.357077309, 5294406.088616831,
                 2511228.6905781724, 9.697531154957, 0.013249195336592957, 2.340356e-06],
                [1112.3524010915671, 1261.167541028615, 1310.427373334314, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 8.46687778040242, 0.016637333018644673, 1.6652576526071669e-06],
                [1098.218987361545, 1228.3923530007805, 1362.8611286943965, 3750480.6814249475, 2857960.75831493,
                 2974008.506144651, 9.613138814249597, 0.013821536692075082, 1.764558300213429e-06],
                [1006.917429294439, 1253.4752967539239, 1434.96949675091, 3991524.1814932027, 2715754.5081717256,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3722169999999998e-06],
                [1138.8994908064808, 1275.607802088896, 1326.3993608570645, 3614288.9880884974, 3448028.61050569,
                 2650987.5459747417, 10.676487893570695, 0.013473647569965668, 2.369752188687355e-06],
                [1089.945804052426, 1238.8150696891787, 1419.872158616546, 3617780.108636307, 3381410.7237955173,
                 5793876.509322431, 9.940957317476279, 0.013890054540421578, 1.937855678932045e-06],
                [1138.8994908064808, 1264.2034079590524, 1326.3993608570645, 3614288.9880884974, 3448028.61050569,
                 2613897.3925167606, 10.676487893570695, 0.013473647569965668, 2.369752188687355e-06],
                [1017.0400643510857, 1253.4752967539239, 1433.4462451932, 3991524.1814932027, 2728507.9047457194,
                 5499059.652247055, 10.200095357793, 0.013625707946, 2.3565571282176328e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.33776038941515e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1181.71642134566, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5326574.538707027, 11.216198135126099, 0.015677646923999998, 2.337928e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5327018.267562188,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.6652576526071669e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.258879312566, 2849919.1458090344, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1165.3708118481113, 1257.5600518091533, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6170892.134969252, 9.763733552583153, 0.016533863886783313, 1.8687363845527149e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3514350.018490984, 3424741.165504974,
                 2532740.0207836498, 11.303133790427447, 0.013744620276, 2.369752188687355e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2727002.312869729, 5177667.749296468,
                 6055735.99685296, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1086.6744945525136, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6170892.134969252, 9.867582517813236, 0.016533863886783313, 1.6399981991408293e-06],
                [1118.574750587253, 1225.477552275013, 1419.57153976711, 2913298.37356719, 4668788.44424598,
                 4235528.204404335, 10.804801586068123, 0.018009563243247886, 2.432848e-06],
                [1099.8334006943442, 1228.280521663599, 1320.7305957642318, 3583733.79956298, 3752115.6375754834,
                 2554942.3385524517, 9.340562434534428, 0.014072187930050743, 2.381564648675539e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1162.985408391334, 1288.241759707222, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 11.275980105130875, 0.01605278676006669, 1.891574e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1133.7319732346987, 1231.6287501442703, 1433.860821727598, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.017327964330254093, 1.8465579641647771e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3380884.8225842332,
                 2541919.4174867366, 11.792786416892714, 0.01367658030408831, 2.4001855210031537e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.3930516661924304e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.3930516661924304e-06],
                [1181.71642134566, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 6040859.706667053, 11.216198135126099, 0.015677646923999998, 2.337928e-06],
                [1090.1434600041982, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.01355763746913406, 2.340356e-06],
                [1162.985408391334, 1282.3691608818883, 1426.265554917064, 2849919.1458090344, 5723333.212275367,
                 4700731.238906128, 9.667633942498014, 0.015939573082178627, 1.8666212688497504e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1162.985408391334, 1271.107772815279, 1426.086012425705, 2849919.1458090344, 6117638.170328333,
                 5244155.843156379, 10.835973545147, 0.015029717961110403, 1.8812150677270692e-06],
                [1064.317474852996, 1125.693111980194, 1373.150130234432, 3388779.352021926, 4629853.564052632,
                 2663148.3146476536, 11.562097953987161, 0.012624837228155834, 2.065492e-06],
                [1113.902144919001, 1125.693111980194, 1373.150130234432, 3591541.589713606, 4629853.564052632,
                 2709285.3525210335, 11.58145858407725, 0.016944948373997893, 2.065492e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6517843.878102322,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1099.8334006943442, 1228.280521663599, 1418.6287677598361, 3627678.357077309, 3682634.5184471733,
                 2561089.0347011206, 9.507093911625585, 0.013744620276, 2.4321979076424337e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1162.985408391334, 1282.3691608818883, 1426.265554917064, 2849919.1458090344, 5723333.212275367,
                 4700731.238906128, 9.667633942498014, 0.015939573082178627, 1.8666212688497504e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 5244155.843156379, 10.835973545147, 0.016005238592, 1.8812150677270692e-06],
                [1100.1063624541289, 1227.4420530805235, 1333.5182217626189, 3624493.3041226277, 3511987.0914596524,
                 2532740.0207836498, 9.064929256226925, 0.013744620276, 2.369752188687355e-06],
                [1006.7804880189733, 1116.961606566379, 1434.546466688771, 3753295.247361683, 2715483.6685275612,
                 5499059.652247055, 10.366563701365079, 0.013625707946, 1.8527773810483195e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1056.2370978971562, 1228.0621474080876, 1331.3214771996602, 3627678.357077309, 2998055.7538120616,
                 2532740.0207836498, 9.697531154957, 0.01319801981388231, 2.340356e-06],
                [1115.466323398068, 1257.9877608217614, 1424.6533094655529, 2711709.7515290566, 5285528.529106242,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.6463307499573493e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2610334.436302432, 5177667.749296468,
                 6055735.99685296, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1100.1063624541289, 1239.4182277433551, 1327.008153045644, 3627678.357077309, 2562820.6700545694,
                 2532740.0207836498, 9.697531154957, 0.01375409951794936, 2.34052361058485e-06],
                [1147.501727915812, 1114.1127163538993, 1304.635397750834, 3029706.7719576024, 3369551.2205330944,
                 4107081.9288311386, 10.885113287182477, 0.018026000660431068, 1.7812109999999999e-06],
                [1000.95387509763, 1107.897956860713, 1430.2475226641252, 3692765.14860116, 2595594.7361568823,
                 5521872.338771292, 9.79652553411393, 0.013500085136179965, 1.8527773810483195e-06],
                [1003.6429065603127, 1107.897956860713, 1436.8036091630076, 3773196.911885251, 2715483.6685275612,
                 5499059.652247055, 10.200095357793, 0.013625707946, 1.8527773810483195e-06],
                [1056.3804291682652, 1125.693111980194, 1373.150130234432, 3726676.39046288, 4629853.564052632,
                 2709285.3525210335, 11.473558022559024, 0.013701864499999999, 2.065492e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1181.71642134566, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5326574.538707027, 11.216198135126099, 0.015677646923999998, 2.337928e-06],
                [1162.985408391334, 1271.107772815279, 1430.9301822203277, 2849919.1458090344, 6292860.96194586,
                 4709676.100805292, 10.847019408563023, 0.016005238592, 1.8608908325858286e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.7036883813014, 2849919.1458090344, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.33776038941515e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.313183931715978e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5327018.267562188,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.6652576526071669e-06],
                [1165.3708118481113, 1257.5600518091533, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6170892.134969252, 9.763733552583153, 0.016533863886783313, 1.8687363845527149e-06],
                [1111.9294185127558, 1231.6287501442703, 1436.9514030052342, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.017327964330254093, 1.8465579641647771e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016564172329980215, 1.64751e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.258879312566, 2849919.1458090344, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.258879312566, 2834886.9850633065, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016790084294285006, 1.6422729938108779e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.440776451404325, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.402883040588248e-06],
                [1118.574750587253, 1225.477552275013, 1419.57153976711, 2913298.37356719, 4668788.44424598,
                 4235528.204404335, 10.804801586068123, 0.018009563243247886, 2.432848e-06],
                [1086.6744945525136, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6170892.134969252, 9.867582517813236, 0.016533863886783313, 1.6399981991408293e-06],
                [1106.9740976067264, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1162.985408391334, 1279.5386822635626, 1426.086012425705, 2849919.1458090344, 6117638.170328333,
                 5244155.843156379, 10.755320844411056, 0.015035631426244598, 1.8812150677270692e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3380884.8225842332,
                 2541919.4174867366, 11.792786416892714, 0.01367658030408831, 2.4001855210031537e-06],
                [1133.7319732346987, 1231.6287501442703, 1433.860821727598, 2849919.1458090344, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.017634941653686658, 1.8669744103046079e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2610334.436302432, 5177667.749296468,
                 6055735.99685296, 9.869052328516, 0.017475501529314635, 1.6415556806410382e-06],
                [1162.985408391334, 1288.241759707222, 1425.8207458483287, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 11.282618025470859, 0.01605278676006669, 1.891574e-06],
                [1173.760206680501, 1241.2917167358653, 1431.0508432709498, 3130727.9042818854, 3611549.2338148607,
                 5245399.450055137, 11.19480477719018, 0.015355060165661974, 2.321674098717642e-06],
                [1012.5978711589274, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3314096.7712910594,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.3930516661924304e-06],
                [1007.3076285560295, 1258.9929268507472, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 5476108.741488709, 9.869052328516, 0.01655187517040234, 1.6953918374461486e-06],
                [1099.8334006943442, 1228.280521663599, 1320.7305957642318, 3583733.79956298, 3752115.6375754834,
                 2554942.3385524517, 9.340562434534428, 0.014072187930050743, 2.377100851183877e-06],
                [1055.185770740802, 1246.5797646495644, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5958879.200228611, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3514350.018490984, 3424741.165504974,
                 2532740.0207836498, 11.303133790427447, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1320.7305957642318, 3583733.79956298, 3752115.6375754834,
                 2554942.3385524517, 9.340562434534428, 0.014072187930050743, 2.381564648675539e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.3930516661924304e-06],
                [1118.574750587253, 1235.490624970862, 1419.57153976711, 2913298.37356719, 4668788.44424598,
                 4235528.204404335, 10.804801586068123, 0.018009563243247886, 2.432848e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.4001855210031537e-06],
                [1162.985408391334, 1288.241759707222, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 11.275980105130875, 0.01605278676006669, 1.891574e-06],
                [1115.466323398068, 1245.52972031989, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3381410.7237955173,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.3930516661924304e-06],
                [1099.8334006943442, 1228.280521663599, 1418.6287677598361, 3627678.357077309, 3682634.5184471733,
                 2561089.0347011206, 9.507093911625585, 0.013744620276, 2.4321979076424337e-06],
                [1133.7319732346987, 1231.6287501442703, 1433.860821727598, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.017327964330254093, 1.8465579641647771e-06],
                [1162.985408391334, 1282.3691608818883, 1426.265554917064, 2849919.1458090344, 5723333.212275367,
                 4700731.238906128, 9.667633942498014, 0.015939573082178627, 1.8666212688497504e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1171.1989534324357, 1241.2917167358653, 1428.5163072367866, 3130727.9042818854, 3611549.2338148607,
                 5235964.403682908, 11.394289259455535, 0.01552981782318728, 2.313183931715978e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.313183931715978e-06],
                [1162.985408391334, 1271.107772815279, 1430.9301822203277, 2849919.1458090344, 6292860.96194586,
                 4709676.100805292, 10.847019408563023, 0.016005238592, 1.8608908325858286e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5327018.267562188,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.6652576526071669e-06],
                [1111.9294185127558, 1231.6287501442703, 1436.9514030052342, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.017327964330254093, 1.8465579641647771e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 6125197.144713911, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1181.0176110655648, 1241.2917167358653, 1431.0508432709498, 3100999.558193504, 3611549.2338148607,
                 5244960.155394779, 11.283340866719415, 0.015355060165661974, 2.321674098717642e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016750155641775035, 1.6399981991408293e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016459766461, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6440186.163520689, 9.869052328516, 0.014655133358103744, 1.64751e-06],
                [1165.3708118481113, 1257.5600518091533, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6170892.134969252, 9.763733552583153, 0.016533863886783313, 1.8687363845527149e-06],
                [1173.760206680501, 1241.2917167358653, 1447.1854809617255, 3130727.9042818854, 3611549.2338148607,
                 5245400.108628566, 11.216198135126099, 0.01552981782318728, 2.33776038941515e-06],
                [1133.7319732346987, 1231.6287501442703, 1424.9158576601822, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8669744103046079e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1165.3708118481113, 1284.2421194708988, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6214366.85519229, 10.377362233237546, 0.016533863886783313, 1.8687363845527149e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016564172329980215, 1.64751e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.258879312566, 2834886.9850633065, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.7036883813014, 2849919.1458090344, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.258879312566, 2849919.1458090344, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1007.3076285560295, 1258.9929268507472, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 5476108.741488709, 9.42512548042827, 0.01655187517040234, 1.6953918374461486e-06],
                [1086.6744945525136, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6170892.134969252, 9.867582517813236, 0.016533863886783313, 1.6399981991408293e-06],
                [1016.1434047882536, 1258.9929268507472, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 5476108.741488709, 9.869052328516, 0.016514742940488783, 1.6953918374461486e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016790084294285006, 1.6422729938108779e-06],
                [1010.594960195638, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3380884.8225842332,
                 2541919.4174867366, 11.792786416892714, 0.01367658030408831, 2.4001855210031537e-06],
                [1162.985408391334, 1279.5386822635626, 1426.086012425705, 2849919.1458090344, 6117638.170328333,
                 5244155.843156379, 10.755320844411056, 0.015035631426244598, 1.8812150677270692e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.402883040588248e-06],
                [1161.49283152393, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.33776038941515e-06],
                [1012.5978711589274, 1114.607364489448, 1440.140121879473, 3826649.954397131, 3314096.7712910594,
                 2541919.4174867366, 11.71199876673876, 0.01367658030408831, 2.3930516661924304e-06],
                [1007.3076285560295, 1258.9929268507472, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 5476108.741488709, 9.869052328516, 0.01655187517040234, 1.6953918374461486e-06],
                [1100.9068545689154, 1228.280521663599, 1329.636923091002, 3514350.018490984, 3482600.221527501,
                 2532740.0207836498, 11.303133790427447, 0.013744620276, 2.3206091934658165e-06],
                [1116.524163794125, 1245.52972031989, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.7184984084868742e-06],
                [1173.760206680501, 1241.2917167358653, 1431.0508432709498, 3130727.9042818854, 3611549.2338148607,
                 5245399.450055137, 11.19480477719018, 0.015355060165661974, 2.321674098717642e-06],
                [1115.466323398068, 1252.0836293813945, 1424.6533094655529, 2610334.436302432, 5177667.749296468,
                 6042928.745306183, 9.869052328516, 0.017969745988824948, 1.6415556806410382e-06],
                [1099.8334006943442, 1228.280521663599, 1320.7305957642318, 3583733.79956298, 3752115.6375754834,
                 2554942.3385524517, 9.340562434534428, 0.014072187930050743, 2.377100851183877e-06],
                [1118.574750587253, 1221.4953942198913, 1419.57153976711, 2913298.37356719, 4668788.44424598,
                 4235528.204404335, 10.804801586068123, 0.018009563243247886, 2.4553884073962524e-06],
                [1106.1493144633912, 1245.52972031989, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01678981800236394, 1.64751e-06],
                [1136.7310977913553, 1228.280521663599, 1322.4595430719855, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1102.3944525784132, 1228.280521663599, 1319.1730980745965, 3627678.357077309, 3552984.980490215,
                 2589233.5511834263, 8.999508966801848, 0.013744620276, 2.381564648675539e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6440186.163520689, 9.869052328516, 0.014655133358103744, 1.64751e-06],
                [1181.0176110655648, 1241.2917167358653, 1431.0508432709498, 3100999.558193504, 3611549.2338148607,
                 5244960.155394779, 11.283340866719415, 0.015355060165661974, 2.321674098717642e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016564172329980215, 1.64751e-06],
                [1114.834792035313, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6462873.262083922, 9.78692463749932, 0.014655133358103744, 1.64751e-06],
                [1111.9294185127558, 1238.9831349130286, 1424.5396463101872, 2841102.2183005502, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.015809764308396975, 1.8473871305138034e-06],
                [1171.1989534324357, 1241.2917167358653, 1428.5163072367866, 3130727.9042818854, 3611549.2338148607,
                 5235964.403682908, 11.394289259455535, 0.01552981782318728, 2.313183931715978e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.258879312566, 2849919.1458090344, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1167.2885683989298, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.015792703010527837, 1.891574e-06],
                [1115.466323398068, 1231.7898086645037, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5868217.479161575, 9.787113063154044, 0.016750975918697033, 1.6520170993838028e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.258879312566, 2834886.9850633065, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016753600050059186, 1.6399981991408293e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.313183931715978e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.653942625364186, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 6125197.144713911, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1133.7319732346987, 1231.6287501442703, 1424.9158576601822, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8669744103046079e-06],
                [1162.985408391334, 1271.107772815279, 1430.9301822203277, 2849919.1458090344, 6292860.96194586,
                 4709676.100805292, 10.847019408563023, 0.016005238592, 1.8608908325858286e-06],
                [1134.3052340247398, 1228.2810960353463, 1430.258879312566, 2834886.9850633065, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638844712814545, 1.8465579641647771e-06],
                [1165.3708118481113, 1284.2421194708988, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6214366.85519229, 10.377362233237546, 0.016533863886783313, 1.8687363845527149e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016456322052715848, 1.6551951573792323e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5327018.267562188,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.6652576526071669e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016750155641775035, 1.6399981991408293e-06],
                [1086.6744945525136, 1257.584793961963, 1424.6533094655529, 2671351.947940796, 5177667.749296468,
                 6170892.134969252, 9.867582517813236, 0.016533863886783313, 1.6399981991408293e-06],
                [1111.9294185127558, 1231.6287501442703, 1436.9514030052342, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.017327964330254093, 1.8465579641647771e-06],
                [1165.3708118481113, 1257.5600518091533, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6170892.134969252, 9.763733552583153, 0.016533863886783313, 1.8687363845527149e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.1006908836548, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.7036883813014, 2849919.1458090344, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1099.8334006943442, 1219.1416303055253, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.3791894603434197e-06],
                [1086.6744945525136, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6008577.894118812, 9.867582517813236, 0.016533863886783313, 1.6399981991408293e-06],
                [1173.760206680501, 1241.2917167358653, 1447.1854809617255, 3130727.9042818854, 3611549.2338148607,
                 5245400.108628566, 11.216198135126099, 0.01552981782318728, 2.33776038941515e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4553912.363352443, 10.835973545147, 0.016005251475292706, 1.907179227136742e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1173.760206680501, 1241.2917167358653, 1431.6889935278664, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.015664942257637034, 2.4579859548301695e-06],
                [1007.3076285560295, 1258.9929268507472, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 5476108.741488709, 9.42512548042827, 0.01655187517040234, 1.6953918374461486e-06],
                [1150.3197745288064, 1284.2421194708988, 1422.392377237153, 2854778.224454453, 5423785.660980307,
                 6239458.140793205, 10.377362233237546, 0.016550747956540105, 1.8687363845527149e-06],
                [1161.49283152393, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.33776038941515e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5327018.267562188,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.6652576526071669e-06],
                [1016.1434047882536, 1258.9929268507472, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 5476108.741488709, 9.869052328516, 0.016514742940488783, 1.6953918374461486e-06],
                [1086.6744945525136, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6170892.134969252, 9.867582517813236, 0.016533863886783313, 1.6399981991408293e-06],
                [1115.466323398068, 1252.0836293813945, 1424.6533094655529, 2610334.436302432, 5177667.749296468,
                 6042928.745306183, 9.869052328516, 0.017969745988824948, 1.6415556806410382e-06],
                [1100.1063624541289, 1228.280521663599, 1327.008153045644, 3627678.357077309, 3038662.9625544758,
                 2532740.0207836498, 9.697531154957, 0.013744620276, 2.340356e-06],
                [1134.3052340247398, 1261.1141597856492, 1430.258879312566, 2834886.9850633065, 6153349.653974743,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5327018.267562188,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.6652576526071669e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4553912.363352443, 10.835973545147, 0.016005251475292706, 1.907179227136742e-06],
                [1181.0176110655648, 1241.2917167358653, 1431.0508432709498, 3100999.558193504, 3611549.2338148607,
                 5244960.155394779, 11.283340866719415, 0.015355060165661974, 2.321674098717642e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1167.2885683989298, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.015792703010527837, 1.891574e-06],
                [1086.181122815101, 1259.3587163655109, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.822108190567562, 0.01508497956653298, 1.64751e-06],
                [1134.3052340247398, 1225.2669802893638, 1430.258879312566, 2849919.1458090344, 5764876.190923505,
                 4700731.238906128, 9.411049254686262, 0.01638956964339133, 1.8465579641647771e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1134.3052340247398, 1228.2810960353463, 1430.258879312566, 2834886.9850633065, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638844712814545, 1.8465579641647771e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016750155641775035, 1.6399981991408293e-06],
                [1114.834792035313, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6462873.262083922, 9.78692463749932, 0.014655133358103744, 1.64751e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1171.1989534324357, 1260.4527545664735, 1423.524821138209, 3130727.9042818854, 3447560.0449528676,
                 5235964.403682908, 11.394289259455535, 0.015068372716108102, 2.313183931715978e-06],
                [1165.3708118481113, 1284.2421194708988, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6214366.85519229, 10.377362233237546, 0.016533863886783313, 1.8687363845527149e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.653942625364186, 0.01508497956653298, 1.64751e-06],
                [1086.6744945525136, 1257.584793961963, 1424.6533094655529, 2671351.947940796, 5177667.749296468,
                 6170892.134969252, 9.867582517813236, 0.016533863886783313, 1.6399981991408293e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016753600050059186, 1.6399981991408293e-06],
                [1111.9294185127558, 1251.4313725537131, 1420.1421242024198, 2841102.2183005502, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.01588730299040333, 1.8535053425111795e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 6125197.144713911, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1111.9294185127558, 1231.6287501442703, 1436.9514030052342, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.017327964330254093, 1.8465579641647771e-06],
                [1171.1989534324357, 1241.2917167358653, 1428.5163072367866, 3130727.9042818854, 3611549.2338148607,
                 5235964.403682908, 11.394289259455535, 0.01552981782318728, 2.313183931715978e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1114.5041669636305, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6462873.262083922, 9.78929894298772, 0.014653655805144156, 1.64751e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.313183931715978e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016456322052715848, 1.6551951573792323e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5370063.8091288535,
                 6125197.144713911, 9.869052328516, 0.01515091929856088, 1.64751e-06],
                [1105.7606376509357, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.934894193259877, 0.014655133358103744, 1.64751e-06],
                [1111.9294185127558, 1238.9831349130286, 1424.5396463101872, 2841102.2183005502, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.015809764308396975, 1.8473871305138034e-06],
                [1173.760206680501, 1241.2917167358653, 1447.1854809617255, 3130727.9042818854, 3611549.2338148607,
                 5245400.108628566, 11.216198135126099, 0.01552981782318728, 2.33776038941515e-06],
                [1133.7319732346987, 1231.6287501442703, 1424.9158576601822, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8669744103046079e-06],
                [1115.466323398068, 1231.7898086645037, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5868217.479161575, 9.787113063154044, 0.016750975918697033, 1.6520170993838028e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016564172329980215, 1.64751e-06],
                [1134.3052340247398, 1228.7883272249376, 1430.258879312566, 2834886.9850633065, 5764876.190923505,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1111.9294185127558, 1238.9831349130286, 1424.5396463101872, 2841102.2183005502, 5849056.825483265,
                 4700731.238906128, 9.36332541177229, 0.015809764308396975, 1.8605033193888066e-06],
                [1167.2885683989298, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.015792703010527837, 1.891574e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6440186.163520689, 9.869052328516, 0.014655133358103744, 1.64751e-06],
                [1162.985408391334, 1271.107772815279, 1430.9301822203277, 2849919.1458090344, 6292860.96194586,
                 4709676.100805292, 10.847019408563023, 0.016005238592, 1.8608908325858286e-06],
                [1086.8792226662247, 1257.584793961963, 1424.6482539664976, 2677849.96572248, 5177667.749296468,
                 6170892.134969252, 10.843270131540745, 0.016533863886783313, 1.8921978305199946e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.1006908836548, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1111.7984874490123, 1258.9929268507472, 1424.6533094655529, 2711709.7515290566, 5755307.133262615,
                 5476108.741488709, 9.42512548042827, 0.01655187517040234, 1.6953918374461486e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2532740.0207836498, 9.578804891430824, 0.013744620276, 2.369752188687355e-06],
                [1099.8334006943442, 1228.280521663599, 1319.2890643444373, 3627678.357077309, 3552984.980490215,
                 2554942.3385524517, 9.340562434534428, 0.013744620276, 2.381564648675539e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1134.3052340247398, 1261.1141597856492, 1430.258879312566, 2834886.9850633065, 6153349.653974743,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1173.760206680501, 1241.2917167358653, 1447.1854809617255, 3130727.9042818854, 3611549.2338148607,
                 5245400.108628566, 11.216198135126099, 0.01552981782318728, 2.33776038941515e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2672187.6112241857, 5194929.277645023,
                 5955635.622891837, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5389154.533109349,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.685798387702239e-06],
                [1086.8792226662247, 1257.584793961963, 1424.6482539664976, 2677849.96572248, 5177667.749296468,
                 6170892.134969252, 10.843270131540745, 0.016533863886783313, 1.8921978305199946e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5868217.479161575, 9.787113063154044, 0.016750975918697033, 1.6520170993838028e-06],
                [1111.9294185127558, 1231.6287501442703, 1436.9514030052342, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.017327964330254093, 1.8465579641647771e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016564172329980215, 1.64751e-06],
                [1086.8792226662247, 1257.584793961963, 1424.6482539664976, 2677849.96572248, 5151835.089616997,
                 6170892.134969252, 10.843270131540745, 0.016533863886783313, 1.8921978305199946e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 6125197.144713911, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1124.392389981206, 1260.5869822935006, 1419.626242400125, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.1006908836548, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1086.181122815101, 1259.3587163655109, 1424.4013736288516, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.64751e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.313183931715978e-06],
                [1167.2885683989298, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.015792703010527837, 1.891574e-06],
                [1105.7606376509357, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.934894193259877, 0.014655133358103744, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016750155641775035, 1.6399981991408293e-06],
                [1114.834792035313, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6462873.262083922, 9.78692463749932, 0.014655133358103744, 1.64751e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1165.3708118481113, 1284.2421194708988, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6214366.85519229, 10.377362233237546, 0.016533863886783313, 1.8687363845527149e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.653942625364186, 0.01508497956653298, 1.64751e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016753600050059186, 1.6399981991408293e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016456322052715848, 1.6551951573792323e-06],
                [1171.1989534324357, 1241.2917167358653, 1428.5163072367866, 3130727.9042818854, 3611549.2338148607,
                 5235964.403682908, 11.394289259455535, 0.01552981782318728, 2.313183931715978e-06],
                [1086.181122815101, 1259.3587163655109, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01508497956653298, 1.64751e-06],
                [1167.2885683989298, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.015792703010527837, 1.891574e-06],
                [1133.0499588683645, 1228.5392853023554, 1410.8868937934, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016307724969914394, 1.64751e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4553912.363352443, 10.835973545147, 0.01575636609759392, 1.907179227136742e-06],
                [1114.5041669636305, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6462873.262083922, 9.78929894298772, 0.014653655805144156, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5125021.87212479,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.014995216559648368, 1.64441131717817e-06],
                [1168.7843286919008, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5151932.1545183025,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1096.0238569072808, 1257.584793961963, 1424.6482539664976, 2677849.96572248, 5177667.749296468,
                 6170892.134969252, 10.507657612312261, 0.016800381450651318, 1.8921978305199946e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5370063.8091288535,
                 6125197.144713911, 9.869052328516, 0.01515091929856088, 1.64751e-06],
                [1111.9294185127558, 1238.9831349130286, 1424.5396463101872, 2841102.2183005502, 5849056.825483265,
                 4700731.238906128, 9.36332541177229, 0.015809764308396975, 1.8605033193888066e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6440186.163520689, 9.869052328516, 0.014655133358103744, 1.64751e-06],
                [1105.7606376509357, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5146910.105384467,
                 6462873.262083922, 9.868257560463114, 0.014655133358103744, 1.64751e-06],
                [1162.985408391334, 1271.107772815279, 1430.9301822203277, 2849919.1458090344, 6292860.96194586,
                 4709676.100805292, 10.847019408563023, 0.016005238592, 1.8608908325858286e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4553912.363352443, 10.835973545147, 0.016005251475292706, 1.907179227136742e-06],
                [1086.181122815101, 1259.3587163655109, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.822108190567562, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.8064274069718, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.6449800619998893e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1173.760206680501, 1241.2917167358653, 1447.1854809617255, 3130727.9042818854, 3611549.2338148607,
                 5245400.108628566, 11.216198135126099, 0.01552981782318728, 2.33776038941515e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2672187.6112241857, 5194929.277645023,
                 5955635.622891837, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5389154.533109349,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.685798387702239e-06],
                [1086.8792226662247, 1257.584793961963, 1424.6482539664976, 2677849.96572248, 5177667.749296468,
                 6170892.134969252, 10.843270131540745, 0.016533863886783313, 1.8921978305199946e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5868217.479161575, 9.787113063154044, 0.016750975918697033, 1.6520170993838028e-06],
                [1111.9294185127558, 1231.6287501442703, 1436.9514030052342, 2849919.1458090344, 5771719.995759816,
                 4700731.238906128, 9.36332541177229, 0.017327964330254093, 1.8465579641647771e-06],
                [1086.8792226662247, 1257.584793961963, 1424.6482539664976, 2677849.96572248, 5151835.089616997,
                 6170892.134969252, 10.843270131540745, 0.016533863886783313, 1.8921978305199946e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.835973545147, 0.016005238592, 1.891574e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 6125197.144713911, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1124.392389981206, 1260.5869822935006, 1419.626242400125, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.1006908836548, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1086.181122815101, 1259.3587163655109, 1424.4013736288516, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.64751e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.313183931715978e-06],
                [1105.7606376509357, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.934894193259877, 0.014655133358103744, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6280169.899133423,
                 4553912.363352443, 10.835973545147, 0.01562869836514684, 1.907179227136742e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016753600050059186, 1.6399981991408293e-06],
                [1114.834792035313, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6462873.262083922, 9.78692463749932, 0.014655133358103744, 1.64751e-06],
                [1086.181122815101, 1259.3587163655109, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01508497956653298, 1.64751e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016750155641775035, 1.6399981991408293e-06],
                [1165.3708118481113, 1284.2421194708988, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6214366.85519229, 10.377362233237546, 0.016533863886783313, 1.8687363845527149e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1107.3715710091938, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6194661.429560538, 10.053256657600741, 0.01508497956653298, 1.6520168892140052e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5868217.479161575, 9.787113063154044, 0.016750975918697033, 1.6475102101697995e-06],
                [1111.7200920428952, 1245.9167513378093, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1111.1608895000384, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.869052328516, 0.014974300156757834, 1.64751e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4700731.238906128, 10.569992232599837, 0.016005238592, 1.891574e-06],
                [1168.7843286919008, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5151932.1545183025,
                 6252960.904134171, 11.437514624265415, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2767648.6297825263, 5298269.598703455,
                 6125197.144713911, 9.615810164209593, 0.016750975918697033, 1.6474993031812003e-06],
                [1114.834792035313, 1251.535389764655, 1424.6533094655529, 2577630.7506229677, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1115.466323398068, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016456322052715848, 1.6551951573792323e-06],
                [1133.0499588683645, 1228.5392853023554, 1410.8868937934, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016307724969914394, 1.64751e-06],
                [1162.985408391334, 1271.9251223025587, 1426.265554917064, 2849919.1458090344, 6572214.362356157,
                 4700731.238906128, 10.141368020101972, 0.01581701564809486, 1.65389119410361e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5125021.87212479,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.014995216559648368, 1.64441131717817e-06],
                [1168.7843286919008, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5151932.1545183025,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4553912.363352443, 10.835973545147, 0.01575636609759392, 1.907179227136742e-06],
                [1134.4136265483053, 1257.584793961963, 1442.1439452584866, 2677849.96572248, 5177667.749296468,
                 6170892.134969252, 10.619004736246616, 0.016800381450651318, 1.8921978305199946e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4553912.363352443, 10.835973545147, 0.016005251475292706, 1.907179227136742e-06],
                [1132.8146763631971, 1259.7696328062204, 1424.6533094655529, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.869052328516, 0.01508497956653298, 1.8557507186605142e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016564172329980215, 1.64751e-06],
                [1134.3052340247398, 1261.1141597856492, 1430.258879312566, 2914544.9227398997, 6153349.653974743,
                 4700731.238906128, 9.427550057683757, 0.01638956964339133, 1.8465579641647771e-06],
                [1084.4879631318756, 1257.584793961963, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1165.3708118481113, 1284.2421194708988, 1425.021955085407, 2854778.224454453, 5290343.222690649,
                 6214366.85519229, 10.843097036079232, 0.016533863886783313, 1.8687363845527149e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5389154.533109349,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.685798387702239e-06],
                [1114.834792035313, 1251.535389764655, 1424.6533094655529, 2577630.7506229677, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.1006908836548, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1114.834792035313, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6499915.346484166, 9.78692463749932, 0.014655133358103744, 1.6455601114196722e-06],
                [1115.466323398068, 1230.6795203644804, 1432.8952971468889, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.779007103319314, 0.017536941207474163, 1.6520170993838028e-06],
                [1052.19590528635, 1257.584793961963, 1309.230014960785, 2711709.7515290566, 5480170.304487934,
                 6125197.144713911, 8.38204902772919, 0.013691625192435415, 1.6470620324356985e-06],
                [1115.466323398068, 1230.6795203644804, 1434.6272470209367, 2767648.6297825263, 5209744.475869942,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1116.5269344841006, 1256.6840668527575, 1424.6533094655529, 2712885.4996207994, 5601130.432463982,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1114.9126539360466, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1114.1144487688819, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.871620683775097, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.014995216559648368, 1.64441131717817e-06],
                [1114.834792035313, 1251.3768537744033, 1424.6533094655529, 2536154.9667941616, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1177.9577488540551, 1257.606216652866, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1168.7843286919008, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5151932.1545183025,
                 6252960.904134171, 11.437514624265415, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2672187.6112241857, 5194929.277645023,
                 5955635.622891837, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1115.466323398068, 1257.584793961963, 1425.5175661985572, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016456322052715848, 1.6551951573792323e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.7134895117258869e-06],
                [1124.392389981206, 1260.5869822935006, 1419.626242400125, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1132.8146763631971, 1259.7696328062204, 1424.6533094655529, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.869052328516, 0.01508497956653298, 1.8557507186605142e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4553912.363352443, 10.835973545147, 0.01575636609759392, 1.907179227136742e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 6150052.4350188095, 9.788257071626864, 0.01710991733724194, 1.653526206759371e-06],
                [1084.4879631318756, 1257.584793961963, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1086.181122815101, 1259.3587163655109, 1424.4013736288516, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.64751e-06],
                [1105.7606376509357, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.934894193259877, 0.014655133358103744, 1.64751e-06],
                [1173.760206680501, 1241.2917167358653, 1430.9982219213007, 3130727.9042818854, 3611549.2338148607,
                 5244960.813968207, 11.216198135126099, 0.01552981782318728, 2.313183931715978e-06],
                [1086.8792226662247, 1257.584793961963, 1424.6482539664976, 2677849.96572248, 5177667.749296468,
                 6170892.134969252, 10.843270131540745, 0.016533863886783313, 1.8921978305199946e-06],
                [1114.834792035313, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6462873.262083922, 9.78692463749932, 0.014655133358103744, 1.64751e-06],
                [1162.985408391334, 1271.107772815279, 1426.265554917064, 2849919.1458090344, 6292860.96194586,
                 4553912.363352443, 10.835973545147, 0.016005251475292706, 1.907179227136742e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5125021.87212479,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1133.0499588683645, 1228.5392853023554, 1410.8868937934, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016307724969914394, 1.64751e-06],
                [1086.181122815101, 1259.3587163655109, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016564172329980215, 1.64751e-06],
                [1162.985408391334, 1271.9251223025587, 1426.265554917064, 2849919.1458090344, 6572214.362356157,
                 4700731.238906128, 10.141368020101972, 0.01581701564809486, 1.65389119410361e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5868217.479161575, 9.787113063154044, 0.016750975918697033, 1.6520170993838028e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 6125197.144713911, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1107.3715710091938, 1260.5869822935006, 1420.7016199249624, 2791507.522811302, 5151932.1545183025,
                 6216074.834299484, 10.053256657600741, 0.01508497956653298, 1.7221590832771696e-06],
                [1052.19590528635, 1257.7433299522147, 1311.5054571133896, 2716201.931261375, 5389154.533109349,
                 6125197.144713911, 8.41622197839763, 0.013733748116950906, 1.685798387702239e-06],
                [1107.3715710091938, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6194661.429560538, 10.053256657600741, 0.01508497956653298, 1.6520168892140052e-06],
                [1003.755240634562, 1257.584793961963, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 5479939.5493022995, 9.867582517813236, 0.016750155641775035, 1.6399981991408293e-06],
                [1134.4136265483053, 1257.584793961963, 1442.1439452584866, 2677849.96572248, 5177667.749296468,
                 6170892.134969252, 10.619004736246616, 0.017158031465326264, 1.8921978305199946e-06],
                [1165.3708118481113, 1284.2421194708988, 1424.6533094655529, 2854778.224454453, 5290343.222690649,
                 6214366.85519229, 10.377362233237546, 0.016533863886783313, 1.8687363845527149e-06],
                [1165.3708118481113, 1284.2421194708988, 1425.021955085407, 2854778.224454453, 5290343.222690649,
                 6214366.85519229, 10.843097036079232, 0.016533863886783313, 1.869440100161175e-06],
                [1165.3708118481113, 1284.2421194708988, 1425.021955085407, 2854778.224454453, 5290343.222690649,
                 6214366.85519229, 10.843097036079232, 0.016533863886783313, 1.8687363845527149e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5389154.533109349,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.685798387702239e-06],
                [1114.834792035313, 1251.535389764655, 1424.6533094655529, 2577630.7506229677, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.7134895117258869e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.1006908836548, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1114.834792035313, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6499915.346484166, 9.78692463749932, 0.014655133358103744, 1.6455601114196722e-06],
                [1115.466323398068, 1230.6795203644804, 1432.8952971468889, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.779007103319314, 0.017536941207474163, 1.6520170993838028e-06],
                [1115.466323398068, 1230.6795203644804, 1434.6272470209367, 2767648.6297825263, 5209744.475869942,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1052.19590528635, 1257.584793961963, 1309.230014960785, 2711709.7515290566, 5480170.304487934,
                 6125197.144713911, 8.38204902772919, 0.013691625192435415, 1.6470620324356985e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.3704468752608303e-06],
                [1084.4879631318756, 1257.584793961963, 1424.6595264578891, 2516749.3841471686, 5151835.089616997,
                 5827511.644970751, 11.162586033172255, 0.015983618896994494, 2.361669097945957e-06],
                [1114.1144487688819, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.871620683775097, 0.01508497956653298, 1.64751e-06],
                [1116.5269344841006, 1256.6840668527575, 1424.6533094655529, 2712885.4996207994, 5601130.432463982,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1114.9126539360466, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.014995216559648368, 1.64441131717817e-06],
                [1177.9577488540551, 1257.606216652866, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1114.834792035313, 1251.3768537744033, 1424.6533094655529, 2536154.9667941616, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2672187.6112241857, 5194929.277645023,
                 5955635.622891837, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1168.7843286919008, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5151932.1545183025,
                 6252960.904134171, 11.437514624265415, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1257.584793961963, 1425.5175661985572, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016456322052715848, 1.6551951573792323e-06],
                [1124.392389981206, 1260.5869822935006, 1419.626242400125, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1086.8792226662247, 1257.584793961963, 1424.6482539664976, 2677849.96572248, 5177667.749296468,
                 6170892.134969252, 10.843270131540745, 0.016533863886783313, 1.8921978305199946e-06],
                [1191.5746337744677, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.377362233237546, 0.017509170428913215, 1.8770823454840815e-06],
                [1084.4879631318756, 1257.584793961963, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1086.181122815101, 1259.3587163655109, 1424.4013736288516, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1133.0499588683645, 1228.5392853023554, 1410.8868937934, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016307724969914394, 1.64751e-06],
                [1172.1322901571223, 1257.606216652866, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1085.9932464823125, 1259.3587163655109, 1417.0087573032533, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.64751e-06],
                [1114.834792035313, 1251.727762110881, 1424.6533094655529, 2536154.9667941616, 5482838.960861618,
                 6398907.147332583, 11.505648022330274, 0.015262337423596692, 1.7225270413854408e-06],
                [1105.7606376509357, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.934894193259877, 0.014655133358103744, 1.64751e-06],
                [1132.8146763631971, 1259.7696328062204, 1424.6533094655529, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.869052328516, 0.01508497956653298, 1.8557507186605142e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.8921978305199946e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1110.9471928513656, 1251.535389764655, 1424.6533094655529, 2592702.2039177953, 5482838.960861618,
                 6398907.147332583, 11.181097051754435, 0.015262337423596692, 1.64751e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5125021.87212479,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1086.181122815101, 1259.3587163655109, 1424.6533094655529, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5868217.479161575, 9.787113063154044, 0.016750975918697033, 1.6520170993838028e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5196368.203882732,
                 6321047.275851353, 9.869052328516, 0.015465136736984927, 1.64751e-06],
                [1105.7606376509357, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5180292.022685467,
                 6462873.262083922, 9.934894193259877, 0.014655133358103744, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5389154.533109349,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.685798387702239e-06],
                [1114.834792035313, 1251.535389764655, 1424.6533094655529, 2577630.7506229677, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.7134895117258869e-06],
                [1114.834792035313, 1270.1788066747688, 1424.6533094655529, 2674149.783107711, 5177667.749296468,
                 6499915.346484166, 9.78692463749932, 0.014655133358103744, 1.6455601114196722e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.1006908836548, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1115.466323398068, 1230.6795203644804, 1432.8952971468889, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.779007103319314, 0.017536941207474163, 1.6520170993838028e-06],
                [1115.466323398068, 1230.6795203644804, 1434.6272470209367, 2767648.6297825263, 5209744.475869942,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1052.19590528635, 1257.584793961963, 1309.230014960785, 2711709.7515290566, 5480170.304487934,
                 6125197.144713911, 8.38204902772919, 0.013691625192435415, 1.6470620324356985e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1114.1144487688819, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6125197.144713911, 9.871620683775097, 0.01508497956653298, 1.64751e-06],
                [1116.5269344841006, 1256.6840668527575, 1424.6533094655529, 2712885.4996207994, 5601130.432463982,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1114.9126539360466, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1124.392389981206, 1260.5869822935006, 1419.626242400125, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1177.9577488540551, 1257.606216652866, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1084.4879631318756, 1257.584793961963, 1424.6595264578891, 2516749.3841471686, 5151835.089616997,
                 5827511.644970751, 11.162586033172255, 0.015983618896994494, 2.361669097945957e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.014995216559648368, 1.64441131717817e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2672187.6112241857, 5194929.277645023,
                 5955635.622891837, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1114.834792035313, 1251.3768537744033, 1424.6533094655529, 2536154.9667941616, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1168.7843286919008, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5151932.1545183025,
                 6252960.904134171, 11.437514624265415, 0.01508497956653298, 1.64751e-06],
                [1115.466323398068, 1257.584793961963, 1425.5175661985572, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 9.869052328516, 0.016456322052715848, 1.6551951573792323e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1084.4879631318756, 1250.5317620004655, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1086.8792226662247, 1257.584793961963, 1424.6482539664976, 2677849.96572248, 5177667.749296468,
                 6170892.134969252, 10.843270131540745, 0.016533863886783313, 1.8921978305199946e-06],
                [1191.5746337744677, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.377362233237546, 0.017509170428913215, 1.8770823454840815e-06],
                [1084.4879631318756, 1257.584793961963, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1086.181122815101, 1259.3587163655109, 1424.4013736288516, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.64751e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1105.7606376509357, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.934894193259877, 0.014655133358103744, 1.64751e-06],
                [1133.0499588683645, 1228.5392853023554, 1410.8868937934, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016307724969914394, 1.64751e-06],
                [1114.834792035313, 1251.727762110881, 1424.6533094655529, 2536154.9667941616, 5417386.548458506,
                 6398907.147332583, 11.505648022330274, 0.01523677004163477, 1.7225270413854408e-06],
                [1085.9932464823125, 1259.3587163655109, 1417.0087573032533, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.64751e-06],
                [1132.8146763631971, 1259.7696328062204, 1424.6533094655529, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.869052328516, 0.01508497956653298, 1.8557507186605142e-06],
                [1172.1322901571223, 1257.606216652866, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.8921978305199946e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5178353.355033892,
                 6346803.504583463, 9.74030460064988, 0.01508497956653298, 1.676386974185345e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.9824134885632, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1116.5269344841006, 1264.399609988531, 1424.6533094655529, 2711709.7515290566, 5677614.129084032,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1115.0621708467986, 1249.4740610346707, 1424.6533094655529, 2711709.7515290566, 5196368.203882732,
                 6547488.591648434, 10.149794340877818, 0.015465136736984927, 1.64751e-06],
                [1115.466323398068, 1266.8929341135959, 1424.678949560473, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 10.468813326349617, 0.016456322052715848, 1.647462780095882e-06],
                [1110.9471928513656, 1251.535389764655, 1424.6533094655529, 2592702.2039177953, 5482838.960861618,
                 6398907.147332583, 11.181097051754435, 0.015262337423596692, 1.64751e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.3704468752608303e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5125021.87212479,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1132.4693790973101, 1260.5869822935006, 1419.626242400125, 2939142.8233972667, 5093097.959680135,
                 6045229.56246013, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.8921978305199946e-06],
                [1191.5746337744677, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.377362233237546, 0.017509170428913215, 1.8770823454840815e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.869052328516, 0.014995216559648368, 1.64441131717817e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5178353.355033892,
                 6346803.504583463, 9.830302894689721, 0.01508497956653298, 1.676386974185345e-06],
                [1124.392389981206, 1260.5869822935006, 1424.4547665346258, 2711709.7515290566, 5151932.1545183025,
                 6383368.992652747, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.9824134885632, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.1006908836548, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.7134895117258869e-06],
                [1105.3288280099578, 1230.6795203644804, 1432.8952971468889, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.779007103319314, 0.017536941207474163, 1.6498374225713848e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5125021.87212479,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1115.466323398068, 1230.6795203644804, 1434.6272470209367, 2767648.6297825263, 5209744.475869942,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1052.19590528635, 1257.584793961963, 1309.230014960785, 2711709.7515290566, 5480170.304487934,
                 6125197.144713911, 8.38204902772919, 0.013691625192435415, 1.6470620324356985e-06],
                [1052.19590528635, 1257.584793961963, 1311.5054571133896, 2711709.7515290566, 5389154.533109349,
                 6125197.144713911, 8.41622197839763, 0.013691625192435415, 1.685798387702239e-06],
                [1084.4879631318756, 1250.5317620004655, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1110.9471928513656, 1251.535389764655, 1424.6533094655529, 2592702.2039177953, 5482838.960861618,
                 6398907.147332583, 11.181097051754435, 0.015262337423596692, 1.64751e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1133.0499588683645, 1228.5392853023554, 1410.8868937934, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016307724969914394, 1.64751e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1084.4879631318756, 1257.584793961963, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.836883214384672, 0.016533863886783313, 2.361669097945957e-06],
                [1132.8146763631971, 1259.7696328062204, 1424.6533094655529, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.869052328516, 0.01508497956653298, 1.8557507186605142e-06],
                [1115.466323398068, 1266.8929341135959, 1424.678949560473, 2711709.7515290566, 5422188.248604289,
                 6125197.144713911, 10.468813326349617, 0.016456322052715848, 1.647462780095882e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1114.9126539360466, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1257.606216652866, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1114.834792035313, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1116.5269344841006, 1256.6840668527575, 1424.6533094655529, 2713757.049709839, 5601130.432463982,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1080.8218098103202, 1267.6616708885738, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6352499.037384299, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1086.209168872005, 1259.3357160096296, 1432.7260303878165, 2707317.764722687, 5151932.1545183025,
                 6243119.477172591, 9.910590583182286, 0.01508497956653298, 1.7134895117258869e-06],
                [1085.9932464823125, 1259.3587163655109, 1417.0087573032533, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.64751e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.721496417372578, 0.01508497956653298, 1.644075174710523e-06],
                [1115.466323398068, 1246.5797646495644, 1433.9202379795624, 2672187.6112241857, 5194929.277645023,
                 5955635.622891837, 9.788257071626864, 0.016750975918697033, 1.653526206759371e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1114.834792035313, 1251.3768537744033, 1424.6533094655529, 2536154.9667941616, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1112.7575720729085, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.87543924567381, 0.01508497956653298, 1.64751e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1116.5269344841006, 1256.6840668527575, 1424.6533094655529, 2712885.4996207994, 5601130.432463982,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1110.9471928513656, 1251.535389764655, 1424.6533094655529, 2592702.2039177953, 5482838.960861618,
                 6398907.147332583, 11.181097051754435, 0.015262337423596692, 1.64751e-06],
                [1084.4879631318756, 1257.584793961963, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1099.7819923693667, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.934894193259877, 0.01457779379810453, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1132.4693790973101, 1260.5869822935006, 1419.626242400125, 2939142.8233972667, 5093097.959680135,
                 6045229.56246013, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1114.834792035313, 1260.7338595060503, 1434.484774021517, 2521375.1087238477, 5318775.920489855,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.7036543310582546e-06],
                [1191.5746337744677, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.377362233237546, 0.017509170428913215, 1.8770823454840815e-06],
                [1115.466323398068, 1230.6795203644804, 1434.6272470209367, 2767648.6297825263, 5209744.475869942,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1124.392389981206, 1260.5869822935006, 1424.4547665346258, 2711709.7515290566, 5151932.1545183025,
                 6383368.992652747, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1085.9932464823125, 1259.3587163655109, 1417.0087573032533, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.9959581371526, 0.01491459237885602, 1.64751e-06],
                [1128.6500541303647, 1269.9796387467936, 1430.0648310577762, 2711709.7515290566, 5178353.355033892,
                 6346803.504583463, 9.830302894689721, 0.01508497956653298, 1.676386974185345e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.7134895117258869e-06],
                [1052.19590528635, 1257.584793961963, 1309.230014960785, 2711709.7515290566, 5480170.304487934,
                 6125197.144713911, 8.38204902772919, 0.013691625192435415, 1.6470620324356985e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5125021.87212479,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1132.8146763631971, 1259.7696328062204, 1424.6533094655529, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.869052328516, 0.01508497956653298, 1.8557507186605142e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1105.3288280099578, 1230.6795203644804, 1432.8952971468889, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.779007103319314, 0.017536941207474163, 1.6498374225713848e-06],
                [1114.834792035313, 1260.5869822935006, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.01508497956653298, 1.6470599805134658e-06],
                [1112.7575720729085, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.87543924567381, 0.01508497956653298, 1.64751e-06],
                [1084.4879631318756, 1250.5317620004655, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1133.0499588683645, 1228.5392853023554, 1410.8868937934, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016307724969914394, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.8921978305199946e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1110.9471928513656, 1251.535389764655, 1424.6533094655529, 2592702.2039177953, 5482838.960861618,
                 6398907.147332583, 11.181097051754435, 0.015262337423596692, 1.64751e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6917187201999296e-06],
                [1114.9126539360466, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1099.7819923693667, 1270.1788066747688, 1424.5392743903153, 2683191.651900613, 5139556.741381487,
                 6462873.262083922, 9.934894193259877, 0.01457779379810453, 1.64751e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.1006908836548, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.721496417372578, 0.01508497956653298, 1.644075174710523e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1251.3768537744033, 1424.6533094655529, 2536154.9667941616, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1171.1989534324357, 1247.317850560859, 1428.5163072367866, 3130727.9042818854, 3393993.1373124444,
                 5235964.403682908, 11.394289259455535, 0.015606591914183083, 2.371767469954206e-06],
                [1054.1924321162223, 1257.606216652866, 1424.5249718058972, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 9.895228528838377, 0.01508497956653298, 1.6586378910639396e-06],
                [1084.4879631318756, 1257.584793961963, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.836883214384672, 0.016533863886783313, 2.361669097945957e-06],
                [1115.466323398068, 1230.6795203644804, 1433.9202379795624, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1085.9932464823125, 1259.3587163655109, 1417.0087573032533, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.64751e-06],
                [1116.5269344841006, 1256.6840668527575, 1424.6533094655529, 2712885.4996207994, 5601130.432463982,
                 6149826.31395066, 9.867582517813236, 0.016126488948190316, 1.6422729938108779e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.9824134885632, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5253162.6621970115,
                 6532829.441213693, 9.405840826327404, 0.01508497956653298, 1.644075174710523e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1099.7819923693667, 1270.1788066747688, 1424.5392743903153, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.934894193259877, 0.01457779379810453, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1104.1468138489206, 1270.1788066747688, 1423.0493808748647, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.934894193259877, 0.01457779379810453, 1.6838594647207929e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1132.4693790973101, 1260.5869822935006, 1419.626242400125, 2939142.8233972667, 5093097.959680135,
                 6045229.56246013, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1115.466323398068, 1230.6795203644804, 1434.6272470209367, 2767648.6297825263, 5209744.475869942,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1191.5746337744677, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.377362233237546, 0.017509170428913215, 1.8770823454840815e-06],
                [1114.834792035313, 1260.7338595060503, 1434.484774021517, 2521375.1087238477, 5318775.920489855,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.7036543310582546e-06],
                [1115.466323398068, 1246.5797646495644, 1424.6533094655529, 2767648.6297825263, 5125021.87212479,
                 6125197.144713911, 9.788257071626864, 0.01678981800236394, 1.64751e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1085.9932464823125, 1259.3587163655109, 1417.0087573032533, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.9959581371526, 0.01491459237885602, 1.64751e-06],
                [1132.8146763631971, 1259.7696328062204, 1424.6533094655529, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.869052328516, 0.01508497956653298, 1.8557507186605142e-06],
                [1116.1148589813238, 1260.5869822935006, 1417.6540141389703, 2727408.669573044, 5151932.1545183025,
                 6346803.504583463, 9.859496987181, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.910590583182286, 0.01508497956653298, 1.7134895117258869e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6917187201999296e-06],
                [1133.0499588683645, 1228.5392853023554, 1410.8868937934, 2767648.6297825263, 5177667.749296468,
                 6125197.144713911, 9.788257071626864, 0.016307724969914394, 1.64751e-06],
                [1084.4879631318756, 1250.5317620004655, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1124.392389981206, 1260.5869822935006, 1424.4547665346258, 2711709.7515290566, 5151932.1545183025,
                 6383368.992652747, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1104.1468138489206, 1269.1713094568, 1400.5041300649148, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.930403598854456, 0.014610675464065541, 1.6838594647207929e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1105.3288280099578, 1230.6795203644804, 1432.8952971468889, 2767648.6297825263, 5194929.277645023,
                 5743715.795215454, 9.779007103319314, 0.017536941207474163, 1.6498374225713848e-06],
                [1112.7575720729085, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.87543924567381, 0.01508497956653298, 1.64751e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.8921978305199946e-06],
                [1080.3528117556502, 1257.584793961963, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.798206749609951, 0.016533863886783313, 2.3951257876327254e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6055568.81090391, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1183.0124979874236, 1257.760750259536, 1424.6465526495679, 2756948.118884614, 4914204.915875496,
                 6252781.810184416, 11.382355790415062, 0.014569681251776666, 1.6475084370490054e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1128.6500541303647, 1269.9796387467936, 1430.0648310577762, 2711709.7515290566, 5178353.355033892,
                 6346803.504583463, 9.830302894689721, 0.01508497956653298, 1.676386974185345e-06],
                [1052.19590528635, 1257.584793961963, 1309.230014960785, 2711709.7515290566, 5480170.304487934,
                 6125197.144713911, 8.38204902772919, 0.013691625192435415, 1.6470620324356985e-06],
                [1104.1468138489206, 1270.1788066747688, 1423.0493808748647, 2680818.9215573217, 5176513.48526453,
                 6456417.596046858, 9.934894193259877, 0.01457779379810453, 1.6706443467875595e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.9824134885632, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1183.39613591346, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1136.9294137228587, 1231.6287501442703, 1423.6910978207807, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252954.02610822, 11.395208016578668, 0.015080737923366064, 1.6475084370490054e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6362919.999533043, 9.575562416329118, 0.01508497956653298, 1.7134895117258869e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5083580.380295775,
                 6252960.904134171, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1110.9471928513656, 1251.535389764655, 1424.6533094655529, 2592702.2039177953, 5482838.960861618,
                 6398907.147332583, 11.181097051754435, 0.015262337423596692, 1.64751e-06],
                [1126.7449349520505, 1269.9796387467936, 1420.7022991611589, 2711709.7515290566, 5253162.6621970115,
                 6505858.782087919, 9.405840826327404, 0.01506930691030413, 1.6438808688553816e-06],
                [1114.834792035313, 1260.5869822935006, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.01508497956653298, 1.6470599805134658e-06],
                [1114.834792035313, 1251.3768537744033, 1424.6533094655529, 2536154.9667941616, 5482838.960861618,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.64751e-06],
                [1109.7223419726379, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6866917168224695e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1136.9294137228587, 1231.6287501442703, 1423.6910978207807, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1183.39613591346, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1132.8146763631971, 1259.7696328062204, 1431.129135340821, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8805722333904149e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5133273.89347622,
                 6015547.613055304, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6362919.999533043, 9.575562416329118, 0.01508497956653298, 1.7134895117258869e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1114.834792035313, 1260.5869822935006, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.01508497956653298, 1.6470599805134658e-06],
                [1080.3528117556502, 1257.584793961963, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.798206749609951, 0.016533863886783313, 2.3951257876327254e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6917187201999296e-06],
                [1114.834792035313, 1260.7338595060503, 1434.484774021517, 2521375.1087238477, 5318775.920489855,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.7036543310582546e-06],
                [1116.1478053426872, 1261.417411886347, 1424.6533094655529, 2536154.9667941616, 5631058.522618111,
                 6398907.147332583, 11.36144205775584, 0.015416514215957088, 1.64751e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.01461627638890642, 1.6475084370490054e-06],
                [1104.1468138489206, 1269.1713094568, 1400.5041300649148, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.930403598854456, 0.014610675464065541, 1.6838594647207929e-06],
                [1183.0124979874236, 1257.760750259536, 1424.6465526495679, 2756948.118884614, 4914204.915875496,
                 6252781.810184416, 11.382355790415062, 0.014569681251776666, 1.6475084370490054e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1179.6176953978943, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6150498.254348851, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1115.466323398068, 1230.6795203644804, 1434.6272470209367, 2767648.6297825263, 5209744.475869942,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1121.0245085454426, 1250.5464241815566, 1419.0316620837257, 2727408.669573044, 5151932.1545183025,
                 6346803.504583463, 9.859496987181, 0.01508497956653298, 1.64751e-06],
                [1191.5746337744677, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.377362233237546, 0.017509170428913215, 1.8770823454840815e-06],
                [1109.7223419726379, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6866917168224695e-06],
                [1136.9294137228587, 1233.4172198509577, 1438.7200703551343, 2852707.4289400186, 5760607.364900568,
                 4684620.5529705705, 9.337909358709057, 0.016540351152655584, 1.8676654427795434e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.9824134885632, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5356946.282826636,
                 6055568.81090391, 9.962874958185285, 0.014989949707629712, 1.64441131717817e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1183.39613591346, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1085.9932464823125, 1259.3587163655109, 1417.0087573032533, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.9959581371526, 0.01491459237885602, 1.64751e-06],
                [1114.9126539360466, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.64751e-06],
                [1100.1373452862736, 1267.3828396561273, 1400.5041300649148, 2680818.9215573217, 5176513.48526453,
                 6478960.857998105, 9.800254974436823, 0.014610675464065541, 1.6838594647207929e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252954.02610822, 11.395208016578668, 0.015080737923366064, 1.6475084370490054e-06],
                [1052.19590528635, 1257.584793961963, 1309.230014960785, 2711709.7515290566, 5480170.304487934,
                 6125197.144713911, 8.38204902772919, 0.013691625192435415, 1.6470620324356985e-06],
                [1084.4879631318756, 1250.5317620004655, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1132.8146763631971, 1259.7696328062204, 1424.6533094655529, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.869052328516, 0.01508497956653298, 1.8557507186605142e-06],
                [1079.1784194680245, 1259.3587163655109, 1419.0880406850756, 2706163.328552395, 5151932.1545183025,
                 5956595.165526332, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.8921978305199946e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.9236251304644705e-06],
                [1132.4968544972521, 1260.5869822935006, 1420.7016199249624, 2775485.1349959243, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6760084475117078e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6055568.81090391, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1133.768536113456, 1260.5869822935006, 1424.6533094655529, 2711435.0608179937, 5177667.749296468,
                 6056468.691639014, 9.869052328516, 0.014127260546408704, 1.64751e-06],
                [1110.9471928513656, 1251.535389764655, 1424.6533094655529, 2592702.2039177953, 5482838.960861618,
                 6398907.147332583, 11.181097051754435, 0.015262337423596692, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1132.8146763631971, 1259.7696328062204, 1431.129135340821, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8805722333904149e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5133273.89347622,
                 6015547.613055304, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1183.39613591346, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6362919.999533043, 9.575562416329118, 0.01508497956653298, 1.7134895117258869e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.01461627638890642, 1.6475084370490054e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6917187201999296e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1183.0124979874236, 1257.760750259536, 1424.6465526495679, 2756948.118884614, 4914204.915875496,
                 6252781.810184416, 11.382355790415062, 0.014569681251776666, 1.6475084370490054e-06],
                [1080.3528117556502, 1257.584793961963, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.798206749609951, 0.016533863886783313, 2.3951257876327254e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1104.1468138489206, 1269.1713094568, 1400.5041300649148, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.930403598854456, 0.014610675464065541, 1.6838594647207929e-06],
                [1115.466323398068, 1230.6795203644804, 1434.6272470209367, 2767648.6297825263, 5209744.475869942,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1121.0245085454426, 1250.5464241815566, 1419.0316620837257, 2727408.669573044, 5151932.1545183025,
                 6346803.504583463, 9.859496987181, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.7338595060503, 1434.484774021517, 2521375.1087238477, 5318775.920489855,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.7036543310582546e-06],
                [1121.8684840370697, 1251.118059868417, 1445.7770632611023, 2680186.09729148, 5148651.658242081,
                 6346803.504583463, 9.859496987181, 0.015088105294695333, 1.639798179111424e-06],
                [1179.6176953978943, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1183.39613591346, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1114.834792035313, 1260.5869822935006, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.01508497956653298, 1.6470599805134658e-06],
                [1116.1478053426872, 1261.417411886347, 1424.6533094655529, 2536154.9667941616, 5631058.522618111,
                 6398907.147332583, 11.36144205775584, 0.015416514215957088, 1.64751e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1100.1373452862736, 1267.3828396561273, 1400.5041300649148, 2680818.9215573217, 5176446.708119623,
                 6478960.857998105, 9.800254974436823, 0.014610675464065541, 1.6838594647207929e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1171.2883088636754, 1260.684467035174, 1418.9934367939197, 2630813.4034174774, 5637476.003693504,
                 6253461.142118783, 11.605501217123276, 0.015271687794596967, 1.646211033468178e-06],
                [1084.4879631318756, 1250.5317620004655, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.9824134885632, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5356946.282826636,
                 6055568.81090391, 9.962874958185285, 0.014989949707629712, 1.64441131717817e-06],
                [1079.1784194680245, 1259.3587163655109, 1419.0880406850756, 2706163.328552395, 5151932.1545183025,
                 5956595.165526332, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1109.7223419726379, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6866917168224695e-06],
                [1172.1322901571223, 1261.2561027220345, 1442.854182165886, 2657348.204338886, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1191.5746337744677, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.377362233237546, 0.017509170428913215, 1.8770823454840815e-06],
                [1080.3528117556502, 1265.7237988058796, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.798206749609951, 0.016533863886783313, 2.362014103822307e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252954.02610822, 11.395208016578668, 0.015080737923366064, 1.6693720419408488e-06],
                [1136.9294137228587, 1233.4172198509577, 1438.7200703551343, 2852707.4289400186, 5760607.364900568,
                 4684620.5529705705, 9.337909358709057, 0.016540351152655584, 1.8676654427795434e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5460891.129329316,
                 6015547.613055304, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1183.39613591346, 1240.6584438348118, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1102.1152991124782, 1257.5624747055258, 1310.4803905865347, 2711709.7515290566, 5167041.861010327,
                 6149853.921081831, 9.837636314132341, 0.01462714264058961, 1.6997971589837513e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.8921978305199946e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.9236251304644705e-06],
                [1084.4879631318756, 1250.5317620004655, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1109.7223419726379, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5225870.872202378,
                 6058545.49793099, 9.869052328516, 0.015398836251961747, 1.6866917168224695e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5133273.89347622,
                 6015547.613055304, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1132.8146763631971, 1259.7696328062204, 1431.129135340821, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8805722333904149e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6362919.999533043, 9.575562416329118, 0.01508497956653298, 1.7134895117258869e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.01461627638890642, 1.6475084370490054e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1179.6176953978943, 1249.0062677413969, 1424.647927352167, 2662699.2546734824, 5382138.607390853,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1104.1468138489206, 1269.1713094568, 1400.5041300649148, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.930403598854456, 0.014610675464065541, 1.6838594647207929e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6917187201999296e-06],
                [1183.39613591346, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1079.1784194680245, 1259.3587163655109, 1419.0880406850756, 2706163.328552395, 5151932.1545183025,
                 5956595.165526332, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1183.0124979874236, 1257.760750259536, 1424.6465526495679, 2756948.118884614, 4914204.915875496,
                 6252781.810184416, 11.382355790415062, 0.014569681251776666, 1.6475084370490054e-06],
                [1109.7223419726379, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6866917168224695e-06],
                [1191.5746337744677, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.377362233237546, 0.017509170428913215, 1.8770823454840815e-06],
                [1121.8684840370697, 1251.118059868417, 1445.7770632611023, 2680186.09729148, 5148651.658242081,
                 6346803.504583463, 9.859496987181, 0.015088105294695333, 1.639798179111424e-06],
                [1183.39613591346, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1171.2883088636754, 1260.684467035174, 1418.9934367939197, 2630813.4034174774, 5637476.003693504,
                 6253461.142118783, 11.605501217123276, 0.015271687794596967, 1.646211033468178e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1080.3528117556502, 1265.7237988058796, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.798206749609951, 0.016533863886783313, 2.362014103822307e-06],
                [1136.9294137228587, 1231.6287501442703, 1443.9824134885632, 2852707.4289400186, 5760607.364900568,
                 4700731.238906128, 9.36332541177229, 0.016540351152655584, 1.8676654427795434e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1080.3528117556502, 1257.584793961963, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.798206749609951, 0.016533863886783313, 2.3951257876327254e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5356946.282826636,
                 6055568.81090391, 9.962874958185285, 0.014989949707629712, 1.64441131717817e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1115.466323398068, 1230.6795203644804, 1434.6272470209367, 2767648.6297825263, 5209744.475869942,
                 5743715.795215454, 9.787113063154044, 0.017536941207474163, 1.6520170993838028e-06],
                [1102.1152991124782, 1257.5624747055258, 1310.4803905865347, 2711709.7515290566, 5167041.861010327,
                 6149853.921081831, 9.837636314132341, 0.01462714264058961, 1.6997971589837513e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.9236251304644705e-06],
                [1079.1784194680245, 1259.3587163655109, 1419.0880406850756, 2706163.328552395, 5151932.1545183025,
                 5956595.165526332, 9.97531880551067, 0.01491459237885602, 1.6463053082056679e-06],
                [1180.7233642687456, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1178.9096100025354, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1172.1322901571223, 1261.2561027220345, 1442.854182165886, 2657348.204338886, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.01508497956653298, 1.64751e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5460891.129329316,
                 6015547.613055304, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1146.7253083068827, 1257.584793961963, 1442.1439452584866, 2712014.0081000105, 5178449.653957896,
                 6170892.134969252, 10.619004736246616, 0.01731720826118376, 1.8921978305199946e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252954.02610822, 11.395208016578668, 0.015080737923366064, 1.6693720419408488e-06],
                [1115.400748739093, 1262.2626810628155, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.015227144226188045, 1.6470599805134658e-06],
                [1109.7223419726379, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5225870.872202378,
                 6058545.49793099, 9.869052328516, 0.015398836251961747, 1.6866917168224695e-06],
                [1114.834792035313, 1260.7338595060503, 1434.484774021517, 2521375.1087238477, 5318775.920489855,
                 6398907.147332583, 11.362419230865815, 0.015262337423596692, 1.7036543310582546e-06],
                [1084.4879631318756, 1250.5317620004655, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1100.1373452862736, 1267.3828396561273, 1400.5041300649148, 2680818.9215573217, 5176446.708119623,
                 6478960.857998105, 9.800254974436823, 0.014610675464065541, 1.6838594647207929e-06],
                [1181.8203100646904, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252954.02610822, 11.395208016578668, 0.015080737923366064, 1.6847380224448772e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1102.1152991124782, 1257.5624747055258, 1310.4803905865347, 2711709.7515290566, 5167041.861010327,
                 6149853.921081831, 9.837636314132341, 0.01462714264058961, 1.6997971589837513e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 5999071.985613948, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1115.400748739093, 1262.2626810628155, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.015227144226188045, 1.6470599805134658e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1178.4227399172075, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.417802244505165, 0.01508497956653298, 1.6571644711735225e-06],
                [1079.1784194680245, 1259.3587163655109, 1419.0880406850756, 2706163.328552395, 5151932.1545183025,
                 5956595.165526332, 9.97531880551067, 0.01491459237885602, 1.6463053082056679e-06],
                [1172.1322901571223, 1261.2561027220345, 1442.854182165886, 2657348.204338886, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5133273.89347622,
                 6015547.613055304, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1084.4879631318756, 1250.5317620004655, 1425.0566077670053, 2516749.3841471686, 5151835.089616997,
                 6051450.579799658, 10.843270131540745, 0.016533863886783313, 2.361669097945957e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1110.0932897166533, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5216394.350278399,
                 6382146.572286638, 10.37995071911957, 0.015398836251961747, 1.6866917168224695e-06],
                [1191.5746337744677, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.184880722099962, 0.017509170428913215, 1.928535330675258e-06],
                [1110.1175336282156, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1080.3528117556502, 1265.7237988058796, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.798206749609951, 0.016533863886783313, 2.362014103822307e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1185.931351058013, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6343949.916530928, 11.54839380449381, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.920980046961, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.014332511765226055, 1.64751e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6204147.819576919, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1183.39613591346, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6917187201999296e-06],
                [1133.9089419004854, 1259.7696328062204, 1436.8742705057894, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8805722333904149e-06],
                [1181.8203100646904, 1260.1461808350282, 1413.9897307285646, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.395208016578668, 0.015392065619194425, 1.64751e-06],
                [1128.7618876139773, 1249.0468225853215, 1445.7770632611023, 2680186.09729148, 5148651.658242081,
                 6346803.504583463, 9.859496987181, 0.015088105294695333, 1.639798179111424e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1179.6176953978943, 1248.028182449465, 1424.647927352167, 2672069.8742411183, 5382138.607390853,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1178.9096100025354, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1160.6095060902956, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 11.561826576709295, 0.01508497956653298, 1.6453299579647577e-06],
                [1171.2883088636754, 1260.684467035174, 1418.9934367939197, 2630813.4034174774, 5637476.003693504,
                 6253461.142118783, 11.605501217123276, 0.015271687794596967, 1.646211033468178e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1079.1784194680245, 1259.3587163655109, 1419.0880406850756, 2706163.328552395, 5151932.1545183025,
                 5956595.165526332, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6009864.991404326, 11.605501217123276, 0.015129553230613428, 1.746960716329272e-06],
                [1080.3528117556502, 1257.584793961963, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.798206749609951, 0.016533863886783313, 2.3951257876327254e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6362919.999533043, 9.575562416329118, 0.01508497956653298, 1.7134895117258869e-06],
                [1086.5911760185782, 1259.3587163655109, 1423.263682540565, 2711709.7515290566, 5151932.1545183025,
                 6362919.999533043, 9.575562416329118, 0.01508497956653298, 1.7134895117258869e-06],
                [1109.7223419726379, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6866917168224695e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1104.1468138489206, 1269.1713094568, 1400.5041300649148, 2680818.9215573217, 5176513.48526453,
                 6462873.262083922, 9.930403598854456, 0.014610675464065541, 1.6838594647207929e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1132.8146763631971, 1259.7365379392072, 1425.2233655690657, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8805722333904149e-06],
                [1171.2883088636754, 1260.684467035174, 1418.9934367939197, 2630813.4034174774, 5603690.352460641,
                 6253693.013477108, 11.605501217123276, 0.015271687794596967, 1.646211033468178e-06],
                [1050.9762404493679, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1180.7233642687456, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1132.8146763631971, 1259.7696328062204, 1431.129135340821, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8805722333904149e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 5999071.985613948, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1110.0932897166533, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5216394.350278399,
                 6382146.572286638, 10.37995071911957, 0.015398836251961747, 1.6866917168224695e-06],
                [1102.1152991124782, 1257.5624747055258, 1310.4803905865347, 2711709.7515290566, 5167041.861010327,
                 6149853.921081831, 9.837636314132341, 0.01462714264058961, 1.6997971589837513e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5640603.666650951,
                 6009864.991404326, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1114.9126539360466, 1260.5869822935006, 1382.8805760930954, 2711709.7515290566, 5140058.622833891,
                 6346803.504583463, 9.887855977162419, 0.01508497956653298, 1.64751e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1128.7618876139773, 1249.0468225853215, 1445.7770632611023, 2680186.09729148, 5148651.658242081,
                 6346803.504583463, 9.859496987181, 0.015088105294695333, 1.639798179111424e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1189.435265267026, 1289.634611933384, 1434.8120791304352, 2862137.984148449, 5290343.222690649,
                 5912724.800574016, 10.184880722099962, 0.017509170428913215, 1.928535330675258e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1179.0771063597485, 1231.4016603382263, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1179.6176953978943, 1248.028182449465, 1424.647927352167, 2672069.8742411183, 5382182.017517347,
                 6253461.142118783, 11.688941994097817, 0.015138422350173299, 1.670666167191741e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5133273.89347622,
                 6015547.613055304, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.920980046961, 1430.981126200897, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.014332511765226055, 1.64751e-06],
                [1114.9126539360466, 1260.920980046961, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.014332511765226055, 1.64751e-06],
                [1110.1175336282156, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1181.8203100646904, 1238.9401764552945, 1416.2870169690996, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6416237398977881e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1185.931351058013, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6343949.916530928, 11.54839380449381, 0.01508497956653298, 1.64751e-06],
                [1079.1784194680245, 1259.3587163655109, 1419.0880406850756, 2706163.328552395, 5151932.1545183025,
                 5956595.165526332, 9.97531880551067, 0.01491459237885602, 1.6463053082056679e-06],
                [1183.39613591346, 1249.0062677413969, 1424.647927352167, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1133.9089419004854, 1259.7696328062204, 1436.8742705057894, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8805722333904149e-06],
                [1179.6327800190472, 1256.6948098061655, 1424.605807429695, 2756730.692920595, 5133273.89347622,
                 6043327.517125797, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1132.8146763631971, 1259.7696328062204, 1431.129135340821, 2711709.7515290566, 5139946.170017449,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8805722333904149e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1115.400748739093, 1262.2626810628155, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.015227144226188045, 1.6470599805134658e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1178.4227399172075, 1243.0428881177108, 1421.4061869993748, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.417802244505165, 0.01508497956653298, 1.6501983845623307e-06],
                [1110.0932897166533, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015398836251961747, 1.6866917168224695e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1118.7612020200572, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5467159.047638733,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6866917168224695e-06],
                [1178.4227399172075, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.417802244505165, 0.01508497956653298, 1.6571644711735225e-06],
                [1179.6176953978943, 1248.028182449465, 1424.647927352167, 2672069.8742411183, 5382138.607390853,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1180.7233642687456, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.111309139135487, 0.017509170428913215, 1.8805380048230116e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6009864.991404326, 11.605501217123276, 0.015129553230613428, 1.746960716329272e-06],
                [1080.3528117556502, 1265.7237988058796, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.798206749609951, 0.016533863886783313, 2.362014103822307e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6204147.819576919, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1086.181122815101, 1259.3587163655109, 1423.694364587315, 2711709.7515290566, 5151932.1545183025,
                 6362919.999533043, 9.575562416329118, 0.01508497956653298, 1.7134895117258869e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1050.9762404493679, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1110.1718615943932, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 10.075539692668276, 0.014995216559648368, 1.64441131717817e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1180.7233642687456, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5912724.800574016, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1128.7618876139773, 1249.0468225853215, 1445.7770632611023, 2680186.09729148, 5148651.658242081,
                 6346803.504583463, 9.859496987181, 0.015088105294695333, 1.639798179111424e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 5999071.985613948, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1183.39613591346, 1249.007116324817, 1424.9780695578336, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1179.0771063597485, 1231.4016603382263, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5640603.666650951,
                 6009864.991404326, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6009864.991404326, 11.605501217123276, 0.015129553230613428, 1.746960716329272e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1180.010905938387, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.111309139135487, 0.017509170428913215, 1.8805380048230116e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1114.9126539360466, 1260.920980046961, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.014332511765226055, 1.64751e-06],
                [1110.1175336282156, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1080.3528117556502, 1265.7237988058796, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.748432217510594, 0.016533863886783313, 2.362014103822307e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1115.400748739093, 1262.2626810628155, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.015227144226188045, 1.6470599805134658e-06],
                [1179.0771063597485, 1217.4447193223818, 1413.6456109965338, 2620530.399487235, 5184803.83682678,
                 6245691.858534137, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1189.435265267026, 1289.634611933384, 1434.8120791304352, 2862137.984148449, 5290343.222690649,
                 5912724.800574016, 10.184880722099962, 0.017509170428913215, 1.928535330675258e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1133.9089419004854, 1259.7696328062204, 1436.8742705057894, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1179.6327800190472, 1256.6948098061655, 1424.605807429695, 2756730.692920595, 5133273.89347622,
                 6043327.517125797, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1189.435265267026, 1294.47809179355, 1434.8120791304352, 2862137.984148449, 5246305.625197027,
                 6349243.745008091, 10.015056351277549, 0.017509170428913215, 1.928535330675258e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1179.0771063597485, 1231.4016603382263, 1416.1995418912618, 2618905.5231839283, 5290384.748336985,
                 6245691.858534137, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1102.1152991124782, 1257.5624747055258, 1310.4803905865347, 2711709.7515290566, 5167041.861010327,
                 6149853.921081831, 9.837636314132341, 0.01462714264058961, 1.6997971589837513e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5133273.89347622,
                 6015547.613055304, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1178.4227399172075, 1243.0428881177108, 1421.4061869993748, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.415666471826182, 0.015086261250349245, 1.6501983845623307e-06],
                [1114.9126539360466, 1260.5869822935006, 1428.2458013755695, 2750643.691680225, 5344138.987577684,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1178.4227399172075, 1243.0428881177108, 1421.4061869993748, 2756948.118884614, 5083580.380295775,
                 6252960.904134171, 11.417802244505165, 0.01508497956653298, 1.6501983845623307e-06],
                [1050.9762404493679, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1172.1322901571223, 1275.4736789918381, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6204147.819576919, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 5999071.985613948, 9.962874958185285, 0.015031653825581753, 1.64441131717817e-06],
                [1172.7862297639358, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5082503.039302015,
                 6252960.904134171, 11.417802244505165, 0.01508497956653298, 1.6571644711735225e-06],
                [1079.1784194680245, 1259.3587163655109, 1419.0880406850756, 2706163.328552395, 5151932.1545183025,
                 5956595.165526332, 9.97531880551067, 0.01491459237885602, 1.6463053082056679e-06],
                [1117.7420821280732, 1260.5508397384126, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6346723.769806807, 9.899845029941671, 0.014904179979953325, 1.6416817287721877e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.959965929147502, 0.014580502784658061, 1.675614930109933e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.920980046961, 1430.981126200897, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.014332511765226055, 1.64751e-06],
                [1110.0932897166533, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015398836251961747, 1.6866917168224695e-06],
                [1172.6173836553487, 1231.4016603382263, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.593330796866331, 0.015109546481814558, 1.716788954519652e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 5999071.985613948, 9.962874958185285, 0.015031653825581753, 1.64441131717817e-06],
                [1085.9977035903282, 1287.212660761374, 1429.0126531036894, 2853237.462187207, 5151932.1545183025,
                 6123092.829186761, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1183.39613591346, 1249.007116324817, 1424.9780695578336, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1179.0771063597485, 1217.4447193223818, 1413.6456109965338, 2620530.399487235, 5184803.83682678,
                 6245691.858534137, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6252960.904134171, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1110.0932897166533, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015173396239601395, 1.6866917168224695e-06],
                [1117.7420821280732, 1268.6265627123964, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6293398.136293111, 9.899845029941671, 0.014904179979953325, 1.6416817287721877e-06],
                [1176.1786883446464, 1281.1513670331767, 1422.5889236131238, 2643981.980613781, 5933402.3025543075,
                 6009864.991404326, 11.603245339559699, 0.015319465719948019, 1.746960716329272e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1110.1175336282156, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6009864.991404326, 11.605501217123276, 0.015129553230613428, 1.746960716329272e-06],
                [1128.7618876139773, 1249.0468225853215, 1445.7770632611023, 2680186.09729148, 5148651.658242081,
                 6346803.504583463, 9.859496987181, 0.015088105294695333, 1.639798179111424e-06],
                [1135.7553633615312, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5381047.991391903,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6453299579647577e-06],
                [1054.1924321162223, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 11.605501217123276, 0.01508497956653298, 1.6581878711425546e-06],
                [1080.3528117556502, 1265.7237988058796, 1428.2534650650523, 2511977.1924710386, 5151835.089616997,
                 6051450.579799658, 10.748432217510594, 0.016533863886783313, 2.362014103822307e-06],
                [1114.9126539360466, 1260.920980046961, 1427.7914453000797, 2711709.7515290566, 5154755.245931415,
                 6346803.504583463, 9.87071144774333, 0.014332511765226055, 1.64751e-06],
                [1109.4847517557757, 1260.5869822935006, 1428.1550395559061, 2749269.6046548923, 5151932.1545183025,
                 6374031.613940251, 10.746163282079246, 0.01508497956653298, 1.64751e-06],
                [1115.400748739093, 1262.2626810628155, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.015227144226188045, 1.6470599805134658e-06],
                [1179.0771063597485, 1231.4016603382263, 1416.1995418912618, 2618905.5231839283, 5290384.748336985,
                 6245691.858534137, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1189.435265267026, 1289.634611933384, 1434.8120791304352, 2862137.984148449, 5290343.222690649,
                 5912724.800574016, 10.184880722099962, 0.017509170428913215, 1.928535330675258e-06],
                [1115.400748739093, 1262.2626810628155, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.015227144226188045, 1.6470599805134658e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1180.010905938387, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.111309139135487, 0.017509170428913215, 1.8805380048230116e-06],
                [1172.7862297639358, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5082503.039302015,
                 6252960.904134171, 11.417802244505165, 0.01508497956653298, 1.6571644711735225e-06],
                [1114.834792035313, 1247.381448125031, 1431.1711871767654, 2711709.7515290566, 5134156.702340882,
                 5999071.985613948, 9.962874958185285, 0.015031653825581753, 1.7398312362925428e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.959965929147502, 0.014580502784658061, 1.675614930109933e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.6434577512893157e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1180.617132887007, 1256.6948098061655, 1424.605807429695, 2756247.4596015695, 5133273.89347622,
                 6015547.613055304, 11.406422461803086, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5348062.169816384,
                 6414156.707362284, 11.170771782933183, 0.015319465719948019, 1.746960716329272e-06],
                [1181.8203100646904, 1250.9235003414021, 1431.1745701883897, 2756948.118884614, 4930457.91981958,
                 6252960.904134171, 11.395208016578668, 0.014569681251776666, 1.6475084370490054e-06],
                [1102.1152991124782, 1257.5624747055258, 1310.4803905865347, 2711709.7515290566, 5167041.861010327,
                 6149853.921081831, 9.837636314132341, 0.01462714264058961, 1.6997971589837513e-06],
                [1128.6500541303647, 1269.9796387467936, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6346803.504583463, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1050.9762404493679, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1111.2176303621834, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.014836825112684983, 1.6890537116861066e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6204147.819576919, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1172.6173836553487, 1231.4016603382263, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.593330796866331, 0.015109546481814558, 1.716788954519652e-06],
                [1172.1322901571223, 1275.4736789918381, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1110.0932897166533, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015398836251961747, 1.6866917168224695e-06],
                [1114.9126539360466, 1269.2798015768649, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6334610.95506902, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1133.9089419004854, 1259.7696328062204, 1436.8742705057894, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.90690597315, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1118.376653988001, 1250.323551925034, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6395049308713066e-06],
                [1172.7862297639358, 1260.6365385324632, 1424.6533094655529, 2756948.118884614, 5190068.531179048,
                 6252960.904134171, 11.191356944469986, 0.015083085791920761, 1.6571644711735225e-06],
                [1114.8364994143328, 1259.8285287095925, 1437.3529725977721, 2711709.7515290566, 5061252.899416794,
                 6609284.374515722, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1172.1322901571223, 1272.8969286837373, 1424.647927352167, 2643981.980613781, 5644960.1495587025,
                 6343918.206439393, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1111.2176303621834, 1265.6413887411613, 1420.7016199249624, 2711709.7515290566, 4972815.830228467,
                 6346803.504583463, 9.867608359905255, 0.014875693737049485, 1.698160937631157e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 5999071.985613948, 9.962874958185285, 0.015031653825581753, 1.64441131717817e-06],
                [1183.39613591346, 1249.007116324817, 1424.9780695578336, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.426790838912128, 0.015319465719948019, 1.746960716329272e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5177667.749296468,
                 6058545.49793099, 11.579324789874297, 0.015227144226188045, 1.6470599805134658e-06],
                [1179.0771063597485, 1217.4447193223818, 1413.6456109965338, 2620530.399487235, 5184803.83682678,
                 6245691.858534137, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1050.9762404493679, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6009864.991404326, 11.605501217123276, 0.015129553230613428, 1.746960716329272e-06],
                [1176.1786883446464, 1281.1513670331767, 1422.5889236131238, 2643981.980613781, 5933402.3025543075,
                 6009864.991404326, 11.603245339559699, 0.015319465719948019, 1.746960716329272e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1117.7420821280732, 1268.6265627123964, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6293398.136293111, 9.899845029941671, 0.014904179979953325, 1.6416817287721877e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6277042.034213948, 9.959965929147502, 0.014877788107490625, 1.6767475913984607e-06],
                [1140.2618729167225, 1259.7696328062204, 1439.7518547085635, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1085.9977035903282, 1259.3587163655109, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6010849.40413067, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1172.6173836553487, 1231.4016603382263, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.593330796866331, 0.015109546481814558, 1.716788954519652e-06],
                [1137.2060277008807, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5530424.3268809505,
                 6058545.49793099, 9.772174739908307, 0.01508497956653298, 1.6453299579647577e-06],
                [1189.435265267026, 1289.634611933384, 1434.8120791304352, 2862137.984148449, 5290343.222690649,
                 5912724.800574016, 10.184880722099962, 0.017509170428913215, 1.928535330675258e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6204147.819576919, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1128.6500541303647, 1260.508213067299, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6267381.396045687, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1180.010905938387, 1284.2421194708988, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.111309139135487, 0.017509170428913215, 1.8805380048230116e-06],
                [1176.1786883446464, 1283.60271590594, 1422.5889236131238, 2643981.980613781, 5933402.3025543075,
                 6019806.243375341, 11.603245339559699, 0.015319465719948019, 1.746960716329272e-06],
                [1110.1175336282156, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1133.9089419004854, 1259.7696328062204, 1436.8742705057894, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1114.9126539360466, 1260.920980046961, 1427.7914453000797, 2711709.7515290566, 5154755.245931415,
                 6346803.504583463, 9.87071144774333, 0.014332511765226055, 1.64751e-06],
                [1110.0932897166533, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015398836251961747, 1.6866917168224695e-06],
                [1109.4847517557757, 1260.5869822935006, 1428.1550395559061, 2749269.6046548923, 5151932.1545183025,
                 6374031.613940251, 10.746163282079246, 0.01508497956653298, 1.64751e-06],
                [1085.9977035903282, 1287.212660761374, 1429.0126531036894, 2853237.462187207, 5151932.1545183025,
                 6123092.829186761, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1172.7862297639358, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5082503.039302015,
                 6252960.904134171, 11.417802244505165, 0.01508497956653298, 1.6571644711735225e-06],
                [1100.4582501781676, 1260.5869822935006, 1428.1550395559061, 2749269.6046548923, 4637750.6030374495,
                 6771063.144279609, 10.746163282079246, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1170.681625634677, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5390118.746328119,
                 6264496.097901617, 11.605501217123276, 0.015319465719948019, 1.7448699564399412e-06],
                [1102.1152991124782, 1257.5624747055258, 1310.4803905865347, 2711709.7515290566, 5167041.861010327,
                 6149853.921081831, 9.837636314132341, 0.01462714264058961, 1.6997971589837513e-06],
                [1117.7420821280732, 1268.6265627123964, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6293398.136293111, 9.927254088826919, 0.014904179979953325, 1.6416817287721877e-06],
                [1106.8749431340277, 1252.420943546848, 1435.4274518724374, 2711709.7515290566, 5177667.749296468,
                 6204147.819576919, 9.607588244206696, 0.014995216559648368, 1.644382598706639e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.959965929147502, 0.014580502784658061, 1.675614930109933e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.3027121805815, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1118.376653988001, 1250.323551925034, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.6395049308713066e-06],
                [1117.7420821280732, 1263.6042552377323, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6316177.007625211, 9.927254088826919, 0.014904179979953325, 1.6692606546872883e-06],
                [1050.9762404493679, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1179.0771063597485, 1217.4447193223818, 1413.6456109965338, 2620530.399487235, 5184803.83682678,
                 6245691.858534137, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1106.8629264804772, 1251.8935813168757, 1435.4274518724374, 2711709.7515290566, 5337967.192104895,
                 6138136.18408686, 9.607588244206696, 0.014995216559648368, 1.644382598706639e-06],
                [1140.2618729167225, 1259.7696328062204, 1439.7518547085635, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1180.010905938387, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017509170428913215, 1.8805380048230116e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2711709.7515290566, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1117.7420821280732, 1268.6265627123964, 1417.260144182994, 2711709.7515290566, 5151932.1545183025,
                 6293398.136293111, 9.899845029941671, 0.014904179979953325, 1.6416817287721877e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.426790838912128, 0.015319465719948019, 1.746960716329272e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1183.39613591346, 1249.007116324817, 1424.9780695578336, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1176.1786883446464, 1283.60271590594, 1422.5889236131238, 2643981.980613781, 5933402.3025543075,
                 6019806.243375341, 11.603245339559699, 0.015319465719948019, 1.746960716329272e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1118.8759320363351, 1268.4658426637905, 1424.8562282482428, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1114.9126539360466, 1260.647368087708, 1420.7016199249624, 2711709.7515290566, 5227000.31504303,
                 6346803.504583463, 9.87071144774333, 0.01477210173962785, 1.64751e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1085.9977035903282, 1287.212660761374, 1429.0126531036894, 2853237.462187207, 5151932.1545183025,
                 6123092.829186761, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6204147.819576919, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1111.2176303621834, 1265.581002946954, 1420.7016199249624, 2711709.7515290566, 4972815.830228467,
                 6346803.504583463, 9.867608359905255, 0.014875693737049485, 1.698160937631157e-06],
                [1114.8364994143328, 1259.8285287095925, 1437.3529725977721, 2711709.7515290566, 5061252.899416794,
                 6609284.374515722, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1172.1322901571223, 1272.8969286837373, 1424.647927352167, 2643981.980613781, 5644960.1495587025,
                 6343918.206439393, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1110.0932897166533, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015398836251961747, 1.6866917168224695e-06],
                [1114.9126539360466, 1260.48746626119, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 5999071.985613948, 9.962874958185285, 0.015031653825581753, 1.64441131717817e-06],
                [1050.9762404493679, 1257.2267980758488, 1440.177953468843, 2711337.2097504647, 5029083.46383821,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1110.0932897166533, 1268.3282815822229, 1435.9138529261822, 2743472.8696363987, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015398836251961747, 1.6866917168224695e-06],
                [1172.6173836553487, 1231.4016603382263, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.593330796866331, 0.015109546481814558, 1.716788954519652e-06],
                [1137.2060277008807, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5530424.3268809505,
                 6058545.49793099, 9.772174739908307, 0.01508497956653298, 1.6453299579647577e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.8370396447197, 0.015156027526283776, 1.64441131717817e-06],
                [1172.7862297639358, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5082503.039302015,
                 6252960.904134171, 11.033638203846577, 0.01532902143707891, 1.6571644711735225e-06],
                [1128.6500541303647, 1260.508213067299, 1420.6927251709847, 2711709.7515290566, 5196098.543422922,
                 6267381.396045687, 9.805607387570063, 0.01508497956653298, 1.644075174710523e-06],
                [1133.9089419004854, 1259.7696328062204, 1436.8742705057894, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1111.2176303621834, 1265.6413887411613, 1420.7016199249624, 2732860.0254997723, 4972815.830228467,
                 6433383.956931086, 9.867608359905255, 0.014875693737049485, 1.698160937631157e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6217837.80643352, 9.962874958185285, 0.014794895554939519, 1.644382598706639e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.01782143850877657, 1.874993741056419e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.7862297639358, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5082503.039302015,
                 6252960.904134171, 11.417802244505165, 0.01508497956653298, 1.6571644711735225e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.839306628863088, 0.01437360244072579, 1.6406324153956573e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1114.9126539360466, 1255.3060555816992, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1272.8969286837373, 1421.7990273266873, 2643981.980613781, 5644960.1495587025,
                 6054341.979799062, 11.605501217123276, 0.01439486405547544, 1.746960716329272e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.3027121805815, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1118.8759320363351, 1268.4658426637905, 1424.8562282482428, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.426790838912128, 0.015319465719948019, 1.746960716329272e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1172.7862297639358, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5082503.039302015,
                 6252960.904134171, 11.14125315048896, 0.01508497956653298, 1.65184419912513e-06],
                [1172.7862297639358, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5082503.039302015,
                 6252960.904134171, 11.417802244505165, 0.01508497956653298, 1.6571644711735225e-06],
                [1114.9126539360466, 1255.3060555816992, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1111.2176303621834, 1265.581002946954, 1420.7016199249624, 2711709.7515290566, 4972815.830228467,
                 6346803.504583463, 9.867608359905255, 0.014875693737049485, 1.698160937631157e-06],
                [1179.0771063597485, 1217.4447193223818, 1413.6456109965338, 2620530.399487235, 5184803.83682678,
                 6245691.858534137, 11.688941994097817, 0.015110828165630823, 1.670666167191741e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1094.20494378046, 1255.3060555816992, 1420.7016199249624, 2640666.855664094, 5135604.332508602,
                 6266674.486555223, 10.091325553538296, 0.01508497956653298, 1.64751e-06],
                [1110.0932897166533, 1277.4127717832648, 1435.9138529261822, 2743472.8696363987, 5202115.561551428,
                 6382146.572286638, 10.62631839334633, 0.015398836251961747, 1.6866917168224695e-06],
                [1176.1786883446464, 1283.60271590594, 1422.5889236131238, 2643981.980613781, 5933402.3025543075,
                 6019806.243375341, 11.603245339559699, 0.015319465719948019, 1.746960716329272e-06],
                [1106.8629264804772, 1251.8935813168757, 1435.4274518724374, 2711709.7515290566, 5337967.192104895,
                 6138136.18408686, 9.607588244206696, 0.014995216559648368, 1.644382598706639e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1172.1322901571223, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7492546775003008e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1260.48746626119, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1183.39613591346, 1249.007116324817, 1424.9780695578336, 2685865.672715818, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1183.39613591346, 1249.007116324817, 1424.9780695578336, 2688406.096439742, 5151840.492070999,
                 6253461.142118783, 11.688941994097817, 0.015110828165630823, 1.7287073342999962e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1128.7178736119527, 1260.508213067299, 1420.090845472287, 2711709.7515290566, 5196098.543422922,
                 6267381.396045687, 9.805607387570063, 0.01508497956653298, 1.6462533731904685e-06],
                [1180.010905938387, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017352132407197105, 1.8805380048230116e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6204147.819576919, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1114.9126539360466, 1254.7974343990907, 1420.4426978781207, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01462354532140832, 1.64751e-06],
                [1050.9762404493679, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1172.6173836553487, 1237.9667862455688, 1416.1995418912618, 2618905.5231839283, 5079617.930562607,
                 6245691.858534137, 11.771103069974284, 0.015512293130991383, 1.716788954519652e-06],
                [1050.9762404493679, 1257.2267980758488, 1440.177953468843, 2711337.2097504647, 5029083.46383821,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1140.2618729167225, 1259.7696328062204, 1439.7518547085635, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1180.010905938387, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017509170428913215, 1.8805380048230116e-06],
                [1141.2570102982986, 1259.7696328062204, 1439.7518547085635, 2753844.2247911957, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.015110704744109062, 1.8717425252296465e-06],
                [1114.9126539360466, 1260.48746626119, 1408.3354294977653, 2794529.276377869, 5197901.329027299,
                 6346803.504583463, 9.87071144774333, 0.01505009711174968, 1.64751e-06],
                [1110.0932897166533, 1263.0364881672328, 1435.9138529261822, 2711709.7515290566, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015398836251961747, 1.6866917168224695e-06],
                [1110.0932897166533, 1268.3282815822229, 1435.9138529261822, 2743472.8696363987, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015398836251961747, 1.6866917168224695e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5478555.735442081,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1114.9126539360466, 1247.339086600727, 1421.8558877581204, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6217837.80643352, 9.962874958185285, 0.014794895554939519, 1.644382598706639e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 5999071.985613948, 9.962874958185285, 0.015031653825581753, 1.64441131717817e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.839306628863088, 0.01437360244072579, 1.6406324153956573e-06],
                [1114.8364994143328, 1259.8285287095925, 1437.3529725977721, 2711709.7515290566, 5061252.899416794,
                 6609284.374515722, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1161.9308463271727, 1239.3344143550214, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.593330796866331, 0.01534698148474761, 1.716788954519652e-06],
                [1172.6173836553487, 1231.4016603382263, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.593330796866331, 0.015109546481814558, 1.716788954519652e-06],
                [1172.1322901571223, 1272.8969286837373, 1424.647927352167, 2643981.980613781, 5644960.1495587025,
                 6343918.206439393, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.3027121805815, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1050.9762404493679, 1253.7997759357147, 1425.0398300804388, 2643910.964641653, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.015713623088508245, 1.6858501133596892e-06],
                [1171.0989638205704, 1269.4644002253424, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015912513251578327, 1.7492546775003008e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017136258349463553, 1.8805380048230116e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1140.2618729167225, 1259.7696328062204, 1439.7518547085635, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1050.9762404493679, 1257.2267980758488, 1440.177953468843, 2711337.2097504647, 5029083.46383821,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1172.1322901571223, 1272.8969286837373, 1424.647927352167, 2643981.980613781, 5644960.1495587025,
                 6343918.206439393, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1114.8364994143328, 1276.755042176067, 1437.607196167975, 2711709.7515290566, 5061252.899416794,
                 6609284.374515722, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1114.9126539360466, 1260.48746626119, 1408.3354294977653, 2794529.276377869, 5197901.329027299,
                 6346803.504583463, 9.87071144774333, 0.01505009711174968, 1.64751e-06],
                [1172.1322901571223, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7492546775003008e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2676965.0315407575, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1161.9308463271727, 1239.3344143550214, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.593330796866331, 0.01534698148474761, 1.716788954519652e-06],
                [1114.9126539360466, 1255.3060555816992, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1114.9126539360466, 1254.7974343990907, 1420.4426978781207, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01462354532140832, 1.64751e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1118.8759320363351, 1268.4658426637905, 1424.8562282482428, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1106.8629264804772, 1251.8935813168757, 1435.4274518724374, 2711709.7515290566, 5337967.192104895,
                 6138136.18408686, 9.607588244206696, 0.014995216559648368, 1.644382598706639e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.426790838912128, 0.015319465719948019, 1.746960716329272e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1094.20494378046, 1255.3060555816992, 1420.7016199249624, 2640666.855664094, 5135604.332508602,
                 6266674.486555223, 10.091325553538296, 0.01508497956653298, 1.64751e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6638462.321443304, 9.638380132352067, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.9126539360466, 1256.6998042531457, 1431.2824071034747, 2711709.7515290566, 5135842.055580591,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.64751e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1180.010905938387, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017352132407197105, 1.8805380048230116e-06],
                [1106.6926400295054, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.46457888998826, 0.01491459237885602, 1.6416646007583103e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5478555.735442081,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.839306628863088, 0.01437360244072579, 1.6406324153956573e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 5999071.985613948, 9.962874958185285, 0.015031653825581753, 1.64441131717817e-06],
                [1094.20494378046, 1255.3060555816992, 1420.7016199249624, 2640666.855664094, 5135604.332508602,
                 6266674.486555223, 10.091325553538296, 0.01508497956653298, 1.64751e-06],
                [1100.1408530781362, 1259.20529625876, 1424.4350629201424, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.963100330742796, 0.014877945336207184, 1.6827081430883913e-06],
                [1176.1786883446464, 1283.60271590594, 1428.888466324537, 2643981.980613781, 5933402.3025543075,
                 6019806.243375341, 11.603245339559699, 0.015319465719948019, 1.7487048378215287e-06],
                [1110.0932897166533, 1268.3282815822229, 1435.9138529261822, 2743472.8696363987, 5183912.901011421,
                 6382146.572286638, 10.641003542772033, 0.015398836251961747, 1.6866917168224695e-06],
                [1176.1786883446464, 1283.60271590594, 1422.5889236131238, 2643981.980613781, 5933402.3025543075,
                 6019806.243375341, 11.603245339559699, 0.015319465719948019, 1.746960716329272e-06],
                [1114.834792035313, 1252.4221946420748, 1416.2679018973663, 2660406.8703632285, 5282085.827710184,
                 6205159.784569372, 9.955251140486949, 0.015045485379389994, 1.64441131717817e-06],
                [1050.9762404493679, 1257.606216652866, 1428.1727195332353, 2676217.667872637, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1141.2570102982986, 1259.7696328062204, 1439.7518547085635, 2753844.2247911957, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.015110704744109062, 1.8717425252296465e-06],
                [1114.9126539360466, 1247.339086600727, 1421.8558877581204, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.7862297639358, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 5082503.039302015,
                 6252960.904134171, 11.14125315048896, 0.01508497956653298, 1.65184419912513e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.8364994143328, 1252.420943546848, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6204147.819576919, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.3027121805815, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1050.9762404493679, 1253.7997759357147, 1425.0398300804388, 2643910.964641653, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.015713623088508245, 1.6858501133596892e-06],
                [1171.0989638205704, 1269.4644002253424, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015912513251578327, 1.7492546775003008e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017136258349463553, 1.8805380048230116e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1140.2618729167225, 1259.7696328062204, 1439.7518547085635, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1050.9762404493679, 1257.2267980758488, 1440.177953468843, 2711337.2097504647, 5029083.46383821,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1172.1322901571223, 1272.8969286837373, 1424.647927352167, 2643981.980613781, 5644960.1495587025,
                 6343918.206439393, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1114.8364994143328, 1276.755042176067, 1437.607196167975, 2711709.7515290566, 5061252.899416794,
                 6609284.374515722, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1114.9126539360466, 1260.48746626119, 1408.3354294977653, 2794529.276377869, 5197901.329027299,
                 6346803.504583463, 9.87071144774333, 0.01505009711174968, 1.64751e-06],
                [1172.1322901571223, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7492546775003008e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2676965.0315407575, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1161.9308463271727, 1239.3344143550214, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.593330796866331, 0.01534698148474761, 1.716788954519652e-06],
                [1114.9126539360466, 1255.3060555816992, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1114.9126539360466, 1254.7974343990907, 1420.4426978781207, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01462354532140832, 1.64751e-06],
                [1106.8629264804772, 1251.8935813168757, 1435.4274518724374, 2711709.7515290566, 5337967.192104895,
                 6138136.18408686, 9.607588244206696, 0.014995216559648368, 1.644382598706639e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.426790838912128, 0.015319465719948019, 1.746960716329272e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1094.20494378046, 1255.3060555816992, 1420.7016199249624, 2640666.855664094, 5135604.332508602,
                 6266674.486555223, 10.091325553538296, 0.01508497956653298, 1.64751e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6638462.321443304, 9.638380132352067, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.9126539360466, 1256.6998042531457, 1431.2824071034747, 2711709.7515290566, 5135842.055580591,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.64751e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1100.1147053565142, 1259.20529625876, 1421.1596772381108, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.015413025995534579, 1.6827081430883913e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6638462.321443304, 9.638380132352067, 0.01491459237885602, 1.6416646007583103e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1100.1408530781362, 1259.20529625876, 1424.749788793609, 2794529.276377869, 5137466.622582068,
                 6100698.446400893, 9.963100330742796, 0.014877945336207184, 1.6416877649487889e-06],
                [1180.010905938387, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017352132407197105, 1.8805380048230116e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.839306628863088, 0.014160961059429754, 1.6406324153956573e-06],
                [1172.7862297639358, 1257.3170602524901, 1433.9096111224062, 2756948.118884614, 5082503.039302015,
                 6252960.904134171, 11.14125315048896, 0.01508497956653298, 1.6511902215333544e-06],
                [1106.6926400295054, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.46457888998826, 0.01491459237885602, 1.6416646007583103e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015508872161102813, 1.7492546775003008e-06],
                [1094.20494378046, 1257.1652413000595, 1420.7016199249624, 2640666.855664094, 5123686.718780826,
                 6266674.486555223, 10.353923395734402, 0.01508497956653298, 1.64751e-06],
                [1116.099438708417, 1248.2528109041914, 1417.21404073115, 2710386.6655561705, 5020955.320062604,
                 6252740.844923852, 10.132163437622156, 0.01439649066491804, 1.6404983713795935e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017136258349463553, 1.8805380048230116e-06],
                [1118.8759320363351, 1268.4658426637905, 1424.8562282482428, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5478555.735442081,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.9126539360466, 1264.1301647186085, 1408.3354294977653, 2794529.276377869, 5224489.970748406,
                 6346803.504583463, 9.87071144774333, 0.01505009711174968, 1.64751e-06],
                [1105.4214399254815, 1260.692682430978, 1435.4274518724374, 2711709.7515290566, 5337967.192104895,
                 6113196.821670636, 9.540736088430256, 0.014995644153602643, 1.644382598706639e-06],
                [1114.9126539360466, 1247.339086600727, 1399.664756324703, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.729534050129528, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5306929.141208601,
                 6351339.408023649, 9.599662957877094, 0.015352438307778995, 1.64751e-06],
                [1116.7644456614257, 1260.48746626119, 1408.8406864975236, 2794529.276377869, 5197901.329027299,
                 6390992.7070193775, 9.87071144774333, 0.01505009711174968, 1.6850174215984935e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.3027121805815, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1050.9762404493679, 1253.7997759357147, 1425.0398300804388, 2643910.964641653, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.015713623088508245, 1.6858501133596892e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017136258349463553, 1.8805380048230116e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1140.2618729167225, 1259.7696328062204, 1439.7518547085635, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1172.1322901571223, 1272.8969286837373, 1424.647927352167, 2643981.980613781, 5644960.1495587025,
                 6343918.206439393, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1050.9762404493679, 1257.2267980758488, 1440.177953468843, 2711337.2097504647, 5029083.46383821,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1114.8364994143328, 1276.755042176067, 1437.607196167975, 2711709.7515290566, 5061252.899416794,
                 6609284.374515722, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1114.9126539360466, 1260.48746626119, 1408.3354294977653, 2794529.276377869, 5197901.329027299,
                 6346803.504583463, 9.87071144774333, 0.01505009711174968, 1.64751e-06],
                [1172.1322901571223, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7492546775003008e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2676965.0315407575, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1161.9308463271727, 1239.3344143550214, 1416.1995418912618, 2618905.5231839283, 5151840.492070999,
                 6245691.858534137, 11.593330796866331, 0.01534698148474761, 1.716788954519652e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1114.9126539360466, 1255.3060555816992, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1114.9126539360466, 1254.7974343990907, 1420.4426978781207, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01462354532140832, 1.64751e-06],
                [1106.8629264804772, 1251.8935813168757, 1435.4274518724374, 2711709.7515290566, 5337967.192104895,
                 6138136.18408686, 9.607588244206696, 0.014995216559648368, 1.644382598706639e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6264496.097901617, 11.426790838912128, 0.015319465719948019, 1.746960716329272e-06],
                [1094.20494378046, 1255.3060555816992, 1420.7016199249624, 2640666.855664094, 5135604.332508602,
                 6266674.486555223, 10.091325553538296, 0.01508497956653298, 1.64751e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6638462.321443304, 9.638380132352067, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.9126539360466, 1256.6998042531457, 1431.2824071034747, 2711709.7515290566, 5135842.055580591,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.64751e-06],
                [1094.20494378046, 1257.1652413000595, 1420.7016199249624, 2640666.855664094, 5123686.718780826,
                 6266674.486555223, 10.353923395734402, 0.01508497956653298, 1.64751e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290195.012571191,
                 5907538.352840094, 10.14357129999074, 0.017136258349463553, 1.8805380048230116e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2642226.6327868286, 5029083.46383821,
                 6252781.131346029, 9.939222256778564, 0.015713623088508245, 1.6858501133596892e-06],
                [1100.1408530781362, 1259.20529625876, 1424.749788793609, 2794529.276377869, 5137466.622582068,
                 6100698.446400893, 9.963100330742796, 0.014877945336207184, 1.6416877649487889e-06],
                [1094.20494378046, 1257.1652413000595, 1407.9165649854249, 2665073.282263369, 5123686.718780826,
                 6266674.486555223, 10.353923395734402, 0.01508497956653298, 1.64751e-06],
                [1118.8759320363351, 1268.4658426637905, 1424.8562282482428, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1101.3512931657685, 1252.2697963391465, 1420.7016199249624, 2794529.276377869, 5137773.120348509,
                 6291895.722338178, 9.959965929147502, 0.014879785728535506, 1.6827081430883913e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6496829729248117e-06],
                [1114.9126539360466, 1256.6998042531457, 1431.2824071034747, 2662311.4857510417, 5137321.181182315,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.8110557423011423e-06],
                [1117.3403886048986, 1247.339086600727, 1390.636966587214, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.730740063432355, 0.01508497956653298, 1.64751e-06],
                [1115.400748739093, 1265.55045099029, 1428.3010571928912, 2713994.109487649, 5048177.206187649,
                 6137077.880026326, 11.579324789874297, 0.01509067020917881, 1.6447615608703846e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6638462.321443304, 9.638380132352067, 0.01491459237885602, 1.6416646007583103e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5478555.735442081,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1115.3505637801043, 1259.3587163655109, 1424.9437503042586, 2710386.6655561705, 5151932.1545183025,
                 6638462.321443304, 9.642834576308625, 0.01491459237885602, 1.6416646007583103e-06],
                [1106.6926400295054, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.46457888998826, 0.01491459237885602, 1.6416646007583103e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.787151112326503, 0.014995216559648368, 1.6865448209421645e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6252740.844923852, 9.839306628863088, 0.014160961059429754, 1.6406324153956573e-06],
                [1100.1147053565142, 1259.20529625876, 1421.1596772381108, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.01647393809951346, 1.6827081430883913e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1116.099438708417, 1248.2528109041914, 1417.21404073115, 2710386.6655561705, 5020955.320062604,
                 6252740.844923852, 10.512801747076804, 0.01439649066491804, 1.6404983713795935e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.3027121805815, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1109.9589493790136, 1259.2797322073343, 1408.3354294977653, 2794529.276377869, 5197901.329027299,
                 6346803.504583463, 9.87071144774333, 0.015213544200567527, 1.64751e-06],
                [1170.815684829177, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7222399231838787e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2674228.063186179, 5073692.871060503,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1100.1408530781362, 1259.20529625876, 1423.8349912868905, 2794529.276377869, 5137466.622582068,
                 6100698.446400893, 9.963100330742796, 0.014877945336207184, 1.6416877649487889e-06],
                [1140.2618729167225, 1259.7696328062204, 1439.7518547085635, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7492546775003008e-06],
                [1114.8364015979141, 1262.2228977124012, 1424.509665833488, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.644233311414821e-06],
                [1114.9126539360466, 1254.7974343990907, 1420.4426978781207, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01462354532140832, 1.64751e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1050.9762404493679, 1257.2267980758488, 1440.177953468843, 2711337.2097504647, 5029083.46383821,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1114.9126539360466, 1260.48746626119, 1408.3354294977653, 2794529.276377869, 5197901.329027299,
                 6346803.504583463, 9.87071144774333, 0.01505009711174968, 1.64751e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.787151112326503, 0.014995216559648368, 1.6865448209421645e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1050.9762404493679, 1253.7997759357147, 1425.0398300804388, 2643910.964641653, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.015713623088508245, 1.6858501133596892e-06],
                [1116.099438708417, 1248.2528109041914, 1417.21404073115, 2710386.6655561705, 5020955.320062604,
                 6252740.844923852, 10.512801747076804, 0.01439649066491804, 1.6404983713795935e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1115.3505637801043, 1250.1440195093382, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6761119331505416e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2676965.0315407575, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6211148.3834170075, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1114.8364994143328, 1276.755042176067, 1437.607196167975, 2711709.7515290566, 5061252.899416794,
                 6609284.374515722, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290195.012571191,
                 5907538.352840094, 10.14357129999074, 0.017136258349463553, 1.8805380048230116e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2642226.6327868286, 5029083.46383821,
                 6252781.131346029, 9.939222256778564, 0.015713623088508245, 1.6858501133596892e-06],
                [1115.400748739093, 1265.55045099029, 1428.3010571928912, 2713994.109487649, 5048177.206187649,
                 6137077.880026326, 11.579324789874297, 0.01509067020917881, 1.6447615608703846e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2676965.0315407575, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1094.20494378046, 1257.1652413000595, 1420.7016199249624, 2640666.855664094, 5123686.718780826,
                 6266674.486555223, 10.353923395734402, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1255.3060555816992, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017136258349463553, 1.8805380048230116e-06],
                [1114.9126539360466, 1256.6998042531457, 1432.7380531614376, 2711709.7515290566, 5135842.055580591,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.64751e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1172.1322901571223, 1273.6063854006757, 1424.647927352167, 2787890.321944607, 4970750.218790298,
                 6264496.097901617, 11.69640114007929, 0.015654224593012078, 1.7492546775003008e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402352374322306, 1.746960716329272e-06],
                [1172.1322901571223, 1272.8969286837373, 1424.647927352167, 2643981.980613781, 5644960.1495587025,
                 6343918.206439393, 11.605501217123276, 0.015319465719948019, 1.746960716329272e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2710386.6655561705, 5151932.1545183025,
                 6638462.321443304, 9.638380132352067, 0.01491459237885602, 1.6416646007583103e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1172.1322901571223, 1261.2561027220345, 1417.0249946056997, 2674527.9391689897, 5624136.048239707,
                 6253461.142118783, 11.605501217123276, 0.014873088291808691, 1.6413780734336877e-06],
                [1106.8629264804772, 1251.8935813168757, 1435.4274518724374, 2711709.7515290566, 5337967.192104895,
                 6138136.18408686, 9.607588244206696, 0.014995216559648368, 1.644382598706639e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6496829729248117e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1050.6347038859697, 1257.5582265374605, 1424.6932084491125, 2676965.0315407575, 5029083.46383821,
                 6295670.361109444, 9.914600670892218, 0.01508497956653298, 1.6487083714263362e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5478555.735442081,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.3027121805815, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1109.9589493790136, 1259.2797322073343, 1408.3354294977653, 2794529.276377869, 5197901.329027299,
                 6346803.504583463, 9.87071144774333, 0.015213544200567527, 1.64751e-06],
                [1170.815684829177, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7222399231838787e-06],
                [1100.1408530781362, 1259.20529625876, 1423.8349912868905, 2794529.276377869, 5137466.622582068,
                 6100698.446400893, 9.963100330742796, 0.014877945336207184, 1.6416877649487889e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2674228.063186179, 5073692.871060503,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1140.2618729167225, 1259.7696328062204, 1439.7518547085635, 2711709.7515290566, 5172210.454034539,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1172.1322901571223, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7492546775003008e-06],
                [1114.9126539360466, 1259.1180510408708, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6277042.034213948, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1254.7974343990907, 1420.4426978781207, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01462354532140832, 1.64751e-06],
                [1114.8364015979141, 1262.2228977124012, 1424.509665833488, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.644233311414821e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2676965.0315407575, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1116.099438708417, 1248.2528109041914, 1417.21404073115, 2710386.6655561705, 5020955.320062604,
                 6252740.844923852, 10.512801747076804, 0.01439649066491804, 1.6404983713795935e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1050.9762404493679, 1257.2267980758488, 1440.177953468843, 2711337.2097504647, 5029083.46383821,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2642226.6327868286, 5029083.46383821,
                 6252781.131346029, 9.939222256778564, 0.015713623088508245, 1.6858501133596892e-06],
                [1114.8364994143328, 1276.755042176067, 1437.607196167975, 2711709.7515290566, 5061252.899416794,
                 6609284.374515722, 9.962874958185285, 0.014995216559648368, 1.644382598706639e-06],
                [1114.9126539360466, 1260.48746626119, 1408.3354294977653, 2794529.276377869, 5197901.329027299,
                 6346803.504583463, 9.87071144774333, 0.01505009711174968, 1.64751e-06],
                [1094.20494378046, 1257.1652413000595, 1420.7016199249624, 2640666.855664094, 5123686.718780826,
                 6266674.486555223, 10.353923395734402, 0.01508497956653298, 1.64751e-06],
                [1115.3505637801043, 1250.1440195093382, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6761119331505416e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017136258349463553, 1.8805380048230116e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.787151112326503, 0.014995216559648368, 1.6865448209421645e-06],
                [1114.9126539360466, 1255.3060555816992, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1050.9762404493679, 1253.7997759357147, 1425.0398300804388, 2643910.964641653, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.015713623088508245, 1.6858501133596892e-06],
                [1172.1322901571223, 1273.6063854006757, 1424.647927352167, 2787890.321944607, 4970750.218790298,
                 6264496.097901617, 11.69640114007929, 0.015654224593012078, 1.7492546775003008e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6211148.3834170075, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290195.012571191,
                 6032540.681365669, 10.14357129999074, 0.017136258349463553, 1.8805380048230116e-06],
                [1115.400748739093, 1265.55045099029, 1428.3010571928912, 2713994.109487649, 5048177.206187649,
                 6137077.880026326, 11.579324789874297, 0.01509067020917881, 1.6447615608703846e-06],
                [1172.1322901571223, 1272.8969286837373, 1437.557746508848, 2643981.980613781, 5644960.1495587025,
                 6319984.010862878, 11.608379366203652, 0.0153098728874533, 1.7558263639829205e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1273.6063854006757, 1424.647927352167, 2787890.321944607, 4970750.218790298,
                 6291269.48214699, 11.70980266070725, 0.015654224593012078, 1.7654351657555759e-06],
                [1115.354899099546, 1259.354435982738, 1406.3414505048033, 2870149.082397822, 5151932.1545183025,
                 6274415.63844293, 9.638380132352067, 0.014911399491080297, 1.6416646007583103e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2681628.57902566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1123.9689889274446, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6471410354130274e-06],
                [1114.9126539360466, 1256.6998042531457, 1432.7380531614376, 2711709.7515290566, 5135842.055580591,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.64751e-06],
                [1176.0145199706367, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5478555.735442081,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1125.2043182911782, 1262.2228977124012, 1424.509665833488, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.644233311414821e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.01508497956653298, 1.670666167191741e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.149246408514118, 0.017229439676543383, 1.874732806720558e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1100.3826382823288, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6414678246117103e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402352374322306, 1.746960716329272e-06],
                [1111.8152136576268, 1261.5968599256498, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.676592761776776, 0.014902455414208876, 1.6820375060380185e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.3027121805815, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1050.9762404493679, 1258.4266239556036, 1440.177953468843, 2711337.2097504647, 4951513.490183301,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1172.1322901571223, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7492546775003008e-06],
                [1125.2043182911782, 1262.2228977124012, 1424.509665833488, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.644233311414821e-06],
                [1115.3505637801043, 1250.1440195093382, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6761119331505416e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1172.1322901571223, 1272.8969286837373, 1437.557746508848, 2643981.980613781, 5644960.1495587025,
                 6319984.010862878, 11.608379366203652, 0.0153098728874533, 1.7558263639829205e-06],
                [1114.9126539360466, 1256.6998042531457, 1432.7380531614376, 2711709.7515290566, 5135842.055580591,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.64751e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.787151112326503, 0.014995216559648368, 1.6865448209421645e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2642226.6327868286, 5029083.46383821,
                 6189310.214897488, 9.939222256778564, 0.015713623088508245, 1.6858501133596892e-06],
                [1100.3826382823288, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6414678246117103e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1124.8314582363391, 1244.7486449823257, 1428.1727195332353, 2711109.3787926026, 5177658.5684807105,
                 6252781.131346029, 9.912726746719038, 0.014876312019105655, 1.6858501133596892e-06],
                [1114.8364994143328, 1276.755042176067, 1430.0554994299264, 2711709.7515290566, 5061252.899416794,
                 6843779.100678993, 9.962874958185285, 0.014995216559648368, 1.6435718248748259e-06],
                [1114.8364015979141, 1262.2228977124012, 1424.509665833488, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.644233311414821e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2642226.6327868286, 5029083.46383821,
                 6252781.131346029, 9.939222256778564, 0.015713623088508245, 1.6858501133596892e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1170.815684829177, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015919094724508868, 1.7222399231838787e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2674228.063186179, 5073692.871060503,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1140.2618729167225, 1259.7696328062204, 1425.9492710629509, 2715399.6210057866, 5204539.152162562,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1159.0274286361612, 1287.8053727475722, 1448.6162624321432, 2854778.224454453, 5360842.436526001,
                 5907538.352840094, 10.175809217601456, 0.017136258349463553, 1.8805380048230116e-06],
                [1116.099438708417, 1248.2528109041914, 1417.21404073115, 2710386.6655561705, 5020955.320062604,
                 6252740.844923852, 10.512801747076804, 0.01439649066491804, 1.6404983713795935e-06],
                [1115.0097241932576, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.945398538287433, 0.01508497956653298, 1.664228124161138e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6211148.3834170075, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1094.20494378046, 1257.1652413000595, 1434.1536421639848, 2640666.855664094, 5123686.718780826,
                 6266674.486555223, 10.336890255390792, 0.01508497956653298, 1.64751e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1100.1408530781362, 1259.20529625876, 1423.8349912868905, 2794529.276377869, 5137466.622582068,
                 6100698.446400893, 9.963100330742796, 0.014877945336207184, 1.6416877649487889e-06],
                [1102.9707247869544, 1260.5869822935006, 1420.5917316831233, 2825130.0490485826, 5151932.1545183025,
                 6329254.498107326, 9.870483944224635, 0.01508497956653298, 1.6471410354130274e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014416923144892024, 1.746960716329272e-06],
                [1106.8161654815144, 1260.5869822935006, 1428.2938999164598, 2756856.4798341207, 4987417.102943326,
                 6252960.904134171, 11.324658824884459, 0.01508497956653298, 1.6859236759203733e-06],
                [1123.2020742221703, 1259.2797322073343, 1408.385208661309, 2794529.276377869, 5197901.329027299,
                 6346803.504583463, 9.87071144774333, 0.015213544200567527, 1.64751e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290195.012571191,
                 6032540.681365669, 10.14357129999074, 0.017136258349463553, 1.8805380048230116e-06],
                [1165.0546070424907, 1263.903713429593, 1437.3522342831861, 2674819.702993904, 5540572.403677184,
                 6211148.3834170075, 11.157304045445803, 0.015319507848894277, 1.7500942903236284e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1094.20494378046, 1257.1652413000595, 1420.7016199249624, 2640666.855664094, 5123686.718780826,
                 6266674.486555223, 10.353923395734402, 0.01508497956653298, 1.64751e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.016034774278074274, 1.670666167191741e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402352374322306, 1.746960716329272e-06],
                [1046.2476557504224, 1253.7081957386629, 1424.647927352167, 2576404.8537268285, 5037123.924454901,
                 6264496.097901617, 10.926743668226985, 0.015654224593012078, 1.7221039311460954e-06],
                [1179.6140307813284, 1284.2421194708988, 1446.7483442856462, 2854778.224454453, 5290343.222690649,
                 5907538.352840094, 10.175809217601456, 0.017136258349463553, 1.8805380048230116e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2674918.453684049, 5029083.46383821,
                 6252781.131346029, 9.66368383513432, 0.01508497956653298, 1.6858501133596892e-06],
                [1172.1322901571223, 1273.6063854006757, 1424.647927352167, 2787890.321944607, 4970750.218790298,
                 6264496.097901617, 11.69640114007929, 0.015654224593012078, 1.7492546775003008e-06],
                [1114.9126539360466, 1255.3060555816992, 1420.7016199249624, 2711709.7515290566, 5135604.332508602,
                 6339744.606807769, 9.869108693672011, 0.01508497956653298, 1.64751e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2676965.0315407575, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1116.099438708417, 1248.2528109041914, 1431.2802890524115, 2618670.60777649, 5020955.320062604,
                 6252740.844923852, 10.403131094036514, 0.01439649066491804, 1.6612822159851205e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1114.834792035313, 1260.5869822935006, 1424.6533094655529, 2711709.7515290566, 5177667.749296468,
                 6058545.49793099, 9.869052328516, 0.01508497956653298, 1.64751e-06],
                [1100.1063624541289, 1231.7664627414742, 1436.3027121805815, 3627678.357077309, 4986879.9039197,
                 2532740.0207836498, 10.181273986642962, 0.016620130673890635, 2.3929656390077357e-06],
                [1100.3826382823288, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6414678246117103e-06],
                [1180.7233642687456, 1290.1144240449926, 1434.8120791304352, 2854778.224454453, 5290343.222690649,
                 6018525.355599652, 10.111309139135487, 0.017509170428913215, 1.874732806720558e-06],
                [1050.9762404493679, 1258.4266239556036, 1440.177953468843, 2711337.2097504647, 4951513.490183301,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.787151112326503, 0.014995216559648368, 1.6865448209421645e-06],
                [1172.1322901571223, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015654224593012078, 1.7492546775003008e-06],
                [1115.3505637801043, 1250.1440195093382, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6761119331505416e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1114.9126539360466, 1256.6998042531457, 1432.7380531614376, 2711709.7515290566, 5135842.055580591,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.64751e-06],
                [1114.8364994143328, 1276.755042176067, 1430.0554994299264, 2711709.7515290566, 5061252.899416794,
                 6843779.100678993, 9.962874958185285, 0.014995216559648368, 1.6435718248748259e-06],
                [1172.1322901571223, 1273.6063854006757, 1424.647927352167, 2787890.321944607, 4970750.218790298,
                 6264496.097901617, 11.69640114007929, 0.015654224593012078, 1.7492546775003008e-06],
                [1114.834792035313, 1252.4221946420748, 1431.2324989413153, 2674228.063186179, 5073692.871060503,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.64441131717817e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2642226.6327868286, 5029083.46383821,
                 6189310.214897488, 9.939222256778564, 0.015713623088508245, 1.6858501133596892e-06],
                [1114.8364015979141, 1262.2228977124012, 1424.509665833488, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.644233311414821e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2676965.0315407575, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5079729.931980113,
                 6205159.784569372, 9.523942410142089, 0.014995216559648368, 1.6455995539949147e-06],
                [1125.2043182911782, 1262.2228977124012, 1424.509665833488, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.644233311414821e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1172.1322901571223, 1272.8969286837373, 1437.557746508848, 2643981.980613781, 5644960.1495587025,
                 6319984.010862878, 11.608379366203652, 0.0153098728874533, 1.7558263639829205e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402352374322306, 1.772746704805386e-06],
                [1089.037314605264, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5198282.044819644,
                 6403335.37885126, 9.87071144774333, 0.015004521873080429, 1.6462972507927568e-06],
                [1171.886230946291, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5541510.5883321315,
                 6254834.865253087, 11.602935611820246, 0.015129553230613428, 1.646211033468178e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2674918.453684049, 5029083.46383821,
                 6252781.131346029, 9.66368383513432, 0.01508497956653298, 1.6858501133596892e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6211148.3834170075, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1172.1322901571223, 1249.0062677413969, 1424.647927352167, 2676217.667872637, 5151840.492070999,
                 6253461.142118783, 11.605501217123276, 0.016034774278074274, 1.670666167191741e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1094.20494378046, 1257.1652413000595, 1434.1536421639848, 2640666.855664094, 5123686.718780826,
                 6266674.486555223, 10.336890255390792, 0.01508497956653298, 1.64751e-06],
                [1140.2618729167225, 1259.7696328062204, 1425.9492710629509, 2715399.6210057866, 5204539.152162562,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1050.9762404493679, 1253.7841068951632, 1446.6888589765636, 2641700.2350165956, 5029083.46383821,
                 6189310.214897488, 9.939222256778564, 0.015713623088508245, 1.6858501133596892e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014416923144892024, 1.746960716329272e-06],
                [1106.8161654815144, 1260.5869822935006, 1428.2938999164598, 2756856.4798341207, 4987417.102943326,
                 6252960.904134171, 11.324658824884459, 0.01508497956653298, 1.6859236759203733e-06],
                [1170.815684829177, 1263.903713429593, 1424.647927352167, 2627915.438331411, 5539495.081817166,
                 6264496.097901617, 11.21124190291663, 0.015919094724508868, 1.7222399231838787e-06],
                [1124.8314582363391, 1244.7486449823257, 1428.1727195332353, 2711109.3787926026, 4957035.652219516,
                 6252781.131346029, 9.912726746719038, 0.015131656963552058, 1.6858501133596892e-06],
                [1159.0274286361612, 1287.8053727475722, 1448.538005613146, 2854778.224454453, 5360842.436526001,
                 5907538.352840094, 10.257090415594687, 0.017136258349463553, 1.8805380048230116e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2675995.973765552, 5086221.215584276,
                 6307046.665064646, 9.939222256778564, 0.0148888930145161, 1.6858501133596892e-06],
                [1114.8364015979141, 1262.2228977124012, 1424.509665833488, 2708245.6581426947, 5177667.749296468,
                 6205159.784569372, 9.962874958185285, 0.014995216559648368, 1.644233311414821e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2642226.6327868286, 5029083.46383821,
                 6252781.131346029, 9.939222256778564, 0.015713623088508245, 1.6858501133596892e-06],
                [1094.20494378046, 1257.1652413000595, 1434.1536421639848, 2640666.855664094, 5123686.718780826,
                 6266674.486555223, 10.336890255390792, 0.01508497956653298, 1.6437081192784193e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.31993774216143, 0.01508497956653298, 1.675718750508336e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1124.6022184104063, 1253.9976612371274, 1428.1727195332353, 2711109.3787926026, 5177658.5684807105,
                 6056075.082738621, 9.912726746719038, 0.014876312019105655, 1.6447260544123108e-06],
                [1102.9707247869544, 1260.5869822935006, 1420.5917316831233, 2825130.0490485826, 5151932.1545183025,
                 6329254.498107326, 9.870483944224635, 0.01508497956653298, 1.6471410354130274e-06],
                [1100.1408530781362, 1259.20529625876, 1423.8349912868905, 2794529.276377869, 5137466.622582068,
                 6100698.446400893, 9.963100330742796, 0.014877945336207184, 1.6416877649487889e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2675995.973765552, 5086221.215584276,
                 6307046.665064646, 9.939222256778564, 0.0148888930145161, 1.6858501133596892e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402352374322306, 1.7961069170362274e-06],
                [1124.0077507460633, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5341886.884732996,
                 6274517.334926196, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.9126539360466, 1256.6998042531457, 1432.7380531614376, 2711709.7515290566, 5135842.055580591,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.64751e-06],
                [1089.037314605264, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5198282.044819644,
                 6403335.37885126, 9.87071144774333, 0.015004521873080429, 1.6462972507927568e-06],
                [1115.3505637801043, 1250.1440195093382, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6761119331505416e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402352374322306, 1.772746704805386e-06],
                [1117.1225010723806, 1245.0461992541775, 1408.2360625510769, 2809265.192909424, 5151932.1545183025,
                 6034517.235726284, 9.899845029941671, 0.014343389674800526, 1.7139568106579685e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402711636037064, 1.8287965466133632e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2769369.5842076726, 4987417.102943326,
                 6252960.904134171, 11.31993774216143, 0.01508497956653298, 1.675718750508336e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4982497.032409398,
                 6316982.864584648, 11.401218941020598, 0.015071570205789586, 1.6571644711735225e-06],
                [1172.1322901571223, 1272.8969286837373, 1437.557746508848, 2643981.980613781, 5644960.1495587025,
                 6319984.010862878, 11.608379366203652, 0.0153098728874533, 1.7558263639829205e-06],
                [1115.3505637801043, 1250.1440195093382, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6202055.6442852765, 9.899845029941671, 0.01491459237885602, 1.6761119331505416e-06],
                [1102.9707247869544, 1260.5869822935006, 1420.5917316831233, 2825130.0490485826, 5151932.1545183025,
                 6329254.498107326, 9.870483944224635, 0.01508497956653298, 1.6471410354130274e-06],
                [1106.8161654815144, 1260.5869822935006, 1428.2938999164598, 2756856.4798341207, 4987417.102943326,
                 6252960.904134171, 11.324658824884459, 0.01508497956653298, 1.6859236759203733e-06],
                [1171.886230946291, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5541510.5883321315,
                 6254834.865253087, 11.602935611820246, 0.015129553230613428, 1.646211033468178e-06],
                [1124.3787600097082, 1256.786782905013, 1432.7380531614376, 2704309.8939345954, 5135842.055580591,
                 6339744.606807769, 10.010198748307836, 0.014604814079047299, 1.64751e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014416923144892024, 1.746960716329272e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1111.4397194962298, 1252.3352159902076, 1431.2324989413153, 2834053.9104807484, 5073692.871060503,
                 6205159.784569372, 9.962970738526543, 0.014995216559648368, 1.64441131717817e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.31993774216143, 0.01508497956653298, 1.675718750508336e-06],
                [1094.9494294625033, 1260.5869822935006, 1420.5917316831233, 2825130.0490485826, 5151932.1545183025,
                 6328379.705405241, 9.703511954746483, 0.01508497956653298, 1.6477426444753308e-06],
                [1114.8364994143328, 1276.755042176067, 1430.0554994299264, 2711709.7515290566, 5061252.899416794,
                 6843779.100678993, 9.962874958185285, 0.014995216559648368, 1.6435718248748259e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.787151112326503, 0.014995216559648368, 1.6865448209421645e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2645497.698072738, 5634195.507417285,
                 6253461.142118783, 11.538003087103846, 0.015129553230613428, 1.646211033468178e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1165.821735989405, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1140.2618729167225, 1259.7696328062204, 1425.9492710629509, 2715399.6210057866, 5204539.152162562,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1102.2121241384732, 1260.5869822935006, 1425.726056523609, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6414678246117103e-06],
                [1172.1322901571223, 1275.3575942303487, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1100.3826382823288, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6414678246117103e-06],
                [1172.1322901571223, 1273.6063854006757, 1424.647927352167, 2787890.321944607, 4970750.218790298,
                 6264496.097901617, 11.69640114007929, 0.015654224593012078, 1.7492546775003008e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1115.3505637801043, 1259.877397589078, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6211148.3834170075, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6211148.3834170075, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1114.9126539360466, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.64751e-06],
                [1045.94074477175, 1257.5582265374605, 1428.1727195332353, 2676965.0315407575, 5029083.46383821,
                 6252781.131346029, 9.912726746719038, 0.01508497956653298, 1.6858501133596892e-06],
                [1050.9762404493679, 1258.4266239556036, 1440.177953468843, 2711337.2097504647, 4951513.490183301,
                 6252781.131346029, 9.840366214118836, 0.01508497956653298, 1.6858501133596892e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.462388216486575, 0.014995216559648368, 1.6865448209421645e-06],
                [1114.9126539360466, 1263.4435351891264, 1420.7016199249624, 2711709.7515290566, 5234415.800976389,
                 6351339.408023649, 9.599662957877094, 0.01508497956653298, 1.64751e-06],
                [1124.8314582363391, 1244.7486449823257, 1428.1727195332353, 2711109.3787926026, 4957035.652219516,
                 6252781.131346029, 9.912726746719038, 0.015131656963552058, 1.6858501133596892e-06],
                [1095.4692686979017, 1253.9910313461303, 1434.1536421639848, 2640666.855664094, 5123686.718780826,
                 6266431.849747426, 10.336890255390792, 0.01508497956653298, 1.64751e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1050.9762404493679, 1253.7997759357147, 1446.6888589765636, 2675995.973765552, 5086221.215584276,
                 6307046.665064646, 9.939222256778564, 0.0148888930145161, 1.6858501133596892e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402352374322306, 1.7961069170362274e-06],
                [1124.0077507460633, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5341886.884732996,
                 6274517.334926196, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06],
                [1114.9126539360466, 1256.6998042531457, 1432.7380531614376, 2711709.7515290566, 5135842.055580591,
                 6339744.606807769, 10.010294528649096, 0.014604814079047299, 1.64751e-06],
                [1089.037314605264, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5198282.044819644,
                 6403335.37885126, 9.87071144774333, 0.015004521873080429, 1.6462972507927568e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6382155.347856246, 10.793393892973041, 0.014402352374322306, 1.772746704805386e-06],
                [1115.3505637801043, 1250.1440195093382, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6761119331505416e-06],
                [1115.3505637801043, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 10.139251868310934, 0.01491459237885602, 1.7006950945250131e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402352374322306, 1.772746704805386e-06],
                [1117.1225010723806, 1245.0461992541775, 1408.2360625510769, 2809265.192909424, 5151932.1545183025,
                 6034517.235726284, 9.899845029941671, 0.014343389674800526, 1.7139568106579685e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014402711636037064, 1.8287965466133632e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2769369.5842076726, 4987417.102943326,
                 6252960.904134171, 11.31993774216143, 0.01508497956653298, 1.675718750508336e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4982497.032409398,
                 6316982.864584648, 11.401218941020598, 0.015071570205789586, 1.6571644711735225e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.923151891580432, 0.01467653963001504, 1.6827081430883913e-06],
                [1172.1322901571223, 1272.8969286837373, 1437.557746508848, 2643981.980613781, 5644960.1495587025,
                 6319984.010862878, 11.608379366203652, 0.0153098728874533, 1.7558263639829205e-06],
                [1096.634295467943, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1115.3505637801043, 1250.1440195093382, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6202055.6442852765, 9.899845029941671, 0.01491459237885602, 1.6761119331505416e-06],
                [1102.9707247869544, 1260.5869822935006, 1420.5917316831233, 2825130.0490485826, 5151932.1545183025,
                 6329254.498107326, 9.870483944224635, 0.01508497956653298, 1.6471410354130274e-06],
                [1103.1986748240922, 1259.20529625876, 1421.6358662409607, 2794529.276377869, 5138339.579249148,
                 6317443.2095390335, 9.959965929147502, 0.014876786169789531, 1.6827081430883913e-06],
                [1106.8161654815144, 1260.5869822935006, 1428.2938999164598, 2756856.4798341207, 4987417.102943326,
                 6252960.904134171, 11.324658824884459, 0.01508497956653298, 1.6859236759203733e-06],
                [1171.886230946291, 1261.2561027220345, 1424.647927352167, 2674527.9391689897, 5541510.5883321315,
                 6254834.865253087, 11.602935611820246, 0.015129553230613428, 1.646211033468178e-06],
                [1124.3787600097082, 1256.786782905013, 1432.7380531614376, 2704309.8939345954, 5135842.055580591,
                 6339744.606807769, 10.010198748307836, 0.014604814079047299, 1.64751e-06],
                [1172.1322901571223, 1261.353970231427, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014416923144892024, 1.746960716329272e-06],
                [1100.1408530781362, 1259.20529625876, 1420.7016199249624, 2794529.276377869, 5137466.622582068,
                 6291895.722338178, 9.959965929147502, 0.014877788107490625, 1.6827081430883913e-06],
                [1111.4397194962298, 1252.3352159902076, 1431.2324989413153, 2834053.9104807484, 5073692.871060503,
                 6205159.784569372, 9.962970738526543, 0.014995216559648368, 1.64441131717817e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.401218941020598, 0.01508497956653298, 1.6571644711735225e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.31993774216143, 0.01508497956653298, 1.675718750508336e-06],
                [1094.9494294625033, 1260.5869822935006, 1420.5917316831233, 2825130.0490485826, 5151932.1545183025,
                 6328379.705405241, 9.703511954746483, 0.01508497956653298, 1.6477426444753308e-06],
                [1106.8161654815144, 1260.5869822935006, 1424.6533094655529, 2756948.118884614, 4987417.102943326,
                 6252960.904134171, 11.29179270201854, 0.01508497956653298, 1.675718750508336e-06],
                [1114.8364994143328, 1276.755042176067, 1430.0554994299264, 2711709.7515290566, 5061252.899416794,
                 6843779.100678993, 9.962874958185285, 0.014995216559648368, 1.6435718248748259e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.787151112326503, 0.014995216559648368, 1.6865448209421645e-06],
                [1123.0873102981457, 1256.786782905013, 1432.7380531614376, 2704309.8939345954, 5135842.055580591,
                 6339744.606807769, 10.010198748307836, 0.0150714007214576, 1.6742977494042081e-06],
                [1172.1322901571223, 1261.2561027220345, 1424.647927352167, 2645497.698072738, 5634195.507417285,
                 6253461.142118783, 11.538003087103846, 0.015129553230613428, 1.646211033468178e-06],
                [1115.400748739093, 1265.5877782356972, 1428.3010571928912, 2714079.301423972, 5048177.206187649,
                 6403353.83122605, 11.579324789874297, 0.015321153721496901, 1.6447615608703846e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1165.0546070424907, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1165.821735989405, 1263.903713429593, 1424.647927352167, 2674819.702993904, 5540572.403677184,
                 6264496.097901617, 11.21124190291663, 0.015844898331152327, 1.7492546775003008e-06],
                [1140.2618729167225, 1259.7696328062204, 1425.9492710629509, 2715399.6210057866, 5204539.152162562,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1140.2618729167225, 1259.7696328062204, 1425.9492710629509, 2715399.6210057866, 5204539.152162562,
                 6129007.601007714, 9.459700490332747, 0.01508497956653298, 1.8717425252296465e-06],
                [1112.0739142572368, 1249.5129354938679, 1421.4288718354624, 2711709.7515290566, 5177667.749296468,
                 6205159.784569372, 9.462388216486575, 0.014995216559648368, 1.6977428700459318e-06],
                [1102.2121241384732, 1260.5869822935006, 1425.726056523609, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6414678246117103e-06],
                [1172.1322901571223, 1275.3575942303487, 1424.647927352167, 2674527.9391689897, 5634195.507417285,
                 6253461.142118783, 11.605501217123276, 0.015129553230613428, 1.646211033468178e-06],
                [1100.3826382823288, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6414678246117103e-06],
                [1112.0739142572368, 1252.4221946420748, 1421.4288718354624, 2711709.7515290566, 5281477.560804546,
                 6011677.073363474, 9.787151112326503, 0.014995216559648368, 1.6768806535715052e-06],
                [1192.3898611011628, 1261.1135898526488, 1424.647927352167, 2643981.980613781, 5539495.081817166,
                 6122658.492254281, 11.426790838912128, 0.014416923144892024, 1.746960716329272e-06],
                [1172.1322901571223, 1273.6063854006757, 1424.647927352167, 2787890.321944607, 4970750.218790298,
                 6264496.097901617, 11.69640114007929, 0.015654224593012078, 1.7492546775003008e-06],
                [1115.3671211181227, 1260.5869822935006, 1420.7016199249624, 2794529.276377869, 5151932.1545183025,
                 6346803.504583463, 9.87071144774333, 0.01508497956653298, 1.6726603388725022e-06],
                [1114.8242474028489, 1259.3587163655109, 1417.21404073115, 2809265.192909424, 5151932.1545183025,
                 6252740.844923852, 9.899845029941671, 0.01491459237885602, 1.6416646007583103e-06]]
            self.y_data = [[2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [84.39742275948912, 33.78314907324829, 265.4014460620532],
                           [932.6069641175573, 400.59055218545626, 749.2704358802002],
                           [1219.4217619547032, 483.40873541839017, 827.8547810464862],
                           [590.3969996969831, 361.1751758113917, 897.3239825360729],
                           [28.069171949228203, 1003.1312033418135, 1027.3516684209444],
                           [3293.6417749350526, 4000.296651387464, 2749.9811204802127],
                           [2689.888797413033, 1995.423712500253, 2913.153419949472],
                           [3641.408349259386, 4149.027032556554, 2979.8648961738068],
                           [64.95870380673843, 2037.0352581288441, 3142.3452838768544],
                           [5050.042406848, 3353.631362261292, 4046.1731460186083],
                           [3211.138664604769, 5346.419128787817, 4918.9261583296375],
                           [4258.619318468715, 5989.02266715898, 5220.525994374762],
                           [8936.189342070667, 9965.239758350204, 7563.000813830158],
                           [6021.767634834487, 8903.495264146026, 7814.127457858662],
                           [6017.482913124648, 9878.662058584368, 9219.928973267353],
                           [10383.663899872603, 11692.453275702828, 9778.423392895807],
                           [9959.714499578859, 12100.473650714735, 10815.113656904918],
                           [4728.725795408572, 9833.320115679087, 11373.539981017966],
                           [7040.3231430446585, 11659.214148088358, 11790.152463711856],
                           [6625.1594545319385, 10968.926234932243, 12625.282931427295],
                           [15642.10277555957, 16893.988266075045, 14863.564593947294],
                           [18607.931674621763, 19082.778723217332, 15013.161810006111],
                           [10626.98560266889, 14816.17950387497, 15506.383314966086],
                           [19339.61783872236, 19682.730138362174, 15674.580866406783],
                           [22189.138934738672, 21973.758051439312, 17867.900673260734],
                           [23690.59769921251, 25783.93771324694, 20809.23996609462],
                           [14967.7196505778, 22248.082939376614, 22731.8285500197],
                           [23423.407106244136, 28591.62813777676, 26664.35093389845],
                           [19075.645612367698, 27207.437259519545, 28128.538674419888],
                           [29997.930337349753, 33405.527835033296, 28873.62124640369],
                           [24820.89628524226, 33586.311728094224, 31719.51145267233],
                           [37282.62405854246, 38046.09208233556, 33387.387843928715],
                           [28410.627000789853, 35686.40139112738, 36220.30055895746],
                           [40471.242966725076, 43007.140870292154, 37868.621210244215],
                           [37398.14699282434, 42565.10811820573, 37870.35351039514],
                           [45479.57029289774, 47732.36757601205, 41595.563139189384],
                           [32552.29129704946, 44403.59499708624, 45751.504789978455],
                           [45786.72576366185, 50791.11058567032, 48223.58538251199],
                           [51020.88136265659, 54357.122848508094, 50465.40361704311],
                           [61881.73733921197, 59644.849846935205, 51400.45602940709],
                           [60457.47777367473, 60647.47618547739, 51432.39988001411],
                           [62646.79677225086, 58733.17045634589, 53097.700319228185],
                           [71247.0421208742, 72098.19782720233, 67199.0151309894],
                           [63957.85300621107, 77501.03448860586, 81158.49902713434],
                           [93260.27723239115, 94410.9579982488, 86854.8936256963],
                           [69247.51335865381, 84142.67184010833, 89262.52599409055],
                           [96023.63022667098, 99075.37552357498, 97413.0256048264],
                           [115089.56740480401, 109259.95186664716, 99354.46311566826],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [23.975982777695172, 6.757366123987301, 191.0233853494245],
                           [84.39742275948912, 33.78314907324829, 265.4014460620532],
                           [28.069171949228203, 1003.1312033418135, 1027.3516684209444],
                           [3293.5437591554783, 130.18670022608515, 25.37306097654482],
                           [932.6069641175573, 400.59055218545626, 749.2704358802002],
                           [64.95870380673843, 2037.0352581288441, 3142.3452838768544],
                           [590.3969996969831, 361.1751758113917, 897.3239825360729],
                           [1195.5075793930594, 526.1504936826086, 932.2775739448363],
                           [615.7980966174894, 2371.6328956458747, 2197.19994390353],
                           [84.94494447429759, 2139.8419082286405, 3265.87568834472],
                           [1219.4217619547032, 483.40873541839017, 827.8547810464862],
                           [2724.5059366381524, 3420.7700687544716, 2292.325941125391],
                           [2144.00329464219, 4811.868522654193, 4324.293371510733],
                           [2689.888797413033, 1995.423712500253, 2913.153419949472],
                           [3227.723554235312, 3931.1954332489513, 2694.6979836887685],
                           [3102.5415684431687, 5156.512214548655, 4696.911509918105],
                           [5050.042406848, 3353.631362261292, 4046.1731460186083],
                           [3211.138664604769, 5346.419128787817, 4918.9261583296375],
                           [3293.6417749350526, 4000.296651387464, 2749.9811204802127],
                           [3636.4519255889963, 7701.036499127755, 8649.018536861058],
                           [3641.408349259386, 4149.027032556554, 2979.8648961738068],
                           [7094.934117160581, 4963.715116596285, 5681.255547927741],
                           [4258.619318468715, 5989.02266715898, 5220.525994374762],
                           [4728.725795408572, 9833.320115679087, 11373.539981017966],
                           [6021.767634834487, 8903.495264146026, 7814.127457858662],
                           [8936.189342070667, 9965.239758350204, 7563.000813830158],
                           [5108.537042399923, 8846.416734250804, 8264.988876899557],
                           [6017.482913124648, 9878.662058584368, 9219.928973267353],
                           [6749.343731353359, 9786.717933043677, 8863.214695950994],
                           [8699.036586466838, 11186.733448158204, 9991.249310163385],
                           [9685.64062202062, 10986.969888115693, 9156.021071890882],
                           [6854.662907936784, 11100.471712774037, 10984.746603997097],
                           [6625.1594545319385, 10968.926234932243, 12625.282931427295],
                           [7040.3231430446585, 11659.214148088358, 11790.152463711856],
                           [9955.857397038104, 11269.85863024913, 9404.157063167368],
                           [10383.663899872603, 11692.453275702828, 9778.423392895807],
                           [9959.714499578859, 12100.473650714735, 10815.113656904918],
                           [7855.450025109316, 13741.228884635286, 13987.2032403042],
                           [9539.580737744978, 15726.640872161857, 18121.942676224597],
                           [15272.943154076545, 16049.89530306775, 12544.873510185207],
                           [10626.98560266889, 14816.17950387497, 15506.383314966086],
                           [15642.10277555957, 16893.988266075045, 14863.564593947294],
                           [11972.885691299613, 16740.618540004456, 15604.373301592324],
                           [14967.7196505778, 22248.082939376614, 22731.8285500197],
                           [18607.931674621763, 19082.778723217332, 15013.161810006111],
                           [15600.764007884087, 20229.29363534731, 19865.958258264185],
                           [17299.3462527832, 22185.51377411193, 20908.129208334474],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [23.975982777695172, 6.757366123987301, 191.0233853494245],
                           [84.39742275948912, 33.78314907324829, 265.4014460620532],
                           [0.8807537900138819, 308.8706699734264, 244.54305628933315],
                           [348.7682075032219, 79.03123825619325, 251.45129387305434],
                           [3293.5437591554783, 130.18670022608515, 25.37306097654482],
                           [222.55731422260303, 146.9065009953099, 554.3482836092844],
                           [28.069171949228203, 1003.1312033418135, 1027.3516684209444],
                           [875.379542803012, 99.28324447428258, 466.83123685949204],
                           [932.6069641175573, 400.59055218545626, 749.2704358802002],
                           [64.95870380673843, 2037.0352581288441, 3142.3452838768544],
                           [590.3969996969831, 361.1751758113917, 897.3239825360729],
                           [419.4517590701555, 1833.403663275858, 1907.3039740444506],
                           [666.5041189662938, 369.17520959179217, 852.6852888192585],
                           [1195.5075793930594, 526.1504936826086, 932.2775739448363],
                           [615.7980966174894, 2371.6328956458747, 2197.19994390353],
                           [84.94494447429759, 2139.8419082286405, 3265.87568834472],
                           [1219.4217619547032, 483.40873541839017, 827.8547810464862],
                           [806.4250159418375, 2573.897296351246, 2788.011669077916],
                           [2189.475081270695, 1751.4863945013165, 2779.906539602099],
                           [1511.207677762512, 2088.7074393216253, 1254.3211433445492],
                           [2724.5059366381524, 3420.7700687544716, 2292.325941125391],
                           [2144.00329464219, 4811.868522654193, 4324.293371510733],
                           [2265.587989319462, 4618.719697500087, 3911.8825835162193],
                           [3337.817233042934, 2204.27579141391, 2869.87751326041],
                           [2689.888797413033, 1995.423712500253, 2913.153419949472],
                           [2850.780079119669, 3539.006290856695, 2384.2371130748134],
                           [5050.042406848, 3353.631362261292, 4046.1731460186083],
                           [3102.5415684431687, 5156.512214548655, 4696.911509918105],
                           [3227.723554235312, 3931.1954332489513, 2694.6979836887685],
                           [3155.6670736396673, 6458.660214702686, 6019.507672510742],
                           [3293.6417749350526, 4000.296651387464, 2749.9811204802127],
                           [3211.138664604769, 5346.419128787817, 4918.9261583296375],
                           [3333.657830110825, 4036.580611849707, 2778.6004285400895],
                           [3636.4519255889963, 7701.036499127755, 8649.018536861058],
                           [3641.408349259386, 4149.027032556554, 2979.8648961738068],
                           [3881.6456447322603, 4524.469986396184, 3141.099951069475],
                           [3641.863977178802, 7708.587433808557, 8656.844594135235],
                           [7094.934117160581, 4963.715116596285, 5681.255547927741],
                           [4028.4909074635952, 7529.888366003115, 7053.717520267663],
                           [3904.834571608711, 8901.031785138372, 10520.56571402266],
                           [4258.619318468715, 5989.02266715898, 5220.525994374762],
                           [4714.951634369974, 7990.049683599002, 7473.374103872339],
                           [4896.420563119286, 7895.896204411419, 7067.661070062295],
                           [8830.145285436984, 6453.843953175076, 7234.323435982808],
                           [4728.725795408572, 9833.320115679087, 11373.539981017966],
                           [5108.537042399923, 8846.416734250804, 8264.988876899557],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [1698.424036289286, 0.01299795111754957, 168.37538497510388],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [10.666841821962958, 0.10829104953011984, 120.397148384552],
                           [1725.6082432411808, 14.39254270781067, 32.963287653975904],
                           [24.29419694458095, 6.387867606153993, 188.9083418160392],
                           [0.8807537900138819, 308.8706699734264, 244.54305628933315],
                           [3293.5437591554783, 130.18670022608515, 25.37306097654482],
                           [23.975982777690717, 6.757366123987301, 191.0233853494245],
                           [23.975982777695172, 6.757366123987301, 191.0233853494245],
                           [3614.1945490081284, 403.56408090706645, 64.64230859699205],
                           [84.39742275948912, 33.78314907324829, 265.4014460620532],
                           [49.500236767375256, 748.8184597207388, 1194.8425780038576],
                           [59.29231378399418, 73.37614553766984, 507.52095741347256],
                           [273.34395127563, 47.55919777853988, 219.62646505048573],
                           [231.49253129291893, 59.0836955422837, 264.88592492996935],
                           [28.066275586976047, 763.7119287944752, 825.5639218708797],
                           [222.55731422260303, 146.9065009953099, 554.3482836092844],
                           [121.64342076581877, 120.4858584700321, 595.5160139781838],
                           [348.7682075032219, 79.03123825619325, 251.45129387305434],
                           [28.069171949228203, 1003.1312033418135, 1027.3516684209444],
                           [451.4713618411107, 250.38998334942517, 710.4584995815518],
                           [64.95870380673843, 2037.0352581288441, 3142.3452838768544],
                           [875.379542803012, 99.28324447428258, 466.83123685949204],
                           [103.35966736125393, 1054.0901300295363, 1140.2307760836932],
                           [932.6069641175573, 400.59055218545626, 749.2704358802002],
                           [573.8775186027166, 348.50456444541703, 878.1386230306048],
                           [84.94494447429759, 2139.8419082285986, 3265.87568834472],
                           [2215.8745136846023, 497.1308621557844, 469.140767742917],
                           [372.330578765052, 1872.1123080729162, 1732.8198568681098],
                           [419.4517590701555, 1833.403663275858, 1907.3039740444506],
                           [666.5041189662938, 369.17520959179217, 852.6852888192585],
                           [84.94494447429759, 2139.8419082286405, 3265.87568834472],
                           [590.3969996969831, 361.1751758113917, 897.3239825360729],
                           [1219.4217619547032, 483.40873541839017, 827.8547810464862],
                           [156.00522571860756, 2373.563916579705, 3498.689531915568],
                           [1195.5075793930594, 526.1504936826086, 932.2775739448363],
                           [645.5498562605851, 401.99901838595474, 957.277825073818],
                           [615.7980966174894, 2371.6328956458747, 2197.19994390353],
                           [1272.8274396706872, 656.4073205730202, 1099.5969877098196],
                           [1049.756553208089, 2433.9776637581167, 2272.2434364372602],
                           [686.8351067732438, 2487.4607641016514, 2300.204812129034],
                           [1253.8496497005235, 844.2912509695493, 1537.8776451832111],
                           [806.4250159418375, 2573.897296351246, 2788.011669077916],
                           [2189.475081270695, 1751.4863945013165, 2779.906539602099],
                           [2287.613789100325, 1348.6367817905837, 1878.2952537142526],
                           [1511.207677762512, 2088.7074393216253, 1254.3211433445492],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [1698.424036289286, 0.01299795111754957, 168.37538497510388],
                           [1731.6323778993533, 16.43284128682213, 28.548313164582687],
                           [10.666841821962958, 0.10829104953011984, 120.397148384552],
                           [1725.6082432411808, 14.39254270781067, 32.963287653975904],
                           [142.6784616644594, 34.32191793803289, 17.293611825289027],
                           [30.995120357671627, 1.129181305210288, 47.36986262786443],
                           [0.7095485344228267, 26.236956930188985, 21.312956679909245],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [24.29419694458095, 6.387867606153993, 188.9083418160392],
                           [15.857182051422825, 28.07431174700396, 40.861335425806786],
                           [0.8807537900138819, 308.8706699734264, 244.54305628933315],
                           [23.975982777695172, 6.757366123987301, 191.02338534941194],
                           [23.975982777690717, 6.757366123987301, 191.0233853494245],
                           [3293.5437591554783, 130.18670022608515, 25.37306097654482],
                           [41.98736182975695, 4.393729977420127, 167.7678514646283],
                           [184.7742346627305, 25.47323030257608, 170.68879273981022],
                           [85.44346461306462, 268.5879004525793, 69.17680850300846],
                           [3614.1945490081284, 403.56408090706645, 64.64230859699205],
                           [21.754435630673875, 715.2488256866449, 764.9568460498299],
                           [23.975982777695172, 6.757366123987301, 191.0233853494245],
                           [28.066275586976047, 763.7119287944752, 825.5639218708797],
                           [49.500236767375256, 748.8184597207388, 1194.8425780038576],
                           [59.29231378399418, 73.37614553766984, 507.52095741347256],
                           [71.6396219959614, 9.050893401635866, 199.6203564923143],
                           [84.39742275948912, 33.78314907324829, 265.4014460620532],
                           [273.34395127563, 47.55919777853988, 219.62646505048573],
                           [231.49253129291893, 59.0836955422837, 264.88592492996935],
                           [28.069171949223385, 1003.1312033418135, 1027.3516684209444],
                           [28.069171949228203, 1003.1312033418135, 1027.3516684209444],
                           [121.64342076581877, 120.4858584700321, 595.5160139781838],
                           [348.7682075032219, 79.03123825619325, 251.45129387305434],
                           [221.141679359564, 145.9022039732921, 552.4093156027258],
                           [324.35512380912814, 159.5829028190041, 290.5371171371144],
                           [103.16861608217945, 1051.7319992608554, 1136.4449105738381],
                           [348.7682075032389, 79.03123825620942, 251.45129387306875],
                           [221.65251822339494, 547.7839636484961, 1144.784349669977],
                           [39.15158125464487, 1200.4924713762389, 1329.7118311068864],
                           [222.55731422260303, 146.9065009953099, 554.3482836092844],
                           [39.14468497062805, 1380.7260489195446, 2655.37490097492],
                           [239.66160535506611, 159.67815422239494, 577.843151942103],
                           [2119.566971197377, 470.2314330348149, 454.9514202697302],
                           [64.95870380673843, 2037.0352581288441, 3142.3452838768544],
                           [785.003281477448, 130.86712665484018, 530.7839050668333],
                           [103.35966736125393, 1054.0901300295363, 1140.2307760836932],
                           [875.379542803012, 99.28324447428258, 466.83123685949204],
                           [94.37644003738922, 2139.4004585170637, 3223.4173407953876],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [1698.424036289286, 0.01299795111754957, 168.37538497510388],
                           [10.666841821962958, 0.10829104953011984, 120.397148384552],
                           [203.594770917816, 19.824142184046142, 11.293382524271141],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [142.6784616644594, 34.32191793803289, 17.293611825289027],
                           [325.5962494985258, 2.0637405787549548, 5.210879374044886],
                           [0.7095485344228267, 26.236956930188985, 21.312956679909245],
                           [0.7835586192627262, 1.676351289965749, 100.34701702002148],
                           [3.5731860780187756, 53.16877490582369, 3.009071202302619],
                           [30.995120357671627, 1.129181305210288, 47.36986262786443],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [1533.6139300255934, 14.159421830163234, 26.415486450323698],
                           [24.29419694458095, 6.387867606153993, 188.9083418160392],
                           [15.857182051422825, 28.07431174700396, 40.861335425806786],
                           [153.2434867525576, 7.000938846496265, 113.48710702569375],
                           [1653.3882605950482, 8.670579299028148, 53.8653020856373],
                           [23.975982777695172, 6.757366123987301, 191.02338534941194],
                           [0.8440182840134394, 80.34587155530498, 1.0382882544643537],
                           [23.975982777690717, 6.757366123987301, 191.0233853494245],
                           [41.98736182975695, 4.393729977420127, 167.7678514646283],
                           [1679.8161835319363, 11.843501245199953, 35.4908477026538],
                           [1731.6323778993533, 16.43284128682213, 28.548313164582687],
                           [1731.6323778993533, 16.43284128682213, 28.548313164582687],
                           [0.8807537900138819, 308.8706699734264, 244.54305628933315],
                           [2737.591321461413, 112.62471052602751, 6.859757602699076],
                           [184.7742346627305, 25.47323030257608, 170.68879273979834],
                           [1725.6082432411808, 14.39254270781067, 32.963287653975904],
                           [5.775639905375375, 190.72659909664773, 76.81780388215864],
                           [1082.1781362040533, 13.305516743552916, 171.28408521191926],
                           [7.106795994094727, 217.37001170891475, 36.21695057138824],
                           [21.31264283001281, 157.55891288396842, 21.479461987177856],
                           [2104.212181935844, 5.23347954486414, 138.81063798812713],
                           [23.975982777695172, 6.757366123987301, 191.0233853494245],
                           [184.7742346627305, 25.47323030257608, 170.68879273981022],
                           [85.44346461306462, 268.5879004525793, 69.17680850300846],
                           [2288.359976084589, 28.994927748993238, 68.79006264030811],
                           [21.754435630673875, 715.2488256866449, 764.9568460498299],
                           [23.975982777699624, 6.757366123989664, 191.0233853494245],
                           [3293.5437591554783, 130.18670022608515, 25.37306097654482],
                           [28.783696784989797, 9.384950799991389, 203.82246493282764],
                           [25.08181172238115, 13.693906521142155, 241.73334424006276],
                           [3489.7925919810664, 367.75052437072355, 52.34093336755523],
                           [71.6396219959614, 9.050893401635866, 199.6203564923143],
                           [34.520498791605654, 326.57694614810976, 418.9483490110867],
                           [28.066275586976047, 763.7119287944752, 825.5639218708797],
                           [31.923406125744755, 489.4423110958731, 605.4021271616411],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [1698.424036289286, 0.01299795111754957, 168.37538497510388],
                           [142.6784616644594, 34.32191793803289, 17.293611825289027],
                           [0.5772211124468531, 74.2102600709729, 0.30155031314624153],
                           [0.6424484977527564, 26.608086273080183, 20.989619496872145],
                           [0.0008582001632676055, 69.44727686411046, 0.3071014948342446],
                           [26.13058550809608, 2.141480924070444, 42.10217361371894],
                           [203.594770917816, 19.824142184046142, 11.293382524271141],
                           [30.995120357671627, 1.129181305210288, 47.36986262786443],
                           [10.666841821962958, 0.10829104953011984, 120.397148384552],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [1595.095376997128, 0.0610148161277292, 150.2366866729055],
                           [207.6312233512305, 15.22131442465849, 6.291022749656092],
                           [325.5962494985258, 2.0637405787549548, 5.210879374044886],
                           [3.5731860780187756, 53.16877490582369, 3.009071202302619],
                           [11.29966006255687, 0.5830898866905679, 104.71988885779116],
                           [0.7835586192627262, 1.676351289965749, 100.34701702002148],
                           [3.20923396464739, 0.9783319000006608, 97.54501629398654],
                           [0.7095485344228267, 26.236956930188985, 21.312956679909245],
                           [4.278577793961269, 15.360752799924295, 33.54765088157793],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [7.045007797487232, 18.17242417586246, 51.9346785536019],
                           [1865.6343979646495, 4.4856785492880515, 112.17277986515508],
                           [1679.762645123694, 1.0718170703870942, 208.28785635417034],
                           [153.2434867525576, 7.000938846496265, 113.48710702569375],
                           [15.857182051422825, 28.07431174700396, 40.861335425806786],
                           [17.541944891867267, 3.832153772764636, 174.53972273700833],
                           [0.05315462890899393, 138.61630888519258, 46.322536093479506],
                           [1679.8161835319363, 11.843501245199953, 35.4908477026538],
                           [29.745105957679694, 2.7021323444944745, 166.9661513268865],
                           [4.804381395324434, 5.799601703728597, 236.36883796813848],
                           [0.8236638930928141, 80.21190969293276, 1.0224442381977954],
                           [224.6297101975616, 19.77992966447502, 14.944769219789107],
                           [1925.8072004787518, 38.88394581165614, 10.644278459309703],
                           [35.10481877686825, 66.5769325905058, 11.834380520576785],
                           [1656.9560860562453, 9.742185462390303, 49.871561672973684],
                           [44.436501237597625, 55.3183631663894, 23.14432518736778],
                           [1533.6139300255934, 14.159421830163234, 26.415486450323698],
                           [1653.3882605950482, 8.670579299028148, 53.8653020856373],
                           [1731.6323778993533, 16.43284128682213, 28.548313164582687],
                           [180.85857012332116, 13.460337053275351, 135.23383236174564],
                           [1725.6082432411808, 14.39254270781067, 32.963287653975904],
                           [0.8440182840134394, 80.34587155530498, 1.0382882544643537],
                           [21.27072387211216, 5.445982970611186, 183.9721434425518],
                           [1731.6323778993533, 16.43284128682213, 28.548313164582687],
                           [2104.212181935844, 5.23347954486414, 138.81063798812713],
                           [41.98736182975695, 4.393729977420127, 167.7678514646283],
                           [184.7742346627305, 25.47323030257608, 170.68879273979834],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [1698.4240362893236, 0.01299795111744588, 168.37538497509206],
                           [1698.424036289286, 0.01299795111754957, 168.37538497510388],
                           [202.35113541078235, 17.063449150127163, 7.426857106060623],
                           [207.6312233512305, 15.22131442465849, 6.291022749656092],
                           [0.5772211124468531, 74.2102600709729, 0.30155031314624153],
                           [0.0008582001632676055, 69.44727686411046, 0.3071014948342446],
                           [30.995120357671627, 1.129181305210288, 47.36986262786443],
                           [325.5962494985258, 2.0637405787549548, 5.210879374044886],
                           [11.29966006255687, 0.5830898866905679, 104.71988885779116],
                           [0.04022760342216386, 4.175523328298983, 83.98280887651758],
                           [1595.095376997128, 0.0610148161277292, 150.2366866729055],
                           [271.78999830670466, 8.31837941503812, 5.23694910379756],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [0.7866796241855081, 0.7482338284857643, 126.95937544720896],
                           [10.666841821962958, 0.10829104953011984, 120.397148384552],
                           [21.513257606901835, 10.778569167839118, 10.748948298837892],
                           [3.5731860780187756, 53.16877490582369, 3.009071202302619],
                           [242.04472754817434, 10.71498387628563, 4.9484867291264365],
                           [1.9860473235411782, 19.190032496109808, 30.48515479005733],
                           [19.35606118638945, 4.205685765701538, 34.632844709404786],
                           [0.7835586192627262, 1.676351289965749, 100.34701702002148],
                           [26.13058550809608, 2.141480924070444, 42.10217361371894],
                           [0.03724473476969815, 33.41040897798797, 10.286782340230328],
                           [3.20923396464739, 0.9783319000006608, 97.54501629398654],
                           [4.278577793961269, 15.360752799924295, 33.54765088157793],
                           [0.6424484977527564, 26.608086273080183, 20.989619496872145],
                           [0.7095485344228267, 26.236956930188985, 21.312956679909245],
                           [6.427609958441836, 23.840182714890954, 44.284965427359715],
                           [15.857182051422825, 28.07431174700396, 40.861335425806786],
                           [23.825439029475444, 53.03309642538138, 17.672888856222958],
                           [7.045007797487232, 18.17242417586246, 51.9346785536019],
                           [1679.8161835319363, 11.843501245199953, 35.4908477026538],
                           [0.05315462890899393, 138.61630888519258, 46.322536093479506],
                           [1679.762645123694, 1.0718170703870942, 208.28785635417034],
                           [0.08109306187714843, 73.45148299822735, 0.6117602408145528],
                           [203.594770917816, 19.824142184046142, 11.293382524271141],
                           [1925.8072004787518, 38.88394581165614, 10.644278459309703],
                           [35.10481877686825, 66.5769325905058, 11.834380520576785],
                           [1499.6921840168209, 21.30156047154324, 10.90709964228792],
                           [20.433942904429184, 2.397233490174813, 150.66191766940582],
                           [224.6297101975616, 19.77992966447502, 14.944769219789107],
                           [3.432233184867408, 7.753965884968214, 266.51532240702346],
                           [1656.9560860562453, 9.742185462390303, 49.871561672973684],
                           [141.19933032184335, 3.75305621640723, 96.05629812221983],
                           [14.726489896274677, 2.6692621943225103, 166.41764257521436],
                           [1703.271436457758, 0.1985189241203648, 185.50258052300742],
                           [1853.7623372405394, 0.343920445102121, 172.23451713038236],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1698.4240362893236, 0.01299795111744588, 168.37538497509206],
                           [1698.424036289286, 0.01299795111754957, 168.37538497510388],
                           [21.513257606901835, 10.778569167839118, 10.748948298837892],
                           [271.78999830670466, 8.31837941503812, 5.23694910379756],
                           [3.20923396464739, 0.9783319000006608, 97.54501629398654],
                           [60.61694367562461, 0.14830423561726147, 36.8651600451074],
                           [10.079984508478312, 24.118566906936138, 10.68647837601356],
                           [202.35113541078235, 17.063449150127163, 7.426857106060623],
                           [0.7095485344228267, 26.236956930188985, 21.312956679909245],
                           [30.995120357671627, 1.129181305210288, 47.36986262786443],
                           [19.35606118638945, 4.205685765701538, 34.632844709404786],
                           [2.066924661280738, 59.205654507634, 0.25671630833363357],
                           [1.9008424218631372, 1.283653785511257, 101.77420454461785],
                           [4.278577793961269, 15.360752799924295, 33.54765088157793],
                           [0.7620934383638515, 0.7578921509282203, 126.97386237967345],
                           [207.6312233512305, 15.22131442465849, 6.291022749656092],
                           [0.7835586192627262, 1.676351289965749, 100.34701702002148],
                           [52.639819051652154, 0.21522488892796612, 69.05420030384893],
                           [0.04022760342216386, 4.175523328298983, 83.98280887651758],
                           [0.0008582001632676055, 69.44727686411046, 0.3071014948342446],
                           [1.9860473235411782, 19.190032496109808, 30.48515479005733],
                           [0.5772211124468531, 74.2102600709729, 0.30155031314624153],
                           [3.9852763910344136, 15.931266049713381, 32.71854054148144],
                           [0.03724473476969815, 33.41040897798797, 10.286782340230328],
                           [10.61893647789901, 0.05439480284717122, 116.89945655820534],
                           [26.13058550809608, 2.141480924070444, 42.10217361371894],
                           [325.5962494985258, 2.0637405787549548, 5.210879374044886],
                           [0.7866796241855081, 0.7482338284857643, 126.95937544720896],
                           [0.3311937245486956, 4.007112370765605, 92.50276814459693],
                           [11.29966006255687, 0.5830898866905679, 104.71988885779116],
                           [3.5731860780187756, 53.16877490582369, 3.009071202302619],
                           [0.025005487349713213, 30.051842860070614, 19.657177301280463],
                           [0.6424484977527564, 26.608086273080183, 20.989619496872145],
                           [1653.228529879998, 0.020397299037547063, 146.85884956036674],
                           [1.0658944194750057, 7.411037756256374, 79.3047727472867],
                           [242.04472754817434, 10.71498387628563, 4.9484867291264365],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [5.872350009128298, 40.23887891083333, 1.5356909696758638],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [201.74103170829272, 36.95869328138994, 37.079974532382174],
                           [6.427609958441836, 23.840182714890954, 44.284965427359715],
                           [22.52980740486441, 97.65496047409178, 0.49527216279703423],
                           [0.08109306187714843, 73.45148299822735, 0.6117602408145528],
                           [203.594770917816, 19.824142184046142, 11.293382524271141],
                           [325.5962494985422, 2.0637405787549548, 5.210879374044886],
                           [23.825439029471003, 53.033096425388, 17.672888856222958],
                           [15.857182051422825, 28.07431174700396, 40.861335425806786],
                           [3.432233184867408, 7.753965884968214, 266.51532240702346],
                           [2612.6108434344833, 130.39766484633668, 0.09942136188993068],
                           [47.59248524935342, 0.0001029496470784214, 78.28045974217127],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [325.5962494985258, 2.0637405787549548, 5.210879374044886],
                           [5.872350009128298, 40.23887891083333, 1.5356909696758638],
                           [271.78999830670466, 8.31837941503812, 5.23694910379756],
                           [30.995120357671627, 1.129181305210288, 47.36986262786443],
                           [2.0669246612820458, 59.205654507627, 0.2567163083327119],
                           [2.3849227590784627, 21.43148324424069, 11.663695340517886],
                           [0.19789479813854127, 30.24494837254427, 18.10507383328362],
                           [0.0008582001632676055, 69.44727686411046, 0.3071014948342446],
                           [3.9852763910344136, 15.931266049713381, 32.71854054148144],
                           [0.7866796241855081, 0.7482338284857643, 126.95937544720896],
                           [60.61694367562461, 0.14830423561726147, 36.8651600451074],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [424.5679676884309, 0.2040284337768011, 0.9304574666914291],
                           [0.03724473476969815, 33.41040897798797, 10.286782340230328],
                           [52.639819051652154, 0.21522488892796612, 69.05420030384893],
                           [0.3311937245486956, 4.007112370765605, 92.50276814459693],
                           [0.7620934383638515, 0.7578921509282203, 126.97386237967345],
                           [26.43958287027143, 8.057308768842947, 29.663315905802882],
                           [0.04022760342216386, 4.175523328298983, 83.98280887651758],
                           [11.29966006255687, 0.5830898866905679, 104.71988885779116],
                           [202.35113541078235, 17.063449150127163, 7.426857106060623],
                           [19.35606118638945, 4.205685765701538, 34.632844709404786],
                           [0.7835586192627262, 1.676351289965749, 100.34701702002148],
                           [59.96877004716973, 0.1838447977803869, 36.301552188555476],
                           [207.6312233512174, 15.22131442465494, 6.291022749656092],
                           [10.61893647789901, 0.05439480284717122, 116.89945655820534],
                           [0.34180295245297704, 29.121580878328974, 8.155141795503095],
                           [28.217163070588278, 6.867308391765894, 15.732851261064393],
                           [10.079984508478312, 24.118566906936138, 10.68647837601356],
                           [242.04472754817434, 10.71498387628563, 4.9484867291264365],
                           [1.9860473235411782, 19.190032496109808, 30.48515479005231],
                           [1.9008424218631372, 1.283653785511257, 101.77420454461785],
                           [3.20923396464739, 0.9783319000006608, 97.54501629398654],
                           [4.278577793961269, 15.360752799924295, 33.54765088157793],
                           [1.4503832325323285, 47.755697670977675, 1.4175516791962168],
                           [2.066924661280738, 59.205654507634, 0.25671630833363357],
                           [21.513257606901835, 10.778569167839118, 10.748948298837892],
                           [0.5772211124468531, 74.2102600709729, 0.30155031314624153],
                           [0.42852253460509365, 20.62553301269027, 20.137503433872496],
                           [0.025005487349713213, 30.051842860070614, 19.657177301280463],
                           [0.0008582001632409618, 69.44727686411804, 0.3071014948342446],
                           [0.00012483735418128362, 6.572194892279124, 76.55242185202863],
                           [26.13058550809608, 2.141480924070444, 42.10217361371894],
                           [8.369473072835572, 11.463734182240307, 20.917637994863085],
                           [0.0008582001632409618, 69.44727686412561, 0.3071014948342446],
                           [1595.095376997128, 0.06101481612750454, 150.23668667291665],
                           [1698.4240362893236, 0.01299795111744588, 168.37538497509206],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [47.59248524935342, 0.0001029496470784214, 78.28045974217127],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [242.04472754817434, 10.71498387628563, 4.9484867291264365],
                           [2.7227719011735223, 0.05128961069658474, 145.03276012135183],
                           [275.82625286629735, 0.15078427238148212, 0.01734290528220513],
                           [0.42852253460509365, 20.62553301269027, 20.137503433872496],
                           [30.995120357671627, 1.129181305210288, 47.36986262786443],
                           [1.218246254921652, 1.9319531478306338, 96.82443488233574],
                           [11.29966006255687, 0.5830898866905679, 104.71988885779116],
                           [3.20923396464739, 0.9783319000006608, 97.54501629398654],
                           [1.8008755553785996, 15.257336753323756, 26.031153026294295],
                           [0.0008582001632676055, 69.44727686411046, 0.3071014948342446],
                           [19.35606118638945, 4.205685765701538, 34.632844709404786],
                           [0.04022760342216386, 4.175523328298983, 83.98280887651758],
                           [13.82580628388404, 7.015709755230711, 27.897061226189386],
                           [0.025005487349713213, 30.051842860070614, 19.657177301280463],
                           [0.7495692855957126, 27.49892598929964, 8.081602306130074],
                           [207.6312233512174, 15.22131442465494, 6.291022749656092],
                           [2.0669246612820458, 59.205654507627, 0.2567163083327119],
                           [28.217163070588278, 6.867308391765894, 15.732851261064393],
                           [8.369473072835572, 11.463734182240307, 20.917637994863085],
                           [0.7866796241855081, 0.7482338284857643, 126.95937544720896],
                           [60.61694367562461, 0.14830423561726147, 36.8651600451074],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [0.03724473476969815, 33.41040897798797, 10.286782340230328],
                           [52.639819051652154, 0.21522488892796612, 69.05420030384893],
                           [0.7620934383638515, 0.7578921509282203, 126.97386237967345],
                           [10.61893647789901, 0.05439480284717122, 116.89945655820534],
                           [202.35113541078235, 17.063449150127163, 7.426857106060623],
                           [0.8804849999484993, 26.800070109854254, 8.436125334600398],
                           [2.066924661278123, 59.205654507634, 0.25671630833363357],
                           [0.003653390798950482, 11.401091572960702, 49.18423992801202],
                           [10.73670420623688, 41.070709988050204, 0.2529137308832787],
                           [59.96877004716973, 0.1838447977803869, 36.301552188555476],
                           [4.673343261366936, 0.013149278162734916, 124.55799761321882],
                           [29.729950214368042, 2.117307870117984, 39.11858839863627],
                           [6.892175959653678, 22.876073962992507, 7.255347916773981],
                           [8.148748476253063, 3.0098107805379932, 60.41038907433773],
                           [0.34180295245297704, 29.121580878328974, 8.155141795503095],
                           [1.4503832325323285, 47.755697670977675, 1.4175516791962168],
                           [21.513257606901835, 10.778569167839118, 10.748948298837892],
                           [43.41513672260691, 2.1586414863343215, 25.62017935093832],
                           [0.5772211124468531, 74.2102600709729, 0.30155031314624153],
                           [0.0008582001632409618, 69.44727686411804, 0.3071014948342446],
                           [5.872350009128298, 40.23887891083333, 1.5356909696758638],
                           [2.3849227590784627, 21.43148324424069, 11.663695340517886],
                           [26.13058550809608, 2.141480924070444, 42.10217361371894],
                           [8.807168268863801, 10.769013870939286, 22.102883259159043],
                           [0.7835586192627262, 1.676351289965749, 100.34701702002148],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [242.04472754817434, 10.71498387628563, 4.9484867291264365],
                           [2.7227719011735223, 0.05128961069658474, 145.03276012135183],
                           [275.82625286629735, 0.15078427238148212, 0.01734290528220513],
                           [0.42852253460509365, 20.62553301269027, 20.137503433872496],
                           [30.995120357671627, 1.129181305210288, 47.36986262786443],
                           [1.218246254921652, 1.9319531478306338, 96.82443488233574],
                           [11.29966006255687, 0.5830898866905679, 104.71988885779116],
                           [1.8008755553785996, 15.257336753323756, 26.031153026294295],
                           [0.0008582001632676055, 69.44727686411046, 0.3071014948342446],
                           [19.35606118638945, 4.205685765701538, 34.632844709404786],
                           [0.04022760342216386, 4.175523328298983, 83.98280887651758],
                           [13.82580628388404, 7.015709755230711, 27.897061226189386],
                           [0.025005487349713213, 30.051842860070614, 19.657177301280463],
                           [0.7495692855957126, 27.49892598929964, 8.081602306130074],
                           [207.6312233512174, 15.22131442465494, 6.291022749656092],
                           [28.217163070588278, 6.867308391765894, 15.732851261064393],
                           [8.369473072835572, 11.463734182240307, 20.917637994863085],
                           [15.713996064922473, 38.95235076544148, 0.08868236580580716],
                           [0.7620934383638515, 0.7578921509282203, 126.97386237967345],
                           [60.61694367562461, 0.14830423561726147, 36.8651600451074],
                           [0.8804849999484993, 26.800070109854254, 8.436125334600398],
                           [0.7866796241855081, 0.7482338284857643, 126.95937544720896],
                           [0.03724473476969815, 33.41040897798797, 10.286782340230328],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [9.333502737286677, 11.366984873899042, 21.545743083230086],
                           [1.2436388048285154, 1.7831732045733766, 98.52579242628498],
                           [0.1582926303963939, 3.017160801383041, 92.10043550224556],
                           [32.008581315959084, 1.9774144501487505, 38.59133882151926],
                           [3.712105637033285, 42.47599599478546, 1.365197394312663],
                           [5.371573448813243, 25.688075886926526, 5.870562529946706],
                           [1.1118394757099235, 9.41021042966404, 67.95833655255579],
                           [0.11800611495690463, 36.247163960971484, 4.653526117999326],
                           [10.61893647789901, 0.05439480284717122, 116.89945655820534],
                           [0.003653390798950482, 11.401091572960702, 49.18423992801202],
                           [1.243733792755906, 24.053657814677262, 19.221417350208554],
                           [4.673343261366936, 0.013149278162734916, 124.55799761321882],
                           [0.13553067043794778, 0.34185931552958, 142.150112539614],
                           [29.729950214368042, 2.117307870117984, 39.11858839863627],
                           [6.892175959653678, 22.876073962992507, 7.255347916773981],
                           [10.73670420623688, 41.070709988050204, 0.2529137308832787],
                           [20.058620746715224, 0.08741721397189463, 82.20869626815225],
                           [5.872350009126094, 40.23887891083333, 1.5356909696758638],
                           [44.8321847953332, 9.805102035428424, 6.462262624957976],
                           [3.209233964644132, 0.9783319000006608, 97.54501629398654],
                           [0.060926531512385904, 41.83349873512364, 6.4116244024863915],
                           [30.380302892634695, 18.656280189776332, 1.5497289247150978],
                           [13.18420291295021, 82.05348708507053, 0.0005481787718279897],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [275.82625286629735, 0.15078427238148212, 0.01734290528220513],
                           [0.11800611495690463, 36.247163960971484, 4.653526117999326],
                           [0.025005487349713213, 30.051842860070614, 19.657177301280463],
                           [51.24389070750225, 0.820048794305253, 31.12526117057384],
                           [0.22324492317015532, 0.23678826355336915, 144.5145140450492],
                           [237.83816444392767, 0.4888703582471729, 0.18990092909694792],
                           [0.0009872251043546506, 0.9053174962382692, 133.8246713127458],
                           [0.13553067043794778, 0.34185931552958, 142.150112539614],
                           [0.14767998891583067, 14.995851403139168, 38.44202307027863],
                           [20.671796358278446, 3.4112033673749256, 36.87557978841176],
                           [30.104680375194803, 1.2879629538311368, 46.43051079576937],
                           [29.729950214368042, 2.117307870117984, 39.11858839863627],
                           [9.525335310777088, 7.194590484053822, 29.52209860374937],
                           [59.90857807060325, 0.012927332442942313, 54.958724603065015],
                           [5.371573448813243, 25.688075886926526, 5.870562529946706],
                           [2.7227719011735223, 0.05128961069658474, 145.03276012135183],
                           [10.275139100655732, 0.03477984786216228, 115.8783702277622],
                           [0.3751041491539514, 41.086912440706506, 1.0190502661720282],
                           [13.82580628388404, 7.015709755230711, 27.897061226189386],
                           [24.742787086399492, 6.959727046187082, 22.667514822070036],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [44.8321847953332, 9.805102035428424, 6.462262624957976],
                           [10.73670420623688, 41.070709988050204, 0.2529137308832787],
                           [2.9626513504066954, 9.148127316088173, 78.41508184297442],
                           [30.380302892634695, 18.656280189776332, 1.5497289247150978],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [0.7495692855957126, 27.49892598929964, 8.081602306130074],
                           [28.217163070588278, 6.867308391765894, 15.732851261064393],
                           [207.6312233512174, 15.22131442465494, 6.291022749656092],
                           [0.42852253460509365, 20.62553301269027, 20.137503433872496],
                           [60.61694367562461, 0.14830423561726147, 36.8651600451074],
                           [5.872350009126094, 40.23887891083333, 1.5356909696758638],
                           [4.673343261366936, 0.013149278162734916, 124.55799761321882],
                           [0.003653390798950482, 11.401091572960702, 49.18423992801202],
                           [0.8804849999484993, 26.800070109854254, 8.436125334600398],
                           [3.209233964644132, 0.9783319000006608, 97.54501629398654],
                           [1.243733792755906, 24.053657814677262, 19.221417350208554],
                           [1.218246254921652, 1.9319531478306338, 96.82443488233574],
                           [0.04022760342216386, 4.175523328298983, 83.98280887651758],
                           [28.356143406397486, 12.282921681935173, 8.311519889383781],
                           [260.2342297501791, 0.29924533793791563, 0.04549845421024115],
                           [9.333502737286677, 11.366984873899042, 21.545743083230086],
                           [0.7866796241855081, 0.7482338284857643, 126.95937544720896],
                           [9.826772540841548, 0.45663708956401294, 88.4724880197565],
                           [0.03724473476969815, 33.41040897798797, 10.286782340230328],
                           [13.199873961335147, 82.22594604399762, 7.327146826308325e-05],
                           [13.18420291295021, 82.05348708507053, 0.0005481787718279897],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [275.82625286629735, 0.15078427238148212, 0.01734290528220513],
                           [0.11800611495690463, 36.247163960971484, 4.653526117999326],
                           [0.3751041491539514, 41.086912440706506, 1.0190502661720282],
                           [0.025005487349713213, 30.051842860070614, 19.657177301280463],
                           [51.24389070750225, 0.820048794305253, 31.12526117057384],
                           [0.22324492317015532, 0.23678826355336915, 144.5145140450492],
                           [0.0009872251043546506, 0.9053174962382692, 133.8246713127458],
                           [237.83816444392767, 0.4888703582471729, 0.18990092909694792],
                           [0.13553067043794778, 0.34185931552958, 142.150112539614],
                           [330.8090254823592, 0.6636789539470053, 0.017061478848771936],
                           [81.19346609933417, 11.831777605522273, 0.1802098350203372],
                           [30.104680375194803, 1.2879629538311368, 46.43051079576937],
                           [0.14767998891583067, 14.995851403139168, 38.44202307027863],
                           [20.671796358278446, 3.4112033673749256, 36.87557978841176],
                           [29.729950214368042, 2.117307870117984, 39.11858839863627],
                           [59.90857807060325, 0.012927332442942313, 54.958724603065015],
                           [9.525335310777088, 7.194590484053822, 29.52209860374937],
                           [2.7227719011735223, 0.05128961069658474, 145.03276012135183],
                           [5.371573448813243, 25.688075886926526, 5.870562529946706],
                           [10.275139100655732, 0.03477984786216228, 115.8783702277622],
                           [13.82580628388404, 7.015709755230711, 27.897061226189386],
                           [24.742787086399492, 6.959727046187082, 22.667514822070036],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [0.42852253460509365, 20.62553301269027, 20.137503433872496],
                           [7.25570681241572, 1.0927440074874795, 91.84582406284943],
                           [30.380302892634695, 18.656280189776332, 1.5497289247150978],
                           [0.7495692855957126, 27.49892598929964, 8.081602306130074],
                           [2.905002864644057, 20.98676899209004, 11.858504885060452],
                           [0.003653390798950482, 11.401091572960702, 49.18423992801202],
                           [52.92090247564465, 0.09441812991911698, 49.16582362283056],
                           [3.512238642109891, 25.231441976293915, 6.40575264540535],
                           [3.5199561155391628, 22.223571130012502, 7.8146256797829405],
                           [28.217163070588278, 6.867308391765894, 15.732851261064393],
                           [44.8321847953332, 9.805102035428424, 6.462262624957976],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [0.4566986680066298, 7.638013993540891, 58.074715910215936],
                           [45.19704602817468, 8.326032118639139, 9.312646035739876],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [0.4970230584818212, 40.96550099403458, 3.063540081490032],
                           [4.673343261366936, 0.013149278162734916, 124.55799761321882],
                           [0.8804849999484993, 26.800070109854254, 8.436125334600398],
                           [1.218246254921652, 1.9319531478306338, 96.82443488233574],
                           [0.031780006833555065, 20.2403204507899, 21.14189852593494],
                           [27.208211047000393, 7.337470566755466, 15.0551599498586],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [275.82625286629735, 0.15078427238148212, 0.01734290528220513],
                           [0.11800611495690463, 36.247163960971484, 4.653526117999326],
                           [0.3751041491539514, 41.086912440706506, 1.0190502661720282],
                           [51.24389070750225, 0.820048794305253, 31.12526117057384],
                           [0.025005487349713213, 30.051842860070614, 19.657177301280463],
                           [0.22324492317015532, 0.23678826355336915, 144.5145140450492],
                           [0.0009872251043546506, 0.9053174962382692, 133.8246713127458],
                           [237.83816444392767, 0.4888703582471729, 0.18990092909694792],
                           [0.13553067043794778, 0.34185931552958, 142.150112539614],
                           [30.104680375194803, 1.2879629538311368, 46.43051079576937],
                           [0.14767998891583067, 14.995851403139168, 38.44202307027863],
                           [20.671796358278446, 3.4112033673749256, 36.87557978841176],
                           [13.82580628388404, 7.015709755230711, 27.897061226189386],
                           [59.90857807060325, 0.012927332442942313, 54.958724603065015],
                           [81.19346609933417, 11.831777605522273, 0.1802098350203372],
                           [29.729950214368042, 2.117307870117984, 39.11858839863627],
                           [2.7227719011735223, 0.05128961069658474, 145.03276012135183],
                           [9.525335310777088, 7.194590484053822, 29.52209860374937],
                           [5.371573448813243, 25.688075886926526, 5.870562529946706],
                           [10.275139100655732, 0.03477984786216228, 115.8783702277622],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [12.335782559414683, 38.085916954368024, 0.2992188070911673],
                           [0.42852253460509365, 20.62553301269027, 20.137503433872496],
                           [7.25570681241572, 1.0927440074874795, 91.84582406284943],
                           [30.380302892634695, 18.656280189776332, 1.5497289247150978],
                           [0.7495692855957126, 27.49892598929964, 8.081602306130074],
                           [24.742787086399492, 6.959727046187082, 22.667514822070036],
                           [2.905002864644057, 20.98676899209004, 11.858504885060452],
                           [28.217163070588278, 6.867308391765894, 15.732851261064393],
                           [0.003653390798950482, 11.401091572960702, 49.18423992801202],
                           [12.649337613137646, 10.523597115597857, 17.270317346601498],
                           [3.512238642109891, 25.231441976293915, 6.40575264540535],
                           [44.8321847953332, 9.805102035428424, 6.462262624957976],
                           [52.92090247564465, 0.09441812991911698, 49.16582362283056],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [0.4566986680066298, 7.638013993540891, 58.074715910215936],
                           [42.48034668437565, 0.22503075967939343, 49.9677826773582],
                           [45.19704602817468, 8.326032118639139, 9.312646035739876],
                           [0.006036194733345661, 30.836478323627077, 19.045513656336457],
                           [0.0010290730588207357, 12.411000033390046, 42.754791797188105],
                           [0.15567679481850077, 25.883351507056847, 10.721017855669343],
                           [1.6883419794546533, 2.880377754231256, 79.9150224492966],
                           [0.4970230584818212, 40.96550099403458, 3.063540081490032],
                           [330.8090254823426, 0.6636789539462643, 0.017061478848771936],
                           [4.673343261366936, 0.013149278162734916, 124.55799761321882],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [2.658419759563575, 44.02221909387343, 0.2523467042002105],
                           [0.4566986680066298, 7.638013993540891, 58.074715910215936],
                           [7.25570681241572, 1.0927440074874795, 91.84582406284943],
                           [24.742787086399492, 6.959727046187082, 22.667514822070036],
                           [29.729950214368042, 2.117307870117984, 39.11858839863627],
                           [33.810734197060384, 1.2235610052258759, 42.00851339130258],
                           [8.119087814578176, 11.534463846273825, 20.646748307888863],
                           [0.006036194733345661, 30.836478323627077, 19.045513656336457],
                           [0.13553067043794778, 0.34185931552958, 142.150112539614],
                           [0.025005487349713213, 30.051842860070614, 19.657177301280463],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [0.3751041491539514, 41.086912440706506, 1.0190502661720282],
                           [0.14160646568913623, 1.5247433585619465, 128.3999636006685],
                           [4.673343261366936, 0.013149278162734916, 124.55799761321882],
                           [0.0009872251043546506, 0.9053174962382692, 133.8246713127458],
                           [237.83816444392767, 0.4888703582471729, 0.18990092909694792],
                           [275.82625286629735, 0.15078427238148212, 0.01734290528220513],
                           [12.335782559414683, 38.085916954368024, 0.2992188070911673],
                           [0.4970230584818212, 40.96550099403458, 3.063540081490032],
                           [23.300198979840427, 6.684207085136835, 23.09188632214682],
                           [0.003653390798950482, 11.401091572960702, 49.18423992801202],
                           [45.19704602816857, 8.326032118644386, 9.3126460357371],
                           [30.880203310585983, 18.315631179109918, 1.6400283186688043],
                           [44.8321847953332, 9.805102035428424, 6.462262624957976],
                           [1.6883419794546533, 2.880377754231256, 79.9150224492966],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [20.671796358278446, 3.4112033673749256, 36.87557978841176],
                           [52.92090247564465, 0.09441812991911698, 49.16582362283056],
                           [2.905002864644057, 20.98676899209004, 11.858504885060452],
                           [0.20966681953708224, 15.555645642410555, 37.57607753936384],
                           [45.19704602817468, 8.326032118639139, 9.312646035739876],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [8.084753969098298, 42.37921320519444, 0.8238994248397645],
                           [0.5747779174395379, 38.39898613996961, 1.4611111716883698],
                           [3.512238642109891, 25.231441976293915, 6.40575264540535],
                           [40.87176720650732, 0.017634838459856948, 59.866266778455056],
                           [2.7227719011735223, 0.05128961069658474, 145.03276012135183],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [9.525335310777088, 7.194590484053822, 29.52209860374937],
                           [2.0814914900905435, 23.22577670944277, 10.336308497213848],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [0.14767998891583067, 14.995851403139168, 38.44202307027863],
                           [0.4970230584818212, 40.96550099403458, 3.063540081490032],
                           [30.380302892634695, 18.656280189776332, 1.5497289247150978],
                           [25.75364969487488, 10.036338242055585, 10.03974656616823],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [2.658419759563575, 44.02221909387343, 0.2523467042002105],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [0.5813684255696072, 19.786932694371394, 15.41750901086601],
                           [7.25570681241572, 1.0927440074874795, 91.84582406284943],
                           [0.0009872251043546506, 0.9053174962382692, 133.8246713127458],
                           [8.119087814578176, 11.534463846273825, 20.646748307888863],
                           [1.2660556585284062, 32.426344757867966, 3.644132864524858],
                           [28.953846560192822, 2.1080861661265375, 37.62160206861866],
                           [45.19704602816857, 8.326032118644386, 9.3126460357371],
                           [0.3751041491539514, 41.086912440706506, 1.0190502661720282],
                           [237.83816444392767, 0.4888703582471729, 0.18990092909694792],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [4.673343261366936, 0.013149278162734916, 124.55799761321882],
                           [44.8321847953332, 9.805102035428424, 6.462262624957976],
                           [46.223778498112445, 7.913853363412893, 9.695973596545315],
                           [0.14160646568913623, 1.5247433585619465, 128.3999636006685],
                           [10.222841372156568, 16.41763242392627, 12.772314032530836],
                           [2.0814914900905435, 23.22577670944277, 10.336308497213848],
                           [12.335782559414683, 38.085916954368024, 0.2992188070911673],
                           [0.003653390798950482, 11.401091572960702, 49.18423992801202],
                           [22.28112539082112, 4.474747451834841, 31.796482081630725],
                           [0.4566986680066298, 7.638013993540891, 58.074715910215936],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [0.49702305848118006, 40.96550099402293, 3.0635400814916243],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [45.19704602817468, 8.326032118639139, 9.312646035739876],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [13.023867757475223, 12.103013652825378, 16.224609284682284],
                           [20.671796358278446, 3.4112033673749256, 36.87557978841176],
                           [33.996777913280305, 6.172065112575482, 14.72640080270051],
                           [0.025005487349713213, 30.051842860070614, 19.657177301280463],
                           [40.87176720650732, 0.017634838459856948, 59.866266778455056],
                           [24.742787086399492, 6.959727046187082, 22.667514822070036],
                           [9.525335310777088, 7.194590484053822, 29.52209860374937],
                           [330.5279481271995, 0.7127311051294718, 0.007228939382119833],
                           [20.09672606555698, 3.5055312949107793, 36.81276615776066],
                           [30.880203310585983, 18.315631179109918, 1.6400283186688043],
                           [0.13553067043794778, 0.34185931552958, 142.150112539614],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [3.512238642109891, 25.231441976293915, 6.40575264540535],
                           [0.14767998891583067, 14.995851403139168, 38.44202307027863],
                           [0.006036194733345661, 30.836478323627077, 19.045513656336457],
                           [19.081473965263168, 3.650424419365363, 34.54959290880227],
                           [23.300198979840427, 6.684207085136835, 23.09188632214682],
                           [25.75364969487488, 10.036338242055585, 10.03974656616823],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [28.45209832490563, 12.383134588612172, 5.726574931820362],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884820439],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [2.658419759563575, 44.02221909387343, 0.2523467042002105],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [0.0009872251043546506, 0.9053174962382692, 133.8246713127458],
                           [7.25570681241572, 1.0927440074874795, 91.84582406284943],
                           [0.5813684255696072, 19.786932694371394, 15.41750901086601],
                           [4.673343261366936, 0.013149278162734916, 124.55799761321882],
                           [45.19704602816857, 8.326032118644386, 9.3126460357371],
                           [1.2660556585284062, 32.426344757867966, 3.644132864524858],
                           [44.8321847953332, 9.805102035428424, 6.462262624957976],
                           [1.8736444433586872, 23.71461044689457, 9.836098886888687],
                           [20.671796358278446, 3.4112033673749256, 36.87557978841176],
                           [0.3751041491539514, 41.086912440706506, 1.0190502661720282],
                           [22.28112539082112, 4.474747451834841, 31.796482081630725],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [13.023867757475223, 12.103013652825378, 16.224609284682284],
                           [0.003653390798950482, 11.401091572960702, 49.18423992801202],
                           [12.335782559414683, 38.085916954368024, 0.2992188070911673],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [8.119087814578176, 11.534463846273825, 20.646748307888863],
                           [35.50279068090353, 8.490114044559714, 9.818315035198417],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [0.14160646568913623, 1.5247433585619465, 128.3999636006685],
                           [2.0814914900905435, 23.22577670944277, 10.336308497213848],
                           [0.4566986680066298, 7.638013993540891, 58.074715910215936],
                           [31.54312744737227, 21.65484057745565, 0.30776119835847343],
                           [15.553439588526743, 8.29743415586849, 23.99123906557556],
                           [87.1873638348221, 0.2303489736064877, 28.961781032324126],
                           [24.742787086399492, 6.959727046187082, 22.667514822070036],
                           [28.953846560192822, 2.1080861661265375, 37.62160206861866],
                           [237.83816444392767, 0.4888703582471729, 0.18990092909694792],
                           [30.369613257968627, 9.937932426810628, 8.46874410495159],
                           [0.006036194733345661, 30.836478323627077, 19.045513656336457],
                           [17.484900945403833, 10.94807331143494, 16.428991600506635],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [3.7360289114384897, 15.320995591214686, 35.52595789669466],
                           [24.89107254885531, 6.961983001478342, 22.55732814019543],
                           [1.5498733652564336, 33.860262496287085, 2.063559150886595],
                           [46.223778498112445, 7.913853363412893, 9.695973596545315],
                           [14.307900189404565, 13.988908191330449, 13.670034273145134],
                           [0.49702305848118006, 40.96550099402293, 3.0635400814916243],
                           [22.25224017339891, 2.7842154134539854, 36.8545896478215],
                           [10.222841372156568, 16.41763242392627, 12.772314032530836],
                           [9.525335310777088, 7.194590484053822, 29.52209860374937],
                           [1.0341294482743668, 33.16800136276841, 3.566779808974607],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884820439],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [3.7360289114384897, 15.320995591214686, 35.52595789669466],
                           [17.484900945403833, 10.94807331143494, 16.428991600506635],
                           [90.10882731223003, 1.2305535382342105, 15.84317596331721],
                           [50.37649582590206, 0.6327219764613276, 43.27363680485479],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [1.5498733652564336, 33.860262496287085, 2.063559150886595],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [46.223778498112445, 7.913853363412893, 9.695973596545315],
                           [10.222841372156568, 16.41763242392627, 12.772314032530836],
                           [31.54312744737227, 21.65484057745565, 0.30776119835847343],
                           [13.023867757475223, 12.103013652825378, 16.224609284682284],
                           [0.5813684255696072, 19.786932694371394, 15.41750901086601],
                           [4.768191710839176, 9.093224970433258, 30.2282044071339],
                           [44.150053108555646, 7.708049633113804, 10.827482630069365],
                           [35.50279068090353, 8.490114044559714, 9.818315035198417],
                           [87.1873638348221, 0.2303489736064877, 28.961781032324126],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [15.046320124994251, 12.829293030727515, 14.375849180822923],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [22.28112539082112, 4.474747451834841, 31.796482081630725],
                           [0.0009872251043546506, 0.9053174962382692, 133.8246713127458],
                           [1.4164939301115462, 52.4847055083062, 0.6615332237581696],
                           [7.25570681241572, 1.0927440074874795, 91.84582406284943],
                           [1.0341294482743668, 33.16800136276841, 3.566779808974607],
                           [1.8253499131750095, 19.341032762166613, 30.100384127592445],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [0.006036194733345661, 30.836478323627077, 19.045513656336457],
                           [0.30206213932240766, 49.416731496408715, 0.6896406055948532],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [24.742787086399492, 6.959727046187082, 22.667514822070036],
                           [15.636666796147429, 17.979383381071905, 6.803659508424539],
                           [1.2660556585284062, 32.426344757867966, 3.644132864524858],
                           [20.671796358278446, 3.4112033673749256, 36.87557978841176],
                           [33.83023389283168, 9.521048002302278, 8.394504506485251],
                           [24.89107254885531, 6.961983001478342, 22.55732814019543],
                           [237.83816444392767, 0.4888703582471729, 0.18990092909694792],
                           [12.335782559414683, 38.085916954368024, 0.2992188070911673],
                           [44.8321847953332, 9.805102035428424, 6.462262624957976],
                           [50.20037757177307, 0.09666360007902673, 52.46963397947609],
                           [0.4566986680066298, 7.638013993540891, 58.074715910215936],
                           [0.35760077933128237, 9.772422092152075, 49.70693542334217],
                           [64.3954919044104, 0.08503730746319846, 41.109055155373966],
                           [15.553439588526743, 8.29743415586849, 23.99123906557556],
                           [138.69158365105196, 0.09003767333735373, 23.370089248552887],
                           [0.49702305848118006, 40.96550099402293, 3.0635400814916243],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884820439],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [90.10882731223003, 1.2305535382342105, 15.84317596331721],
                           [50.37649582590206, 0.6327219764613276, 43.27363680485479],
                           [17.484900945403833, 10.94807331143494, 16.428991600506635],
                           [1.5498733652564336, 33.860262496287085, 2.063559150886595],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [44.150053108555646, 7.708049633113804, 10.827482630069365],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [46.223778498112445, 7.913853363412893, 9.695973596545315],
                           [13.023867757475223, 12.103013652825378, 16.224609284682284],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [87.1873638348221, 0.2303489736064877, 28.961781032324126],
                           [31.54312744737227, 21.65484057745565, 0.30776119835847343],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [35.50279068090353, 8.490114044559714, 9.818315035198417],
                           [0.0009872251043546506, 0.9053174962382692, 133.8246713127458],
                           [1.4164939301115462, 52.4847055083062, 0.6615332237581696],
                           [0.5813684255696072, 19.786932694371394, 15.41750901086601],
                           [3.631423256843593, 15.118079648267617, 17.28551384951113],
                           [15.046320124994251, 12.829293030727515, 14.375849180822923],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [15.636666796147429, 17.979383381071905, 6.803659508424539],
                           [10.222841372156568, 16.41763242392627, 12.772314032530836],
                           [4.768191710839176, 9.093224970433258, 30.2282044071339],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [33.848791321091916, 9.511951939528668, 8.402872677132219],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [4.564383451318267, 17.090081906455943, 15.403684656504605],
                           [12.335782559414683, 38.085916954368024, 0.2992188070911673],
                           [0.006036194733345661, 30.836478323627077, 19.045513656336457],
                           [0.30206213932240766, 49.416731496408715, 0.6896406055948532],
                           [50.20037757177307, 0.09666360007902673, 52.46963397947609],
                           [1.0341294482743668, 33.16800136276841, 3.566779808974607],
                           [1.3348119977641206, 57.77209194753253, 0.024821663568878268],
                           [7.25570681241572, 1.0927440074874795, 91.84582406284943],
                           [65.18113961112365, 3.6978479595792413, 12.594089078062588],
                           [24.051294882739846, 8.954310302679842, 17.64298857353434],
                           [1.8253499131750095, 19.341032762166613, 30.100384127592445],
                           [24.742787086399492, 6.959727046187082, 22.667514822070036],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [0.7222922968351293, 43.11833015731655, 0.9627714481267619],
                           [3.2672327056938735, 30.3639568871338, 3.6964012136329876],
                           [102.3075652338219, 0.001092712178348667, 34.66160207628678],
                           [0.4566986680066298, 7.638013993540891, 58.074715910215936],
                           [0.35760077933128237, 9.772422092152075, 49.70693542334217],
                           [12.335782559414683, 38.085916954368024, 0.2992188070911673],
                           [15.263877862284307, 3.7149945924623946, 43.17316200958201],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884820439],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [50.37649582590206, 0.6327219764613276, 43.27363680485479],
                           [90.10882731223003, 1.2305535382342105, 15.84317596331721],
                           [1.5498733652564336, 33.860262496287085, 2.063559150886595],
                           [44.150053108555646, 7.708049633113804, 10.827482630069365],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [1.965972712237809, 33.34321702553822, 2.8175589491851123],
                           [35.50279068090353, 8.490114044559714, 9.818315035198417],
                           [13.023867757475223, 12.103013652825378, 16.224609284682284],
                           [17.484900945403833, 10.94807331143494, 16.428991600506635],
                           [50.20037757177307, 0.09666360007902673, 52.46963397947609],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [87.1873638348221, 0.2303489736064877, 28.961781032324126],
                           [1.0341294482743668, 33.16800136276841, 3.566779808974607],
                           [7.25570681241572, 1.0927440074874795, 91.84582406284943],
                           [3.631423256843593, 15.118079648267617, 17.28551384951113],
                           [15.636666796147429, 17.979383381071905, 6.803659508424539],
                           [4.564383451318267, 17.090081906455943, 15.403684656504605],
                           [46.223778498112445, 7.913853363412893, 9.695973596545315],
                           [65.18113961112365, 3.6978479595792413, 12.594089078062588],
                           [0.006036194733345661, 30.836478323627077, 19.045513656336457],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [31.54312744737227, 21.65484057745565, 0.30776119835847343],
                           [0.30206213932240766, 49.416731496408715, 0.6896406055948532],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [0.0009872251043546506, 0.9053174962382692, 133.8246713127458],
                           [102.3075652338219, 0.001092712178348667, 34.66160207628678],
                           [0.35760077933128237, 9.772422092152075, 49.70693542334217],
                           [41.94853186808705, 0.8544070377122558, 43.83971327314418],
                           [13.417919941445112, 0.0009685852550144952, 114.96989412637986],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [14.619542807631488, 13.194698430206008, 14.007318722294348],
                           [1.3348119977641206, 57.77209194753253, 0.024821663568878268],
                           [24.742787086399492, 6.959727046187082, 22.667514822070036],
                           [0.7222922968351293, 43.11833015731655, 0.9627714481267619],
                           [0.4566986680066298, 7.638013993540891, 58.074715910215936],
                           [24.051294882739846, 8.954310302679842, 17.64298857353434],
                           [10.22403310520745, 12.933154165515383, 19.16170164728055],
                           [15.263877862284307, 3.7149945924623946, 43.17316200958201],
                           [0.5813684255696072, 19.786932694371394, 15.41750901086601],
                           [12.335782559414683, 38.085916954368024, 0.2992188070911673],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [33.848791321091916, 9.511951939528668, 8.402872677132219],
                           [23.454722191793852, 10.5287409315765, 14.514691341783633],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884820439],
                           [102.3075652338219, 0.001092712178348667, 34.66160207628678],
                           [24.994679989142753, 3.815355660370556, 33.76774153757404],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [10.22403310520745, 12.933154165515383, 19.16170164728055],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [20.443924078461226, 10.039445000003397, 17.341220128559968],
                           [41.94853186808705, 0.8544070377122558, 43.83971327314418],
                           [1.3348119977641206, 57.77209194753253, 0.024821663568878268],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [50.37649582590206, 0.6327219764613276, 43.27363680485479],
                           [12.335782559414683, 38.085916954368024, 0.2992188070911673],
                           [14.055225249182767, 6.714650601003579, 28.79874846202873],
                           [18.571246379102494, 4.9611484101148315, 30.80015587195707],
                           [15.826025009791802, 0.05591073962038981, 101.15367643351473],
                           [0.5605752789986421, 32.38025898940134, 4.576144536494646],
                           [65.18113961110897, 3.697847959577492, 12.59408907805936],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [6.549009069819267, 23.46740339228924, 6.8968409318139],
                           [26.38728595012448, 20.926259079410624, 0.714832110735247],
                           [1.3699780762232898, 28.197229782390245, 6.276658014766033],
                           [15.636666796147429, 17.979383381071905, 6.803659508424539],
                           [13.023867757475223, 12.103013652825378, 16.224609284682284],
                           [87.011494218145, 1.512423343119, 14.883592101749498],
                           [17.246707215651895, 6.0131615140884955, 32.87807084065664],
                           [3.4620982366229334, 15.527094308681374, 16.806683431158156],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [0.11655062342594713, 46.12492496746356, 0.47257330617820387],
                           [14.619542807631488, 13.194698430206008, 14.007318722294348],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [10.6360533046946, 16.218993080463402, 12.371814384785296],
                           [4.564383451318267, 17.090081906455943, 15.403684656504605],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [50.20037757177307, 0.09666360007902673, 52.46963397947609],
                           [7.372815727903613, 31.152300278218597, 1.4585570858564976],
                           [31.54312744737227, 21.65484057745565, 0.30776119835847343],
                           [1.5498733652564336, 33.860262496287085, 2.063559150886595],
                           [1.686033855151136, 33.33031240900092, 2.195121808503979],
                           [1.0341294482743668, 33.16800136276841, 3.566779808974607],
                           [46.223778498112445, 7.913853363412893, 9.695973596545315],
                           [35.50279068090353, 8.490114044559714, 9.818315035198417],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [94.76924635186782, 0.8488634757550659, 17.41210711229579],
                           [7.757011039288594, 12.511731771696875, 20.284732709277748],
                           [15.036177216417082, 7.765590976669071, 23.958448931591604],
                           [13.417919941445112, 0.0009685852550144952, 114.96989412637986],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [90.10882731223003, 1.2305535382342105, 15.84317596331721],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884820439],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [24.994679989142753, 3.815355660370556, 33.76774153757404],
                           [18.571246379102494, 4.9611484101148315, 30.80015587195707],
                           [102.3075652338219, 0.001092712178348667, 34.66160207628678],
                           [46.223778498100074, 7.913853363410335, 9.695973596548146],
                           [20.5529311980434, 9.843676677853658, 16.896908279858],
                           [16.854146298511154, 5.941330043835413, 30.967216317813527],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [3.4620982366229334, 15.527094308681374, 16.806683431158156],
                           [14.055225249182767, 6.714650601003579, 28.79874846202873],
                           [19.769809707681357, 0.03113083700716666, 109.23040687972221],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [23.132636550072803, 6.328881824129333, 23.879924843715422],
                           [0.060168978294185674, 46.11249605231558, 0.578398391292279],
                           [50.37649582590206, 0.6327219764613276, 43.27363680485479],
                           [21.87492204482864, 24.31007383109233, 0.2354745907651458],
                           [26.38728595012448, 20.926259079410624, 0.714832110735247],
                           [0.5605752789986421, 32.38025898940134, 4.576144536494646],
                           [14.55645173625101, 30.77649186243637, 0.2953559267772053],
                           [46.223778498112445, 7.913853363412893, 9.695973596545315],
                           [6.549009069819267, 23.46740339228924, 6.8968409318139],
                           [41.94853186808705, 0.8544070377122558, 43.83971327314418],
                           [15.636666796147429, 17.979383381071905, 6.803659508424539],
                           [87.011494218145, 1.512423343119, 14.883592101749498],
                           [42.005986794634644, 1.8272066753124638, 36.443119552535094],
                           [90.10882731223003, 1.2305535382342105, 15.84317596331721],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [10.22403310520745, 12.933154165515383, 19.16170164728055],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [0.04716500943905092, 56.21963976820116, 0.0015354641300302113],
                           [10.1155671236392, 10.652621878237817, 21.137829677240344],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [0.0573651635057149, 47.90914480356339, 0.5504059283195805],
                           [20.443924078461226, 10.039445000003397, 17.341220128559968],
                           [0.11655062342594713, 46.12492496746356, 0.47257330617820387],
                           [13.91964130619412, 0.0014191441052082555, 114.26098897988942],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [5.685888655518544, 14.787874834546074, 17.174074602065392],
                           [7.372815727903613, 31.152300278218597, 1.4585570858564976],
                           [65.18113961110897, 3.697847959577492, 12.59408907805936],
                           [1.3699780762232898, 28.197229782390245, 6.276658014766033],
                           [1.5498733652564336, 33.860262496287085, 2.063559150886595],
                           [35.33103708240217, 3.5542748562579423, 26.965352759342053],
                           [15.036177216417082, 7.765590976669071, 23.958448931591604],
                           [0.014396050845362954, 41.887048555164746, 1.9941393415629105],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [13.417919941445112, 0.0009685852550144952, 114.96989412637986],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884820439],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [3.4620982366229334, 15.527094308681374, 16.806683431158156],
                           [24.994679989142753, 3.815355660370556, 33.76774153757404],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [45.79437005015109, 3.370051627818931e-05, 62.74342884819718],
                           [15.536021940198495, 18.07634227757434, 6.743621346707772],
                           [23.132636550072803, 6.328881824129333, 23.879924843715422],
                           [20.5529311980434, 9.843676677853658, 16.896908279858],
                           [7.372815727903613, 31.152300278218597, 1.4585570858564976],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [13.440549090163813, 0.0004266679456497624, 113.07885961629518],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [26.38728595012448, 20.926259079410624, 0.714832110735247],
                           [0.5605752789986421, 32.38025898940134, 4.576144536494646],
                           [71.02303757312367, 2.598190559617718, 14.66761096911895],
                           [46.223778498100074, 7.913853363410335, 9.695973596548146],
                           [4.198649546781141, 28.0361413997774, 3.9745226693823965],
                           [10.22403310520745, 12.933154165515383, 19.16170164728055],
                           [0.020304267136820035, 47.375235121051226, 0.42738606442809546],
                           [19.769809707681357, 0.03113083700716666, 109.23040687972221],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [73.8710262116265, 3.088928964135834, 11.912423740365377],
                           [42.005986794634644, 1.8272066753124638, 36.443119552535094],
                           [1.391312623938801, 6.453052092865291, 61.33703508990467],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [5.017159377753604, 23.893560113393278, 6.676756492939675],
                           [102.3075652338219, 0.001092712178348667, 34.66160207628678],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [50.37649582590206, 0.6327219764613276, 43.27363680485479],
                           [35.33103708240217, 3.5542748562579423, 26.965352759342053],
                           [0.042436372246521595, 56.000516047924734, 0.0031808619213400707],
                           [14.158376710939255, 13.34185876603863, 12.00434977560262],
                           [14.055225249182767, 6.714650601003579, 28.79874846202873],
                           [0.04716500943905092, 56.21963976820116, 0.0015354641300302113],
                           [15.036177216417082, 7.765590976669071, 23.958448931591604],
                           [8.124219953614286, 16.3231820509106, 13.102422175174226],
                           [1.3699780762232898, 28.197229782390245, 6.276658014766033],
                           [23.68738588244616, 3.7454730603692696, 35.28035675170802],
                           [16.91088136467953, 12.528474742830426, 14.491029510637139],
                           [41.94853186808705, 0.8544070377122558, 43.83971327314418],
                           [10.298791707550937, 13.958090775507177, 14.11119066198783],
                           [47.563622279791616, 12.977396708382019, 2.200275157409278],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [5.685888655516375, 14.787874834546074, 17.174074602065392],
                           [21.87492204482864, 24.31007383109233, 0.2354745907651458],
                           [10.1155671236392, 10.652621878237817, 21.137829677240344],
                           [23.145976918764514, 9.532735990291828, 15.050934917136093],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [23.68738588244616, 3.7454730603692696, 35.28035675170802],
                           [77.35779943953624, 0.0029279447649296422, 42.55956926220304],
                           [15.536021940198495, 18.07634227757434, 6.743621346707772],
                           [35.33103708240217, 3.5542748562579423, 26.965352759342053],
                           [0.020304267136820035, 47.375235121051226, 0.42738606442809546],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [3.5710333945842927, 24.24110284275886, 7.239351846405419],
                           [4.198649546781141, 28.0361413997774, 3.9745226693823965],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [15.828910644873933, 11.056539973353988, 15.210326879373437],
                           [42.001022713271006, 0.5550003220933705, 44.62298308982553],
                           [23.767023520231813, 7.618964675991613, 20.03154579718517],
                           [14.055225249182767, 6.714650601003579, 28.79874846202873],
                           [0.5605752789986421, 32.38025898940134, 4.576144536494646],
                           [5.685888655516375, 14.787874834546074, 17.174074602065392],
                           [7.372815727903613, 31.152300278218597, 1.4585570858564976],
                           [3.4620982366229334, 15.527094308681374, 16.806683431158156],
                           [14.51709249202439, 6.983205119480762, 28.393455696095884],
                           [1.8413621505780446, 30.567483868545093, 4.600220013723792],
                           [71.02303757312367, 2.598190559617718, 14.66761096911895],
                           [22.592520991011675, 23.802916001995857, 0.29039655111389684],
                           [1.0199500926971496, 36.54208253338295, 1.9907365613689634],
                           [10.22403310520745, 12.933154165518655, 19.16170164727657],
                           [5.017159377753604, 23.893560113393278, 6.676756492939675],
                           [19.769809707681357, 0.03113083700716666, 109.23040687972221],
                           [10.22403310520745, 12.933154165515383, 19.16170164728055],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [13.440549090163813, 0.0004266679456497624, 113.07885961629518],
                           [16.91088136467953, 12.528474742830426, 14.491029510637139],
                           [17.77491605587731, 15.302594775444643, 8.957227304852244],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [47.563622279791616, 12.977396708382019, 2.200275157409278],
                           [32.907633653489505, 0.5152045821579239, 51.80329417704964],
                           [19.732176969287597, 6.852298047753575, 24.003319539372534],
                           [50.37649582590206, 0.6327219764613276, 43.27363680485479],
                           [40.300497447626604, 1.5802947421865192, 33.74257773766528],
                           [46.223778498100074, 7.913853363410335, 9.695973596548146],
                           [102.3075652338219, 0.001092712178348667, 34.66160207628678],
                           [32.87768630373118, 0.5312197537280611, 51.555541534077854],
                           [15.036177216417082, 7.765590976669071, 23.958448931591604],
                           [7.744734130670457, 25.87681934066833, 2.845728782049847],
                           [1.3699780762232898, 28.197229782390245, 6.276658014766033],
                           [23.145976918764514, 9.532735990291828, 15.050934917136093],
                           [8.124219953614286, 16.3231820509106, 13.102422175174226],
                           [10.1155671236392, 10.652621878237817, 21.137829677240344],
                           [26.235252334726773, 1.739717185890004, 43.65880847117003],
                           [73.8710262116265, 3.088928964135834, 11.912423740365377],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [0.00010479847342511498, 323.75294183481395, 185.38830897650539],
                           [18.92944748113145, 4.645362952929202, 34.867632869214574],
                           [13.49495298910005, 15.580086994884157, 11.351220626404432],
                           [1.8953681908642714, 52.79504592307766, 0.2803457071305837],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [3.936603120281482, 27.89620922958773, 3.9946297768245733],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [72.6474008909111, 0.061634583228155355, 47.97738114038364],
                           [23.68738588244616, 3.7454730603692696, 35.28035675170802],
                           [15.536021940198495, 18.07634227757434, 6.743621346707772],
                           [11.11090686095937, 16.918787980767476, 9.626808619911325],
                           [16.665018449667855, 7.674327606222968, 26.80672375573778],
                           [0.020304267136820035, 47.375235121051226, 0.42738606442809546],
                           [15.036177216417082, 7.765590976669071, 23.958448931591604],
                           [4.198649546781141, 28.0361413997774, 3.9745226693823965],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [5.685888655516375, 14.787874834546074, 17.174074602065392],
                           [7.372815727903613, 31.152300278218597, 1.4585570858564976],
                           [23.767023520231813, 7.618964675991613, 20.03154579718517],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [19.732176969287597, 6.852298047753575, 24.003319539372534],
                           [42.001022713271006, 0.5550003220933705, 44.62298308982553],
                           [25.544935821325875, 15.617103445530853, 4.895071756249814],
                           [80.78421081888365, 1.9341542023799023, 14.437199599340184],
                           [35.33103708240217, 3.5542748562579423, 26.965352759342053],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [39.32968357728305, 1.1320461617891886, 42.12527101232664],
                           [23.145976918764514, 9.532735990291828, 15.050934917136093],
                           [1.3662437820712143, 25.697892655583175, 8.591476768532909],
                           [19.769809707681357, 0.03113083700716666, 109.23040687972221],
                           [1.3699780762232898, 28.197229782390245, 6.276658014766033],
                           [19.44401779363834, 4.09562895133428, 35.16801716953991],
                           [13.440549090163813, 0.0004266679456497624, 113.07885961629518],
                           [28.789567792127414, 5.27495080261387, 24.22505542719145],
                           [0.5605752789986421, 32.38025898940134, 4.576144536494646],
                           [73.8710262116265, 3.088928964135834, 11.912423740365377],
                           [22.592520991011675, 23.802916001995857, 0.29039655111389684],
                           [10.1155671236392, 10.652621878237817, 21.137829677240344],
                           [1.0199500926971496, 36.54208253338295, 1.9907365613689634],
                           [77.35779943953624, 0.0029279447649296422, 42.55956926220304],
                           [16.91088136467953, 12.528474742830426, 14.491029510637139],
                           [44.120822180735786, 0.3361708620912494, 44.70735475835474],
                           [14.055225249182767, 6.714650601003579, 28.79874846202873],
                           [23.595432717218294, 7.193328383612917, 20.680816708817105],
                           [102.3075652338219, 0.001092712178348667, 34.66160207628678],
                           [39.266149088789355, 0.8794991637282905, 42.259359052167646],
                           [9.481263882352115, 12.785117574041903, 16.682387085512286],
                           [47.563622279791616, 12.977396708382019, 2.200275157409278],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [4.969530117022278e-05, 322.9326014486039, 184.7193699826495],
                           [18.92944748113145, 4.645362952929202, 34.867632869214574],
                           [18.483628227056787, 9.868914741737836, 16.418438261782583],
                           [15.036177216417082, 7.765590976669071, 23.958448931591604],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [0.020304267136820035, 47.375235121051226, 0.42738606442809546],
                           [0.0010142018904553424, 42.185978271379234, 1.532712130518071],
                           [80.78421081888365, 1.9341542023799023, 14.437199599340184],
                           [6.634657715814117, 1.0389400567378968, 93.0636082809769],
                           [5.685888655516375, 14.787874834546074, 17.174074602065392],
                           [42.001022713271006, 0.5550003220933705, 44.62298308982553],
                           [11.11090686095937, 16.918787980767476, 9.626808619911325],
                           [35.33103708241298, 3.5542748562562276, 26.96535275934678],
                           [15.536021940198495, 18.07634227757434, 6.743621346707772],
                           [28.789567792127414, 5.27495080261387, 24.22505542719145],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [19.732176969287597, 6.852298047753575, 24.003319539372534],
                           [28.67107159956282, 1.187219677842409, 46.537244556296045],
                           [1.0132430493799036, 26.05795494163298, 9.789906229561902],
                           [2.8922278925545, 32.98108546023323, 1.6529946783499792],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [77.35779943953624, 0.0029279447649296422, 42.55956926220304],
                           [1.3699780762232898, 28.197229782390245, 6.276658014766033],
                           [72.3487113522709, 0.0538250290084471, 47.75835010776051],
                           [1.8953681908642714, 52.79504592307766, 0.2803457071305837],
                           [3.936603120281482, 27.89620922958773, 3.9946297768245733],
                           [10.1155671236392, 10.652621878237817, 21.137829677240344],
                           [25.919894247102977, 6.930918481300885, 19.828384131942027],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [23.68738588244616, 3.7454730603692696, 35.28035675170802],
                           [0.8334202479833682, 31.389917080775984, 4.375816495282417],
                           [4.081189845319247, 19.88902953496945, 11.666378596656344],
                           [23.145976918764514, 9.532735990291828, 15.050934917136093],
                           [1.3662437820712143, 25.697892655583175, 8.591476768532909],
                           [2.390392541621507, 19.061312828200602, 15.054508336833504],
                           [37.022138239374435, 0.800095305221367, 48.43995793225644],
                           [19.44401779363834, 4.09562895133428, 35.16801716953991],
                           [73.8710262116265, 3.088928964135834, 11.912423740365377],
                           [27.0374148507065, 8.1072336259179, 15.119450758906076],
                           [2.640775774690694, 32.38660332995622, 2.2162888233464964],
                           [4.888974467310375, 0.4111500648304522, 111.4751462582192],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [16.91088136467953, 12.528474742830426, 14.491029510637139],
                           [46.618022740839756, 7.8853476736645405, 7.972575823479614],
                           [4.198649546781141, 28.0361413997774, 3.9745226693823965],
                           [1.9329783063503818, 22.82139152469313, 10.472348308424968],
                           [100.76273872779629, 2.31347472334977, 8.280536011814613],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [4.969530117022278e-05, 322.9326014486039, 184.7193699826495],
                           [28.67107159956282, 1.187219677842409, 46.537244556296045],
                           [11.11090686095937, 16.918787980767476, 9.626808619911325],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [38.05208411706507, 2.459447264177809, 32.772530601798174],
                           [16.91088136467953, 12.528474742830426, 14.491029510637139],
                           [1.9329783063503818, 22.82139152469313, 10.472348308424968],
                           [72.3487113522709, 0.0538250290084471, 47.75835010776051],
                           [0.020304267136820035, 47.375235121051226, 0.42738606442809546],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [28.34472399046336, 0.6945556029884049, 52.03905208307363],
                           [18.33823402708983, 5.573314066915118, 29.703458284629868],
                           [28.789567792127414, 5.27495080261387, 24.22505542719145],
                           [0.0010142018904553424, 42.185978271379234, 1.532712130518071],
                           [4.198649546781141, 28.0361413997774, 3.9745226693823965],
                           [34.213278881310416, 0.43924170169983184, 52.89964510092818],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [25.919894247102977, 6.930918481300885, 19.828384131942027],
                           [15.536021940198495, 18.07634227757434, 6.743621346707772],
                           [35.33103708241298, 3.5542748562562276, 26.96535275934678],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [1.0132430493799036, 26.05795494163298, 9.789906229561902],
                           [13.885144415109245, 19.92448230959155, 5.7010403989985035],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [19.65720355413345, 4.140702508813989, 34.7762120959616],
                           [10.191000154413182, 0.6165078649720065, 92.20952074712163],
                           [19.732176969287597, 6.852298047753575, 24.003319539372534],
                           [1.3699780762232898, 28.197229782390245, 6.276658014766033],
                           [8.411996741305195, 28.126426831059653, 1.4543178589760866],
                           [15.036177216417082, 7.765590976669071, 23.958448931591604],
                           [29.79556766182098, 1.4266646444826396, 45.94479732687259],
                           [0.8334202479833682, 31.389917080775984, 4.375816495282417],
                           [80.78421081888365, 1.9341542023799023, 14.437199599340184],
                           [6.634657715814117, 1.0389400567378968, 93.0636082809769],
                           [29.186298375462552, 22.48214273426865, 0.38328901032536666],
                           [21.659520773248992, 10.301486975014665, 14.426804301764028],
                           [10.1155671236392, 10.652621878237817, 21.137829677240344],
                           [4.081189845319247, 19.88902953496945, 11.666378596656344],
                           [0.07313107333303914, 12.325354605364216, 53.307779943114674],
                           [1.7849005853610493, 37.51233349393413, 1.0785567709264525],
                           [2.640775774690694, 32.38660332995622, 2.2162888233464964],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [23.68738588244616, 3.7454730603692696, 35.28035675170802],
                           [46.618022740839756, 7.8853476736645405, 7.972575823479614],
                           [1.8953681908642714, 52.79504592307766, 0.2803457071305837],
                           [29.942828587379125, 2.7009124456953866, 35.24153947867063],
                           [23.145976918764514, 9.532735990291828, 15.050934917136093],
                           [3.936603120281482, 27.89620922958773, 3.9946297768245733],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [4.969530117022278e-05, 322.9326014486039, 184.7193699826495],
                           [10.979224447074506, 0.9108981776282215, 72.13107981750976],
                           [41.209318398206655, 0.417266052346584, 83.99385451097919],
                           [15.945579207879776, 0.2895610607302289, 89.20952020973053],
                           [35.33103708241298, 3.5542748562562276, 26.96535275934678],
                           [80.78421081888365, 1.9341542023799023, 14.437199599340184],
                           [0.8334202479833682, 31.389917080775984, 4.375816495282417],
                           [3.936603120281482, 27.89620922958773, 3.9946297768245733],
                           [15.27716721971955, 5.7617658102087095, 27.876911851892167],
                           [21.659520773248992, 10.301486975014665, 14.426804301764028],
                           [34.213278881310416, 0.43924170169983184, 52.89964510092818],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [11.865923042862734, 10.143337916933055, 20.356886840722375],
                           [29.942828587379125, 2.7009124456953866, 35.24153947867063],
                           [1.9329783063503818, 22.82139152469313, 10.472348308424968],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [8.411996741305195, 28.126426831059653, 1.4543178589760866],
                           [33.55836959810893, 7.361975792892662, 15.285424277666465],
                           [28.67107159956282, 1.187219677842409, 46.537244556296045],
                           [0.0010142018904553424, 42.185978271379234, 1.532712130518071],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [2.787815831545738, 31.821947897701147, 3.5027969003150905],
                           [11.11090686095937, 16.918787980767476, 9.626808619911325],
                           [1.0132430493799036, 26.05795494163298, 9.789906229561902],
                           [28.34472399046336, 0.694555602987647, 52.03905208306707],
                           [0.21169437567373317, 47.6335467669177, 0.3124560210420021],
                           [3.75574768293957, 37.69112252010349, 0.1411948721507353],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [4.198649546781141, 28.0361413997774, 3.9745226693823965],
                           [10.191000154413182, 0.6165078649720065, 92.20952074712163],
                           [51.198212504373984, 0.04020046603248372, 50.57594136995785],
                           [0.07313107333303914, 12.325354605364216, 53.307779943114674],
                           [46.618022740839756, 7.8853476736645405, 7.972575823479614],
                           [23.68738588244616, 3.7454730603692696, 35.28035675170802],
                           [28.34472399045852, 0.6945556029884049, 52.03905208306707],
                           [21.06522946296845, 19.788172284068878, 2.7423149680640106],
                           [22.269933815413957, 8.363505140257223, 18.671356370674683],
                           [4.081189845319247, 19.88902953496945, 11.666378596656344],
                           [28.789567792127414, 5.27495080261387, 24.22505542719145],
                           [12.479517400970124, 7.9320252781908005, 26.242870906747406],
                           [15.036177216417082, 7.765590976669071, 23.958448931591604],
                           [29.186298375462552, 22.48214273426865, 0.38328901032536666],
                           [1.7849005853610493, 37.51233349393413, 1.0785567709264525],
                           [38.05208411706507, 2.459447264177809, 32.772530601798174],
                           [19.732176969287597, 6.852298047753575, 24.003319539372534],
                           [1.3699780762232898, 28.197229782390245, 6.276658014766033],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [4.969530117022278e-05, 322.9326014486039, 184.7193699826495],
                           [10.979224447074506, 0.9108981776282215, 72.13107981750976],
                           [41.209318398206655, 0.417266052346584, 83.99385451097919],
                           [15.945579207879776, 0.2895610607302289, 89.20952020973053],
                           [35.33103708241298, 3.5542748562562276, 26.96535275934678],
                           [80.78421081888365, 1.9341542023799023, 14.437199599340184],
                           [0.8334202479833682, 31.389917080775984, 4.375816495282417],
                           [3.936603120281482, 27.89620922958773, 3.9946297768245733],
                           [15.27716721971955, 5.7617658102087095, 27.876911851892167],
                           [21.659520773248992, 10.301486975014665, 14.426804301764028],
                           [34.213278881310416, 0.43924170169983184, 52.89964510092818],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [11.865923042862734, 10.143337916933055, 20.356886840722375],
                           [29.942828587379125, 2.7009124456953866, 35.24153947867063],
                           [1.9329783063503818, 22.82139152469313, 10.472348308424968],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [8.411996741305195, 28.126426831059653, 1.4543178589760866],
                           [0.0010142018904553424, 42.185978271379234, 1.532712130518071],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [2.787815831545738, 31.821947897701147, 3.5027969003150905],
                           [11.11090686095937, 16.918787980767476, 9.626808619911325],
                           [1.0132430493799036, 26.05795494163298, 9.789906229561902],
                           [28.34472399046336, 0.694555602987647, 52.03905208306707],
                           [0.21169437567373317, 47.6335467669177, 0.3124560210420021],
                           [3.75574768293957, 37.69112252010349, 0.1411948721507353],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [7.957127803994528, 15.644270968811636, 14.86963388127132],
                           [0.21169437567373317, 47.63354676692398, 0.3124560210409853],
                           [4.198649546781141, 28.0361413997774, 3.9745226693823965],
                           [66.69013448259132, 0.3875774192081382, 34.65328885617088],
                           [10.191000154413182, 0.6165078649720065, 92.20952074712163],
                           [57.03370678854614, 10.20700063660658, 2.6023250583807918],
                           [24.299470426437416, 7.2168856212595305, 21.433277025804152],
                           [51.198212504373984, 0.04020046603248372, 50.57594136995785],
                           [30.91521955662019, 1.8773511278085573, 38.82605794456882],
                           [17.308178440430193, 3.5424690672274823, 39.156184335310066],
                           [20.268034553908244, 22.430976427412403, 1.4009796983155207],
                           [15.94557920788704, 0.28956106072973953, 89.20952020974771],
                           [28.67107159955795, 1.187219677841418, 46.537244556296045],
                           [0.07313107333303914, 12.325354605364216, 53.307779943114674],
                           [33.55836959810893, 7.3619757928901945, 15.285424277666465],
                           [25.655288206010113, 7.97106447690314, 17.431522387941303],
                           [10.358137907016964, 12.299447688056697, 17.149259621403566],
                           [11.122655946275792, 19.14273259200534, 7.462418438260423],
                           [0.2453921806542392, 21.11902754953728, 17.0922567373013],
                           [14.760734577531963, 20.1425600532704, 4.528079111749302],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [4.969530117022278e-05, 322.9326014486039, 184.7193699826495],
                           [10.979224447074506, 0.9108981776282215, 72.13107981750976],
                           [15.945579207879776, 0.2895610607302289, 89.20952020973053],
                           [35.33103708241298, 3.5542748562562276, 26.96535275934678],
                           [80.78421081888365, 1.9341542023799023, 14.437199599340184],
                           [3.936603120281482, 27.89620922958773, 3.9946297768245733],
                           [0.8334202479833682, 31.389917080775984, 4.375816495282417],
                           [15.27716721971955, 5.7617658102087095, 27.876911851892167],
                           [21.659520773248992, 10.301486975014665, 14.426804301764028],
                           [34.213278881310416, 0.43924170169983184, 52.89964510092818],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [11.865923042862734, 10.143337916933055, 20.356886840722375],
                           [29.942828587379125, 2.7009124456953866, 35.24153947867063],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [1.9329783063503818, 22.82139152469313, 10.472348308424968],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [8.411996741305195, 28.126426831059653, 1.4543178589760866],
                           [0.0010142018904553424, 42.185978271379234, 1.532712130518071],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [1.0132430493799036, 26.05795494163298, 9.789906229561902],
                           [2.787815831545738, 31.821947897701147, 3.5027969003150905],
                           [11.11090686095937, 16.918787980767476, 9.626808619911325],
                           [28.34472399046336, 0.694555602987647, 52.03905208306707],
                           [0.21169437567373317, 47.6335467669177, 0.3124560210420021],
                           [3.75574768293957, 37.69112252010349, 0.1411948721507353],
                           [17.308178440430193, 3.5424690672274823, 39.156184335310066],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [17.928975332677098, 0.09229963341718674, 93.62800100058587],
                           [6.177595486458452, 2.4698963180521005, 62.00102574415646],
                           [66.69013448259132, 0.3875774192081382, 34.65328885617088],
                           [6.840799323080443, 12.077423718510678, 22.21465034387381],
                           [28.67107159955795, 1.187219677841418, 46.537244556296045],
                           [8.493352780138059, 36.281092657326205, 0.014902858888112356],
                           [2.8857058801703928, 30.479305759404593, 4.237328303227819],
                           [27.59652424154268, 30.38556510921814, 0.9497557369881103],
                           [15.138398500381655, 15.138034359836492, 10.335059429540564],
                           [40.961146842337435, 0.9676644844849699, 43.20438702984144],
                           [0.1657368014261809, 27.90804897379983, 10.500329203065355],
                           [0.21169437567373317, 47.63354676692398, 0.3124560210409853],
                           [0.07313107333303914, 12.325354605364216, 53.307779943114674],
                           [0.7045322373238427, 51.654916839569786, 0.06915557245166713],
                           [51.198212504373984, 0.04020046603248372, 50.57594136995785],
                           [6.5232547360587265, 22.47163172083615, 6.588615825003919],
                           [57.03370678854614, 10.20700063660658, 2.6023250583807918],
                           [1.4427922108135132, 19.125392608138892, 37.29977879946837],
                           [33.55836959810893, 7.3619757928901945, 15.285424277666465],
                           [125.72912767130768, 0.44887454871085264, 33.140471070500226],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [4.969530117022278e-05, 322.9326014486039, 184.7193699826495],
                           [10.848784548129368, 14.585615790507841, 13.848277099136396],
                           [33.644052006785955, 0.15197937522811636, 60.48864526396182],
                           [48.016818992929664, 0.019997889497120912, 61.43751570511156],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [67.26778562427383, 0.35151766355133673, 35.002875881881764],
                           [80.78421081888365, 1.9341542023799023, 14.437199599340184],
                           [35.33103708241298, 3.5542748562562276, 26.96535275934678],
                           [34.213278881310416, 0.43924170169983184, 52.89964510092818],
                           [17.648576925977572, 6.491284630305452, 27.16183217044371],
                           [8.411996741305195, 28.126426831059653, 1.4543178589760866],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [33.55836959810893, 7.3619757928901945, 15.285424277666465],
                           [0.8334202479833682, 31.389917080775984, 4.375816495282417],
                           [21.659520773248992, 10.301486975014665, 14.426804301764028],
                           [2.787815831542701, 31.821947897706277, 3.5027969003150905],
                           [0.1657368014261809, 27.90804897379983, 10.500329203065355],
                           [6.5232547360587265, 22.47163172083615, 6.588615825003919],
                           [0.16573680142655117, 27.908048973795026, 10.500329203068302],
                           [10.979224447074506, 0.9108981776282215, 72.13107981750976],
                           [125.72912767130768, 0.44887454871085264, 33.140471070500226],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [9.730741172355417, 32.79513262195397, 0.326174989962993],
                           [11.865923042862734, 10.143337916933055, 20.356886840722375],
                           [1.775091666403272, 19.8840680700664, 16.35503138255625],
                           [15.27716721971955, 5.7617658102087095, 27.876911851892167],
                           [17.928975332677098, 0.09229963341718674, 93.62800100058587],
                           [6.177595486458452, 2.4698963180521005, 62.00102574415646],
                           [40.961146842337435, 0.9676644844849699, 43.20438702984144],
                           [11.865923042862734, 10.143337916933055, 20.356886840722375],
                           [17.308178440430193, 3.5424690672274823, 39.156184335310066],
                           [1.9329783063503818, 22.82139152469313, 10.472348308424968],
                           [15.945579207879776, 0.2895610607302289, 89.20952020973053],
                           [3.512313800206984, 38.31362547402685, 0.10560220707265709],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [1.3263325547101132, 28.37651509260147, 6.192161418935841],
                           [7.502454589243345, 18.068824420582754, 13.850754900738357],
                           [78.1808891803457, 7.062394128647856, 2.887561012791642],
                           [3.936603120281482, 27.89620922958773, 3.9946297768245733],
                           [0.21169437567373317, 47.63354676692398, 0.3124560210409853],
                           [1.0132430493799036, 26.05795494163298, 9.789906229561902],
                           [0.25702124145260663, 49.720978376030494, 0.0051449379281967595],
                           [0.0010142018904553424, 42.185978271379234, 1.532712130518071],
                           [2.8857058801703928, 30.479305759404593, 4.237328303227819],
                           [2.787815831545738, 31.821947897701147, 3.5027969003150905],
                           [10.603080225829217, 8.001120404322569, 27.46788830295603],
                           [0.07313107333303914, 12.325354605364216, 53.307779943114674],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [4.969530117022278e-05, 322.9326014486039, 184.7193699826495],
                           [10.848784548129368, 14.585615790507841, 13.848277099136396],
                           [33.644052006785955, 0.15197937522811636, 60.48864526396182],
                           [67.26778562427383, 0.35151766355133673, 35.002875881881764],
                           [48.016818992929664, 0.019997889497120912, 61.43751570511156],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [80.78421081888365, 1.9341542023799023, 14.437199599340184],
                           [34.213278881310416, 0.43924170169983184, 52.89964510092818],
                           [35.33103708241298, 3.5542748562562276, 26.96535275934678],
                           [8.411996741305195, 28.126426831059653, 1.4543178589760866],
                           [17.648576925977572, 6.491284630305452, 27.16183217044371],
                           [11.865923042862734, 10.143337916933055, 20.356886840722375],
                           [125.72912767130768, 0.44887454871085264, 33.140471070500226],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [0.16573680142655117, 27.908048973795026, 10.500329203068302],
                           [0.8334202479833682, 31.389917080775984, 4.375816495282417],
                           [2.787815831542701, 31.821947897706277, 3.5027969003150905],
                           [33.55836959810893, 7.3619757928901945, 15.285424277666465],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [0.1657368014261809, 27.90804897379983, 10.500329203065355],
                           [6.177595486458452, 2.4698963180521005, 62.00102574415646],
                           [15.27716721971955, 5.7617658102087095, 27.876911851892167],
                           [21.659520773248992, 10.301486975014665, 14.426804301764028],
                           [17.308178440430193, 3.5424690672274823, 39.156184335310066],
                           [9.730741172355417, 32.79513262195397, 0.326174989962993],
                           [15.945579207879776, 0.2895610607302289, 89.20952020973053],
                           [6.5232547360587265, 22.47163172083615, 6.588615825003919],
                           [1.9329783063503818, 22.82139152469313, 10.472348308424968],
                           [10.979224447074506, 0.9108981776282215, 72.13107981750976],
                           [7.502454589243345, 18.068824420582754, 13.850754900738357],
                           [1.775091666403272, 19.8840680700664, 16.35503138255625],
                           [5.117143467233962, 4.21656406355104, 63.611277297031215],
                           [40.961146842337435, 0.9676644844849699, 43.20438702984144],
                           [1.7788504108090206, 35.72132929181334, 1.4401007551116924],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [4.589561259182364, 24.717268886336505, 8.3288692313527],
                           [7.917273251952683, 34.00504436487846, 0.5340496239119189],
                           [13.553413271041665, 8.274972797683311, 23.885481752801898],
                           [34.341780499945955, 3.744353314618373, 26.27446892037511],
                           [3.512313800206984, 38.31362547402685, 0.10560220707265709],
                           [0.5989222033096018, 15.725632880845758, 47.16729988383343],
                           [25.754683060256887, 3.0954680695729855, 35.612344989821345],
                           [22.834754562993897, 7.748430604687635, 20.2292918467861],
                           [18.207499499728478, 0.000488719267692604, 104.1335619536117],
                           [1.0132430493799036, 26.05795494163298, 9.789906229561902],
                           [15.741021008969707, 12.66889296313699, 13.159151253468197],
                           [78.1808891803457, 7.062394128647856, 2.887561012791642],
                           [39.95378295355896, 2.5436423410566866, 28.511305191447192],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [4.969530117022278e-05, 322.9326014486039, 184.7193699826495],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [13.568919374578707, 9.28552707709384, 21.02638712414344],
                           [34.213278881310416, 0.43924170169983184, 52.89964510092818],
                           [25.754683060256887, 3.0954680695729855, 35.612344989821345],
                           [9.730741172355417, 32.79513262195397, 0.326174989962993],
                           [1.0132430493799036, 26.057954941623695, 9.789906229559056],
                           [1.7788504108090206, 35.72132929181334, 1.4401007551116924],
                           [3.512313800206984, 38.31362547402685, 0.10560220707265709],
                           [6.5232547360587265, 22.47163172083615, 6.588615825003919],
                           [13.42826214964404, 0.2969379836602582, 78.86251240842077],
                           [15.741021008969707, 12.66889296313699, 13.159151253468197],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [0.06414357225630674, 59.65279533879202, 0.5189952051185565],
                           [0.08620072234100284, 31.17234295370835, 4.648285398527095],
                           [17.648576925977572, 6.491284630305452, 27.16183217044371],
                           [6.177595486458452, 2.4698963180521005, 62.00102574415646],
                           [33.55836959810893, 7.3619757928901945, 15.285424277666465],
                           [23.567764007177725, 0.2491453785945735, 68.78827713756199],
                           [2.787815831542701, 31.821947897706277, 3.5027969003150905],
                           [48.016818992929664, 0.019997889497120912, 61.43751570511156],
                           [73.50398278133706, 3.482896952707202, 11.20546207795856],
                           [1.6433995032970752, 8.83123430046554, 50.76288723226265],
                           [125.72912767130768, 0.44887454871085264, 33.140471070500226],
                           [19.998964180612745, 11.778163891068973, 12.374842252433247],
                           [1.775091666403272, 19.8840680700664, 16.35503138255625],
                           [13.938820174761217, 4.834064639243414, 35.21687437498169],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [67.26778562428129, 0.35151766355133673, 35.002875881876385],
                           [3.5916433670601497, 32.44308855401593, 2.1537207537904863],
                           [77.3740915605802, 6.944298066855472, 3.1406487119579944],
                           [4.09259010202973, 32.135468088739884, 2.400014030512113],
                           [19.55149539111877, 7.834853793476079, 22.111768060448572],
                           [0.1657368014261809, 27.90804897379983, 10.500329203065355],
                           [5.117143467233962, 4.21656406355104, 63.611277297031215],
                           [7.001171821956641, 23.746204826272525, 5.109445612167707],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [17.308178440430193, 3.5424690672274823, 39.156184335310066],
                           [0.16573680142655117, 27.908048973795026, 10.500329203068302],
                           [2.009288706702401, 10.175471780920072, 39.504237766471455],
                           [78.1808891803457, 7.062394128647856, 2.887561012791642],
                           [0.5153511319475005, 14.22475567250081, 29.75024813715785],
                           [15.945579207879776, 0.2895610607302289, 89.20952020973053],
                           [30.945956114037564, 1.6283157595178808, 39.49619771076921],
                           [7.502454589240854, 18.06882442057889, 13.850754900738357],
                           [1.9329783063503818, 22.82139152469313, 10.472348308424968],
                           [11.865923042862734, 10.143337916933055, 20.356886840722375],
                           [91.32444617713155, 0.03467767916189289, 34.72531554478345],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [45.794370050144934, 3.370051627818931e-05, 62.74342884819718],
                           [4.969530117022278e-05, 322.9326014486039, 184.7193699826495],
                           [15.741021008969707, 12.66889296313699, 13.159151253468197],
                           [11.921582477209201, 0.010696879501812119, 111.99234020025884],
                           [13.568919374578707, 9.28552707709384, 21.02638712414344],
                           [6.5232547360587265, 22.47163172083615, 6.588615825003919],
                           [34.213278881310416, 0.43924170169983184, 52.89964510092818],
                           [9.730741172355417, 32.79513262195397, 0.326174989962993],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [3.512313800206984, 38.31362547402685, 0.10560220707265709],
                           [0.08620072234100284, 31.17234295370835, 4.648285398527095],
                           [7.502454589240854, 18.06882442057889, 13.850754900738357],
                           [48.016818992929664, 0.019997889497120912, 61.43751570511156],
                           [13.42826214964404, 0.2969379836602582, 78.86251240842077],
                           [17.648576925977572, 6.491284630305452, 27.16183217044371],
                           [11.865923042862734, 10.143337916933055, 20.356886840722375],
                           [52.19601767336568, 0.012048451572420016, 58.599293778324956],
                           [25.754683060256887, 3.0954680695729855, 35.612344989821345],
                           [1.0132430493799036, 26.057954941623695, 9.789906229559056],
                           [1.7788504108090206, 35.72132929181334, 1.4401007551116924],
                           [75.85720578393419, 10.106971184092746, 0.7909807142208257],
                           [1.2557190804621323, 42.7735888326932, 0.1621793952830308],
                           [0.38572987212313936, 37.58487236564446, 2.5380268243885564],
                           [0.1657368014261809, 27.90804897379983, 10.500329203065355],
                           [30.945956114037564, 1.6283157595178808, 39.49619771076921],
                           [33.5583695981142, 7.361975792887726, 15.28542427767002],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [1.775091666403272, 19.8840680700664, 16.35503138255625],
                           [2.009288706702401, 10.175471780920072, 39.504237766471455],
                           [33.55836959810893, 7.3619757928901945, 15.285424277666465],
                           [13.938820174761217, 4.834064639243414, 35.21687437498169],
                           [73.50398278133706, 3.482896952707202, 11.20546207795856],
                           [2.787815831542701, 31.821947897706277, 3.5027969003150905],
                           [13.724632587895004, 0.2556411069134793, 79.55103975160362],
                           [77.3740915605802, 6.944298066855472, 3.1406487119579944],
                           [4.09259010202973, 32.135468088739884, 2.400014030512113],
                           [23.567764007177725, 0.2491453785945735, 68.78827713756199],
                           [17.35278536904067, 6.9567653197445285, 25.729606637832507],
                           [0.468497935489441, 12.635002742619966, 42.88928518972179],
                           [0.0444884774397116, 51.2552747881172, 0.0014207543613834243],
                           [20.209606037940745, 5.115663236452663, 30.150521967600802],
                           [6.177595486458452, 2.4698963180521005, 62.00102574415646],
                           [14.053292035731255, 4.556278308743968, 36.41578461926068],
                           [4.806147917236756, 29.090926712194147, 3.686132939110452],
                           [0.16573680142655117, 27.908048973795026, 10.500329203068302],
                           [38.265823650983165, 1.780854180513313, 36.0310441801937],
                           [3.5916433670601497, 32.44308855401593, 2.1537207537904863],
                           [67.26778562428129, 0.35151766355133673, 35.002875881876385],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [0.0444884774397116, 51.2552747881172, 0.0014207543613834243],
                           [73.76142286933303, 13.384772717299988, 0.02122006048442061],
                           [3.0155043370236925, 41.975682290494596, 0.04261583496903493],
                           [3.512313800206984, 38.31362547402685, 0.10560220707265709],
                           [1.2557190804621323, 42.7735888326932, 0.1621793952830308],
                           [9.730741172355417, 32.79513262195397, 0.326174989962993],
                           [75.85720578393419, 10.106971184092746, 0.7909807142208257],
                           [65.98977541027942, 21.20126125103483, 0.8213841310781534],
                           [70.80731894650805, 18.834926366858642, 0.8327125544453596],
                           [1.3108851957800465, 41.14855830078085, 0.8478749815772986],
                           [0.5048993271536816, 42.72779895561976, 0.920050413235479],
                           [1.7788504108090206, 35.72132929181334, 1.4401007551116924],
                           [16.453657342370146, 23.88763462419785, 1.959284217362945],
                           [3.5916433670601497, 32.44308855401593, 2.1537207537904863],
                           [4.09259010202973, 32.135468088739884, 2.400014030512113],
                           [0.38572987212313936, 37.58487236564446, 2.5380268243885564],
                           [10.82961734613227, 23.69366574965547, 2.582567835035866],
                           [77.3740915605802, 6.944298066855472, 3.1406487119579944],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [8.81102934815731, 26.067903003068803, 3.320384473512983],
                           [2.787815831542701, 31.821947897706277, 3.5027969003150905],
                           [4.806147917236756, 29.090926712194147, 3.686132939110452],
                           [6.607988584315748, 26.0305186767819, 3.95166903637265],
                           [0.08620072234100284, 31.17234295370835, 4.648285398527095],
                           [6.5232547360587265, 22.47163172083615, 6.588615825003919],
                           [3.1207625172342466, 24.522582035850547, 7.524481618694249],
                           [1.0132430493799036, 26.057954941623695, 9.789906229559056],
                           [0.1657368014261809, 27.90804897379983, 10.500329203065355],
                           [0.16573680142655117, 27.908048973795026, 10.500329203068302],
                           [0.2418340377984187, 27.101582128295203, 10.98173918634317],
                           [73.50398278133706, 3.482896952707202, 11.20546207795856],
                           [15.301424730542088, 12.903437521385733, 12.884795330609126],
                           [7.942836549519044, 16.560413086442907, 12.900659188307579],
                           [15.741021008969707, 12.66889296313699, 13.159151253468197],
                           [7.502454589240854, 18.06882442057889, 13.850754900738357],
                           [33.55836959810893, 7.3619757928901945, 15.285424277666465],
                           [33.55836959810893, 7.3619757928901945, 15.285424277666465],
                           [33.55836959810893, 7.3619757928901945, 15.285424277666465],
                           [33.5583695981142, 7.361975792887726, 15.28542427767002],
                           [35.226861331836695, 6.658304231573319, 16.319703399514708],
                           [1.775091666403272, 19.884068070074512, 16.35503138255257],
                           [1.775091666403272, 19.8840680700664, 16.35503138255625],
                           [26.20562764859457, 6.7948304804854445, 20.054555139332127],
                           [11.865923042862734, 10.143337916933055, 20.356886840722375],
                           [13.568919374578707, 9.28552707709384, 21.02638712414344],
                           [26.16302666059811, 6.082773093580554, 21.706194622428523],
                           [10.580421409921414, 9.475172182857191, 23.10864706699485],
                           [17.35278536904067, 6.9567653197445285, 25.729606637832507],
                           [8.507737226530791, 8.734686582586823, 26.992333162605647],
                           [1.3716580973586394, 60.48945308210783, 7.643134134527229e-06],
                           [0.0444884774397116, 51.2552747881172, 0.0014207543613834243],
                           [73.76142286933303, 13.384772717299988, 0.02122006048442061],
                           [3.0155043370236925, 41.975682290494596, 0.04261583496903493],
                           [3.512313800206984, 38.31362547402685, 0.10560220707265709],
                           [1.2557190804621323, 42.7735888326932, 0.1621793952830308],
                           [78.10414705353116, 10.493696884843606, 0.1753646920877865],
                           [9.730741172355417, 32.79513262195397, 0.326174989962993],
                           [13.45182003230224, 30.198846812475832, 0.39325612207759963],
                           [75.85720578393419, 10.106971184092746, 0.7909807142208257],
                           [65.98977541027942, 21.20126125103483, 0.8213841310781534],
                           [70.80731894650805, 18.834926366858642, 0.8327125544453596],
                           [1.3108851957800465, 41.14855830078085, 0.8478749815772986],
                           [0.5048993271536816, 42.72779895561976, 0.920050413235479],
                           [33.071197601625755, 18.57049251529015, 1.1133387186214947],
                           [1.7788504108090206, 35.72132929181334, 1.4401007551116924],
                           [0.9660874424366744, 39.13472538804914, 1.665602608249315],
                           [16.453657342370146, 23.88763462419785, 1.959284217362945],
                           [3.5916433670601497, 32.44308855401593, 2.1537207537904863],
                           [20.188711557554992, 20.663839923950285, 2.3967672201884174],
                           [4.09259010202973, 32.135468088739884, 2.400014030512113],
                           [0.38572987212313936, 37.58487236564446, 2.5380268243885564],
                           [10.82961734613227, 23.69366574965547, 2.582567835035866],
                           [77.3740915605802, 6.944298066855472, 3.1406487119579944],
                           [22.67433878969055, 18.53263721058495, 3.227549447592607],
                           [8.81102934815731, 26.067903003068803, 3.320384473512983],
                           [2.787815831542701, 31.821947897706277, 3.5027969003150905],
                           [4.806147917236756, 29.090926712194147, 3.686132939110452],
                           [6.607988584315748, 26.0305186767819, 3.95166903637265],
                           [5.751710783453857, 27.08553466426195, 4.3998297560450865],
                           [0.08620072234100284, 31.17234295370835, 4.648285398527095],
                           [6.5232547360587265, 22.47163172083615, 6.588615825003919],
                           [2.0182737129630763, 25.52262429984685, 7.104111633863058],
                           [3.1207625172342466, 24.522582035850547, 7.524481618694249],
                           [1.0132430493799036, 26.057954941623695, 9.789906229559056],
                           [0.1657368014261809, 27.90804897379983, 10.500329203065355],
                           [0.1657368014261809, 27.90804897379983, 10.500329203065355],
                           [0.16573680142655117, 27.908048973795026, 10.500329203068302],
                           [0.2418340377984187, 27.101582128295203, 10.98173918634317],
                           [73.50398278134486, 3.482896952707202, 11.205462077955515],
                           [73.50398278133706, 3.482896952707202, 11.20546207795856],
                           [18.057802498489657, 11.853615505106328, 12.785131182739574],
                           [15.301424730542088, 12.903437521385733, 12.884795330609126],
                           [7.942836549519044, 16.560413086442907, 12.900659188307579],
                           [15.741021008969707, 12.66889296313699, 13.159151253468197],
                           [13.32475597526631, 13.986157143150956, 13.509317703366962],
                           [120.76014217043587, 0.4092241554237587, 13.68124404764909],
                           [7.502454589240854, 18.06882442057889, 13.850754900738357],
                           [25.463041766622105, 9.235517200479267, 14.299888748655532],
                           [33.05634350376728, 7.578692570402744, 14.986918178103068]]
        else:
            raise Exception("Dataset '{}' not available.".format(set))

        return self._train_data()


class SurrogateBenchmarkLoudspeakerSingle(SurrogateBenchmarkData):
    def __init__(self, n_test=5, n_train=None, verbose=1):
        super().__init__(n_test, n_train, verbose)

    def values(self, set="100"):
        if set == "100":
            self.x_data = [[0.010150303759728387, 0.01697294189638882, 0.02493268307657625, 0.03262220919667486],
                           [0.013530510427096964, 0.033226314871835805, 0.024650506272602977, 0.02833948806246614],
                           [0.011565265353732014, 0.025212271409448796, 0.02433093743137681, 0.011717779294835192],
                           [0.025908666949089634, 0.031327075753886484, 0.023004423561901903, 0.030381057412195843],
                           [0.02763217480023264, 0.0301647673544252, 0.02828617936794623, 0.021841524209525332],
                           [0.021625127175843955, 0.03409281664007212, 0.030637004618269563, 0.021047728141974326],
                           [0.023594561075776805, 0.019657871573664186, 0.01290604181309714, 0.01530826574641353],
                           [0.0072876427012460755, 0.02281329181574809, 0.02732800415859666, 0.016801755040398596],
                           [0.027414937695923406, 0.03419353391280126, 0.029131039255773185, 0.03202823959835993],
                           [0.02546704660456456, 0.01784506385810669, 0.019964156251784784, 0.006412704304038479],
                           [0.019195539041571435, 0.02186760632793575, 0.018059187000255474, 0.03443770304312658],
                           [0.00943563516473122, 0.018035868370735078, 0.022897419500969712, 0.02917992335319057],
                           [0.031735178884243535, 0.021512746949688636, 0.015376702863693791, 0.017803549462774133],
                           [0.033684679750091234, 0.034401177913857856, 0.027089739322106926, 0.006761894977497129],
                           [0.0241173115450804, 0.03179825796713009, 0.02314911475353878, 0.008106150461566777],
                           [0.012003894965117442, 0.023122504434072697, 0.010305555968312948, 0.03241634895527622],
                           [0.026221731399593717, 0.031658485004896444, 0.029603370278114945, 0.008392643300679407],
                           [0.014234025436909076, 0.022788424682727966, 0.01708949058244938, 0.031524268339810965],
                           [0.01667028486092592, 0.019189599172621166, 0.028883782803684505, 0.0274428058599092],
                           [0.01108820691111207, 0.02202196386123985, 0.01740515637240259, 0.023374599542970474],
                           [0.008558878561716455, 0.026253913590469427, 0.021228227539096495, 0.006793736539441184],
                           [0.028885521473738133, 0.015585091187466895, 0.014133058019227884, 0.016204934433073012],
                           [0.017355118163606825, 0.033912001697556884, 0.012551052051653992, 0.007381815849224524],
                           [0.024351261896307035, 0.016380679838732995, 0.011169390322370989, 0.00978231986146855],
                           [0.014459887866188025, 0.02747947543867366, 0.02369904614860335, 0.03327345998172842],
                           [0.006748978689455202, 0.026900731732853695, 0.02951392010753904, 0.015882065409369088],
                           [0.02946297935321869, 0.029733726436583174, 0.020943503738562744, 0.011196125334825006],
                           [0.005823544607936883, 0.020048288922575083, 0.026962366507560266, 0.025020185951146735],
                           [0.02259253594377702, 0.01595111632154294, 0.015860709485566397, 0.01908183797504244],
                           [0.009334752905051528, 0.023820687680418597, 0.016105670751473698, 0.013785096115266154],
                           [0.008007774807559803, 0.017456610296636713, 0.02774667803066338, 0.008634969605095268],
                           [0.022729775991667343, 0.020570028299811315, 0.012754348845386725, 0.005203125394740378],
                           [0.009055355941229802, 0.033559702176315605, 0.030114743510108115, 0.005445624127668913],
                           [0.03392296276159212, 0.028063331187096995, 0.014974068102234, 0.016266662509838048],
                           [0.005278610432175765, 0.028648234673752757, 0.031189483022229243, 0.01249700313450952],
                           [0.02329832840026382, 0.015246125730915687, 0.013669302397360337, 0.03183509432998215],
                           [0.01056274051249402, 0.020283551239259574, 0.011436673317955483, 0.008977419923253235],
                           [0.03343516607437224, 0.03305582461456873, 0.01387546816153956, 0.010953484600227482],
                           [0.010718209946215026, 0.029589217695773353, 0.029800090081067723, 0.016525844027007932],
                           [0.03232004703674809, 0.027846138418424475, 0.012036059448764642, 0.024543768891606355],
                           [0.03089072115911865, 0.01846073528830258, 0.02239846235573246, 0.026931931885770227],
                           [0.020633415579999644, 0.027268926094808032, 0.01775114503513531, 0.017458020852175533],
                           [0.006808681505625685, 0.028943813171851768, 0.023881375979967843, 0.033748086697799703],
                           [0.02217227163933811, 0.024059926195938358, 0.03146395986106095, 0.02069203835404088],
                           [0.028319672586567743, 0.01860737253187464, 0.017263995337200053, 0.010136349089680148],
                           [0.015723098252972547, 0.024176692864085112, 0.01684180969348372, 0.011972950213724215],
                           [0.026630820708436594, 0.027525746485134256, 0.013079209361764989, 0.018714356949521452],
                           [0.020100676041060514, 0.01544395688704537, 0.02818115481269818, 0.015374613088204744],
                           [0.03134719435664559, 0.024760717697078258, 0.019457211434036573, 0.02154368917328295],
                           [0.017001538220615314, 0.03348230981993944, 0.02564322461484906, 0.027325449053876898],
                           [0.012710775459508349, 0.024649162911385053, 0.014304896472292827, 0.013472266593028404],
                           [0.018273766115469672, 0.021751294760155138, 0.020998665099751422, 0.025194218919404456],
                           [0.02951527185089268, 0.016775862404020927, 0.02183428100500934, 0.01476713538586065],
                           [0.005921013203560085, 0.03230117880324343, 0.012161771527730925, 0.022383670669380754],
                           [0.021933858024674614, 0.029862734636365004, 0.025907660669235166, 0.026531924996018603],
                           [0.006425221211814998, 0.017653174759058858, 0.026676251075811203, 0.01834532898869261],
                           [0.026462976045302563, 0.01940056262167527, 0.01985838002076693, 0.0293641465164007],
                           [0.032119488261741304, 0.017271156695046076, 0.018797967809989116, 0.024362972554342614],
                           [0.018624025589539843, 0.019739037748103504, 0.020667868029965425, 0.00961130325055599],
                           [0.033035584909082705, 0.029405090014791815, 0.01818308695851261, 0.019436543209151332],
                           [0.02068380027319103, 0.016262276694419237, 0.010623904166544556, 0.019953725160868675],
                           [0.009944858967424247, 0.026584961578074582, 0.025936570977275697, 0.02035469323772124],
                           [0.034376712151260196, 0.025043743073896604, 0.025122184721465123, 0.013231499513005666],
                           [0.014745761514698582, 0.019004388315576527, 0.01763777556515595, 0.0059469458014879415],
                           [0.007905621766571858, 0.025364203786439346, 0.020301769665043323, 0.012809965677988117],
                           [0.008461448995903616, 0.02455461283139606, 0.016562180334125577, 0.021381574542263544],
                           [0.007366171563872563, 0.022486481162506598, 0.015219559539707182, 0.0328097967697411],
                           [0.027811755499956767, 0.03243255489605519, 0.011656056329453268, 0.02600651860743927],
                           [0.024483420653107053, 0.030923613978199406, 0.021728467659377238, 0.034073230293627715],
                           [0.032734755450596266, 0.02337183446322095, 0.019211183822506483, 0.022719688018966545],
                           [0.03125037661523947, 0.030391876779439064, 0.02917704728460483, 0.02814403775337945],
                           [0.021293692787745813, 0.02605964891020253, 0.013246325802518246, 0.017381865490978865],
                           [0.032468746342332884, 0.0319763394739479, 0.01958322812836358, 0.020181399784564766],
                           [0.011284272701140304, 0.021161196207723908, 0.027577565990838264, 0.010733935997299309],
                           [0.014030622699300166, 0.030647478887429035, 0.02639241574617466, 0.019679350635005085],
                           [0.025306011524417204, 0.025908355586721478, 0.015796329023283054, 0.02958364469284142],
                           [0.017888817054931363, 0.02915626528131679, 0.01636900421631987, 0.03003498288866381],
                           [0.01958526794823269, 0.016580907512745373, 0.026338902025886103, 0.025436435993249813],
                           [0.02998283120319546, 0.030990384067581915, 0.018925678936639406, 0.030151777864107943],
                           [0.018977299020479517, 0.03291884122136579, 0.022099478327363538, 0.033593204199786265],
                           [0.019856468199157824, 0.022360868889034994, 0.024498782492695697, 0.023589017613564214],
                           [0.012409341499155669, 0.020254965549298856, 0.025351118256252744, 0.023193383103676943],
                           [0.02097987929820613, 0.03266010297682065, 0.011815443216920529, 0.014262024706579904],
                           [0.016029939236071534, 0.026808688487906726, 0.02033019796518541, 0.022418147201644374],
                           [0.023245036385912635, 0.02349651191018673, 0.018396248373900567, 0.030997529921837973],
                           [0.03029372387027348, 0.01511862990312232, 0.010838144375251277, 0.005839746904841575],
                           [0.02477231199072754, 0.031407800022736576, 0.014923146387119637, 0.01819735207232457],
                           [0.005451117261385388, 0.02820383016814483, 0.030220906926010288, 0.023975618405768128],
                           [0.016270230432987694, 0.018890130617315665, 0.01001289395024211, 0.025928779617023487],
                           [0.013040242899390063, 0.01614744137322141, 0.022593174936688235, 0.009394264093379578],
                           [0.018384817138336944, 0.03046228797864859, 0.024105423472674276, 0.01042427102594811],
                           [0.015217052207400897, 0.0236036899961888, 0.014521062762512527, 0.014672159625822408],
                           [0.015545172328638307, 0.020771372159989046, 0.02144362500393844, 0.0073188952863916116],
                           [0.02801490488778815, 0.021326636021743783, 0.02862928341139239, 0.007747697298383526],
                           [0.01739661921560268, 0.02641673301194324, 0.01350298738784228, 0.026661478298099384],
                           [0.02891317985928948, 0.02865266162107393, 0.027891368451784755, 0.014086357248234108],
                           [0.026934411473261814, 0.025713265071474714, 0.030876387279124194, 0.012370305767679297],
                           [0.030371734043607705, 0.021014028907476907, 0.02344297933915599, 0.028774387544950923],
                           [0.012120739322555245, 0.01819058499658016, 0.01093908795710849, 0.030852912112602014],
                           [0.013696711528398426, 0.028419879042536178, 0.03076551343470775, 0.02772328751808993]]
            self.y_data = [[-1.1677787174932213], [-1.4376189055244304], [-1.373031841547464], [-1.418284379808004],
                           [-1.4472175402701346], [-1.4678023075075393], [-1.1315998565061176], [-1.3446011003353122],
                           [-1.4466950076671972], [-1.252837475441959], [-1.2817882246317909], [-1.196710769045755],
                           [-1.2432349265160059], [-1.4531688406960073], [-1.4143901748848717], [-1.2508596028049808],
                           [-1.4658454972921302], [-1.2869391669700287], [-1.2798796004582127], [-1.25198939154187],
                           [-1.3410832836473319], [-1.1644342078248715], [-1.1497430903659036], [-1.0136005730113757],
                           [-1.390202999683518], [-1.4180385501806574], [-1.3697481342223934], [-1.2662792344635885],
                           [-1.1648773554274352], [-1.216581673572063], [-1.2163881843915731], [-1.0650951669200552],
                           [-1.474556704736409], [-1.254829474412331], [-1.437236036681136], [-1.2003188559157671],
                           [-1.0135535893836642], [-1.2027761220321853], [-1.4510479214181775], [-1.2974318319673104],
                           [-1.338717542375065], [-1.308386960257777], [-1.400660928816329], [-1.3900006544388914],
                           [-1.2211398493174253], [-1.2354689346872805], [-1.2475196025570692], [-1.2566261928225821],
                           [-1.3379433337889417], [-1.4441632932164339], [-1.1883381064300071], [-1.2944499150961304],
                           [-1.3070687651692565], [-1.3041392986108218], [-1.430185933492152], [-1.205874142972806],
                           [-1.3038682776652917], [-1.3020720185977983], [-1.2436306652217028], [-1.335003752332767],
                           [-1.0917902713408896], [-1.4002140198137842], [-1.4048816326463143], [-1.1555358067828163],
                           [-1.316505783211063], [-1.2743357512001627], [-1.2602454606833946], [-1.3292170276957846],
                           [-1.4073971029949153], [-1.3339095282201456], [-1.4418306946630326], [-1.221683260414382],
                           [-1.3665596554240165], [-1.3225927853523605], [-1.442704737548234], [-1.3344259540502865],
                           [-1.3672024792791126], [-1.2490061904203373], [-1.3901825545668798], [-1.4190611832705025],
                           [-1.3357253175574686], [-1.2830343859835514], [-1.1937768048145723], [-1.3541280294843032],
                           [-1.3218911473445714], [-0.9938771795190015], [-1.291172394469776], [-1.4261189651750479],
                           [-1.1481801394432716], [-1.1636940145521077], [-1.4224079251343404], [-1.188523014651399],
                           [-1.2657606932804693], [-1.392418855362544], [-1.3079358715201335], [-1.4410299975865952],
                           [-1.4239129199532308], [-1.3585960817341027], [-1.138856229821183], [-1.4313288412989704]]
        else:
            raise Exception("Dataset '{}' not available.".format(set))

        return self._train_data()

    def limits(self):
        return [[0.005, 0.0345], [0.005, 0.0345], [0.01, 0.0315], [0.005, 0.0345]]


class SurrogateBenchmarkLoudspeakerMulti(SurrogateBenchmarkData):
    def __init__(self, n_test=5, n_train=None, verbose=1):
        super().__init__(n_test, n_train, verbose)

    def values(self, set="200"):
        if set == "200":
            self.x_data = [[0.031596547228888036, 0.024455767599596667, 0.025681363437764892, 0.008146928355161495],
                           [0.0053506872587475865, 0.015050096800889843, 0.021554449417183576, 0.013702647870250065],
                           [0.01372245731754565, 0.031586716036533555, 0.014789109883139105, 0.023225693376730545],
                           [0.023979148693263907, 0.028365495367850935, 0.028553012766689284, 0.026154649288020127],
                           [0.024112289532094888, 0.03342713197720861, 0.01730014148040283, 0.01735765634218753],
                           [0.023474071210216777, 0.03411606123185678, 0.02355871431962417, 0.029878716446682746],
                           [0.01615176726970779, 0.024954940797104443, 0.022117720607934155, 0.013535640641665483],
                           [0.027173453582080197, 0.028616609363903582, 0.018846799540808666, 0.03196404648655364],
                           [0.0313756607125538, 0.021158286045629703, 0.02818366715759331, 0.015231136313539693],
                           [0.01924150329370518, 0.018475566501551098, 0.011902121046787468, 0.024278800350613534],
                           [0.010609115039285614, 0.03222525378283371, 0.0238653575623129, 0.025582095385223665],
                           [0.02890865016117975, 0.016191516152956047, 0.015382234271253613, 0.021806537486891972],
                           [0.010052469386487382, 0.029194488620731294, 0.030681495398063814, 0.02336339772160652],
                           [0.009019560515831358, 0.0167382887181109, 0.018430992387852, 0.017164137861635797],
                           [0.01653656305914035, 0.019834735365639584, 0.028387197832469492, 0.020149469236651146],
                           [0.01367803384320598, 0.03017272190554613, 0.010535081930764897, 0.01296162831656113],
                           [0.013944945033750548, 0.021945735293253475, 0.016984466655810295, 0.016442909188611875],
                           [0.008503083160661059, 0.030621282496970367, 0.02510912711230751, 0.03352650812351297],
                           [0.02620475563093311, 0.017855627478541924, 0.013251605155252701, 0.007958856554297686],
                           [0.00957997484252315, 0.02723536617785858, 0.013810555624097153, 0.032597801170996324],
                           [0.030869985412323577, 0.02680053347695115, 0.02490053356163649, 0.01180298339793384],
                           [0.01943247058229589, 0.03353283542689722, 0.026591066423600845, 0.018042804859064142],
                           [0.02367398615790557, 0.016491516972593444, 0.02437868726853966, 0.031261974200913895],
                           [0.01487625972178529, 0.028912913665735306, 0.02475546949597728, 0.015075728771294879],
                           [0.022524968940027288, 0.029837244463074394, 0.03140480561425161, 0.00966170208204549],
                           [0.03422006276744364, 0.015227647073510481, 0.014680446441621767, 0.007367430265066158],
                           [0.006489716610763896, 0.03298672767704233, 0.025924436793999223, 0.022006039245722687],
                           [0.016783766179845973, 0.01561498315895851, 0.025811823492901932, 0.02698298773756593],
                           [0.025148763742588023, 0.02559617873219286, 0.016453378893652993, 0.011066215417246738],
                           [0.00718548859999438, 0.02749745374407167, 0.017921501619996706, 0.031071333784088252],
                           [0.033939944464841124, 0.033863951873429964, 0.021381156160418073, 0.014336689992194347],
                           [0.011278632798054428, 0.029692278996054977, 0.014359279026411605, 0.027448746869836123],
                           [0.015333941515416265, 0.021478670738882967, 0.013685739699074847, 0.010801759410192423],
                           [0.027788501088594455, 0.030973531101083453, 0.02034603394851103, 0.022391144630361973],
                           [0.019925624445500407, 0.017131405982369664, 0.028275542511113944, 0.028522194778666184],
                           [0.010182682066510679, 0.021297418894969073, 0.010782886590912658, 0.02820176260237296],
                           [0.00517775146697478, 0.02675089956952193, 0.012861571938557607, 0.006886156746533002],
                           [0.032846238713656696, 0.027119460712201518, 0.02867362157482793, 0.034036571858133795],
                           [0.010911168081137894, 0.024299486433581645, 0.01614501843285703, 0.005444770840206862],
                           [0.011973567313146638, 0.01936317337827006, 0.019414535110758007, 0.015966150272500326],
                           [0.006148030108326572, 0.025682320089902788, 0.029614430546923302, 0.028045031048794966],
                           [0.006014756400703651, 0.018588831415906525, 0.010556762406729016, 0.014745315250716356],
                           [0.026867791612244275, 0.027009047136606835, 0.030257203662179835, 0.020765765116835616],
                           [0.031547829120519406, 0.020313333101954715, 0.015899344149245688, 0.013220306018657903],
                           [0.024190002038874735, 0.022192931667984, 0.01257929731803215, 0.022179098047675178],
                           [0.03266903657955847, 0.030088167016172583, 0.018200433485628206, 0.009425432477128185],
                           [0.022555399604766428, 0.026471268145055286, 0.017132883429618575, 0.02715580523959015],
                           [0.005798132024085593, 0.015362911793026741, 0.014882069224156127, 0.017606038839520194],
                           [0.011653105794929723, 0.01610712990707375, 0.029443859629469382, 0.03389708858225541],
                           [0.016262747718269063, 0.032084018863951326, 0.029173618381289776, 0.008778511656313714],
                           [0.015594255640382856, 0.026157340263055984, 0.014515697724483117, 0.019932696878169308],
                           [0.030410614599137815, 0.020155814873361773, 0.02757538601228439, 0.0064124580029933965],
                           [0.012808462823972732, 0.019182008823031657, 0.028962440239952404, 0.010458588855301085],
                           [0.031819645579452355, 0.019677536678548525, 0.028876183445810354, 0.01750539880677692],
                           [0.013216760745327072, 0.03236366803976731, 0.026285016475569627, 0.02106358632634513],
                           [0.020496982434974505, 0.02230216406727377, 0.018077648573437764, 0.028681807289015475],
                           [0.032984984431077426, 0.024774029849028396, 0.01925222791640567, 0.01949930505129398],
                           [0.027117835196241443, 0.020180098554777162, 0.03058054043468001, 0.017937375391093992],
                           [0.009261580070076884, 0.0327352439785559, 0.018640477656659292, 0.006031477117894337],
                           [0.029444625482766624, 0.029585115066650996, 0.023081743338045788, 0.02309865609429287],
                           [0.011923958955447414, 0.03152445568966573, 0.011607755359674213, 0.02503569244490069],
                           [0.025302524279055775, 0.02080870410936252, 0.028735181512237383, 0.02697489600389275],
                           [0.03305495480026131, 0.027825846564626687, 0.027124488693305117, 0.031404049351031006],
                           [0.023335939506089063, 0.020916890223363325, 0.02177502045103645, 0.021480247742732213],
                           [0.0125434689922025, 0.025507341947562105, 0.013012225164965509, 0.01821778722689935],
                           [0.009752078590612015, 0.01631581502070385, 0.020735405327008107, 0.006320805711677656],
                           [0.031249396747725013, 0.025236487860462564, 0.03080384651321473, 0.018303712732084827],
                           [0.017421361432743157, 0.015414201924208188, 0.014001719887275037, 0.005843514544670641],
                           [0.026460472365229654, 0.01999489624873364, 0.015198980579337287, 0.032423859215430874],
                           [0.018125811086671915, 0.030858271687685, 0.011791094751186393, 0.01224269794434934],
                           [0.009489539575270545, 0.020533797584572473, 0.023314636238656847, 0.0050419718566161],
                           [0.03387277072534, 0.01802316471948524, 0.012238121708315435, 0.024728000751746373],
                           [0.026585000669379636, 0.024480071287696407, 0.012261564581186504, 0.02185706869346309],
                           [0.02085291032906219, 0.03249919908566711, 0.012659174905624128, 0.023987409836253953],
                           [0.028032370877214212, 0.029267719685310604, 0.022328955096572525, 0.009070216202357841],
                           [0.017100901751803062, 0.021662345449533892, 0.03020793497821956, 0.022617665694465924],
                           [0.022072358762459606, 0.028511543876087383, 0.011468014230068395, 0.005709481456427309],
                           [0.033493854084436406, 0.016987053452035914, 0.02318494394610772, 0.01558690547961629],
                           [0.020959726442146387, 0.015683466802136087, 0.026850875302819772, 0.016969860034844044],
                           [0.007049328720985048, 0.021067521837349338, 0.015585671759373958, 0.009248729569968393],
                           [0.022281243085809706, 0.03292770880632073, 0.013142220560040802, 0.01627058301247227],
                           [0.008563942815604084, 0.03054475077872513, 0.029268327720911765, 0.01137545479318599],
                           [0.014934158264058413, 0.023208397420167694, 0.027502847682059135, 0.015350509505657629],
                           [0.019564301881820213, 0.02260544267448463, 0.021446737689514363, 0.025476804896939344],
                           [0.03443280573293673, 0.01748137873300205, 0.01991742046387654, 0.019860182166897795],
                           [0.029705208393110756, 0.01834617300771656, 0.01514167947263307, 0.01584833875740082],
                           [0.012889072161020126, 0.01792717085861756, 0.022835966796688967, 0.02682049589418588],
                           [0.016863032411282362, 0.023578171793767366, 0.016046168510914564, 0.007630908112496726],
                           [0.022959653137478227, 0.0220224379527476, 0.028004141593160864, 0.01563378669136765],
                           [0.018819387714503196, 0.021372103456305507, 0.013630679318015173, 0.012440334367310048],
                           [0.024391589682416264, 0.024903179623926304, 0.029683012584567778, 0.010370390935050128],
                           [0.019155172483611444, 0.01645021070001125, 0.020609254708751884, 0.010182831707561138],
                           [0.021141339157535032, 0.02243938911446109, 0.02620333363641658, 0.03412829611572282],
                           [0.02850132952822549, 0.03234952546732089, 0.02950151035738629, 0.020854513219541567],
                           [0.018176475227837267, 0.015872004860940807, 0.01294128353942152, 0.023533556494373508],
                           [0.018952135972949265, 0.030383709541667886, 0.026535481565314663, 0.033112432311447795],
                           [0.007246345903786322, 0.03041185988961677, 0.02015553523600653, 0.021309632222263133],
                           [0.010464946276885417, 0.019423824039172487, 0.010269897208325307, 0.02288179972921052],
                           [0.007854488110124487, 0.02419172548921976, 0.014989427488876031, 0.0320544703978761],
                           [0.020479277828000075, 0.03408941841145842, 0.027867479458389258, 0.024380166915388163],
                           [0.020171798196768023, 0.02371882394125089, 0.022580865933314345, 0.009893824095191829],
                           [0.006625744465168094, 0.018995691651238017, 0.026088958912120444, 0.03300764032731823],
                           [0.01423367799804981, 0.03444684165391437, 0.017069667900107756, 0.019436129812651786],
                           [0.025607367874295673, 0.026921956608721542, 0.021015257791414645, 0.009856837015727439],
                           [0.022822272689067717, 0.03174057523328129, 0.030981811272937365, 0.013060557711064912],
                           [0.018558191790621854, 0.03200442113956674, 0.011035594885209976, 0.03175626779578522],
                           [0.008750976910559559, 0.02880373536493281, 0.017320559912139655, 0.033461061522018914],
                           [0.026773341340084262, 0.022879277797244493, 0.014509170419925606, 0.010135167929545703],
                           [0.021789504126109673, 0.018781750539738764, 0.01531841742755345, 0.030230362351881822],
                           [0.015156286734767488, 0.025978325920488723, 0.020304098537346563, 0.028749740991604926],
                           [0.027327822779673307, 0.025787578463560532, 0.013952523124252308, 0.012632110323607736],
                           [0.01595510695321184, 0.01831104726909958, 0.025357054962635392, 0.01667455710456067],
                           [0.014689625941146707, 0.01775592496468153, 0.0249813027236406, 0.0069536479905883165],
                           [0.031070931810254138, 0.015146069629718206, 0.020777462100772248, 0.011309426649081297],
                           [0.022147784148526722, 0.023085221386055