# -*- coding: utf-8 -*-
# Copyright 2022 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import warnings
from typing import Callable, Dict, Optional, Sequence, Tuple, Union

from google.api_core import grpc_helpers
from google.api_core import operations_v1
from google.api_core import gapic_v1
import google.auth  # type: ignore
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.grpc import SslCredentials  # type: ignore

import grpc  # type: ignore

from google.cloud.eventarc_v1.types import channel
from google.cloud.eventarc_v1.types import channel_connection
from google.cloud.eventarc_v1.types import discovery
from google.cloud.eventarc_v1.types import eventarc
from google.cloud.eventarc_v1.types import trigger
from google.longrunning import operations_pb2  # type: ignore
from .base import EventarcTransport, DEFAULT_CLIENT_INFO


class EventarcGrpcTransport(EventarcTransport):
    """gRPC backend transport for Eventarc.

    Eventarc allows users to subscribe to various events that are
    provided by Google Cloud services and forward them to supported
    destinations.

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends protocol buffers over the wire using gRPC (which is built on
    top of HTTP/2); the ``grpcio`` package must be installed.
    """

    _stubs: Dict[str, Callable]

    def __init__(
        self,
        *,
        host: str = "eventarc.googleapis.com",
        credentials: ga_credentials.Credentials = None,
        credentials_file: str = None,
        scopes: Sequence[str] = None,
        channel: grpc.Channel = None,
        api_mtls_endpoint: str = None,
        client_cert_source: Callable[[], Tuple[bytes, bytes]] = None,
        ssl_channel_credentials: grpc.ChannelCredentials = None,
        client_cert_source_for_mtls: Callable[[], Tuple[bytes, bytes]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        api_audience: Optional[str] = None,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
                This argument is ignored if ``channel`` is provided.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional(Sequence[str])): A list of scopes. This argument is
                ignored if ``channel`` is provided.
            channel (Optional[grpc.Channel]): A ``Channel`` instance through
                which to make calls.
            api_mtls_endpoint (Optional[str]): Deprecated. The mutual TLS endpoint.
                If provided, it overrides the ``host`` argument and tries to create
                a mutual TLS channel with client SSL credentials from
                ``client_cert_source`` or application default SSL credentials.
            client_cert_source (Optional[Callable[[], Tuple[bytes, bytes]]]):
                Deprecated. A callback to provide client SSL certificate bytes and
                private key bytes, both in PEM format. It is ignored if
                ``api_mtls_endpoint`` is None.
            ssl_channel_credentials (grpc.ChannelCredentials): SSL credentials
                for the grpc channel. It is ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Optional[Callable[[], Tuple[bytes, bytes]]]):
                A callback to provide client certificate bytes and private key bytes,
                both in PEM format. It is used to configure a mutual TLS channel. It is
                ignored if ``channel`` or ``ssl_channel_credentials`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you're developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.

        Raises:
          google.auth.exceptions.MutualTLSChannelError: If mutual TLS transport
              creation failed for any reason.
          google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """
        self._grpc_channel = None
        self._ssl_channel_credentials = ssl_channel_credentials
        self._stubs: Dict[str, Callable] = {}
        self._operations_client: Optional[operations_v1.OperationsClient] = None

        if api_mtls_endpoint:
            warnings.warn("api_mtls_endpoint is deprecated", DeprecationWarning)
        if client_cert_source:
            warnings.warn("client_cert_source is deprecated", DeprecationWarning)

        if channel:
            # Ignore credentials if a channel was passed.
            credentials = False
            # If a channel was explicitly provided, set it.
            self._grpc_channel = channel
            self._ssl_channel_credentials = None

        else:
            if api_mtls_endpoint:
                host = api_mtls_endpoint

                # Create SSL credentials with client_cert_source or application
                # default SSL credentials.
                if client_cert_source:
                    cert, key = client_cert_source()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )
                else:
                    self._ssl_channel_credentials = SslCredentials().ssl_credentials

            else:
                if client_cert_source_for_mtls and not ssl_channel_credentials:
                    cert, key = client_cert_source_for_mtls()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )

        # The base transport sets the host, credentials and scopes
        super().__init__(
            host=host,
            credentials=credentials,
            credentials_file=credentials_file,
            scopes=scopes,
            quota_project_id=quota_project_id,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            api_audience=api_audience,
        )

        if not self._grpc_channel:
            self._grpc_channel = type(self).create_channel(
                self._host,
                # use the credentials which are saved
                credentials=self._credentials,
                # Set ``credentials_file`` to ``None`` here as
                # the credentials that we saved earlier should be used.
                credentials_file=None,
                scopes=self._scopes,
                ssl_credentials=self._ssl_channel_credentials,
                quota_project_id=quota_project_id,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )

        # Wrap messages. This must be done after self._grpc_channel exists
        self._prep_wrapped_messages(client_info)

    @classmethod
    def create_channel(
        cls,
        host: str = "eventarc.googleapis.com",
        credentials: ga_credentials.Credentials = None,
        credentials_file: str = None,
        scopes: Optional[Sequence[str]] = None,
        quota_project_id: Optional[str] = None,
        **kwargs,
    ) -> grpc.Channel:
        """Create and return a gRPC channel object.
        Args:
            host (Optional[str]): The host for the channel to use.
            credentials (Optional[~.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify this application to the service. If
                none are specified, the client will attempt to ascertain
                the credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is mutually exclusive with credentials.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            kwargs (Optional[dict]): Keyword arguments, which are passed to the
                channel creation.
        Returns:
            grpc.Channel: A gRPC channel object.

        Raises:
            google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """

        return grpc_helpers.create_channel(
            host,
            credentials=credentials,
            credentials_file=credentials_file,
            quota_project_id=quota_project_id,
            default_scopes=cls.AUTH_SCOPES,
            scopes=scopes,
            default_host=cls.DEFAULT_HOST,
            **kwargs,
        )

    @property
    def grpc_channel(self) -> grpc.Channel:
        """Return the channel designed to connect to this service."""
        return self._grpc_channel

    @property
    def operations_client(self) -> operations_v1.OperationsClient:
        """Create the client designed to process long-running operations.

        This property caches on the instance; repeated calls return the same
        client.
        """
        # Quick check: Only create a new client if we do not already have one.
        if self._operations_client is None:
            self._operations_client = operations_v1.OperationsClient(self.grpc_channel)

        # Return the client from cache.
        return self._operations_client

    @property
    def get_trigger(self) -> Callable[[eventarc.GetTriggerRequest], trigger.Trigger]:
        r"""Return a callable for the get trigger method over gRPC.

        Get a single trigger.

        Returns:
            Callable[[~.GetTriggerRequest],
                    ~.Trigger]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_trigger" not in self._stubs:
            self._stubs["get_trigger"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/GetTrigger",
                request_serializer=eventarc.GetTriggerRequest.serialize,
                response_deserializer=trigger.Trigger.deserialize,
            )
        return self._stubs["get_trigger"]

    @property
    def list_triggers(
        self,
    ) -> Callable[[eventarc.ListTriggersRequest], eventarc.ListTriggersResponse]:
        r"""Return a callable for the list triggers method over gRPC.

        List triggers.

        Returns:
            Callable[[~.ListTriggersRequest],
                    ~.ListTriggersResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_triggers" not in self._stubs:
            self._stubs["list_triggers"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/ListTriggers",
                request_serializer=eventarc.ListTriggersRequest.serialize,
                response_deserializer=eventarc.ListTriggersResponse.deserialize,
            )
        return self._stubs["list_triggers"]

    @property
    def create_trigger(
        self,
    ) -> Callable[[eventarc.CreateTriggerRequest], operations_pb2.Operation]:
        r"""Return a callable for the create trigger method over gRPC.

        Create a new trigger in a particular project and
        location.

        Returns:
            Callable[[~.CreateTriggerRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_trigger" not in self._stubs:
            self._stubs["create_trigger"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/CreateTrigger",
                request_serializer=eventarc.CreateTriggerRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["create_trigger"]

    @property
    def update_trigger(
        self,
    ) -> Callable[[eventarc.UpdateTriggerRequest], operations_pb2.Operation]:
        r"""Return a callable for the update trigger method over gRPC.

        Update a single trigger.

        Returns:
            Callable[[~.UpdateTriggerRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_trigger" not in self._stubs:
            self._stubs["update_trigger"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/UpdateTrigger",
                request_serializer=eventarc.UpdateTriggerRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["update_trigger"]

    @property
    def delete_trigger(
        self,
    ) -> Callable[[eventarc.DeleteTriggerRequest], operations_pb2.Operation]:
        r"""Return a callable for the delete trigger method over gRPC.

        Delete a single trigger.

        Returns:
            Callable[[~.DeleteTriggerRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_trigger" not in self._stubs:
            self._stubs["delete_trigger"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/DeleteTrigger",
                request_serializer=eventarc.DeleteTriggerRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["delete_trigger"]

    @property
    def get_channel(self) -> Callable[[eventarc.GetChannelRequest], channel.Channel]:
        r"""Return a callable for the get channel method over gRPC.

        Get a single Channel.

        Returns:
            Callable[[~.GetChannelRequest],
                    ~.Channel]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_channel" not in self._stubs:
            self._stubs["get_channel"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/GetChannel",
                request_serializer=eventarc.GetChannelRequest.serialize,
                response_deserializer=channel.Channel.deserialize,
            )
        return self._stubs["get_channel"]

    @property
    def list_channels(
        self,
    ) -> Callable[[eventarc.ListChannelsRequest], eventarc.ListChannelsResponse]:
        r"""Return a callable for the list channels method over gRPC.

        List channels.

        Returns:
            Callable[[~.ListChannelsRequest],
                    ~.ListChannelsResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_channels" not in self._stubs:
            self._stubs["list_channels"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/ListChannels",
                request_serializer=eventarc.ListChannelsRequest.serialize,
                response_deserializer=eventarc.ListChannelsResponse.deserialize,
            )
        return self._stubs["list_channels"]

    @property
    def create_channel_(
        self,
    ) -> Callable[[eventarc.CreateChannelRequest], operations_pb2.Operation]:
        r"""Return a callable for the create channel method over gRPC.

        Create a new channel in a particular project and
        location.

        Returns:
            Callable[[~.CreateChannelRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_channel_" not in self._stubs:
            self._stubs["create_channel_"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/CreateChannel",
                request_serializer=eventarc.CreateChannelRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["create_channel_"]

    @property
    def update_channel(
        self,
    ) -> Callable[[eventarc.UpdateChannelRequest], operations_pb2.Operation]:
        r"""Return a callable for the update channel method over gRPC.

        Update a single channel.

        Returns:
            Callable[[~.UpdateChannelRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_channel" not in self._stubs:
            self._stubs["update_channel"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/UpdateChannel",
                request_serializer=eventarc.UpdateChannelRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["update_channel"]

    @property
    def delete_channel(
        self,
    ) -> Callable[[eventarc.DeleteChannelRequest], operations_pb2.Operation]:
        r"""Return a callable for the delete channel method over gRPC.

        Delete a single channel.

        Returns:
            Callable[[~.DeleteChannelRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_channel" not in self._stubs:
            self._stubs["delete_channel"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/DeleteChannel",
                request_serializer=eventarc.DeleteChannelRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["delete_channel"]

    @property
    def get_provider(
        self,
    ) -> Callable[[eventarc.GetProviderRequest], discovery.Provider]:
        r"""Return a callable for the get provider method over gRPC.

        Get a single Provider.

        Returns:
            Callable[[~.GetProviderRequest],
                    ~.Provider]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_provider" not in self._stubs:
            self._stubs["get_provider"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/GetProvider",
                request_serializer=eventarc.GetProviderRequest.serialize,
                response_deserializer=discovery.Provider.deserialize,
            )
        return self._stubs["get_provider"]

    @property
    def list_providers(
        self,
    ) -> Callable[[eventarc.ListProvidersRequest], eventarc.ListProvidersResponse]:
        r"""Return a callable for the list providers method over gRPC.

        List providers.

        Returns:
            Callable[[~.ListProvidersRequest],
                    ~.ListProvidersResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_providers" not in self._stubs:
            self._stubs["list_providers"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/ListProviders",
                request_serializer=eventarc.ListProvidersRequest.serialize,
                response_deserializer=eventarc.ListProvidersResponse.deserialize,
            )
        return self._stubs["list_providers"]

    @property
    def get_channel_connection(
        self,
    ) -> Callable[
        [eventarc.GetChannelConnectionRequest], channel_connection.ChannelConnection
    ]:
        r"""Return a callable for the get channel connection method over gRPC.

        Get a single ChannelConnection.

        Returns:
            Callable[[~.GetChannelConnectionRequest],
                    ~.ChannelConnection]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_channel_connection" not in self._stubs:
            self._stubs["get_channel_connection"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/GetChannelConnection",
                request_serializer=eventarc.GetChannelConnectionRequest.serialize,
                response_deserializer=channel_connection.ChannelConnection.deserialize,
            )
        return self._stubs["get_channel_connection"]

    @property
    def list_channel_connections(
        self,
    ) -> Callable[
        [eventarc.ListChannelConnectionsRequest],
        eventarc.ListChannelConnectionsResponse,
    ]:
        r"""Return a callable for the list channel connections method over gRPC.

        List channel connections.

        Returns:
            Callable[[~.ListChannelConnectionsRequest],
                    ~.ListChannelConnectionsResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_channel_connections" not in self._stubs:
            self._stubs["list_channel_connections"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/ListChannelConnections",
                request_serializer=eventarc.ListChannelConnectionsRequest.serialize,
                response_deserializer=eventarc.ListChannelConnectionsResponse.deserialize,
            )
        return self._stubs["list_channel_connections"]

    @property
    def create_channel_connection(
        self,
    ) -> Callable[[eventarc.CreateChannelConnectionRequest], operations_pb2.Operation]:
        r"""Return a callable for the create channel connection method over gRPC.

        Create a new ChannelConnection in a particular
        project and location.

        Returns:
            Callable[[~.CreateChannelConnectionRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_channel_connection" not in self._stubs:
            self._stubs["create_channel_connection"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/CreateChannelConnection",
                request_serializer=eventarc.CreateChannelConnectionRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["create_channel_connection"]

    @property
    def delete_channel_connection(
        self,
    ) -> Callable[[eventarc.DeleteChannelConnectionRequest], operations_pb2.Operation]:
        r"""Return a callable for the delete channel connection method over gRPC.

        Delete a single ChannelConnection.

        Returns:
            Callable[[~.DeleteChannelConnectionRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_channel_connection" not in self._stubs:
            self._stubs["delete_channel_connection"] = self.grpc_channel.unary_unary(
                "/google.cloud.eventarc.v1.Eventarc/DeleteChannelConnection",
                request_serializer=eventarc.DeleteChannelConnectionRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["delete_channel_connection"]

    def close(self):
        self.grpc_channel.close()

    @property
    def kind(self) -> str:
        return "grpc"


__all__ = ("EventarcGrpcTransport",)
