"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Scheduler = exports.BaseOS = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const info_1 = require("./info");
/**
 * @experimental
 */
var BaseOS;
(function (BaseOS) {
    BaseOS["CENTOS_7"] = "centos7";
    BaseOS["RHEL_7"] = "rhel7";
    BaseOS["AMZN2"] = "amazonlinux2";
})(BaseOS = exports.BaseOS || (exports.BaseOS = {}));
/**
 * @experimental
 */
class Scheduler extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        const urlsuffix = cdk.Stack.of(this).urlSuffix;
        const stack = cdk.Stack.of(this);
        const region = cdk.Stack.of(this).region;
        const s3InstallBucket = props.s3InstallBucket;
        const s3InstallFolder = props.s3InstallFolder;
        const baseOs = props.baseOs ? props.baseOs.toString() : BaseOS.AMZN2.toString();
        const clusterId = props.network.clusterId;
        const socaVersion = info_1.SocaInfo.Data.Version;
        const ldapUserName = props.ldapUserName;
        const ldapUserPassword = props.ldapUserPassword;
        // amazon linux 2 AMI if customAmi is undefined
        const socaInstallAmi = props.customAmi ? ec2.MachineImage.genericLinux({
            [region]: props.customAmi,
        }) : new ec2.AmazonLinuxImage({
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
        const socaInstallAmiId = socaInstallAmi.getImage(this).imageId;
        // const bootscript: string = fs.readFileSync(path.join(__dirname, '../assets/user-data'), 'utf-8');
        const userData = ec2.UserData.forLinux();
        // userData.addCommands(bootscript)
        userData.addCommands(`
export PATH=$PATH:/usr/local/bin
# Deactivate shell to make sure users won't access the cluster if it's not ready
echo '
************* SOCA FIRST TIME CONFIGURATION *************
Hold on, cluster is not ready yet.
Please wait ~30 minutes as SOCA is being installed.
Once cluster is ready to use, this message will be replaced automatically and you will be able to SSH.
*********************************************************' > /etc/nologin

if [ "${baseOs}" == "amazonlinux2" ] || [ "${baseOs}" == "rhel7" ];
  then
    usermod --shell /usr/sbin/nologin ec2-user
fi

if [ "${baseOs}" == "centos7" ];
  then
    usermod --shell /usr/sbin/nologin centos
fi

# Install awscli
if [ "${baseOs}" == "centos7" ] || [ "${baseOs}" == "rhel7" ];
then
  EASY_INSTALL=$(which easy_install-2.7)
  $EASY_INSTALL pip
  PIP=$(which pip2.7)
  $PIP install awscli
fi

# Disable automatic motd update if using ALI
if [ "${baseOs}" == "amazonlinux2" ];
then
  /usr/sbin/update-motd --disable
  rm /etc/cron.d/update-motd
  rm -f /etc/update-motd.d/*
fi

AWS=$(which aws)
echo export "SOCA_BASE_OS=${baseOs}" >> /etc/environment
echo export "SOCA_CONFIGURATION=${clusterId}" >> /etc/environment
echo export "AWS_DEFAULT_REGION=${region} " >> /etc/environment
echo export "SOCA_INSTALL_BUCKET=${s3InstallBucket}" >> /etc/environment
echo export "SOCA_INSTALL_BUCKET_FOLDER=${s3InstallFolder}" >> /etc/environment
echo export "SOCA_VERSION=${socaVersion}" >> /etc/environment
echo export "SOCA_INSTALL_AMI=${socaInstallAmiId}" >> /etc/environment
source /etc/environment

# Tag EBS disks manually as CFN ASG does not support it
AWS_AVAIL_ZONE=$(curl http://169.254.169.254/latest/meta-data/placement/availability-zone)
AWS_REGION=$(echo \"$AWS_AVAIL_ZONE\" | sed "s/[a-z]$//")
AWS_INSTANCE_ID=$(curl http://169.254.169.254/latest/meta-data/instance-id)
EBS_IDS=$(aws ec2 describe-volumes --filters Name=attachment.instance-id,Values="$AWS_INSTANCE_ID" --region $AWS_REGION --query "Volumes[*].[VolumeId]" --out text | tr "\n" " ")
$AWS ec2 create-tags --resources $EBS_IDS --region $AWS_REGION --tags Key=Name,Value="${clusterId} Root Disk" "Key=soca:ClusterId,Value=${clusterId}"

# Tag Network Adapter for the Scheduler
ENI_IDS=$(aws ec2 describe-network-interfaces --filters Name=attachment.instance-id,Values="$AWS_INSTANCE_ID" --region $AWS_REGION --query "NetworkInterfaces[*].[NetworkInterfaceId]" --out text | tr "\n" " ")
$AWS ec2 create-tags --resources $ENI_IDS --region $AWS_REGION --tags Key=Name,Value="${clusterId} Scheduler Network Adapter" "Key=soca:ClusterId,Value=${clusterId}"

echo "@reboot /bin/aws s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/SchedulerPostReboot.sh /root && /bin/bash /root/SchedulerPostReboot.sh ${s3InstallBucket} ${s3InstallFolder} ${ldapUserName} '${ldapUserPassword}' >> /root/PostRebootConfig.log 2>&1" | crontab -
AWS=$(which aws)
$AWS s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/config.cfg /root/
$AWS s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/requirements.txt /root/
$AWS s3 cp s3://${s3InstallBucket}/${s3InstallFolder}/scripts/Scheduler.sh /root/
/bin/bash /root/Scheduler.sh ${props.storage.efsDataDns} ${props.storage.efsAppsDns} >> /root/Scheduler.sh.log 2>&1
`);
        const scheduler = new ec2.Instance(this, 'Scheduler', {
            vpc: props.network.vpc,
            instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : new ec2.InstanceType('m5.xlarge'),
            machineImage: socaInstallAmi,
            userData,
            securityGroup: props.schedulerSecurityGroup,
        });
        scheduler.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'pricing:GetProducts',
                'budgets:ViewBudget',
                'ec2:DescribeInstances',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
                'ec2:DescribeImages',
                'ec2:DescribeInstanceAttribute',
                'ec2:DescribeInstanceTypes',
                'ec2:DescribeReservedInstances',
                'ec2:DescribeSpotInstanceRequests',
                'ec2:DescribeVpcClassicLink',
                'ec2:DescribeVolumes',
                'ec2:DescribePlacementGroups',
                'ec2:DescribeKeyPairs',
                'ec2:DescribeLaunchTemplates',
                'ec2:DescribeLaunchTemplateVersions',
                'ec2:DescribeNetworkInterfaces',
                'ec2:DescribeSpotFleetRequests',
                'ec2:DescribeSpotFleetInstances',
                'fsx:DescribeFileSystems',
                'autoscaling:DescribeAutoScalingGroups',
                'autoscaling:DescribeScalingActivities',
                'autoscaling:DescribeLaunchConfigurations',
                'elasticloadbalancing:DescribeRules',
                'elasticloadbalancing:DescribeListeners',
                'elasticloadbalancing:DescribeTargetGroups',
                'savingsplans:DescribeSavingsPlans',
                'servicequotas:ListServiceQuotas',
            ],
            resources: ['*'],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
                'autoscaling:DeleteAutoScalingGroup',
                'autoscaling:CreateAutoScalingGroup',
                'autoscaling:DetachInstances',
                'ec2:DeleteLaunchTemplate',
                'ec2:CreateLaunchTemplate',
                'fsx:CreateDataRepositoryTask',
            ],
            resources: ['*'],
            conditions: {
                StringLikeIfExists: {
                    'autoscaling:LaunchConfigurationName': props.network.clusterId,
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:CreateTags',
            ],
            resources: [
                stack.formatArn({ service: 'ec2', resource: 'volume' }),
                stack.formatArn({ service: 'ec2', resource: 'network-interface' }),
                stack.formatArn({ service: 'ec2', resource: 'instance' }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'cloudformation:CreateStack',
                'cloudformation:DeleteStack',
                'cloudformation:DescribeStacks',
            ],
            resources: ['*'],
            conditions: {
                'ForAllValues:StringEquals': {
                    'cloudformation:TemplateURL': `https://s3.${urlsuffix}/${s3InstallBucket}/${s3InstallFolder}/templates/ComputeNode.template`,
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:RunInstances',
                'ec2:TerminateInstances',
                'ec2:CreatePlacementGroup',
                'ec2:DeletePlacementGroup',
            ],
            resources: [
                ...['subnet', 'key-pair', 'instance', 'launch-template', 'volume', 'security-group', 'placement-group', 'network-interface'].map(resource => stack.formatArn({ service: 'ec2', resource })).concat(['snapshot', 'image'].map(resource => stack.formatArn({ service: 'ec2', resource, account: '' }))),
            ],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({ service: 'ec2', resource: 'vpc', resourceName: props.network.vpc.vpcId }),
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'lambda:InvokeFunction',
            ],
            resources: [
                stack.formatArn({
                    service: 'lambda',
                    resource: 'function',
                    resourceName: `${props.network.clusterId}-Metrics`,
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'fsx:CreateFileSystem',
            ],
            resources: [
                stack.formatArn({
                    service: 'fsx',
                    resource: 'file-system',
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'fsx:DeleteFileSystem',
            ],
            resources: [
                stack.formatArn({
                    service: 'fsx',
                    resource: 'file-system',
                }),
            ],
            conditions: {
                StringLike: {
                    'aws:ResourceTag/soca:ClusterId': props.network.clusterId,
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'iam:CreateServiceLinkedRole',
                'iam:AttachRolePolicy',
                'iam:PutRolePolicy',
            ],
            resources: [
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/s3.data-source.lustre.fsx.amazonaws.com/*',
                }),
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/autoscaling.amazonaws.com/*',
                }),
                stack.formatArn({
                    service: 'iam',
                    resource: 'role',
                    region: '',
                    resourceName: 'aws-service-role/spotfleet.amazonaws.com/*',
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ses:SendEmail',
            ],
            resources: [
                stack.formatArn({
                    service: 'ses',
                    resource: 'identity',
                }),
            ],
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:CreatePlacementGroup',
                'ec2:DeletePlacementGroup',
                'ec2:RequestSpotFleet',
                'ec2:ModifySpotFleetRequest',
                'ec2:CancelSpotFleetRequests',
            ],
            resources: ['*'],
            conditions: {
                'ForAllValues:ArnEqualsIfExists': {
                    'ec2:Vpc': stack.formatArn({
                        service: 'ec2',
                        resource: 'vpc',
                        resourceName: props.network.vpc.vpcId,
                    }),
                },
            },
        }));
        scheduler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject',
                's3:ListBucket',
                's3:PutObject',
            ],
            resources: [
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: props.s3InstallBucket,
                }),
                stack.formatArn({
                    service: 's3',
                    account: '',
                    resource: '',
                    region: '',
                    resourceName: `${props.s3InstallBucket}/*`,
                }),
            ],
        }));
        const eip = new ec2.CfnEIP(this, 'EIPScheduler', {
            instanceId: scheduler.instanceId,
            domain: 'vpc',
        });
        this.publicIp = eip.ref;
        new cdk.CfnOutput(this, 'SchedulerEIP', { value: eip.ref });
    }
}
exports.Scheduler = Scheduler;
_a = JSII_RTTI_SYMBOL_1;
Scheduler[_a] = { fqn: "cdk-soca.Scheduler", version: "0.2.5" };
//# sourceMappingURL=data:application/json;base64,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