import sys
import numpy as np
import nanonet.tb as tb
import nanonet.negf as negf
from nanonet.negf.hamiltonian_chain import HamiltonianChain
from nanonet.negf.recursive_greens_functions import recursive_gf

np.warnings.filterwarnings('ignore')


def simple_chain_greens_function(energy, h_0, h_r):
    """Analytical expression for the chain response function
    according to []

    Parameters
    ----------
    energy :
        param h_0:
    h_r :
        return:
    h_0 :
        

    Returns
    -------

    """

    gf = (energy - h_0) / (2 * h_r ** 2)

    for j, E in enumerate(energy):
        if E - h_0 <= -2 * h_r:
            gf[0, j] = gf[0, j] + 1.0 / (2 * h_r ** 2) * np.sqrt((E - h_0) ** 2 - 4 * h_r ** 2)
        elif E - h_0 >= 2 * h_r:
            gf[0, j] = gf[0, j] - 1.0 / (2 * h_r ** 2) * np.sqrt((E - h_0) ** 2 - 4 * h_r ** 2)
        else:
            gf[0, j] = gf[0, j] - 1.0j / (2 * h_r ** 2) * np.sqrt(4 * h_r ** 2 - (E - h_0) ** 2)

    return gf


def single_atom_chain():
    """Test set for a single-atom chain.
    
    :return:

    Parameters
    ----------

    Returns
    -------

    """

    sys.path.insert(0, '/home/mk/TB_project/tb')

    a = tb.Orbitals('A')
    a.add_orbital('s', 0.7)
    tb.set_tb_params(PARAMS_A_A={'ss_sigma': 0.5})

    xyz_file = """1
    H cell
    A1       0.0000000000    0.0000000000    0.0000000000
    """

    h = tb.Hamiltonian(xyz=xyz_file, nn_distance=1.1)
    h.initialize()
    h.set_periodic_bc([[0, 0, 1.0]])
    h_l, h_0, h_r = h.get_hamiltonians()
    num_sites = h_0.shape[0]

    energy = np.linspace(-3.0, 3.0, 300)

    tr = np.zeros((energy.shape[0]))
    dos = np.zeros((energy.shape[0]))

    sgf_l = []
    sgf_r = []

    for j, E in enumerate(energy):
        se_l, se_r = negf.surface_greens_function(E, h_l, h_0, h_r, iterate=False)
        sgf_l.append(se_l)
        sgf_r.append(se_r)
        gf = np.linalg.pinv(E * np.identity(num_sites) - h_0 - se_l - se_r)
        gamma_l = 1j * (se_l - se_l.conj().T)
        gamma_r = 1j * (se_r - se_r.conj().T)
        tr[j] = np.real(np.trace(gamma_l.dot(gf).dot(gamma_r).dot(gf.conj().T)))
        dos[j] = np.real(np.trace(1j * (gf - gf.conj().T)))

    sgf_l = np.array(sgf_l)
    sgf_r = np.array(sgf_r)

    return energy, dos, tr, h, sgf_l, sgf_r


def complex_chain():
    """ """
    sys.path.insert(0, '/home/mk/TB_project/tb')

    a = tb.Orbitals('A')
    a.add_orbital('s', -0.7)
    b = tb.Orbitals('B')
    b.add_orbital('s', -0.5)
    c = tb.Orbitals('C')
    c.add_orbital('s', -0.3)

    tb.set_tb_params(PARAMS_A_A={'ss_sigma': -0.5},
                     PARAMS_B_B={'ss_sigma': -0.5},
                     PARAMS_A_B={'ss_sigma': -0.5},
                     PARAMS_B_C={'ss_sigma': -0.5},
                     PARAMS_A_C={'ss_sigma': -0.5})

    xyz_file = """4
    H cell
    A1       0.0000000000    0.0000000000    0.0000000000
    B2       0.0000000000    0.0000000000    1.0000000000
    A2       0.0000000000    1.0000000000    0.0000000000
    B3       0.0000000000    1.0000000000    1.0000000000
    """

    h = tb.Hamiltonian(xyz=xyz_file, nn_distance=1.1)
    h.initialize()
    h.set_periodic_bc([[0, 0, 2.0]])
    h_l, h_0, h_r = h.get_hamiltonians()

    energy = np.linspace(-3.0, 1.5, 700)

    sgf_l = []
    sgf_r = []

    for E in energy:
        left_se, right_se = negf.surface_greens_function(E, h_l, h_0, h_r, iterate=5)
        sgf_l.append(left_se)
        sgf_r.append(right_se)

    sgf_l = np.array(sgf_l)
    sgf_r = np.array(sgf_r)

    num_sites = h_0.shape[0]
    gf = np.linalg.pinv(np.multiply.outer(energy, np.identity(num_sites)) - h_0 - sgf_l - sgf_r)

    tr = np.zeros((energy.shape[0]))
    dos = np.zeros((energy.shape[0]))

    for j, E in enumerate(energy):
        gf0 = gf[j, :, :]
        gamma_l = 1j * (sgf_l[j, :, :] - sgf_l[j, :, :].conj().T)
        gamma_r = 1j * (sgf_r[j, :, :] - sgf_r[j, :, :].conj().T)
        tr[j] = np.real(np.trace(gamma_l.dot(gf0).dot(gamma_r).dot(gf0.conj().T)))
        dos[j] = np.real(np.trace(1j * (gf0 - gf0.conj().T)))

    return energy, dos, tr, h, sgf_l, sgf_r


def test_single_atom_chain():
    """ """
    energy, dos, tr, h, sgf_l, sgf_r = single_atom_chain()

    h_l, h_0, h_r = h.get_hamiltonians()

    np.testing.assert_allclose(sgf_l, sgf_r, rtol=1e-2)
    expected = h_l * simple_chain_greens_function(energy, h_0, h_r) * h_r
    np.testing.assert_allclose(np.squeeze(sgf_r), np.squeeze(expected), rtol=1e-2)


def test_complex_chain():
    """ """
    energy, dos, tr, h, sgf_l, sgf_r = complex_chain()

    np.testing.assert_allclose(dos, expected_dos_of_complex_chain(), rtol=1e-2)
    np.testing.assert_allclose(tr, expected_tr_of_complex_chain(), rtol=1e-2)


def run_for_periods(single_period_test, periods):
    """

    Parameters
    ----------
    single_period_test :
        
    periods :
        

    Returns
    -------

    """
    energy, dos, tr, h, sgf_l, sgf_r = single_period_test()
    h_l, h_0, h_r = h.get_hamiltonians()
    cell = h.ct.pcv

    h_chain = HamiltonianChain(h_l, h_0, h_r, h.get_site_coordinates())
    h_chain.translate(cell[0], periods, periods)

    num_sites = h_0.shape[0]
    num_sites1 = 2 * periods * num_sites + num_sites
    num_periods = 2 * periods + 1

    tr1 = np.zeros((energy.shape[0]))
    dos1 = np.zeros((energy.shape[0]))

    for j, E in enumerate(energy):
        h_chain.add_self_energies(sgf_l[j, :, :], sgf_r[j, :, :])
        gf = np.linalg.pinv(E * np.identity(num_sites1) - h_chain.get_matrix())
        h_chain.remove_self_energies()
        se_l, se_r = h_chain.translate_self_energies(sgf_l[j, :, :], sgf_r[j, :, :])

        gamma_l = 1j * (se_l - se_l.conj().T)
        gamma_r = 1j * (se_r - se_r.conj().T)
        tr1[j] = np.real(np.trace(gamma_l.dot(gf).dot(gamma_r).dot(gf.conj().T)))
        dos1[j] = np.real(np.trace(1j * (gf - gf.conj().T))) / num_periods

    print('hi')
    np.testing.assert_allclose(dos, dos1, atol=15)
    np.testing.assert_allclose(tr, tr1, atol=0.2)


def run_for_periods_recursive(single_period_test, periods):
    """

    Parameters
    ----------
    single_period_test :
        
    periods :
        

    Returns
    -------

    """
    energy, dos, tr, h, sgf_l, sgf_r = single_period_test()
    h_l, h_0, h_r = h.get_hamiltonians()
    cell = h.ct.pcv

    h_chain = HamiltonianChain(h_l, h_0, h_r, h.get_site_coordinates())
    h_chain.translate(cell[0], periods, periods)

    num_periods = 2 * periods + 1

    dos1 = np.zeros((energy.shape[0]))

    for j, E in enumerate(energy):

        h_chain.add_self_energies(sgf_l[j, :, :], sgf_r[j, :, :])
        _, grd, grl, gru, gr_left = recursive_gf(E, h_chain.h_l, h_chain.h_0, h_chain.h_r)
        h_chain.remove_self_energies()

        for jj in range(len(grd)):
            dos1[j] = dos1[j] + np.real(np.trace(1j * (grd[jj] - grd[jj].conj().T))) / num_periods

    # dos1[dos1 > 100] = 0
    np.testing.assert_allclose(dos, dos1, atol=25)


def test_gf_single_atom_chain_several_periods():
    """ """
    for j in range(5):
        run_for_periods(single_atom_chain, periods=j)


def test_gf_complex_chain_several_periods():
    """ """
    for j in range(5):
        run_for_periods(complex_chain, periods=j)


def test_gf_single_atom_chain_several_periods_recursive():
    """ """
    for j in range(5):
        run_for_periods_recursive(single_atom_chain, periods=j)


def test_gf_complex_chain_several_periods_recursive():
    """ """
    for j in range(5):
        run_for_periods_recursive(complex_chain, periods=j)


def expected_dos_of_complex_chain():
    """ """
    return np.array([8.71733068e-10, 9.16642348e-10, 9.64117652e-10, 1.01431709e-09,
                     1.06741601e-09, 1.12359914e-09, 1.18306191e-09, 1.24601712e-09,
                     1.31269013e-09, 1.38332359e-09, 1.45817603e-09, 1.53752718e-09,
                     1.62167369e-09, 1.71093617e-09, 1.80565932e-09, 1.90620715e-09,
                     2.01298238e-09, 2.12640699e-09, 2.24693861e-09, 2.37506933e-09,
                     2.51132694e-09, 2.65628532e-09, 2.81056073e-09, 2.97480963e-09,
                     3.14974698e-09, 3.33614415e-09, 3.53482698e-09, 3.74670136e-09,
                     3.97273097e-09, 4.21396779e-09, 4.47153859e-09, 4.74667359e-09,
                     5.04070457e-09, 5.35505912e-09, 5.69130369e-09, 6.05113008e-09,
                     6.43637571e-09, 6.84902717e-09, 7.29125432e-09, 7.76540917e-09,
                     8.27405635e-09, 8.81998644e-09, 9.40622866e-09, 1.00361090e-08,
                     1.07132315e-08, 1.14415310e-08, 1.22253236e-08, 1.30693285e-08,
                     1.39786811e-08, 1.49590313e-08, 1.60165598e-08, 1.71580478e-08,
                     1.83909182e-08, 1.97233593e-08, 2.11642926e-08, 2.27236872e-08,
                     2.44123472e-08, 2.62422640e-08, 2.82266921e-08, 3.03801701e-08,
                     3.27188793e-08, 3.52606354e-08, 3.80252461e-08, 4.10345673e-08,
                     4.43128345e-08, 4.78871615e-08, 5.17874233e-08, 5.60470969e-08,
                     6.07034321e-08, 6.57979408e-08, 7.13769776e-08, 7.74924745e-08,
                     8.42025023e-08, 9.15723051e-08, 9.96749906e-08, 1.08592744e-07,
                     1.18418354e-07, 1.29256159e-07, 1.41223798e-07, 1.54455210e-07,
                     1.69101194e-07, 1.85332973e-07, 2.03345638e-07, 2.23360712e-07,
                     2.45631490e-07, 2.70446281e-07, 2.98136950e-07, 3.29081885e-07,
                     3.63718672e-07, 4.02547858e-07, 4.46151465e-07, 4.95197404e-07,
                     5.50466430e-07, 6.12862492e-07, 6.83439534e-07, 7.63429093e-07,
                     8.54272576e-07, 9.57672767e-07, 1.07562030e-06, 1.21048493e-06,
                     1.36506806e-06, 1.54270600e-06, 1.74744533e-06, 1.98396917e-06,
                     2.25808093e-06, 2.57673038e-06, 2.94837194e-06, 3.38331561e-06,
                     3.89417951e-06, 4.49655018e-06, 5.20969494e-06, 6.05763570e-06,
                     7.07052539e-06, 8.28641940e-06, 9.75362517e-06, 1.15341228e-05,
                     1.37079482e-05, 1.63794070e-05, 1.96855970e-05, 2.38090378e-05,
                     2.89950590e-05, 3.55775818e-05, 4.40179773e-05, 5.49629153e-05,
                     6.93340804e-05, 8.84713982e-05, 1.14365440e-04, 1.50044349e-04,
                     2.00251747e-04, 2.72663221e-04, 3.80197585e-04, 5.45641012e-04,
                     8.11558752e-04, 1.26365777e-03, 2.09082008e-03, 3.76816664e-03,
                     7.73782632e-03, 1.99576058e-02, 8.68655876e-02, 4.76984699e+01,
                     3.61049293e+01, 2.53303270e+01, 2.06428422e+01, 1.78724402e+01,
                     1.59930692e+01, 1.46120518e+01, 1.35429823e+01, 1.26842814e+01,
                     1.19753536e+01, 1.13775191e+01, 1.08647338e+01, 1.04187305e+01,
                     1.00263471e+01, 9.67778432e+00, 9.36555614e+00, 9.08382821e+00,
                     8.82803168e+00, 8.59448390e+00, 8.38020547e+00, 8.18273502e+00,
                     8.00003684e+00, 7.83039905e+00, 7.67236491e+00, 7.52472610e+00,
                     7.38641423e+00, 7.25652400e+00, 7.13426063e+00, 7.01893181e+00,
                     6.90993391e+00, 6.80673746e+00, 6.70886375e+00, 6.61589595e+00,
                     6.52745733e+00, 6.44321514e+00, 6.36286399e+00, 6.28613593e+00,
                     6.21278400e+00, 6.14258396e+00, 6.07533866e+00, 6.01086085e+00,
                     5.94898257e+00, 5.88954930e+00, 5.83242310e+00, 5.77747068e+00,
                     5.72457728e+00, 5.67362884e+00, 5.62452577e+00, 5.57717485e+00,
                     5.53148661e+00, 5.48738437e+00, 5.44478907e+00, 5.40363503e+00,
                     5.36385165e+00, 5.32538418e+00, 5.28817430e+00, 5.25216899e+00,
                     5.21731670e+00, 5.18357445e+00, 5.15089621e+00, 5.11924324e+00,
                     5.08857522e+00, 5.05886011e+00, 5.03006102e+00, 5.00214849e+00,
                     4.97509180e+00, 4.94886358e+00, 4.92343805e+00, 4.89879017e+00,
                     4.87489861e+00, 4.85173850e+00, 4.82929321e+00, 4.80754207e+00,
                     4.78646675e+00, 4.76605334e+00, 4.74628412e+00, 4.72714661e+00,
                     4.70862563e+00, 4.69071074e+00, 4.67339172e+00, 4.65665653e+00,
                     4.64049655e+00, 4.62490378e+00, 4.60987228e+00, 4.59539474e+00,
                     4.58146617e+00, 4.56808519e+00, 4.55524402e+00, 4.54294511e+00,
                     4.53118445e+00, 4.51996398e+00, 4.50928695e+00, 4.49915241e+00,
                     4.48957028e+00, 4.48054003e+00, 4.47207327e+00, 4.46417822e+00,
                     4.45686512e+00, 4.45014719e+00, 4.44404093e+00, 4.43856316e+00,
                     4.43373320e+00, 4.42957606e+00, 4.42611912e+00, 4.42339206e+00,
                     4.42143024e+00, 4.42027360e+00, 4.41996760e+00, 4.42056439e+00,
                     4.42212390e+00, 4.42471235e+00, 4.42840891e+00, 4.43330124e+00,
                     4.43949155e+00, 4.44709887e+00, 4.45625986e+00, 4.46713303e+00,
                     4.47990527e+00, 4.49479563e+00, 4.51205881e+00, 4.53200573e+00,
                     4.55499710e+00, 4.58148203e+00, 4.61199337e+00, 4.64718977e+00,
                     4.68789241e+00, 4.73512572e+00, 4.79019371e+00, 4.85478490e+00,
                     4.93111804e+00, 5.02216798e+00, 5.13203534e+00, 5.26651215e+00,
                     5.43410775e+00, 5.64787248e+00, 5.92896315e+00, 6.31438350e+00,
                     6.87599880e+00, 7.77734586e+00, 9.50763319e+00, 1.48577678e+01,
                     2.75441863e-01, 3.59748045e-02, 1.48378298e-02, 8.47270034e-03,
                     5.71796853e-03, 4.31497912e-03, 3.56679553e-03, 3.22124997e-03,
                     3.21205957e-03, 3.61333987e-03, 4.72979200e-03, 7.50855005e-03,
                     1.54795559e-02, 4.95321404e-02, 6.62221692e-01, 4.45843672e+01,
                     2.77898990e+01, 2.19044312e+01, 1.86683152e+01, 1.65524546e+01,
                     1.50325701e+01, 1.38739016e+01, 1.29535814e+01, 1.22002550e+01,
                     1.15693009e+01, 1.10313248e+01, 1.05660890e+01, 1.01594960e+01,
                     9.80242106e+00, 9.49361144e+00, 9.29727899e+00, 2.86088047e+01,
                     1.89655468e+01, 1.65701072e+01, 1.53191264e+01, 1.45018620e+01,
                     1.39048488e+01, 1.34387222e+01, 1.30585414e+01, 1.27388248e+01,
                     1.24638707e+01, 1.22233581e+01, 1.20101710e+01, 1.18191944e+01,
                     1.16466384e+01, 1.14896076e+01, 1.13458728e+01, 1.12136329e+01,
                     1.10914519e+01, 1.09781506e+01, 1.08727497e+01, 1.07744275e+01,
                     1.06824921e+01, 1.05963486e+01, 1.05154824e+01, 1.04394511e+01,
                     1.03678747e+01, 1.03004020e+01, 1.02367434e+01, 1.01766359e+01,
                     1.01198353e+01, 1.00661351e+01, 1.00153459e+01, 9.96730294e+00,
                     9.92184476e+00, 9.87883718e+00, 9.83815746e+00, 9.79968710e+00,
                     9.76332745e+00, 9.72898132e+00, 9.69656325e+00, 9.66599565e+00,
                     9.63720359e+00, 9.61012650e+00, 9.58470109e+00, 9.56087283e+00,
                     9.53859305e+00, 9.51781649e+00, 9.49849659e+00, 9.48060035e+00,
                     9.46409225e+00, 9.44894060e+00, 9.43511563e+00, 9.42259507e+00,
                     9.41135200e+00, 9.40136849e+00, 9.39262694e+00, 9.38511429e+00,
                     9.37881290e+00, 9.37371764e+00, 9.36981130e+00, 9.36709462e+00,
                     9.36555905e+00, 9.36520481e+00, 9.36603020e+00, 9.36803566e+00,
                     9.37122498e+00, 9.37560353e+00, 9.38118198e+00, 9.38796386e+00,
                     9.39596837e+00, 9.40520344e+00, 9.41568529e+00, 9.42743853e+00,
                     9.44047874e+00, 9.45483018e+00, 9.47052356e+00, 9.48758467e+00,
                     9.50604645e+00, 9.52594625e+00, 9.54732414e+00, 9.57022146e+00,
                     9.59468788e+00, 9.62077146e+00, 9.64853252e+00, 9.67803141e+00,
                     9.70933859e+00, 9.74252505e+00, 9.77767463e+00, 9.81487471e+00,
                     9.85422448e+00, 9.89583071e+00, 9.93981091e+00, 9.98629733e+00,
                     1.00354306e+01, 1.00873698e+01, 1.01423004e+01, 1.02004099e+01,
                     1.02619193e+01, 1.03270788e+01, 1.03961661e+01, 1.04694944e+01,
                     1.05474183e+01, 1.06303480e+01, 1.07187533e+01, 1.08131771e+01,
                     1.09142499e+01, 1.10227094e+01, 1.11394413e+01, 1.12654975e+01,
                     1.14021474e+01, 1.15509645e+01, 1.17139041e+01, 1.18934405e+01,
                     1.20927760e+01, 1.23161791e+01, 1.25694557e+01, 1.28608200e+01,
                     1.32024124e+01, 1.36131063e+01, 1.41245077e+01, 1.47947813e+01,
                     1.57471126e+01, 1.73087013e+01, 2.08542179e+01, 1.34608582e+01,
                     9.34383582e+00, 9.61011617e+00, 9.93934898e+00, 1.03155874e+01,
                     1.07443192e+01, 1.12367478e+01, 1.18091084e+01, 1.24848294e+01,
                     1.32985044e+01, 1.43034891e+01, 1.55876131e+01, 1.73077719e+01,
                     1.97802727e+01, 2.37765069e+01, 3.19715650e+01, 7.30616453e+01,
                     1.61264621e-01, 2.89988312e-02, 1.12605563e-02, 6.16615128e-03,
                     4.24771035e-03, 3.48698148e-03, 3.28495461e-03, 3.45223092e-03,
                     3.97553953e-03, 4.98721630e-03, 6.87440222e-03, 1.07395123e-02,
                     2.06276540e-02, 6.37185580e-02, 6.32528461e+01, 1.17153082e+01,
                     8.65003647e+00, 7.35770263e+00, 6.62291324e+00, 6.14411379e+00,
                     5.80640534e+00, 5.55552225e+00, 5.36216340e+00, 5.20905144e+00,
                     5.08524368e+00, 4.98347955e+00, 4.89872499e+00, 4.82739354e+00,
                     4.76683839e+00, 4.71508290e+00, 4.67060234e+00, 4.63221686e+00,
                     4.59898747e+00, 4.57016617e+00, 4.54514473e+00, 4.52342658e+00,
                     4.50460376e+00, 4.48833418e+00, 4.47433073e+00, 4.46235363e+00,
                     4.45219827e+00, 4.44368840e+00, 4.43667566e+00, 4.43103233e+00,
                     4.42664227e+00, 4.42341106e+00, 4.42125688e+00, 4.42010325e+00,
                     4.41988821e+00, 4.42055259e+00, 4.42204789e+00, 4.42433238e+00,
                     4.42736656e+00, 4.43111913e+00, 4.43555862e+00, 4.44065979e+00,
                     4.44640159e+00, 4.45276363e+00, 4.45972917e+00, 4.46728457e+00,
                     4.47541706e+00, 4.48411663e+00, 4.49337501e+00, 4.50318454e+00,
                     4.51354136e+00, 4.52444112e+00, 4.53588079e+00, 4.54786034e+00,
                     4.56037984e+00, 4.57343928e+00, 4.58704060e+00, 4.60118901e+00,
                     4.61588899e+00, 4.63114527e+00, 4.64696282e+00, 4.66335226e+00,
                     4.68031896e+00, 4.69787590e+00, 4.71602887e+00, 4.73479310e+00,
                     4.75418031e+00, 4.77420361e+00, 4.79487701e+00, 4.81621665e+00,
                     4.83824211e+00, 4.86096610e+00, 4.88441208e+00, 4.90860144e+00,
                     4.93355367e+00, 4.95929321e+00, 4.98584700e+00, 5.01323612e+00,
                     5.04149603e+00, 5.07065432e+00, 5.10074369e+00, 5.13179507e+00,
                     5.16384879e+00, 5.19694466e+00, 5.23112332e+00, 5.26642594e+00,
                     5.30290509e+00, 5.34060800e+00, 5.37959146e+00, 5.41990947e+00,
                     5.46162972e+00, 5.50481477e+00, 5.54953828e+00, 5.59588068e+00,
                     5.64391955e+00, 5.69374597e+00, 5.74545702e+00, 5.79915788e+00,
                     5.85496378e+00, 5.91300005e+00, 5.97339040e+00, 6.03628880e+00,
                     6.10185135e+00, 6.17025688e+00, 6.24168931e+00, 6.31636512e+00,
                     6.39451296e+00, 6.47638584e+00, 6.56227156e+00, 6.65248218e+00,
                     6.74736639e+00, 6.84731786e+00, 6.95277865e+00, 7.06423949e+00,
                     7.18226816e+00, 7.30750295e+00, 7.44067094e+00, 7.58259911e+00,
                     7.73426324e+00, 7.89678736e+00, 8.07146552e+00, 8.25985983e+00,
                     8.46379102e+00, 8.68546420e+00, 8.92752159e+00, 9.19321410e+00,
                     9.48654976e+00, 9.81253504e+00, 1.01776199e+01, 1.05901010e+01,
                     1.10610156e+01, 1.16052963e+01, 1.22439013e+01, 1.30071251e+01,
                     1.39410066e+01, 1.51193277e+01, 1.66698743e+01, 1.88391097e+01,
                     2.21842740e+01, 2.83723485e+01, 4.71199673e+01, 4.83771379e-01,
                     4.36469846e-02, 1.32397492e-02, 5.73635111e-03, 2.97048931e-03,
                     1.71488885e-03, 1.06589338e-03, 6.99271587e-04, 4.78026136e-04,
                     3.37619527e-04, 2.44873668e-04, 1.81576817e-04, 1.37185925e-04,
                     1.05328324e-04, 8.20082842e-05, 6.46412145e-05, 5.15108124e-05,
                     4.14493825e-05, 3.36471152e-05, 2.75313563e-05, 2.26906453e-05,
                     1.88252206e-05, 1.57135629e-05, 1.31899699e-05, 1.11293338e-05,
                     9.43608640e-06, 8.03650852e-06, 6.87335957e-06, 5.90177419e-06,
                     5.08631999e-06, 4.39881991e-06, 3.81679439e-06, 3.32208191e-06,
                     2.90000927e-06, 2.53864654e-06, 2.22821487e-06, 1.96068985e-06,
                     1.72940074e-06, 1.52894299e-06, 1.35469003e-06, 1.20282036e-06,
                     1.07012540e-06, 9.53907586e-07, 8.51886490e-07, 7.62130815e-07,
                     6.82999925e-07, 6.13094263e-07, 5.51217892e-07, 4.96344935e-07,
                     4.47595148e-07, 4.04209834e-07, 3.65532853e-07, 3.30998406e-07,
                     3.00114002e-07, 2.72451274e-07, 2.47638212e-07, 2.25349370e-07,
                     2.05300593e-07, 1.87242449e-07, 1.70956500e-07, 1.56249996e-07,
                     1.42953812e-07, 1.30918138e-07, 1.20011179e-07, 1.10115785e-07,
                     1.01128739e-07, 9.29573390e-08, 8.55203903e-08, 7.87450150e-08,
                     7.25659970e-08, 6.69257040e-08, 6.17724027e-08, 5.70594811e-08,
                     5.27457877e-08, 4.87938806e-08, 4.51704206e-08, 4.18453953e-08,
                     3.87915811e-08, 3.59848115e-08, 3.34030666e-08, 3.10264404e-08,
                     2.88371020e-08, 2.68187564e-08, 2.49567525e-08, 2.32378126e-08,
                     2.16498170e-08, 2.01818428e-08, 1.88239341e-08, 1.75669852e-08,
                     1.64028055e-08, 1.53238760e-08, 1.43233280e-08, 1.33949527e-08])


def expected_tr_of_complex_chain():
    """ """
    return np.array([3.16736609e-20, 3.49411800e-20, 3.85646747e-20, 4.25847343e-20,
                     4.70473911e-20, 5.20039879e-20, 5.75120587e-20, 6.36364602e-20,
                     7.04498182e-20, 7.80339854e-20, 8.64808201e-20, 9.58941115e-20,
                     1.06390247e-19, 1.18100866e-19, 1.31174474e-19, 1.45777798e-19,
                     1.62100913e-19, 1.80357064e-19, 2.00787975e-19, 2.23667351e-19,
                     2.49304663e-19, 2.78052054e-19, 3.10308762e-19, 3.46526018e-19,
                     3.87218055e-19, 4.32969924e-19, 4.84445629e-19, 5.42404709e-19,
                     6.07709123e-19, 6.81345148e-19, 7.64434652e-19, 8.58263978e-19,
                     9.64303440e-19, 1.08423015e-18, 1.21997327e-18, 1.37374240e-18,
                     1.54807296e-18, 1.74587391e-18, 1.97049645e-18, 2.22579156e-18,
                     2.51620213e-18, 2.84684929e-18, 3.22363671e-18, 3.65340579e-18,
                     4.14405086e-18, 4.70471359e-18, 5.34600494e-18, 6.08024400e-18,
                     6.92172005e-18, 7.88706803e-18, 8.99566157e-18, 1.02700949e-17,
                     1.17367104e-17, 1.34263480e-17, 1.53749825e-17, 1.76249915e-17,
                     2.02258379e-17, 2.32357497e-17, 2.67232903e-17, 3.07690385e-17,
                     3.54682679e-17, 4.09333940e-17, 4.72975802e-17, 5.47183690e-17,
                     6.33826387e-17, 7.35131686e-17, 8.53741985e-17, 9.92816710e-17,
                     1.15612767e-16, 1.34818528e-16, 1.57439581e-16, 1.84125520e-16,
                     2.15657334e-16, 2.52977087e-16, 2.97221796e-16, 3.49767098e-16,
                     4.12282726e-16, 4.86795867e-16, 5.75773228e-16, 6.82232984e-16,
                     8.09859769e-16, 9.63171187e-16, 1.14772624e-15, 1.37036780e-15,
                     1.63955706e-15, 1.96575803e-15, 2.36199634e-15, 2.84445734e-15,
                     3.43340715e-15, 4.15416756e-15, 5.03865506e-15, 6.12698067e-15,
                     7.47004098e-15, 9.13235586e-15, 1.11961241e-14, 1.37664557e-14,
                     1.69782006e-14, 2.10055380e-14, 2.60731534e-14, 3.24738721e-14,
                     4.05895750e-14, 5.09213461e-14, 6.41345522e-14, 8.10980660e-14,
                     1.02982629e-13, 1.31351938e-13, 1.68315912e-13, 2.16737207e-13,
                     2.80523298e-13, 3.65054549e-13, 4.77780456e-13, 6.29107001e-13,
                     8.33697169e-13, 1.11238960e-12, 1.49505720e-12, 2.02502464e-12,
                     2.76576823e-12, 3.81139946e-12, 5.30315483e-12, 7.45615006e-12,
                     1.06027212e-11, 1.52646992e-11, 2.22767790e-11, 3.30003094e-11,
                     4.97043511e-11, 7.62662229e-11, 1.19497791e-10, 1.91742339e-10,
                     3.16197800e-10, 5.38310215e-10, 9.51618015e-10, 1.76032261e-09,
                     3.44343278e-09, 7.23321966e-09, 1.66852552e-08, 4.38608670e-08,
                     1.40457684e-07, 6.32132342e-07, 6.12718236e-06, 5.26893738e-02,
                     9.99985767e-01, 9.99999990e-01, 9.99999252e-01, 9.99998619e-01,
                     9.99998434e-01, 9.99998510e-01, 9.99998709e-01, 9.99998956e-01,
                     9.99999188e-01, 9.99999396e-01, 9.99999573e-01, 9.99999712e-01,
                     9.99999819e-01, 9.99999897e-01, 9.99999947e-01, 9.99999979e-01,
                     9.99999995e-01, 9.99999999e-01, 9.99999995e-01, 9.99999985e-01,
                     9.99999971e-01, 9.99999957e-01, 9.99999943e-01, 9.99999930e-01,
                     9.99999918e-01, 9.99999910e-01, 9.99999903e-01, 9.99999900e-01,
                     9.99999899e-01, 9.99999899e-01, 9.99999903e-01, 9.99999908e-01,
                     9.99999915e-01, 9.99999922e-01, 9.99999931e-01, 9.99999939e-01,
                     9.99999948e-01, 9.99999956e-01, 9.99999965e-01, 9.99999972e-01,
                     9.99999979e-01, 9.99999984e-01, 9.99999989e-01, 9.99999993e-01,
                     9.99999996e-01, 9.99999998e-01, 9.99999999e-01, 9.99999999e-01,
                     9.99999998e-01, 9.99999996e-01, 9.99999994e-01, 9.99999991e-01,
                     9.99999989e-01, 9.99999985e-01, 9.99999982e-01, 9.99999979e-01,
                     9.99999975e-01, 9.99999972e-01, 9.99999969e-01, 9.99999966e-01,
                     9.99999964e-01, 9.99999962e-01, 9.99999961e-01, 9.99999960e-01,
                     9.99999960e-01, 9.99999959e-01, 9.99999960e-01, 9.99999961e-01,
                     9.99999962e-01, 9.99999964e-01, 9.99999966e-01, 9.99999968e-01,
                     9.99999971e-01, 9.99999973e-01, 9.99999976e-01, 9.99999979e-01,
                     9.99999982e-01, 9.99999984e-01, 9.99999987e-01, 9.99999989e-01,
                     9.99999991e-01, 9.99999993e-01, 9.99999994e-01, 9.99999995e-01,
                     9.99999996e-01, 9.99999996e-01, 9.99999996e-01, 9.99999995e-01,
                     9.99999994e-01, 9.99999993e-01, 9.99999991e-01, 9.99999989e-01,
                     9.99999986e-01, 9.99999983e-01, 9.99999980e-01, 9.99999976e-01,
                     9.99999972e-01, 9.99999969e-01, 9.99999965e-01, 9.99999961e-01,
                     9.99999957e-01, 9.99999953e-01, 9.99999950e-01, 9.99999947e-01,
                     9.99999944e-01, 9.99999941e-01, 9.99999939e-01, 9.99999937e-01,
                     9.99999936e-01, 9.99999936e-01, 9.99999935e-01, 9.99999936e-01,
                     9.99999938e-01, 9.99999939e-01, 9.99999942e-01, 9.99999945e-01,
                     9.99999949e-01, 9.99999953e-01, 9.99999957e-01, 9.99999962e-01,
                     9.99999968e-01, 9.99999973e-01, 9.99999978e-01, 9.99999982e-01,
                     9.99999986e-01, 9.99999988e-01, 9.99999987e-01, 9.99999982e-01,
                     9.99999970e-01, 9.99999949e-01, 9.99999913e-01, 9.99999852e-01,
                     9.99999748e-01, 9.99999568e-01, 9.99999252e-01, 9.99998639e-01,
                     9.99997317e-01, 9.99993897e-01, 9.99981090e-01, 9.99844260e-01,
                     2.41155205e-04, 1.38510879e-05, 3.85200875e-06, 1.67375602e-06,
                     9.08670378e-07, 5.66150678e-07, 3.88527329e-07, 2.87416366e-07,
                     2.27189149e-07, 1.93263053e-07, 1.84804105e-07, 2.27851862e-07,
                     4.85988331e-07, 2.60139937e-06, 1.25097397e-04, 9.99938886e-01,
                     9.99999508e-01, 9.99999695e-01, 9.99998887e-01, 9.99998595e-01,
                     9.99998620e-01, 9.99998820e-01, 9.99999094e-01, 9.99999400e-01,
                     9.99999718e-01, 1.00000007e+00, 1.00000054e+00, 1.00000131e+00,
                     1.00000310e+00, 1.00000964e+00, 1.00007901e+00, 1.99948660e+00,
                     1.99997391e+00, 1.99999319e+00, 1.99999738e+00, 1.99999880e+00,
                     1.99999940e+00, 1.99999967e+00, 1.99999979e+00, 1.99999986e+00,
                     1.99999989e+00, 1.99999990e+00, 1.99999990e+00, 1.99999990e+00,
                     1.99999990e+00, 1.99999989e+00, 1.99999989e+00, 1.99999989e+00,
                     1.99999990e+00, 1.99999990e+00, 1.99999990e+00, 1.99999991e+00,
                     1.99999992e+00, 1.99999992e+00, 1.99999993e+00, 1.99999993e+00,
                     1.99999994e+00, 1.99999994e+00, 1.99999995e+00, 1.99999995e+00,
                     1.99999995e+00, 1.99999996e+00, 1.99999996e+00, 1.99999996e+00,
                     1.99999996e+00, 1.99999996e+00, 1.99999996e+00, 1.99999996e+00,
                     1.99999996e+00, 1.99999996e+00, 1.99999996e+00, 1.99999996e+00,
                     1.99999995e+00, 1.99999995e+00, 1.99999995e+00, 1.99999995e+00,
                     1.99999995e+00, 1.99999995e+00, 1.99999995e+00, 1.99999996e+00,
                     1.99999996e+00, 1.99999996e+00, 1.99999996e+00, 1.99999996e+00,
                     1.99999996e+00, 1.99999996e+00, 1.99999996e+00, 1.99999997e+00,
                     1.99999997e+00, 1.99999997e+00, 1.99999997e+00, 1.99999997e+00,
                     1.99999997e+00, 1.99999997e+00, 1.99999997e+00, 1.99999997e+00,
                     1.99999997e+00, 1.99999997e+00, 1.99999997e+00, 1.99999997e+00,
                     1.99999996e+00, 1.99999996e+00, 1.99999996e+00, 1.99999996e+00,
                     1.99999996e+00, 1.99999996e+00, 1.99999995e+00, 1.99999995e+00,
                     1.99999995e+00, 1.99999995e+00, 1.99999995e+00, 1.99999994e+00,
                     1.99999994e+00, 1.99999994e+00, 1.99999994e+00, 1.99999994e+00,
                     1.99999994e+00, 1.99999994e+00, 1.99999994e+00, 1.99999994e+00,
                     1.99999994e+00, 1.99999994e+00, 1.99999994e+00, 1.99999994e+00,
                     1.99999993e+00, 1.99999993e+00, 1.99999993e+00, 1.99999993e+00,
                     1.99999993e+00, 1.99999993e+00, 1.99999992e+00, 1.99999992e+00,
                     1.99999992e+00, 1.99999991e+00, 1.99999991e+00, 1.99999991e+00,
                     1.99999990e+00, 1.99999990e+00, 1.99999990e+00, 1.99999990e+00,
                     1.99999989e+00, 1.99999989e+00, 1.99999989e+00, 1.99999988e+00,
                     1.99999986e+00, 1.99999984e+00, 1.99999980e+00, 1.99999972e+00,
                     1.99999958e+00, 1.99999934e+00, 1.99999887e+00, 1.99999786e+00,
                     1.99999546e+00, 1.99998769e+00, 1.99993861e+00, 1.01084896e+00,
                     1.00002609e+00, 1.00000541e+00, 1.00000198e+00, 1.00000083e+00,
                     1.00000026e+00, 9.99999879e-01, 9.99999557e-01, 9.99999257e-01,
                     9.99998971e-01, 9.99998728e-01, 9.99998592e-01, 9.99998677e-01,
                     9.99999176e-01, 1.00000005e+00, 9.99995356e-01, 9.99209374e-01,
                     1.55173083e-05, 1.17270080e-06, 3.37837256e-07, 2.10569010e-07,
                     1.97673388e-07, 2.21488341e-07, 2.71095781e-07, 3.53903752e-07,
                     4.93023547e-07, 7.42865615e-07, 1.24438662e-06, 2.44419233e-06,
                     6.34112831e-06, 3.15609704e-05, 4.84053543e-01, 9.99950233e-01,
                     9.99989979e-01, 9.99996497e-01, 9.99998480e-01, 9.99999278e-01,
                     9.99999643e-01, 9.99999825e-01, 9.99999916e-01, 9.99999964e-01,
                     9.99999988e-01, 9.99999998e-01, 1.00000000e+00, 9.99999998e-01,
                     9.99999994e-01, 9.99999989e-01, 9.99999984e-01, 9.99999978e-01,
                     9.99999973e-01, 9.99999969e-01, 9.99999965e-01, 9.99999962e-01,
                     9.99999959e-01, 9.99999957e-01, 9.99999956e-01, 9.99999955e-01,
                     9.99999954e-01, 9.99999954e-01, 9.99999954e-01, 9.99999955e-01,
                     9.99999956e-01, 9.99999957e-01, 9.99999959e-01, 9.99999961e-01,
                     9.99999963e-01, 9.99999965e-01, 9.99999968e-01, 9.99999970e-01,
                     9.99999973e-01, 9.99999975e-01, 9.99999978e-01, 9.99999981e-01,
                     9.99999983e-01, 9.99999986e-01, 9.99999988e-01, 9.99999990e-01,
                     9.99999992e-01, 9.99999994e-01, 9.99999995e-01, 9.99999996e-01,
                     9.99999997e-01, 9.99999998e-01, 9.99999998e-01, 9.99999998e-01,
                     9.99999998e-01, 9.99999997e-01, 9.99999996e-01, 9.99999995e-01,
                     9.99999994e-01, 9.99999992e-01, 9.99999990e-01, 9.99999988e-01,
                     9.99999986e-01, 9.99999983e-01, 9.99999981e-01, 9.99999978e-01,
                     9.99999976e-01, 9.99999973e-01, 9.99999971e-01, 9.99999969e-01,
                     9.99999968e-01, 9.99999966e-01, 9.99999965e-01, 9.99999964e-01,
                     9.99999963e-01, 9.99999964e-01, 9.99999964e-01, 9.99999965e-01,
                     9.99999966e-01, 9.99999968e-01, 9.99999970e-01, 9.99999972e-01,
                     9.99999975e-01, 9.99999978e-01, 9.99999980e-01, 9.99999983e-01,
                     9.99999987e-01, 9.99999989e-01, 9.99999992e-01, 9.99999995e-01,
                     9.99999997e-01, 9.99999998e-01, 9.99999999e-01, 9.99999999e-01,
                     9.99999998e-01, 9.99999997e-01, 9.99999994e-01, 9.99999991e-01,
                     9.99999987e-01, 9.99999981e-01, 9.99999975e-01, 9.99999968e-01,
                     9.99999961e-01, 9.99999953e-01, 9.99999944e-01, 9.99999936e-01,
                     9.99999928e-01, 9.99999921e-01, 9.99999914e-01, 9.99999908e-01,
                     9.99999905e-01, 9.99999902e-01, 9.99999902e-01, 9.99999904e-01,
                     9.99999909e-01, 9.99999917e-01, 9.99999926e-01, 9.99999938e-01,
                     9.99999952e-01, 9.99999966e-01, 9.99999979e-01, 9.99999991e-01,
                     9.99999998e-01, 9.99999999e-01, 9.99999990e-01, 9.99999967e-01,
                     9.99999927e-01, 9.99999865e-01, 9.99999776e-01, 9.99999656e-01,
                     9.99999503e-01, 9.99999314e-01, 9.99999093e-01, 9.99998857e-01,
                     9.99998632e-01, 9.99998476e-01, 9.99998478e-01, 9.99998835e-01,
                     9.99999655e-01, 9.99999051e-01, 9.99914384e-01, 7.93471156e-05,
                     2.14044760e-06, 3.31495072e-07, 8.69401490e-08, 2.97975648e-08,
                     1.20460204e-08, 5.45340313e-09, 2.68402889e-09, 1.40844259e-09,
                     7.77864590e-10, 4.47985569e-10, 2.67209112e-10, 1.64208038e-10,
                     1.03543248e-10, 6.67759783e-11, 4.39276571e-11, 2.94126240e-11,
                     2.00083190e-11, 1.38072491e-11, 9.65281988e-12, 6.82903298e-12,
                     4.88428519e-12, 3.52865470e-12, 2.57307287e-12, 1.89254095e-12,
                     1.40324785e-12, 1.04830054e-12, 7.88671503e-13, 5.97284320e-13,
                     4.55167759e-13, 3.48907643e-13, 2.68947893e-13, 2.08406231e-13,
                     1.62301274e-13, 1.26998461e-13, 9.98247787e-14, 7.88046678e-14,
                     6.24643796e-14, 4.97103598e-14, 3.97098378e-14, 3.18360256e-14,
                     2.56120414e-14, 2.06735794e-14, 1.67408376e-14, 1.35980408e-14,
                     1.10781012e-14, 9.05101718e-15, 7.41531294e-15, 6.09142778e-15,
                     5.01682000e-15, 4.14210645e-15, 3.42815728e-15, 2.84391806e-15,
                     2.36460204e-15, 1.97038935e-15, 1.64540203e-15, 1.37686343e-15,
                     1.15447206e-15, 9.69894410e-16, 8.16379564e-16, 6.88433968e-16,
                     5.81587590e-16, 4.92184894e-16, 4.17238380e-16, 3.54292889e-16,
                     3.01334390e-16, 2.56696013e-16, 2.19009457e-16, 1.87138585e-16,
                     1.60140674e-16, 1.37235445e-16, 1.17772555e-16, 1.01208440e-16,
                     8.70916183e-17, 7.50424657e-17, 6.47437573e-17, 5.59290988e-17,
                     4.83739660e-17, 4.18902110e-17, 3.63186002e-17, 3.15245416e-17,
                     2.73945846e-17, 2.38322339e-17, 2.07558423e-17, 1.80960470e-17,
                     1.57937045e-17, 1.37985812e-17, 1.20677614e-17, 1.05645392e-17,
                     9.25765250e-18, 8.12025597e-18, 7.12932735e-18, 6.26516641e-18])


def expected_dens_of_complex_chain():
    """ """
    return np.array([1.75500165e-009, 1.94141418e-009, 2.14990898e-009, 2.38339649e-009,
                     2.64519687e-009, 2.93912081e-009, 3.26954760e-009, 3.64151425e-009,
                     4.06082037e-009, 4.53417774e-009, 5.06932795e-009, 5.67527028e-009,
                     6.36242582e-009, 7.14292665e-009, 8.03093462e-009, 9.04297357e-009,
                     1.01983954e-008, 1.15198875e-008, 1.30341863e-008, 1.47727622e-008,
                     1.67728342e-008, 1.90785085e-008, 2.17422028e-008, 2.48262874e-008,
                     2.84054184e-008, 3.25689791e-008, 3.74244807e-008, 4.31015088e-008,
                     4.97570867e-008, 5.75816394e-008, 6.68074352e-008, 7.77188556e-008,
                     9.06650001e-008, 1.06076790e-007, 1.24488153e-007, 1.46564013e-007,
                     1.73136072e-007, 2.05250736e-007, 2.44230643e-007, 2.91758057e-007,
                     3.49986954e-007, 4.21689175e-007, 5.10461322e-007, 6.21000762e-007,
                     7.59494048e-007, 9.34153710e-007, 1.15597991e-006, 1.43986696e-006,
                     1.80616052e-006, 2.28301799e-006, 2.90984109e-006, 3.74268001e-006,
                     4.86197366e-006, 6.38580905e-006, 8.49027041e-006, 1.14434327e-005,
                     1.56628121e-005, 2.18162973e-005, 3.10041388e-005, 4.51035478e-005,
                     6.74498688e-005, 1.04263885e-004, 1.67855151e-004, 2.84429077e-004,
                     5.15275607e-004, 1.02287333e-003, 2.32132609e-003, 6.55010839e-003,
                     2.83190315e-002, 4.59070221e-001, 1.99961429e+003, 1.21710838e+003,
                     9.00438887e+002, 7.69336566e+002, 6.69870620e+002, 6.92710984e+002,
                     5.76315634e+002, 5.43499881e+002, 4.96997077e+002, 6.99191021e+002,
                     5.08181800e+002, 4.63674262e+002, 4.40442012e+002, 4.63671439e+002,
                     1.39075718e+003, 5.02284914e+002, 4.30857267e+002, 4.19692744e+002,
                     4.59899959e+002, 6.80429140e+002, 6.38738539e+002, 4.63767717e+002,
                     4.28272780e+002, 4.34320853e+002, 4.80131962e+002, 5.28126425e+002,
                     4.99130437e+002, 4.53548744e+002, 4.30944804e+002, 4.25164662e+002,
                     4.31358504e+002, 4.37788901e+002, 4.33299091e+002, 4.22603427e+002,
                     4.13508902e+002, 4.06942800e+002, 4.02532003e+002, 3.99989688e+002,
                     3.97475807e+002, 3.93833789e+002, 3.89969678e+002, 3.86696447e+002,
                     3.83810557e+002, 3.81153362e+002, 3.78703105e+002, 3.76248751e+002,
                     3.74048156e+002, 3.72680238e+002, 3.71751134e+002, 3.70195210e+002,
                     3.68076650e+002, 3.66377483e+002, 3.65080844e+002, 3.63781900e+002,
                     3.63929224e+002, 3.67430118e+002, 3.72314717e+002, 3.73530304e+002,
                     3.70544627e+002, 3.67920022e+002, 3.66397797e+002, 3.63765728e+002,
                     3.65940850e+002, 3.89282704e+002, 4.37064506e+002, 4.58767266e+002,
                     2.72352713e+002, 2.28651475e+002, 2.26328063e+002, 1.74131265e+002,
                     5.51658825e-001, 6.93619199e-002, 3.55426644e-001, 1.13364217e-001,
                     1.10480516e-002, 1.06030553e-002, 3.35495520e-002, 2.32996167e+000,
                     6.68369838e+002, 5.38996085e+002, 4.07754574e+002, 3.81878915e+002,
                     3.15578140e+002, 4.81880946e+002, 2.91053669e+002, 2.61928287e+002,
                     2.48182111e-002, 3.53095426e-008, 2.71134307e-015, 9.38055579e-022,
                     2.23816545e-028, 7.43916543e-035, 4.06771143e-041, 7.40209084e-048,
                     2.31152234e-054, 7.73469385e-061, 2.83472754e-067, 1.16327180e-073,
                     2.95071233e-080, 8.27469784e-087, 2.59838502e-093, 8.69263195e-100,
                     3.04352280e-106, 1.02651344e-112, 3.21132001e-119, 9.95483599e-126,
                     3.12004549e-132, 1.00225006e-138, 3.27951099e-145, 1.06683642e-151,
                     3.43603462e-158, 1.10956632e-164, 3.59334144e-171, 1.16038344e-177,
                     3.74437856e-184, 1.21097958e-190, 3.91613914e-197, 1.26553721e-203,
                     4.10064276e-210, 1.33354390e-216, 4.33776174e-223, 1.40823401e-229,
                     4.56933539e-236, 1.48394686e-242, 4.82488438e-249, 1.57120391e-255,
                     5.12103448e-262, 1.66821902e-268, 5.43547394e-275, 1.77268868e-281,
                     5.77933879e-288, 1.88618611e-294, 6.18372041e-301, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000, 0.00000000e+000, 0.00000000e+000,
                     0.00000000e+000, 0.00000000e+000])


def test_double_barrier_density_recursive(single_period_test=complex_chain, periods=20):
    """

    Parameters
    ----------
    single_period_test :
         (Default value = complex_chain)
    periods :
         (Default value = 20)

    Returns
    -------

    """
    from nanonet.negf.field import Field1D

    def qw(coord, coords_of_steps, jumps, width=1):
        """

        Parameters
        ----------
        coord :
            
        coords_of_steps :
            
        jumps :
            
        width :
             (Default value = 1)

        Returns
        -------

        """
        ans = 0
        for j, item in enumerate(coords_of_steps):
            ans += jumps[j] * 0.5 * (np.tanh(coord - item) / width + 1.0)

        return ans

    def z_dependence(coord):
        """

        Parameters
        ----------
        coord :
            

        Returns
        -------

        """
        coords_of_steps = [-15.0, -11.0, 11.0, 15.0]
        jumps = [-1.7, 1.7, -1.7, 1.7]

        return qw(coord, coords_of_steps, jumps)

    f = Field1D(z_dependence, axis=2)

    ef1 = 0.25 - 1.5
    ef2 = 0.5 - 1.5
    tempr = 10

    energy, dos, tr, h, sgf_l, sgf_r = single_period_test()
    h_l, h_0, h_r = h.get_hamiltonians()
    cell = h.ct.pcv

    h_chain = HamiltonianChain(h_l, h_0, h_r, h.get_site_coordinates())
    h_chain.translate(cell[0], periods, periods)
    h_chain.add_field(f)

    num_periods = 2 * periods + 1

    dos1 = np.zeros((energy.shape[0]))
    tr = np.zeros((energy.shape[0]))
    dens = np.zeros((energy.shape[0], num_periods))

    for j, E in enumerate(energy):

        h_chain.add_self_energies(sgf_l[j, :, :], sgf_r[j, :, :], energy=E, tempr=tempr, ef1=ef1, ef2=ef2)
        g_trans, grd, grl, gru, gr_left, gnd, gnl, gnu, gn_left = recursive_gf(E,
                                                                               h_chain.h_l,
                                                                               h_chain.h_0,
                                                                               h_chain.h_r,
                                                                               s_in=h_chain.sgf)
        h_chain.remove_self_energies()

        gamma_l = 1j * (sgf_l[j, :, :] - sgf_l[j, :, :].conj().T)
        gamma_r = 1j * (sgf_r[j, :, :] - sgf_r[j, :, :].conj().T)

        tr[j] = np.real(np.trace(gamma_l.dot(g_trans).dot(gamma_r).dot(g_trans.conj().T)))

        for jj in range(num_periods):
            dos1[j] = dos1[j] + np.real(np.trace(1j * (grd[jj] - grd[jj].conj().T))) / num_periods
            dens[j, jj] = 2 * np.trace(gnd[jj])

    np.testing.assert_allclose(np.sum(dens, axis=1)[::2], expected_dens_of_complex_chain(), rtol=1e-2)


if __name__ == '__main__':
    # test_double_barrier_density_recursive()
    # test_gf_complex_chain_several_periods()
    # test_gf_single_atom_chain_several_periods()
    # test_double_barrier_density_recursive(complex_chain, 20)
    # test_complex_chain()
    test_gf_complex_chain_several_periods_recursive()
    # test_single_atom_chain()
