# prefect-hex

Visit the full docs [here](https://PrefectHQ.github.io/prefect-hex) to see additional examples and the API reference.

<p align="center">
    <a href="https://pypi.python.org/pypi/prefect-hex/" alt="PyPI version">
        <img alt="PyPI" src="https://img.shields.io/pypi/v/prefect-hex?color=0052FF&labelColor=090422"></a>
    <a href="https://github.com/PrefectHQ/prefect-hex/" alt="Stars">
        <img src="https://img.shields.io/github/stars/PrefectHQ/prefect-hex?color=0052FF&labelColor=090422" /></a>
    <a href="https://pepy.tech/badge/prefect-hex/" alt="Downloads">
        <img src="https://img.shields.io/pypi/dm/prefect-hex?color=0052FF&labelColor=090422" /></a>
    <a href="https://github.com/PrefectHQ/prefect-hex/pulse" alt="Activity">
        <img src="https://img.shields.io/github/commit-activity/m/PrefectHQ/prefect-hex?color=0052FF&labelColor=090422" /></a>
    <br>
    <a href="https://prefect-community.slack.com" alt="Slack">
        <img src="https://img.shields.io/badge/slack-join_community-red.svg?color=0052FF&labelColor=090422&logo=slack" /></a>
    <a href="https://discourse.prefect.io/" alt="Discourse">
        <img src="https://img.shields.io/badge/discourse-browse_forum-red.svg?color=0052FF&labelColor=090422&logo=discourse" /></a>
</p>

## Welcome!

Prefect integrations for interacting with Hex. 

Hex is a powerful platform for collaborative data science and analytics. For information getting started with Hex, check out Hex's [quickstart guide](https://learn.hex.tech/quickstart).

The tasks within this collection were created by a code generator using Hex's OpenAPI spec.

Hex's REST API documentation can be found [here](https://learn.hex.tech/docs/develop-logic/hex-api/api-reference).

## Getting Started

### Python setup

Requires an installation of Python 3.7+.

We recommend using a Python virtual environment manager such as pipenv, conda or virtualenv.

These tasks are designed to work with Prefect 2.0. For more information about how to use Prefect, please refer to the [Prefect documentation](https://orion-docs.prefect.io/).

### Installation

Install `prefect-hex` with `pip`:

```bash
pip install prefect-hex
```

A list of available blocks in `prefect-hex` and their setup instructions can be found [here](https://PrefectHQ.github.io/prefect-hex/#blocks-catalog).


### Gather and store authentication

1. Create new token on https://app.hex.tech/ Settings page:

![image](https://user-images.githubusercontent.com/15331990/201996947-07765380-50c4-4c61-9044-bd93e4b8efc7.png)

2. Store token on https://app.prefect.cloud/ Blocks page:

![image](https://user-images.githubusercontent.com/15331990/201997292-b3a18254-229f-4689-aaec-07a990cdaf87.png)

3. Copy project ID from browser URL (in red):

![image](https://user-images.githubusercontent.com/15331990/202002588-55a895b2-de89-438f-ac96-c86940946336.png)

### Write and run a flow

#### Trigger a Hex project run and wait for completion
```python
from prefect import flow
from prefect_hex import HexCredentials
from prefect_hex.project import trigger_project_run_and_wait_for_completion

@flow
def trigger_project_run_and_wait_for_completion_flow(project_id: str):
    hex_credentials = HexCredentials.load("hex-token")
    project_metadata = trigger_project_run_and_wait_for_completion(
        project_id=project_id,
        hex_credentials=hex_credentials
    )
    return project_metadata

trigger_project_run_and_wait_for_completion_flow(
    project_id="012345c6-b67c-1234-1b2c-66e4ad07b9f3"
)
```

#### Run project, get status, cancel run, and get list of projects
```python
from prefect import flow
from prefect_hex import HexCredentials
from prefect_hex.project import (
    get_project_runs,
    run_project,
    get_run_status,
    cancel_run,
)

@flow
def example_hex_flow():
    # load stored credentials
    hex_credentials = HexCredentials.load("hex-token")

    # run project
    project_id='5a8591dd-4039-49df-9202-96385ba3eff8',
    project_run = run_project(project_id=project_id, hex_credentials=hex_credentials)

    # get status
    run_id = project_run.run_id
    project_run_status = get_run_status(
        project_id=project_id, run_id=run_id, hex_credentials=hex_credentials
    )
    print(project_run_status.run_url)

    # cancel run if needed
    cancel_run(project_id=project_id, run_id=run_id, hex_credentials=hex_credentials)

    # get list of project runs
    project_runs = get_project_runs(
        project_id=project_id, hex_credentials=hex_credentials
    )

    return project_runs

example_hex_flow()
```

For more tips on how to use tasks and flows in a Collection, check out [Using Collections](https://orion-docs.prefect.io/collections/usage/)!

## Resources

### Blog Posts

- [Create Observable and Reproducible Notebooks with Hex](https://towardsdatascience.com/create-observable-and-reproducible-notebooks-with-hex-460e75818a09) by Khuyen Tran

### Videos

- [Create Observable and Reproducible Notebooks with Hex: Why Hex (Part 1)](https://youtu.be/_BjqCrun4nE)

If you encounter any bugs while using `prefect-hex`, feel free to open an issue in the [prefect-hex](https://github.com/PrefectHQ/prefect-hex) repository.

If you have any questions or issues while using `prefect-hex`, you can find help in either the [Prefect Discourse forum](https://discourse.prefect.io/) or the [Prefect Slack community](https://prefect.io/slack).

Feel free to star or watch [`prefect-hex`](https://github.com/PrefectHQ/prefect-hex) for updates too!

## Contributing
 
-```bash
-git clone https://github.com/PrefectHQ/prefect-hex.git
If you'd like to help contribute to fix an issue or add a feature to `prefect-hex`, please [propose changes through a pull request from a fork of the repository](https://docs.github.com/en/pull-requests/collaborating-with-pull-requests/proposing-changes-to-your-work-with-pull-requests/creating-a-pull-request-from-a-fork).
 
-cd prefect-hex/
Here are the steps:
 
1. [Fork the repository](https://docs.github.com/en/get-started/quickstart/fork-a-repo#forking-a-repository)
2. [Clone the forked repository](https://docs.github.com/en/get-started/quickstart/fork-a-repo#cloning-your-forked-repository)
3. Install the repository and its dependencies:
```
 pip install -e ".[dev]"
```
4. Make desired changes
5. Add tests
6. Insert an entry to [CHANGELOG.md](https://github.com/PrefectHQ/prefect-hex/blob/main/CHANGELOG.md)
7. Install `pre-commit` to perform quality checks prior to commit:
```
 pre-commit install
 ```
8. `git commit`, `git push`, and create a pull request install