# Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
# SPDX-License-Identifier: Apache-2.0

import json
import logging
import os
import uuid

import boto3

logger = logging.getLogger(__name__)

s3 = boto3.client('s3')

def lambda_handler(event, context):
    try:
        log_level = os.environ.get('LOG_LEVEL', 'INFO')
        textract_endpoint_url = os.environ.get('TEXTRACT_ENDPOINT_URL', None)

        output_bucket = os.environ.get('OUTPUT_BUCKET')
        output_bucket_prefix = os.environ.get('OUTPUT_BUCKET_PREFIX')

        logger.setLevel(log_level)
        logger.info(json.dumps(event))
        logger.info(f"LOG_LEVEL: {log_level} \n \
                    OUTPUT_BUCKET: {output_bucket} \n \
                    OUTPUT_BUCKET_PREFIX: {output_bucket_prefix} \n \
                    TEXTRACT_ENDPOINT_URL: {textract_endpoint_url}")

        if textract_endpoint_url:
            textract = boto3.client("textract",
                                    endpoint_url=textract_endpoint_url)
        else:
            textract = boto3.client("textract")

        s3_bucket = event['s3_bucket']
        s3_key = event['s3_key']
        test_run_id = ""
        if "test_run_id" in event:
            test_run_id = event['test_run_id']

        supported_suffixes = ['.png', '.jpg', '.jpeg']

        # May want to add actual check of the file itself
        if s3_key:
            s3_filename, ext = os.path.splitext(s3_key)
            if ext not in supported_suffixes:
                raise Exception(f"not supported suffix: {ext}")
        else:
            raise ValueError(f"no key set")

        # execution_id = event['ExecutionId']

        # logger.info(f"s3_bucket: {s3_bucket} \n \
        #             s3_key: {s3_key} \n \
        #             execution_id: {execution_id}")

        uuid_key = str(uuid.uuid4())
        logger.debug(f"uuid_key: {uuid_key}")
        textract_results = textract.analyze_document(
            Document={'S3Object': {
                'Bucket': s3_bucket,
                'Name': s3_key
            }},
            FeatureTypes=['FORMS', 'TABLES'],
        )
        if test_run_id:
            output_bucket_key = output_bucket_prefix + "/" + test_run_id + "/" + s3_filename + ".json"
        else:
            output_bucket_key = output_bucket_prefix + "/" + s3_filename + ".json"
        s3.put_object(Body=bytes(
            json.dumps(textract_results, indent=4).encode('UTF-8')),
                      Bucket=output_bucket,
                      Key=output_bucket_key)
        logger.debug(f"Textract-response: {textract_results}")
        result_location = output_bucket + "/" + output_bucket_key
        return {
            "textract_result_location": result_location,
            "textract_endpoint_url": textract_endpoint_url
        }
    except Exception as e:
        logger.error(e)
        raise ValueError(e)
