"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractStepFunctionsStartExecution = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
/**
 * Verifies that a validation pattern is supported for a service integration
 *
 */
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` +
        (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * A Step Functions Task to call StartExecution on another state machine.
 *
 * It supports three service integration patterns: REQUEST_RESPONSE, RUN_JOB, and WAIT_FOR_TASK_TOKEN.
 */
class TextractStepFunctionsStartExecution extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractStepFunctionsStartExecution.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        // need DynamoDB table for taskToken
        new dynamodb.Table(this, 'TextractTaskTokenTable', {
            partitionKey: { name: 'ID', type: dynamodb.AttributeType.STRING },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            timeToLiveAttribute: 'ttltimestamp',
        });
        const workflow_chain = sfn.Chain.start(new sfn.Pass(this, 'PassState'));
        const myFunction = new lambda.Function(this, 'SampleFunction', {
            runtime: lambda.Runtime.PYTHON_3_9,
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/sync/')),
            memorySize: 2048,
            handler: 'sync_main.lambda_handler',
            environment: {
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
            },
        });
        // single page sync call
        const myStep = new tasks.LambdaInvoke(this, 'simple sample', {
            lambdaFunction: myFunction,
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            outputPath: '$.Payload',
        });
        const wait1Second = new sfn.Wait(this, 'Wait 1 Second', {
            time: sfn.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(1)),
        });
        workflow_chain.next(myStep.next(wait1Second));
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            stateMachineName: 'bla-somename',
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
     * @internal
     */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? { ...this.props.input.value, ...associateWithParentEntry } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
     * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
     *
     * the scoped access policy should be generated accordingly.
     *
     * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
     */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['states:DescribeExecution', 'states:StopExecution'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractStepFunctionsStartExecution = TextractStepFunctionsStartExecution;
_a = JSII_RTTI_SYMBOL_1;
TextractStepFunctionsStartExecution[_a] = { fqn: "schadem-cdk-construct-sfn-test.TextractStepFunctionsStartExecution", version: "0.0.5" };
TextractStepFunctionsStartExecution.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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