import { aws_iam, aws_kms, aws_lambda, ITaggable, ResourceProps, TagManager } from 'aws-cdk-lib';
import { Construct } from 'constructs';
export declare enum PublicKeyFormat {
    OPENSSH = "OPENSSH",
    PEM = "PEM"
}
/**
 * Definition of EC2 Key Pair
 */
export interface KeyPairProps extends ResourceProps {
    /**
     * Name of the Key Pair
     *
     * In AWS Secrets Manager the key will be prefixed with `ec2-ssh-key/`.
     *
     * The name can be up to 255 characters long. Valid characters include _, -, a-z, A-Z, and 0-9.
     */
    readonly name: string;
    /**
     * The description for the key in AWS Secrets Manager
     * @default - ''
     */
    readonly description?: string;
    /**
     * The KMS key used to encrypt the Secrets Manager secrets with
     *
     * This needs to be a key created in the same stack. You cannot use a key imported via ARN, because the keys access policy will need to be modified.
     *
     * @default - `alias/aws/secretsmanager`
     */
    readonly kms?: aws_kms.Key;
    /**
     * The KMS key to use to encrypt the private key with
     *
     * This needs to be a key created in the same stack. You cannot use a key imported via ARN, because the keys access policy will need to be modified.
     *
     * If no value is provided, the property `kms` will be used instead.
     *
     * @default - `this.kms`
     */
    readonly kmsPrivateKey?: aws_kms.Key;
    /**
     * The KMS key to use to encrypt the public key with
     *
     * This needs to be a key created in the same stack. You cannot use a key imported via ARN, because the keys access policy will need to be modified.
     *
     * If no value is provided, the property `kms` will be used instead.
     *
     * @default - `this.kms`
     */
    readonly kmsPublicKey?: aws_kms.Key;
    /**
     * Import a public key instead of creating it
     *
     * If no public key is provided, a new key pair will be created.
     */
    readonly publicKey?: string;
    /**
     * Store the public key as a secret
     *
     * @default - false
     */
    readonly storePublicKey?: boolean;
    /**
     * Expose the public key as property `publicKeyValue`
     *
     * @default - false
     */
    readonly exposePublicKey?: boolean;
    /**
     * Format for public key.
     *
     * Relevant only if the public key is stored and/or exposed.
     *
     * @default - OPENSSH
     */
    readonly publicKeyFormat?: PublicKeyFormat;
    /**
     * When the resource is destroyed, after how many days the private and public key in the AWS Secrets Manager should be deleted.
     *
     * Valid values are 0 and 7 to 30
     *
     * @default 0
     */
    readonly removeKeySecretsAfterDays?: number;
    /**
     * Prefix for the secret in AWS Secrets Manager.
     *
     * @default `ec2-ssh-key/`
     */
    readonly secretPrefix?: string;
    /**
     * A prefix for all resource names.
     *
     * By default all resources are prefixed with the stack name to avoid collisions with other stacks. This might cause problems when you work with long stack names and can be overridden through this parameter.
     *
     * @default Name of the stack
     */
    readonly resourcePrefix?: string;
}
/**
 * An EC2 Key Pair
 */
export declare class KeyPair extends Construct implements ITaggable {
    /**
     * The lambda function that is created
     */
    readonly lambda: aws_lambda.IFunction;
    /**
     * ARN of the private key in AWS Secrets Manager
     */
    readonly privateKeyArn: string;
    /**
     * ARN of the public key in AWS Secrets Manager
     */
    readonly publicKeyArn: string;
    /**
     * The public key.
     *
     * Only filled, when `exposePublicKey = true`
     */
    readonly publicKeyValue: string;
    /**
     * Name of the Key Pair
     */
    readonly keyPairName: string;
    /**
     * ID of the Key Pair
     */
    readonly keyPairID: string;
    /**
     * Resource tags
     */
    readonly tags: TagManager;
    readonly prefix: string;
    /**
     * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
     */
    constructor(scope: Construct, id: string, props: KeyPairProps);
    private ensureLambda;
    /**
     * Grants read access to the private key in AWS Secrets Manager
     */
    grantReadOnPrivateKey(grantee: aws_iam.IGrantable): aws_iam.Grant;
    /**
     * Grants read access to the public key in AWS Secrets Manager
     */
    grantReadOnPublicKey(grantee: aws_iam.IGrantable): aws_iam.Grant;
    private grantRead;
}
