import { Dialog, showDialog, showErrorMessage } from '@jupyterlab/apputils';
import List from '@material-ui/core/List';
import ListItem from '@material-ui/core/ListItem';
import ClearIcon from '@material-ui/icons/Clear';
import * as React from 'react';
import { classes } from 'typestyle';
import { activeListItemClass, filterClass, filterClearClass, filterInputClass, filterWrapperClass, listItemClass, listItemIconClass, listWrapperClass, newBranchButtonClass, wrapperClass } from '../style/BranchMenu';
import { sleep } from '../utils';
import { Alert } from './Alert';
import { NewBranchDialog } from './NewBranchDialog';
import { SuspendModal } from './SuspendModal';
const CHANGES_ERR_MSG = 'The current branch contains files with uncommitted changes. Please commit or discard these changes before switching to or creating another branch.';
/**
 * Callback invoked upon encountering an error when switching branches.
 *
 * @private
 * @param err - error
 */
function onBranchError(err) {
    if (err.message.includes('following files would be overwritten')) {
        showDialog({
            title: 'Unable to switch branch',
            body: (React.createElement(React.Fragment, null,
                React.createElement("p", null, "Your changes to the following files would be overwritten by switching:"),
                React.createElement(List, null, err.message
                    .split('\n')
                    .slice(1, -3)
                    .map(renderFileName)),
                React.createElement("span", null, "Please commit, stash, or discard your changes before you switch branches."))),
            buttons: [Dialog.okButton({ label: 'Dismiss' })]
        });
    }
    else {
        showErrorMessage('Error switching branch', err.message);
    }
}
/**
 * Renders a file name.
 *
 * @private
 * @param filename - file name
 * @returns React element
 */
function renderFileName(filename) {
    return React.createElement(ListItem, { key: filename }, filename);
}
/**
 * React component for rendering a branch menu.
 */
export class BranchMenu extends React.Component {
    /**
     * Returns a React component for rendering a branch menu.
     *
     * @param props - component properties
     * @returns React component
     */
    constructor(props) {
        super(props);
        /**
         * Callback invoked upon a change to the menu filter.
         *
         * @param event - event object
         */
        this._onFilterChange = (event) => {
            this.setState({
                filter: event.target.value
            });
        };
        /**
         * Callback invoked to reset the menu filter.
         */
        this._resetFilter = () => {
            this.setState({
                filter: ''
            });
        };
        /**
         * Callback invoked upon clicking a button to create a new branch.
         *
         * @param event - event object
         */
        this._onNewBranchClick = () => {
            if (!this.props.branching) {
                showErrorMessage('Creating a new branch is disabled', CHANGES_ERR_MSG);
                return;
            }
            this.setState({
                branchDialog: true
            });
        };
        /**
         * Callback invoked upon closing a dialog to create a new branch.
         */
        this._onNewBranchDialogClose = () => {
            this.setState({
                branchDialog: false
            });
        };
        /**
         * Callback invoked upon clicking on the feedback modal.
         *
         * @param event - event object
         */
        this._onFeedbackModalClick = () => {
            this._suspend(false);
        };
        /**
         * Callback invoked upon closing a feedback alert.
         *
         * @param event - event object
         */
        this._onFeedbackAlertClose = () => {
            this.setState({
                alert: false
            });
        };
        const repo = this.props.model.pathRepository;
        this.state = {
            filter: '',
            branchDialog: false,
            current: repo ? this.props.model.currentBranch.name : '',
            branches: repo ? this.props.model.branches : [],
            suspend: false,
            alert: false,
            log: {
                severity: 'info',
                message: ''
            }
        };
    }
    /**
     * Callback invoked immediately after mounting a component (i.e., inserting into a tree).
     */
    componentDidMount() {
        this._addListeners();
    }
    /**
     * Callback invoked when a component will no longer be mounted.
     */
    componentWillUnmount() {
        this._removeListeners();
    }
    /**
     * Renders the component.
     *
     * @returns React element
     */
    render() {
        return (React.createElement("div", { className: wrapperClass },
            this._renderFilter(),
            this._renderBranchList(),
            this._renderNewBranchDialog(),
            this._renderFeedback()));
    }
    /**
     * Renders a branch input filter.
     *
     * @returns React element
     */
    _renderFilter() {
        return (React.createElement("div", { className: filterWrapperClass },
            React.createElement("div", { className: filterClass },
                React.createElement("input", { className: filterInputClass, type: "text", onChange: this._onFilterChange, value: this.state.filter, placeholder: "Filter", title: "Filter branch menu" }),
                this.state.filter ? (React.createElement("button", { className: filterClearClass },
                    React.createElement(ClearIcon, { titleAccess: "Clear the current filter", fontSize: "small", onClick: this._resetFilter }))) : null),
            React.createElement("input", { className: newBranchButtonClass, type: "button", title: "Create a new branch", value: "New Branch", onClick: this._onNewBranchClick })));
    }
    /**
     * Renders a
     *
     * @returns React element
     */
    _renderBranchList() {
        return (React.createElement("div", { className: listWrapperClass },
            React.createElement(List, { disablePadding: true }, this._renderItems())));
    }
    /**
     * Renders menu items.
     *
     * @returns array of React elements
     */
    _renderItems() {
        return this.state.branches.map(this._renderItem, this);
    }
    /**
     * Renders a menu item.
     *
     * @param branch - branch
     * @param idx - item index
     * @returns React element
     */
    _renderItem(branch, idx) {
        // Perform a "simple" filter... (TODO: consider implementing fuzzy filtering)
        if (this.state.filter && !branch.name.includes(this.state.filter)) {
            return null;
        }
        const isActive = branch.name === this.state.current;
        return (React.createElement(ListItem, { button: true, title: `Switch to branch: ${branch.name}`, className: classes(listItemClass, isActive ? activeListItemClass : null), key: branch.name, onClick: this._onBranchClickFactory(branch.name) },
            React.createElement("span", { className: classes('jp-git-icon', listItemIconClass, isActive && 'jp-git-selected') }),
            branch.name));
    }
    /**
     * Renders a dialog for creating a new branch.
     *
     * @returns React element
     */
    _renderNewBranchDialog() {
        return (React.createElement(NewBranchDialog, { open: this.state.branchDialog, suspend: this.props.suspend, model: this.props.model, onClose: this._onNewBranchDialogClose }));
    }
    /**
     * Renders a component to provide UI feedback.
     *
     * @returns React element
     */
    _renderFeedback() {
        return (React.createElement(React.Fragment, null,
            React.createElement(SuspendModal, { open: this.props.suspend && this.state.suspend, onClick: this._onFeedbackModalClick }),
            React.createElement(Alert, { open: this.state.alert, message: this.state.log.message, severity: this.state.log.severity, onClose: this._onFeedbackAlertClose })));
    }
    /**
     * Adds model listeners.
     */
    _addListeners() {
        // When the HEAD changes, decent probability that we've switched branches:
        this.props.model.headChanged.connect(this._syncState, this);
        // When the status changes, we may have checked out a new branch (e.g., via the command-line and not via the extension) or changed repositories:
        this.props.model.statusChanged.connect(this._syncState, this);
    }
    /**
     * Removes model listeners.
     */
    _removeListeners() {
        this.props.model.headChanged.disconnect(this._syncState, this);
        this.props.model.statusChanged.disconnect(this._syncState, this);
    }
    /**
     * Syncs the component state with the underlying model.
     */
    _syncState() {
        const repo = this.props.model.pathRepository;
        this.setState({
            current: repo ? this.props.model.currentBranch.name : '',
            branches: repo ? this.props.model.branches : []
        });
    }
    /**
     * Sets the suspension state.
     *
     * @param bool - boolean indicating whether to suspend UI interaction
     */
    _suspend(bool) {
        if (this.props.suspend) {
            this.setState({
                suspend: bool
            });
        }
    }
    /**
     * Sets the current component log message.
     *
     * @param msg - log message
     */
    _log(msg) {
        this.setState({
            alert: true,
            log: msg
        });
    }
    /**
     * Returns a callback which is invoked upon clicking a branch name.
     *
     * @param branch - branch name
     * @returns callback
     */
    _onBranchClickFactory(branch) {
        const self = this;
        return onClick;
        /**
         * Callback invoked upon clicking a branch name.
         *
         * @private
         * @param event - event object
         * @returns promise which resolves upon attempting to switch branches
         */
        async function onClick() {
            if (!self.props.branching) {
                showErrorMessage('Switching branches is disabled', CHANGES_ERR_MSG);
                return;
            }
            const opts = {
                branchname: branch
            };
            self._log({
                severity: 'info',
                message: 'Switching branch...'
            });
            self._suspend(true);
            let result;
            try {
                result = await Promise.all([
                    sleep(1000),
                    self.props.model.checkout(opts)
                ]);
            }
            catch (err) {
                self._suspend(false);
                self._log({
                    severity: 'error',
                    message: 'Failed to switch branch.'
                });
                return onBranchError(err);
            }
            self._suspend(false);
            const res = result[1];
            if (res.code !== 0) {
                self._log({
                    severity: 'error',
                    message: 'Failed to switch branch.'
                });
                showErrorMessage('Error switching branch', res.message);
                return;
            }
            self._log({
                severity: 'success',
                message: 'Switched branch.'
            });
        }
    }
}
//# sourceMappingURL=BranchMenu.js.map