import * as React from 'react';
import { classes } from 'typestyle';
import List from '@material-ui/core/List';
import ListItem from '@material-ui/core/ListItem';
import Dialog from '@material-ui/core/Dialog';
import DialogActions from '@material-ui/core/DialogActions';
import ClearIcon from '@material-ui/icons/Clear';
import { sleep } from '../utils';
import { actionsWrapperClass, activeListItemClass, branchDialogClass, buttonClass, cancelButtonClass, closeButtonClass, contentWrapperClass, createButtonClass, errorMessageClass, filterClass, filterClearClass, filterInputClass, filterWrapperClass, listItemBoldTitleClass, listItemClass, listItemContentClass, listItemDescClass, listItemIconClass, listItemTitleClass, listWrapperClass, nameInputClass, titleClass, titleWrapperClass } from '../style/NewBranchDialog';
import { SuspendModal } from './SuspendModal';
import { Alert } from './Alert';
const BRANCH_DESC = {
    current: 'The current branch. Pick this if you want to build on work done in this branch.',
    default: 'The default branch. Pick this if you want to start fresh from the default branch.'
};
/**
 * React component for rendering a dialog to create a new branch.
 */
export class NewBranchDialog extends React.Component {
    /**
     * Returns a React component for rendering a branch menu.
     *
     * @param props - component properties
     * @returns React component
     */
    constructor(props) {
        super(props);
        /**
         * Callback invoked upon closing the dialog.
         *
         * @param event - event object
         */
        this._onClose = () => {
            this.props.onClose();
            this.setState({
                name: '',
                filter: '',
                error: ''
            });
        };
        /**
         * Callback invoked upon a change to the menu filter.
         *
         * @param event - event object
         */
        this._onFilterChange = (event) => {
            this.setState({
                filter: event.target.value
            });
        };
        /**
         * Callback invoked to reset the menu filter.
         */
        this._resetFilter = () => {
            this.setState({
                filter: ''
            });
        };
        /**
         * Callback invoked upon a change to the branch name input element.
         *
         * @param event - event object
         */
        this._onNameChange = (event) => {
            this.setState({
                name: event.target.value,
                error: ''
            });
        };
        /**
         * Callback invoked upon clicking a button to create a new branch.
         *
         * @param event - event object
         */
        this._onCreate = () => {
            // Create the branch:
            this._createBranch(this.state.name);
        };
        /**
         * Callback invoked upon clicking on the feedback modal.
         *
         * @param event - event object
         */
        this._onFeedbackModalClick = () => {
            this._suspend(false);
        };
        /**
         * Callback invoked upon closing a feedback alert.
         *
         * @param event - event object
         */
        this._onFeedbackAlertClose = () => {
            this.setState({
                alert: false
            });
        };
        const repo = this.props.model.pathRepository;
        this.state = {
            name: '',
            base: repo ? this.props.model.currentBranch.name : '',
            filter: '',
            current: repo ? this.props.model.currentBranch.name : '',
            branches: repo ? this.props.model.branches : [],
            error: '',
            suspend: false,
            alert: false,
            log: {
                severity: 'info',
                message: ''
            }
        };
    }
    /**
     * Callback invoked immediately after mounting a component (i.e., inserting into a tree).
     */
    componentDidMount() {
        this._addListeners();
    }
    /**
     * Callback invoked when a component will no longer be mounted.
     */
    componentWillUnmount() {
        this._removeListeners();
    }
    /**
     * Renders the component.
     *
     * @returns React element
     */
    render() {
        return (React.createElement(React.Fragment, null,
            this._renderDialog(),
            this._renderFeedback()));
    }
    /**
     * Renders a dialog for creating a new branch.
     *
     * @returns React element
     */
    _renderDialog() {
        return (React.createElement(Dialog, { classes: {
                paper: branchDialogClass
            }, open: this.props.open, onClose: this._onClose },
            React.createElement("div", { className: titleWrapperClass },
                React.createElement("p", { className: titleClass }, "Create a Branch"),
                React.createElement("button", { className: closeButtonClass },
                    React.createElement(ClearIcon, { titleAccess: "Close this dialog", fontSize: "small", onClick: this._onClose }))),
            React.createElement("div", { className: contentWrapperClass },
                this.state.error ? (React.createElement("p", { className: errorMessageClass }, this.state.error)) : null,
                React.createElement("p", null, "Name"),
                React.createElement("input", { className: nameInputClass, type: "text", onChange: this._onNameChange, value: this.state.name, placeholder: "", title: "Enter a branch name" }),
                React.createElement("p", null, "Create branch based on..."),
                React.createElement("div", { className: filterWrapperClass },
                    React.createElement("div", { className: filterClass },
                        React.createElement("input", { className: filterInputClass, type: "text", onChange: this._onFilterChange, value: this.state.filter, placeholder: "Filter", title: "Filter branch menu" }),
                        this.state.filter ? (React.createElement("button", { className: filterClearClass },
                            React.createElement(ClearIcon, { titleAccess: "Clear the current filter", fontSize: "small", onClick: this._resetFilter }))) : null)),
                React.createElement("div", { className: listWrapperClass },
                    React.createElement(List, { disablePadding: true }, this._renderItems()))),
            React.createElement(DialogActions, { className: actionsWrapperClass },
                React.createElement("input", { className: classes(buttonClass, cancelButtonClass), type: "button", title: "Close this dialog without creating a new branch", value: "Cancel", onClick: this._onClose }),
                React.createElement("input", { className: classes(buttonClass, createButtonClass), type: "button", title: "Create a new branch", value: "Create Branch", onClick: this._onCreate, disabled: this.state.name === '' || this.state.error !== '' }))));
    }
    /**
     * Renders branch menu items.
     *
     * @returns array of React elements
     */
    _renderItems() {
        const current = this.props.model.currentBranch.name;
        return this.state.branches
            .slice()
            .sort(comparator)
            .map(this._renderItem, this);
        /**
         * Comparator function for sorting branches.
         *
         * @private
         * @param a - first branch
         * @param b - second branch
         * @returns integer indicating sort order
         */
        function comparator(a, b) {
            if (a.name === current) {
                return -1;
            }
            else if (b.name === current) {
                return 1;
            }
            if (a.name === 'master') {
                return -1;
            }
            else if (b.name === 'master') {
                return 1;
            }
            return 0;
        }
    }
    /**
     * Renders a branch menu item.
     *
     * @param branch - branch
     * @param idx - item index
     * @returns React element
     */
    _renderItem(branch, idx) {
        // Perform a "simple" filter... (TODO: consider implementing fuzzy filtering)
        if (this.state.filter && !branch.name.includes(this.state.filter)) {
            return null;
        }
        const isBase = branch.name === this.state.base;
        const isCurr = branch.name === this.state.current;
        let isBold;
        let desc;
        if (isCurr) {
            isBold = true;
            desc = BRANCH_DESC['current'];
        }
        return (React.createElement(ListItem, { button: true, title: `Create a new branch based on: ${branch.name}`, className: classes(listItemClass, isBase ? activeListItemClass : null), key: branch.name, onClick: this._onBranchClickFactory(branch.name) },
            React.createElement("span", { className: classes('jp-git-icon', listItemIconClass, 'jp-Icon-16', isBold && 'jp-git-selected') }),
            React.createElement("div", { className: listItemContentClass },
                React.createElement("p", { className: classes(listItemTitleClass, isBold ? listItemBoldTitleClass : null) }, branch.name),
                desc ? React.createElement("p", { className: listItemDescClass }, desc) : null)));
    }
    /**
     * Renders a component to provide UI feedback.
     *
     * @returns React element
     */
    _renderFeedback() {
        return (React.createElement(React.Fragment, null,
            React.createElement(SuspendModal, { open: this.props.suspend && this.state.suspend, onClick: this._onFeedbackModalClick }),
            React.createElement(Alert, { open: this.state.alert, message: this.state.log.message, severity: this.state.log.severity, onClose: this._onFeedbackAlertClose })));
    }
    /**
     * Adds model listeners.
     */
    _addListeners() {
        // When the HEAD changes, decent probability that we've switched branches:
        this.props.model.headChanged.connect(this._syncState, this);
        // When the status changes, we may have checked out a new branch (e.g., via the command-line and not via the extension) or changed repositories:
        this.props.model.statusChanged.connect(this._syncState, this);
    }
    /**
     * Removes model listeners.
     */
    _removeListeners() {
        this.props.model.headChanged.disconnect(this._syncState, this);
        this.props.model.statusChanged.disconnect(this._syncState, this);
    }
    /**
     * Syncs the component state with the underlying model.
     */
    _syncState() {
        const repo = this.props.model.pathRepository;
        this.setState({
            base: repo ? this.state.base : '',
            current: repo ? this.props.model.currentBranch.name : '',
            branches: repo ? this.props.model.branches : []
        });
    }
    /**
     * Sets the suspension state.
     *
     * @param bool - boolean indicating whether to suspend UI interaction
     */
    _suspend(bool) {
        if (this.props.suspend) {
            this.setState({
                suspend: bool
            });
        }
    }
    /**
     * Sets the current component log message.
     *
     * @param msg - log message
     */
    _log(msg) {
        this.setState({
            alert: true,
            log: msg
        });
    }
    /**
     * Returns a callback which is invoked upon clicking a branch name.
     *
     * @param branch - branch name
     * @returns callback
     */
    _onBranchClickFactory(branch) {
        const self = this;
        return onClick;
        /**
         * Callback invoked upon clicking a branch name.
         *
         * @private
         * @param event - event object
         */
        function onClick() {
            self.setState({
                base: branch
            });
        }
    }
    /**
     * Creates a new branch.
     *
     * @param branch - branch name
     * @returns promise which resolves upon attempting to create a new branch
     */
    async _createBranch(branch) {
        let result;
        const opts = {
            newBranch: true,
            branchname: branch
        };
        this._log({
            severity: 'info',
            message: 'Creating branch...'
        });
        this._suspend(true);
        try {
            result = await Promise.all([
                sleep(1000),
                this.props.model.checkout(opts)
            ]);
        }
        catch (err) {
            this._suspend(false);
            this.setState({
                error: err.message.replace(/^fatal:/, '')
            });
            this._log({
                severity: 'error',
                message: 'Failed to create branch.'
            });
            return;
        }
        this._suspend(false);
        const res = result[1];
        if (res.code !== 0) {
            this.setState({
                error: res.message
            });
            this._log({
                severity: 'error',
                message: 'Failed to create branch.'
            });
            return;
        }
        this._log({
            severity: 'success',
            message: 'Branch created.'
        });
        // Close the branch dialog:
        this.props.onClose();
        // Reset the branch name and filter:
        this.setState({
            name: '',
            filter: ''
        });
    }
}
//# sourceMappingURL=NewBranchDialog.js.map