import { JupyterFrontEnd } from '@jupyterlab/application';
import { IChangedArgs } from '@jupyterlab/coreutils';
import { IDocumentManager } from '@jupyterlab/docmanager';
import { ISettingRegistry } from '@jupyterlab/settingregistry';
import { CommandRegistry } from '@lumino/commands';
import { JSONObject } from '@lumino/coreutils';
import { ISignal } from '@lumino/signaling';
import { Git, IGitExtension } from './tokens';
/**
 * Class for creating a model for retrieving info from, and interacting with, a remote Git repository.
 */
export declare class GitExtension implements IGitExtension {
    /**
     * Returns an extension model.
     *
     * @param app - frontend application
     * @param settings - plugin settings
     * @returns extension model
     */
    constructor(serverRoot: string, app?: JupyterFrontEnd, docmanager?: IDocumentManager, settings?: ISettingRegistry.ISettings);
    /**
     * Branch list for the current repository.
     */
    get branches(): Git.IBranch[];
    /**
     * List of available Git commands.
     */
    get commands(): CommandRegistry | null;
    /**
     * The current repository branch.
     */
    get currentBranch(): Git.IBranch;
    /**
     * Boolean indicating whether the model has been disposed.
     */
    get isDisposed(): boolean;
    /**
     * Boolean indicating whether the model is ready.
     */
    get isReady(): boolean;
    /**
     * Promise which fulfills when the model is ready.
     */
    get ready(): Promise<void>;
    /**
     * Git repository path.
     *
     * ## Notes
     *
     * -   This is the full path of the top-level folder.
     * -   The return value is `null` if a repository path is not defined.
     */
    get pathRepository(): string | null;
    set pathRepository(v: string | null);
    /**
     * The Jupyter front-end application shell.
     */
    get shell(): JupyterFrontEnd.IShell | null;
    /**
     * A list of modified files.
     *
     * ## Notes
     *
     * -   The file list corresponds to the list of files from `git status`.
     */
    get status(): Git.IStatusFile[];
    /**
     * A signal emitted when the `HEAD` of the Git repository changes.
     */
    get headChanged(): ISignal<IGitExtension, void>;
    /**
     * A signal emitted when the current marking of the Git repository changes.
     */
    get markChanged(): ISignal<IGitExtension, void>;
    /**
     * A signal emitted when the current Git repository changes.
     */
    get repositoryChanged(): ISignal<IGitExtension, IChangedArgs<string | null>>;
    /**
     * A signal emitted when the current status of the Git repository changes.
     */
    get statusChanged(): ISignal<IGitExtension, Git.IStatusFile[]>;
    /**
     * A signal emitted whenever a model event occurs.
     */
    get logger(): ISignal<IGitExtension, string>;
    /**
     * Add one or more files to the repository staging area.
     *
     * ## Notes
     *
     * -   If no filename is provided, all files are added.
     *
     * @param filename - files to add
     * @returns promise which resolves upon adding files to the repository staging area
     */
    add(...filename: string[]): Promise<Response>;
    /**
     * Add all "unstaged" files to the repository staging area.
     *
     * @returns promise which resolves upon adding files to the repository staging area
     */
    addAllUnstaged(): Promise<Response>;
    /**
     * Add all untracked files to the repository staging area.
     *
     * @returns promise which resolves upon adding files to the repository staging area
     */
    addAllUntracked(): Promise<Response>;
    /**
     * Add a remote Git repository to the current repository.
     *
     * @param url - remote repository URL
     * @param name - remote name
     * @returns promise which resolves upon adding a remote
     */
    addRemote(url: string, name?: string): Promise<void>;
    /**
     * Retrieve the repository commit log.
     *
     * ## Notes
     *
     * -  This API can be used to implicitly check if the current folder is a Git repository.
     *
     * @param count - number of commits to retrieve
     * @returns promise which resolves upon retrieving the repository commit log
     */
    allHistory(count?: number): Promise<Git.IAllHistory>;
    /**
     * Checkout a branch.
     *
     * ## Notes
     *
     * -   If a branch name is provided, checkout the provided branch (with or without creating it)
     * -   If a filename is provided, checkout the file, discarding all changes.
     * -   If nothing is provided, checkout all files, discarding all changes.
     *
     * TODO: Refactor into separate endpoints for each kind of checkout request
     *
     * @param options - checkout options
     * @returns promise which resolves upon performing a checkout
     */
    checkout(options?: Git.ICheckoutOptions): Promise<Git.ICheckoutResult>;
    /**
     * Clone a repository.
     *
     * @param path - local path into which the repository will be cloned
     * @param url - Git repository URL
     * @param auth - remote repository authentication information
     * @returns promise which resolves upon cloning a repository
     */
    clone(path: string, url: string, auth?: Git.IAuth): Promise<Git.ICloneResult>;
    /**
     * Commit all staged file changes.
     *
     * @param message - commit message
     * @returns promise which resolves upon committing file changes
     */
    commit(message: string): Promise<Response>;
    /**
     * Get (or set) Git configuration options.
     *
     * @param options - configuration options to set
     * @returns promise which resolves upon either getting or setting configuration options
     */
    config(options?: JSONObject): Promise<Response>;
    /**
     * Revert changes made after a specified commit.
     *
     * @param message - commit message
     * @param hash - commit identifier (hash)
     * @returns promise which resolves upon reverting changes
     */
    revertCommit(message: string, hash: string): Promise<Response>;
    /**
     * Fetch commit information.
     *
     * @param hash - commit hash
     * @returns promise which resolves upon retrieving commit information
     */
    detailedLog(hash: string): Promise<Git.ISingleCommitFilePathInfo>;
    /**
     * Initialize a new Git repository at a specified path.
     *
     * @param path - path at which initialize a Git repository
     * @returns promise which resolves upon initializing a Git repository
     */
    init(path: string): Promise<Response>;
    /**
     * Retrieve commit logs.
     *
     * @param count - number of commits
     * @returns promise which resolves upon retrieving commit logs
     */
    log(count?: number): Promise<Git.ILogResult>;
    /**
     * Fetch changes from a remote repository.
     *
     * @param auth - remote authentication information
     * @returns promise which resolves upon fetching changes
     */
    pull(auth?: Git.IAuth): Promise<Git.IPushPullResult>;
    /**
     * Push local changes to a remote repository.
     *
     * @param auth - remote authentication information
     * @returns promise which resolves upon pushing changes
     */
    push(auth?: Git.IAuth): Promise<Git.IPushPullResult>;
    /**
     * Refresh the repository.
     *
     * @returns promise which resolves upon refreshing the repository
     */
    refresh(): Promise<void>;
    /**
     * Refresh the list of repository branches.
     *
     * @returns promise which resolves upon refreshing repository branches
     */
    refreshBranch(): Promise<void>;
    /**
     * Refresh the repository status.
     *
     * @returns promise which resolves upon refreshing the repository status
     */
    refreshStatus(): Promise<void>;
    /**
     * Move files from the "staged" to the "unstaged" area.
     *
     * ## Notes
     *
     * -  If no filename is provided, moves all files from the "staged" to the "unstaged" area.
     *
     * @param filename - file path to be reset
     * @returns promise which resolves upon moving files
     */
    reset(filename?: string): Promise<Response>;
    /**
     * Reset the repository to a specified commit.
     *
     * ## Notes
     *
     * -   If a commit hash is not provided, resets the repository to `HEAD`.
     *
     * @param hash - commit identifier (hash)
     * @returns promises which resolves upon resetting the repository
     */
    resetToCommit(hash?: string): Promise<Response>;
    /**
     * Retrieve the prefix path of a directory `path` with respect to the root repository directory.
     *
     * @param path - directory path
     * @returns promise which resolves upon retrieving the prefix path
     */
    showPrefix(path: string): Promise<Git.IShowPrefixResult>;
    /**
     * Retrieve the top level repository path.
     *
     * @param path - current path
     * @returns promise which resolves upon retrieving the top level repository path
     */
    showTopLevel(path: string): Promise<Git.IShowTopLevelResult>;
    /**
     * Retrieve the list of tags in the repository.
     *
     * @returns promise which resolves upon retrieving the tag list
     */
    tags(): Promise<Git.ITagResult>;
    /**
     * Checkout the specified tag version
     *
     * @param tag - selected tag version
     * @returns promise which resolves upon checking out the tag version of the repository
     */
    checkoutTag(tag: string): Promise<Git.ICheckoutResult>;
    /**
     * Add a file to the current marker object.
     *
     * @param fname - filename
     * @param mark - mark to set
     */
    addMark(fname: string, mark: boolean): void;
    /**
     * Return the current mark associated with a specified filename.
     *
     * @param fname - filename
     * @returns mark
     */
    getMark(fname: string): boolean;
    /**
     * Toggle the mark for a file in the current marker object
     *
     * @param fname - filename
     */
    toggleMark(fname: string): void;
    /**
     * Register a new diff provider for specified file types
     *
     * @param filetypes File type list
     * @param callback Callback to use for the provided file types
     */
    registerDiffProvider(filetypes: string[], callback: Git.IDiffCallback): void;
    /**
     * Return the path of a file relative to the Jupyter server root.
     *
     * ## Notes
     *
     * -   If no path is provided, returns the Git repository top folder relative path.
     * -   If no Git repository selected, returns `null`
     *
     * @param path - file path relative to the top folder of the Git repository
     * @returns relative path
     */
    getRelativeFilePath(path?: string): string | null;
    /**
     * Dispose of model resources.
     */
    dispose(): void;
    /**
     * Make request to ensure gitignore.
     *
     */
    ensureGitignore(): Promise<Response>;
    /**
     * Make request to ignore one file.
     *
     * @param filePath File to ignore
     * @param useExtension Whether to ignore the file or its extension
     */
    ignore(filePath: string, useExtension: boolean): Promise<Response>;
    /**
     * Get list of files changed between two commits or two branches
     * @param base id of base commit or base branch for comparison
     * @param remote id of remote commit or remote branch for comparison
     * @param singleCommit id of a single commit
     *
     * @returns the names of the changed files
     */
    changedFiles(base?: string, remote?: string, singleCommit?: string): Promise<Git.IChangedFilesResult>;
    /**
     * Make request for a list of all git branches in the repository
     * Retrieve a list of repository branches.
     *
     * @returns promise which resolves upon fetching repository branches
     */
    protected _branch(): Promise<Git.IBranchResult>;
    /**
     * Set the repository status.
     *
     * @param v - repository status
     */
    protected _setStatus(v: Git.IStatusFile[]): void;
    /**
     * Set the marker object for a repository path and branch.
     *
     * @returns branch marker
     */
    private _setMarker;
    /**
     * Adds a task to the list of pending model tasks.
     *
     * @param task - task name
     * @returns task identifier
     */
    private _addTask;
    /**
     * Removes a task from the list of pending model tasks.
     *
     * @param id - task identifier
     */
    private _removeTask;
    /**
     * Generates a unique task identifier.
     *
     * @returns task identifier
     */
    private _generateTaskID;
    /**
     * if file is open in JupyterLab find the widget and ensure the JupyterLab
     * version matches the version on disk. Do nothing if the file has unsaved changes
     *
     * @param path path to the file to be reverted
     */
    private _revertFile;
    private _status;
    private _pathRepository;
    private _branches;
    private _currentBranch;
    private _serverRoot;
    private _app;
    private _docmanager;
    private _diffProviders;
    private _isDisposed;
    private _markerCache;
    private _currentMarker;
    private _readyPromise;
    private _pendingReadyPromise;
    private _poll;
    private _taskList;
    private _taskID;
    private _settings;
    private _headChanged;
    private _markChanged;
    private _repositoryChanged;
    private _statusChanged;
    private _logger;
}
export declare class BranchMarker implements Git.IBranchMarker {
    private _refresh;
    constructor(_refresh: () => void);
    add(fname: string, mark?: boolean): void;
    get(fname: string): boolean;
    set(fname: string, mark: boolean): void;
    toggle(fname: string): void;
    private _marks;
}
export declare class Markers {
    private _refresh;
    constructor(_refresh: () => void);
    get(path: string, branch: string): BranchMarker;
    static markerKey(path: string, branch: string): string;
    private _branchMarkers;
}
