import { Construct, Duration, IResource, Resource } from '@aws-cdk/core';
import { IAliasRecordTarget } from './alias-record-target';
import { IHostedZone } from './hosted-zone-ref';
/**
 * A record set
 */
export interface IRecordSet extends IResource {
    /**
     * The domain name of the record
     */
    readonly domainName: string;
}
/**
 * The record type.
 */
export declare enum RecordType {
    A = "A",
    AAAA = "AAAA",
    CAA = "CAA",
    CNAME = "CNAME",
    MX = "MX",
    NAPTR = "NAPTR",
    NS = "NS",
    PTR = "PTR",
    SOA = "SOA",
    SPF = "SPF",
    SRV = "SRV",
    TXT = "TXT"
}
/**
 * Options for a RecordSet.
 */
export interface RecordSetOptions {
    /**
     * The hosted zone in which to define the new record.
     */
    readonly zone: IHostedZone;
    /**
     * The domain name for this record.
     *
     * @default zone root
     */
    readonly recordName?: string;
    /**
     * The resource record cache time to live (TTL).
     *
     * @default Duration.minutes(30)
     */
    readonly ttl?: Duration;
    /**
     * A comment to add on the record.
     *
     * @default no comment
     */
    readonly comment?: string;
}
/**
 * Type union for a record that accepts multiple types of target.
 */
export declare class RecordTarget {
    readonly values?: string[] | undefined;
    readonly aliasTarget?: IAliasRecordTarget | undefined;
    /**
     * Use string values as target.
     */
    static fromValues(...values: string[]): RecordTarget;
    /**
     * Use an alias as target.
     */
    static fromAlias(aliasTarget: IAliasRecordTarget): RecordTarget;
    /**
     * Use ip adresses as target.
     */
    static fromIpAddresses(...ipAddresses: string[]): RecordTarget;
    protected constructor(values?: string[] | undefined, aliasTarget?: IAliasRecordTarget | undefined);
}
/**
 * Construction properties for a RecordSet.
 */
export interface RecordSetProps extends RecordSetOptions {
    /**
     * The record type.
     */
    readonly recordType: RecordType;
    /**
     * The target for this record, either `RecordTarget.fromValues()` or
     * `RecordTarget.fromAlias()`.
     */
    readonly target: RecordTarget;
}
/**
 * A record set.
 */
export declare class RecordSet extends Resource implements IRecordSet {
    readonly domainName: string;
    constructor(scope: Construct, id: string, props: RecordSetProps);
}
/**
 * @deprecated Use RecordTarget
 */
export declare class AddressRecordTarget extends RecordTarget {
}
/**
 * Construction properties for a ARecord.
 */
export interface ARecordProps extends RecordSetOptions {
    /**
     * The target.
     */
    readonly target: RecordTarget;
}
/**
 * A DNS A record
 *
 * @resource AWS::Route53::RecordSet
 */
export declare class ARecord extends RecordSet {
    constructor(scope: Construct, id: string, props: ARecordProps);
}
/**
 * Construction properties for a AaaaRecord.
 */
export interface AaaaRecordProps extends RecordSetOptions {
    /**
     * The target.
     */
    readonly target: AddressRecordTarget;
}
/**
 * A DNS AAAA record
 *
 * @resource AWS::Route53::RecordSet
 */
export declare class AaaaRecord extends RecordSet {
    constructor(scope: Construct, id: string, props: AaaaRecordProps);
}
/**
 * Construction properties for a CnameRecord.
 */
export interface CnameRecordProps extends RecordSetOptions {
    /**
     * The domain name.
     */
    readonly domainName: string;
}
/**
 * A DNS CNAME record
 *
 * @resource AWS::Route53::RecordSet
 */
export declare class CnameRecord extends RecordSet {
    constructor(scope: Construct, id: string, props: CnameRecordProps);
}
/**
 * Construction properties for a TxtRecord.
 */
export interface TxtRecordProps extends RecordSetOptions {
    /**
     * The text values.
     */
    readonly values: string[];
}
/**
 * A DNS TXT record
 *
 * @resource AWS::Route53::RecordSet
 */
export declare class TxtRecord extends RecordSet {
    constructor(scope: Construct, id: string, props: TxtRecordProps);
}
/**
 * Properties for a SRV record value.
 */
export interface SrvRecordValue {
    /**
     * The priority.
     */
    readonly priority: number;
    /**
     * The weight.
     */
    readonly weight: number;
    /**
     * The port.
     */
    readonly port: number;
    /**
     * The server host name.
     */
    readonly hostName: string;
}
/**
 * Construction properties for a SrvRecord.
 */
export interface SrvRecordProps extends RecordSetOptions {
    /**
     * The values.
     */
    readonly values: SrvRecordValue[];
}
/**
 * A DNS SRV record
 *
 * @resource AWS::Route53::RecordSet
 */
export declare class SrvRecord extends RecordSet {
    constructor(scope: Construct, id: string, props: SrvRecordProps);
}
/**
 * The CAA tag.
 */
export declare enum CaaTag {
    /**
     * Explicity authorizes a single certificate authority to issue a
     * certificate (any type) for the hostname.
     */
    ISSUE = "issue",
    /**
     * Explicity authorizes a single certificate authority to issue a
     * wildcard certificate (and only wildcard) for the hostname.
     */
    ISSUEWILD = "issuewild",
    /**
     * Specifies a URL to which a certificate authority may report policy
     * violations.
     */
    IODEF = "iodef"
}
/**
 * Properties for a CAA record value.
 */
export interface CaaRecordValue {
    /**
     * The flag.
     */
    readonly flag: number;
    /**
     * The tag.
     */
    readonly tag: CaaTag;
    /**
     * The value associated with the tag.
     */
    readonly value: string;
}
/**
 * Construction properties for a CaaRecord.
 */
export interface CaaRecordProps extends RecordSetOptions {
    /**
     * The values.
     */
    readonly values: CaaRecordValue[];
}
/**
 * A DNS CAA record
 *
 * @resource AWS::Route53::RecordSet
 */
export declare class CaaRecord extends RecordSet {
    constructor(scope: Construct, id: string, props: CaaRecordProps);
}
/**
 * Construction properties for a CaaAmazonRecord.
 */
export interface CaaAmazonRecordProps extends RecordSetOptions {
}
/**
 * A DNS Amazon CAA record.
 *
 * A CAA record to restrict certificate authorities allowed
 * to issue certificates for a domain to Amazon only.
 *
 * @resource AWS::Route53::RecordSet
 */
export declare class CaaAmazonRecord extends CaaRecord {
    constructor(scope: Construct, id: string, props: CaaAmazonRecordProps);
}
/**
 * Properties for a MX record value.
 */
export interface MxRecordValue {
    /**
     * The priority.
     */
    readonly priority: number;
    /**
     * The mail server host name.
     */
    readonly hostName: string;
}
/**
 * Construction properties for a MxRecord.
 */
export interface MxRecordProps extends RecordSetOptions {
    /**
     * The values.
     */
    readonly values: MxRecordValue[];
}
/**
 * A DNS MX record
 *
 * @resource AWS::Route53::RecordSet
 */
export declare class MxRecord extends RecordSet {
    constructor(scope: Construct, id: string, props: MxRecordProps);
}
/**
 * Construction properties for a ZoneDelegationRecord
 */
export interface ZoneDelegationRecordProps extends RecordSetOptions {
    /**
     * The name servers to report in the delegation records.
     */
    readonly nameServers: string[];
}
/**
 * A record to delegate further lookups to a different set of name servers.
 */
export declare class ZoneDelegationRecord extends RecordSet {
    constructor(scope: Construct, id: string, props: ZoneDelegationRecordProps);
}
