"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPool = exports.AccountRecovery = exports.Mfa = exports.VerificationEmailStyle = exports.UserPoolOperation = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const punycode_1 = require("punycode/");
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
const user_pool_resource_server_1 = require("./user-pool-resource-server");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /**
     * A custom user pool operation.
     */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
/**
 * Creates a challenge in a custom auth flow.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * The email verification style.
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    Mfa["OFF"] = "OFF";
    Mfa["OPTIONAL"] = "OPTIONAL";
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * How will a user be able to recover their account?
 *
 * When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
 * You can choose the preferred way to send codes below.
 * We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
 */
var AccountRecovery;
(function (AccountRecovery) {
    AccountRecovery[AccountRecovery["EMAIL_AND_PHONE_WITHOUT_MFA"] = 0] = "EMAIL_AND_PHONE_WITHOUT_MFA";
    AccountRecovery[AccountRecovery["PHONE_WITHOUT_MFA_AND_EMAIL"] = 1] = "PHONE_WITHOUT_MFA_AND_EMAIL";
    AccountRecovery[AccountRecovery["EMAIL_ONLY"] = 2] = "EMAIL_ONLY";
    AccountRecovery[AccountRecovery["PHONE_ONLY_WITHOUT_MFA"] = 3] = "PHONE_ONLY_WITHOUT_MFA";
    AccountRecovery[AccountRecovery["PHONE_AND_EMAIL"] = 4] = "PHONE_AND_EMAIL";
    AccountRecovery[AccountRecovery["NONE"] = 5] = "NONE";
})(AccountRecovery = exports.AccountRecovery || (exports.AccountRecovery = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Get all identity providers registered with this user pool.
         */
        this.identityProviders = [];
    }
    /**
     * Add a new app client to this user pool.
     */
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    /**
     * Associate a domain to this user pool.
     */
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    /**
     * Add a new resource server to this user pool.
     */
    addResourceServer(id, options) {
        return new user_pool_resource_server_1.UserPoolResourceServer(this, id, {
            userPool: this,
            ...options,
        });
    }
    /**
     * Register an identity provider with this user pool.
     */
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * Define a Cognito User Pool.
 */
class UserPool extends UserPoolBase {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_a = props.userInvitation) === null || _a === void 0 ? void 0 : _a.emailBody,
            emailSubject: (_b = props.userInvitation) === null || _b === void 0 ? void 0 : _b.emailSubject,
            smsMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: encodePuny((_d = props.emailSettings) === null || _d === void 0 ? void 0 : _d.from),
                replyToEmailAddress: encodePuny((_e = props.emailSettings) === null || _e === void 0 ? void 0 : _e.replyTo),
            }),
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
            accountRecoverySetting: this.accountRecovery(props),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends UserPoolBase {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        return UserPool.fromUserPoolId(scope, id, core_1.Stack.of(scope).parseArn(userPoolArn).resourceName);
    }
    /**
     * Add a lambda trigger to a user pool operation.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        this.triggers[operation.operationName] = fn.functionArn;
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn,
        });
    }
    verificationMessageConfiguration(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailStyle) !== null && _b !== void 0 ? _b : VerificationEmailStyle.CODE;
        const emailSubject = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailSubject) !== null && _d !== void 0 ? _d : 'Verify your new account';
        const smsMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage) !== null && _f !== void 0 ? _f : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailBody) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases) !== null && _a !== void 0 ? _a : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a, _b, _c;
        if (props.enableSmsRole === false && props.smsRole) {
            throw new Error('enableSmsRole cannot be disabled when smsRole is specified');
        }
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
            };
        }
        if (props.enableSmsRole === false) {
            return undefined;
        }
        const mfaConfiguration = this.mfaConfiguration(props);
        const phoneVerification = ((_a = props.signInAliases) === null || _a === void 0 ? void 0 : _a.phone) === true || ((_b = props.autoVerify) === null || _b === void 0 ? void 0 : _b.phone) === true;
        const roleRequired = (mfaConfiguration === null || mfaConfiguration === void 0 ? void 0 : mfaConfiguration.includes('SMS_MFA')) || phoneVerification;
        if (!roleRequired && props.enableSmsRole === undefined) {
            return undefined;
        }
        const smsRoleExternalId = core_1.Names.uniqueId(this).substr(0, 1223); // sts:ExternalId max length of 1224
        const smsRole = (_c = props.smsRole) !== null && _c !== void 0 ? _c : new aws_iam_1.Role(this, 'smsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                conditions: {
                    StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                },
            }),
            inlinePolicies: {
                /*
                  * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                  * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                  * Seems like a case of being excessively strict.
                  */
                'sns-publish': new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: ['sns:Publish'],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        return {
            externalId: smsRoleExternalId,
            snsCallerArn: smsRole.roleArn,
        };
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_b = props.passwordPolicy.minLength) !== null && _b !== void 0 ? _b : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.requireLowercase,
            requireUppercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireUppercase,
            requireNumbers: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireDigits,
            requireSymbols: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => {
                var _a, _b;
                return ({
                    name: attr_names_1.StandardAttributeNames[attrName],
                    mutable: (_a = attr.mutable) !== null && _a !== void 0 ? _a : true,
                    required: (_b = attr.required) !== null && _b !== void 0 ? _b : false,
                });
            });
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
    accountRecovery(props) {
        var _a;
        const accountRecovery = (_a = props.accountRecovery) !== null && _a !== void 0 ? _a : AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL;
        switch (accountRecovery) {
            case AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_email', priority: 1 },
                        { name: 'verified_phone_number', priority: 2 },
                    ],
                };
            case AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_phone_number', priority: 1 },
                        { name: 'verified_email', priority: 2 },
                    ],
                };
            case AccountRecovery.EMAIL_ONLY:
                return {
                    recoveryMechanisms: [{ name: 'verified_email', priority: 1 }],
                };
            case AccountRecovery.PHONE_ONLY_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [{ name: 'verified_phone_number', priority: 1 }],
                };
            case AccountRecovery.NONE:
                return {
                    recoveryMechanisms: [{ name: 'admin_only', priority: 1 }],
                };
            case AccountRecovery.PHONE_AND_EMAIL:
                return undefined;
            default:
                throw new Error(`Unsupported AccountRecovery type - ${accountRecovery}`);
        }
    }
}
exports.UserPool = UserPool;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
function encodePuny(input) {
    return input !== undefined ? punycode_1.toASCII(input) : input;
}
//# sourceMappingURL=data:application/json;base64,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