"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Archiver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * Every week
 */
const DEFAULT_CRON_EXPRESSION = 'cron(0 0 ? * 1 *)';
class Archiver extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.logGroupKmsKey = this.createLogGroupKey();
        this.logGroup = this.createLogGroup();
        this.topic = new aws_cdk_lib_1.aws_sns.Topic(this, 'notifications', {
            displayName: 'archiver-notifications',
        });
        this.bucket = this.createArchiveBucket();
        this.bucket.addEventNotification(aws_cdk_lib_1.aws_s3.EventType.OBJECT_CREATED, new aws_cdk_lib_1.aws_s3_notifications.SnsDestination(this.topic));
        this.createProjects();
        this.createCfnOutputs();
    }
    createCfnOutputs() {
        new aws_cdk_lib_1.CfnOutput(this, 's3-bucket-arn', {
            description: 'ARN of the S3 bucket storing the repositories.',
            value: this.bucket.bucketArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'log-group-arn', {
            description: 'ARN of the Cloudwatch Log group storing the code build logs.',
            value: this.logGroup.logGroupArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'log-group-key', {
            description: 'ARN of the KMS key used to encrypt the Cloudwatch logs.',
            value: this.logGroupKmsKey.keyArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'sns-topic-arn', {
            description: 'ARN of the SNS topic.',
            value: this.topic.topicArn,
        });
    }
    createLogGroup() {
        const loggroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'loggroup', {
            encryptionKey: this.logGroupKmsKey,
            retention: this.props.retentionDays
                ? this.props.retentionDays
                : aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        loggroup.node.addDependency(this.logGroupKmsKey);
        return loggroup;
    }
    /**
     *Create the S3 bucket that will later store the repositories.
     *
     * @private
     * @return {*}
     * @memberof Archiver
     */
    createArchiveBucket() {
        return new aws_cdk_lib_1.aws_s3.Bucket(this, 'destination', {
            blockPublicAccess: aws_cdk_lib_1.aws_s3.BlockPublicAccess.BLOCK_ALL,
            encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(360),
                    transitions: [
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.GLACIER,
                            transitionAfter: aws_cdk_lib_1.Duration.days(90),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.DEEP_ARCHIVE,
                            transitionAfter: aws_cdk_lib_1.Duration.days(180),
                        },
                    ],
                },
            ],
            publicReadAccess: false,
            versioned: true,
        });
    }
    createLogGroupKey() {
        const key = new aws_cdk_lib_1.aws_kms.Key(this, 'loggroupKey', {
            description: 'Repository Archiver',
            enableKeyRotation: true,
            pendingWindow: aws_cdk_lib_1.Duration.days(7),
            keyUsage: aws_cdk_lib_1.aws_kms.KeyUsage.ENCRYPT_DECRYPT,
            keySpec: aws_cdk_lib_1.aws_kms.KeySpec.SYMMETRIC_DEFAULT,
            alias: 'archiver-loggroup-key',
        });
        key.grantEncryptDecrypt(new aws_cdk_lib_1.aws_iam.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`));
        return key;
    }
    /**
     * Creates for each backup configuration a separate CodeBuild project.
     *
     * @private
     * @memberof Archiver
     */
    createProjects() {
        this.props.backupConfigurations.forEach((element) => {
            const project = this.createProject(element);
            project.enableBatchBuilds();
            this.bucket.grantReadWrite(project);
            new aws_cdk_lib_1.aws_events.Rule(this, 'ScheduleRule-' + element.organizationName + '-' + element.projectName, {
                enabled: true,
                schedule: aws_cdk_lib_1.aws_events.Schedule.expression(DEFAULT_CRON_EXPRESSION),
                targets: [new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project)],
                description: 'Trigger for Azure DevOps git repositories backup',
            });
            console.log(element);
        });
    }
    /**
     * Create a CodeBuild project.
     *
     * @private
     * @param {BackupConfiguration} element
     * @return {*}
     * @memberof Archiver
     */
    createProject(element) {
        return new aws_cdk_lib_1.aws_codebuild.Project(this, 'archiver-' + element.organizationName + '-' + element.projectName, {
            timeout: aws_cdk_lib_1.Duration.hours(5),
            projectName: 'AzureDevOpsGitBackup' +
                '-' +
                element.organizationName +
                '-' +
                element.projectName,
            description: 'Backup Azure DevOps git repositories to an S3 bucket.',
            checkSecretsInPlainTextEnvVariables: true,
            concurrentBuildLimit: 90,
            environment: {
                environmentVariables: {
                    TOKEN: {
                        value: element.secretArn + ':pat',
                        type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    },
                    ORGANIZATION: { value: element.organizationName },
                    PROJECT: { value: element.projectName },
                },
                buildImage: aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_6_0,
            },
            logging: {
                cloudWatch: {
                    enabled: true,
                    logGroup: this.logGroup,
                },
            },
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject({
                version: 0.2,
                batch: {
                    'fail-fast': false,
                    'build-list': this.createBatchConfiguration(element.repositoryNames),
                },
                phases: {
                    build: {
                        commands: [
                            'git clone --mirror "https://${TOKEN}@dev.azure.com/${ORGANIZATION}/${PROJECT}/_git/${REPOSITORY}"',
                            'tar czf ${REPOSITORY}.tgz ./${REPOSITORY}.git',
                            'aws s3 cp ./${REPOSITORY}.tgz ' +
                                this.bucket.s3UrlForObject() +
                                '/${ORGANIZATION}/${PROJECT}/${REPOSITORY}.tgz',
                        ],
                    },
                },
            }),
        });
    }
    createBatchConfiguration(repositoryNames) {
        const output = [];
        repositoryNames.forEach((element) => {
            output.push({
                identifier: 'build_' + element.replace(/-/g, '_'),
                env: {
                    variables: {
                        REPOSITORY: element,
                    },
                },
            });
        });
        return output;
    }
}
exports.Archiver = Archiver;
_a = JSII_RTTI_SYMBOL_1;
Archiver[_a] = { fqn: "azure-devops-repository-archiver.Archiver", version: "0.0.15" };
//# sourceMappingURL=data:application/json;base64,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