
import datetime
import argparse
import collections
import textwrap
import operator
import os
import pathlib
import sys
import re
from xml.etree import ElementTree
from xml.etree.ElementTree import Element
from typing import Mapping, Tuple, Optional

# PyPI
try:
    from jinja2 import Environment
except ModuleNotFoundError as error:
    sys.exit("Dependency jinja2 is needed to run this tool. "
             "Try 'python3 -m pip install --user jinja2'.")

# Package
import emqxlwm2m.loadobjects


PY_TEMPLATE = """\
\"\"\"Generated by emqxlwm2m.codegen at {{ date_time }}

LwM2M Object: {{ obj.name }}
ID: {{ obj.oid }}, URN: {{ obj.urn }}, {{ obj.mandatory_str }}, {{ obj.multiple_str }}
\"\"\"

from emqxlwm2m.lwm2m import *


{% for res in obj.resources %}
class {{ res.name_class }}(Resource):
    \"\"\"{{ res.name }} {{res.rid}} - {{ res.type|title }}, {{ res.multiple_str }}, {{ res.mandatory_str }}

{% if res.description %}
{{ res.description }}
{% endif %}
    \"\"\"

    rid = {{res.rid}}
    operations = {{ res.operations }}
    type = '{{ res.type }}'
    range = '{{ res.range_enumeration }}'
    unit = '{{ res.units }}'
    mandatory = {{ res.mandatory }}
    multiple = {{ res.multiple }}


{% endfor %}
class {{ obj.name_class }}(ObjectDef):
    \"\"\"{{ obj.name }} Object {{ obj.oid }} - {{ obj.mandatory_str }}, {{ obj.multiple_str }}

{{ obj.description }}
    \"\"\"

    oid = {{ obj.oid }}
    mandatory = {{ obj.mandatory }}
    multiple = {{ obj.multiple }}
{% for res in obj.resources %}

    # ID={{ res.rid }}, {{ res.type|title }}, {{ res.operations }}, {{ res.multiple_str }}, {{ res.mandatory_str }}, range: {{ res.range_enumeration }}, unit: {{ res.units }}
    {{ res.name_snake }} = {{ res.name_class }}({{ res.default_value }})
{% endfor %}
"""

INIT_TEMPLATE = """\
\"\"\"Generated by emqxlwm2m.codegen at {{ date_time }}\"\"\"

{% for obj in objs %}
from .{{ obj.name_module }} import {{ obj.name_class }}
{% endfor %}

__all__ = [
{% for obj in objs %}
    '{{ obj.name_class }}',
{% endfor %}
]
"""


def generate_python_code(template, **kwargs):
    jinja_env = Environment(trim_blocks=True)
    now = format(datetime.datetime.now(), '%Y-%m-%d %H:%M:%S')
    return jinja_env.from_string(template).render(date_time=now, **kwargs)


if __name__ == '__main__':
    parser = argparse.ArgumentParser(
        prog='python3 -m emqxlwm2m.codegen',
        description='Parses an LwM2M object definition XML and '
                    'generates Python skeleton.')
    parser.add_argument(
        'input',
        nargs='*',
        help='Input XML files or directories containing XML files.')
    parser.add_argument(
        '-o', '--output', default='/dev/stdout',
        help='Output filename or output directory if multiple inputs.')
    parser.add_argument(
        '-f', '--force', action='store_true',
        help='Overwrite exiting output files.')

    args = parser.parse_args()
    mode = 'w' if args.force else 'x'
    inputs = set(args.input)
    xmlfiles = emqxlwm2m.loadobjects.load_objects(sorted(inputs))
    output_is_file = os.path.isfile(args.output)
    output_is_dir = os.path.isdir(args.output)
    if len(xmlfiles) > 1 and not output_is_dir:
        sys.exit(
            'Output must be directory if multiple xml files are provided.')

    objects = emqxlwm2m.loadobjects.load_objects(xmlfiles)
    for xmlfile, obj in objects.items():
        content = generate_python_code(PY_TEMPLATE, obj=obj)
        if output_is_dir:
            filename = os.path.join(args.output, f'{obj.name_module}.py')
        elif output_is_file:
            filename = args.output
        else:
            print(content)
            continue
        with open(filename, mode) as f:
            print(content, file=f)

    if output_is_dir:
        content = generate_python_code(INIT_TEMPLATE, objs=objects.values())
        filename = os.path.join(args.output, '__init__.py')
        with open(filename, mode) as f:
            print(content, file=f)
