
# PyDPFT demo


A package for Density Potential Functional Theory introduced by Julian Schwinger and Berge Englert.

A project developed by Ding Ruiqi from National University of Singapore Physics Department for his bachelor thesis

`Author:` Ding Ruiqi

`Date:` 6 April 2020

`Thesis:` https://tesla-cat.github.io/about/Bachelor-Thesis.pdf 

`Presentation:` https://github.com/tesla-cat/PyDPFT/Thesis/PPT

`PyPI:` https://pypi.org/project/PyDPFT/

`Github:` https://github.com/tesla-cat/PyDPFT

**Run this notebook in Google Colab:** https://colab.research.google.com/drive/1bC64gVM2WMk85MH2clTqDnBbaLhLCwTr


`Highlights:`
  - multi-GPU acceleration
  - optimized using Tensor operation (instead of for-loop) 
  - implemented from 1D to 3D with auto detection

## import



```python
!pip install PyDPFT
from PyDPFT import PyDPFT
from PyDPFT.plot import plot
```

    Collecting PyDPFT
      Downloading https://files.pythonhosted.org/packages/b9/f4/c9989112b6c1beafbd7c592ebe07864dcae9d8d93f5878af1207187d3331/PyDPFT-0.0.7-py3-none-any.whl
    Installing collected packages: PyDPFT
    Successfully installed PyDPFT-0.0.7


## 1D Hartree interaction


```python
config = {
    'space':{'x':[-5,5,100]},
    'loop':{'Imax':1000,'precision':1e-6,'mix':0.05},
    'const':{'epsilon':1e-3},
    'rho':{'N':32},
    'Vint':{'name':'Hartree','coef':1},
}

dpft = PyDPFT(config)
Vext = 1e3*dpft.x ** 2
Vx,Vint,rho,N = dpft(Vext)
plot(dpft,Vx,Vint,rho)
```

    PyDPFT: Written by Ding Ruiqi from NUS for his bachelor thesis
    PyDPFT: Detected dim = 1
    PyDPFT: Using 1 GPUs !
    PyDPFT: Starting the self consistent loop
    PyDPFT: Converged after 223 iterations in 2.0902912616729736 seconds!



![png](output_3_1.png)


## 2D Dipole-dipole interaction in momentum (p) space


```python
config = {
    'space':{'x':[-5,5,50],'y':[-5,5,50]},
    'loop':{'Imax':1000,'precision':1e-6,'mix':0.05},
    'const':{'epsilon':1e-2,'mu':[0.7, 0.7]},
    'rho':{'N':32},
    'Vint':{'name':'Dipole-p','coef':.1},
}

dpft = PyDPFT(config)
Vext = dpft.xx**2 + dpft.yy**2
Vx,Vint,rho,N = dpft(Vext)
plot(dpft,Vx,Vint,rho)
```

    PyDPFT: Written by Ding Ruiqi from NUS for his bachelor thesis
    PyDPFT: Detected dim = 2
    PyDPFT: Using 1 GPUs !
    PyDPFT: Starting the self consistent loop
    PyDPFT: Converged after 219 iterations in 11.653722524642944 seconds!



![png](output_5_1.png)


## 3D Dipole-dipole interaction in position (x) space


```python
config = {
    'space':{'x':[-5,5,20],'y':[-5,5,20],'z':[-5,5,20]},
    'loop':{'Imax':1000,'precision':1e-6,'mix':0.05},
    'const':{'epsilon':1e-2,'mu':[0.7, 0.7, 0]},
    'rho':{'N':32},
    'Vint':{'name':'Dipole-x','coef':5},
}

dpft = PyDPFT(config)
Vext = dpft.xx**2 + dpft.yy**2 + dpft.zz**2
Vx,Vint,rho,N = dpft(Vext)
plot(dpft,Vx,Vint,rho)
```

    PyDPFT: Written by Ding Ruiqi from NUS for his bachelor thesis
    PyDPFT: Detected dim = 3
    PyDPFT: Using 1 GPUs !
    PyDPFT: Starting the self consistent loop
    PyDPFT: Converged after 142 iterations in 4.736774921417236 seconds!



![png](output_7_1.png)

