"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLake = exports.LakeType = exports.Permissions = exports.Stage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const athena = require("@aws-cdk/aws-athena");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const lf = require("@aws-cdk/aws-lakeformation");
const lambda = require("@aws-cdk/aws-lambda");
const aws_lambda_python_1 = require("@aws-cdk/aws-lambda-python");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const data_set_1 = require("./data-sets/data-set");
const kinesis_ops_1 = require("./data-streams/kinesis-ops");
const kinesis_stream_1 = require("./data-streams/kinesis-stream");
const s3_delivery_stream_1 = require("./data-streams/s3-delivery-stream");
const glue_crawler_1 = require("./etl/glue-crawler");
const glue_job_1 = require("./etl/glue-job");
const glue_job_ops_1 = require("./etl/glue-job-ops");
const glue_table_1 = require("./etl/glue-table");
const data_lake_admin_1 = require("./personas/data-lake-admin");
const data_lake_creator_1 = require("./personas/data-lake-creator");
const pipeline_1 = require("./pipeline");
const utils_1 = require("./utils");
/**
 * @experimental
 */
var Stage;
(function (Stage) {
    Stage["ALPHA"] = "alpha";
    Stage["BETA"] = "beta";
    Stage["GAMMA"] = "gamma";
    Stage["PROD"] = "prod";
})(Stage = exports.Stage || (exports.Stage = {}));
/**
 * @experimental
 */
var Permissions;
(function (Permissions) {
    Permissions["ALTER"] = "ALTER";
    Permissions["CREATE_DATABASE"] = "CREATE_DATABASE";
    Permissions["CREATE_TABLE"] = "CREATE_TABLE";
    Permissions["DATA_LOCATION_ACCESS"] = "DATA_LOCATION_ACCESS";
    Permissions["DELETE"] = "DELETE";
    Permissions["DESCRIBE"] = "DESCRIBE";
    Permissions["DROP"] = "DROP";
    Permissions["INSERT"] = "INSERT";
    Permissions["SELECT"] = "SELECT";
    Permissions["ASSOCIATE"] = "ASSOCIATE";
})(Permissions = exports.Permissions || (exports.Permissions = {}));
/**
 * @experimental
 */
var LakeType;
(function (LakeType) {
    LakeType["DATA_PRODUCT"] = "DATA_PRODUCT";
    LakeType["CENTRAL_CATALOG"] = "CENTRAL_CATALOG";
    LakeType["CONSUMER"] = "CONSUMER";
    LakeType["DATA_PRODUCT_AND_CATALOG"] = "DATA_PRODUCT_AND_CATALOG";
})(LakeType = exports.LakeType || (exports.LakeType = {}));
/**
 * (experimental) A CDK construct to create a DataLake.
 *
 * @experimental
 */
class DataLake extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * @experimental
         */
        this.dataSets = {};
        /**
         * @experimental
         */
        this.dataStreams = {};
        /**
         * @experimental
         */
        this.databases = {};
        this.downloadLocations = {}; //used for the Custom Resource to allow downloading of existing datasets into datalake
        this.stageName = props.stageName;
        this.crossAccountAccess = props.crossAccountAccess ? props.crossAccountAccess : undefined;
        this.vpc = props.vpc ? props.vpc : undefined;
        this.lakeType = props.lakeType;
        if (props.logBucketProps) {
            this.logBucketProps = props.logBucketProps;
        }
        else {
            this.logBucketProps = {
                lifecycleRules: [
                    {
                        expiration: cdk.Duration.days(30),
                    },
                ],
                removalPolicy: cdk.RemovalPolicy.DESTROY,
                autoDeleteObjects: true,
            };
        }
        if (this.vpc) {
            this.glueSecurityGroup = new ec2.SecurityGroup(this, 'glue-sg', {
                description: 'Glue self referential allow in out',
                vpc: this.vpc,
                securityGroupName: utils_1.buildUniqueName({
                    name: 'glue',
                    resourceUse: 'datalake',
                    stage: this.stageName,
                }, 80),
            });
            this.glueSecurityGroup.connections.allowFrom(this.glueSecurityGroup, ec2.Port.allTcp());
            this.glueSecurityGroup.connections.allowTo(this.glueSecurityGroup, ec2.Port.allTcp());
            new cdk.CfnOutput(this, 'GlueSecurityGroupName', { value: this.glueSecurityGroup.securityGroupName });
        }
        // make this optional
        this.logBucket = new s3.Bucket(this, 'datalake-log-bucket', {
            bucketName: utils_1.buildS3BucketName({
                stage: props.stageName,
                resourceUse: 'log-bucket',
                name: props.name,
            }),
            ...this.logBucketProps,
        });
        new cdk.CfnOutput(this, 'DataLakeLogBucket', { value: this.logBucket.bucketName });
        if (props.datalakeAdminRole) {
            this.datalakeAdminRole = props.datalakeAdminRole;
        }
        else {
            this.datalakeAdminRole = new data_lake_admin_1.DataLakeAdministrator(this, `${props.name}-datalake-admin-role`, {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'datalake-admin',
                    stage: this.stageName,
                }, 60),
            }).role;
        }
        if (props.datalakeCreatorRole) {
            this.datalakeDbCreatorRole = props.datalakeCreatorRole;
        }
        else {
            this.datalakeDbCreatorRole = new data_lake_creator_1.DataLakeCreator(this, `${props.name}-datalake-creator-role`, {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'datalake-creator',
                    stage: this.stageName,
                }, 60),
            }).role;
        }
        if (props.createDefaultDatabase) {
            this.databases[props.name] = this.createDatabase(props.name);
            new cdk.CfnOutput(this, 'DataLakeDefaultDatabase', { value: props.name });
        }
        if (this.crossAccountAccess) {
            this.createCrossAccountGlueCatalogResourcePolicy(this.crossAccountAccess.consumerAccountIds, this.crossAccountAccess.dataCatalogOwnerAccountId);
        }
        if (props.createAthenaWorkgroup) {
            this.athenaWorkgroup = new athena.CfnWorkGroup(this, 'workgroup', {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'workgroup',
                    stage: this.stageName,
                }, 60),
                description: 'Default Data Lake Workgroup',
                state: 'ENABLED',
                recursiveDeleteOption: true,
                workGroupConfiguration: {
                    enforceWorkGroupConfiguration: true,
                    resultConfiguration: {
                        outputLocation: `s3://${this.logBucket.bucketName}/results/`,
                    },
                    engineVersion: {
                        selectedEngineVersion: 'Athena engine version 2',
                        effectiveEngineVersion: 'Athena engine version 2',
                    },
                },
            });
            new cdk.CfnOutput(this, 'DataLakeAthenaWorkgroup', { value: this.athenaWorkgroup.name });
        }
        if (props.policyTags) {
            this.createPolicyTagsCustomResource(props.policyTags, this.datalakeAdminRole);
        }
        if (props.dataProducts && props.dataProducts.length > 0) {
            props.dataProducts.forEach((product) => {
                this.databases[product.databaseName] = this.createDatabase(product.databaseName);
                product.pipelines.forEach((pipe) => {
                    this.addPipeline(pipe, product);
                });
            });
        }
    }
    /**
     * @experimental
     */
    createDownloaderCustomResource(stageName) {
        // download the data sets with the custom resource after successfull creation of resource
        const onEvent = new aws_lambda_python_1.PythonFunction(this, 'DataloaderHandler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/download-data'),
            timeout: cdk.Duration.minutes(15),
            functionName: utils_1.buildLambdaFunctionName({
                name: 'load-data',
                resourceUse: 'cr',
                stage: stageName,
            }),
        });
        // create readable and writable buckets for the datasets and set the appropriate S3 access
        onEvent.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:*'],
            resources: ['*'],
        }));
        const dataLoadProvider = new cr.Provider(this, 'DataloaderProvider', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        // CR to download the static datasets form the dataSets var passed in.
        new cdk.CustomResource(this, 'LoadDatalakeCustomResource', {
            serviceToken: dataLoadProvider.serviceToken,
            properties: {
                dataSets: this.downloadLocations,
                stackName: cdk.Stack.name,
                regionName: core_1.Aws.REGION,
            },
        });
    }
    createDatabase(databaseName) {
        const db = new glue.Database(this, `${databaseName}-database`, {
            databaseName: `${databaseName}`,
        });
        const dbPerm = new lf.CfnPermissions(this, `${databaseName}-lf-db-creator-permission`, {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: this.datalakeDbCreatorRole.roleArn,
            },
            resource: {
                databaseResource: {
                    name: databaseName,
                },
            },
            permissions: [
                Permissions.ALTER,
                Permissions.CREATE_TABLE,
                Permissions.DROP,
            ],
        });
        dbPerm.node.addDependency(db);
        return db;
    }
    addDataStream(pipeline, dataSet) {
        const schemaName = pipeline.name;
        const dataStreamStack = new cdk.NestedStack(cdk.Stack.of(this), `${schemaName}-datastream-stack`);
        if (!pipeline.streamProperties) {
            throw Error("Cannot create a stream pipeline without 'sreamProperties'");
        }
        this.dataStreams[schemaName] = new kinesis_stream_1.KinesisStream(dataStreamStack, 'DataStream', {
            shardCount: 1,
            streamName: pipeline.streamProperties.streamName,
        });
        const deliveryStream = new s3_delivery_stream_1.S3DeliveryStream(dataStreamStack, 'deliveryStream', {
            compression: s3_delivery_stream_1.CompressionType.UNCOMPRESSED,
            kinesisStream: this.dataStreams[schemaName].stream,
            s3Bucket: s3.Bucket.fromBucketName(this, 'get-bucket-for-kinesis', utils_1.getDataSetBucketName(pipeline.dataSetDropLocation, dataSet)),
            s3Prefix: pipeline.destinationPrefix,
        });
        new kinesis_ops_1.KinesisOps(dataStreamStack, 'kinesis-ops', {
            stream: this.dataStreams[schemaName],
            deliveryStream: deliveryStream,
        });
        if (pipeline.streamProperties.lambdaDataGenerator) {
            const dataGeneratorFunction = new lambda.Function(dataStreamStack, 'data-generator-function', {
                code: pipeline.streamProperties.lambdaDataGenerator.code,
                handler: pipeline.streamProperties.lambdaDataGenerator.handler,
                timeout: pipeline.streamProperties.lambdaDataGenerator.timeout,
                runtime: pipeline.streamProperties.lambdaDataGenerator.runtime,
                functionName: pipeline.streamProperties.lambdaDataGenerator.functionName,
                environment: {
                    KINESIS_STREAM: this.dataStreams[schemaName].stream.streamName,
                },
            });
            this.dataStreams[schemaName].stream.grantWrite(dataGeneratorFunction);
            const rule = new events.Rule(this, 'Rule', {
                schedule: pipeline.streamProperties.lambdaDataGenerator.schedule,
                ruleName: pipeline.streamProperties.lambdaDataGenerator.ruleName,
            });
            rule.addTarget(new targets.LambdaFunction(dataGeneratorFunction));
        }
        return this.dataStreams[schemaName];
    }
    addPipeline(pipeline, dataProduct) {
        const schemaName = pipeline.name;
        const dataSetStack = dataProduct.accountId == core_1.Aws.ACCOUNT_ID ? new cdk.NestedStack(cdk.Stack.of(this), `${schemaName}-dataset-stack`) : this;
        // create the dataSet
        this.dataSets[schemaName] = new data_set_1.DataSet(dataSetStack, schemaName, {
            pipeline: pipeline,
            dataProduct: dataProduct,
            logBucket: this.logBucket,
            stage: this.stageName,
            s3BucketProps: dataProduct.s3BucketProps,
            lakeType: this.lakeType,
        });
        const ds = this.dataSets[schemaName];
        const catelogAccountId = dataProduct.dataCatalogAccountId ? dataProduct.dataCatalogAccountId : core_1.Aws.ACCOUNT_ID;
        if (this.lakeType === LakeType.DATA_PRODUCT || this.lakeType === LakeType.DATA_PRODUCT_AND_CATALOG) {
            this.createPipelineResources(pipeline, dataProduct, ds);
        }
        // only create the table if the lake has a catelog
        if (pipeline.table && (this.lakeType === LakeType.CENTRAL_CATALOG || this.lakeType === LakeType.DATA_PRODUCT_AND_CATALOG)) {
            const table = new glue_table_1.GlueTable(this, `${pipeline.name}-table`, {
                catalogId: pipeline.table.catalogId,
                columns: pipeline.table.columns,
                databaseName: this.databases[dataProduct.databaseName].databaseName,
                description: pipeline.table.description,
                inputFormat: pipeline.table.inputFormat,
                outputFormat: pipeline.table.outputFormat,
                parameters: pipeline.table.parameters,
                partitionKeys: pipeline.table.partitionKeys,
                s3Location: `s3://${utils_1.getDataSetBucketName(pipeline.dataSetDropLocation, ds)}/${pipeline.destinationPrefix}`,
                serdeParameters: pipeline.table.serdeParameters,
                serializationLibrary: pipeline.table.serializationLibrary,
                tableName: pipeline.table.tableName,
            });
            table.node.addDependency(this.databases[dataProduct.databaseName]);
        }
        // find the correct metadata catalog account
        if (catelogAccountId == core_1.Aws.ACCOUNT_ID) {
            // refactor to only register the needed buckets from the data product account
            const rawDlResource = this.registerDataLakeLocation(this.datalakeAdminRole.roleArn, this.datalakeDbCreatorRole.roleArn, ds.rawBucketName);
            const trustedDlResource = this.registerDataLakeLocation(this.datalakeAdminRole.roleArn, this.datalakeDbCreatorRole.roleArn, ds.trustedBucketName);
            const refinedDlResource = this.registerDataLakeLocation(this.datalakeAdminRole.roleArn, this.datalakeDbCreatorRole.roleArn, ds.refinedBucketName);
            if (!pipeline.table) {
                const bucketName = utils_1.getDataSetBucketName(pipeline.dataSetDropLocation, ds);
                const name = bucketName.replace(/\W/g, '');
                // only create a crawler for the drop location of the data in the data product of the pipeline
                const crawler = new glue_crawler_1.GlueCrawler(this, `data-lake-crawler-${name}`, {
                    name: utils_1.buildGlueCrawlerName({
                        stage: this.stageName,
                        resourceUse: 'crawler',
                        name: pipeline.name,
                    }),
                    databaseName: dataProduct.databaseName,
                    bucketName: bucketName,
                    bucketPrefix: pipeline.destinationPrefix,
                    roleName: utils_1.buildRoleName({
                        stage: this.stageName,
                        resourceUse: 'crawler-role',
                        name: pipeline.name,
                    }),
                });
                crawler.node.addDependency(rawDlResource);
                crawler.node.addDependency(trustedDlResource);
                crawler.node.addDependency(refinedDlResource);
            }
        }
    }
    // this is a jumbled mess clean up once refecto
    createPipelineResources(pipeline, dataProduct, ds) {
        switch (pipeline.type) {
            case pipeline_1.DataPipelineType.S3: {
                if (ds.downloadLocations) {
                    this.downloadLocations[pipeline.name] = ds.downloadLocations;
                }
                break;
            }
            case pipeline_1.DataPipelineType.STREAM: {
                this.addDataStream(pipeline, ds);
                break;
            }
            case pipeline_1.DataPipelineType.JDBC: {
                this.createJDBCConnection(pipeline);
                break;
            }
        }
        if (pipeline.job) {
            const jobScript = utils_1.packageAsset(this, `${pipeline.name}Script`, pipeline.job.jobScript);
            pipeline.job.jobArgs['--TempDir'] = `s3://${this.logBucket.bucketName}/temp/`;
            pipeline.job.jobArgs['--spark-event-logs-path'] = `s3://${this.logBucket.bucketName}/logs/`;
            // rethink how this works not all jobs write to S3
            if (pipeline.job.destinationLocation) {
                pipeline.job.jobArgs['--DESTINATION_BUCKET'] = utils_1.getDataSetBucketName(pipeline.job.destinationLocation, ds);
            }
            const job = new glue_job_1.GlueJob(this, `${pipeline.name}-etl-job`, {
                deploymentBucket: jobScript.bucket,
                jobScript: utils_1.toS3Path(jobScript),
                name: pipeline.job.name,
                workerType: pipeline.job.workerType,
                description: pipeline.job.description,
                glueVersion: pipeline.job.glueVersion,
                jobArgs: pipeline.job.jobArgs,
                maxCapacity: pipeline.job.maxCapacity,
                maxConcurrentRuns: pipeline.job.maxConcurrentRuns,
                maxRetries: pipeline.job.maxRetries,
                numberOfWorkers: pipeline.job.numberOfWorkers,
                roleName: pipeline.job.roleName,
                timeout: pipeline.job.timeout,
                jobType: pipeline.job.jobType,
                readAccessBuckets: [
                    this.logBucket,
                ],
                writeAccessBuckets: [
                    this.logBucket,
                    s3.Bucket.fromBucketName(this, 'raw-bucket-role', ds.rawBucketName),
                    s3.Bucket.fromBucketName(this, 'refined-bucket-role', ds.refinedBucketName),
                    s3.Bucket.fromBucketName(this, 'trusted-bucket-role', ds.trustedBucketName),
                ],
            });
            new glue_job_ops_1.GlueJobOps(this, `${pipeline.name}-etl-job-ops`, {
                job: job,
            });
            if (pipeline.streamProperties) {
                this.dataStreams[pipeline.name].stream.grantRead(job.role);
            }
            new lf.CfnPermissions(this, `${pipeline.name}-create-table-perm`, {
                dataLakePrincipal: {
                    dataLakePrincipalIdentifier: job.role.roleArn,
                },
                resource: {
                    databaseResource: {
                        name: dataProduct.databaseName,
                    },
                },
                permissions: [
                    Permissions.ALTER,
                    Permissions.CREATE_TABLE,
                    Permissions.DESCRIBE,
                ],
            });
            if (pipeline.table) {
                new lf.CfnPermissions(this, `${pipeline.name}-access-table-perm`, {
                    dataLakePrincipal: {
                        dataLakePrincipalIdentifier: job.role.roleArn,
                    },
                    resource: {
                        tableResource: {
                            databaseName: dataProduct.databaseName,
                            name: pipeline.table.tableName,
                        },
                    },
                    permissions: [
                        Permissions.SELECT,
                        Permissions.DESCRIBE,
                    ],
                });
            }
        }
    }
    createJDBCConnection(pipeline) {
        if (this.vpc && this.glueSecurityGroup) {
            new glue.Connection(this, `${pipeline.name}-glue-connection`, {
                type: glue.ConnectionType.JDBC,
                connectionName: `${pipeline.name}-jdbc`,
                description: `JDBC connection for glue to use on pipeline ${pipeline.name}`,
                subnet: this.vpc.isolatedSubnets[0],
                securityGroups: [this.glueSecurityGroup],
                properties: {
                    JDBC_CONNECTION_URL: pipeline.jdbcProperties.jdbc,
                    USERNAME: pipeline.jdbcProperties.username,
                    PASSWORD: pipeline.jdbcProperties.password,
                },
            });
        }
        else {
            throw new Error('VPC required to create a JDBC pipeline.');
        }
    }
    createPolicyTagsCustomResource(policyTags, datalakeAdminRole) {
        const onEvent = new aws_lambda_python_1.PythonFunction(this, 'create-policy-tags-handler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/create-tags-handler'),
            role: this.datalakeAdminRole,
            functionName: utils_1.buildLambdaFunctionName({
                name: 'create-tags',
                resourceUse: 'cr',
                stage: this.stageName,
            }),
        });
        const myProvider = new cr.Provider(this, 'policy-tags-provider', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const outputs = new cdk.CustomResource(this, 'tag-creation-custom-resource', {
            serviceToken: myProvider.serviceToken,
            properties: {
                policyTags: policyTags,
                stackName: cdk.Stack.name,
                regionName: core_1.Aws.REGION,
                catalogId: core_1.Aws.ACCOUNT_ID,
            },
        });
        outputs.node.addDependency(datalakeAdminRole);
    }
    /**
     * @experimental
     */
    createCrossAccountGlueCatalogResourcePolicy(consumerAccountIds, dataCatalogOwnerAccountId) {
        const onCatalogEvent = new aws_lambda_python_1.PythonFunction(this, 'enable-hybrid-catalog-handler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/enable-hybrid-catalog'),
            role: this.datalakeAdminRole,
            functionName: utils_1.buildLambdaFunctionName({
                name: 'create-catalog',
                resourceUse: 'cr',
                stage: this.stageName,
            }),
        });
        const catalogProvider = new cr.Provider(this, 'hybrid-catalog-provider', {
            onEventHandler: onCatalogEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        new cdk.CustomResource(this, 'hybrid-catalog-custom-resource', {
            serviceToken: catalogProvider.serviceToken,
            properties: {
                stackName: cdk.Stack.name,
                regionName: core_1.Aws.REGION,
                consumerAccountIds: consumerAccountIds,
                producerAccountId: dataCatalogOwnerAccountId,
            },
        });
    }
    registerDataLakeLocation(dataLakeAdminRoleArn, dataLakeDbCreatorRoleArn, bucketName) {
        const name = bucketName.replace(/\W/g, '');
        const dlResource = new lf.CfnResource(this, `lf-resource-${name}`, {
            resourceArn: `arn:aws:s3:::${bucketName}`,
            useServiceLinkedRole: false,
            roleArn: dataLakeDbCreatorRoleArn,
        });
        this.createDataLocationAccessPermission(`${name}-creator`, dataLakeDbCreatorRoleArn, bucketName, dlResource);
        this.createDataLocationAccessPermission(`${name}-admin`, dataLakeAdminRoleArn, bucketName, dlResource);
        return dlResource;
    }
    createDataLocationAccessPermission(name, roleArn, bucketName, resource) {
        const perm = new lf.CfnPermissions(this, `datalake-creator-perm-${name}`, {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: roleArn,
            },
            resource: {
                dataLocationResource: {
                    s3Resource: `arn:aws:s3:::${bucketName}`,
                },
            },
            permissions: [
                Permissions.DATA_LOCATION_ACCESS,
            ],
        });
        perm.node.addDependency(resource);
        return perm;
    }
}
exports.DataLake = DataLake;
_a = JSII_RTTI_SYMBOL_1;
DataLake[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.DataLake", version: "0.0.39" };
//# sourceMappingURL=data:application/json;base64,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