"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLake = exports.LakeType = exports.Permissions = exports.Stage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const athena = require("@aws-cdk/aws-athena");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const lf = require("@aws-cdk/aws-lakeformation");
const lambda = require("@aws-cdk/aws-lambda");
const aws_lambda_python_1 = require("@aws-cdk/aws-lambda-python");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const data_set_1 = require("./data-sets/data-set");
const kinesis_ops_1 = require("./data-streams/kinesis-ops");
const kinesis_stream_1 = require("./data-streams/kinesis-stream");
const s3_delivery_stream_1 = require("./data-streams/s3-delivery-stream");
const glue_crawler_1 = require("./etl/glue-crawler");
const glue_job_1 = require("./etl/glue-job");
const glue_job_ops_1 = require("./etl/glue-job-ops");
const glue_table_1 = require("./etl/glue-table");
const data_lake_admin_1 = require("./personas/data-lake-admin");
const data_lake_creator_1 = require("./personas/data-lake-creator");
const pipeline_1 = require("./pipeline");
const utils_1 = require("./utils");
/**
 * @experimental
 */
var Stage;
(function (Stage) {
    Stage["ALPHA"] = "alpha";
    Stage["BETA"] = "beta";
    Stage["GAMMA"] = "gamma";
    Stage["PROD"] = "prod";
})(Stage = exports.Stage || (exports.Stage = {}));
/**
 * @experimental
 */
var Permissions;
(function (Permissions) {
    Permissions["ALTER"] = "ALTER";
    Permissions["CREATE_DATABASE"] = "CREATE_DATABASE";
    Permissions["CREATE_TABLE"] = "CREATE_TABLE";
    Permissions["DATA_LOCATION_ACCESS"] = "DATA_LOCATION_ACCESS";
    Permissions["DELETE"] = "DELETE";
    Permissions["DESCRIBE"] = "DESCRIBE";
    Permissions["DROP"] = "DROP";
    Permissions["INSERT"] = "INSERT";
    Permissions["SELECT"] = "SELECT";
    Permissions["ASSOCIATE"] = "ASSOCIATE";
})(Permissions = exports.Permissions || (exports.Permissions = {}));
/**
 * @experimental
 */
var LakeType;
(function (LakeType) {
    LakeType["DATA_PRODUCT"] = "DATA_PRODUCT";
    LakeType["CENTRAL_CATALOG"] = "CENTRAL_CATALOG";
    LakeType["CONSUMER"] = "CONSUMER";
    LakeType["DATA_PRODUCT_AND_CATALOG"] = "DATA_PRODUCT_AND_CATALOG";
})(LakeType = exports.LakeType || (exports.LakeType = {}));
/**
 * (experimental) A CDK construct to create a DataLake.
 *
 * @experimental
 */
class DataLake extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * @experimental
         */
        this.dataSets = {};
        /**
         * @experimental
         */
        this.dataStreams = {};
        /**
         * @experimental
         */
        this.databases = {};
        this.downloadLocations = {}; //used for the Custom Resource to allow downloading of existing datasets into datalake
        this.stageName = props.stageName;
        this.crossAccountAccess = props.crossAccountAccess ? props.crossAccountAccess : undefined;
        this.vpc = props.vpc ? props.vpc : undefined;
        this.lakeType = props.lakeType;
        if (props.logBucketProps) {
            this.logBucketProps = props.logBucketProps;
        }
        else {
            this.logBucketProps = {
                lifecycleRules: [
                    {
                        expiration: cdk.Duration.days(30),
                    },
                ],
                removalPolicy: cdk.RemovalPolicy.DESTROY,
                autoDeleteObjects: true,
            };
        }
        if (this.vpc) {
            this.glueSecurityGroup = new ec2.SecurityGroup(this, 'glue-sg', {
                description: 'Glue self referential allow in out',
                vpc: this.vpc,
                securityGroupName: utils_1.buildUniqueName({
                    name: 'glue',
                    resourceUse: 'datalake',
                    stage: this.stageName,
                }, 80),
            });
            this.glueSecurityGroup.connections.allowFrom(this.glueSecurityGroup, ec2.Port.allTcp());
            this.glueSecurityGroup.connections.allowTo(this.glueSecurityGroup, ec2.Port.allTcp());
            new cdk.CfnOutput(this, 'GlueSecurityGroupName', { value: this.glueSecurityGroup.securityGroupName });
        }
        // make this optional
        this.logBucket = new s3.Bucket(this, 'datalake-log-bucket', {
            bucketName: utils_1.buildS3BucketName({
                stage: props.stageName,
                resourceUse: 'log-bucket',
                name: props.name,
            }),
            ...this.logBucketProps,
        });
        new cdk.CfnOutput(this, 'DataLakeLogBucket', { value: this.logBucket.bucketName });
        if (props.datalakeAdminRole) {
            this.datalakeAdminRole = props.datalakeAdminRole;
        }
        else {
            this.datalakeAdminRole = new data_lake_admin_1.DataLakeAdministrator(this, `${props.name}-datalake-admin-role`, {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'datalake-admin',
                    stage: this.stageName,
                }, 60),
            }).role;
        }
        if (props.datalakeCreatorRole) {
            this.datalakeDbCreatorRole = props.datalakeCreatorRole;
        }
        else {
            this.datalakeDbCreatorRole = new data_lake_creator_1.DataLakeCreator(this, `${props.name}-datalake-creator-role`, {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'datalake-creator',
                    stage: this.stageName,
                }, 60),
            }).role;
        }
        if (props.createDefaultDatabase) {
            this.databases[props.name] = this.createDatabase(props.name);
            new cdk.CfnOutput(this, 'DataLakeDefaultDatabase', { value: props.name });
        }
        if (this.crossAccountAccess) {
            this.createCrossAccountGlueCatalogResourcePolicy(this.crossAccountAccess.consumerAccountIds, this.crossAccountAccess.dataCatalogOwnerAccountId);
        }
        if (props.createAthenaWorkgroup) {
            this.athenaWorkgroup = new athena.CfnWorkGroup(this, 'workgroup', {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'workgroup',
                    stage: this.stageName,
                }, 60),
                description: 'Default Data Lake Workgroup',
                state: 'ENABLED',
                recursiveDeleteOption: true,
                workGroupConfiguration: {
                    enforceWorkGroupConfiguration: true,
                    resultConfiguration: {
                        outputLocation: `s3://${this.logBucket.bucketName}/results/`,
                    },
                    engineVersion: {
                        selectedEngineVersion: 'Athena engine version 2',
                        effectiveEngineVersion: 'Athena engine version 2',
                    },
                },
            });
            new cdk.CfnOutput(this, 'DataLakeAthenaWorkgroup', { value: this.athenaWorkgroup.name });
        }
        if (props.policyTags) {
            this.createPolicyTagsCustomResource(props.policyTags, this.datalakeAdminRole);
        }
        if (props.dataProducts && props.dataProducts.length > 0) {
            props.dataProducts.forEach((product) => {
                this.databases[product.databaseName] = this.createDatabase(product.databaseName);
                product.pipelines.forEach((pipe) => {
                    this.addPipeline(pipe, product);
                });
            });
        }
    }
    /**
     * @experimental
     */
    createDownloaderCustomResource(stageName) {
        // download the data sets with the custom resource after successfull creation of resource
        const onEvent = new aws_lambda_python_1.PythonFunction(this, 'DataloaderHandler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/download-data'),
            timeout: cdk.Duration.minutes(15),
            functionName: utils_1.buildLambdaFunctionName({
                name: 'load-data',
                resourceUse: 'cr',
                stage: stageName,
            }),
        });
        // create readable and writable buckets for the datasets and set the appropriate S3 access
        onEvent.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:*'],
            resources: ['*'],
        }));
        const dataLoadProvider = new cr.Provider(this, 'DataloaderProvider', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        // CR to download the static datasets form the dataSets var passed in.
        new cdk.CustomResource(this, 'LoadDatalakeCustomResource', {
            serviceToken: dataLoadProvider.serviceToken,
            properties: {
                dataSets: this.downloadLocations,
                stackName: cdk.Stack.name,
                regionName: core_1.Aws.REGION,
            },
        });
    }
    createDatabase(databaseName) {
        const db = new glue.Database(this, `${databaseName}-database`, {
            databaseName: `${databaseName}`,
        });
        const dbPerm = new lf.CfnPermissions(this, `${databaseName}-lf-db-creator-permission`, {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: this.datalakeDbCreatorRole.roleArn,
            },
            resource: {
                databaseResource: {
                    name: databaseName,
                },
            },
            permissions: [
                Permissions.ALTER,
                Permissions.CREATE_TABLE,
                Permissions.DROP,
            ],
        });
        dbPerm.node.addDependency(db);
        return db;
    }
    addDataStream(pipeline, dataSet) {
        const schemaName = pipeline.name;
        const dataStreamStack = new cdk.NestedStack(cdk.Stack.of(this), `${schemaName}-datastream-stack`);
        if (!pipeline.streamProperties) {
            throw Error("Cannot create a stream pipeline without 'sreamProperties'");
        }
        this.dataStreams[schemaName] = new kinesis_stream_1.KinesisStream(dataStreamStack, 'DataStream', {
            shardCount: 1,
            streamName: pipeline.streamProperties.streamName,
        });
        const deliveryStream = new s3_delivery_stream_1.S3DeliveryStream(dataStreamStack, 'deliveryStream', {
            compression: s3_delivery_stream_1.CompressionType.UNCOMPRESSED,
            kinesisStream: this.dataStreams[schemaName].stream,
            s3Bucket: s3.Bucket.fromBucketName(this, 'get-bucket-for-kinesis', utils_1.getDataSetBucketName(pipeline.dataSetDropLocation, dataSet)),
            s3Prefix: pipeline.destinationPrefix,
        });
        new kinesis_ops_1.KinesisOps(dataStreamStack, 'kinesis-ops', {
            stream: this.dataStreams[schemaName],
            deliveryStream: deliveryStream,
        });
        if (pipeline.streamProperties.lambdaDataGenerator) {
            const dataGeneratorFunction = new lambda.Function(dataStreamStack, 'data-generator-function', {
                code: pipeline.streamProperties.lambdaDataGenerator.code,
                handler: pipeline.streamProperties.lambdaDataGenerator.handler,
                timeout: pipeline.streamProperties.lambdaDataGenerator.timeout,
                runtime: pipeline.streamProperties.lambdaDataGenerator.runtime,
                functionName: pipeline.streamProperties.lambdaDataGenerator.functionName,
                environment: {
                    KINESIS_STREAM: this.dataStreams[schemaName].stream.streamName,
                },
            });
            this.dataStreams[schemaName].stream.grantWrite(dataGeneratorFunction);
            const rule = new events.Rule(this, 'Rule', {
                schedule: pipeline.streamProperties.lambdaDataGenerator.schedule,
                ruleName: pipeline.streamProperties.lambdaDataGenerator.ruleName,
            });
            rule.addTarget(new targets.LambdaFunction(dataGeneratorFunction));
        }
        return this.dataStreams[schemaName];
    }
    addPipeline(pipeline, dataProduct) {
        const schemaName = pipeline.name;
        const dataSetStack = dataProduct.accountId == core_1.Aws.ACCOUNT_ID ? new cdk.NestedStack(cdk.Stack.of(this), `${schemaName}-dataset-stack`) : this;
        // create the dataSet
        this.dataSets[schemaName] = new data_set_1.DataSet(dataSetStack, schemaName, {
            pipeline: pipeline,
            dataProduct: dataProduct,
            logBucket: this.logBucket,
            stage: this.stageName,
            accountId: core_1.Aws.ACCOUNT_ID,
            region: core_1.Aws.REGION,
            s3BucketProps: dataProduct.s3BucketProps,
        });
        const ds = this.dataSets[schemaName];
        const catelogAccountId = dataProduct.dataCatalogAccountId ? dataProduct.dataCatalogAccountId : core_1.Aws.ACCOUNT_ID;
        if (this.lakeType === LakeType.DATA_PRODUCT || this.lakeType === LakeType.DATA_PRODUCT_AND_CATALOG) {
            this.createPipelineResources(pipeline, dataProduct, ds);
        }
        // only create the table if the lake has a catelog
        if (pipeline.table && (this.lakeType === LakeType.CENTRAL_CATALOG || this.lakeType === LakeType.DATA_PRODUCT_AND_CATALOG)) {
            const table = new glue_table_1.GlueTable(this, `${pipeline.name}-table`, {
                catalogId: pipeline.table.catalogId,
                columns: pipeline.table.columns,
                databaseName: this.databases[dataProduct.databaseName].databaseName,
                description: pipeline.table.description,
                inputFormat: pipeline.table.inputFormat,
                outputFormat: pipeline.table.outputFormat,
                parameters: pipeline.table.parameters,
                partitionKeys: pipeline.table.partitionKeys,
                s3Location: `s3://${utils_1.getDataSetBucketName(pipeline.dataSetDropLocation, ds)}/${pipeline.destinationPrefix}`,
                serdeParameters: pipeline.table.serdeParameters,
                serializationLibrary: pipeline.table.serializationLibrary,
                tableName: pipeline.table.tableName,
            });
            table.node.addDependency(this.databases[dataProduct.databaseName]);
        }
        // find the correct metadata catalog account
        if (catelogAccountId == core_1.Aws.ACCOUNT_ID) {
            // refactor to only register the needed buckets from the data product account
            const rawDlResource = this.registerDataLakeLocation(this.datalakeAdminRole.roleArn, this.datalakeDbCreatorRole.roleArn, ds.rawBucketName);
            const trustedDlResource = this.registerDataLakeLocation(this.datalakeAdminRole.roleArn, this.datalakeDbCreatorRole.roleArn, ds.trustedBucketName);
            const refinedDlResource = this.registerDataLakeLocation(this.datalakeAdminRole.roleArn, this.datalakeDbCreatorRole.roleArn, ds.refinedBucketName);
            if (!pipeline.table) {
                const bucketName = utils_1.getDataSetBucketName(pipeline.dataSetDropLocation, ds);
                const name = bucketName.replace(/\W/g, '');
                // only create a crawler for the drop location of the data in the data product of the pipeline
                const crawler = new glue_crawler_1.GlueCrawler(this, `data-lake-crawler-${name}`, {
                    name: utils_1.buildGlueCrawlerName({
                        stage: this.stageName,
                        resourceUse: 'crawler',
                        name: pipeline.name,
                    }),
                    databaseName: dataProduct.databaseName,
                    bucketName: bucketName,
                    bucketPrefix: pipeline.destinationPrefix,
                    roleName: utils_1.buildRoleName({
                        stage: this.stageName,
                        resourceUse: 'crawler-role',
                        name: pipeline.name,
                    }),
                });
                crawler.node.addDependency(rawDlResource);
                crawler.node.addDependency(trustedDlResource);
                crawler.node.addDependency(refinedDlResource);
            }
        }
    }
    // this is a jumbled mess clean up once refecto
    createPipelineResources(pipeline, dataProduct, ds) {
        switch (pipeline.type) {
            case pipeline_1.DataPipelineType.S3: {
                if (ds.downloadLocations) {
                    this.downloadLocations[pipeline.name] = ds.downloadLocations;
                }
                break;
            }
            case pipeline_1.DataPipelineType.STREAM: {
                this.addDataStream(pipeline, ds);
                break;
            }
            case pipeline_1.DataPipelineType.JDBC: {
                this.createJDBCConnection(pipeline);
                break;
            }
        }
        if (pipeline.job) {
            const jobScript = utils_1.packageAsset(this, `${pipeline.name}Script`, pipeline.job.jobScript);
            pipeline.job.jobArgs['--TempDir'] = `s3://${this.logBucket.bucketName}/temp/`;
            pipeline.job.jobArgs['--spark-event-logs-path'] = `s3://${this.logBucket.bucketName}/logs/`;
            // rethink how this works not all jobs write to S3
            if (pipeline.job.destinationLocation) {
                pipeline.job.jobArgs['--DESTINATION_BUCKET'] = utils_1.getDataSetBucketName(pipeline.job.destinationLocation, ds);
            }
            const job = new glue_job_1.GlueJob(this, `${pipeline.name}-etl-job`, {
                deploymentBucket: jobScript.bucket,
                jobScript: utils_1.toS3Path(jobScript),
                name: pipeline.job.name,
                workerType: pipeline.job.workerType,
                description: pipeline.job.description,
                glueVersion: pipeline.job.glueVersion,
                jobArgs: pipeline.job.jobArgs,
                maxCapacity: pipeline.job.maxCapacity,
                maxConcurrentRuns: pipeline.job.maxConcurrentRuns,
                maxRetries: pipeline.job.maxRetries,
                numberOfWorkers: pipeline.job.numberOfWorkers,
                roleName: pipeline.job.roleName,
                timeout: pipeline.job.timeout,
                jobType: pipeline.job.jobType,
                readAccessBuckets: [
                    this.logBucket,
                ],
                writeAccessBuckets: [
                    this.logBucket,
                    s3.Bucket.fromBucketName(this, 'raw-bucket-role', ds.rawBucketName),
                    s3.Bucket.fromBucketName(this, 'refined-bucket-role', ds.refinedBucketName),
                    s3.Bucket.fromBucketName(this, 'trusted-bucket-role', ds.trustedBucketName),
                ],
            });
            new glue_job_ops_1.GlueJobOps(this, `${pipeline.name}-etl-job-ops`, {
                job: job,
            });
            if (pipeline.streamProperties) {
                this.dataStreams[pipeline.name].stream.grantRead(job.role);
            }
            new lf.CfnPermissions(this, `${pipeline.name}-create-table-perm`, {
                dataLakePrincipal: {
                    dataLakePrincipalIdentifier: job.role.roleArn,
                },
                resource: {
                    databaseResource: {
                        name: dataProduct.databaseName,
                    },
                },
                permissions: [
                    Permissions.ALTER,
                    Permissions.CREATE_TABLE,
                    Permissions.DESCRIBE,
                ],
            });
            if (pipeline.table) {
                new lf.CfnPermissions(this, `${pipeline.name}-access-table-perm`, {
                    dataLakePrincipal: {
                        dataLakePrincipalIdentifier: job.role.roleArn,
                    },
                    resource: {
                        tableResource: {
                            databaseName: dataProduct.databaseName,
                            name: pipeline.table.tableName,
                        },
                    },
                    permissions: [
                        Permissions.SELECT,
                        Permissions.DESCRIBE,
                    ],
                });
            }
        }
    }
    createJDBCConnection(pipeline) {
        if (this.vpc && this.glueSecurityGroup) {
            new glue.Connection(this, `${pipeline.name}-glue-connection`, {
                type: glue.ConnectionType.JDBC,
                connectionName: `${pipeline.name}-jdbc`,
                description: `JDBC connection for glue to use on pipeline ${pipeline.name}`,
                subnet: this.vpc.isolatedSubnets[0],
                securityGroups: [this.glueSecurityGroup],
                properties: {
                    JDBC_CONNECTION_URL: pipeline.jdbcProperties.jdbc,
                    USERNAME: pipeline.jdbcProperties.username,
                    PASSWORD: pipeline.jdbcProperties.password,
                },
            });
        }
        else {
            throw new Error('VPC required to create a JDBC pipeline.');
        }
    }
    createPolicyTagsCustomResource(policyTags, datalakeAdminRole) {
        const onEvent = new aws_lambda_python_1.PythonFunction(this, 'create-policy-tags-handler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/create-tags-handler'),
            role: this.datalakeAdminRole,
            functionName: utils_1.buildLambdaFunctionName({
                name: 'create-tags',
                resourceUse: 'cr',
                stage: this.stageName,
            }),
        });
        const myProvider = new cr.Provider(this, 'policy-tags-provider', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const outputs = new cdk.CustomResource(this, 'tag-creation-custom-resource', {
            serviceToken: myProvider.serviceToken,
            properties: {
                policyTags: policyTags,
                stackName: cdk.Stack.name,
                regionName: core_1.Aws.REGION,
                catalogId: core_1.Aws.ACCOUNT_ID,
            },
        });
        outputs.node.addDependency(datalakeAdminRole);
    }
    /**
     * @experimental
     */
    createCrossAccountGlueCatalogResourcePolicy(consumerAccountIds, dataCatalogOwnerAccountId) {
        const onCatalogEvent = new aws_lambda_python_1.PythonFunction(this, 'enable-hybrid-catalog-handler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/enable-hybrid-catalog'),
            role: this.datalakeAdminRole,
            functionName: utils_1.buildLambdaFunctionName({
                name: 'create-catalog',
                resourceUse: 'cr',
                stage: this.stageName,
            }),
        });
        const catalogProvider = new cr.Provider(this, 'hybrid-catalog-provider', {
            onEventHandler: onCatalogEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        new cdk.CustomResource(this, 'hybrid-catalog-custom-resource', {
            serviceToken: catalogProvider.serviceToken,
            properties: {
                stackName: cdk.Stack.name,
                regionName: core_1.Aws.REGION,
                consumerAccountIds: consumerAccountIds,
                producerAccountId: dataCatalogOwnerAccountId,
            },
        });
    }
    registerDataLakeLocation(dataLakeAdminRoleArn, dataLakeDbCreatorRoleArn, bucketName) {
        const name = bucketName.replace(/\W/g, '');
        const dlResource = new lf.CfnResource(this, `lf-resource-${name}`, {
            resourceArn: `arn:aws:s3:::${bucketName}`,
            useServiceLinkedRole: false,
            roleArn: dataLakeDbCreatorRoleArn,
        });
        this.createDataLocationAccessPermission(`${name}-creator`, dataLakeDbCreatorRoleArn, bucketName, dlResource);
        this.createDataLocationAccessPermission(`${name}-admin`, dataLakeAdminRoleArn, bucketName, dlResource);
        return dlResource;
    }
    createDataLocationAccessPermission(name, roleArn, bucketName, resource) {
        const perm = new lf.CfnPermissions(this, `datalake-creator-perm-${name}`, {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: roleArn,
            },
            resource: {
                dataLocationResource: {
                    s3Resource: `arn:aws:s3:::${bucketName}`,
                },
            },
            permissions: [
                Permissions.DATA_LOCATION_ACCESS,
            ],
        });
        perm.node.addDependency(resource);
        return perm;
    }
}
exports.DataLake = DataLake;
_a = JSII_RTTI_SYMBOL_1;
DataLake[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.DataLake", version: "0.0.36" };
//# sourceMappingURL=data:application/json;base64,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