"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisOps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
/**
 * @experimental
 */
class KinesisOps extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.stream = props.stream;
        this.streamName = props.stream.stream.streamName;
        this.deliveryStream = props.deliveryStream;
        this.dashboard = new cloudwatch.Dashboard(this, 'dashboard', {
            dashboardName: `Kinesis_${this.streamName}`,
        });
        this.inputStreamIteratorAgeCriticalAlarm = new cloudwatch.Alarm(this, 'inputStream-iterator-age-critical-alarm', {
            alarmName: `${this.streamName} inputStream IteratorAge Long`,
            alarmDescription: 'Alarms if maximum iterator age of inputStream is more than 10 minute',
            metric: this.stream.metricGetRecordsIteratorAgeMilliseconds(),
            threshold: 600000,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            period: cdk.Duration.minutes(5),
            evaluationPeriods: 12,
            ...(props.inputStreamIteratorAgeCritical || {}),
        });
        this.inputStreamIteratorAgeWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-iterator-age-warning-alarm', {
            alarmName: `${this.streamName} inputStream IteratorAge Long Warning`,
            alarmDescription: 'Alarms if maximum iterator age of inputStream is more than 5 minute',
            metric: this.stream.metricGetRecordsIteratorAgeMilliseconds(),
            threshold: 30000,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            period: cdk.Duration.minutes(5),
            evaluationPeriods: 12,
            ...(props.inputStreamIteratorAgeWarning || {}),
        });
        this.inputStreamReadThroughputWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-read-throughput-warning-alarm', {
            alarmName: `${this.streamName} inputStream ReadThroughput Exceed Warning`,
            alarmDescription: 'Alarms if read provisioned throughput of inputStream is exceeded for least 2 hours',
            metric: this.stream.metricReadProvisionedThroughputExceeded(),
            threshold: 0.15,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            period: cdk.Duration.minutes(10),
            evaluationPeriods: 12,
            ...(props.inputStreamReadThroughputWarning || {}),
        });
        this.inputStreamWriteThroughputWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-write-throughput-warning-alarm', {
            alarmName: `${this.streamName} inputStream WriteThroughput Exceed Warning`,
            alarmDescription: 'Alarms if write provisioned throughput of inputStream is exceeded for least 12 hours',
            metric: this.stream.metricWriteProvisionedThroughputExceeded(),
            threshold: 0.15,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            period: cdk.Duration.minutes(60),
            evaluationPeriods: 12,
            ...(props.inputStreamWriteThroughputWarning || {}),
        });
        this.inputStreamGetRecordsWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-get-records-warning-alarm', {
            alarmName: `${this.streamName} inputStream GetRecords Success Low Warning`,
            alarmDescription: 'Alarms if GetRecords of inputStream not very successful for least 30 minutes',
            metric: this.stream.metricGetRecordsSuccess(),
            threshold: 0.9,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            period: cdk.Duration.minutes(5),
            evaluationPeriods: 6,
            ...(props.inputStreamGetRecordsWarning || {}),
        });
        this.inputStreamPutRecordsWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-put-records-warning-alarm', {
            alarmName: `${this.streamName} inputStream PutRecords Success Low Warning`,
            alarmDescription: 'Alarms if PutRecords of inputStream not very successful for least 12 hours',
            metric: this.stream.metricPutRecordsSuccess(),
            threshold: 0.9,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            period: cdk.Duration.minutes(60),
            evaluationPeriods: 12,
            ...(props.inputStreamPutRecordsWarning || {}),
        });
        this.firehoseDeliveryToS3WarningAlarm = new cloudwatch.Alarm(this, 'deliveryStream-delivery-to-s3-warning-alarm', {
            alarmName: `${this.streamName} Firehose DeliveryToS3 Failure Warning`,
            alarmDescription: 'Alarms if firehose DeliveryToS3 failed for atleast 60 minutes',
            metric: this.deliveryStream.metricDeliveryToS3Success(),
            statistic: cloudwatch.Statistic.AVERAGE,
            threshold: 1,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            period: cdk.Duration.minutes(5),
            evaluationPeriods: 12,
            treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
            ...(props.firehoseDeliveryToS3Warning || {}),
        });
        this.firehoseDeliveryToS3CriticalAlarm = new cloudwatch.Alarm(this, 'deliveryStream-delivery-to-s3-critical-alarm', {
            alarmName: `${this.streamName} Firehose DeliveryToS3 Failure Critical`,
            alarmDescription: 'Alarms if firehose DeliveryToS3 failed for atleast 24 hours',
            metric: this.deliveryStream.metricDeliveryToS3Success(),
            statistic: cloudwatch.Statistic.AVERAGE,
            threshold: 1,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            period: cdk.Duration.hours(1),
            evaluationPeriods: 24,
            treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
            ...(props.firehoseDeliveryToS3Critical || {}),
        });
        this.alarmsSev2 = [
            this.inputStreamIteratorAgeCriticalAlarm,
            this.firehoseDeliveryToS3CriticalAlarm,
        ];
        this.alarmsSev3 = [
            this.inputStreamIteratorAgeWarningAlarm,
            this.inputStreamReadThroughputWarningAlarm,
            this.inputStreamWriteThroughputWarningAlarm,
            this.inputStreamGetRecordsWarningAlarm,
            this.inputStreamPutRecordsWarningAlarm,
            this.firehoseDeliveryToS3WarningAlarm,
        ];
        this.setupDashboard();
    }
    alarmWidgets(alarms, severity) {
        return alarms.map(alarm => new cloudwatch.AlarmWidget({
            title: `${alarm.alarmName} - Sev ${severity}`,
            alarm,
        }));
    }
    addWidgets(widgets) {
        for (let i = 0; i < widgets.length; i += 4) {
            this.dashboard.addWidgets(...widgets.slice(i, i + 4));
        }
    }
    setupDashboard() {
        const widgets = [
            new cloudwatch.GraphWidget({
                title: 'Kinesis Stream (Ingress)',
                left: [
                    this.stream.metricIncomingRecords({
                        label: 'Incoming Records',
                        statistic: 'sum',
                    }),
                ],
                right: [
                    this.stream.metricIncomingBytes({
                        label: 'Incoming Bytes',
                        statistic: 'sum',
                    }),
                ],
            }),
            new cloudwatch.GraphWidget({
                title: 'Kinesis Stream (Throttling)',
                left: [
                    this.stream.metricGetRecordsSuccess(),
                    this.stream.metricPutRecordsSuccess(),
                ],
                right: [
                    this.stream.metricReadProvisionedThroughputExceeded({
                        label: 'Throttled Reads',
                        statistic: 'sum',
                    }),
                    this.stream.metricWriteProvisionedThroughputExceeded({
                        label: 'Throttled Writes',
                        statistic: 'sum',
                    }),
                ],
            }),
            new cloudwatch.GraphWidget({
                title: 'Kinesis Stream (Delay)',
                left: [
                    this.stream.metricGetRecordsIteratorAgeMilliseconds({
                        label: 'Time-lag behind Kinesis Stream',
                        statistic: 'max',
                    }),
                ],
            }),
            new cloudwatch.GraphWidget({
                title: 'Firehose Delivery Stream (Ingress/Egress)',
                left: [
                    this.deliveryStream.metricIncomingRecords({
                        label: 'Incoming Records',
                        statistic: 'sum',
                    }),
                    this.deliveryStream.metricDeliveryToS3Records({
                        label: 'Outgoing Records',
                        statistic: 'sum',
                    }),
                ],
                right: [
                    this.deliveryStream.metricDeliveryToS3Success(),
                ],
            }),
            new cloudwatch.GraphWidget({
                title: 'Firehose Data Freshness',
                left: [
                    this.deliveryStream.metricDeliveryToS3DataFreshness({
                        label: 'Freshness',
                        statistic: 'max',
                        period: cdk.Duration.minutes(5),
                    }),
                ],
            }),
        ];
        widgets.push(...this.alarmWidgets(this.alarmsSev2, 2));
        widgets.push(...this.alarmWidgets(this.alarmsSev3, 3));
        this.addWidgets(widgets);
    }
}
exports.KinesisOps = KinesisOps;
_a = JSII_RTTI_SYMBOL_1;
KinesisOps[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.KinesisOps", version: "0.0.36" };
//# sourceMappingURL=data:application/json;base64,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