"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlueJob = exports.GlueJobType = exports.GlueVersion = exports.GlueWorkerType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const eventtargets = require("@aws-cdk/aws-events-targets");
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
/**
 * @experimental
 */
var GlueWorkerType;
(function (GlueWorkerType) {
    GlueWorkerType["STANDARD"] = "Standard";
    GlueWorkerType["G1_X"] = "G.1X";
    GlueWorkerType["G2_X"] = "G.2X";
})(GlueWorkerType = exports.GlueWorkerType || (exports.GlueWorkerType = {}));
/**
 * @experimental
 */
var GlueVersion;
(function (GlueVersion) {
    GlueVersion["V_0"] = "0.9";
    GlueVersion["V_1"] = "1.0";
    GlueVersion["V_2"] = "2.0";
})(GlueVersion = exports.GlueVersion || (exports.GlueVersion = {}));
/**
 * @experimental
 */
var GlueJobType;
(function (GlueJobType) {
    GlueJobType["GLUE_ETL"] = "glueetl";
    GlueJobType["GLUE_STREAMING"] = "gluestreaming";
})(GlueJobType = exports.GlueJobType || (exports.GlueJobType = {}));
/**
 * @experimental
 */
class GlueJob extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * @experimental
         */
        this.allExecutionAttemptsFailedEventSource = 'custom.aws.glue.allExecutionAttemptsFailed';
        /**
         * @experimental
         */
        this.allExecutionAttemptsFailedEventDetailType = 'All Execution Attempts Failed';
        this.role = this.createGlueJobRole(props);
        this.job = new glue.CfnJob(this, `${props.name}-glue-job`, {
            name: props.name,
            description: props.description,
            workerType: props.workerType,
            numberOfWorkers: props.numberOfWorkers,
            role: this.role.roleName,
            maxRetries: props.maxRetries || 0,
            executionProperty: {
                maxConcurrentRuns: props.maxConcurrentRuns || 3,
            },
            glueVersion: props.glueVersion || GlueVersion.V_1,
            command: {
                pythonVersion: '3',
                scriptLocation: props.jobScript,
                name: props.jobType,
            },
            timeout: props.timeout || GlueJob.DAY_IN_MINUTES,
            defaultArguments: {
                '--job-language': 'python',
                '--enable-metrics': true,
                '--enable-continuous-cloudwatch-log': true,
                '--region': cdk.Stack.of(this).region,
                '--enable-glue-datacatalog': true,
                '--enable-continuous-log-filter': true,
                '--enable-spark-ui': true,
                ...props.jobArgs,
            },
        });
        this.name = props.name;
        this.metricSuccessRule = this.jobRule('SuccessRule', this.name, 'SUCCEEDED');
        this.metricFailureRule = this.jobRule('FailureRule', this.name, 'FAILED');
        this.metricTimeoutRule = this.jobRule('TimeoutRule', this.name, 'TIMEOUT');
        this.executionFailureRule = new events.Rule(scope, `${this.name}-execution-failure-rule`, {
            description: `Glue job ${this.name} failed or timed out on an attempt. There might be job retries after this error.`,
            eventPattern: {
                source: ['aws.glue'],
                detailType: ['Glue Job State Change'],
                detail: {
                    state: ['FAILED', 'TIMEOUT'],
                    jobName: [this.name],
                },
            },
        });
        this.lambdaFunction = this.createLambdaFunction();
        this.executionFailureRule.addTarget(new eventtargets.LambdaFunction(this.lambdaFunction));
        this.allExecutionAttemptsFailedRule = new events.Rule(this, `${this.name}-all-execution-attempts-failed-rule`, {
            description: `Glue job ${this.name} failed or timed out on the last attempt. There will be no retries of the job after this error.`,
            eventPattern: {
                source: [this.allExecutionAttemptsFailedEventSource],
                detailType: [this.allExecutionAttemptsFailedEventDetailType],
                detail: {
                    jobName: [this.name],
                },
            },
        });
    }
    createGlueJobRole(props) {
        const role = new iam.Role(this, 'Role', {
            roleName: props.roleName || props.name + 'Role',
            assumedBy: new iam.ServicePrincipal('glue'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AWSGlueConsoleFullAccess'),
            ],
        });
        role.addToPolicy(new iam.PolicyStatement({ actions: ['lakeformation:GetDataAccess'], resources: ['*'] }));
        props.deploymentBucket.grantRead(role);
        if (props.readAccessBuckets) {
            props.readAccessBuckets.forEach(bucket => {
                bucket.grantRead(role);
            });
        }
        if (props.writeAccessBuckets) {
            props.writeAccessBuckets.forEach(bucket => {
                bucket.grantWrite(role);
            });
        }
        return role;
    }
    jobRule(id, jobName, ...states) {
        return new events.Rule(this, id, {
            ruleName: jobName + states.join(''),
            description: `Event triggered when Glue job ${jobName} is in ${states.join(' or ')} state(s)`,
            eventPattern: {
                source: ['aws.glue'],
                detailType: ['Glue Job State Change'],
                detail: {
                    state: states,
                    jobName: [jobName],
                },
            },
        });
    }
    /**
     * @experimental
     */
    metricSuccess(props) {
        return this.ruleMetric(this.metricSuccessRule, props);
    }
    /**
     * @experimental
     */
    metricFailure(props) {
        return this.ruleMetric(this.metricFailureRule, props);
    }
    /**
     * @experimental
     */
    metricAllExecutionAttemptsFailed(props) {
        return new cloudwatch.Metric({
            metricName: 'TriggeredRules',
            namespace: 'AWS/Events',
            dimensions: {
                RuleName: this.allExecutionAttemptsFailedRule.ruleName,
            },
            statistic: 'Sum',
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * @experimental
     */
    metricTimeout(props) {
        return this.ruleMetric(this.metricTimeoutRule, props);
    }
    ruleMetric({ ruleName }, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Events',
            metricName: 'TriggeredRules',
            dimensions: { RuleName: ruleName },
            statistic: cloudwatch.Statistic.SUM,
            ...props,
        }).attachTo(this);
    }
    /**
     * @experimental
     */
    metric(metricName, dimensionType, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Glue',
            metricName,
            dimensions: {
                JobName: this.name,
                JobRunId: 'ALL',
                Type: dimensionType,
            },
            ...props,
        });
    }
    /**
     * @experimental
     */
    jvmHeapUsageMetric(props) {
        return this.metric('glue.ALL.jvm.heap.usage', 'gauge', props);
    }
    /**
     * @experimental
     */
    elapsedTimeMetric(props) {
        return this.metric('glue.driver.aggregate.elapsedTime', 'count', props);
    }
    /**
     * @experimental
     */
    diskSpaceUsedMbMetric(props) {
        return this.metric('glue.driver.BlockManager.disk.diskSpaceUsed_MB', 'gauge', props);
    }
    /**
     * @experimental
     */
    runTimeInMiliseconds(props) {
        return this.metric('glue.driver.aggregate.elapsedTime', 'gauge', props);
    }
    createLambdaFunction() {
        const lambdaFunction = new lambda.SingletonFunction(this, `GlueExecutionFailListenerLambdaSingleton${this.name}`, {
            description: 'Checks if an error of a Glue job was on the last attempt (no more retries) in which case the function sends out an event.',
            environment: {
                eventToSendSource: this.allExecutionAttemptsFailedEventSource,
                eventToSendDetailType: this.allExecutionAttemptsFailedEventDetailType,
            },
            uuid: 'GlueExecutionFailListenerLambda',
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: cdk.Duration.minutes(1),
            code: lambda.Code.fromInline(`
import boto3
import json
import os
import re

def handler(event, context):
    try:
        jobRunId = event['detail']['jobRunId']
        jobName = event['detail']['jobName']
    except:
        raise Exception(f'Received an malformed event. ({event})')
        
    # get the current execution attempt, we parse it from the jobRunId which has a _attempt_# suffix on retries
    try:
        curExecutionAttempt = int(re.findall('_attempt_(\\d*)$', jobRunId)[0])
    except IndexError:
        curExecutionAttempt = 0
    
    # get the number of MaxRetries for this glue job
    try:
        glue_client = boto3.client('glue')
        maxRetries = glue_client.get_job(JobName=jobName)['Job']['MaxRetries']
    except Exception as e:
        raise Exception(f'Failed to access the Glue API to get the MaxRetries parameter. ({e})')
        
    # is this the last execution? if yes we send out the event
    isLastExecutionAttempt = curExecutionAttempt == maxRetries
    print(f'Job name: {jobName}, is last execution attempt: {isLastExecutionAttempt}, current attempt: {curExecutionAttempt}, max retry attempts: {maxRetries}')
    if isLastExecutionAttempt:
        event_client = boto3.client('events')
        event_client.put_events(Entries=[{
            'Source': os.environ['eventToSendSource'],
            'Detail': json.dumps(event['detail']),
            'DetailType': os.environ['eventToSendDetailType']
        }])
    `),
        });
        const region = cdk.Stack.of(this).region;
        const accountId = cdk.Stack.of(this).account;
        lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [`arn:aws:events:${region}:${accountId}:event-bus/default`],
        }));
        lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['glue:GetJob'],
            resources: [`arn:aws:glue:${region}:${accountId}:job/${this.name}`],
        }));
        return lambdaFunction;
    }
}
exports.GlueJob = GlueJob;
_a = JSII_RTTI_SYMBOL_1;
GlueJob[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.GlueJob", version: "0.0.36" };
GlueJob.DAY_IN_MINUTES = 1440;
//# sourceMappingURL=data:application/json;base64,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