"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScheduledJobWorkflow = void 0;
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const cdk = require("@aws-cdk/core");
const utils_1 = require("../utils");
class ScheduledJobWorkflow extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stateMachineRole = new iam.Role(scope, 'StateMachineJobExecutionRole', {
            roleName: utils_1.buildRoleName({
                name: props.name,
                resourceUse: 'datalake',
                stage: props.stageName,
            }),
            assumedBy: new iam.ServicePrincipal('states'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3FullAccess'),
            ],
        });
        const jobState = new tasks.GlueStartJobRun(this, 'GlueExecutionStep', {
            glueJobName: props.jobName,
            integrationPattern: sfn.IntegrationPattern.RUN_JOB,
            arguments: sfn.TaskInput.fromObject(props.jobArguments),
            timeout: props.jobTimeout,
            resultPath: '$.jobOutput',
        });
        const stateMachineDefinition = sfn.Chain.start(jobState).toSingleState('Run Job pipeline', {
            comment: 'Container for glue job states',
        }).addCatch(this.getStateMachineFailureHandlerState(), {});
        this.stateMachine = new sfn.StateMachine(this, 'GlueStateMachine', {
            definition: stateMachineDefinition,
            logs: {
                destination: new logs.LogGroup(this, `DataLakeWorkflow-${props.name}`, {
                    retention: logs.RetentionDays.SIX_MONTHS,
                }),
                includeExecutionData: true,
                level: sfn.LogLevel.ERROR,
            },
            tracingEnabled: true,
            role: stateMachineRole,
        });
        this.rule = new events.Rule(this, 'Rule', {
            schedule: props.schedule,
            ruleName: utils_1.buildEventRuleName({
                name: props.name,
                resourceUse: 'datalake',
                stage: props.stageName,
            }),
        });
        this.rule.addTarget(new targets.SfnStateMachine(this.stateMachine));
    }
    getStateMachineFailureHandlerState() {
        return new sfn.Fail(this, 'Handle failures', {
            comment: 'Handle failures for entire state machine',
        });
    }
}
exports.ScheduledJobWorkflow = ScheduledJobWorkflow;
//# sourceMappingURL=data:application/json;base64,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