import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IAccelerator } from './accelerator';
import { EndpointGroup, EndpointGroupOptions } from './endpoint-group';
/**
 * Interface of the Listener.
 *
 * @stability stable
 */
export interface IListener extends cdk.IResource {
    /**
     * The ARN of the listener.
     *
     * @stability stable
     * @attribute true
     */
    readonly listenerArn: string;
}
/**
 * Construct options for Listener.
 *
 * @stability stable
 */
export interface ListenerOptions {
    /**
     * Name of the listener.
     *
     * @default - logical ID of the resource
     * @stability stable
     */
    readonly listenerName?: string;
    /**
     * The list of port ranges for the connections from clients to the accelerator.
     *
     * @stability stable
     */
    readonly portRanges: PortRange[];
    /**
     * The protocol for the connections from clients to the accelerator.
     *
     * @default ConnectionProtocol.TCP
     * @stability stable
     */
    readonly protocol?: ConnectionProtocol;
    /**
     * Client affinity to direct all requests from a user to the same endpoint.
     *
     * If you have stateful applications, client affinity lets you direct all
     * requests from a user to the same endpoint.
     *
     * By default, each connection from each client is routed to seperate
     * endpoints. Set client affinity to SOURCE_IP to route all connections from
     * a single client to the same endpoint.
     *
     * @default ClientAffinity.NONE
     * @stability stable
     */
    readonly clientAffinity?: ClientAffinity;
}
/**
 * Construct properties for Listener.
 *
 * @stability stable
 */
export interface ListenerProps extends ListenerOptions {
    /**
     * The accelerator for this listener.
     *
     * @stability stable
     */
    readonly accelerator: IAccelerator;
}
/**
 * The list of port ranges for the connections from clients to the accelerator.
 *
 * @stability stable
 */
export interface PortRange {
    /**
     * The first port in the range of ports, inclusive.
     *
     * @stability stable
     */
    readonly fromPort: number;
    /**
     * The last port in the range of ports, inclusive.
     *
     * @default - same as `fromPort`
     * @stability stable
     */
    readonly toPort?: number;
}
/**
 * The protocol for the connections from clients to the accelerator.
 *
 * @stability stable
 */
export declare enum ConnectionProtocol {
    /**
     * TCP.
     *
     * @stability stable
     */
    TCP = "TCP",
    /**
     * UDP.
     *
     * @stability stable
     */
    UDP = "UDP"
}
/**
 * Client affinity gives you control over whether to always route each client to the same specific endpoint.
 *
 * @see https://docs.aws.amazon.com/global-accelerator/latest/dg/about-listeners.html#about-listeners-client-affinity
 * @stability stable
 */
export declare enum ClientAffinity {
    /**
     * Route traffic based on the 5-tuple `(source IP, source port, destination IP, destination port, protocol)`.
     *
     * @stability stable
     */
    NONE = "NONE",
    /**
     * Route traffic based on the 2-tuple `(source IP, destination IP)`.
     *
     * The result is that multiple connections from the same client will be routed the same.
     *
     * @stability stable
     */
    SOURCE_IP = "SOURCE_IP"
}
/**
 * The construct for the Listener.
 *
 * @stability stable
 */
export declare class Listener extends cdk.Resource implements IListener {
    /**
     * import from ARN.
     *
     * @stability stable
     */
    static fromListenerArn(scope: Construct, id: string, listenerArn: string): IListener;
    /**
     * The ARN of the listener.
     *
     * @stability stable
     */
    readonly listenerArn: string;
    /**
     * The name of the listener.
     *
     * @stability stable
     * @attribute true
     */
    readonly listenerName: string;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ListenerProps);
    /**
     * Add a new endpoint group to this listener.
     *
     * @stability stable
     */
    addEndpointGroup(id: string, options?: EndpointGroupOptions): EndpointGroup;
}
