"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossAccountDestination = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const logs_generated_1 = require("./logs.generated");
/**
 * A new CloudWatch Logs Destination for use in cross-account scenarios
 *
 * CrossAccountDestinations are used to subscribe a Kinesis stream in a
 * different account to a CloudWatch Subscription.
 *
 * Consumers will hardly ever need to use this class. Instead, directly
 * subscribe a Kinesis stream using the integration class in the
 * `@aws-cdk/aws-logs-destinations` package; if necessary, a
 * `CrossAccountDestination` will be created automatically.
 *
 * @resource AWS::Logs::Destination
 */
class CrossAccountDestination extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.destinationName ||
                // In the underlying model, the name is not optional, but we make it so anyway.
                cdk.Lazy.stringValue({ produce: () => this.generateUniqueName() }),
        });
        /**
         * Policy object of this CrossAccountDestination object
         */
        this.policyDocument = new iam.PolicyDocument();
        this.resource = new logs_generated_1.CfnDestination(this, 'Resource', {
            destinationName: this.physicalName,
            // Must be stringified policy
            destinationPolicy: this.lazyStringifiedPolicyDocument(),
            roleArn: props.role.roleArn,
            targetArn: props.targetArn,
        });
        this.destinationArn = this.getResourceArnAttribute(this.resource.attrArn, {
            service: 'logs',
            resource: 'destination',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.destinationName = this.getResourceNameAttribute(this.resource.ref);
    }
    addToPolicy(statement) {
        this.policyDocument.addStatements(statement);
    }
    bind(_scope, _sourceLogGroup) {
        return { arn: this.destinationArn };
    }
    /**
     * Generate a unique Destination name in case the user didn't supply one
     */
    generateUniqueName() {
        // Combination of stack name and LogicalID, which are guaranteed to be unique.
        return cdk.Stack.of(this).stackName + '-' + this.resource.logicalId;
    }
    /**
     * Return a stringified JSON version of the PolicyDocument
     */
    lazyStringifiedPolicyDocument() {
        return cdk.Lazy.stringValue({ produce: () => this.policyDocument.isEmpty ? '' : cdk.Stack.of(this).toJsonString(this.policyDocument),
        });
    }
}
exports.CrossAccountDestination = CrossAccountDestination;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3Jvc3MtYWNjb3VudC1kZXN0aW5hdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNyb3NzLWFjY291bnQtZGVzdGluYXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0NBQXdDO0FBQ3hDLHFDQUFxQztBQUVyQyxxREFBa0Q7QUEyQmxEOzs7Ozs7Ozs7Ozs7R0FZRztBQUNILE1BQWEsdUJBQXdCLFNBQVEsR0FBRyxDQUFDLFFBQVE7SUF1QnZELFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBbUM7UUFDL0UsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLGVBQWU7Z0JBQ2pDLCtFQUErRTtnQkFDL0UsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEVBQUUsQ0FBQztTQUNyRSxDQUFDLENBQUM7UUEzQkw7O1dBRUc7UUFDYSxtQkFBYyxHQUF1QixJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQztRQTBCNUUsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLCtCQUFjLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNuRCxlQUFlLEVBQUUsSUFBSSxDQUFDLFlBQWE7WUFDbkMsNkJBQTZCO1lBQzdCLGlCQUFpQixFQUFFLElBQUksQ0FBQyw2QkFBNkIsRUFBRTtZQUN2RCxPQUFPLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQzNCLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztTQUMzQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRTtZQUN4RSxPQUFPLEVBQUUsTUFBTTtZQUNmLFFBQVEsRUFBRSxhQUFhO1lBQ3ZCLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUMvQixHQUFHLEVBQUUsR0FBRztTQUNULENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDMUUsQ0FBQztJQUVNLFdBQVcsQ0FBQyxTQUE4QjtRQUMvQyxJQUFJLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRU0sSUFBSSxDQUFDLE1BQXFCLEVBQUUsZUFBMEI7UUFDM0QsT0FBTyxFQUFFLEdBQUcsRUFBRSxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7SUFDdEMsQ0FBQztJQUVEOztPQUVHO0lBQ0ssa0JBQWtCO1FBQ3hCLDhFQUE4RTtRQUM5RSxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsR0FBRyxHQUFHLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUM7SUFDdEUsQ0FBQztJQUVEOztPQUVHO0lBQ0ssNkJBQTZCO1FBQ25DLE9BQU8sR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQzFDLElBQUksQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDO1NBQ3hGLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQXZFRCwwREF1RUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBJTG9nR3JvdXAgfSBmcm9tICcuL2xvZy1ncm91cCc7XG5pbXBvcnQgeyBDZm5EZXN0aW5hdGlvbiB9IGZyb20gJy4vbG9ncy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUxvZ1N1YnNjcmlwdGlvbkRlc3RpbmF0aW9uLCBMb2dTdWJzY3JpcHRpb25EZXN0aW5hdGlvbkNvbmZpZyB9IGZyb20gJy4vc3Vic2NyaXB0aW9uLWZpbHRlcic7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYSBDcm9zc0FjY291bnREZXN0aW5hdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIENyb3NzQWNjb3VudERlc3RpbmF0aW9uUHJvcHMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGxvZyBkZXN0aW5hdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgQXV0b21hdGljYWxseSBnZW5lcmF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IGRlc3RpbmF0aW9uTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHJvbGUgdG8gYXNzdW1lIHRoYXQgZ3JhbnRzIHBlcm1pc3Npb25zIHRvIHdyaXRlIHRvICd0YXJnZXQnLlxuICAgKlxuICAgKiBUaGUgcm9sZSBtdXN0IGJlIGFzc3VtYWJsZSBieSAnbG9ncy57UkVHSU9OfS5hbWF6b25hd3MuY29tJy5cbiAgICovXG4gIHJlYWRvbmx5IHJvbGU6IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogVGhlIGxvZyBkZXN0aW5hdGlvbiB0YXJnZXQncyBBUk5cbiAgICovXG4gIHJlYWRvbmx5IHRhcmdldEFybjogc3RyaW5nO1xufVxuXG4vKipcbiAqIEEgbmV3IENsb3VkV2F0Y2ggTG9ncyBEZXN0aW5hdGlvbiBmb3IgdXNlIGluIGNyb3NzLWFjY291bnQgc2NlbmFyaW9zXG4gKlxuICogQ3Jvc3NBY2NvdW50RGVzdGluYXRpb25zIGFyZSB1c2VkIHRvIHN1YnNjcmliZSBhIEtpbmVzaXMgc3RyZWFtIGluIGFcbiAqIGRpZmZlcmVudCBhY2NvdW50IHRvIGEgQ2xvdWRXYXRjaCBTdWJzY3JpcHRpb24uXG4gKlxuICogQ29uc3VtZXJzIHdpbGwgaGFyZGx5IGV2ZXIgbmVlZCB0byB1c2UgdGhpcyBjbGFzcy4gSW5zdGVhZCwgZGlyZWN0bHlcbiAqIHN1YnNjcmliZSBhIEtpbmVzaXMgc3RyZWFtIHVzaW5nIHRoZSBpbnRlZ3JhdGlvbiBjbGFzcyBpbiB0aGVcbiAqIGBAYXdzLWNkay9hd3MtbG9ncy1kZXN0aW5hdGlvbnNgIHBhY2thZ2U7IGlmIG5lY2Vzc2FyeSwgYVxuICogYENyb3NzQWNjb3VudERlc3RpbmF0aW9uYCB3aWxsIGJlIGNyZWF0ZWQgYXV0b21hdGljYWxseS5cbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpMb2dzOjpEZXN0aW5hdGlvblxuICovXG5leHBvcnQgY2xhc3MgQ3Jvc3NBY2NvdW50RGVzdGluYXRpb24gZXh0ZW5kcyBjZGsuUmVzb3VyY2UgaW1wbGVtZW50cyBJTG9nU3Vic2NyaXB0aW9uRGVzdGluYXRpb24ge1xuICAvKipcbiAgICogUG9saWN5IG9iamVjdCBvZiB0aGlzIENyb3NzQWNjb3VudERlc3RpbmF0aW9uIG9iamVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBvbGljeURvY3VtZW50OiBpYW0uUG9saWN5RG9jdW1lbnQgPSBuZXcgaWFtLlBvbGljeURvY3VtZW50KCk7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoaXMgQ3Jvc3NBY2NvdW50RGVzdGluYXRpb24gb2JqZWN0XG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZXN0aW5hdGlvbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGlzIENyb3NzQWNjb3VudERlc3RpbmF0aW9uIG9iamVjdFxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGVzdGluYXRpb25Bcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGlubmVyIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IHJlc291cmNlOiBDZm5EZXN0aW5hdGlvbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENyb3NzQWNjb3VudERlc3RpbmF0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuZGVzdGluYXRpb25OYW1lIHx8XG4gICAgICAgIC8vIEluIHRoZSB1bmRlcmx5aW5nIG1vZGVsLCB0aGUgbmFtZSBpcyBub3Qgb3B0aW9uYWwsIGJ1dCB3ZSBtYWtlIGl0IHNvIGFueXdheS5cbiAgICAgICAgY2RrLkxhenkuc3RyaW5nVmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmdlbmVyYXRlVW5pcXVlTmFtZSgpIH0pLFxuICAgIH0pO1xuXG4gICAgdGhpcy5yZXNvdXJjZSA9IG5ldyBDZm5EZXN0aW5hdGlvbih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBkZXN0aW5hdGlvbk5hbWU6IHRoaXMucGh5c2ljYWxOYW1lISxcbiAgICAgIC8vIE11c3QgYmUgc3RyaW5naWZpZWQgcG9saWN5XG4gICAgICBkZXN0aW5hdGlvblBvbGljeTogdGhpcy5sYXp5U3RyaW5naWZpZWRQb2xpY3lEb2N1bWVudCgpLFxuICAgICAgcm9sZUFybjogcHJvcHMucm9sZS5yb2xlQXJuLFxuICAgICAgdGFyZ2V0QXJuOiBwcm9wcy50YXJnZXRBcm4sXG4gICAgfSk7XG5cbiAgICB0aGlzLmRlc3RpbmF0aW9uQXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZSh0aGlzLnJlc291cmNlLmF0dHJBcm4sIHtcbiAgICAgIHNlcnZpY2U6ICdsb2dzJyxcbiAgICAgIHJlc291cmNlOiAnZGVzdGluYXRpb24nLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIHNlcDogJzonLFxuICAgIH0pO1xuICAgIHRoaXMuZGVzdGluYXRpb25OYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUodGhpcy5yZXNvdXJjZS5yZWYpO1xuICB9XG5cbiAgcHVibGljIGFkZFRvUG9saWN5KHN0YXRlbWVudDogaWFtLlBvbGljeVN0YXRlbWVudCkge1xuICAgIHRoaXMucG9saWN5RG9jdW1lbnQuYWRkU3RhdGVtZW50cyhzdGF0ZW1lbnQpO1xuICB9XG5cbiAgcHVibGljIGJpbmQoX3Njb3BlOiBjZGsuQ29uc3RydWN0LCBfc291cmNlTG9nR3JvdXA6IElMb2dHcm91cCk6IExvZ1N1YnNjcmlwdGlvbkRlc3RpbmF0aW9uQ29uZmlnIHtcbiAgICByZXR1cm4geyBhcm46IHRoaXMuZGVzdGluYXRpb25Bcm4gfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSBhIHVuaXF1ZSBEZXN0aW5hdGlvbiBuYW1lIGluIGNhc2UgdGhlIHVzZXIgZGlkbid0IHN1cHBseSBvbmVcbiAgICovXG4gIHByaXZhdGUgZ2VuZXJhdGVVbmlxdWVOYW1lKCk6IHN0cmluZyB7XG4gICAgLy8gQ29tYmluYXRpb24gb2Ygc3RhY2sgbmFtZSBhbmQgTG9naWNhbElELCB3aGljaCBhcmUgZ3VhcmFudGVlZCB0byBiZSB1bmlxdWUuXG4gICAgcmV0dXJuIGNkay5TdGFjay5vZih0aGlzKS5zdGFja05hbWUgKyAnLScgKyB0aGlzLnJlc291cmNlLmxvZ2ljYWxJZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gYSBzdHJpbmdpZmllZCBKU09OIHZlcnNpb24gb2YgdGhlIFBvbGljeURvY3VtZW50XG4gICAqL1xuICBwcml2YXRlIGxhenlTdHJpbmdpZmllZFBvbGljeURvY3VtZW50KCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIGNkay5MYXp5LnN0cmluZ1ZhbHVlKHsgcHJvZHVjZTogKCkgPT5cbiAgICAgIHRoaXMucG9saWN5RG9jdW1lbnQuaXNFbXB0eSA/ICcnIDogY2RrLlN0YWNrLm9mKHRoaXMpLnRvSnNvblN0cmluZyh0aGlzLnBvbGljeURvY3VtZW50KSxcbiAgICB9KTtcbiAgfVxufVxuIl19