#
# Copyright © 2012–2022 Michal Čihař <michal@cihar.com>
#
# This file is part of Weblate <https://weblate.org/>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#


from rest_framework import routers


class WeblateRouter(routers.DefaultRouter):
    def get_lookup_regex(self, viewset, lookup_prefix=""):
        """Get lookup regex for a viewset.

        Given a viewset, return the portion of URL regex that is used
        to match against a single instance.

        Note that lookup_prefix is not used directly inside REST rest_framework
        itself, but is required in order to nicely support nested router
        implementations, such as drf-nested-routers.

        https://github.com/alanjds/drf-nested-routers
        """
        base_regex = "(?P<{lookup_prefix}{lookup_url_kwarg}>{lookup_value})"
        # Use `pk` as default field, unset set.  Default regex should not
        # consume `.json` style suffixes and should break at '/' boundaries.
        lookup_field = getattr(viewset, "lookup_field", "pk")
        lookup_fields = getattr(viewset, "lookup_fields", None)
        if lookup_fields is None:
            lookup_fields = [lookup_field]

        lookup_value = getattr(viewset, "lookup_value_regex", "[^/]+")

        result = []

        for field in lookup_fields:
            result.append(
                base_regex.format(
                    lookup_prefix=lookup_prefix,
                    lookup_url_kwarg=field,
                    lookup_value=lookup_value,
                )
            )
        return "/".join(result)
