"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractResolvedServiceName = exports.resolveServiceName = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const variants_1 = require("./variants");
const assets_1 = require("../../generated/assets");
const utils_1 = require("../../utils");
/** @internal */
const SERVICE_NAME_LOOKUP = Object.fromEntries([
    ...Object.keys(assets_1.AwsAsset.Services).flatMap((key) => {
        return [
            // constant self lookup
            [key, key],
            [(0, utils_1.normalizeComparisonString)(key), key],
        ];
    }),
    ...Object.entries(variants_1.NORMALIZED_SERVICE_VARIANTS).flatMap(([key, variants]) => {
        if (variants == null)
            return [];
        return variants.map((variant) => [(0, utils_1.normalizeComparisonString)(variant), key]);
    }),
]);
/** @internal */
function resolveServiceName(value) {
    const comparableValue = (0, utils_1.normalizeComparisonString)(value);
    // constant lookup first for perf
    if (comparableValue in SERVICE_NAME_LOOKUP) {
        return SERVICE_NAME_LOOKUP[comparableValue];
    }
    // attempt know mappings
    if (comparableValue.startsWith("media") &&
        SERVICE_NAME_LOOKUP[`elemental${comparableValue}`]) {
        const serviceName = SERVICE_NAME_LOOKUP[`elemental${comparableValue}`];
        SERVICE_NAME_LOOKUP[comparableValue] = serviceName;
        return serviceName;
    }
    // prefixed lookup (WAFRegion => WAF, CodeGuruProfile => CodeGuru)
    for (const serviceName of Object.keys(assets_1.AwsAsset.Services)) {
        if (comparableValue.startsWith((0, utils_1.normalizeComparisonString)(serviceName))) {
            // add to lookup for constant lookkup in subsequent calls;
            SERVICE_NAME_LOOKUP[comparableValue] = serviceName;
            return serviceName;
        }
    }
    throw new Error(`Failed to resolve serviceName ${value} (${comparableValue})`);
}
exports.resolveServiceName = resolveServiceName;
/**
 * Extract service name parts from value that is or contains a service name or variant of service name.
 * @internal
 */
function extractResolvedServiceName(value) {
    const parts = (0, utils_1.normalizeIdentifier)(value).split("_");
    for (let i = parts.length; i > 0; i--) {
        try {
            const match = parts.slice(0, i).join("_");
            const serviceName = resolveServiceName(match);
            let rest = value.replace(match, "");
            if (rest.length === 0) {
                rest = undefined;
            }
            const resolvedValue = `${serviceName}${rest}`;
            return { serviceName, match, rest, resolvedValue };
        }
        catch {
            continue;
        }
    }
    return;
}
exports.extractResolvedServiceName = extractResolvedServiceName;
//# sourceMappingURL=data:application/json;base64,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