"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseAwsUrl = exports.AWS_URL_PATTERN = exports.normalizeComparisonString = exports.normalizeIdentifier = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
/**
 * Normalize string value to standard identifier form.
 *
 * Removes extraneous prefixes (Amazon, AWS), replaces all non-alphanumerics with underscore (_),
 * and converts to lower-case.
 * @param value - Value to normalize
 * @returns Returns normalized identifier string.
 */
function normalizeIdentifier(value) {
    value = value.replace(/^(Amazon|AWS)-?/i, ""); // remove prefix
    value = value.replace(/[^A-Za-z0-9]/g, "_"); // only alphanum + _
    value = value.replace(/_+/g, "_"); // de-dupe _
    return value.toLowerCase();
}
exports.normalizeIdentifier = normalizeIdentifier;
const ComparisonDictionary = {
    acl: "access_control_list",
    eni: "elastic_network_interface",
    eip: "elastic_ip_address",
};
/**
 * Normalizes string value for consistent comparison with variants between systems.
 * @param value - The value to normalize for comparison
 */
function normalizeComparisonString(value) {
    value = value.replace(/\.(png|svg)$/, "");
    value = normalizeIdentifier(value);
    Object.entries(ComparisonDictionary).forEach(([_key, _value]) => {
        _value = _value.replace(new RegExp(_key, "ig"), _value);
    });
    value = value.replace(/[_-]+(amazon|aws)[_-]+/gi, "");
    value = value.replace(/_(for|on|and|the|of|&)_/gi, "");
    value = value.replace(/v\d+/g, "");
    value = value.replace(/_/g, "");
    return value;
}
exports.normalizeComparisonString = normalizeComparisonString;
/**
 * RegEx pattern for https://aws.amazon.com based urls.
 */
exports.AWS_URL_PATTERN = /^https:\/\/(?<domain>(?:(?:docs)\.)?aws\.amazon\.com)\/(?<service>[^\/]+)(?:\/(?<resource>[^#]+))?\/?(?:#(?<dataType>.+))?$/i;
/** Parses https://aws.amazon.com based url into common definition. */
function parseAwsUrl(url) {
    url = url.replace(/\/(index\.html.*)?$/, "");
    const groups = url.match(exports.AWS_URL_PATTERN)
        ?.groups;
    if (groups) {
        const code = normalizeIdentifier(`${groups.service}_${groups.resource || ""}_${groups.dataType || ""}`);
        return {
            ...groups,
            code,
        };
    }
    throw new Error(`Failed to parse aws url: ${url}`);
}
exports.parseAwsUrl = parseAwsUrl;
//# sourceMappingURL=data:application/json;base64,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