"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDiagram = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_arch_1 = require("../../../aws-arch");
const cdk_graph_1 = require("../../../cdk-graph");
const uniqBy = require("lodash.uniqby"); // eslint-disable-line @typescript-eslint/no-require-imports
const Diagram = require("./entities");
const entities_1 = require("./entities");
const theme_1 = require("./theme");
const debug_1 = require("../debug");
/**
 * EdgeResolver class resolves edges within a diagram for rendering
 * @internal
 */
class EdgeResolver {
    constructor() {
        /** @internal */
        this._edges = new Map();
    }
    /** Adds diagram edge to the resolver to be resolved */
    trackEdge(edge) {
        let fromId;
        let toId;
        if (edge instanceof Diagram.Edge) {
            fromId = edge.graphEdge.source.uuid;
            toId = edge.graphEdge.target.uuid;
        }
        else {
            fromId = edge.from.graphNode.uuid;
            toId = edge.to.graphNode.uuid;
        }
        const key = fromId < toId ? `${fromId}:${toId}` : `${toId}:${fromId}`;
        const edgeSet = this._edges.get(key) || [];
        this._edges.set(key, edgeSet.concat([edge]));
    }
    /** Resolve all edges based on diagram options */
    resolveEdges(options) {
        const compact = options.preset === cdk_graph_1.FilterPreset.COMPACT;
        const resolvedEdges = [];
        for (let edges of this._edges.values()) {
            if (compact) {
                edges = edges.filter((edge) => !edge.isVerbose);
            }
            if (edges.length === 0) {
                continue;
            }
            edges.sort((a, b) => {
                const _a = getEdgeRank(a);
                const _b = getEdgeRank(b);
                if (_a === _b)
                    return 0;
                if (_a < _b)
                    return -1;
                return 1;
            });
            edges = uniqBy(edges, getEdgeRank);
            // only return highest ranked edge unless verbose
            if (compact) {
                resolvedEdges.push(edges[0]);
                continue;
            }
            for (const _edge1 of edges) {
                for (const _edge2 of edges) {
                    if (_edge1 === _edge2)
                        continue;
                    const _id1 = _edge1.attributes.get("id");
                    const _id2 = _edge2.attributes.get("id");
                    let _sameHead1 = _edge1.attributes.get("samehead");
                    _edge1.attributes.set("samehead", !_sameHead1 ? _id2 : `${_sameHead1},${_id2}`);
                    let _sameTail1 = _edge1.attributes.get("sametail");
                    _edge1.attributes.set("sametail", !_sameTail1 ? _id2 : `${_sameTail1},${_id2}`);
                    let _sameHead2 = _edge2.attributes.get("samehead");
                    _edge2.attributes.set("samehead", !_sameHead2 ? _id1 : `${_sameHead2},${_id1}`);
                    let _sameTail2 = _edge2.attributes.get("sametail");
                    _edge2.attributes.set("sametail", !_sameTail2 ? _id1 : `${_sameTail2},${_id1}`);
                }
            }
            resolvedEdges.push(...edges);
        }
        return resolvedEdges;
    }
}
/**
 * Build a {@link Diagram.Diagram Diagram} for a given {@link Graph.Store} based on {@link DiagramOptions Options}
 * @internal
 */
function buildDiagram(store, options) {
    const { title } = options;
    const edgeResolve = new EdgeResolver();
    theme_1.GraphTheme.init(options.theme);
    const entities = new Map();
    const diagram = new Diagram.Diagram(title, aws_arch_1.AwsArchitecture.assetDirectory);
    function visit(gNode, parent) {
        if (gNode.isDestroyed)
            return;
        let entity;
        switch (gNode.nodeType) {
            case cdk_graph_1.NodeTypeEnum.RESOURCE: {
                entity = new Diagram.ResourceNode(gNode);
                break;
            }
            case cdk_graph_1.NodeTypeEnum.CFN_RESOURCE: {
                entity = new Diagram.CfnResourceNode(gNode);
                break;
            }
            case cdk_graph_1.NodeTypeEnum.NESTED_STACK: {
                entity = new Diagram.NestedStackCluster(gNode);
                break;
            }
            case cdk_graph_1.NodeTypeEnum.STACK: {
                if (theme_1.GraphTheme.instance.rendering.stack &&
                    new RegExp(theme_1.GraphTheme.instance.rendering.stack).test(gNode.id) ===
                        false) {
                    // Ignore non-matching root stacks
                    return;
                }
                entity = new Diagram.StackCluster(gNode);
                break;
            }
            default: {
                if (gNode.isLeaf) {
                    if (gNode.hasFlag(cdk_graph_1.FlagEnum.CUSTOM_RESOURCE)) {
                        entity = new Diagram.CustomResourceNode(gNode);
                    }
                    else {
                        entity = new Diagram.Node(gNode);
                    }
                }
                else {
                    entity = new Diagram.Cluster(gNode);
                    gNode.addFlag(cdk_graph_1.FlagEnum.CLUSTER);
                }
                break;
            }
        }
        if (entity instanceof entities_1.ImageNode && entity.image) {
            diagram.trackImage(entity.image);
        }
        if (parent instanceof Diagram.Container && parent.linkChildren) {
            edgeResolve.trackEdge(new entities_1.ChildLink(parent, entity));
        }
        if (gNode.isLeaf) {
            entities.set(gNode.uuid, entity);
            parent.addNode(entity);
        }
        else {
            if (entity instanceof Diagram.Node) {
                entity = asContainer(entity);
            }
            parent.addSubgraph(entity);
            entities.set(gNode.uuid, entity);
            gNode.children.forEach((child) => visit(child, entity));
        }
    }
    if (store.stages.length > 1) {
        const stageRendering = theme_1.GraphTheme.instance.rendering.stage || "last";
        let stages;
        switch (stageRendering) {
            case "all": {
                stages = store.stages;
                break;
            }
            case "first": {
                stages = store.stages.slice(0, 1);
                break;
            }
            case "last": {
                stages = store.stages.slice(-1);
                break;
            }
            default: {
                stages = store.stages.filter((stage) => stage.id.match(stageRendering));
            }
        }
        // traverse all stages
        stages.forEach((gStage) => {
            const dStage = new Diagram.StageCluster(gStage);
            diagram.addSubgraph(dStage);
            entities.set(gStage.uuid, dStage);
            gStage.children.forEach((child) => visit(child, dStage));
        });
    }
    else if (store.rootStacks.length) {
        // traverse all root stack
        store.rootStacks.forEach((gStack) => {
            const dStack = new Diagram.StackCluster(gStack);
            diagram.addSubgraph(dStack);
            entities.set(gStack.uuid, dStack);
            gStack.children.forEach((child) => visit(child, dStack));
        });
    }
    else {
        store.root.children.forEach((gChild) => {
            if (gChild.isGraphContainer) {
                gChild.children.forEach((_gChild) => {
                    visit(_gChild, diagram);
                });
            }
            else {
                visit(gChild, diagram);
            }
        });
    }
    // apply all edges
    store.edges.forEach((gEdge) => {
        if (gEdge.isDestroyed)
            return;
        const dSource = entities.get(gEdge.source.uuid);
        const dTarget = entities.get(gEdge.target.uuid);
        if (!dSource || !dTarget) {
            debug_1.IS_DEBUG &&
                console.warn("Diagram.Edge unresolved source and/or target:", `source(${gEdge.source} => ${dSource})`, `target(${gEdge.target} => ${dTarget})`);
            return;
        }
        let edge = undefined;
        switch (gEdge.edgeType) {
            case cdk_graph_1.EdgeTypeEnum.REFERENCE: {
                edge = new Diagram.ReferenceLink(gEdge, dSource, dTarget);
                break;
            }
            case cdk_graph_1.EdgeTypeEnum.DEPENDENCY: {
                edge = new Diagram.DependencyLink(gEdge, dSource, dTarget);
                break;
            }
        }
        if (edge) {
            entities.set(gEdge.uuid, edge);
            edgeResolve.trackEdge(edge);
        }
    });
    edgeResolve.resolveEdges(options).forEach((edge) => {
        diagram.addEdge(edge);
    });
    return diagram;
}
exports.buildDiagram = buildDiagram;
/**
 * Wrap a {@link Diagram.Node} with {@link Diagram.Container} to support adding child {@link Diagram.Node}s
 * @internal
 */
function asContainer(node) {
    const container = new Diagram.Container(node.graphNode);
    container.addNode(node);
    return container;
}
/**
 * Get the rank score of an {@link Diagram.BaseEdge Edge} used to sort and prioritize edges
 * @internal
 */
function getEdgeRank(edge) {
    if (edge instanceof Diagram.ChildLink) {
        return 0;
    }
    if (edge instanceof Diagram.ReferenceLink) {
        return 1;
    }
    if (edge instanceof Diagram.DependencyLink) {
        return 2;
    }
    return 3;
}
//# sourceMappingURL=data:application/json;base64,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