"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getBaseTheme = exports.GraphTheme = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_arch_1 = require("../../../../aws-arch");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const theme_1 = require("./theme");
const fonts_1 = require("../../fonts");
const DEFAULT_RENDERING = {
    resourceIconMin: theme_1.GraphThemeRenderingIconTarget.DATA,
    resourceIconMax: theme_1.GraphThemeRenderingIconTarget.CATEGORY,
    cfnResourceIconMin: theme_1.GraphThemeRenderingIconTarget.DATA,
    cfnResourceIconMax: theme_1.GraphThemeRenderingIconTarget.SERVICE,
};
/** GraphTheme class is the implementation of diagram graph based theme */
class GraphTheme {
    /** Initializes the theme based on {@link aws_arch.Theme} definition */
    static init(config) {
        if (config == null || typeof config === "string") {
            this._instance = new GraphTheme(generateGraphThemeFromAwsTheme(aws_arch_1.aws_arch.resolveTheme(config)));
        }
        else if (Object.keys(config).length <= 2) {
            const { theme, rendering } = config;
            this._instance = new GraphTheme(generateGraphThemeFromAwsTheme(aws_arch_1.aws_arch.resolveTheme(theme), rendering));
        }
        else {
            this._instance = new GraphTheme(config);
        }
        return this.instance;
    }
    /** Get the current singleton instance for the theme being utilized for the diagram */
    static get instance() {
        if (this._instance == null) {
            throw new Error("Must init GraphTheme before requesting instance");
        }
        return this._instance;
    }
    /** @internal */
    constructor(theme) {
        this.rendering = theme.rendering;
        this.graph = theme.graph;
        this.subgraph = theme.subgraph;
        this.cluster = theme.cluster;
        this.node = theme.node;
        this.edge = theme.edge;
        this.cloud = theme.cloud;
        this.stage = theme.stage;
        this.stack = theme.stack;
        this.nestedStack = theme.nestedStack;
        this.imageNode = theme.imageNode;
        this.cfnResourceNode = theme.cfnResourceNode;
        this.resourceNode = theme.resourceNode;
        this.childLink = theme.childLink;
        this.referenceLink = theme.referenceLink;
        this.dependencyLink = theme.dependencyLink;
        this.awsTheme = theme.awsTheme;
    }
}
exports.GraphTheme = GraphTheme;
/** Get the base theme */
function getBaseTheme(rendering) {
    return cloneDeep({
        rendering: {
            ...DEFAULT_RENDERING,
            ...rendering,
        },
        graph: GRAPH_ATTRIBUTES,
        subgraph: SUBGRAPH_ATTRIBUTES,
        cluster: CLUSTER_ATTRIBUTES,
        node: NODE_ATTRIBUTES,
        edge: EDGE_ATTRIBUTES,
        cloud: CLOUD_ATTRIBUTES,
        stage: STAGE_ATTRIBUTES,
        stack: STACK_ATTRIBUTES,
        nestedStack: NESTED_STACK_ATTRIBUTES,
        imageNode: IMAGE_NODE_ATTRIBUTES,
        cfnResourceNode: CFN_RESOURCE_NODE_ATTRIBUTES,
        resourceNode: RESOURCE_NODE_ATTRIBUTES,
        childLink: CHILD_LINK_ATTRIBUTES,
        referenceLink: REFERENCE_LINK_ATTRIBUTES,
        dependencyLink: DEPENDENCY_LINK_ATTRIBUTES,
    });
}
exports.getBaseTheme = getBaseTheme;
/** Generate {@link IGraphTheme} from {@link aws_arch.Theme} */
function generateGraphThemeFromAwsTheme(awsTheme, rendering) {
    const theme = getBaseTheme(rendering);
    Object.assign(theme, {
        awsTheme,
    });
    Object.assign(theme.graph, {
        bgcolor: awsTheme.backgrounds.base,
        fontcolor: awsTheme.text.default,
    });
    Object.assign(theme.node, {
        fontcolor: awsTheme.text.primary,
    });
    Object.assign(theme.edge, awsArrowToEdgeAtts(awsTheme.arrows.default));
    // containers
    Object.assign(theme.cloud, awsGroupToSubgraphAtts(awsTheme.groups.cloud));
    Object.assign(theme.stage, awsGroupToSubgraphAtts(awsTheme.groups.awsAccount, true));
    Object.assign(theme.stack, awsGroupToSubgraphAtts(awsTheme.groups.genericAlt, true));
    Object.assign(theme.nestedStack, awsGroupToSubgraphAtts(awsTheme.groups.genericAlt, true));
    // nodes
    Object.assign(theme.cfnResourceNode, {
        color: awsTheme.text.secondary,
        fontcolor: awsTheme.text.secondary,
    });
    Object.assign(theme.resourceNode, {
        color: awsTheme.text.primary,
        fontcolor: awsTheme.text.primary,
    });
    Object.assign(theme.childLink, awsArrowToEdgeAtts(awsTheme.arrows.child));
    Object.assign(theme.referenceLink, awsArrowToEdgeAtts(awsTheme.arrows.reference));
    Object.assign(theme.dependencyLink, awsArrowToEdgeAtts(awsTheme.arrows.dependency));
    return theme;
}
/** Base graph attributes */
const GRAPH_ATTRIBUTES = {
    ...fonts_1.GraphFonts.REGULAR,
    center: true,
    compound: true,
    concentrate: true,
    dpi: 300,
    fontcolor: "#222222",
    fontnames: "ps",
    fontsize: 14,
    forcelabels: true,
    labelloc: "tc",
    nodesep: 0.8,
    pad: "%2,%1",
    // @ts-ignore
    rankdir: "TB",
    ranksep: 0.75,
    ratio: "compress",
    remincross: true,
    size: "%1024,%1024!",
    splines: "ortho",
    stylesheet: fonts_1.FONT_STYLESHEET,
};
/** Base subgraph attributes */
const SUBGRAPH_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    labelloc: "tc",
    fontsize: 12,
    style: "rounded,solid",
};
/** Base cluster attributes */
const CLUSTER_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT,
    labelloc: "tc",
    fontsize: 12,
    style: "rounded,dashed",
};
/** Base cloud container attributes */
const CLOUD_ATTRIBUTES = {
    ...fonts_1.GraphFonts.BOLD,
    style: "solid",
};
/** Base stage attributes */
const STAGE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.BOLD_ITALIC,
    style: "dashed",
    margin: 6,
    rank: "same",
};
/** Base stack attributes */
const STACK_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT,
    style: "solid,bold,filled",
    fillcolor: "#5A6B861A",
    margin: 10,
};
/** Base nested stack attributes */
const NESTED_STACK_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    style: "solid,filled",
    fillcolor: "#5A6B861A",
    margin: 6,
};
/** Base node attributes */
const NODE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.REGULAR,
    shape: "box",
    style: "solid",
    fixedsize: false,
    width: 0.25,
    height: 0.25,
    labelloc: "c",
    imagescale: true,
    fontsize: 10,
    penwidth: 0,
};
/** Base image based node attributes */
const IMAGE_NODE_ATTRIBUTES = {
    shape: "box",
    style: "solid,rounded",
    fixedsize: true,
    width: 1,
    height: 1,
    labelloc: "b",
    imagescale: true,
    imagepos: "tc",
    fillcolor: "transparent",
};
/** Base cfn resource node attributes */
const CFN_RESOURCE_NODE_ATTRIBUTES = {
    width: 1,
    height: 1,
    fixedsize: true,
    imagescale: true,
    style: "solid,rounded",
    color: "#999999",
    fontcolor: "#999999",
};
/** Base resource node attributes */
const RESOURCE_NODE_ATTRIBUTES = {
    width: 1,
    height: 1,
    fixedsize: true,
    imagescale: true,
    color: "#666666",
    fontcolor: "#666666",
};
/** Base edge attributes */
const EDGE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    dir: "both",
    color: "#545B64",
    penwidth: 0.75,
    arrowhead: "dot",
    arrowtail: "dot",
    arrowsize: 0.5,
    fontsize: 9,
    style: "solid",
};
/** Base child link attributes */
const CHILD_LINK_ATTRIBUTES = {
    penwidth: 1,
    arrowhead: "none",
    arrowtail: "normal",
    arrowsize: 1,
    style: "solid",
};
/** Base reference link attributes */
const REFERENCE_LINK_ATTRIBUTES = {
    penwidth: 1,
    arrowhead: "none",
    arrowtail: "normal",
    arrowsize: 0.75,
    style: "solid",
};
/** Base dependency link attributes */
const DEPENDENCY_LINK_ATTRIBUTES = {
    penwidth: 0.75,
    arrowhead: "dot",
    arrowtail: "odot",
    arrowsize: 0.75,
    style: "dotted",
};
/** Convert {@link aws_arch.GroupFormat} to {@link Dot.SubgraphAttributesObject} */
function awsGroupToSubgraphAtts(group, ignoreStyle = false) {
    const attributes = {};
    if (!ignoreStyle) {
        if (group.borderStyle === "none") {
            attributes.style = "solid";
            attributes.penwidth = 0;
        }
        else {
            attributes.style = group.borderStyle;
        }
    }
    attributes.color = group.color || "transparent";
    attributes.bgcolor = group.bgcolor || "transparent";
    attributes.fillcolor = group.bgcolor || "transparent";
    attributes.pencolor = (group.borderColor || "transparent");
    if (group.color && group.color !== "transparent") {
        attributes.fontcolor = group.color;
    }
    return attributes;
}
/** Convert {@link aws_arch.ArrowFormat} to {@link Dot.EdgeAttributesObject} */
function awsArrowToEdgeAtts(arrow) {
    return {
        color: arrow.color,
        fontcolor: arrow.color,
        arrowtail: arrow.tail,
        arrowhead: arrow.head,
        penwidth: arrow.width,
        style: arrow.style,
    };
}
//# sourceMappingURL=data:application/json;base64,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