"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.inferImportCfnType = exports.tokenizeImportArn = exports.resolveImportedConstructArnToken = exports.isImportConstruct = exports.inferFlags = exports.extractUnresolvedReferences = exports.IGNORE_REF_PATTERN = exports.extractInspectableAttributes = exports.inferNodeProps = exports.tryGetLogicalId = exports.getConstructUUID = exports.generateConsistentUUID = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const shorthash = require("shorthash2"); // eslint-disable-line @typescript-eslint/no-require-imports
const traverse = require("traverse"); // eslint-disable-line @typescript-eslint/no-require-imports
const types_1 = require("./types");
const cdk_internals_1 = require("../cdk-internals");
/**
 * Generate deterministic UUID based on given value and prefix.
 * @param value The value to hash as UUID
 * @param {string} [prefix=""] Optional prefix used to prevent value conflicts
 */
function generateConsistentUUID(value, prefix = "") {
    return prefix + shorthash(JSON.stringify(value));
}
exports.generateConsistentUUID = generateConsistentUUID;
/** Get UUID for a given construct */
function getConstructUUID(construct) {
    return aws_cdk_lib_1.Names.uniqueResourceName(construct, {});
}
exports.getConstructUUID = getConstructUUID;
/** Try to get *logicalId* for given construct */
function tryGetLogicalId(construct) {
    if (aws_cdk_lib_1.CfnElement.isCfnElement(construct)) {
        const stack = aws_cdk_lib_1.Stack.of(construct);
        return stack.resolve(stack.getLogicalId(construct));
    }
    return undefined;
}
exports.tryGetLogicalId = tryGetLogicalId;
/** Infer node props from construct */
function inferNodeProps(construct) {
    const uuid = getConstructUUID(construct);
    const logicalId = tryGetLogicalId(construct);
    const metadata = construct.node.metadata.filter((entry) => {
        if (entry.type === types_1.MetadataTypeEnum.LOGICAL_ID)
            return false;
        return true;
    });
    const attributes = cloneDeep(extractInspectableAttributes(construct) || {});
    const cfnType = attributes[types_1.CfnAttributesEnum.TYPE];
    if (cfnType) {
        // @ts-ignore
        delete attributes[types_1.CfnAttributesEnum.TYPE];
    }
    const cfnProps = attributes[types_1.CfnAttributesEnum.PROPS] || {};
    let tags = {};
    // normalize tags
    if (typeof cfnProps === "object" && "tags" in cfnProps) {
        const _tags = cfnProps.tags;
        // remove the tags from the attributes since we normalize
        // @ts-ignore
        delete cfnProps.tags;
        if (Array.isArray(_tags)) {
            tags = Object.fromEntries(_tags.map(({ key, value }) => [key, value]));
        }
        else {
            tags = _tags;
        }
    }
    const constructInfo = (0, cdk_internals_1.constructInfoFromConstruct)(construct);
    const flags = inferFlags(construct, constructInfo, tags);
    return {
        uuid,
        attributes,
        metadata,
        tags,
        logicalId,
        cfnType,
        constructInfo,
        dependencies: obtainDependencies(construct),
        unresolvedReferences: extractUnresolvedReferences(uuid, attributes),
        flags,
    };
}
exports.inferNodeProps = inferNodeProps;
function obtainDependencies(construct) {
    if (aws_cdk_lib_1.CfnResource.isCfnResource(construct)) {
        return construct.obtainDependencies().map(getConstructUUID);
    }
    return construct.node.dependencies.map(getConstructUUID);
}
/** Extract inspectable attributes from construct */
function extractInspectableAttributes(construct) {
    // check if a construct implements IInspectable
    function canInspect(inspectable) {
        return inspectable.inspect !== undefined;
    }
    const inspector = new aws_cdk_lib_1.TreeInspector();
    // get attributes from the inspector
    if (canInspect(construct)) {
        construct.inspect(inspector);
        return aws_cdk_lib_1.Stack.of(construct).resolve(inspector.attributes);
    }
    return undefined;
}
exports.extractInspectableAttributes = extractInspectableAttributes;
/** Pattern of ignored references. Those which are resolved during deploy-time. */
exports.IGNORE_REF_PATTERN = /^AWS::/;
/** Extract unresolved references from attributes for a given source */
function extractUnresolvedReferences(source, from) {
    const references = [];
    traverse(from).forEach(function () {
        switch (this.key) {
            case types_1.ReferenceTypeEnum.ATTRIBUTE: {
                const [logicalId, attribute] = this.node;
                references.push({
                    source,
                    referenceType: types_1.ReferenceTypeEnum.ATTRIBUTE,
                    target: logicalId,
                    value: attribute,
                });
                this.block();
                break;
            }
            case types_1.ReferenceTypeEnum.REF: {
                if (typeof this.node === "string") {
                    if (!exports.IGNORE_REF_PATTERN.test(this.node)) {
                        references.push({
                            source,
                            referenceType: types_1.ReferenceTypeEnum.REF,
                            target: this.node,
                        });
                    }
                }
                else {
                    console.warn(`Found non-string "Ref"`, this.node);
                }
                this.block();
                break;
            }
            case types_1.ReferenceTypeEnum.IMPORT: {
                // "Fn::ImportValue": "Ada:ExportsOutputFnGetAttCommonStackA8F9EE77OutputsAdaCommonStackCounterTable5D6ADA16ArnED1AF27F"
                // "Fn::ImportValue": "Stage-Ada:ExportsOutputFnGetAttCommonStackA8F9EE77OutputsAdaCommonStackCounterTable5D6ADA16ArnED1AF27F"
                references.push({
                    source,
                    referenceType: types_1.ReferenceTypeEnum.IMPORT,
                    // NB: remove stage - separator
                    target: this.node.replace("-", ""),
                });
                this.block();
                break;
            }
            case "Fn::Join": {
                if (Array.isArray(this.node) &&
                    this.node.flatMap(String).join("").startsWith("arn:")) {
                    const potentialImportArn = {
                        "Fn::Join": this.node,
                    };
                    references.push({
                        source,
                        referenceType: types_1.ReferenceTypeEnum.IMPORT_ARN,
                        target: tokenizeImportArn(potentialImportArn),
                    });
                }
                break;
            }
        }
    });
    return references;
}
exports.extractUnresolvedReferences = extractUnresolvedReferences;
// https://github.com/aws/aws-cdk/blob/main/packages/%40aws-cdk/custom-resources/lib/aws-custom-resource/aws-custom-resource.ts#L357
const AWS_PROVIDER_FUNCTION_UUID = "679f53fac002430cb0da5b7982bd2287";
/** Infer construct flags  */
function inferFlags(construct, constructInfo, tags) {
    const flags = new Set();
    const fqn = constructInfo?.fqn;
    if (isImportConstruct(construct)) {
        flags.add(types_1.FlagEnum.IMPORT);
    }
    else {
        if (fqn && types_1.ExtraneousFqns.includes(fqn)) {
            flags.add(types_1.FlagEnum.EXTRANEOUS);
        }
        if (fqn && types_1.AssetFqns.includes(fqn)) {
            flags.add(types_1.FlagEnum.ASSET);
        }
    }
    if (fqn && _isCfnFqn(fqn)) {
        flags.add(types_1.FlagEnum.CFN_FQN);
    }
    if (construct.node.id === "Exports" && aws_cdk_lib_1.Stack.isStack(construct.node.scope)) {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    if (construct.node.id.startsWith("SsmParameterValue:")) {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    if (fqn === types_1.ConstructInfoFqnEnum.LAMBDA &&
        aws_cdk_lib_1.Resource.isOwnedResource(construct)) {
        if (construct.node.id === `AWS${AWS_PROVIDER_FUNCTION_UUID}`) {
            flags.add(types_1.FlagEnum.AWS_API_CALL_LAMBDA);
            flags.add(types_1.FlagEnum.EXTRANEOUS);
        }
    }
    if (fqn && types_1.CustomResourceFqns.includes(fqn)) {
        flags.add(types_1.FlagEnum.CUSTOM_RESOURCE);
        if (fqn === types_1.ConstructInfoFqnEnum.AWS_CUSTOM_RESOURCE) {
            flags.add(types_1.FlagEnum.AWS_CUSTOM_RESOURCE);
        }
    }
    // https://github.com/aws/aws-cdk/blob/37f031f1f1c41bbfb6f8e8a56f73b5966e365ff6/packages/%40aws-cdk/aws-s3/lib/bucket.ts#L21
    if (tags && tags["aws-cdk:auto-delete-objects"] === "true") {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    if (/^Custom::(CDK(BucketDeployment)|S3AutoDeleteObjects)/i.test(construct.node.id)) {
        flags.add(types_1.FlagEnum.EXTRANEOUS);
    }
    return Array.from(flags.values());
}
exports.inferFlags = inferFlags;
/**
 * Indicates if given construct is an import (eg: `s3.Bucket.fromBucketArn()`)
 */
function isImportConstruct(construct) {
    if (!aws_cdk_lib_1.Resource.isResource(construct)) {
        return false;
    }
    // CDK import constructs extend based resource classes via `class Import extends XXXBase` syntax.
    // https://github.com/aws/aws-cdk/blob/main/packages/%40aws-cdk/aws-s3/lib/bucket.ts#L1621
    return construct.constructor.name === "Import";
}
exports.isImportConstruct = isImportConstruct;
/**
 * Resolve an imported resources arn to tokenized hash value of arn.
 * @see {@link tokenizeImportArn}
 * @param construct {Construct} Imported resource to resolve arn for.
 * @returns If construct is an imported resource and able to infer the arn for it then the tokenized arn value is returned, otherwise undefined
 */
function resolveImportedConstructArnToken(construct) {
    if (!isImportConstruct(construct)) {
        return undefined;
    }
    for (const [key, desc] of Object.entries(Object.getOwnPropertyDescriptors(construct))) {
        if (key.endsWith("Arn") &&
            typeof desc.value === "string" &&
            desc.value.startsWith("arn:")) {
            return tokenizeImportArn(aws_cdk_lib_1.Stack.of(construct).resolve(desc.value));
        }
    }
    return undefined;
}
exports.resolveImportedConstructArnToken = resolveImportedConstructArnToken;
/**
 * Generate token for imported resource arn used to resolve references.
 *
 * Imported resources are CDK `s3.Bucket.fromBucketArn()` like resources
 * that are external from the application.
 * @param value The value to tokenize, which is usually an object with nested `Fn:Join: ...["arn:", ...]` format.
 * @returns Consistent string hash prefixed with `ImportArn-` prefix.
 */
function tokenizeImportArn(value) {
    return generateConsistentUUID(value, "ImportArn-");
}
exports.tokenizeImportArn = tokenizeImportArn;
/**
 * Infers CloudFormation Type for a given import resource.
 * @param construct {Construct} Import construct such as `s3.Bucket.fromBucketArn()`.
 * @param constructInfo {ConstructInfo} Construct info like fqn
 * @returns Returns Cloudformation resource type if it can be inferred, otherwise undefined.
 */
function inferImportCfnType(construct, constructInfo) {
    if (!isImportConstruct(construct) || !constructInfo) {
        return undefined;
    }
    const [source, pkg, resourceBase] = constructInfo.fqn.split(".");
    if (source !== "aws-cdk-lib" ||
        !pkg.startsWith("aws_") ||
        !resourceBase ||
        !resourceBase.endsWith("Base")) {
        return undefined;
    }
    try {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const pkgModule = require(`aws-cdk-lib/${pkg.replace("_", "-")}`);
        const cfnResource = "Cfn" + resourceBase.replace(/Base$/, "");
        if (cfnResource in pkgModule) {
            return pkgModule[cfnResource].CFN_RESOURCE_TYPE_NAME;
        }
    }
    catch (error) {
        // ignore
    }
    return undefined;
}
exports.inferImportCfnType = inferImportCfnType;
/** @internal */
function _isCfnFqn(fqn) {
    return /^aws-cdk-lib\.[^.]+\.Cfn[^.]+$/.test(fqn);
}
//# sourceMappingURL=data:application/json;base64,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