"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudscapeReactTsWebsiteProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const Mustache = require("mustache");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const web_1 = require("projen/lib/web");
/**
 * Synthesizes a Cloudscape React Typescript Website Project.
 *
 * @pjid cloudscape-react-ts-website
 */
class CloudscapeReactTsWebsiteProject extends web_1.ReactTypeScriptProject {
    constructor(options) {
        super({
            ...options,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            name: options.name,
            sampleCode: false,
            prettier: options.prettier || true,
            packageManager: options.parent && options.parent instanceof javascript_1.NodeProject
                ? options.parent.package.packageManager
                : options.packageManager ?? javascript_1.NodePackageManager.PNPM,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../samples/cloudscape-react-ts-website/README.md"))
                    .toString(),
            },
            gitignore: [
                "public/runtime-config.json",
                "public/api.json",
                "public/*/api.json",
                ...(options.gitignore || []),
            ],
        });
        this.typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        this.typeSafeWebSocketApis = options.typeSafeWebSocketApis ?? [];
        this.addDeps("@aws-northstar/ui", "@cloudscape-design/components", "@cloudscape-design/board-components", "react-router-dom");
        this.testTask.reset();
        const lintTask = this.tasks.tryFind("eslint");
        lintTask && this.testTask.spawn(lintTask);
        this.testTask.exec("react-scripts test --watchAll=false --passWithNoTests");
        this.applicationName = options.applicationName ?? "Sample App";
        this.publicDir = options.publicDir ?? "public";
        const srcDir = path.resolve(__dirname, "../samples/cloudscape-react-ts-website/src");
        const publicDir = path.resolve(__dirname, "../samples/cloudscape-react-ts-website/public");
        this.typeSafeApis.forEach((typeSafeApi) => {
            const hooks = typeSafeApi.library?.typescriptReactQueryHooks;
            const libraryHooksPackage = hooks?.package?.packageName;
            const libraryHooksPackageVersion = hooks?.package?.manifest.version;
            if (!libraryHooksPackage) {
                throw new Error("Cannot pass in a Type Safe Api without React Hooks Library configured!");
            }
            this.addDeps(`${libraryHooksPackage}@${libraryHooksPackageVersion}`);
            this.setupApiExplorer(typeSafeApi);
        });
        this.typeSafeWebSocketApis.forEach((typeSafeWebSocketApi) => {
            const hooks = typeSafeWebSocketApi.library?.typescriptWebSocketHooks;
            const client = typeSafeWebSocketApi.library?.typescriptWebSocketClient;
            if (!hooks || !client) {
                throw new Error("Cannot pass in a Type Safe WebSocket Api without React Hooks Library configured");
            }
            this.addDeps(`${hooks.package.packageName}@${hooks.package.manifest.version}`, `${client.package.packageName}@${client.package.manifest.version}`);
            this.setupApiExplorer(typeSafeWebSocketApi);
        });
        const apis = this.typeSafeApis.map((tsApi, idx) => ({
            apiName: tsApi.model.apiName,
            isLast: idx === this.typeSafeApis.length - 1,
            apiNameSafe: tsApi.model.apiName
                ?.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""),
            hooksPackage: tsApi.library?.typescriptReactQueryHooks?.package?.packageName,
        }));
        const webSocketApis = this.typeSafeWebSocketApis.map((tsApi, idx) => ({
            apiName: tsApi.model.apiName,
            isLast: idx === this.typeSafeWebSocketApis.length - 1,
            apiNameSafe: tsApi.model.apiName
                ?.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""),
            hooksPackage: tsApi.library?.typescriptWebSocketHooks?.package?.packageName,
            clientPackage: tsApi.library?.typescriptWebSocketClient?.package?.packageName,
        }));
        const mustacheConfig = {
            applicationName: this.applicationName,
            hasApis: apis.length + webSocketApis.length > 0,
            typeSafeApis: apis,
            typeSafeApisReversed: [...apis].reverse(),
            typeSafeWebSocketApis: webSocketApis,
            typeSafeWebSocketApisReversed: [...webSocketApis].reverse(),
        };
        new projen_1.SampleDir(this, this.srcdir, {
            files: {
                ...Object.fromEntries(this.buildSampleDirEntries(srcDir, [], mustacheConfig)),
            },
        });
        new projen_1.SampleDir(this, this.publicDir, {
            files: {
                ...Object.fromEntries(this.buildSampleDirEntries(publicDir, [], mustacheConfig)),
            },
        });
        // Linting is managed as part of the test task already, so disable react-scripts running eslint again
        this.tasks.addEnvironment("DISABLE_ESLINT_PLUGIN", "true");
        // Relax EsLint and TSC for dev
        this.tasks.tryFind("dev")?.env("ESLINT_NO_DEV_ERRORS", "true");
        this.tasks.tryFind("dev")?.env("TSC_COMPILE_ON_ERROR", "true");
    }
    setupApiExplorer(tsApi) {
        this.addDevDeps("@types/swagger-ui-react", "@types/uuid");
        this.addDeps("swagger-ui-react@5.5.0", "aws4fetch", "uuid", "@aws-crypto/sha256-js", "@aws-sdk/signature-v4", "@aws-sdk/protocol-http");
        const targetApiSpecFolder = `public/${tsApi.model.apiName}`;
        const targetApiSpecPath = `${targetApiSpecFolder}/api.json`;
        this.preCompileTask.exec(`rm -rf ${targetApiSpecFolder}`);
        this.preCompileTask.exec(`mkdir -p ${targetApiSpecFolder} && cp ${path.relative(this.outdir, tsApi.model.outdir)}/.api.json ${targetApiSpecPath}`);
    }
    buildSampleDirEntries(dir, pathPrefixes = [], mustacheConfig) {
        return fs
            .readdirSync(dir, { withFileTypes: true })
            .filter((f) => mustacheConfig.typeSafeApis.length +
            mustacheConfig.typeSafeWebSocketApis.length >
            0 || !`${pathPrefixes.join("/")}${f.name}`.includes("ApiExplorer"))
            .filter((f) => mustacheConfig.typeSafeApis.length > 0 ||
            !`${pathPrefixes.join("/")}${f.name}`.includes("TypeSafeApiClient"))
            .filter((f) => mustacheConfig.typeSafeWebSocketApis.length > 0 ||
            !`${pathPrefixes.join("/")}${f.name}`.includes("TypeSafeWebSocketApiClient"))
            .flatMap((f) => f.isDirectory()
            ? this.buildSampleDirEntries(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig)
            : [
                [
                    `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`,
                    Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig),
                ],
            ]);
    }
}
exports.CloudscapeReactTsWebsiteProject = CloudscapeReactTsWebsiteProject;
_a = JSII_RTTI_SYMBOL_1;
CloudscapeReactTsWebsiteProject[_a] = { fqn: "@aws/pdk.cloudscape_react_ts_website.CloudscapeReactTsWebsiteProject", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,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