"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructureJavaProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const infrastructure_commands_1 = require("../../components/infrastructure-commands");
const consts_1 = require("../../consts");
/**
 * Synthesizes a Infrastructure Java Project.
 */
class InfrastructureJavaProject extends awscdk_1.AwsCdkJavaApp {
    constructor(options) {
        const groupId = options.groupId ?? "software.aws.infra";
        const artifactId = options.artifactId ?? "infra";
        super({
            ...options,
            cdkVersion: options.cdkVersion ?? "2.133.0",
            cdkVersionPinning: true,
            sample: false,
            junit: false,
            groupId,
            artifactId,
            mainClass: `${groupId}.Main`,
            version: options.version ?? "0.0.0",
            name: options.name,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/java/README.md"))
                    .toString(),
            },
        });
        // Pin constructs version
        this.deps.removeDependency("software.constructs/constructs", projen_1.DependencyType.RUNTIME);
        [
            "software.constructs/constructs@10.3.0",
            "io.github.cdklabs/cdknag@2.15.16",
        ].forEach((d) => this.addDependency(d));
        infrastructure_commands_1.InfrastructureCommands.ensure(this);
        this.pom.addPlugin("org.apache.maven.plugins/maven-surefire-plugin@3.1.2");
        this.pom.addPlugin("org.apache.maven.plugins/maven-compiler-plugin@3.8.1", {
            configuration: {
                release: "11",
            },
        });
        if (options.junit !== false) {
            [
                "org.junit.jupiter/junit-jupiter-api@5.10.2",
                "org.junit.jupiter/junit-jupiter-engine@5.10.2",
                "io.github.origin-energy/java-snapshot-testing-junit5@4.0.7",
                "io.github.origin-energy/java-snapshot-testing-plugin-jackson@4.0.7",
                "org.slf4j/slf4j-simple@2.0.0-alpha0",
            ].forEach((d) => this.addTestDependency(d));
            this.testTask.exec("mvn test");
        }
        this.addDependency(`software.aws/pdk@${monorepo_1.ProjectUtils.getPdkVersion()}`);
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/java/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/java/test");
        const typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        const cloudscapeReactTsWebsites = [
            ...(options.cloudscapeReactTsWebsites || []),
            ...(options.cloudscapeReactTsWebsite
                ? [options.cloudscapeReactTsWebsite]
                : []),
        ];
        typeSafeApis.forEach((tsApi) => {
            if (!tsApi.infrastructure.java) {
                throw new Error("Cannot pass in a Type Safe Api without Java Infrastructure configured!");
            }
            monorepo_1.NxProject.ensure(this).addJavaDependency(tsApi.infrastructure.java);
            // Ensure handlers are built before infra
            tsApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        });
        cloudscapeReactTsWebsites.forEach((csWebsite) => {
            // Ensure website is built before infra
            monorepo_1.NxProject.ensure(this).addImplicitDependency(csWebsite);
        });
        const mustacheConfig = {
            stackName: options.stackName || consts_1.DEFAULT_STACK_NAME,
            groupId,
            typeSafeApis: this.generateTypeSafeMustacheConfig(groupId, typeSafeApis),
            cloudscapeReactTsWebsites: cloudscapeReactTsWebsites.map((csWebsite) => {
                const websiteName = this.capitalize(csWebsite.package.packageName
                    .replace(/[^a-z0-9_]+/gi, "")
                    .replace(/^[0-9]+/gi, ""));
                return {
                    websiteName,
                    websiteNameLowercase: websiteName.toLowerCase(),
                    groupId,
                    websiteDistRelativePath: path.relative(this.outdir, `${csWebsite.outdir}/build`),
                    typeSafeApis: this.generateTypeSafeMustacheConfig(groupId, csWebsite.typeSafeApis),
                };
            }),
        };
        options.sample !== false &&
            this.emitSampleFiles(srcDir, ["src", "main"], mustacheConfig);
        options.sample !== false &&
            this.emitSampleFiles(testDir, ["src", "test"], mustacheConfig);
    }
    generateTypeSafeMustacheConfig(groupId, typeSafeApis) {
        return typeSafeApis?.map((tsApi, idx) => {
            const apiName = this.capitalize(tsApi.model
                .apiName.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""));
            return {
                apiName,
                apiNameLowercase: apiName?.toLowerCase(),
                groupId,
                infraPackage: `${tsApi?.infrastructure.java?.pom.groupId}.${tsApi?.infrastructure.java?.pom.name}.infra`,
                isLast: idx === typeSafeApis.length - 1,
            };
        });
    }
    capitalize(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true }).forEach((f) => {
            if (f.isDirectory()) {
                return this.emitSampleFiles(`${dir}/${f.name}`, [
                    ...pathPrefixes,
                    ...(f.name === "groupId"
                        ? mustacheConfig.groupId.split(".")
                        : [f.name]),
                ], mustacheConfig);
            }
            else if (f.name.endsWith("ApiConstruct.java.mustache")) {
                mustacheConfig.typeSafeApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiName}.java`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("WebsiteConstruct.java.mustache")) {
                mustacheConfig.cloudscapeReactTsWebsites.forEach((csWebsite) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${csWebsite.websiteName}.java`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), csWebsite),
                    });
                });
            }
            else {
                const contents = Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig);
                return new projen_1.SampleFile(this, `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`, {
                    contents,
                    sourcePath: (!contents && `${dir}/${f.name}`) || undefined,
                });
            }
        });
    }
}
exports.InfrastructureJavaProject = InfrastructureJavaProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructureJavaProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructureJavaProject", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,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