"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructurePyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const infrastructure_commands_1 = require("../../components/infrastructure-commands");
const consts_1 = require("../../consts");
/**
 * Synthesizes a Infrastructure Python Project.
 */
class InfrastructurePyProject extends awscdk_1.AwsCdkPythonApp {
    constructor(options) {
        const moduleName = options.moduleName ?? "infra";
        super({
            ...options,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            sample: false,
            poetry: true,
            moduleName,
            appEntrypoint: "main.py",
            pytest: options.pytest ?? false,
            version: options.version ?? "0.0.0",
            authorName: options.authorName ?? "pdkuser",
            authorEmail: options.authorEmail ?? "user@pdk.com",
            name: options.name,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/python/README.md"))
                    .toString(),
            },
        });
        infrastructure_commands_1.InfrastructureCommands.ensure(this);
        ["pytest@^7", "syrupy@^4"].forEach((devDep) => this.addDevDependency(devDep));
        ["aws_pdk@^0", "cdk_nag@^2", "python@^3.9"].forEach((dep) => this.addDependency(dep));
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/python/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/python/test");
        const typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        const cloudscapeReactTsWebsites = [
            ...(options.cloudscapeReactTsWebsites || []),
            ...(options.cloudscapeReactTsWebsite
                ? [options.cloudscapeReactTsWebsite]
                : []),
        ];
        typeSafeApis.forEach((tsApi) => {
            if (!tsApi.infrastructure.python) {
                throw new Error("Cannot pass in a Type Safe Api without Python Infrastructure configured!");
            }
            monorepo_1.NxProject.ensure(this).addPythonPoetryDependency(tsApi.infrastructure.python);
            // Ensure handlers are built before infra
            tsApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        });
        cloudscapeReactTsWebsites.forEach((csWebsite) => {
            // Ensure website is built before infra
            monorepo_1.NxProject.ensure(this).addImplicitDependency(csWebsite);
        });
        const mustacheConfig = {
            stackName: options.stackName || consts_1.DEFAULT_STACK_NAME,
            moduleName,
            typeSafeApis: this.generateTypeSafeMustacheConfig(moduleName, typeSafeApis),
            cloudscapeReactTsWebsites: cloudscapeReactTsWebsites.map((csWebsite) => {
                const websiteName = this.capitalize(csWebsite.package.packageName
                    .replace(/[^a-z0-9_]+/gi, "")
                    .replace(/^[0-9]+/gi, ""));
                return {
                    websiteName,
                    websiteNameLowercase: websiteName.toLowerCase(),
                    websiteDistRelativePath: path.relative(this.outdir, `${csWebsite.outdir}/build`),
                    typeSafeApis: this.generateTypeSafeMustacheConfig(moduleName, csWebsite.typeSafeApis),
                };
            }),
        };
        const tstDir = "tests";
        options.sample !== false &&
            this.emitSampleFiles(srcDir, [this.moduleName], mustacheConfig);
        options.sample !== false &&
            this.emitSampleFiles(testDir, [tstDir], mustacheConfig);
        this.testTask.reset(`poetry run pytest ${tstDir}/ \${CI:-'--snapshot-update'}`);
    }
    generateTypeSafeMustacheConfig(moduleName, typeSafeApis) {
        return typeSafeApis?.map((tsApi, idx) => {
            const apiName = this.capitalize(tsApi.model
                .apiName.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""));
            return {
                apiName,
                apiNameLowercase: apiName?.toLowerCase(),
                infraPackage: tsApi.infrastructure.python?.moduleName,
                moduleName,
                isLast: idx === typeSafeApis.length - 1,
            };
        });
    }
    capitalize(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true }).forEach((f) => {
            if (f.isDirectory()) {
                return this.emitSampleFiles(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig);
            }
            else if (f.name.endsWith("api.py.mustache")) {
                mustacheConfig.typeSafeApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiNameLowercase}.py`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("website.py.mustache")) {
                mustacheConfig.cloudscapeReactTsWebsites.forEach((csWebsite) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${csWebsite.websiteNameLowercase}.py`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), csWebsite),
                    });
                });
            }
            else {
                const contents = Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig);
                return new projen_1.SampleFile(this, `${path.join(...(f.name !== "main.py.mustache" ? pathPrefixes : []), // emit at the root so package imports work correctly :(
                f.name.replace(".mustache", ""))}`, {
                    contents,
                    sourcePath: (!contents && `${dir}/${f.name}`) || undefined,
                });
            }
        });
    }
}
exports.InfrastructurePyProject = InfrastructurePyProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructurePyProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructurePyProject", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5mcmFzdHJ1Y3R1cmUtcHktcHJvamVjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImluZnJhc3RydWN0dXJlLXB5LXByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTtzQ0FDc0M7QUFDdEMseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUU3Qiw0Q0FBMEM7QUFFMUMscUNBQXFDO0FBQ3JDLG1DQUFvQztBQUNwQyw4Q0FBb0Q7QUFFcEQsc0ZBQWtGO0FBQ2xGLHlDQUFrRDtBQW9DbEQ7O0dBRUc7QUFDSCxNQUFhLHVCQUF3QixTQUFRLHdCQUFlO0lBQzFELFlBQVksT0FBdUM7UUFDakQsTUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsSUFBSSxPQUFPLENBQUM7UUFFakQsS0FBSyxDQUFDO1lBQ0osR0FBRyxPQUFPO1lBQ1YsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVLElBQUksT0FBTztZQUN6QyxNQUFNLEVBQUUsS0FBSztZQUNiLE1BQU0sRUFBRSxJQUFJO1lBQ1osVUFBVTtZQUNWLGFBQWEsRUFBRSxTQUFTO1lBQ3hCLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxJQUFJLEtBQUs7WUFDL0IsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPLElBQUksT0FBTztZQUNuQyxVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVUsSUFBSSxTQUFTO1lBQzNDLFdBQVcsRUFBRSxPQUFPLENBQUMsV0FBVyxJQUFJLGNBQWM7WUFDbEQsSUFBSSxFQUFFLE9BQU8sQ0FBQyxJQUFJO1lBQ2xCLE1BQU0sRUFBRTtnQkFDTixRQUFRLEVBQUUsRUFBRTtxQkFDVCxZQUFZLENBQ1gsSUFBSSxDQUFDLE9BQU8sQ0FDVixTQUFTLEVBQ1Qsa0RBQWtELENBQ25ELENBQ0Y7cUJBQ0EsUUFBUSxFQUFFO2FBQ2Q7U0FDRixDQUFDLENBQUM7UUFFSCxnREFBc0IsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFcEMsQ0FBQyxXQUFXLEVBQUUsV0FBVyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FDNUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUM5QixDQUFDO1FBQ0YsQ0FBQyxZQUFZLEVBQUUsWUFBWSxFQUFFLGFBQWEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQzFELElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQ3hCLENBQUM7UUFFRixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUN6QixTQUFTLEVBQ1QsNENBQTRDLENBQzdDLENBQUM7UUFDRixNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUMxQixTQUFTLEVBQ1QsNkNBQTZDLENBQzlDLENBQUM7UUFFRixNQUFNLFlBQVksR0FBRztZQUNuQixHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksSUFBSSxFQUFFLENBQUM7WUFDL0IsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7U0FDdEQsQ0FBQztRQUNGLE1BQU0seUJBQXlCLEdBQUc7WUFDaEMsR0FBRyxDQUFDLE9BQU8sQ0FBQyx5QkFBeUIsSUFBSSxFQUFFLENBQUM7WUFDNUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyx3QkFBd0I7Z0JBQ2xDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQztnQkFDcEMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztTQUNSLENBQUM7UUFFRixZQUFZLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUU7WUFDN0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFO2dCQUNoQyxNQUFNLElBQUksS0FBSyxDQUNiLDBFQUEwRSxDQUMzRSxDQUFDO2FBQ0g7WUFDRCxvQkFBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyx5QkFBeUIsQ0FDOUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQzVCLENBQUM7WUFDRix5Q0FBeUM7WUFDekMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUU7Z0JBQ3RDLG9CQUFTLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLHFCQUFxQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ3hELENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCx5QkFBeUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRTtZQUM5Qyx1Q0FBdUM7WUFDdkMsb0JBQVMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMscUJBQXFCLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDMUQsQ0FBQyxDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRztZQUNyQixTQUFTLEVBQUUsT0FBTyxDQUFDLFNBQVMsSUFBSSwyQkFBa0I7WUFDbEQsVUFBVTtZQUNWLFlBQVksRUFBRSxJQUFJLENBQUMsOEJBQThCLENBQy9DLFVBQVUsRUFDVixZQUFZLENBQ2I7WUFDRCx5QkFBeUIsRUFBRSx5QkFBeUIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRTtnQkFDckUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FDakMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxXQUFXO3FCQUMxQixPQUFPLENBQUMsZUFBZSxFQUFFLEVBQUUsQ0FBQztxQkFDNUIsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FDNUIsQ0FBQztnQkFDRixPQUFPO29CQUNMLFdBQVc7b0JBQ1gsb0JBQW9CLEVBQUUsV0FBVyxDQUFDLFdBQVcsRUFBRTtvQkFDL0MsdUJBQXVCLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FDcEMsSUFBSSxDQUFDLE1BQU0sRUFDWCxHQUFHLFNBQVMsQ0FBQyxNQUFNLFFBQVEsQ0FDNUI7b0JBQ0QsWUFBWSxFQUFFLElBQUksQ0FBQyw4QkFBOEIsQ0FDL0MsVUFBVSxFQUNWLFNBQVMsQ0FBQyxZQUFZLENBQ3ZCO2lCQUNGLENBQUM7WUFDSixDQUFDLENBQUM7U0FDSCxDQUFDO1FBRUYsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDO1FBRXZCLE9BQU8sQ0FBQyxNQUFNLEtBQUssS0FBSztZQUN0QixJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sRUFBRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUNsRSxPQUFPLENBQUMsTUFBTSxLQUFLLEtBQUs7WUFDdEIsSUFBSSxDQUFDLGVBQWUsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxNQUFNLENBQUMsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUUxRCxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FDakIscUJBQXFCLE1BQU0sK0JBQStCLENBQzNELENBQUM7SUFDSixDQUFDO0lBRU8sOEJBQThCLENBQ3BDLFVBQWtCLEVBQ2xCLFlBQW1DO1FBRW5DLE9BQU8sWUFBWSxFQUFFLEdBQUcsQ0FBQyxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsRUFBRTtZQUN0QyxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsVUFBVSxDQUM3QixLQUFLLENBQUMsS0FBSztpQkFDUixPQUFRLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxFQUFFLENBQUM7aUJBQ3JDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQzVCLENBQUM7WUFDRixPQUFPO2dCQUNMLE9BQU87Z0JBQ1AsZ0JBQWdCLEVBQUUsT0FBTyxFQUFFLFdBQVcsRUFBRTtnQkFDeEMsWUFBWSxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLFVBQVU7Z0JBQ3JELFVBQVU7Z0JBQ1YsTUFBTSxFQUFFLEdBQUcsS0FBSyxZQUFZLENBQUMsTUFBTSxHQUFHLENBQUM7YUFDeEMsQ0FBQztRQUNKLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLFVBQVUsQ0FBQyxJQUFZO1FBQzdCLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFTyxlQUFlLENBQ3JCLEdBQVcsRUFDWCxlQUF5QixFQUFFLEVBQzNCLGNBQW1CO1FBRW5CLEVBQUUsQ0FBQyxXQUFXLENBQUMsR0FBRyxFQUFFLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUU7WUFDekQsSUFBSSxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUU7Z0JBQ25CLE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FDekIsR0FBRyxHQUFHLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxFQUNsQixDQUFDLEdBQUcsWUFBWSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFDekIsY0FBYyxDQUNmLENBQUM7YUFDSDtpQkFBTSxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDLEVBQUU7Z0JBQzdDLGNBQWMsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBVSxFQUFFLEVBQUU7b0JBQ2pELElBQUksbUJBQVUsQ0FDWixJQUFJLEVBQ0osR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsWUFBWSxFQUFFLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixLQUFLLENBQUMsRUFBRSxFQUMvRDt3QkFDRSxRQUFRLEVBQUUsUUFBUSxDQUFDLE1BQU0sQ0FDdkIsRUFBRSxDQUFDLFlBQVksQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxRQUFRLEVBQUUsRUFDOUMsS0FBSyxDQUNOO3FCQUNGLENBQ0YsQ0FBQztnQkFDSixDQUFDLENBQUMsQ0FBQzthQUNKO2lCQUFNLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMscUJBQXFCLENBQUMsRUFBRTtnQkFDakQsY0FBYyxDQUFDLHlCQUF5QixDQUFDLE9BQU8sQ0FBQyxDQUFDLFNBQWMsRUFBRSxFQUFFO29CQUNsRSxJQUFJLG1CQUFVLENBQ1osSUFBSSxFQUNKLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FDVixHQUFHLFlBQVksRUFDZixHQUFHLFNBQVMsQ0FBQyxvQkFBb0IsS0FBSyxDQUN2QyxFQUFFLEVBQ0g7d0JBQ0UsUUFBUSxFQUFFLFFBQVEsQ0FBQyxNQUFNLENBQ3ZCLEVBQUUsQ0FBQyxZQUFZLENBQUMsR0FBRyxHQUFHLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsUUFBUSxFQUFFLEVBQzlDLFNBQVMsQ0FDVjtxQkFDRixDQUNGLENBQUM7Z0JBQ0osQ0FBQyxDQUFDLENBQUM7YUFDSjtpQkFBTTtnQkFDTCxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsTUFBTSxDQUM5QixFQUFFLENBQUMsWUFBWSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLFFBQVEsRUFBRSxFQUM5QyxjQUFjLENBQ2YsQ0FBQztnQkFDRixPQUFPLElBQUksbUJBQVUsQ0FDbkIsSUFBSSxFQUNKLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FDVixHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSx3REFBd0Q7Z0JBQ2hILENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FDaEMsRUFBRSxFQUNIO29CQUNFLFFBQVE7b0JBQ1IsVUFBVSxFQUFFLENBQUMsQ0FBQyxRQUFRLElBQUksR0FBRyxHQUFHLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDLElBQUksU0FBUztpQkFDM0QsQ0FDRixDQUFDO2FBQ0g7UUFDSCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBeE1ILDBEQXlNQyIsInNvdXJjZXNDb250ZW50IjpbIi8qISBDb3B5cmlnaHQgW0FtYXpvbi5jb21dKGh0dHA6Ly9hbWF6b24uY29tLyksIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMCAqL1xuaW1wb3J0ICogYXMgZnMgZnJvbSBcImZzXCI7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBDbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVQcm9qZWN0IH0gZnJvbSBcIkBhd3MvY2xvdWRzY2FwZS1yZWFjdC10cy13ZWJzaXRlXCI7XG5pbXBvcnQgeyBOeFByb2plY3QgfSBmcm9tIFwiQGF3cy9tb25vcmVwb1wiO1xuaW1wb3J0IHsgVHlwZVNhZmVBcGlQcm9qZWN0IH0gZnJvbSBcIkBhd3MvdHlwZS1zYWZlLWFwaVwiO1xuaW1wb3J0ICogYXMgTXVzdGFjaGUgZnJvbSBcIm11c3RhY2hlXCI7XG5pbXBvcnQgeyBTYW1wbGVGaWxlIH0gZnJvbSBcInByb2plblwiO1xuaW1wb3J0IHsgQXdzQ2RrUHl0aG9uQXBwIH0gZnJvbSBcInByb2plbi9saWIvYXdzY2RrXCI7XG5pbXBvcnQgeyBBd3NDZGtQeXRob25BcHBPcHRpb25zIH0gZnJvbSBcIi4vYXdzLWNkay1weS1hcHAtb3B0aW9uc1wiO1xuaW1wb3J0IHsgSW5mcmFzdHJ1Y3R1cmVDb21tYW5kcyB9IGZyb20gXCIuLi8uLi9jb21wb25lbnRzL2luZnJhc3RydWN0dXJlLWNvbW1hbmRzXCI7XG5pbXBvcnQgeyBERUZBVUxUX1NUQUNLX05BTUUgfSBmcm9tIFwiLi4vLi4vY29uc3RzXCI7XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBvcHRpb25zIGZvciB0aGUgSW5mcmFzdHJ1Y3R1cmVQeVByb2plY3QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSW5mcmFzdHJ1Y3R1cmVQeVByb2plY3RPcHRpb25zIGV4dGVuZHMgQXdzQ2RrUHl0aG9uQXBwT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBTdGFjayBuYW1lLlxuICAgKlxuICAgKiBAZGVmYXVsdCBpbmZyYS1kZXZcbiAgICovXG4gIHJlYWRvbmx5IHN0YWNrTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVHlwZVNhZmVBcGkgaW5zdGFuY2UgdG8gdXNlIHdoZW4gc2V0dGluZyB1cCB0aGUgaW5pdGlhbCBwcm9qZWN0IHNhbXBsZSBjb2RlLlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgdHlwZVNhZmVBcGlzXG4gICAqL1xuICByZWFkb25seSB0eXBlU2FmZUFwaT86IFR5cGVTYWZlQXBpUHJvamVjdDtcblxuICAvKipcbiAgICogQ2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlUHJvamVjdCBpbnN0YW5jZSB0byB1c2Ugd2hlbiBzZXR0aW5nIHVwIHRoZSBpbml0aWFsIHByb2plY3Qgc2FtcGxlIGNvZGUuXG4gICAqIEBkZXByZWNhdGVkIHVzZSBjbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVzXG4gICAqL1xuICByZWFkb25seSBjbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGU/OiBDbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVQcm9qZWN0O1xuXG4gIC8qKlxuICAgKiBUeXBlU2FmZUFwaSBpbnN0YW5jZXMgdG8gdXNlIHdoZW4gc2V0dGluZyB1cCB0aGUgaW5pdGlhbCBwcm9qZWN0IHNhbXBsZSBjb2RlLlxuICAgKi9cbiAgcmVhZG9ubHkgdHlwZVNhZmVBcGlzPzogVHlwZVNhZmVBcGlQcm9qZWN0W107XG5cbiAgLyoqXG4gICAqIENsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZVByb2plY3QgaW5zdGFuY2VzIHRvIHVzZSB3aGVuIHNldHRpbmcgdXAgdGhlIGluaXRpYWwgcHJvamVjdCBzYW1wbGUgY29kZS5cbiAgICovXG4gIHJlYWRvbmx5IGNsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZXM/OiBDbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVQcm9qZWN0W107XG59XG5cbi8qKlxuICogU3ludGhlc2l6ZXMgYSBJbmZyYXN0cnVjdHVyZSBQeXRob24gUHJvamVjdC5cbiAqL1xuZXhwb3J0IGNsYXNzIEluZnJhc3RydWN0dXJlUHlQcm9qZWN0IGV4dGVuZHMgQXdzQ2RrUHl0aG9uQXBwIHtcbiAgY29uc3RydWN0b3Iob3B0aW9uczogSW5mcmFzdHJ1Y3R1cmVQeVByb2plY3RPcHRpb25zKSB7XG4gICAgY29uc3QgbW9kdWxlTmFtZSA9IG9wdGlvbnMubW9kdWxlTmFtZSA/PyBcImluZnJhXCI7XG5cbiAgICBzdXBlcih7XG4gICAgICAuLi5vcHRpb25zLFxuICAgICAgY2RrVmVyc2lvbjogb3B0aW9ucy5jZGtWZXJzaW9uID8/IFwiMi4xLjBcIixcbiAgICAgIHNhbXBsZTogZmFsc2UsXG4gICAgICBwb2V0cnk6IHRydWUsXG4gICAgICBtb2R1bGVOYW1lLFxuICAgICAgYXBwRW50cnlwb2ludDogXCJtYWluLnB5XCIsXG4gICAgICBweXRlc3Q6IG9wdGlvbnMucHl0ZXN0ID8/IGZhbHNlLFxuICAgICAgdmVyc2lvbjogb3B0aW9ucy52ZXJzaW9uID8/IFwiMC4wLjBcIixcbiAgICAgIGF1dGhvck5hbWU6IG9wdGlvbnMuYXV0aG9yTmFtZSA/PyBcInBka3VzZXJcIixcbiAgICAgIGF1dGhvckVtYWlsOiBvcHRpb25zLmF1dGhvckVtYWlsID8/IFwidXNlckBwZGsuY29tXCIsXG4gICAgICBuYW1lOiBvcHRpb25zLm5hbWUsXG4gICAgICByZWFkbWU6IHtcbiAgICAgICAgY29udGVudHM6IGZzXG4gICAgICAgICAgLnJlYWRGaWxlU3luYyhcbiAgICAgICAgICAgIHBhdGgucmVzb2x2ZShcbiAgICAgICAgICAgICAgX19kaXJuYW1lLFxuICAgICAgICAgICAgICBcIi4uLy4uLy4uL3NhbXBsZXMvaW5mcmFzdHJ1Y3R1cmUvcHl0aG9uL1JFQURNRS5tZFwiXG4gICAgICAgICAgICApXG4gICAgICAgICAgKVxuICAgICAgICAgIC50b1N0cmluZygpLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIEluZnJhc3RydWN0dXJlQ29tbWFuZHMuZW5zdXJlKHRoaXMpO1xuXG4gICAgW1wicHl0ZXN0QF43XCIsIFwic3lydXB5QF40XCJdLmZvckVhY2goKGRldkRlcCkgPT5cbiAgICAgIHRoaXMuYWRkRGV2RGVwZW5kZW5jeShkZXZEZXApXG4gICAgKTtcbiAgICBbXCJhd3NfcGRrQF4wXCIsIFwiY2RrX25hZ0BeMlwiLCBcInB5dGhvbkBeMy45XCJdLmZvckVhY2goKGRlcCkgPT5cbiAgICAgIHRoaXMuYWRkRGVwZW5kZW5jeShkZXApXG4gICAgKTtcblxuICAgIGNvbnN0IHNyY0RpciA9IHBhdGgucmVzb2x2ZShcbiAgICAgIF9fZGlybmFtZSxcbiAgICAgIFwiLi4vLi4vLi4vc2FtcGxlcy9pbmZyYXN0cnVjdHVyZS9weXRob24vc3JjXCJcbiAgICApO1xuICAgIGNvbnN0IHRlc3REaXIgPSBwYXRoLnJlc29sdmUoXG4gICAgICBfX2Rpcm5hbWUsXG4gICAgICBcIi4uLy4uLy4uL3NhbXBsZXMvaW5mcmFzdHJ1Y3R1cmUvcHl0aG9uL3Rlc3RcIlxuICAgICk7XG5cbiAgICBjb25zdCB0eXBlU2FmZUFwaXMgPSBbXG4gICAgICAuLi4ob3B0aW9ucy50eXBlU2FmZUFwaXMgfHwgW10pLFxuICAgICAgLi4uKG9wdGlvbnMudHlwZVNhZmVBcGkgPyBbb3B0aW9ucy50eXBlU2FmZUFwaV0gOiBbXSksXG4gICAgXTtcbiAgICBjb25zdCBjbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVzID0gW1xuICAgICAgLi4uKG9wdGlvbnMuY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlcyB8fCBbXSksXG4gICAgICAuLi4ob3B0aW9ucy5jbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVcbiAgICAgICAgPyBbb3B0aW9ucy5jbG91ZHNjYXBlUmVhY3RUc1dlYnNpdGVdXG4gICAgICAgIDogW10pLFxuICAgIF07XG5cbiAgICB0eXBlU2FmZUFwaXMuZm9yRWFjaCgodHNBcGkpID0+IHtcbiAgICAgIGlmICghdHNBcGkuaW5mcmFzdHJ1Y3R1cmUucHl0aG9uKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBcIkNhbm5vdCBwYXNzIGluIGEgVHlwZSBTYWZlIEFwaSB3aXRob3V0IFB5dGhvbiBJbmZyYXN0cnVjdHVyZSBjb25maWd1cmVkIVwiXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgICBOeFByb2plY3QuZW5zdXJlKHRoaXMpLmFkZFB5dGhvblBvZXRyeURlcGVuZGVuY3koXG4gICAgICAgIHRzQXBpLmluZnJhc3RydWN0dXJlLnB5dGhvblxuICAgICAgKTtcbiAgICAgIC8vIEVuc3VyZSBoYW5kbGVycyBhcmUgYnVpbHQgYmVmb3JlIGluZnJhXG4gICAgICB0c0FwaS5hbGwuaGFuZGxlcnM/LmZvckVhY2goKGhhbmRsZXIpID0+IHtcbiAgICAgICAgTnhQcm9qZWN0LmVuc3VyZSh0aGlzKS5hZGRJbXBsaWNpdERlcGVuZGVuY3koaGFuZGxlcik7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIGNsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZXMuZm9yRWFjaCgoY3NXZWJzaXRlKSA9PiB7XG4gICAgICAvLyBFbnN1cmUgd2Vic2l0ZSBpcyBidWlsdCBiZWZvcmUgaW5mcmFcbiAgICAgIE54UHJvamVjdC5lbnN1cmUodGhpcykuYWRkSW1wbGljaXREZXBlbmRlbmN5KGNzV2Vic2l0ZSk7XG4gICAgfSk7XG5cbiAgICBjb25zdCBtdXN0YWNoZUNvbmZpZyA9IHtcbiAgICAgIHN0YWNrTmFtZTogb3B0aW9ucy5zdGFja05hbWUgfHwgREVGQVVMVF9TVEFDS19OQU1FLFxuICAgICAgbW9kdWxlTmFtZSxcbiAgICAgIHR5cGVTYWZlQXBpczogdGhpcy5nZW5lcmF0ZVR5cGVTYWZlTXVzdGFjaGVDb25maWcoXG4gICAgICAgIG1vZHVsZU5hbWUsXG4gICAgICAgIHR5cGVTYWZlQXBpc1xuICAgICAgKSxcbiAgICAgIGNsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZXM6IGNsb3Vkc2NhcGVSZWFjdFRzV2Vic2l0ZXMubWFwKChjc1dlYnNpdGUpID0+IHtcbiAgICAgICAgY29uc3Qgd2Vic2l0ZU5hbWUgPSB0aGlzLmNhcGl0YWxpemUoXG4gICAgICAgICAgY3NXZWJzaXRlLnBhY2thZ2UucGFja2FnZU5hbWVcbiAgICAgICAgICAgIC5yZXBsYWNlKC9bXmEtejAtOV9dKy9naSwgXCJcIilcbiAgICAgICAgICAgIC5yZXBsYWNlKC9eWzAtOV0rL2dpLCBcIlwiKVxuICAgICAgICApO1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIHdlYnNpdGVOYW1lLFxuICAgICAgICAgIHdlYnNpdGVOYW1lTG93ZXJjYXNlOiB3ZWJzaXRlTmFtZS50b0xvd2VyQ2FzZSgpLFxuICAgICAgICAgIHdlYnNpdGVEaXN0UmVsYXRpdmVQYXRoOiBwYXRoLnJlbGF0aXZlKFxuICAgICAgICAgICAgdGhpcy5vdXRkaXIsXG4gICAgICAgICAgICBgJHtjc1dlYnNpdGUub3V0ZGlyfS9idWlsZGBcbiAgICAgICAgICApLFxuICAgICAgICAgIHR5cGVTYWZlQXBpczogdGhpcy5nZW5lcmF0ZVR5cGVTYWZlTXVzdGFjaGVDb25maWcoXG4gICAgICAgICAgICBtb2R1bGVOYW1lLFxuICAgICAgICAgICAgY3NXZWJzaXRlLnR5cGVTYWZlQXBpc1xuICAgICAgICAgICksXG4gICAgICAgIH07XG4gICAgICB9KSxcbiAgICB9O1xuXG4gICAgY29uc3QgdHN0RGlyID0gXCJ0ZXN0c1wiO1xuXG4gICAgb3B0aW9ucy5zYW1wbGUgIT09IGZhbHNlICYmXG4gICAgICB0aGlzLmVtaXRTYW1wbGVGaWxlcyhzcmNEaXIsIFt0aGlzLm1vZHVsZU5hbWVdLCBtdXN0YWNoZUNvbmZpZyk7XG4gICAgb3B0aW9ucy5zYW1wbGUgIT09IGZhbHNlICYmXG4gICAgICB0aGlzLmVtaXRTYW1wbGVGaWxlcyh0ZXN0RGlyLCBbdHN0RGlyXSwgbXVzdGFjaGVDb25maWcpO1xuXG4gICAgdGhpcy50ZXN0VGFzay5yZXNldChcbiAgICAgIGBwb2V0cnkgcnVuIHB5dGVzdCAke3RzdERpcn0vIFxcJHtDSTotJy0tc25hcHNob3QtdXBkYXRlJ31gXG4gICAgKTtcbiAgfVxuXG4gIHByaXZhdGUgZ2VuZXJhdGVUeXBlU2FmZU11c3RhY2hlQ29uZmlnKFxuICAgIG1vZHVsZU5hbWU6IHN0cmluZyxcbiAgICB0eXBlU2FmZUFwaXM/OiBUeXBlU2FmZUFwaVByb2plY3RbXVxuICApIHtcbiAgICByZXR1cm4gdHlwZVNhZmVBcGlzPy5tYXAoKHRzQXBpLCBpZHgpID0+IHtcbiAgICAgIGNvbnN0IGFwaU5hbWUgPSB0aGlzLmNhcGl0YWxpemUoXG4gICAgICAgIHRzQXBpLm1vZGVsXG4gICAgICAgICAgLmFwaU5hbWUhLnJlcGxhY2UoL1teYS16MC05X10rL2dpLCBcIlwiKVxuICAgICAgICAgIC5yZXBsYWNlKC9eWzAtOV0rL2dpLCBcIlwiKVxuICAgICAgKTtcbiAgICAgIHJldHVybiB7XG4gICAgICAgIGFwaU5hbWUsXG4gICAgICAgIGFwaU5hbWVMb3dlcmNhc2U6IGFwaU5hbWU/LnRvTG93ZXJDYXNlKCksXG4gICAgICAgIGluZnJhUGFja2FnZTogdHNBcGkuaW5mcmFzdHJ1Y3R1cmUucHl0aG9uPy5tb2R1bGVOYW1lLFxuICAgICAgICBtb2R1bGVOYW1lLFxuICAgICAgICBpc0xhc3Q6IGlkeCA9PT0gdHlwZVNhZmVBcGlzLmxlbmd0aCAtIDEsXG4gICAgICB9O1xuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBjYXBpdGFsaXplKHdvcmQ6IHN0cmluZykge1xuICAgIHJldHVybiB3b3JkLmNoYXJBdCgwKS50b1VwcGVyQ2FzZSgpICsgd29yZC5zbGljZSgxKTtcbiAgfVxuXG4gIHByaXZhdGUgZW1pdFNhbXBsZUZpbGVzKFxuICAgIGRpcjogc3RyaW5nLFxuICAgIHBhdGhQcmVmaXhlczogc3RyaW5nW10gPSBbXSxcbiAgICBtdXN0YWNoZUNvbmZpZzogYW55XG4gICkge1xuICAgIGZzLnJlYWRkaXJTeW5jKGRpciwgeyB3aXRoRmlsZVR5cGVzOiB0cnVlIH0pLmZvckVhY2goKGYpID0+IHtcbiAgICAgIGlmIChmLmlzRGlyZWN0b3J5KCkpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuZW1pdFNhbXBsZUZpbGVzKFxuICAgICAgICAgIGAke2Rpcn0vJHtmLm5hbWV9YCxcbiAgICAgICAgICBbLi4ucGF0aFByZWZpeGVzLCBmLm5hbWVdLFxuICAgICAgICAgIG11c3RhY2hlQ29uZmlnXG4gICAgICAgICk7XG4gICAgICB9IGVsc2UgaWYgKGYubmFtZS5lbmRzV2l0aChcImFwaS5weS5tdXN0YWNoZVwiKSkge1xuICAgICAgICBtdXN0YWNoZUNvbmZpZy50eXBlU2FmZUFwaXMuZm9yRWFjaCgodHNBcGk6IGFueSkgPT4ge1xuICAgICAgICAgIG5ldyBTYW1wbGVGaWxlKFxuICAgICAgICAgICAgdGhpcyxcbiAgICAgICAgICAgIGAke3BhdGguam9pbiguLi5wYXRoUHJlZml4ZXMsIGAke3RzQXBpLmFwaU5hbWVMb3dlcmNhc2V9LnB5YCl9YCxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgY29udGVudHM6IE11c3RhY2hlLnJlbmRlcihcbiAgICAgICAgICAgICAgICBmcy5yZWFkRmlsZVN5bmMoYCR7ZGlyfS8ke2YubmFtZX1gKS50b1N0cmluZygpLFxuICAgICAgICAgICAgICAgIHRzQXBpXG4gICAgICAgICAgICAgICksXG4gICAgICAgICAgICB9XG4gICAgICAgICAgKTtcbiAgICAgICAgfSk7XG4gICAgICB9IGVsc2UgaWYgKGYubmFtZS5lbmRzV2l0aChcIndlYnNpdGUucHkubXVzdGFjaGVcIikpIHtcbiAgICAgICAgbXVzdGFjaGVDb25maWcuY2xvdWRzY2FwZVJlYWN0VHNXZWJzaXRlcy5mb3JFYWNoKChjc1dlYnNpdGU6IGFueSkgPT4ge1xuICAgICAgICAgIG5ldyBTYW1wbGVGaWxlKFxuICAgICAgICAgICAgdGhpcyxcbiAgICAgICAgICAgIGAke3BhdGguam9pbihcbiAgICAgICAgICAgICAgLi4ucGF0aFByZWZpeGVzLFxuICAgICAgICAgICAgICBgJHtjc1dlYnNpdGUud2Vic2l0ZU5hbWVMb3dlcmNhc2V9LnB5YFxuICAgICAgICAgICAgKX1gLFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBjb250ZW50czogTXVzdGFjaGUucmVuZGVyKFxuICAgICAgICAgICAgICAgIGZzLnJlYWRGaWxlU3luYyhgJHtkaXJ9LyR7Zi5uYW1lfWApLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgY3NXZWJzaXRlXG4gICAgICAgICAgICAgICksXG4gICAgICAgICAgICB9XG4gICAgICAgICAgKTtcbiAgICAgICAgfSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBjb25zdCBjb250ZW50cyA9IE11c3RhY2hlLnJlbmRlcihcbiAgICAgICAgICBmcy5yZWFkRmlsZVN5bmMoYCR7ZGlyfS8ke2YubmFtZX1gKS50b1N0cmluZygpLFxuICAgICAgICAgIG11c3RhY2hlQ29uZmlnXG4gICAgICAgICk7XG4gICAgICAgIHJldHVybiBuZXcgU2FtcGxlRmlsZShcbiAgICAgICAgICB0aGlzLFxuICAgICAgICAgIGAke3BhdGguam9pbihcbiAgICAgICAgICAgIC4uLihmLm5hbWUgIT09IFwibWFpbi5weS5tdXN0YWNoZVwiID8gcGF0aFByZWZpeGVzIDogW10pLCAvLyBlbWl0IGF0IHRoZSByb290IHNvIHBhY2thZ2UgaW1wb3J0cyB3b3JrIGNvcnJlY3RseSA6KFxuICAgICAgICAgICAgZi5uYW1lLnJlcGxhY2UoXCIubXVzdGFjaGVcIiwgXCJcIilcbiAgICAgICAgICApfWAsXG4gICAgICAgICAge1xuICAgICAgICAgICAgY29udGVudHMsXG4gICAgICAgICAgICBzb3VyY2VQYXRoOiAoIWNvbnRlbnRzICYmIGAke2Rpcn0vJHtmLm5hbWV9YCkgfHwgdW5kZWZpbmVkLFxuICAgICAgICAgIH1cbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgfVxufVxuIl19