"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructureTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const javascript_1 = require("projen/lib/javascript");
const infrastructure_commands_1 = require("../../components/infrastructure-commands");
const consts_1 = require("../../consts");
/**
 * Synthesizes a Infrastructure Typescript Project.
 */
class InfrastructureTsProject extends awscdk_1.AwsCdkTypeScriptApp {
    constructor(options) {
        super({
            ...options,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            prettier: options.prettier || true,
            packageManager: options.parent && options.parent instanceof javascript_1.NodeProject
                ? options.parent.package.packageManager
                : options.packageManager,
            cdkVersion: options.cdkVersion ?? "2.1.0",
            name: options.name,
            sampleCode: false,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/typescript/README.md"))
                    .toString(),
            },
        });
        infrastructure_commands_1.InfrastructureCommands.ensure(this);
        this.addDeps("@aws/pdk", "cdk-nag");
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/typescript/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/typescript/test");
        const typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        const typeSafeWebSocketApis = options.typeSafeWebSocketApis ?? [];
        const cloudscapeReactTsWebsites = [
            ...(options.cloudscapeReactTsWebsites || []),
            ...(options.cloudscapeReactTsWebsite
                ? [options.cloudscapeReactTsWebsite]
                : []),
        ];
        [...typeSafeApis, ...typeSafeWebSocketApis].forEach((tsApi) => {
            if (!tsApi.infrastructure.typescript) {
                throw new Error("Cannot pass in a Type Safe Api without Typescript Infrastructure configured!");
            }
            this.addDeps(`${tsApi.infrastructure.typescript?.package.packageName}@${tsApi.infrastructure.typescript?.package.manifest.version}`);
            // Ensure handlers are built before infra
            tsApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        });
        cloudscapeReactTsWebsites.forEach((csWebsite) => {
            // Ensure website is built before infra
            this.addDevDeps(`${csWebsite.package.packageName}@${csWebsite.package.manifest.version}`);
        });
        const mustacheConfig = {
            stackName: options.stackName || consts_1.DEFAULT_STACK_NAME,
            typeSafeApis: this.generateTypeSafeMustacheConfig(typeSafeApis),
            typeSafeWebSocketApis: this.generateTypeSafeMustacheConfig(typeSafeWebSocketApis),
            cloudscapeReactTsWebsites: cloudscapeReactTsWebsites.map((csWebsite) => {
                const websiteName = this.capitalize(csWebsite.package.packageName
                    .replace(/[^a-z0-9_]+/gi, "")
                    .replace(/^[0-9]+/gi, ""));
                return {
                    websiteName,
                    websiteNameLowercase: websiteName.toLowerCase(),
                    websiteDistRelativePath: path.relative(this.outdir, `${csWebsite.outdir}/build`),
                    typeSafeApis: this.generateTypeSafeMustacheConfig(csWebsite.typeSafeApis),
                    typeSafeWebSocketApis: this.generateTypeSafeMustacheConfig(csWebsite.typeSafeWebSocketApis),
                };
            }),
        };
        options.sampleCode !== false &&
            this.emitSampleFiles(srcDir, ["src"], mustacheConfig);
        options.sampleCode !== false &&
            this.emitSampleFiles(testDir, ["test"], mustacheConfig);
        const eslintTask = this.tasks.tryFind("eslint");
        this.testTask.reset();
        this.testTask.exec("jest --passWithNoTests ${CI:-'--updateSnapshot'}");
        eslintTask && this.testTask.spawn(eslintTask);
    }
    generateTypeSafeMustacheConfig(typeSafeApis) {
        return (typeSafeApis ?? []).map((tsApi, idx) => {
            const apiName = this.capitalize(tsApi.model
                .apiName.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""));
            return {
                apiName,
                apiNameLowercase: apiName?.toLowerCase(),
                infraPackage: tsApi.infrastructure.typescript?.package.packageName,
                isLast: idx === typeSafeApis.length - 1,
            };
        });
    }
    capitalize(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true }).forEach((f) => {
            if (f.isDirectory()) {
                this.emitSampleFiles(`${dir}/${f.name}`, [...pathPrefixes, f.name], mustacheConfig);
            }
            else if (f.name.endsWith("websocketapi.ts.mustache")) {
                mustacheConfig.typeSafeWebSocketApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiNameLowercase}.ts`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("api.ts.mustache")) {
                mustacheConfig.typeSafeApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiNameLowercase}.ts`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("website.ts.mustache")) {
                mustacheConfig.cloudscapeReactTsWebsites.forEach((csWebsite) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${csWebsite.websiteNameLowercase}.ts`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), csWebsite),
                    });
                });
            }
            else {
                new projen_1.SampleFile(this, `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`, {
                    contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig),
                });
            }
        });
    }
}
exports.InfrastructureTsProject = InfrastructureTsProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructureTsProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructureTsProject", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,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