"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxConfigurator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const nx_project_1 = require("./nx-project");
const nx_workspace_1 = require("./nx-workspace");
const utils_1 = require("../utils");
const DEFAULT_PYTHON_VERSION = "3";
const DEFAULT_LICENSE = "Apache-2.0";
/**
 * Configues common NX related tasks and methods.
 */
class NxConfigurator extends projen_1.Component {
    constructor(project, options) {
        super(project);
        this.nxPlugins = {};
        project.addGitIgnore(".nx/cache");
        project.addTask("run-many", {
            receiveArgs: true,
            exec: utils_1.NodePackageUtils.command.exec(utils_1.NodePackageUtils.getNodePackageManager(this.project, javascript_1.NodePackageManager.NPM), "nx", "run-many"),
            description: "Run task against multiple workspace projects",
        });
        project.addTask("graph", {
            receiveArgs: true,
            exec: utils_1.NodePackageUtils.command.exec(utils_1.NodePackageUtils.getNodePackageManager(this.project, javascript_1.NodePackageManager.NPM), "nx", "graph"),
            description: "Generate dependency graph for monorepo",
        });
        this.licenseOptions = options?.licenseOptions;
        this.nx = nx_workspace_1.NxWorkspace.of(project) || new nx_workspace_1.NxWorkspace(project);
        this.nx.affected.defaultBase = options?.defaultReleaseBranch ?? "mainline";
    }
    patchPoetryEnv(project) {
        // Since the root monorepo is a poetry project and sets the VIRTUAL_ENV, and poetry env info -p will print
        // the virtual env set in the VIRTUAL_ENV variable if set, we need to unset it to ensure the local project's
        // env is used.
        if (utils_1.ProjectUtils.isNamedInstanceOf(project.depsManager, python_1.Poetry)) {
            ["install", "install:ci"].forEach((t) => {
                const task = project.tasks.tryFind(t);
                // Setup env
                const createVenvCmd = "poetry env use python$PYTHON_VERSION";
                !task?.steps.find((s) => s.exec === createVenvCmd) &&
                    task?.prependExec(createVenvCmd);
                // Ensure the projen & pdk bins are removed from the venv as we always want to use the npx variant
                const removeBinsCmd = "rm -f `poetry env info -p`/bin/projen `poetry env info -p`/bin/pdk";
                !task?.steps.find((s) => s.exec === removeBinsCmd) &&
                    task?.exec(removeBinsCmd);
                const pythonVersion = project.deps.tryGetDependency("python")?.version;
                task.env("PYTHON_VERSION", pythonVersion && !pythonVersion?.startsWith("^")
                    ? pythonVersion
                    : `$(pyenv latest ${pythonVersion?.substring(1).split(".")[0] ||
                        DEFAULT_PYTHON_VERSION} | cut -d '.' -f 1,2 || echo '')`);
            });
            project.tasks.addEnvironment("VIRTUAL_ENV", "$(env -u VIRTUAL_ENV poetry env info -p || echo '')");
            project.tasks.addEnvironment("PATH", "$(echo $(env -u VIRTUAL_ENV poetry env info -p || echo '')/bin:$PATH)");
        }
    }
    patchPythonProjects(projects) {
        projects.forEach((p) => {
            if (utils_1.ProjectUtils.isNamedInstanceOf(p, python_1.PythonProject)) {
                this.patchPoetryEnv(p);
            }
            this.patchPythonProjects(p.subprojects);
        });
    }
    /**
     * Overrides "build" related project tasks (build, compile, test, etc.) with `npx nx run-many` format.
     * @param task - The task or task name to override
     * @param options - Nx run-many options
     * @param overrideOptions - Options for overriding the task
     * @returns - The task that was overridden
     * @internal
     */
    _overrideNxBuildTask(task, options, overrideOptions) {
        if (typeof task === "string") {
            task = this.project.tasks.tryFind(task);
        }
        if (task == null) {
            return;
        }
        if (overrideOptions?.force) {
            // @ts-ignore - private property
            task._locked = false;
        }
        task.reset(this.execNxRunManyCommand(options), {
            receiveArgs: true,
        });
        task.description += " for all affected projects";
        if (overrideOptions?.disableReset) {
            // Prevent any further resets of the task to force it to remain as the overridden nx build task
            task.reset = () => { };
        }
        return task;
    }
    /**
     * Adds a command to upgrade all python subprojects to the given task
     * @param monorepo the monorepo project
     * @param task the task to add the command to
     * @internal
     */
    _configurePythonSubprojectUpgradeDeps(monorepo, task) {
        // Upgrade deps for
        const pythonSubprojects = monorepo.subprojects.filter((p) => utils_1.ProjectUtils.isNamedInstanceOf(p, python_1.PythonProject));
        if (pythonSubprojects.length > 0) {
            task.exec(this.execNxRunManyCommand({
                target: "install",
                projects: pythonSubprojects.map((p) => p.name),
            }), { receiveArgs: true });
        }
    }
    /**
     * Returns the install task or creates one with nx installation command added.
     *
     * Note: this should only be called from non-node projects
     *
     * @param nxPlugins additional plugins to install
     * @returns install task
     */
    ensureNxInstallTask(nxPlugins) {
        this.nxPlugins = nxPlugins;
        const installTask = this.project.tasks.tryFind("install") ?? this.project.addTask("install");
        installTask.exec("pnpm i --no-frozen-lockfile");
        (this.project.tasks.tryFind("install:ci") ??
            this.project.addTask("install:ci")).exec("pnpm i --frozen-lockfile");
        return installTask;
    }
    /**
     * Helper to format `npx nx run-many ...` style command execution in package manager.
     * @param options
     */
    execNxRunManyCommand(options) {
        return utils_1.NodePackageUtils.command.exec(utils_1.NodePackageUtils.getNodePackageManager(this.project, javascript_1.NodePackageManager.NPM), ...this.composeNxRunManyCommand(options));
    }
    /**
     * Helper to format `npx nx run-many ...` style command
     * @param options
     */
    composeNxRunManyCommand(options) {
        const args = [];
        if (options.configuration) {
            args.push(`--configuration=${options.configuration}`);
        }
        if (options.runner) {
            args.push(`--runner=${options.runner}`);
        }
        if (options.parallel) {
            args.push(`--parallel=${options.parallel}`);
        }
        if (options.skipCache) {
            args.push("--skip-nx-cache");
        }
        if (options.ignoreCycles) {
            args.push("--nx-ignore-cycles");
        }
        if (options.noBail !== true) {
            args.push("--nx-bail");
        }
        if (options.projects && options.projects.length) {
            args.push(`--projects=${options.projects.join(",")}`);
        }
        if (options.exclude) {
            args.push(`--exclude=${options.exclude}`);
        }
        if (options.verbose) {
            args.push("--verbose");
        }
        return [
            "nx",
            "run-many",
            `--target=${options.target}`,
            `--output-style=${options.outputStyle || "stream"}`,
            ...args,
        ];
    }
    /**
     * Add project task that executes `npx nx run-many ...` style command.
     */
    addNxRunManyTask(name, options) {
        return this.project.addTask(name, {
            receiveArgs: true,
            exec: this.execNxRunManyCommand(options),
        });
    }
    /**
     * Create an implicit dependency between two Projects. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     * @throws error if this is called on a dependent which does not have a NXProject component attached.
     */
    addImplicitDependency(dependent, dependee) {
        nx_project_1.NxProject.ensure(dependent).addImplicitDependency(dependee);
    }
    /**
     * Adds a dependency between two Java Projects in the monorepo.
     * @param dependent project you want to have the dependency
     * @param dependee project you wish to depend on
     */
    addJavaDependency(dependent, dependee) {
        nx_project_1.NxProject.ensure(dependent).addJavaDependency(dependee);
    }
    /**
     * Adds a dependency between two Python Projects in the monorepo. The dependent must have Poetry enabled.
     * @param dependent project you want to have the dependency (must be a Poetry Python Project)
     * @param dependee project you wish to depend on
     * @throws error if the dependent does not have Poetry enabled
     */
    addPythonPoetryDependency(dependent, dependee) {
        nx_project_1.NxProject.ensure(dependent).addPythonPoetryDependency(dependee);
    }
    /**
     * Ensures that all non-root projects have NxProject applied.
     * @internal
     */
    _ensureNxProjectGraph() {
        function _ensure(_project) {
            if (_project.root === _project)
                return;
            nx_project_1.NxProject.ensure(_project);
            _project.subprojects.forEach((p) => {
                _ensure(p);
            });
        }
        this.project.subprojects.forEach(_ensure);
    }
    /**
     * Emits package.json for non-node NX monorepos.
     * @internal
     */
    _emitPackageJson() {
        if (!utils_1.ProjectUtils.isNamedInstanceOf(this.project, javascript_1.NodeProject) &&
            !this.project.tryFindFile("package.json")) {
            new projen_1.JsonFile(this.project, "package.json", {
                obj: {
                    devDependencies: {
                        ...this.nxPlugins,
                        nx: "^16",
                        "@nx/devkit": "^16",
                    },
                    private: true,
                    engines: {
                        node: ">=16",
                        pnpm: ">=8 <9",
                    },
                    scripts: Object.fromEntries(this.project.tasks.all
                        .filter((t) => t.name !== "install")
                        .map((c) => [
                        c.name,
                        utils_1.NodePackageUtils.command.projen(javascript_1.NodePackageManager.PNPM, c.name),
                    ])),
                },
            }).synthesize();
        }
        if (!utils_1.ProjectUtils.isNamedInstanceOf(this.project, javascript_1.NodeProject) &&
            !this.project.tryFindFile("pnpm-workspace.yaml")) {
            new projen_1.YamlFile(this.project, "pnpm-workspace.yaml", {
                obj: {
                    packages: this.project.subprojects
                        .filter((p) => utils_1.ProjectUtils.isNamedInstanceOf(p, javascript_1.NodeProject))
                        .map((p) => path.relative(this.project.outdir, p.outdir)),
                },
            }).synthesize();
        }
        if (!utils_1.ProjectUtils.isNamedInstanceOf(this.project, javascript_1.NodeProject) &&
            !this.project.tryFindFile(".npmrc")) {
            new projen_1.IniFile(this.project, ".npmrc", {
                obj: {
                    "resolution-mode": "highest",
                    yes: "true",
                    "prefer-workspace-packages": "true",
                    "link-workspace-packages": "true",
                },
            }).synthesize();
        }
        else if (utils_1.ProjectUtils.isNamedInstanceOf(this.project, javascript_1.NodeProject) &&
            this.project.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            this.project.npmrc.addConfig("prefer-workspace-packages", "true");
            this.project.npmrc.addConfig("link-workspace-packages", "true");
            this.project.npmrc.addConfig("yes", "true");
        }
    }
    _invokeInstallCITasks() {
        const cmd = utils_1.NodePackageUtils.command.exec(utils_1.ProjectUtils.isNamedInstanceOf(this.project, javascript_1.NodeProject)
            ? this.project.package.packageManager
            : javascript_1.NodePackageManager.NPM, ...this.composeNxRunManyCommand({
            target: "install:ci",
        }));
        const task = this.project.tasks.tryFind("install:ci");
        task?.steps?.length &&
            task.steps.length > 0 &&
            !task?.steps.find((s) => s.exec === cmd) &&
            task?.exec(cmd, { receiveArgs: true });
    }
    /**
     * Add licenses to any subprojects which don't already have a license.
     */
    _addLicenses() {
        [this.project, ...this.project.subprojects]
            .filter((p) => !this.licenseOptions?.disableDefaultLicenses &&
            p.components.find((c) => c instanceof projen_1.License) === undefined)
            .forEach((p) => {
            if (!this.licenseOptions) {
                new projen_1.License(p, {
                    spdx: DEFAULT_LICENSE,
                });
                if (utils_1.ProjectUtils.isNamedInstanceOf(p, java_1.JavaProject)) {
                    // Force all Java projects to use Apache 2.0
                    p.tryFindObjectFile("pom.xml")?.addOverride("project.licenses", [
                        {
                            license: {
                                name: "Apache License 2.0",
                                url: "https://www.apache.org/licenses/LICENSE-2.0",
                                distribution: "repo",
                                comments: "An OSI-approved license",
                            },
                        },
                    ]);
                }
            }
            else if (!!this.licenseOptions?.licenseText) {
                new projen_1.TextFile(p, "LICENSE", {
                    marker: false,
                    committed: true,
                    lines: this.licenseOptions.licenseText.split("\n"),
                });
            }
            else if (this.licenseOptions.spdx) {
                new projen_1.License(p, {
                    spdx: this.licenseOptions.spdx,
                    copyrightOwner: this.licenseOptions?.copyrightOwner,
                });
            }
            else {
                throw new Error("Either spdx or licenseText must be specified.");
            }
        });
    }
    preSynthesize() {
        this._ensureNxProjectGraph();
        this._emitPackageJson();
        this._invokeInstallCITasks();
        this.patchPythonProjects([this.project]);
        this._addLicenses();
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.resetDefaultTask();
    }
    /**
     * Ensures subprojects don't have a default task
     */
    resetDefaultTask() {
        this.project.subprojects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
        });
    }
}
exports.NxConfigurator = NxConfigurator;
_a = JSII_RTTI_SYMBOL_1;
NxConfigurator[_a] = { fqn: "@aws/pdk.monorepo.NxConfigurator", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,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