"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const targets_1 = require("./targets");
const utils_1 = require("../../utils");
const common_1 = require("../../utils/common");
const nx_workspace_1 = require("../nx-workspace");
// List of tasks that are excluded from nx tasks for node projects
const NODE_LIFECYCLE_TASKS = [
    "preinstall",
    "install",
    "postinstall",
    "preinstall:ci",
    "install:ci",
    "postinstall:ci",
];
/**
 * Component which manages the project specific NX Config and is added to all NXMonorepo subprojects.
 * @experimental
 */
class NxProject extends projen_1.Component {
    /**
     * Retrieves an instance of NXProject if one is associated to the given project.
     *
     * @param project project instance.
     */
    static of(project) {
        return project.components.find((c) => utils_1.ProjectUtils.isNamedInstanceOf(c, NxProject));
    }
    /**
     * Retrieves an instance of NXProject if one is associated to the given project,
     * otherwise created a NXProject instance for the project.
     *
     * @param project project instance.
     */
    static ensure(project) {
        return NxProject.of(project) || new NxProject(project);
    }
    constructor(project) {
        // Make sure we only ever have 1 instance of NxProject component per project
        if (NxProject.of(project))
            throw new Error(`Project ${project.name} already has associated NxProject component.`);
        const _existingFile = project.tryFindObjectFile("project.json");
        if (_existingFile &&
            !utils_1.ProjectUtils.isNamedInstanceOf(_existingFile, projen_1.JsonFile)) {
            throw new Error(`Project "${project.name}" contains a "project.json" file that is not a JsonFile instance. NxProject is unable to support this project.`);
        }
        super(project);
        /**
         * Named inputs
         * @see https://nx.dev/reference/nx-json#inputs-&-namedinputs
         */
        this.namedInputs = {};
        /**
         * Targets configuration
         * @see https://nx.dev/reference/project-configuration
         */
        this.targets = {};
        /**
         * Project tag annotations
         *
         * @see https://nx.dev/reference/project-configuration#tags
         */
        this.tags = [];
        /**
         * Implicit dependencies
         *
         * @see https://nx.dev/reference/project-configuration#implicitdependencies
         */
        this.implicitDependencies = [];
        /**
         * Explicit list of scripts for Nx to include.
         * @see https://nx.dev/reference/project-configuration#ignoring-package.json-scripts
         */
        this.includedScripts = [];
        const _obj = {
            name: () => this.project.name,
            root: () => path.relative(this.project.root.outdir, this.project.outdir),
            namedInputs: () => (0, common_1.asUndefinedIfEmpty)(this.namedInputs),
            targets: () => (0, common_1.asUndefinedIfEmpty)(this.targets),
            tags: () => (0, common_1.asUndefinedIfEmpty)(this.tags),
            implicitDependencies: () => (0, common_1.asUndefinedIfEmpty)(this.implicitDependencies),
            includedScripts: () => (0, common_1.asUndefinedIfEmpty)(this.includedScripts),
        };
        this.file =
            _existingFile ||
                new projen_1.JsonFile(project, "project.json", {
                    readonly: true,
                    marker: true,
                    obj: _obj,
                });
        if (_existingFile) {
            project.logger.warn(`[NxProject] Project "${project.name}" defined independent project.json file, which might conflict with NxProject managed properties [${Object.keys(_obj).join(",")}]`);
            Object.entries(_obj).forEach(([key, value]) => {
                _existingFile.addOverride(key, value);
            });
        }
        if (nx_workspace_1.NxWorkspace.of(project)?.autoInferProjectTargets) {
            this.inferTargets();
        }
    }
    /**
     * Automatically infer targets based on project type.
     * @experimental
     */
    inferTargets() {
        const _inferredBuildTarget = (0, targets_1.inferBuildTarget)(this.project);
        if (_inferredBuildTarget) {
            this.targets.build = _inferredBuildTarget;
        }
    }
    /** Merge configuration into existing config */
    merge(config) {
        Object.entries(config).forEach(([key, value]) => {
            switch (key) {
                case "tags": {
                    this.addTag(...value);
                    break;
                }
                case "implicitDependencies": {
                    this.addImplicitDependency(...value);
                    break;
                }
                case "namedInputs": {
                    Object.entries(value).forEach(([_key, _value]) => {
                        this.setNamedInput(_key, _value);
                    });
                    break;
                }
                case "targets": {
                    Object.entries(value).forEach(([_key, _value]) => {
                        this.setTarget(_key, _value, true);
                    });
                    break;
                }
                default: {
                    this.file.addOverride(key, value);
                }
            }
        });
    }
    /** Add tag */
    addTag(...tags) {
        this.tags.push(...tags);
    }
    /**
     * Adds an implicit dependency between the dependant (this project) and dependee.
     *
     * @param dependee project to add the implicit dependency on.
     */
    addImplicitDependency(...dependee) {
        this.implicitDependencies.push(...dependee.map((_d) => (typeof _d === "string" ? _d : _d.name)));
    }
    /**
     * Adds a dependency between two Java Projects in the monorepo.
     * @param dependee project you wish to depend on
     */
    addJavaDependency(dependee) {
        if (!(this.project instanceof java_1.JavaProject)) {
            throw Error("Cannot call addJavaDependency on a project that is not a JavaProject");
        }
        // Add implicit dependency for build order
        this.addImplicitDependency(dependee);
        // Add dependency in pom.xml
        this.project.addDependency(`${dependee.pom.groupId}/${dependee.pom.artifactId}@${dependee.pom.version}`);
        // Add a repository so that the dependency in the pom can be resolved
        this.project.pom.addRepository({
            id: dependee.name,
            url: `file://${path.join(path.relative(this.project.outdir, dependee.outdir), dependee.packaging.distdir)}`,
        });
    }
    /**
     * Adds a dependency between two Python Projects in the monorepo. The dependent must have Poetry enabled.
     * @param dependee project you wish to depend on
     * @throws error if the dependent does not have Poetry enabled
     */
    addPythonPoetryDependency(dependee) {
        // Check we're adding the dependency to a poetry python project
        if (!(this.project instanceof python_1.PythonProject) ||
            !utils_1.ProjectUtils.isNamedInstanceOf(this.project.depsManager, python_1.Poetry)) {
            throw new Error(`${this.project.name} must be a PythonProject with Poetry enabled to add this dependency`);
        }
        // Add implicit dependency for build order
        this.addImplicitDependency(dependee);
        // Add local path dependency
        this.project.addDependency(`${dependee.name}@{path="${path.relative(this.project.outdir, dependee.outdir)}", develop=true}`);
    }
    /** Set `namedInputs` helper */
    setNamedInput(name, inputs) {
        this.namedInputs[name] = inputs;
    }
    /** @internal */
    _getTargetDefaults(name) {
        return nx_workspace_1.NxWorkspace.of(this.project)?.targetDefaults[name] || {};
    }
    /** Set `targets` helper */
    setTarget(name, target, includeDefaults = false) {
        let _default = {};
        if (includeDefaults) {
            if (this.targets[name]) {
                _default = this.targets[name];
            }
            else {
                (_default = this._getTargetDefaults(includeDefaults === true ? name : includeDefaults)),
                    this.targets[name] || {};
            }
        }
        const mergedTarget = (0, common_1.deepMerge)([_default, target], {
            append: true,
        });
        this.targets[name] = {
            ...mergedTarget,
            outputs: mergedTarget.outputs
                ? [...new Set(mergedTarget.outputs)]
                : undefined,
        };
    }
    /**
     * Add input and output files to build target
     * @param inputs Input files
     * @param outputs Output files
     */
    addBuildTargetFiles(inputs, outputs) {
        this.setTarget("build", {
            inputs: inputs || [],
            outputs: outputs || [],
        }, true);
    }
    /** @interface */
    synthesize() {
        const projectPath = path.relative(this.project.root.outdir, this.project.outdir);
        const isNodeProject = utils_1.NodePackageUtils.isNodeProject(this.project);
        const packageManager = utils_1.NodePackageUtils.tryFindNodePackage(this.project, true)?.packageManager ||
            javascript_1.NodePackageManager.NPM;
        this.project.tasks.all
            .filter((task) => {
            if (this.includedScripts.length &&
                !this.includedScripts.includes(task.name)) {
                // Exclude tasks that are not in explicit "includeScripts" when defined
                return false;
            }
            if (task.name in this.targets) {
                // always include tasks that were explicitly added to nx targets
                return true;
            }
            if (NODE_LIFECYCLE_TASKS.includes(task.name) &&
                utils_1.NodePackageUtils.isNodeProject(this.project)) {
                // exclude node lifecycle tasks for node based projects
                return false;
            }
            return true;
        })
            .forEach((task) => {
            // Non-NodeProject don't have package.json so exec bubbles to the root.
            const command = isNodeProject
                ? utils_1.NodePackageUtils.command.projen(packageManager, task.name)
                : utils_1.NodePackageUtils.command.downloadExec(packageManager, "projen", task.name);
            const _target = this.targets[task.name] || {};
            _target.executor = _target.executor || "nx:run-commands";
            _target.options = {
                command,
                cwd: projectPath,
                ..._target.options,
            };
            this.targets[task.name] = _target;
        });
        super.synthesize();
    }
}
exports.NxProject = NxProject;
_a = JSII_RTTI_SYMBOL_1;
NxProject[_a] = { fqn: "@aws/pdk.monorepo.NxProject", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,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