"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonorepoPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const nx_configurator_1 = require("../../components/nx-configurator");
const nx_project_1 = require("../../components/nx-project");
const projen_dependency_1 = require("../../components/projen-dependency");
const utils_1 = require("../../utils");
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid monorepo-py
 */
class MonorepoPythonProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            poetry: true,
            pytest: options.pytest ?? false,
            version: options.version ?? "0.0.0",
            authorName: options.authorName ?? "pdkuser",
            authorEmail: options.authorEmail ?? "user@pdk.com",
            projenrcPythonOptions: {
                ...options.projenrcPythonOptions,
                projenVersion: options.projenrcPythonOptions?.projenVersion ??
                    projen_dependency_1.DEFAULT_PROJEN_VERSION,
            },
        });
        this.projenVersion =
            options.projenrcPythonOptions?.projenVersion ?? projen_dependency_1.DEFAULT_PROJEN_VERSION;
        // Remove dependency on typescript package which projen incorrectly adds to initial .projenrc.py
        // See: https://github.com/projen/projen/issues/2475
        this.deps.removeDependency("@aws/pdk");
        this.addDevDependency("aws_pdk@^0");
        this.nxConfigurator = new nx_configurator_1.NxConfigurator(this, {
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            licenseOptions: options.licenseOptions,
        });
        // Setup python NX plugin
        this.nx.plugins.push("@nxlv/python");
        this.installTask = this.nxConfigurator.ensureNxInstallTask({
            "@nxlv/python": "^16",
        });
        // Map tasks to nx run-many
        this.nxConfigurator._overrideNxBuildTask(this.buildTask, { target: "build" }, { force: true });
        this.nxConfigurator._overrideNxBuildTask(this.preCompileTask, {
            target: "pre-compile",
        });
        this.nxConfigurator._overrideNxBuildTask(this.compileTask, {
            target: "compile",
        });
        this.nxConfigurator._overrideNxBuildTask(this.postCompileTask, {
            target: "post-compile",
        });
        this.nxConfigurator._overrideNxBuildTask(this.testTask, {
            target: "test",
        });
        this.nxConfigurator._overrideNxBuildTask(this.packageTask, {
            target: "package",
        });
    }
    /**
     * @inheritdoc
     */
    get nx() {
        return this.nxConfigurator.nx;
    }
    /**
     * @inheritdoc
     */
    execNxRunManyCommand(options) {
        return this.nxConfigurator.execNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    composeNxRunManyCommand(options) {
        return this.nxConfigurator.composeNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    addNxRunManyTask(name, options) {
        return this.nxConfigurator.addNxRunManyTask(name, options);
    }
    /**
     * @inheritdoc
     */
    addImplicitDependency(dependent, dependee) {
        this.nxConfigurator.addImplicitDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addJavaDependency(dependent, dependee) {
        this.nxConfigurator.addJavaDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addPythonPoetryDependency(dependent, dependee) {
        this.nxConfigurator.addPythonPoetryDependency(dependent, dependee);
    }
    addUpgradeDepsTask() {
        const upgradeDepsTask = new projen_1.Task("upgrade-deps", {
            description: "Upgrade dependencies in the monorepo",
        });
        this.nxConfigurator._overrideNxBuildTask(upgradeDepsTask, {
            target: "upgrade",
        });
        // Spawn the install task for python projects since this will update the lockfile to the latest versions satisfying
        // the pyproject.toml file
        // TODO: remove in favour of the "upgrade" task if ever implemented for python
        upgradeDepsTask.spawn(this.depsManager.installTask);
        this.nxConfigurator._configurePythonSubprojectUpgradeDeps(this, upgradeDepsTask);
    }
    /**
     * @inheritdoc
     */
    preSynthesize() {
        // Calling before super() to ensure proper pre-synth of NxProject component and its nested components
        this.nxConfigurator.preSynthesize();
        this.addUpgradeDepsTask();
        super.preSynthesize();
        (0, projen_dependency_1.syncProjenVersions)(this.subprojects, this.projenVersion);
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.installPythonSubprojects();
        this.nxConfigurator.synth();
        super.synth();
    }
    /**
     * @inheritdoc
     *
     * NOTE: Be sure to ensure the VIRTUAL_ENV is unset during postSynthesize as the individual poetry envs will only be created if a existing VIRTUAL_ENV cannot be found.
     */
    postSynthesize() {
        const vEnv = process.env.VIRTUAL_ENV;
        process.env.VIRTUAL_ENV = "";
        super.postSynthesize();
        process.env.VIRTUAL_ENV = vEnv;
    }
    /**
     * Ensures all python subprojects have their install target called after the monorepo install task.
     */
    installPythonSubprojects() {
        const installProjects = this.subprojects.filter((project) => isPythonProject(project));
        if (installProjects.length > 0) {
            const nxRunManyInstall = this.composeNxRunManyCommand({
                target: "install",
                projects: installProjects.map((project) => project.name),
                parallel: 1,
            });
            this.installTask.exec(utils_1.NodePackageUtils.command.exec(javascript_1.NodePackageManager.NPM, ...nxRunManyInstall));
            // Ensure that install-py follows dependency order
            installProjects.forEach((p) => {
                nx_project_1.NxProject.ensure(p).setTarget("install", {
                    dependsOn: ["^install"],
                });
            });
        }
    }
    /**
     * Ensures subprojects don't have a default task and that all Python subpackages are configured to use Poetry.
     */
    validateSubProjects() {
        this.subprojects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isPythonProject(subProject) && !isPoetryConfigured(subProject)) {
                throw new Error(`${subProject.name} must use poetry to manage dependencies when using NXMonorepoPythonProject`);
            }
        });
    }
}
exports.MonorepoPythonProject = MonorepoPythonProject;
_a = JSII_RTTI_SYMBOL_1;
MonorepoPythonProject[_a] = { fqn: "@aws/pdk.monorepo.MonorepoPythonProject", version: "0.23.39" };
/**
 * Determines if the passed in project is of type PythonProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type PythonProject.
 */
function isPythonProject(project) {
    return utils_1.ProjectUtils.isNamedInstanceOf(project, python_1.PythonProject);
}
/**
 * Determines if the passed in project uses Poetry.
 *
 * @param project PythonProject instance.
 * @returns true if the project uses Poetry.
 */
function isPoetryConfigured(project) {
    return (project.components.find((c) => utils_1.ProjectUtils.isNamedInstanceOf(c, python_1.Poetry)) !== undefined);
}
//# sourceMappingURL=data:application/json;base64,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