"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonorepoTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const nx_configurator_1 = require("../../components/nx-configurator");
const nx_project_1 = require("../../components/nx-project");
const projen_dependency_1 = require("../../components/projen-dependency");
const utils_1 = require("../../utils");
/**
 * This project type will bootstrap a monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid monorepo-ts
 */
class MonorepoTsProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const defaultReleaseBranch = options.defaultReleaseBranch ?? "main";
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            projenCommand: options.packageManager
                ? utils_1.NodePackageUtils.command.projen(options.packageManager)
                : undefined,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            sampleCode: false,
            gitignore: [".tmp", ...(options.gitignore ?? [])],
            defaultReleaseBranch,
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts", ".projenrc.ts"],
            },
            peerDeps: ["nx@^16", ...(options.peerDeps || [])],
            devDeps: ["nx@^16", "@aws/pdk@^0", ...(options.devDeps || [])],
            deps: [
                "aws-cdk-lib",
                "cdk-nag",
                "@aws-cdk/aws-cognito-identitypool-alpha@latest",
                ...(options.deps || []),
            ],
            projenVersion: options.projenVersion ?? projen_dependency_1.DEFAULT_PROJEN_VERSION,
        });
        this.subNodeProjectResolves = [];
        this.projenVersion = options.projenVersion ?? projen_dependency_1.DEFAULT_PROJEN_VERSION;
        this.nxConfigurator = new nx_configurator_1.NxConfigurator(this, {
            defaultReleaseBranch,
            licenseOptions: options.licenseOptions,
        });
        this._options = options;
        // engines
        this.package.addEngine("node", ">=16");
        this.package.setScript("install:ci", utils_1.NodePackageUtils.command.exec(this.package.packageManager, "projen install:ci"));
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.BUN: {
                this.package.addEngine("bun", ">=1");
                break;
            }
            case javascript_1.NodePackageManager.PNPM: {
                // https://pnpm.io/package_json
                // https://github.com/pnpm/pnpm/releases/tag/v8.0.0
                this.package.addEngine("pnpm", ">=8 <9");
                break;
            }
            case javascript_1.NodePackageManager.YARN_CLASSIC:
            case javascript_1.NodePackageManager.YARN: {
                this.package.addEngine("yarn", ">=1 <2");
                break;
            }
            case javascript_1.NodePackageManager.YARN_BERRY:
            case javascript_1.NodePackageManager.YARN2: {
                this.package.addEngine("yarn", ">=2");
                // Yarn Berry cannot call yarn exec without an install first! Use NPX instead.
                this.package.setScript("install:ci", utils_1.NodePackageUtils.command.exec(javascript_1.NodePackageManager.NPM, "projen install:ci"));
                this.package.setScript("install", utils_1.NodePackageUtils.command.exec(javascript_1.NodePackageManager.NPM, "projen install"));
                this.gitignore.addPatterns(".yarn/*", ".pnp.cjs", "!.yarn/releases", "!.yarn/plugins");
                break;
            }
            case javascript_1.NodePackageManager.NPM: {
                // Allow older versions of peer deps to resolv compatibility issues
                this.tasks.tryFind("install")?.reset("npm install --legacy-peer-deps");
                this.tasks.tryFind("install:ci")?.reset("npm ci --legacy-peer-deps");
                break;
            }
        }
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // Add alias task for "projen" to synthesize workspace
        this.package.setScript("synth-workspace", utils_1.NodePackageUtils.command.projen(this.package.packageManager));
        // Map tasks to nx run-many
        if (options.scripts == null || options.scripts.build == null) {
            this.nxConfigurator._overrideNxBuildTask(this.buildTask, { target: "build" }, { force: true });
        }
        if (options.scripts == null || options.scripts["pre-compile"] == null) {
            this.nxConfigurator._overrideNxBuildTask(this.preCompileTask, {
                target: "pre-compile",
            });
        }
        if (options.scripts == null || options.scripts.compile == null) {
            this.nxConfigurator._overrideNxBuildTask(this.compileTask, {
                target: "compile",
            });
        }
        if (options.scripts == null || options.scripts["post-compile"] == null) {
            this.nxConfigurator._overrideNxBuildTask(this.postCompileTask, {
                target: "post-compile",
            });
        }
        if (options.scripts == null || options.scripts.test == null) {
            this.nxConfigurator._overrideNxBuildTask(this.testTask, {
                target: "test",
            });
        }
        if (options.scripts == null || options.scripts.eslint == null) {
            // The Projenrc component of TypeScriptProject resets the eslint task as part of preSynthesize which would undo
            // our changes, so we disable further resets.
            this.nxConfigurator._overrideNxBuildTask(this.eslint?.eslintTask, { target: "eslint" }, { disableReset: true });
        }
        if (options.scripts == null || options.scripts.package == null) {
            this.nxConfigurator._overrideNxBuildTask(this.packageTask, {
                target: "package",
            });
        }
        if (options.scripts == null || options.scripts.prepare == null) {
            this.nxConfigurator._overrideNxBuildTask("prepare", {
                target: "prepare",
            });
        }
        if (options.scripts == null || options.scripts.watch == null) {
            this.nxConfigurator._overrideNxBuildTask(this.watchTask, {
                target: "watch",
                noBail: false,
                ignoreCycles: true,
                skipCache: true,
                outputStyle: "stream",
            });
        }
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2", "wrap-ansi@^7.0.0");
        if (this.package.packageManager !== javascript_1.NodePackageManager.BUN) {
            this.package.addPackageResolutions("@zkochan/js-yaml@npm:js-yaml@4.1.0");
        }
    }
    /**
     * @inheritdoc
     */
    get nx() {
        return this.nxConfigurator.nx;
    }
    /**
     * @inheritdoc
     */
    execNxRunManyCommand(options) {
        return this.nxConfigurator.execNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    composeNxRunManyCommand(options) {
        return this.nxConfigurator.composeNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    addNxRunManyTask(name, options) {
        return this.nxConfigurator.addNxRunManyTask(name, options);
    }
    /**
     * @inheritdoc
     */
    addImplicitDependency(dependent, dependee) {
        this.nxConfigurator.addImplicitDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addJavaDependency(dependent, dependee) {
        this.nxConfigurator.addJavaDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addPythonPoetryDependency(dependent, dependee) {
        this.nxConfigurator.addPythonPoetryDependency(dependent, dependee);
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.sortedSubProjects
            .filter((s) => utils_1.ProjectUtils.isNamedInstanceOf(s, javascript_1.NodeProject))
            .map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    /** Get consistently sorted list of subprojects */
    get sortedSubProjects() {
        return this.subprojects
            .slice()
            .sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * Create symbolic links to all local workspace bins. This enables the usage of bins the same
     * way as consumers of the packages have when installing from the registry.
     */
    linkLocalWorkspaceBins() {
        const bins = [];
        this.subprojects.forEach((subProject) => {
            if (utils_1.ProjectUtils.isNamedInstanceOf(subProject, javascript_1.NodeProject) &&
                subProject.name !== "@aws/pdk") {
                const pkgBins = subProject.package.manifest.bin() || {};
                bins.push(...Object.entries(pkgBins).map(([cmd, bin]) => {
                    const resolvedBin = path.join("$PWD", path.relative(this.outdir, subProject.outdir), bin);
                    return [cmd, resolvedBin];
                }));
            }
        });
        const linkTask = this.addTask("workspace:bin:link", {
            steps: bins.map(([cmd, bin]) => ({
                exec: `ln -s ${bin} ${utils_1.NodePackageUtils.command.bin(this.package.packageManager, cmd)} &>/dev/null; exit 0;`,
            })),
        });
        (this.tasks.tryFind("prepare") || this.addTask("prepare")).spawn(linkTask);
    }
    addUpgradeDepsTask() {
        if (this._options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack@^12");
            const upgradeDepsTask = this.addTask(this._options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps", {
                description: "Upgrade dependencies in the monorepo",
            });
            // Run the upgrade task for any non-node subprojects
            const nonNodeSubprojects = this.subprojects.filter((p) => !utils_1.NodePackageUtils.isNodeProject(p));
            if (nonNodeSubprojects.length > 0) {
                this.nxConfigurator._overrideNxBuildTask(upgradeDepsTask, {
                    target: "upgrade",
                    projects: nonNodeSubprojects.map((p) => p.name),
                });
            }
            this.nxConfigurator._configurePythonSubprojectUpgradeDeps(this, upgradeDepsTask);
            // Upgrade node subprojects together
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "npm-check-updates", "--deep", "--rejectVersion", "0.0.0", "-u", "--dep", "prod,dev,peer,optional,bundle", "--target=minor"));
            // Sync dependency versions across node subprojects
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "syncpack", "fix-mismatches"));
            upgradeDepsTask.exec(`rm ${this.package.lockFile}`);
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.install(this.package.packageManager));
            upgradeDepsTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, "projen"));
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: this._options.monorepoUpgradeDepsOptions?.syncpackConfig ||
                    syncpack_options_1.Syncpack.DEFAULT_CONFIG,
                readonly: true,
            });
        }
    }
    preSynthesize() {
        this.addUpgradeDepsTask();
        utils_1.NodePackageUtils.removeProjenScript(this);
        this.nxConfigurator.preSynthesize();
        super.preSynthesize();
        if (this._options.workspaceConfig?.linkLocalWorkspaceBins === true) {
            this.linkLocalWorkspaceBins();
        }
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                // Remove any subproject .npmrc files since only the root one matters
                subProject.tryRemoveFile(".npmrc");
                utils_1.NodePackageUtils.removeProjenScript(subProject);
            }
        });
        (0, projen_dependency_1.syncProjenVersions)(this.subprojects, this.projenVersion);
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.installNonNodeDependencies();
        this.resolveLocalBunDependencies();
        // Disable node warnings if configured
        if (this._options.disableNodeWarnings) {
            this.disableNodeWarnings();
        }
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                const subNodeProject = subProject;
                const subNodeProjectResolver = 
                // @ts-ignore - `resolveDepsAndWritePackageJson` is private
                subNodeProject.package.resolveDepsAndWritePackageJson;
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => {
                    this.subNodeProjectResolves.push(() => subNodeProjectResolver.apply(subNodeProject.package));
                };
                // @ts-ignore - `resolveDepsAndWritePackageJson` is private
                subNodeProject.package.resolveDepsAndWritePackageJson = () => { };
            }
        });
        this.nxConfigurator.synth();
        super.synth();
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        super.postSynthesize();
        this.resolveSubNodeProjects();
    }
    /**
     * Resolve sub `NodePackage` dependencies.
     */
    resolveSubNodeProjects() {
        if (this.subNodeProjectResolves.length) {
            if (!this.package.file.changed) {
                // Force workspace install deps since it would not have been invoked during `postSynthesis`.
                // @ts-ignore - `installDependencies` is private
                this.package.installDependencies();
            }
            const completedResolves = this.subNodeProjectResolves.map((resolve) => resolve());
            if (completedResolves.some(Boolean)) {
                // Indicates that a subproject dependency has been resolved from '*', so update the lockfile.
                // @ts-ignore - `installDependencies` is private
                this.package.installDependencies();
            }
        }
        this.subNodeProjectResolves = [];
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subprojects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (utils_1.NodePackageUtils.isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.yarn?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.yarn?.disableNoHoistBundled !== true &&
            [
                javascript_1.NodePackageManager.YARN,
                javascript_1.NodePackageManager.YARN2,
                javascript_1.NodePackageManager.YARN_BERRY,
                javascript_1.NodePackageManager.YARN_CLASSIC,
            ].includes(this.package.packageManager)) {
            const noHoistBundled = this.subprojects.flatMap((sub) => {
                if (utils_1.ProjectUtils.isNamedInstanceOf(sub, javascript_1.NodeProject)) {
                    return getBundledDeps(sub).flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        this.package.addField("workspaces", {
            packages: this.workspacePackages,
            nohoist: noHoist,
        });
    }
    /**
     * Ensures the install task for non-node projects is executed postinstall.
     *
     * @private
     */
    installNonNodeDependencies() {
        const installProjects = this.subprojects.filter((project) => !utils_1.ProjectUtils.isNamedInstanceOf(project, javascript_1.NodeProject) &&
            project.tasks.tryFind("install"));
        if (installProjects.length > 0) {
            // TODO: Install error on clean repo for postinstall (https://cloud.nx.app/runs/MptQr0BxgF) (https://github.com/nrwl/nx/issues/11210)
            const postinstallTask = this.tasks.tryFind("postinstall") ?? this.addTask("postinstall");
            const nxRunManyInstall = this.composeNxRunManyCommand({
                target: "install",
                projects: installProjects.map((project) => project.name),
                parallel: 1,
            });
            postinstallTask.exec(utils_1.NodePackageUtils.command.exec(this.package.packageManager, ...nxRunManyInstall));
            // Ensure that install-py follows dependency order
            installProjects.forEach((p) => {
                nx_project_1.NxProject.ensure(p).setTarget("install", {
                    dependsOn: ["^install"],
                });
            });
        }
    }
    /**
     * Suppress Node warnings from being presented in the console when running builds.
     */
    disableNodeWarnings() {
        this.tasks.addEnvironment("NODE_NO_WARNINGS", "1");
        this.subprojects.forEach((subProject) => subProject.tasks.addEnvironment("NODE_NO_WARNINGS", "1"));
    }
    /**
     * Resolve all local workspace dependencies to keep bun happy.
     */
    resolveLocalBunDependencies() {
        if (this.package.packageManager !== javascript_1.NodePackageManager.BUN) {
            return;
        }
        const nodeSubProjectNames = new Set(this.subprojects
            .filter((s) => utils_1.NodePackageUtils.isNodeProject(s))
            .map((s) => s.package.packageName));
        this.subprojects.forEach((subProject) => {
            if (utils_1.NodePackageUtils.isNodeProject(subProject)) {
                subProject.deps.all
                    .filter((dep) => nodeSubProjectNames.has(dep.name))
                    .forEach((d) => {
                    switch (d.type) {
                        case projen_1.DependencyType.RUNTIME:
                            subProject.addDeps(`${d.name}@workspace:*`);
                            break;
                        case projen_1.DependencyType.BUILD:
                            subProject.addDevDeps(`${d.name}@workspace:*`);
                            break;
                        case projen_1.DependencyType.PEER:
                            subProject.addPeerDeps(`${d.name}@workspace:*`);
                            break;
                        default:
                            console.warn(`Cannot update local dependency due to unsupported type: ${d.type}`);
                    }
                });
            }
        });
    }
}
exports.MonorepoTsProject = MonorepoTsProject;
_a = JSII_RTTI_SYMBOL_1;
MonorepoTsProject[_a] = { fqn: "@aws/pdk.monorepo.MonorepoTsProject", version: "0.23.39" };
/**
 * Gets bundled dependencies for a given project
 */
function getBundledDeps(project) {
    return project.deps.all.filter((dep) => dep.type === projen_1.DependencyType.BUNDLED);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibW9ub3JlcG8tdHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJtb25vcmVwby10cy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBO3NDQUNzQztBQUN0Qyw2QkFBNkI7QUFDN0IsbUNBT2dCO0FBRWhCLHNEQUF3RTtBQUV4RSxzREFBMEQ7QUFDMUQseURBQThDO0FBRTlDLHNFQUkwQztBQUMxQyw0REFBd0Q7QUFFeEQsMEVBRzRDO0FBRTVDLHVDQUE2RDtBQThHN0Q7Ozs7O0dBS0c7QUFDSCxNQUFhLGlCQUNYLFNBQVEsOEJBQWlCO0lBa0J6QixZQUFZLE9BQWlDO1FBQzNDLE1BQU0sb0JBQW9CLEdBQUcsT0FBTyxDQUFDLG9CQUFvQixJQUFJLE1BQU0sQ0FBQztRQUNwRSxLQUFLLENBQUM7WUFDSixHQUFHLE9BQU87WUFDVixNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sSUFBSSxLQUFLO1lBQy9CLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTyxJQUFJLEtBQUs7WUFDakMsYUFBYSxFQUFFLE9BQU8sQ0FBQyxjQUFjO2dCQUNuQyxDQUFDLENBQUMsd0JBQWdCLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDO2dCQUN6RCxDQUFDLENBQUMsU0FBUztZQUNiLFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUSxJQUFJLElBQUk7WUFDbEMsVUFBVSxFQUFFLElBQUk7WUFDaEIsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPLElBQUksS0FBSztZQUNqQyxJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUksSUFBSSxLQUFLO1lBQzNCLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFNBQVMsRUFBRSxDQUFDLE1BQU0sRUFBRSxHQUFHLENBQUMsT0FBTyxDQUFDLFNBQVMsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUNqRCxvQkFBb0I7WUFDcEIsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhLElBQUk7Z0JBQ3RDLElBQUksRUFBRSxDQUFDLEdBQUcsQ0FBQztnQkFDWCxjQUFjLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQzthQUNwQztZQUNELFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUSxJQUFJO2dCQUM1QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFLEdBQUc7aUJBQ2I7Z0JBQ0QsT0FBTyxFQUFFLENBQUMsU0FBUyxFQUFFLGNBQWMsQ0FBQzthQUNyQztZQUNELFFBQVEsRUFBRSxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsT0FBTyxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUNqRCxPQUFPLEVBQUUsQ0FBQyxRQUFRLEVBQUUsYUFBYSxFQUFFLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQzlELElBQUksRUFBRTtnQkFDSixhQUFhO2dCQUNiLFNBQVM7Z0JBQ1QsZ0RBQWdEO2dCQUNoRCxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksSUFBSSxFQUFFLENBQUM7YUFDeEI7WUFDRCxhQUFhLEVBQUUsT0FBTyxDQUFDLGFBQWEsSUFBSSwwQ0FBc0I7U0FDL0QsQ0FBQyxDQUFDO1FBckNHLDJCQUFzQixHQUF5QixFQUFFLENBQUM7UUF1Q3hELElBQUksQ0FBQyxhQUFhLEdBQUcsT0FBTyxDQUFDLGFBQWEsSUFBSSwwQ0FBc0IsQ0FBQztRQUVyRSxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksZ0NBQWMsQ0FBQyxJQUFJLEVBQUU7WUFDN0Msb0JBQW9CO1lBQ3BCLGNBQWMsRUFBRSxPQUFPLENBQUMsY0FBYztTQUN2QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsUUFBUSxHQUFHLE9BQU8sQ0FBQztRQUV4QixVQUFVO1FBQ1YsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUNwQixZQUFZLEVBQ1osd0JBQWdCLENBQUMsT0FBTyxDQUFDLElBQUksQ0FDM0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQzNCLG1CQUFtQixDQUNwQixDQUNGLENBQUM7UUFFRixRQUFRLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFO1lBQ25DLEtBQUssK0JBQWtCLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQzNCLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFDckMsTUFBTTthQUNQO1lBQ0QsS0FBSywrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDNUIsK0JBQStCO2dCQUMvQixtREFBbUQ7Z0JBQ25ELElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztnQkFDekMsTUFBTTthQUNQO1lBQ0QsS0FBSywrQkFBa0IsQ0FBQyxZQUFZLENBQUM7WUFDckMsS0FBSywrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDNUIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO2dCQUN6QyxNQUFNO2FBQ1A7WUFDRCxLQUFLLCtCQUFrQixDQUFDLFVBQVUsQ0FBQztZQUNuQyxLQUFLLCtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUM3QixJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7Z0JBQ3RDLDhFQUE4RTtnQkFDOUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQ3BCLFlBQVksRUFDWix3QkFBZ0IsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUMzQiwrQkFBa0IsQ0FBQyxHQUFHLEVBQ3RCLG1CQUFtQixDQUNwQixDQUNGLENBQUM7Z0JBQ0YsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQ3BCLFNBQVMsRUFDVCx3QkFBZ0IsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUMzQiwrQkFBa0IsQ0FBQyxHQUFHLEVBQ3RCLGdCQUFnQixDQUNqQixDQUNGLENBQUM7Z0JBQ0YsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQ3hCLFNBQVMsRUFDVCxVQUFVLEVBQ1YsaUJBQWlCLEVBQ2pCLGdCQUFnQixDQUNqQixDQUFDO2dCQUNGLE1BQU07YUFDUDtZQUNELEtBQUssK0JBQWtCLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQzNCLG1FQUFtRTtnQkFDbkUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEVBQUUsS0FBSyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7Z0JBQ3ZFLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxFQUFFLEtBQUssQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO2dCQUNyRSxNQUFNO2FBQ1A7U0FDRjtRQUVELElBQUksQ0FBQyxlQUFlLEdBQUcsT0FBTyxDQUFDLGVBQWUsQ0FBQztRQUMvQyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsT0FBTyxDQUFDLGVBQWUsRUFBRSxrQkFBa0IsSUFBSSxFQUFFLENBQUM7UUFFM0UseUNBQXlDO1FBQ3pDLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUV2QyxzREFBc0Q7UUFDdEQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQ3BCLGlCQUFpQixFQUNqQix3QkFBZ0IsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLENBQzdELENBQUM7UUFFRiwyQkFBMkI7UUFDM0IsSUFBSSxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLEtBQUssSUFBSSxJQUFJLEVBQUU7WUFDNUQsSUFBSSxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsQ0FDdEMsSUFBSSxDQUFDLFNBQVMsRUFDZCxFQUFFLE1BQU0sRUFBRSxPQUFPLEVBQUUsRUFDbkIsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQ2hCLENBQUM7U0FDSDtRQUNELElBQUksT0FBTyxDQUFDLE9BQU8sSUFBSSxJQUFJLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsSUFBSSxJQUFJLEVBQUU7WUFDckUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFO2dCQUM1RCxNQUFNLEVBQUUsYUFBYTthQUN0QixDQUFDLENBQUM7U0FDSjtRQUNELElBQUksT0FBTyxDQUFDLE9BQU8sSUFBSSxJQUFJLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxFQUFFO1lBQzlELElBQUksQ0FBQyxjQUFjLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRTtnQkFDekQsTUFBTSxFQUFFLFNBQVM7YUFDbEIsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLElBQUksSUFBSSxFQUFFO1lBQ3RFLElBQUksQ0FBQyxjQUFjLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRTtnQkFDN0QsTUFBTSxFQUFFLGNBQWM7YUFDdkIsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxJQUFJLElBQUksRUFBRTtZQUMzRCxJQUFJLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUU7Z0JBQ3RELE1BQU0sRUFBRSxNQUFNO2FBQ2YsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsTUFBTSxJQUFJLElBQUksRUFBRTtZQUM3RCwrR0FBK0c7WUFDL0csNkNBQTZDO1lBQzdDLElBQUksQ0FBQyxjQUFjLENBQUMsb0JBQW9CLENBQ3RDLElBQUksQ0FBQyxNQUFNLEVBQUUsVUFBVSxFQUN2QixFQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFDcEIsRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLENBQ3ZCLENBQUM7U0FDSDtRQUNELElBQUksT0FBTyxDQUFDLE9BQU8sSUFBSSxJQUFJLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxFQUFFO1lBQzlELElBQUksQ0FBQyxjQUFjLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRTtnQkFDekQsTUFBTSxFQUFFLFNBQVM7YUFDbEIsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksRUFBRTtZQUM5RCxJQUFJLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLFNBQVMsRUFBRTtnQkFDbEQsTUFBTSxFQUFFLFNBQVM7YUFDbEIsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxJQUFJLE9BQU8sQ0FBQyxPQUFPLElBQUksSUFBSSxJQUFJLE9BQU8sQ0FBQyxPQUFPLENBQUMsS0FBSyxJQUFJLElBQUksRUFBRTtZQUM1RCxJQUFJLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUU7Z0JBQ3ZELE1BQU0sRUFBRSxPQUFPO2dCQUNmLE1BQU0sRUFBRSxLQUFLO2dCQUNiLFlBQVksRUFBRSxJQUFJO2dCQUNsQixTQUFTLEVBQUUsSUFBSTtnQkFDZixXQUFXLEVBQUUsUUFBUTthQUN0QixDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxPQUFPLENBQUMscUJBQXFCLENBQ2hDLCtCQUErQixFQUMvQixrQkFBa0IsQ0FDbkIsQ0FBQztRQUVGLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssK0JBQWtCLENBQUMsR0FBRyxFQUFFO1lBQzFELElBQUksQ0FBQyxPQUFPLENBQUMscUJBQXFCLENBQUMsb0NBQW9DLENBQUMsQ0FBQztTQUMxRTtJQUNILENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsRUFBRTtRQUNYLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUM7SUFDaEMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksb0JBQW9CLENBQUMsT0FBMEI7UUFDcEQsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7T0FFRztJQUNJLHVCQUF1QixDQUFDLE9BQTBCO1FBQ3ZELE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyx1QkFBdUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUM5RCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxnQkFBZ0IsQ0FBQyxJQUFZLEVBQUUsT0FBMEI7UUFDOUQsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxxQkFBcUIsQ0FDMUIsU0FBa0IsRUFDbEIsUUFBMEI7UUFFMUIsSUFBSSxDQUFDLGNBQWMsQ0FBQyxxQkFBcUIsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDakUsQ0FBQztJQUVEOztPQUVHO0lBQ0ksaUJBQWlCLENBQ3RCLFNBQXNCLEVBQ3RCLFFBQXFCO1FBRXJCLElBQUksQ0FBQyxjQUFjLENBQUMsaUJBQWlCLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQzdELENBQUM7SUFFRDs7T0FFRztJQUNJLHlCQUF5QixDQUM5QixTQUF3QixFQUN4QixRQUF1QjtRQUV2QixJQUFJLENBQUMsY0FBYyxDQUFDLHlCQUF5QixDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNyRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksb0JBQW9CLENBQUMsR0FBRyxZQUFzQjtRQUNuRCxnSEFBZ0g7UUFDaEgsaURBQWlEO1FBQ2pELE1BQU0sbUNBQW1DLEdBQUcsSUFBSSxDQUFDLGlCQUFpQjthQUMvRCxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLG9CQUFZLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxFQUFFLHdCQUFXLENBQUMsQ0FBQzthQUM3RCxHQUFHLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztRQUNoRSxNQUFNLHlCQUF5QixHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQ2xFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQ3pCLEdBQUcsbUNBQW1DLENBQUMsTUFBTSxDQUMzQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyx5QkFBeUIsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQzdDLENBQ0YsQ0FBQztRQUVGLG1DQUFtQztRQUNuQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLEdBQUcsWUFBWSxDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVELGtEQUFrRDtJQUNsRCxJQUFXLGlCQUFpQjtRQUMxQixPQUFPLElBQUksQ0FBQyxXQUFXO2FBQ3BCLEtBQUssRUFBRTthQUNQLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7O09BR0c7SUFDTyxzQkFBc0I7UUFDOUIsTUFBTSxJQUFJLEdBQXVCLEVBQUUsQ0FBQztRQUVwQyxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFVBQVUsRUFBRSxFQUFFO1lBQ3RDLElBQ0Usb0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLEVBQUUsd0JBQVcsQ0FBQztnQkFDdkQsVUFBVSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQzlCO2dCQUNBLE1BQU0sT0FBTyxHQUNYLFVBQVUsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxJQUFJLEVBQUUsQ0FBQztnQkFDMUMsSUFBSSxDQUFDLElBQUksQ0FDUCxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLEVBQUUsRUFBRTtvQkFDNUMsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FDM0IsTUFBTSxFQUNOLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxVQUFVLENBQUMsTUFBTSxDQUFDLEVBQzdDLEdBQUcsQ0FDSixDQUFDO29CQUNGLE9BQU8sQ0FBQyxHQUFHLEVBQUUsV0FBVyxDQUFxQixDQUFDO2dCQUNoRCxDQUFDLENBQUMsQ0FDSCxDQUFDO2FBQ0g7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUVILE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsb0JBQW9CLEVBQUU7WUFDbEQsS0FBSyxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztnQkFDL0IsSUFBSSxFQUFFLFNBQVMsR0FBRyxJQUFJLHdCQUFnQixDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQ2hELElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUMzQixHQUFHLENBQ0osdUJBQXVCO2FBQ3pCLENBQUMsQ0FBQztTQUNKLENBQUMsQ0FBQztRQUVILENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUM3RSxDQUFDO0lBRU8sa0JBQWtCO1FBQ3hCLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsS0FBSyxLQUFLLEVBQUU7WUFDL0MsSUFBSSxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsRUFBRSxjQUFjLENBQUMsQ0FBQztZQUVyRCxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUNsQyxJQUFJLENBQUMsUUFBUSxDQUFDLDBCQUEwQixFQUFFLFFBQVEsSUFBSSxjQUFjLEVBQ3BFO2dCQUNFLFdBQVcsRUFBRSxzQ0FBc0M7YUFDcEQsQ0FDRixDQUFDO1lBQ0Ysb0RBQW9EO1lBQ3BELE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQ2hELENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLHdCQUFnQixDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FDMUMsQ0FBQztZQUNGLElBQUksa0JBQWtCLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtnQkFDakMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsQ0FBQyxlQUFlLEVBQUU7b0JBQ3hELE1BQU0sRUFBRSxTQUFTO29CQUNqQixRQUFRLEVBQUUsa0JBQWtCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO2lCQUNoRCxDQUFDLENBQUM7YUFDSjtZQUVELElBQUksQ0FBQyxjQUFjLENBQUMscUNBQXFDLENBQ3ZELElBQUksRUFDSixlQUFlLENBQ2hCLENBQUM7WUFFRixvQ0FBb0M7WUFDcEMsZUFBZSxDQUFDLElBQUksQ0FDbEIsd0JBQWdCLENBQUMsT0FBTyxDQUFDLElBQUksQ0FDM0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQzNCLG1CQUFtQixFQUNuQixRQUFRLEVBQ1IsaUJBQWlCLEVBQ2pCLE9BQU8sRUFDUCxJQUFJLEVBQ0osT0FBTyxFQUNQLCtCQUErQixFQUMvQixnQkFBZ0IsQ0FDakIsQ0FDRixDQUFDO1lBQ0YsbURBQW1EO1lBQ25ELGVBQWUsQ0FBQyxJQUFJLENBQ2xCLHdCQUFnQixDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQzNCLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUMzQixVQUFVLEVBQ1YsZ0JBQWdCLENBQ2pCLENBQ0YsQ0FBQztZQUNGLGVBQWUsQ0FBQyxJQUFJLENBQUMsTUFBTSxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7WUFDcEQsZUFBZSxDQUFDLElBQUksQ0FDbEIsd0JBQWdCLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxDQUM5RCxDQUFDO1lBQ0YsZUFBZSxDQUFDLElBQUksQ0FDbEIsd0JBQWdCLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxRQUFRLENBQUMsQ0FDckUsQ0FBQztZQUVGLElBQUksaUJBQVEsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQ3JDLEdBQUcsRUFDRCxJQUFJLENBQUMsUUFBUSxDQUFDLDBCQUEwQixFQUFFLGNBQWM7b0JBQ3hELDJCQUFRLENBQUMsY0FBYztnQkFDekIsUUFBUSxFQUFFLElBQUk7YUFDZixDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFRCxhQUFhO1FBQ1gsSUFBSSxDQUFDLGtCQUFrQixFQUFFLENBQUM7UUFFMUIsd0JBQWdCLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDMUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUNwQyxLQUFLLENBQUMsYUFBYSxFQUFFLENBQUM7UUFFdEIsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLGVBQWUsRUFBRSxzQkFBc0IsS0FBSyxJQUFJLEVBQUU7WUFDbEUsSUFBSSxDQUFDLHNCQUFzQixFQUFFLENBQUM7U0FDL0I7UUFFRCxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFVBQVUsRUFBRSxFQUFFO1lBQ3RDLElBQUksd0JBQWdCLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxFQUFFO2dCQUM5QyxxRUFBcUU7Z0JBQ3JFLFVBQVUsQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUM7Z0JBQ25DLHdCQUFnQixDQUFDLGtCQUFrQixDQUFDLFVBQVUsQ0FBQyxDQUFDO2FBQ2pEO1FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFBLHNDQUFrQixFQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7T0FFRztJQUNILEtBQUs7UUFDSCxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztRQUMzQixJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7UUFDdkIsSUFBSSxDQUFDLDBCQUEwQixFQUFFLENBQUM7UUFDbEMsSUFBSSxDQUFDLDJCQUEyQixFQUFFLENBQUM7UUFFbkMsc0NBQXNDO1FBQ3RDLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRTtZQUNyQyxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztTQUM1QjtRQUVELHlHQUF5RztRQUN6Ryw0RkFBNEY7UUFDNUYsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxVQUFVLEVBQUUsRUFBRTtZQUN0QyxJQUFJLHdCQUFnQixDQUFDLGFBQWEsQ0FBQyxVQUFVLENBQUMsRUFBRTtnQkFDOUMsTUFBTSxjQUFjLEdBQWdCLFVBQXlCLENBQUM7Z0JBQzlELE1BQU0sc0JBQXNCO2dCQUMxQiwyREFBMkQ7Z0JBQzNELGNBQWMsQ0FBQyxPQUFPLENBQUMsOEJBQThCLENBQUM7Z0JBQ3hELGdEQUFnRDtnQkFDaEQsY0FBYyxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsR0FBRyxHQUFHLEVBQUU7b0JBQ2hELElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQ3BDLHNCQUFzQixDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQ3JELENBQUM7Z0JBQ0osQ0FBQyxDQUFDO2dCQUNGLDJEQUEyRDtnQkFDM0QsY0FBYyxDQUFDLE9BQU8sQ0FBQyw4QkFBOEIsR0FBRyxHQUFHLEVBQUUsR0FBRSxDQUFDLENBQUM7YUFDbEU7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDNUIsS0FBSyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQ2hCLENBQUM7SUFFRDs7T0FFRztJQUNILGNBQWM7UUFDWixLQUFLLENBQUMsY0FBYyxFQUFFLENBQUM7UUFDdkIsSUFBSSxDQUFDLHNCQUFzQixFQUFFLENBQUM7SUFDaEMsQ0FBQztJQUVEOztPQUVHO0lBQ0ssc0JBQXNCO1FBQzVCLElBQUksSUFBSSxDQUFDLHNCQUFzQixDQUFDLE1BQU0sRUFBRTtZQUN0QyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFO2dCQUM5Qiw0RkFBNEY7Z0JBQzVGLGdEQUFnRDtnQkFDaEQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO2FBQ3BDO1lBQ0QsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLENBQUMsc0JBQXNCLENBQUMsR0FBRyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FDcEUsT0FBTyxFQUFFLENBQ1YsQ0FBQztZQUNGLElBQUksaUJBQWlCLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxFQUFFO2dCQUNuQyw2RkFBNkY7Z0JBQzdGLGdEQUFnRDtnQkFDaEQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO2FBQ3BDO1NBQ0Y7UUFDRCxJQUFJLENBQUMsc0JBQXNCLEdBQUcsRUFBRSxDQUFDO0lBQ25DLENBQUM7SUFFRDs7T0FFRztJQUNLLG1CQUFtQjtRQUN6QixJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFVBQWUsRUFBRSxFQUFFO1lBQzNDLDRFQUE0RTtZQUM1RSxVQUFVLENBQUMsV0FBVyxFQUFFLEtBQUssRUFBRSxDQUFDO1lBRWhDLElBQ0Usd0JBQWdCLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQztnQkFDMUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQ2pFO2dCQUNBLE1BQU0sSUFBSSxLQUFLLENBQ2IsR0FBRyxVQUFVLENBQUMsSUFBSSwrREFBK0QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEdBQUcsQ0FDaEgsQ0FBQzthQUNIO1FBQ0gsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxlQUFlO1FBQ3JCLGlIQUFpSDtRQUNqSCwyQ0FBMkM7UUFDM0MsSUFBSSxDQUFDLG9CQUFvQixFQUFFLENBQUM7UUFFNUIsSUFBSSxPQUFPLEdBQUcsSUFBSSxDQUFDLGVBQWUsRUFBRSxJQUFJLEVBQUUsT0FBTyxDQUFDO1FBQ2xELGdJQUFnSTtRQUNoSSxJQUNFLElBQUksQ0FBQyxlQUFlLEVBQUUsSUFBSSxFQUFFLHFCQUFxQixLQUFLLElBQUk7WUFDMUQ7Z0JBQ0UsK0JBQWtCLENBQUMsSUFBSTtnQkFDdkIsK0JBQWtCLENBQUMsS0FBSztnQkFDeEIsK0JBQWtCLENBQUMsVUFBVTtnQkFDN0IsK0JBQWtCLENBQUMsWUFBWTthQUNoQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxFQUN2QztZQUNBLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUU7Z0JBQ3RELElBQUksb0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLEVBQUUsd0JBQVcsQ0FBQyxFQUFFO29CQUNwRCxPQUFPLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDO3dCQUMxQyxHQUFHLEdBQUcsQ0FBQyxJQUFJLElBQUksR0FBRyxDQUFDLElBQUksRUFBRTt3QkFDekIsR0FBRyxHQUFHLENBQUMsSUFBSSxJQUFJLEdBQUcsQ0FBQyxJQUFJLElBQUk7cUJBQzVCLENBQUMsQ0FBQztpQkFDSjtnQkFDRCxPQUFPLEVBQUUsQ0FBQztZQUNaLENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxjQUFjLENBQUMsTUFBTSxFQUFFO2dCQUN6QixPQUFPLEdBQUcsQ0FBQyxHQUFHLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxFQUFFLEdBQUcsY0FBYyxDQUFDLENBQUM7YUFDbkQ7U0FDRjtRQUVELHFDQUFxQztRQUNyQyxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxLQUFLLCtCQUFrQixDQUFDLElBQUksRUFBRTtZQUMzRCxJQUFJLGlCQUFRLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO2dCQUN4QyxRQUFRLEVBQUUsSUFBSTtnQkFDZCxHQUFHLEVBQUU7b0JBQ0gsUUFBUSxFQUFFLElBQUksQ0FBQyxpQkFBaUI7aUJBQ2pDO2FBQ0YsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7WUFDbEMsUUFBUSxFQUFFLElBQUksQ0FBQyxpQkFBaUI7WUFDaEMsT0FBTyxFQUFFLE9BQU87U0FDakIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSywwQkFBMEI7UUFDaEMsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQzdDLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FDVixDQUFDLG9CQUFZLENBQUMsaUJBQWlCLENBQUMsT0FBTyxFQUFFLHdCQUFXLENBQUM7WUFDckQsT0FBTyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQ25DLENBQUM7UUFFRixJQUFJLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzlCLHFJQUFxSTtZQUNySSxNQUFNLGVBQWUsR0FDbkIsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUVuRSxNQUFNLGdCQUFnQixHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQztnQkFDcEQsTUFBTSxFQUFFLFNBQVM7Z0JBQ2pCLFFBQVEsRUFBRSxlQUFlLENBQUMsR0FBRyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO2dCQUN4RCxRQUFRLEVBQUUsQ0FBQzthQUNaLENBQUMsQ0FBQztZQUVILGVBQWUsQ0FBQyxJQUFJLENBQ2xCLHdCQUFnQixDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQzNCLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUMzQixHQUFHLGdCQUFnQixDQUNwQixDQUNGLENBQUM7WUFFRixrREFBa0Q7WUFDbEQsZUFBZSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFO2dCQUM1QixzQkFBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO29CQUN2QyxTQUFTLEVBQUUsQ0FBQyxVQUFVLENBQUM7aUJBQ3hCLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxtQkFBbUI7UUFDekIsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDbkQsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUN0QyxVQUFVLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLENBQUMsQ0FDekQsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNLLDJCQUEyQjtRQUNqQyxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxLQUFLLCtCQUFrQixDQUFDLEdBQUcsRUFBRTtZQUMxRCxPQUFPO1NBQ1I7UUFFRCxNQUFNLG1CQUFtQixHQUFHLElBQUksR0FBRyxDQUNqQyxJQUFJLENBQUMsV0FBVzthQUNiLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsd0JBQWdCLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ2hELEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUUsQ0FBaUIsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQ3RELENBQUM7UUFFRixJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFVBQVUsRUFBRSxFQUFFO1lBQ3RDLElBQUksd0JBQWdCLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxFQUFFO2dCQUM5QyxVQUFVLENBQUMsSUFBSSxDQUFDLEdBQUc7cUJBQ2hCLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsbUJBQW1CLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztxQkFDbEQsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUU7b0JBQ2IsUUFBUSxDQUFDLENBQUMsSUFBSSxFQUFFO3dCQUNkLEtBQUssdUJBQWMsQ0FBQyxPQUFPOzRCQUN6QixVQUFVLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksY0FBYyxDQUFDLENBQUM7NEJBQzVDLE1BQU07d0JBQ1IsS0FBSyx1QkFBYyxDQUFDLEtBQUs7NEJBQ3ZCLFVBQVUsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxjQUFjLENBQUMsQ0FBQzs0QkFDL0MsTUFBTTt3QkFDUixLQUFLLHVCQUFjLENBQUMsSUFBSTs0QkFDdEIsVUFBVSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLGNBQWMsQ0FBQyxDQUFDOzRCQUNoRCxNQUFNO3dCQUNSOzRCQUNFLE9BQU8sQ0FBQyxJQUFJLENBQ1YsMkRBQTJELENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FDcEUsQ0FBQztxQkFDTDtnQkFDSCxDQUFDLENBQUMsQ0FBQzthQUNOO1FBQ0gsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDOztBQTVuQkgsOENBNm5CQzs7O0FBRUQ7O0dBRUc7QUFDSCxTQUFTLGNBQWMsQ0FBQyxPQUFnQjtJQUN0QyxPQUFPLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksS0FBSyx1QkFBYyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0FBQy9FLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7XG4gIERlcGVuZGVuY3ksXG4gIERlcGVuZGVuY3lUeXBlLFxuICBKc29uRmlsZSxcbiAgUHJvamVjdCxcbiAgVGFzayxcbiAgWWFtbEZpbGUsXG59IGZyb20gXCJwcm9qZW5cIjtcbmltcG9ydCB7IEphdmFQcm9qZWN0IH0gZnJvbSBcInByb2plbi9saWIvamF2YVwiO1xuaW1wb3J0IHsgTm9kZVBhY2thZ2VNYW5hZ2VyLCBOb2RlUHJvamVjdCB9IGZyb20gXCJwcm9qZW4vbGliL2phdmFzY3JpcHRcIjtcbmltcG9ydCB7IFB5dGhvblByb2plY3QgfSBmcm9tIFwicHJvamVuL2xpYi9weXRob25cIjtcbmltcG9ydCB7IFR5cGVTY3JpcHRQcm9qZWN0IH0gZnJvbSBcInByb2plbi9saWIvdHlwZXNjcmlwdFwiO1xuaW1wb3J0IHsgU3luY3BhY2sgfSBmcm9tIFwiLi9zeW5jcGFjay1vcHRpb25zXCI7XG5pbXBvcnQgeyBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnMgfSBmcm9tIFwiLi90eXBlc2NyaXB0LXByb2plY3Qtb3B0aW9uc1wiO1xuaW1wb3J0IHtcbiAgTnhDb25maWd1cmF0b3IsXG4gIElOeFByb2plY3RDb3JlLFxuICBMaWNlbnNlT3B0aW9ucyxcbn0gZnJvbSBcIi4uLy4uL2NvbXBvbmVudHMvbngtY29uZmlndXJhdG9yXCI7XG5pbXBvcnQgeyBOeFByb2plY3QgfSBmcm9tIFwiLi4vLi4vY29tcG9uZW50cy9ueC1wcm9qZWN0XCI7XG5pbXBvcnQgeyBOeFdvcmtzcGFjZSB9IGZyb20gXCIuLi8uLi9jb21wb25lbnRzL254LXdvcmtzcGFjZVwiO1xuaW1wb3J0IHtcbiAgREVGQVVMVF9QUk9KRU5fVkVSU0lPTixcbiAgc3luY1Byb2plblZlcnNpb25zLFxufSBmcm9tIFwiLi4vLi4vY29tcG9uZW50cy9wcm9qZW4tZGVwZW5kZW5jeVwiO1xuaW1wb3J0IHsgTnggfSBmcm9tIFwiLi4vLi4vbngtdHlwZXNcIjtcbmltcG9ydCB7IE5vZGVQYWNrYWdlVXRpbHMsIFByb2plY3RVdGlscyB9IGZyb20gXCIuLi8uLi91dGlsc1wiO1xuXG4vKipcbiAqIFdvcmtzcGFjZSBjb25maWd1cmF0aW9ucy5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vY2xhc3NpYy55YXJucGtnLmNvbS9sYW5nL2VuL2RvY3Mvd29ya3NwYWNlcy9cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBXb3Jrc3BhY2VDb25maWcge1xuICAvKipcbiAgICogWWFybiB3b3Jrc3BhY2UgY29uZmlnXG4gICAqL1xuICByZWFkb25seSB5YXJuPzogWWFybldvcmtzcGFjZUNvbmZpZztcblxuICAvKipcbiAgICogTGlua3MgYWxsIGxvY2FsIHdvcmtzcGFjZSBwcm9qZWN0IGJpbnMgc28gdGhleSBjYW4gYmUgdXNlZCBmb3IgbG9jYWwgZGV2ZWxvcG1lbnQuXG4gICAqXG4gICAqIFBhY2thZ2UgYmlucyBhcmUgb25seSBsaW5rZWQgd2hlbiBpbnN0YWxsZWQgZnJvbSB0aGUgcmVnaXN0cnksIGhvd2V2ZXIgaXQgaXMgdmVyeSB1c2VmdWxcbiAgICogZm9yIG1vbm9yZXBvIGRldmVsb3BtZW50IHRvIGFsc28gdXRpbGl6ZSB0aGVzZSBiaW4gc2NyaXB0cy4gV2hlbiBlbmFibGVkLCB0aGlzIGZsYWcgd2lsbFxuICAgKiByZWN1cnNpdmVseSBsaW5rIGFsbCBiaW5zIGZyb20gcGFja2FnZXMuanNvbiBmaWxlcyB0byB0aGUgcm9vdCBub2RlX21vZHVsZXMvLmJpbi5cbiAgICovXG4gIHJlYWRvbmx5IGxpbmtMb2NhbFdvcmtzcGFjZUJpbnM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGFkZGl0aW9uYWwgcGFja2FnZSBnbG9icyB0byBpbmNsdWRlIGluIHRoZSB3b3Jrc3BhY2UuXG4gICAqXG4gICAqIEFsbCBwYWNrYWdlcyB3aGljaCBhcmUgcGFyZW50ZWQgYnkgdGhlIG1vbm9yZXBvIGFyZSBhdXRvbWF0aWNhbGx5IGFkZGVkIHRvIHRoZSB3b3Jrc3BhY2UsIGJ1dCB5b3UgY2FuIHVzZSB0aGlzXG4gICAqIHByb3BlcnR5IHRvIHNwZWNpZnkgYW55IGFkZGl0aW9uYWwgcGF0aHMgdG8gcGFja2FnZXMgd2hpY2ggbWF5IG5vdCBiZSBtYW5hZ2VkIGJ5IHByb2plbi5cbiAgICovXG4gIHJlYWRvbmx5IGFkZGl0aW9uYWxQYWNrYWdlcz86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIFlhcm4gcmVsYXRlZCB3b3Jrc3BhY2UgY29uZmlnXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgWWFybldvcmtzcGFjZUNvbmZpZyB7XG4gIC8qKlxuICAgKiBMaXN0IG9mIHBhY2thZ2UgZ2xvYnMgdG8gZXhjbHVkZSBmcm9tIGhvaXN0aW5nIGluIHRoZSB3b3Jrc3BhY2UuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9jbGFzc2ljLnlhcm5wa2cuY29tL2Jsb2cvMjAxOC8wMi8xNS9ub2hvaXN0L1xuICAgKi9cbiAgcmVhZG9ubHkgbm9Ib2lzdD86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBEaXNhYmxlIGF1dG9tYXRpY2FsbHkgYXBwbHlpbmcgYG5vSG9pc3RgIGxvZ2ljIGZvciBhbGwgc3ViLXByb2plY3QgXCJidW5kbGVkRGVwZW5kZW5jaWVzXCIuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBkaXNhYmxlTm9Ib2lzdEJ1bmRsZWQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIE1vbm9yZXBvIFVwZ3JhZGUgRGVwcyB0YXNrLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE1vbm9yZXBvVXBncmFkZURlcHNPcHRpb25zIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIHRhc2sgdG8gY3JlYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCB1cGdyYWRlLWRlcHNcbiAgICovXG4gIHJlYWRvbmx5IHRhc2tOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTeW5jcGFjayBjb25maWd1cmF0aW9uLlxuICAgKlxuICAgKiBObyBtZXJnaW5nIGlzIHBlcmZvcm1lZCBhbmQgYXMgc3VjaCBhIGNvbXBsZXRlIHN5bmNwYWNrQ29uZmlnIGlzIHJlcXVpcmVkIGlmIHN1cHBsaWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTeW5jcGFjay5ERUZBVUxUX0NPTkZJR1xuICAgKi9cbiAgcmVhZG9ubHkgc3luY3BhY2tDb25maWc/OiBTeW5jcGFjay5TeW5jcGFja0NvbmZpZztcbn1cblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIHRoZSBNb25vcmVwb1RzUHJvamVjdC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNb25vcmVwb1RzUHJvamVjdE9wdGlvbnMgZXh0ZW5kcyBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnMge1xuICAvKipcbiAgICogQ29uZmlndXJhdGlvbiBmb3Igd29ya3NwYWNlLlxuICAgKi9cbiAgcmVhZG9ubHkgd29ya3NwYWNlQ29uZmlnPzogV29ya3NwYWNlQ29uZmlnO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGluY2x1ZGUgYW4gdXBncmFkZS1kZXBzIHRhc2sgYXQgdGhlIHJvb3Qgb2YgdGhlIG1vbm9yZXBvIHdoaWNoIHdpbGwgdXBncmFkZSBhbGwgZGVwZW5kZW5jaWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBtb25vcmVwb1VwZ3JhZGVEZXBzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogTW9ub3JlcG8gVXBncmFkZSBEZXBzIG9wdGlvbnMuXG4gICAqXG4gICAqIFRoaXMgaXMgb25seSB1c2VkIGlmIG1vbm9yZXBvVXBncmFkZURlcHMgaXMgdHJ1ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgdW5kZWZpbmVkXG4gICAqL1xuICByZWFkb25seSBtb25vcmVwb1VwZ3JhZGVEZXBzT3B0aW9ucz86IE1vbm9yZXBvVXBncmFkZURlcHNPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBEaXNhYmxlIG5vZGUgd2FybmluZ3MgZnJvbSBiZWluZyBlbWl0dGVkIGR1cmluZyBidWlsZCB0YXNrc1xuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGlzYWJsZU5vZGVXYXJuaW5ncz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIERlZmF1bHQgbGljZW5zZSB0byBhcHBseSB0byBhbGwgUERLIG1hbmFnZWQgcGFja2FnZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IEFwYWNoZS0yLjBcbiAgICovXG4gIHJlYWRvbmx5IGxpY2Vuc2VPcHRpb25zPzogTGljZW5zZU9wdGlvbnM7XG59XG5cbi8qKlxuICogVGhpcyBwcm9qZWN0IHR5cGUgd2lsbCBib290c3RyYXAgYSBtb25vcmVwbyB3aXRoIHN1cHBvcnQgZm9yIHBvbHlnb3RcbiAqIGJ1aWxkcywgYnVpbGQgY2FjaGluZywgZGVwZW5kZW5jeSBncmFwaCB2aXN1YWxpemF0aW9uIGFuZCBtdWNoIG1vcmUuXG4gKlxuICogQHBqaWQgbW9ub3JlcG8tdHNcbiAqL1xuZXhwb3J0IGNsYXNzIE1vbm9yZXBvVHNQcm9qZWN0XG4gIGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3RcbiAgaW1wbGVtZW50cyBJTnhQcm9qZWN0Q29yZVxue1xuICAvKipcbiAgICogVmVyc2lvbiBvZiBwcm9qZW4gdXNlZCBieSB0aGUgbW9ub3JlcG8gYW5kIGl0cyBzdWJwcm9qZWN0c1xuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBwcm9qZW5WZXJzaW9uOiBzdHJpbmc7XG5cbiAgLy8gaW1tdXRhYmxlIGRhdGEgc3RydWN0dXJlc1xuICBwcml2YXRlIHJlYWRvbmx5IHdvcmtzcGFjZUNvbmZpZz86IFdvcmtzcGFjZUNvbmZpZztcbiAgcHJpdmF0ZSByZWFkb25seSB3b3Jrc3BhY2VQYWNrYWdlczogc3RyaW5nW107XG5cbiAgcHVibGljIHJlYWRvbmx5IG54Q29uZmlndXJhdG9yOiBOeENvbmZpZ3VyYXRvcjtcblxuICBwcml2YXRlIHJlYWRvbmx5IF9vcHRpb25zOiBNb25vcmVwb1RzUHJvamVjdE9wdGlvbnM7XG5cbiAgcHJpdmF0ZSBzdWJOb2RlUHJvamVjdFJlc29sdmVzOiBBcnJheTwoKSA9PiBib29sZWFuPiA9IFtdO1xuXG4gIGNvbnN0cnVjdG9yKG9wdGlvbnM6IE1vbm9yZXBvVHNQcm9qZWN0T3B0aW9ucykge1xuICAgIGNvbnN0IGRlZmF1bHRSZWxlYXNlQnJhbmNoID0gb3B0aW9ucy5kZWZhdWx0UmVsZWFzZUJyYW5jaCA/PyBcIm1haW5cIjtcbiAgICBzdXBlcih7XG4gICAgICAuLi5vcHRpb25zLFxuICAgICAgZ2l0aHViOiBvcHRpb25zLmdpdGh1YiA/PyBmYWxzZSxcbiAgICAgIHBhY2thZ2U6IG9wdGlvbnMucGFja2FnZSA/PyBmYWxzZSxcbiAgICAgIHByb2plbkNvbW1hbmQ6IG9wdGlvbnMucGFja2FnZU1hbmFnZXJcbiAgICAgICAgPyBOb2RlUGFja2FnZVV0aWxzLmNvbW1hbmQucHJvamVuKG9wdGlvbnMucGFja2FnZU1hbmFnZXIpXG4gICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgcHJldHRpZXI6IG9wdGlvbnMucHJldHRpZXIgPz8gdHJ1ZSxcbiAgICAgIHByb2plbnJjVHM6IHRydWUsXG4gICAgICByZWxlYXNlOiBvcHRpb25zLnJlbGVhc2UgPz8gZmFsc2UsXG4gICAgICBqZXN0OiBvcHRpb25zLmplc3QgPz8gZmFsc2UsXG4gICAgICBzYW1wbGVDb2RlOiBmYWxzZSwgLy8gcm9vdCBzaG91bGQgbmV2ZXIgaGF2ZSBzYW1wbGUgY29kZSxcbiAgICAgIGdpdGlnbm9yZTogW1wiLnRtcFwiLCAuLi4ob3B0aW9ucy5naXRpZ25vcmUgPz8gW10pXSxcbiAgICAgIGRlZmF1bHRSZWxlYXNlQnJhbmNoLFxuICAgICAgZXNsaW50T3B0aW9uczogb3B0aW9ucy5lc2xpbnRPcHRpb25zID8/IHtcbiAgICAgICAgZGlyczogW1wiLlwiXSxcbiAgICAgICAgaWdub3JlUGF0dGVybnM6IFtcInBhY2thZ2VzLyoqLyouKlwiXSxcbiAgICAgIH0sXG4gICAgICB0c2NvbmZpZzogb3B0aW9ucy50c2NvbmZpZyA/PyB7XG4gICAgICAgIGNvbXBpbGVyT3B0aW9uczoge1xuICAgICAgICAgIHJvb3REaXI6IFwiLlwiLFxuICAgICAgICB9LFxuICAgICAgICBpbmNsdWRlOiBbXCIqKi8qLnRzXCIsIFwiLnByb2plbnJjLnRzXCJdLFxuICAgICAgfSxcbiAgICAgIHBlZXJEZXBzOiBbXCJueEBeMTZcIiwgLi4uKG9wdGlvbnMucGVlckRlcHMgfHwgW10pXSxcbiAgICAgIGRldkRlcHM6IFtcIm54QF4xNlwiLCBcIkBhd3MvcGRrQF4wXCIsIC4uLihvcHRpb25zLmRldkRlcHMgfHwgW10pXSxcbiAgICAgIGRlcHM6IFtcbiAgICAgICAgXCJhd3MtY2RrLWxpYlwiLFxuICAgICAgICBcImNkay1uYWdcIixcbiAgICAgICAgXCJAYXdzLWNkay9hd3MtY29nbml0by1pZGVudGl0eXBvb2wtYWxwaGFAbGF0ZXN0XCIsXG4gICAgICAgIC4uLihvcHRpb25zLmRlcHMgfHwgW10pLFxuICAgICAgXSxcbiAgICAgIHByb2plblZlcnNpb246IG9wdGlvbnMucHJvamVuVmVyc2lvbiA/PyBERUZBVUxUX1BST0pFTl9WRVJTSU9OLFxuICAgIH0pO1xuXG4gICAgdGhpcy5wcm9qZW5WZXJzaW9uID0gb3B0aW9ucy5wcm9qZW5WZXJzaW9uID8/IERFRkFVTFRfUFJPSkVOX1ZFUlNJT047XG5cbiAgICB0aGlzLm54Q29uZmlndXJhdG9yID0gbmV3IE54Q29uZmlndXJhdG9yKHRoaXMsIHtcbiAgICAgIGRlZmF1bHRSZWxlYXNlQnJhbmNoLFxuICAgICAgbGljZW5zZU9wdGlvbnM6IG9wdGlvbnMubGljZW5zZU9wdGlvbnMsXG4gICAgfSk7XG4gICAgdGhpcy5fb3B0aW9ucyA9IG9wdGlvbnM7XG5cbiAgICAvLyBlbmdpbmVzXG4gICAgdGhpcy5wYWNrYWdlLmFkZEVuZ2luZShcIm5vZGVcIiwgXCI+PTE2XCIpO1xuICAgIHRoaXMucGFja2FnZS5zZXRTY3JpcHQoXG4gICAgICBcImluc3RhbGw6Y2lcIixcbiAgICAgIE5vZGVQYWNrYWdlVXRpbHMuY29tbWFuZC5leGVjKFxuICAgICAgICB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIsXG4gICAgICAgIFwicHJvamVuIGluc3RhbGw6Y2lcIlxuICAgICAgKVxuICAgICk7XG5cbiAgICBzd2l0Y2ggKHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcikge1xuICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuQlVOOiB7XG4gICAgICAgIHRoaXMucGFja2FnZS5hZGRFbmdpbmUoXCJidW5cIiwgXCI+PTFcIik7XG4gICAgICAgIGJyZWFrO1xuICAgICAgfVxuICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTToge1xuICAgICAgICAvLyBodHRwczovL3BucG0uaW8vcGFja2FnZV9qc29uXG4gICAgICAgIC8vIGh0dHBzOi8vZ2l0aHViLmNvbS9wbnBtL3BucG0vcmVsZWFzZXMvdGFnL3Y4LjAuMFxuICAgICAgICB0aGlzLnBhY2thZ2UuYWRkRW5naW5lKFwicG5wbVwiLCBcIj49OCA8OVwiKTtcbiAgICAgICAgYnJlYWs7XG4gICAgICB9XG4gICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOX0NMQVNTSUM6XG4gICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOOiB7XG4gICAgICAgIHRoaXMucGFja2FnZS5hZGRFbmdpbmUoXCJ5YXJuXCIsIFwiPj0xIDwyXCIpO1xuICAgICAgICBicmVhaztcbiAgICAgIH1cbiAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLllBUk5fQkVSUlk6XG4gICAgICBjYXNlIE5vZGVQYWNrYWdlTWFuYWdlci5ZQVJOMjoge1xuICAgICAgICB0aGlzLnBhY2thZ2UuYWRkRW5naW5lKFwieWFyblwiLCBcIj49MlwiKTtcbiAgICAgICAgLy8gWWFybiBCZXJyeSBjYW5ub3QgY2FsbCB5YXJuIGV4ZWMgd2l0aG91dCBhbiBpbnN0YWxsIGZpcnN0ISBVc2UgTlBYIGluc3RlYWQuXG4gICAgICAgIHRoaXMucGFja2FnZS5zZXRTY3JpcHQoXG4gICAgICAgICAgXCJpbnN0YWxsOmNpXCIsXG4gICAgICAgICAgTm9kZVBhY2thZ2VVdGlscy5jb21tYW5kLmV4ZWMoXG4gICAgICAgICAgICBOb2RlUGFja2FnZU1hbmFnZXIuTlBNLFxuICAgICAgICAgICAgXCJwcm9qZW4gaW5zdGFsbDpjaVwiXG4gICAgICAgICAgKVxuICAgICAgICApO1xuICAgICAgICB0aGlzLnBhY2thZ2Uuc2V0U2NyaXB0KFxuICAgICAgICAgIFwiaW5zdGFsbFwiLFxuICAgICAgICAgIE5vZGVQYWNrYWdlVXRpbHMuY29tbWFuZC5leGVjKFxuICAgICAgICAgICAgTm9kZVBhY2thZ2VNYW5hZ2VyLk5QTSxcbiAgICAgICAgICAgIFwicHJvamVuIGluc3RhbGxcIlxuICAgICAgICAgIClcbiAgICAgICAgKTtcbiAgICAgICAgdGhpcy5naXRpZ25vcmUuYWRkUGF0dGVybnMoXG4gICAgICAgICAgXCIueWFybi8qXCIsXG4gICAgICAgICAgXCIucG5wLmNqc1wiLFxuICAgICAgICAgIFwiIS55YXJuL3JlbGVhc2VzXCIsXG4gICAgICAgICAgXCIhLnlhcm4vcGx1Z2luc1wiXG4gICAgICAgICk7XG4gICAgICAgIGJyZWFrO1xuICAgICAgfVxuICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuTlBNOiB7XG4gICAgICAgIC8vIEFsbG93IG9sZGVyIHZlcnNpb25zIG9mIHBlZXIgZGVwcyB0byByZXNvbHYgY29tcGF0aWJpbGl0eSBpc3N1ZXNcbiAgICAgICAgdGhpcy50YXNrcy50cnlGaW5kKFwiaW5zdGFsbFwiKT8ucmVzZXQoXCJucG0gaW5zdGFsbCAtLWxlZ2FjeS1wZWVyLWRlcHNcIik7XG4gICAgICAgIHRoaXMudGFza3MudHJ5RmluZChcImluc3RhbGw6Y2lcIik/LnJlc2V0KFwibnBtIGNpIC0tbGVnYWN5LXBlZXItZGVwc1wiKTtcbiAgICAgICAgYnJlYWs7XG4gICAgICB9XG4gICAgfVxuXG4gICAgdGhpcy53b3Jrc3BhY2VDb25maWcgPSBvcHRpb25zLndvcmtzcGFjZUNvbmZpZztcbiAgICB0aGlzLndvcmtzcGFjZVBhY2thZ2VzID0gb3B0aW9ucy53b3Jrc3BhY2VDb25maWc/LmFkZGl0aW9uYWxQYWNrYWdlcyA/PyBbXTtcblxuICAgIC8vIE5ldmVyIHB1Ymxpc2ggYSBtb25vcmVwbyByb290IHBhY2thZ2UuXG4gICAgdGhpcy5wYWNrYWdlLmFkZEZpZWxkKFwicHJpdmF0ZVwiLCB0cnVlKTtcblxuICAgIC8vIEFkZCBhbGlhcyB0YXNrIGZvciBcInByb2plblwiIHRvIHN5bnRoZXNpemUgd29ya3NwYWNlXG4gICAgdGhpcy5wYWNrYWdlLnNldFNjcmlwdChcbiAgICAgIFwic3ludGgtd29ya3NwYWNlXCIsXG4gICAgICBOb2RlUGFja2FnZVV0aWxzLmNvbW1hbmQucHJvamVuKHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcilcbiAgICApO1xuXG4gICAgLy8gTWFwIHRhc2tzIHRvIG54IHJ1bi1tYW55XG4gICAgaWYgKG9wdGlvbnMuc2NyaXB0cyA9PSBudWxsIHx8IG9wdGlvbnMuc2NyaXB0cy5idWlsZCA9PSBudWxsKSB7XG4gICAgICB0aGlzLm54Q29uZmlndXJhdG9yLl9vdmVycmlkZU54QnVpbGRUYXNrKFxuICAgICAgICB0aGlzLmJ1aWxkVGFzayxcbiAgICAgICAgeyB0YXJnZXQ6IFwiYnVpbGRcIiB9LFxuICAgICAgICB7IGZvcmNlOiB0cnVlIH1cbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLnNjcmlwdHMgPT0gbnVsbCB8fCBvcHRpb25zLnNjcmlwdHNbXCJwcmUtY29tcGlsZVwiXSA9PSBudWxsKSB7XG4gICAgICB0aGlzLm54Q29uZmlndXJhdG9yLl9vdmVycmlkZU54QnVpbGRUYXNrKHRoaXMucHJlQ29tcGlsZVRhc2ssIHtcbiAgICAgICAgdGFyZ2V0OiBcInByZS1jb21waWxlXCIsXG4gICAgICB9KTtcbiAgICB9XG4gICAgaWYgKG9wdGlvbnMuc2NyaXB0cyA9PSBudWxsIHx8IG9wdGlvbnMuc2NyaXB0cy5jb21waWxlID09IG51bGwpIHtcbiAgICAgIHRoaXMubnhDb25maWd1cmF0b3IuX292ZXJyaWRlTnhCdWlsZFRhc2sodGhpcy5jb21waWxlVGFzaywge1xuICAgICAgICB0YXJnZXQ6IFwiY29tcGlsZVwiLFxuICAgICAgfSk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLnNjcmlwdHMgPT0gbnVsbCB8fCBvcHRpb25zLnNjcmlwdHNbXCJwb3N0LWNvbXBpbGVcIl0gPT0gbnVsbCkge1xuICAgICAgdGhpcy5ueENvbmZpZ3VyYXRvci5fb3ZlcnJpZGVOeEJ1aWxkVGFzayh0aGlzLnBvc3RDb21waWxlVGFzaywge1xuICAgICAgICB0YXJnZXQ6IFwicG9zdC1jb21waWxlXCIsXG4gICAgICB9KTtcbiAgICB9XG4gICAgaWYgKG9wdGlvbnMuc2NyaXB0cyA9PSBudWxsIHx8IG9wdGlvbnMuc2NyaXB0cy50ZXN0ID09IG51bGwpIHtcbiAgICAgIHRoaXMubnhDb25maWd1cmF0b3IuX292ZXJyaWRlTnhCdWlsZFRhc2sodGhpcy50ZXN0VGFzaywge1xuICAgICAgICB0YXJnZXQ6IFwidGVzdFwiLFxuICAgICAgfSk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLnNjcmlwdHMgPT0gbnVsbCB8fCBvcHRpb25zLnNjcmlwdHMuZXNsaW50ID09IG51bGwpIHtcbiAgICAgIC8vIFRoZSBQcm9qZW5yYyBjb21wb25lbnQgb2YgVHlwZVNjcmlwdFByb2plY3QgcmVzZXRzIHRoZSBlc2xpbnQgdGFzayBhcyBwYXJ0IG9mIHByZVN5bnRoZXNpemUgd2hpY2ggd291bGQgdW5kb1xuICAgICAgLy8gb3VyIGNoYW5nZXMsIHNvIHdlIGRpc2FibGUgZnVydGhlciByZXNldHMuXG4gICAgICB0aGlzLm54Q29uZmlndXJhdG9yLl9vdmVycmlkZU54QnVpbGRUYXNrKFxuICAgICAgICB0aGlzLmVzbGludD8uZXNsaW50VGFzayxcbiAgICAgICAgeyB0YXJnZXQ6IFwiZXNsaW50XCIgfSxcbiAgICAgICAgeyBkaXNhYmxlUmVzZXQ6IHRydWUgfVxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKG9wdGlvbnMuc2NyaXB0cyA9PSBudWxsIHx8IG9wdGlvbnMuc2NyaXB0cy5wYWNrYWdlID09IG51bGwpIHtcbiAgICAgIHRoaXMubnhDb25maWd1cmF0b3IuX292ZXJyaWRlTnhCdWlsZFRhc2sodGhpcy5wYWNrYWdlVGFzaywge1xuICAgICAgICB0YXJnZXQ6IFwicGFja2FnZVwiLFxuICAgICAgfSk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLnNjcmlwdHMgPT0gbnVsbCB8fCBvcHRpb25zLnNjcmlwdHMucHJlcGFyZSA9PSBudWxsKSB7XG4gICAgICB0aGlzLm54Q29uZmlndXJhdG9yLl9vdmVycmlkZU54QnVpbGRUYXNrKFwicHJlcGFyZVwiLCB7XG4gICAgICAgIHRhcmdldDogXCJwcmVwYXJlXCIsXG4gICAgICB9KTtcbiAgICB9XG4gICAgaWYgKG9wdGlvbnMuc2NyaXB0cyA9PSBudWxsIHx8IG9wdGlvbnMuc2NyaXB0cy53YXRjaCA9PSBudWxsKSB7XG4gICAgICB0aGlzLm54Q29uZmlndXJhdG9yLl9vdmVycmlkZU54QnVpbGRUYXNrKHRoaXMud2F0Y2hUYXNrLCB7XG4gICAgICAgIHRhcmdldDogXCJ3YXRjaFwiLFxuICAgICAgICBub0JhaWw6IGZhbHNlLFxuICAgICAgICBpZ25vcmVDeWNsZXM6IHRydWUsXG4gICAgICAgIHNraXBDYWNoZTogdHJ1ZSxcbiAgICAgICAgb3V0cHV0U3R5bGU6IFwic3RyZWFtXCIsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICB0aGlzLnBhY2thZ2UuYWRkUGFja2FnZVJlc29sdXRpb25zKFxuICAgICAgXCJAdHlwZXMvYmFiZWxfX3RyYXZlcnNlQDcuMTguMlwiLFxuICAgICAgXCJ3cmFwLWFuc2lAXjcuMC4wXCJcbiAgICApO1xuXG4gICAgaWYgKHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlciAhPT0gTm9kZVBhY2thZ2VNYW5hZ2VyLkJVTikge1xuICAgICAgdGhpcy5wYWNrYWdlLmFkZFBhY2thZ2VSZXNvbHV0aW9ucyhcIkB6a29jaGFuL2pzLXlhbWxAbnBtOmpzLXlhbWxANC4xLjBcIik7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEBpbmhlcml0ZG9jXG4gICAqL1xuICBwdWJsaWMgZ2V0IG54KCk6IE54V29ya3NwYWNlIHtcbiAgICByZXR1cm4gdGhpcy5ueENvbmZpZ3VyYXRvci5ueDtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW5oZXJpdGRvY1xuICAgKi9cbiAgcHVibGljIGV4ZWNOeFJ1bk1hbnlDb21tYW5kKG9wdGlvbnM6IE54LlJ1bk1hbnlPcHRpb25zKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5ueENvbmZpZ3VyYXRvci5leGVjTnhSdW5NYW55Q29tbWFuZChvcHRpb25zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW5oZXJpdGRvY1xuICAgKi9cbiAgcHVibGljIGNvbXBvc2VOeFJ1bk1hbnlDb21tYW5kKG9wdGlvbnM6IE54LlJ1bk1hbnlPcHRpb25zKTogc3RyaW5nW10ge1xuICAgIHJldHVybiB0aGlzLm54Q29uZmlndXJhdG9yLmNvbXBvc2VOeFJ1bk1hbnlDb21tYW5kKG9wdGlvbnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbmhlcml0ZG9jXG4gICAqL1xuICBwdWJsaWMgYWRkTnhSdW5NYW55VGFzayhuYW1lOiBzdHJpbmcsIG9wdGlvbnM6IE54LlJ1bk1hbnlPcHRpb25zKTogVGFzayB7XG4gICAgcmV0dXJuIHRoaXMubnhDb25maWd1cmF0b3IuYWRkTnhSdW5NYW55VGFzayhuYW1lLCBvcHRpb25zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW5oZXJpdGRvY1xuICAgKi9cbiAgcHVibGljIGFkZEltcGxpY2l0RGVwZW5kZW5jeShcbiAgICBkZXBlbmRlbnQ6IFByb2plY3QsXG4gICAgZGVwZW5kZWU6IHN0cmluZyB8IFByb2plY3RcbiAgKTogdm9pZCB7XG4gICAgdGhpcy5ueENvbmZpZ3VyYXRvci5hZGRJbXBsaWNpdERlcGVuZGVuY3koZGVwZW5kZW50LCBkZXBlbmRlZSk7XG4gIH1cblxuICAvKipcbiAgICogQGluaGVyaXRkb2NcbiAgICovXG4gIHB1YmxpYyBhZGRKYXZhRGVwZW5kZW5jeShcbiAgICBkZXBlbmRlbnQ6IEphdmFQcm9qZWN0LFxuICAgIGRlcGVuZGVlOiBKYXZhUHJvamVjdFxuICApOiB2b2lkIHtcbiAgICB0aGlzLm54Q29uZmlndXJhdG9yLmFkZEphdmFEZXBlbmRlbmN5KGRlcGVuZGVudCwgZGVwZW5kZWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbmhlcml0ZG9jXG4gICAqL1xuICBwdWJsaWMgYWRkUHl0aG9uUG9ldHJ5RGVwZW5kZW5jeShcbiAgICBkZXBlbmRlbnQ6IFB5dGhvblByb2plY3QsXG4gICAgZGVwZW5kZWU6IFB5dGhvblByb2plY3RcbiAgKTogdm9pZCB7XG4gICAgdGhpcy5ueENvbmZpZ3VyYXRvci5hZGRQeXRob25Qb2V0cnlEZXBlbmRlbmN5KGRlcGVuZGVudCwgZGVwZW5kZWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBvbmUgb3IgbW9yZSBhZGRpdGlvbmFsIHBhY2thZ2UgZ2xvYnMgdG8gdGhlIHdvcmtzcGFjZS5cbiAgICogQHBhcmFtIHBhY2thZ2VHbG9icyBwYXRocyB0byB0aGUgcGFja2FnZSB0byBpbmNsdWRlIGluIHRoZSB3b3Jrc3BhY2UgKGZvciBleGFtcGxlIHBhY2thZ2VzL215LXBhY2thZ2UpXG4gICAqL1xuICBwdWJsaWMgYWRkV29ya3NwYWNlUGFja2FnZXMoLi4ucGFja2FnZUdsb2JzOiBzdHJpbmdbXSkge1xuICAgIC8vIEFueSBzdWJwcm9qZWN0cyB0aGF0IHdlcmUgYWRkZWQgc2luY2UgdGhlIGxhc3QgY2FsbCB0byB0aGlzIG1ldGhvZCBuZWVkIHRvIGJlIGFkZGVkIGZpcnN0LCBpbiBvcmRlciB0byBlbnN1cmVcbiAgICAvLyB3ZSBhZGQgdGhlIHdvcmtzcGFjZSBwYWNrYWdlcyBpbiBhIHNhbmUgb3JkZXIuXG4gICAgY29uc3QgcmVsYXRpdmVTdWJQcm9qZWN0V29ya3NwYWNlUGFja2FnZXMgPSB0aGlzLnNvcnRlZFN1YlByb2plY3RzXG4gICAgICAuZmlsdGVyKChzKSA9PiBQcm9qZWN0VXRpbHMuaXNOYW1lZEluc3RhbmNlT2YocywgTm9kZVByb2plY3QpKVxuICAgICAgLm1hcCgocHJvamVjdCkgPT4gcGF0aC5yZWxhdGl2ZSh0aGlzLm91dGRpciwgcHJvamVjdC5vdXRkaXIpKTtcbiAgICBjb25zdCBleGlzdGluZ1dvcmtzcGFjZVBhY2thZ2VzID0gbmV3IFNldCh0aGlzLndvcmtzcGFjZVBhY2thZ2VzKTtcbiAgICB0aGlzLndvcmtzcGFjZVBhY2thZ2VzLnB1c2goXG4gICAgICAuLi5yZWxhdGl2ZVN1YlByb2plY3RXb3Jrc3BhY2VQYWNrYWdlcy5maWx0ZXIoXG4gICAgICAgIChwa2cpID0+ICFleGlzdGluZ1dvcmtzcGFjZVBhY2thZ2VzLmhhcyhwa2cpXG4gICAgICApXG4gICAgKTtcblxuICAgIC8vIEFkZCB0aGUgYWRkaXRpb25hbCBwYWNrYWdlcyBuZXh0XG4gICAgdGhpcy53b3Jrc3BhY2VQYWNrYWdlcy5wdXNoKC4uLnBhY2thZ2VHbG9icyk7XG4gIH1cblxuICAvKiogR2V0IGNvbnNpc3RlbnRseSBzb3J0ZWQgbGlzdCBvZiBzdWJwcm9qZWN0cyAqL1xuICBwdWJsaWMgZ2V0IHNvcnRlZFN1YlByb2plY3RzKCk6IFByb2plY3RbXSB7XG4gICAgcmV0dXJuIHRoaXMuc3VicHJvamVjdHNcbiAgICAgIC5zbGljZSgpXG4gICAgICAuc29ydCgoYSwgYikgPT4gYS5uYW1lLmxvY2FsZUNvbXBhcmUoYi5uYW1lKSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIHN5bWJvbGljIGxpbmtzIHRvIGFsbCBsb2NhbCB3b3Jrc3BhY2UgYmlucy4gVGhpcyBlbmFibGVzIHRoZSB1c2FnZSBvZiBiaW5zIHRoZSBzYW1lXG4gICAqIHdheSBhcyBjb25zdW1lcnMgb2YgdGhlIHBhY2thZ2VzIGhhdmUgd2hlbiBpbnN0YWxsaW5nIGZyb20gdGhlIHJlZ2lzdHJ5LlxuICAgKi9cbiAgcHJvdGVjdGVkIGxpbmtMb2NhbFdvcmtzcGFjZUJpbnMoKTogdm9pZCB7XG4gICAgY29uc3QgYmluczogW3N0cmluZywgc3RyaW5nXVtdID0gW107XG5cbiAgICB0aGlzLnN1YnByb2plY3RzLmZvckVhY2goKHN1YlByb2plY3QpID0+IHtcbiAgICAgIGlmIChcbiAgICAgICAgUHJvamVjdFV0aWxzLmlzTmFtZWRJbnN0YW5jZU9mKHN1YlByb2plY3QsIE5vZGVQcm9qZWN0KSAmJlxuICAgICAgICBzdWJQcm9qZWN0Lm5hbWUgIT09IFwiQGF3cy9wZGtcIlxuICAgICAgKSB7XG4gICAgICAgIGNvbnN0IHBrZ0JpbnM6IFJlY29yZDxzdHJpbmcsIHN0cmluZz4gPVxuICAgICAgICAgIHN1YlByb2plY3QucGFja2FnZS5tYW5pZmVzdC5iaW4oKSB8fCB7fTtcbiAgICAgICAgYmlucy5wdXNoKFxuICAgICAgICAgIC4uLk9iamVjdC5lbnRyaWVzKHBrZ0JpbnMpLm1hcCgoW2NtZCwgYmluXSkgPT4ge1xuICAgICAgICAgICAgY29uc3QgcmVzb2x2ZWRCaW4gPSBwYXRoLmpvaW4oXG4gICAgICAgICAgICAgIFwiJFBXRFwiLFxuICAgICAgICAgICAgICBwYXRoLnJlbGF0aXZlKHRoaXMub3V0ZGlyLCBzdWJQcm9qZWN0Lm91dGRpciksXG4gICAgICAgICAgICAgIGJpblxuICAgICAgICAgICAgKTtcbiAgICAgICAgICAgIHJldHVybiBbY21kLCByZXNvbHZlZEJpbl0gYXMgW3N0cmluZywgc3RyaW5nXTtcbiAgICAgICAgICB9KVxuICAgICAgICApO1xuICAgICAgfVxuICAgIH0pO1xuXG4gICAgY29uc3QgbGlua1Rhc2sgPSB0aGlzLmFkZFRhc2soXCJ3b3Jrc3BhY2U6YmluOmxpbmtcIiwge1xuICAgICAgc3RlcHM6IGJpbnMubWFwKChbY21kLCBiaW5dKSA9PiAoe1xuICAgICAgICBleGVjOiBgbG4gLXMgJHtiaW59ICR7Tm9kZVBhY2thZ2VVdGlscy5jb21tYW5kLmJpbihcbiAgICAgICAgICB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIsXG4gICAgICAgICAgY21kXG4gICAgICAgICl9ICY+L2Rldi9udWxsOyBleGl0IDA7YCxcbiAgICAgIH0pKSxcbiAgICB9KTtcblxuICAgICh0aGlzLnRhc2tzLnRyeUZpbmQoXCJwcmVwYXJlXCIpIHx8IHRoaXMuYWRkVGFzayhcInByZXBhcmVcIikpLnNwYXduKGxpbmtUYXNrKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkVXBncmFkZURlcHNUYXNrKCkge1xuICAgIGlmICh0aGlzLl9vcHRpb25zLm1vbm9yZXBvVXBncmFkZURlcHMgIT09IGZhbHNlKSB7XG4gICAgICB0aGlzLmFkZERldkRlcHMoXCJucG0tY2hlY2stdXBkYXRlc1wiLCBcInN5bmNwYWNrQF4xMlwiKTtcblxuICAgICAgY29uc3QgdXBncmFkZURlcHNUYXNrID0gdGhpcy5hZGRUYXNrKFxuICAgICAgICB0aGlzLl9vcHRpb25zLm1vbm9yZXBvVXBncmFkZURlcHNPcHRpb25zPy50YXNrTmFtZSB8fCBcInVwZ3JhZGUtZGVwc1wiLFxuICAgICAgICB7XG4gICAgICAgICAgZGVzY3JpcHRpb246IFwiVXBncmFkZSBkZXBlbmRlbmNpZXMgaW4gdGhlIG1vbm9yZXBvXCIsXG4gICAgICAgIH1cbiAgICAgICk7XG4gICAgICAvLyBSdW4gdGhlIHVwZ3JhZGUgdGFzayBmb3IgYW55IG5vbi1ub2RlIHN1YnByb2plY3RzXG4gICAgICBjb25zdCBub25Ob2RlU3VicHJvamVjdHMgPSB0aGlzLnN1YnByb2plY3RzLmZpbHRlcihcbiAgICAgICAgKHApID0+ICFOb2RlUGFja2FnZVV0aWxzLmlzTm9kZVByb2plY3QocClcbiAgICAgICk7XG4gICAgICBpZiAobm9uTm9kZVN1YnByb2plY3RzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgdGhpcy5ueENvbmZpZ3VyYXRvci5fb3ZlcnJpZGVOeEJ1aWxkVGFzayh1cGdyYWRlRGVwc1Rhc2ssIHtcbiAgICAgICAgICB0YXJnZXQ6IFwidXBncmFkZVwiLFxuICAgICAgICAgIHByb2plY3RzOiBub25Ob2RlU3VicHJvamVjdHMubWFwKChwKSA9PiBwLm5hbWUpLFxuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgICAgdGhpcy5ueENvbmZpZ3VyYXRvci5fY29uZmlndXJlUHl0aG9uU3VicHJvamVjdFVwZ3JhZGVEZXBzKFxuICAgICAgICB0aGlzLFxuICAgICAgICB1cGdyYWRlRGVwc1Rhc2tcbiAgICAgICk7XG5cbiAgICAgIC8vIFVwZ3JhZGUgbm9kZSBzdWJwcm9qZWN0cyB0b2dldGhlclxuICAgICAgdXBncmFkZURlcHNUYXNrLmV4ZWMoXG4gICAgICAgIE5vZGVQYWNrYWdlVXRpbHMuY29tbWFuZC5leGVjKFxuICAgICAgICAgIHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcixcbiAgICAgICAgICBcIm5wbS1jaGVjay11cGRhdGVzXCIsXG4gICAgICAgICAgXCItLWRlZXBcIixcbiAgICAgICAgICBcIi0tcmVqZWN0VmVyc2lvblwiLFxuICAgICAgICAgIFwiMC4wLjBcIixcbiAgICAgICAgICBcIi11XCIsXG4gICAgICAgICAgXCItLWRlcFwiLFxuICAgICAgICAgIFwicHJvZCxkZXYscGVlcixvcHRpb25hbCxidW5kbGVcIixcbiAgICAgICAgICBcIi0tdGFyZ2V0PW1pbm9yXCJcbiAgICAgICAgKVxuICAgICAgKTtcbiAgICAgIC8vIFN5bmMgZGVwZW5kZW5jeSB2ZXJzaW9ucyBhY3Jvc3Mgbm9kZSBzdWJwcm9qZWN0c1xuICAgICAgdXBncmFkZURlcHNUYXNrLmV4ZWMoXG4gICAgICAgIE5vZGVQYWNrYWdlVXRpbHMuY29tbWFuZC5leGVjKFxuICAgICAgICAgIHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcixcbiAgICAgICAgICBcInN5bmNwYWNrXCIsXG4gICAgICAgICAgXCJmaXgtbWlzbWF0Y2hlc1wiXG4gICAgICAgIClcbiAgICAgICk7XG4gICAgICB1cGdyYWRlRGVwc1Rhc2suZXhlYyhgcm0gJHt0aGlzLnBhY2thZ2UubG9ja0ZpbGV9YCk7XG4gICAgICB1cGdyYWRlRGVwc1Rhc2suZXhlYyhcbiAgICAgICAgTm9kZVBhY2thZ2VVdGlscy5jb21tYW5kLmluc3RhbGwodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyKVxuICAgICAgKTtcbiAgICAgIHVwZ3JhZGVEZXBzVGFzay5leGVjKFxuICAgICAgICBOb2RlUGFja2FnZVV0aWxzLmNvbW1hbmQuZXhlYyh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIsIFwicHJvamVuXCIpXG4gICAgICApO1xuXG4gICAgICBuZXcgSnNvbkZpbGUodGhpcywgXCIuc3luY3BhY2tyYy5qc29uXCIsIHtcbiAgICAgICAgb2JqOlxuICAgICAgICAgIHRoaXMuX29wdGlvbnMubW9ub3JlcG9VcGdyYWRlRGVwc09wdGlvbnM/LnN5bmNwYWNrQ29uZmlnIHx8XG4gICAgICAgICAgU3luY3BhY2suREVGQVVMVF9DT05GSUcsXG4gICAgICAgIHJlYWRvbmx5OiB0cnVlLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgcHJlU3ludGhlc2l6ZSgpOiB2b2lkIHtcbiAgICB0aGlzLmFkZFVwZ3JhZGVEZXBzVGFzaygpO1xuXG4gICAgTm9kZVBhY2thZ2VVdGlscy5yZW1vdmVQcm9qZW5TY3JpcHQodGhpcyk7XG4gICAgdGhpcy5ueENvbmZpZ3VyYXRvci5wcmVTeW50aGVzaXplKCk7XG4gICAgc3VwZXIucHJlU3ludGhlc2l6ZSgpO1xuXG4gICAgaWYgKHRoaXMuX29wdGlvbnMud29ya3NwYWNlQ29uZmlnPy5saW5rTG9jYWxXb3Jrc3BhY2VCaW5zID09PSB0cnVlKSB7XG4gICAgICB0aGlzLmxpbmtMb2NhbFdvcmtzcGFjZUJpbnMoKTtcbiAgICB9XG5cbiAgICB0aGlzLnN1YnByb2plY3RzLmZvckVhY2goKHN1YlByb2plY3QpID0+IHtcbiAgICAgIGlmIChOb2RlUGFja2FnZVV0aWxzLmlzTm9kZVByb2plY3Qoc3ViUHJvamVjdCkpIHtcbiAgICAgICAgLy8gUmVtb3ZlIGFueSBzdWJwcm9qZWN0IC5ucG1yYyBmaWxlcyBzaW5jZSBvbmx5IHRoZSByb290IG9uZSBtYXR0ZXJzXG4gICAgICAgIHN1YlByb2plY3QudHJ5UmVtb3ZlRmlsZShcIi5ucG1yY1wiKTtcbiAgICAgICAgTm9kZVBhY2thZ2VVdGlscy5yZW1vdmVQcm9qZW5TY3JpcHQoc3ViUHJvamVjdCk7XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICBzeW5jUHJvamVuVmVyc2lvbnModGhpcy5zdWJwcm9qZWN0cywgdGhpcy5wcm9qZW5WZXJzaW9uKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW5oZXJpdERvY1xuICAgKi9cbiAgc3ludGgoKSB7XG4gICAgdGhpcy52YWxpZGF0ZVN1YlByb2plY3RzKCk7XG4gICAgdGhpcy51cGRhdGVXb3Jrc3BhY2UoKTtcbiAgICB0aGlzLmluc3RhbGxOb25Ob2RlRGVwZW5kZW5jaWVzKCk7XG4gICAgdGhpcy5yZXNvbHZlTG9jYWxCdW5EZXBlbmRlbmNpZXMoKTtcblxuICAgIC8vIERpc2FibGUgbm9kZSB3YXJuaW5ncyBpZiBjb25maWd1cmVkXG4gICAgaWYgKHRoaXMuX29wdGlvbnMuZGlzYWJsZU5vZGVXYXJuaW5ncykge1xuICAgICAgdGhpcy5kaXNhYmxlTm9kZVdhcm5pbmdzKCk7XG4gICAgfVxuXG4gICAgLy8gUHJldmVudCBzdWIgTm9kZVByb2plY3QgcGFja2FnZXMgZnJvbSBgcG9zdFN5bnRoZXNpc2Agd2hpY2ggd2lsbCBjYXVzZSBpbmRpdmlkdWFsL2V4dHJhbmVvdXMgaW5zdGFsbHMuXG4gICAgLy8gVGhlIHdvcmtzcGFjZSBwYWNrYWdlIGluc3RhbGwgd2lsbCBoYW5kbGUgYWxsIHRoZSBzdWIgTm9kZVByb2plY3QgcGFja2FnZXMgYXV0b21hdGljYWxseS5cbiAgICB0aGlzLnN1YnByb2plY3RzLmZvckVhY2goKHN1YlByb2plY3QpID0+IHtcbiAgICAgIGlmIChOb2RlUGFja2FnZVV0aWxzLmlzTm9kZVByb2plY3Qoc3ViUHJvamVjdCkpIHtcbiAgICAgICAgY29uc3Qgc3ViTm9kZVByb2plY3Q6IE5vZGVQcm9qZWN0ID0gc3ViUHJvamVjdCBhcyBOb2RlUHJvamVjdDtcbiAgICAgICAgY29uc3Qgc3ViTm9kZVByb2plY3RSZXNvbHZlciA9XG4gICAgICAgICAgLy8gQHRzLWlnbm9yZSAtIGByZXNvbHZlRGVwc0FuZFdyaXRlUGFja2FnZUpzb25gIGlzIHByaXZhdGVcbiAgICAgICAgICBzdWJOb2RlUHJvamVjdC5wYWNrYWdlLnJlc29sdmVEZXBzQW5kV3JpdGVQYWNrYWdlSnNvbjtcbiAgICAgICAgLy8gQHRzLWlnbm9yZSAtIGBpbnN0YWxsRGVwZW5kZW5jaWVzYCBpcyBwcml2YXRlXG4gICAgICAgIHN1Yk5vZGVQcm9qZWN0LnBhY2thZ2UuaW5zdGFsbERlcGVuZGVuY2llcyA9ICgpID0+IHtcbiAgICAgICAgICB0aGlzLnN1Yk5vZGVQcm9qZWN0UmVzb2x2ZXMucHVzaCgoKSA9PlxuICAgICAgICAgICAgc3ViTm9kZVByb2plY3RSZXNvbHZlci5hcHBseShzdWJOb2RlUHJvamVjdC5wYWNrYWdlKVxuICAgICAgICAgICk7XG4gICAgICAgIH07XG4gICAgICAgIC8vIEB0cy1pZ25vcmUgLSBgcmVzb2x2ZURlcHNBbmRXcml0ZVBhY2thZ2VKc29uYCBpcyBwcml2YXRlXG4gICAgICAgIHN1Yk5vZGVQcm9qZWN0LnBhY2thZ2UucmVzb2x2ZURlcHNBbmRXcml0ZVBhY2thZ2VKc29uID0gKCkgPT4ge307XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICB0aGlzLm54Q29uZmlndXJhdG9yLnN5bnRoKCk7XG4gICAgc3VwZXIuc3ludGgoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW5oZXJpdERvY1xuICAgKi9cbiAgcG9zdFN5bnRoZXNpemUoKTogdm9pZCB7XG4gICAgc3VwZXIucG9zdFN5bnRoZXNpemUoKTtcbiAgICB0aGlzLnJlc29sdmVTdWJOb2RlUHJvamVjdHMoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXNvbHZlIHN1YiBgTm9kZVBhY2thZ2VgIGRlcGVuZGVuY2llcy5cbiAgICovXG4gIHByaXZhdGUgcmVzb2x2ZVN1Yk5vZGVQcm9qZWN0cygpIHtcbiAgICBpZiAodGhpcy5zdWJOb2RlUHJvamVjdFJlc29sdmVzLmxlbmd0aCkge1xuICAgICAgaWYgKCF0aGlzLnBhY2thZ2UuZmlsZS5jaGFuZ2VkKSB7XG4gICAgICAgIC8vIEZvcmNlIHdvcmtzcGFjZSBpbnN0YWxsIGRlcHMgc2luY2UgaXQgd291bGQgbm90IGhhdmUgYmVlbiBpbnZva2VkIGR1cmluZyBgcG9zdFN5bnRoZXNpc2AuXG4gICAgICAgIC8vIEB0cy1pZ25vcmUgLSBgaW5zdGFsbERlcGVuZGVuY2llc2AgaXMgcHJpdmF0ZVxuICAgICAgICB0aGlzLnBhY2thZ2UuaW5zdGFsbERlcGVuZGVuY2llcygpO1xuICAgICAgfVxuICAgICAgY29uc3QgY29tcGxldGVkUmVzb2x2ZXMgPSB0aGlzLnN1Yk5vZGVQcm9qZWN0UmVzb2x2ZXMubWFwKChyZXNvbHZlKSA9PlxuICAgICAgICByZXNvbHZlKClcbiAgICAgICk7XG4gICAgICBpZiAoY29tcGxldGVkUmVzb2x2ZXMuc29tZShCb29sZWFuKSkge1xuICAgICAgICAvLyBJbmRpY2F0ZXMgdGhhdCBhIHN1YnByb2plY3QgZGVwZW5kZW5jeSBoYXMgYmVlbiByZXNvbHZlZCBmcm9tICcqJywgc28gdXBkYXRlIHRoZSBsb2NrZmlsZS5cbiAgICAgICAgLy8gQHRzLWlnbm9yZSAtIGBpbnN0YWxsRGVwZW5kZW5jaWVzYCBpcyBwcml2YXRlXG4gICAgICAgIHRoaXMucGFja2FnZS5pbnN0YWxsRGVwZW5kZW5jaWVzKCk7XG4gICAgICB9XG4gICAgfVxuICAgIHRoaXMuc3ViTm9kZVByb2plY3RSZXNvbHZlcyA9IFtdO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuc3VyZXMgc3VicHJvamVjdHMgZG9uJ3QgaGF2ZSBhIGRlZmF1bHQgdGFzayBhbmQgdGhhdCBhbGwgcGFja2FnZXMgdXNlIHRoZSBzYW1lIHBhY2thZ2UgbWFuYWdlci5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVTdWJQcm9qZWN0cygpIHtcbiAgICB0aGlzLnN1YnByb2plY3RzLmZvckVhY2goKHN1YlByb2plY3Q6IGFueSkgPT4ge1xuICAgICAgLy8gRGlzYWJsZSBkZWZhdWx0IHRhc2sgb24gc3VicHJvamVjdHMgYXMgdGhpcyBpc24ndCBzdXBwb3J0ZWQgaW4gYSBtb25vcmVwb1xuICAgICAgc3ViUHJvamVjdC5kZWZhdWx0VGFzaz8ucmVzZXQoKTtcblxuICAgICAgaWYgKFxuICAgICAgICBOb2RlUGFja2FnZVV0aWxzLmlzTm9kZVByb2plY3Qoc3ViUHJvamVjdCkgJiZcbiAgICAgICAgc3ViUHJvamVjdC5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyICE9PSB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXJcbiAgICAgICkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgYCR7c3ViUHJvamVjdC5uYW1lfSBwYWNrYWdlTWFuYWdlciBkb2VzIG5vdCBtYXRjaCB0aGUgbW9ub3JlcG8gcGFja2FnZU1hbmFnZXI6ICR7dGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyfS5gXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgc3VibW9kdWxlIGVudHJ5IHRvIHRoZSBhcHByb3ByaWF0ZSB3b3Jrc3BhY2UgZmlsZS5cbiAgICovXG4gIHByaXZhdGUgdXBkYXRlV29ya3NwYWNlKCkge1xuICAgIC8vIEEgZmluYWwgY2FsbCB0byBhZGRXb3Jrc3BhY2VQYWNrYWdlcyB3aWxsIHVwZGF0ZSB0aGUgbGlzdCBvZiB3b3Jrc3BhY2UgcGFja2FnZXMgd2l0aCBhbnkgc3VicHJvamVjdHMgdGhhdCBoYXZlXG4gICAgLy8gbm90IHlldCBiZWVuIGFkZGVkLCBpbiB0aGUgY29ycmVjdCBvcmRlclxuICAgIHRoaXMuYWRkV29ya3NwYWNlUGFja2FnZXMoKTtcblxuICAgIGxldCBub0hvaXN0ID0gdGhpcy53b3Jrc3BhY2VDb25maWc/Lnlhcm4/Lm5vSG9pc3Q7XG4gICAgLy8gQXV0b21hdGljYWxseSBhZGQgYWxsIHN1Yi1wcm9qZWN0IFwiYnVuZGxlZERlcGVuZGVuY2llc1wiIHRvIHdvcmtzcGFjZSBcImhvaG9pc3RcIiwgb3RoZXJ3aXNlIHRoZXkgYXJlIG5vdCBidW5kbGVkIGluIG5wbSBwYWNrYWdlXG4gICAgaWYgKFxuICAgICAgdGhpcy53b3Jrc3BhY2VDb25maWc/Lnlhcm4/LmRpc2FibGVOb0hvaXN0QnVuZGxlZCAhPT0gdHJ1ZSAmJlxuICAgICAgW1xuICAgICAgICBOb2RlUGFja2FnZU1hbmFnZXIuWUFSTixcbiAgICAgICAgTm9kZVBhY2thZ2VNYW5hZ2VyLllBUk4yLFxuICAgICAgICBOb2RlUGFja2FnZU1hbmFnZXIuWUFSTl9CRVJSWSxcbiAgICAgICAgTm9kZVBhY2thZ2VNYW5hZ2VyLllBUk5fQ0xBU1NJQyxcbiAgICAgIF0uaW5jbHVkZXModGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyKVxuICAgICkge1xuICAgICAgY29uc3Qgbm9Ib2lzdEJ1bmRsZWQgPSB0aGlzLnN1YnByb2plY3RzLmZsYXRNYXAoKHN1YikgPT4ge1xuICAgICAgICBpZiAoUHJvamVjdFV0aWxzLmlzTmFtZWRJbnN0YW5jZU9mKHN1YiwgTm9kZVByb2plY3QpKSB7XG4gICAgICAgICAgcmV0dXJuIGdldEJ1bmRsZWREZXBzKHN1YikuZmxhdE1hcCgoZGVwKSA9PiBbXG4gICAgICAgICAgICBgJHtzdWIubmFtZX0vJHtkZXAubmFtZX1gLFxuICAgICAgICAgICAgYCR7c3ViLm5hbWV9LyR7ZGVwLm5hbWV9LypgLFxuICAgICAgICAgIF0pO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBbXTtcbiAgICAgIH0pO1xuXG4gICAgICBpZiAobm9Ib2lzdEJ1bmRsZWQubGVuZ3RoKSB7XG4gICAgICAgIG5vSG9pc3QgPSBbLi4uKG5vSG9pc3QgfHwgW10pLCAuLi5ub0hvaXN0QnVuZGxlZF07XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gQWRkIHdvcmtzcGFjZXMgZm9yIGVhY2ggc3VicHJvamVjdFxuICAgIGlmICh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIgPT09IE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNKSB7XG4gICAgICBuZXcgWWFtbEZpbGUodGhpcywgXCJwbnBtLXdvcmtzcGFjZS55YW1sXCIsIHtcbiAgICAgICAgcmVhZG9ubHk6IHRydWUsXG4gICAgICAgIG9iajoge1xuICAgICAgICAgIHBhY2thZ2VzOiB0aGlzLndvcmtzcGFjZVBhY2thZ2VzLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuICAgIHRoaXMucGFja2FnZS5hZGRGaWVsZChcIndvcmtzcGFjZXNcIiwge1xuICAgICAgcGFja2FnZXM6IHRoaXMud29ya3NwYWNlUGFja2FnZXMsXG4gICAgICBub2hvaXN0OiBub0hvaXN0LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEVuc3VyZXMgdGhlIGluc3RhbGwgdGFzayBmb3Igbm9uLW5vZGUgcHJvamVjdHMgaXMgZXhlY3V0ZWQgcG9zdGluc3RhbGwuXG4gICAqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcml2YXRlIGluc3RhbGxOb25Ob2RlRGVwZW5kZW5jaWVzKCkge1xuICAgIGNvbnN0IGluc3RhbGxQcm9qZWN0cyA9IHRoaXMuc3VicHJvamVjdHMuZmlsdGVyKFxuICAgICAgKHByb2plY3QpID0+XG4gICAgICAgICFQcm9qZWN0VXRpbHMuaXNOYW1lZEluc3RhbmNlT2YocHJvamVjdCwgTm9kZVByb2plY3QpICYmXG4gICAgICAgIHByb2plY3QudGFza3MudHJ5RmluZChcImluc3RhbGxcIilcbiAgICApO1xuXG4gICAgaWYgKGluc3RhbGxQcm9qZWN0cy5sZW5ndGggPiAwKSB7XG4gICAgICAvLyBUT0RPOiBJbnN0YWxsIGVycm9yIG9uIGNsZWFuIHJlcG8gZm9yIHBvc3RpbnN0YWxsIChodHRwczovL2Nsb3VkLm54LmFwcC9ydW5zL01wdFFyMEJ4Z0YpIChodHRwczovL2dpdGh1Yi5jb20vbnJ3bC9ueC9pc3N1ZXMvMTEyMTApXG4gICAgICBjb25zdCBwb3N0aW5zdGFsbFRhc2sgPVxuICAgICAgICB0aGlzLnRhc2tzLnRyeUZpbmQoXCJwb3N0aW5zdGFsbFwiKSA/PyB0aGlzLmFkZFRhc2soXCJwb3N0aW5zdGFsbFwiKTtcblxuICAgICAgY29uc3QgbnhSdW5NYW55SW5zdGFsbCA9IHRoaXMuY29tcG9zZU54UnVuTWFueUNvbW1hbmQoe1xuICAgICAgICB0YXJnZXQ6IFwiaW5zdGFsbFwiLFxuICAgICAgICBwcm9qZWN0czogaW5zdGFsbFByb2plY3RzLm1hcCgocHJvamVjdCkgPT4gcHJvamVjdC5uYW1lKSxcbiAgICAgICAgcGFyYWxsZWw6IDEsXG4gICAgICB9KTtcblxuICAgICAgcG9zdGluc3RhbGxUYXNrLmV4ZWMoXG4gICAgICAgIE5vZGVQYWNrYWdlVXRpbHMuY29tbWFuZC5leGVjKFxuICAgICAgICAgIHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcixcbiAgICAgICAgICAuLi5ueFJ1bk1hbnlJbnN0YWxsXG4gICAgICAgIClcbiAgICAgICk7XG5cbiAgICAgIC8vIEVuc3VyZSB0aGF0IGluc3RhbGwtcHkgZm9sbG93cyBkZXBlbmRlbmN5IG9yZGVyXG4gICAgICBpbnN0YWxsUHJvamVjdHMuZm9yRWFjaCgocCkgPT4ge1xuICAgICAgICBOeFByb2plY3QuZW5zdXJlKHApLnNldFRhcmdldChcImluc3RhbGxcIiwge1xuICAgICAgICAgIGRlcGVuZHNPbjogW1wiXmluc3RhbGxcIl0sXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFN1cHByZXNzIE5vZGUgd2FybmluZ3MgZnJvbSBiZWluZyBwcmVzZW50ZWQgaW4gdGhlIGNvbnNvbGUgd2hlbiBydW5uaW5nIGJ1aWxkcy5cbiAgICovXG4gIHByaXZhdGUgZGlzYWJsZU5vZGVXYXJuaW5ncygpIHtcbiAgICB0aGlzLnRhc2tzLmFkZEVudmlyb25tZW50KFwiTk9ERV9OT19XQVJOSU5HU1wiLCBcIjFcIik7XG4gICAgdGhpcy5zdWJwcm9qZWN0cy5mb3JFYWNoKChzdWJQcm9qZWN0KSA9PlxuICAgICAgc3ViUHJvamVjdC50YXNrcy5hZGRFbnZpcm9ubWVudChcIk5PREVfTk9fV0FSTklOR1NcIiwgXCIxXCIpXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXNvbHZlIGFsbCBsb2NhbCB3b3Jrc3BhY2UgZGVwZW5kZW5jaWVzIHRvIGtlZXAgYnVuIGhhcHB5LlxuICAgKi9cbiAgcHJpdmF0ZSByZXNvbHZlTG9jYWxCdW5EZXBlbmRlbmNpZXMoKTogdm9pZCB7XG4gICAgaWYgKHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlciAhPT0gTm9kZVBhY2thZ2VNYW5hZ2VyLkJVTikge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IG5vZGVTdWJQcm9qZWN0TmFtZXMgPSBuZXcgU2V0KFxuICAgICAgdGhpcy5zdWJwcm9qZWN0c1xuICAgICAgICAuZmlsdGVyKChzKSA9PiBOb2RlUGFja2FnZVV0aWxzLmlzTm9kZVByb2plY3QocykpXG4gICAgICAgIC5tYXAoKHMpID0+IChzIGFzIE5vZGVQcm9qZWN0KS5wYWNrYWdlLnBhY2thZ2VOYW1lKVxuICAgICk7XG5cbiAgICB0aGlzLnN1YnByb2plY3RzLmZvckVhY2goKHN1YlByb2plY3QpID0+IHtcbiAgICAgIGlmIChOb2RlUGFja2FnZVV0aWxzLmlzTm9kZVByb2plY3Qoc3ViUHJvamVjdCkpIHtcbiAgICAgICAgc3ViUHJvamVjdC5kZXBzLmFsbFxuICAgICAgICAgIC5maWx0ZXIoKGRlcCkgPT4gbm9kZVN1YlByb2plY3ROYW1lcy5oYXMoZGVwLm5hbWUpKVxuICAgICAgICAgIC5mb3JFYWNoKChkKSA9PiB7XG4gICAgICAgICAgICBzd2l0Y2ggKGQudHlwZSkge1xuICAgICAgICAgICAgICBjYXNlIERlcGVuZGVuY3lUeXBlLlJVTlRJTUU6XG4gICAgICAgICAgICAgICAgc3ViUHJvamVjdC5hZGREZXBzKGAke2QubmFtZX1Ad29ya3NwYWNlOipgKTtcbiAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgICAgY2FzZSBEZXBlbmRlbmN5VHlwZS5CVUlMRDpcbiAgICAgICAgICAgICAgICBzdWJQcm9qZWN0LmFkZERldkRlcHMoYCR7ZC5uYW1lfUB3b3Jrc3BhY2U6KmApO1xuICAgICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgICBjYXNlIERlcGVuZGVuY3lUeXBlLlBFRVI6XG4gICAgICAgICAgICAgICAgc3ViUHJvamVjdC5hZGRQZWVyRGVwcyhgJHtkLm5hbWV9QHdvcmtzcGFjZToqYCk7XG4gICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgICAgICAgY29uc29sZS53YXJuKFxuICAgICAgICAgICAgICAgICAgYENhbm5vdCB1cGRhdGUgbG9jYWwgZGVwZW5kZW5jeSBkdWUgdG8gdW5zdXBwb3J0ZWQgdHlwZTogJHtkLnR5cGV9YFxuICAgICAgICAgICAgICAgICk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBHZXRzIGJ1bmRsZWQgZGVwZW5kZW5jaWVzIGZvciBhIGdpdmVuIHByb2plY3RcbiAqL1xuZnVuY3Rpb24gZ2V0QnVuZGxlZERlcHMocHJvamVjdDogUHJvamVjdCk6IERlcGVuZGVuY3lbXSB7XG4gIHJldHVybiBwcm9qZWN0LmRlcHMuYWxsLmZpbHRlcigoZGVwKSA9PiBkZXAudHlwZSA9PT0gRGVwZW5kZW5jeVR5cGUuQlVORExFRCk7XG59XG4iXX0=