"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deepMerge = exports.asUndefinedIfEmpty = exports.isEmpty = exports.cloneDeep = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const util_1 = require("projen/lib/util");
/**
 * Utility to deeply clone a value
 * @param value Value to clone
 * @returns Cloned value
 */
function cloneDeep(value) {
    return JSON.parse(JSON.stringify(value));
}
exports.cloneDeep = cloneDeep;
/** Indicates if value is considered empty */
function isEmpty(value) {
    if (value == null)
        return true;
    if (typeof value === "boolean") {
        return false;
    }
    if (typeof value === "string") {
        return value.length === 0;
    }
    if (Array.isArray(value)) {
        return value.length === 0;
    }
    if ((0, util_1.isObject)(value)) {
        return Object.keys(value).length === 0;
    }
    return false;
}
exports.isEmpty = isEmpty;
/** Replace empty value with undefined */
function asUndefinedIfEmpty(value) {
    if (isEmpty(value))
        return undefined;
    return value;
}
exports.asUndefinedIfEmpty = asUndefinedIfEmpty;
/**
 * Recursively merge objects together into a new object with extends support for appending arrays.
 *
 * This is a clone of [projen/lib/util#deepMerge](https://github.com/projen/projen/blob/55ac3657a270285db63e1a6008b3848b36775626/src/util.ts#L218-L281)
 * with added functionality to support appending arrays.
 *
 * @see [projen/lib/util#deepMerge](https://github.com/projen/projen/blob/55ac3657a270285db63e1a6008b3848b36775626/src/util.ts#L218-L281)
 */
function deepMerge(objects, options) {
    const append = options?.append ?? false;
    const destructive = options?.destructive ?? false;
    objects = objects.map(cloneDeep);
    if (append === false)
        return (0, util_1.deepMerge)(objects, destructive);
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            if ((0, util_1.isObject)(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = value;
                }
                if (Array.isArray(value)) {
                    if (Array.isArray(target[key])) {
                        target[key].push(...value);
                    }
                    else {
                        target[key] = value;
                    }
                }
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof output === "object" &&
                    Object.keys(output).length === 0 &&
                    destructive) {
                    delete target[key];
                }
            }
            else if (value === undefined && destructive) {
                delete target[key];
            }
            else if (Array.isArray(value)) {
                if (Array.isArray(target[key])) {
                    // Append to existing array
                    target[key].push(...value);
                }
                else {
                    // Override with array value
                    target[key] = value;
                }
            }
            else if (typeof value !== "undefined") {
                target[key] = value;
            }
        }
    }
    const others = objects.filter((x) => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach((other) => mergeOne(into, other));
    return into;
}
exports.deepMerge = deepMerge;
//# sourceMappingURL=data:application/json;base64,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