"use strict";
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.FeatureBranches = void 0;
const path = require("path");
const pdk_nag_1 = require("../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
class FeatureBranches extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const buildCommands = props.synthShellStepPartialProps?.commands &&
            props.synthShellStepPartialProps.commands.length > 0
            ? props.synthShellStepPartialProps.commands
            : ["npx nx run-many --target=build --all"];
        const installCommands = props.synthShellStepPartialProps?.installCommands &&
            props.synthShellStepPartialProps.installCommands.length > 0
            ? props.synthShellStepPartialProps.installCommands
            : [
                "npm install -g aws-cdk",
                "yarn install --frozen-lockfile || npx projen && yarn install --frozen-lockfile",
            ];
        const cdkCommand = props.cdkCommand ?? "npx cdk";
        const createFeatureBranchProject = new aws_codebuild_1.Project(this, "CreateFeatureBranchProject", {
            ...props.codeBuildDefaults,
            description: "Build project to deploy feature branch pipelines",
            source: aws_codebuild_1.Source.codeCommit({ repository: props.codeRepository }),
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_7_0,
                computeType: aws_codebuild_1.ComputeType.SMALL,
                ...props.codeBuildDefaults?.buildEnvironment,
                privileged: props.dockerEnabledForSynth,
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObjectToYaml({
                version: "0.2",
                phases: {
                    install: {
                        commands: installCommands,
                    },
                    build: {
                        commands: [
                            ...buildCommands,
                            `cd ${props.cdkSrcDir}`,
                            `${cdkCommand} synth`,
                            `${cdkCommand} deploy --require-approval=never`,
                        ],
                    },
                },
                artifacts: {
                    files: ["**/*"],
                },
            }),
        });
        if (props.codeBuildDefaults?.rolePolicy) {
            props.codeBuildDefaults.rolePolicy.forEach((policy) => {
                createFeatureBranchProject.addToRolePolicy(policy);
            });
        }
        createFeatureBranchProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ["sts:AssumeRole"],
            resources: [`arn:*:iam::${aws_cdk_lib_1.Stack.of(this).account}:role/*`],
            conditions: {
                "ForAnyValue:StringEquals": {
                    "iam:ResourceTag/aws-cdk:bootstrap-role": [
                        "image-publishing",
                        "file-publishing",
                        "deploy",
                    ],
                },
            },
        }));
        const createFeatureBranchFunction = new aws_lambda_1.Function(this, "LambdaTriggerCreateBranch", {
            runtime: aws_lambda_1.Runtime.PYTHON_3_12,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "lambda/create_branch")),
            handler: "create_branch.handler",
            environment: {
                CODEBUILD_PROJECT: createFeatureBranchProject.projectName,
                MAIN_BRANCH: props.defaultBranchName,
            },
        });
        createFeatureBranchFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ["codebuild:StartBuild"],
            resources: [createFeatureBranchProject.projectArn],
        }));
        const destroyFeatureBranchFunction = new aws_lambda_1.Function(this, "LambdaTriggerDestroyBranch", {
            runtime: aws_lambda_1.Runtime.PYTHON_3_12,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "lambda/destroy_branch")),
            handler: "destroy_branch.handler",
            environment: {
                MAIN_BRANCH: props.defaultBranchName,
                REPO_NAME: props.codeRepository.repositoryName,
            },
        });
        destroyFeatureBranchFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ["cloudformation:DeleteStack"],
            resources: [
                aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: "cloudformation",
                    resource: "stack",
                    resourceName: "*/*",
                }),
            ],
            conditions: {
                "ForAllValues:StringEquals": {
                    "aws:TagKeys": ["FeatureBranch", "RepoName"],
                },
            },
        }));
        destroyFeatureBranchFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ["tag:GetResources"],
            resources: ["*"],
        }));
        props.codeRepository.onReferenceCreated("BranchCreateTrigger", {
            target: new aws_events_targets_1.LambdaFunction(createFeatureBranchFunction),
            description: "AWS CodeCommit reference created event.",
            eventPattern: {
                detail: {
                    referenceType: ["branch"],
                    referenceName: props.branchNamePrefixes.map((prefix) => ({ prefix })),
                },
            },
        });
        props.codeRepository.onReferenceDeleted("BranchDestroyTrigger", {
            target: new aws_events_targets_1.LambdaFunction(destroyFeatureBranchFunction),
            description: "AWS CodeCommit reference deleted event.",
            eventPattern: {
                detail: {
                    referenceType: ["branch"],
                    referenceName: props.branchNamePrefixes.map((prefix) => {
                        return { prefix };
                    }),
                },
            },
        });
        const stack = aws_cdk_lib_1.Stack.of(this);
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(createFeatureBranchProject.role, [
                {
                    id: RuleId,
                    reason: "CodeBuild requires get, list, and pull access to the CodeCommit repository.",
                    appliesTo: [
                        "Action::codecommit:Get*",
                        "Action::codecommit:List*",
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodeBuild requires access to create report groups that are dynamically determined.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:codebuild:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:report-group/<[a-zA-Z0-9]*CreateFeatureBranchProject.*>-\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodeBuild requires access to manage logs and streams whose names are dynamically determined.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/codebuild/<[a-zA-Z0-9]*CreateFeatureBranchProject.*>:\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodeBuild requires access to assume a role from within the current account limited by a condition in order to deploy.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:\\*:iam::${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:role/\\*$/g`,
                        },
                    ],
                },
            ], true);
            cdk_nag_1.NagSuppressions.addResourceSuppressions(destroyFeatureBranchFunction.role, [
                {
                    id: RuleId,
                    reason: "The DestroyBranch Lambda requires access to delete any stacks with specific tags.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:cloudformation:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:stack/\\*/\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "The DestroyBranch Lambda requires access to look up CloudFormation stacks by tag. The Resource Group Tagging API must use 'Resource': '*'.",
                    appliesTo: ["Resource::*"],
                },
            ], true);
        });
        ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(createFeatureBranchFunction, [
                {
                    id: RuleId,
                    reason: "Lambda functions use the default AWS LambdaBasicExecutionRole managed role.",
                    appliesTo: [
                        {
                            regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole$/g`,
                        },
                    ],
                },
            ], true);
            cdk_nag_1.NagSuppressions.addResourceSuppressions(destroyFeatureBranchFunction, [
                {
                    id: RuleId,
                    reason: "Lambda functions use the default AWS LambdaBasicExecutionRole managed role.",
                    appliesTo: [
                        {
                            regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole$/g`,
                        },
                    ],
                },
            ], true);
        });
        [
            "AwsSolutions-CB4",
            "AwsPrototyping-CodeBuildProjectKMSEncryptedArtifacts",
        ].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(createFeatureBranchProject, [
                {
                    id: RuleId,
                    reason: "Encryption of Codebuild is not required.",
                },
            ]);
        });
    }
}
exports.FeatureBranches = FeatureBranches;
//# sourceMappingURL=data:application/json;base64,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