"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsiteOrigin = exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using by default a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps = (props) => {
            this.validateEncryptionSettings(props);
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
            props.websiteBucket && this.validateBucketConfig(props.websiteBucket);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error("validateRuntimeConfig only accepts non-null RuntimeOptions.");
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error("RuntimeOptions.jsonFileName must be a json file.");
            }
        };
        this.validateBucketConfig = (bucket) => {
            if (bucket.isWebsite) {
                throw new Error("Website buckets cannot be configured as websites as this will break Cloudfront hosting!");
            }
        };
        this.validateEncryptionSettings = ({ defaultWebsiteBucketEncryption, defaultWebsiteBucketEncryptionKey, }) => {
            if (defaultWebsiteBucketEncryptionKey &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS) {
                throw new Error("Bucket encryption should be set to KMS if providing a defaultWebsiteBucketEncryptionKey.");
            }
            if (defaultWebsiteBucketEncryption &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.S3_MANAGED) {
                throw new Error("Only KMS and S3_MANAGED encryption are supported on the default bucket.");
            }
        };
        this.suppressCDKNagViolations = (props) => {
            const stack = aws_cdk_lib_1.Stack.of(this);
            !props.distributionProps?.certificate &&
                [
                    "AwsSolutions-CFR4",
                    "AwsPrototyping-CloudFrontDistributionHttpsViewerNoOutdatedSSL",
                ].forEach((RuleId) => {
                    cdk_nag_1.NagSuppressions.addResourceSuppressions(this.cloudFrontDistribution, [
                        {
                            id: RuleId,
                            reason: "Certificate is not mandatory therefore the Cloudfront certificate will be used.",
                        },
                    ]);
                });
            ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                    {
                        id: RuleId,
                        reason: "Latest runtime cannot be configured. CDK will need to upgrade the BucketDeployment construct accordingly.",
                    },
                ], true);
            });
            ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                    {
                        id: RuleId,
                        reason: "All Policies have been scoped to a Bucket. Given Buckets can contain arbitrary content, wildcard resources with bucket scope are required.",
                        appliesTo: [
                            {
                                regex: "/^Action::s3:.*$/g",
                            },
                            {
                                regex: `/^Resource::.*$/g`,
                            },
                        ],
                    },
                ], true);
            });
            ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                    {
                        id: RuleId,
                        reason: "Buckets can contain arbitrary content, therefore wildcard resources under a bucket are required.",
                        appliesTo: [
                            {
                                regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole$/g`,
                            },
                        ],
                    },
                ], true);
            });
            ["AwsSolutions-S1", "AwsPrototyping-S3BucketLoggingEnabled"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                    {
                        id: RuleId,
                        reason: "Access Log buckets should not have s3 bucket logging",
                    },
                ], true);
            });
        };
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        this.validateProps(props);
        const accessLogsBucket = new aws_s3_1.Bucket(this, "AccessLogsBucket", {
            versioned: false,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            objectOwnership: aws_s3_1.ObjectOwnership.OBJECT_WRITER,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        // S3 Bucket to hold website files
        this.websiteBucket =
            props.websiteBucket ??
                new aws_s3_1.Bucket(this, "WebsiteBucket", {
                    versioned: true,
                    enforceSSL: true,
                    autoDeleteObjects: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                    objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_ENFORCED,
                    encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                    publicReadAccess: false,
                    blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                    serverAccessLogsPrefix: "website-access-logs",
                    serverAccessLogsBucket: accessLogsBucket,
                });
        // Web ACL
        const { distributionProps } = props;
        const webAclArn = distributionProps?.webAclId ??
            (props.webAclProps?.disable
                ? undefined
                : new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", props.webAclProps)
                    .webAclArn);
        // Cloudfront Distribution
        const logBucket = props.distributionProps?.logBucket ||
            new aws_s3_1.Bucket(this, "DistributionLogBucket", {
                enforceSSL: true,
                autoDeleteObjects: true,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_PREFERRED,
                encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                publicReadAccess: false,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                serverAccessLogsPrefix: "distribution-access-logs",
                serverAccessLogsBucket: accessLogsBucket,
            });
        const originAccessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, "OriginAccessIdentity");
        this.websiteBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [this.websiteBucket.bucketArn],
            actions: ["s3:ListBucket"],
            principals: [originAccessIdentity.grantPrincipal],
        }));
        const defaultRootObject = distributionProps?.defaultRootObject ?? "index.html";
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            ...distributionProps,
            webAclId: webAclArn,
            enableLogging: true,
            logBucket: logBucket,
            defaultBehavior: {
                ...distributionProps?.defaultBehavior,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket, {
                    originAccessIdentity,
                }),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            defaultRootObject,
            errorResponses: distributionProps?.errorResponses ?? [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
            ],
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            ...props.bucketDeploymentProps,
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(props.runtimeOptions?.jsonFileName ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, props.runtimeOptions?.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
        });
        new aws_cdk_lib_1.CfnOutput(this, "DistributionDomainName", {
            value: this.cloudFrontDistribution.domainName,
        });
        this.suppressCDKNagViolations(props);
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws/pdk.static_website.StaticWebsite", version: "0.23.39" };
/**
 * If passing in distributionProps, the default behaviour.origin is a required parameter. An instance of this class can be passed in
 * to make the compiler happy.
 */
class StaticWebsiteOrigin {
    bind(_scope, _options) {
        throw new Error("This should never be called");
    }
}
exports.StaticWebsiteOrigin = StaticWebsiteOrigin;
_b = JSII_RTTI_SYMBOL_1;
StaticWebsiteOrigin[_b] = { fqn: "@aws/pdk.static_website.StaticWebsiteOrigin", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,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